"""Persistent properties
"""

import prel
import datetime

class pprop(object):
    """Base class of all persistent properties

    All pprop objects have the following attributes

    name    -- The name of the attribute for the property
    key     -- The key used to store values in the instance __dict__
    ptype   -- The type of the objects that the property stores
    default -- The default for properties that have not been set yet
    """
    def _fastget(self, instance, database=None):
        """Quick version of __get__, used by the database interfaces

        Should return a value that the database can pass to SQL, may call back
        into the database to get a conversion function
        """
        return instance.__dict__.get(self.key, self.default)

    def _fastset(self, instance, value, database=None):
        """Quick version of __set__, used by the database interfaces

        Gets the value directly form the database, may call back to the
        database to get a conversion function
        """
        instance.__dict__[self.key] = value

class pval(pprop):
    """Persistent (atomic) values

    Atomic values have a type (attribute ptype) and a default. Assignment to
    persistent values are type checked. Access to a persistent values returns
    the default if no value has been assigned yet
    """
    def __init__(self, default=None):
        """Create a persistent property

        Optionally, you can provide an alternative default
        """
        if default is not None:
            if isinstance(default, self.ptype):
                self.default = default
            else:
                raise TypeError, ("Default: expected %s, got %s" %
                                  (self.ptype, type(default)))

    def __get__(self, instance, cls=None):
        """Property getter

        If called as instance.prop return the value of prop. Return the
        default if necessary

        If called as cls.prop return an attr_ref object.
        """
        if instance is None:
            return prel.value_attr(cls, self)
        else:
            return instance.__dict__.get(self.key, self.default)

    def __set__(self, instance, value):
        """Property setter

        Check the type of the value, and store in instance.__dict__. Set the
        dirty bit of instance.
        """
        if isinstance(value, self.ptype):
            instance.__dict__[self.key] = value
            instance._dirty = True
        else:
            raise TypeError, "Expected %s, got %s" % (self.ptype, type(value))

class pint(pval):
    """A persistent value of type 'int'
    """
    ptype = int
    default = 0
    def create_column(self, database):
        """Ask database to create a column for integers"""
        return database.create_int_column(self)

class pfloat(pval):
    """A persistent value of type 'float'
    """
    ptype = (int, float)
    default = 0.0
    def create_column(self, database):
        """Ask database to create column for floats"""
        return database.create_float_column(self)

class pstr(pval):
    """A persistent value of type 'str'
    """
    ptype = str
    default = ''
    def create_column(self, database):
        """Ask database to create column for text with arbitrary length"""
        return database.create_str_column(self)
    def _fastget(self, instance, database=None):
        s = instance.__dict__.get(self.key, self.default)
        return database.quote_str(s)

class pdate(pval):
    """A persistent value of type 'datetime.date'
    """
    ptype = datetime.date
    default = datetime.date(1,1,1)
    def create_column(self, database):
        """Ask database to create column for date"""
        return database.create_date_column(self)
    def _fastget(self, instance, database):
        s = instance.__dict__.get(self.key, self.default)
        return database.quote_date(s)
    def _fastset(self, instance, val, database):
        instance.__dict__[self.key] = database.unquote_date(val)

class ptime(pval):
    """A persistent value of type 'datetime.time'
    """
    ptype = datetime.time
    default = datetime.time()
    def create_column(self, database):
        """Ask database to create column for time"""
        return database.create_time_column(self)
    def _fastget(self, instance, database=None):
        s = instance.__dict__.get(self.key, self.default)
        return database.quote_time(s)
    def _fastset(self, instance, val, database):
        instance.__dict__[self.key] = database.unquote_time(val)
    
class pdatetime(pval):
    """A persistent value of type 'datetime.datetime'
    """
    ptype = datetime.datetime
    default = datetime.datetime(1, 1, 1)
    def create_column(self, database):
        """Ask database to create column for datetime"""
        return database.create_datetime_column(self)
    def _fastget(self, instance, database=None):
        s = instance.__dict__.get(self.key, self.default)
        return database.quote_datetime(s)
    def _fastset(self, instance, val, database):
        instance.__dict__[self.key] = database.unquote_datetime(val)

## class pchar(pval):
##     ptype = str
##     def __init__(self, length, default=None):
##         if not isinstance(length, (int, long)):
##             raise TypeError, "'length' of pchar() must be an integer"
##         if length<1 or length>255:
##             raise ValueError, "'length' pf pchar() must be between 1 and 255"
        

class pref(pprop):
    """A reference to another persistent object

    A reference can have three types of values:
    None       -- No object referred to
    obj        -- an object --> isinstance(obj, ptype) == True
    (cls, oid) -- a tuple, implies: issubclass(cls, ptype)

    If the value is the tuple, then the getter will return cls(oid=oid).
    Hence, objects are instantiated when first referred to.
    """
    def __init__(self, ptype, default=None):
        self.ptype = ptype
        if default is None or self._value_ok(default):
            self.default = default
        else:
            raise TypeError, "Expected %s, got %s" % (self.ptype, type(default))

    def create_column(self, database):
        """Ask database to create columns for object references"""
        return database.create_ref_columns(self)

    def _value_ok(self, value):
        """Return True if the value is a valid reference

        Valid references imply value==None, or isinstance(value, ptype), or
        isinstance(value, tuple) and issubclass(value[0], ptype)
        """
        if isinstance(value, self.ptype):
            return True
        elif (isinstance(value, tuple) and issubclass(value[0], self.ptype) and
            (value[1] is None or isinstance(value[1], (long, int)))):
            return True
        else:
            return False

    def __get__(self, instance, cls=None):
        """Property getter

        If called as instance.prop return the value of prop. Return the
        default if necessary. If the return value is a tuple (cls, oid), return
        cls(oid=oid) in stead.

        If called as cls.prop return an attr_ref object.
        """
        if instance is None:
            return prel.ref_attr(cls, self)
        else:
            val = instance.__dict__.get(self.key, self.default)
            if isinstance(val, tuple):
                if val[1] is None:
                    val = val[0]()
                else:
                    val = val[0](oid=val[1])
            instance.__dict__[self.key] = val
            return val

    def __set__(self, instance, value):
        """Property setter

        Check the type of the value, and store in instance.__dict__. Set the
        dirty bit of instance.
        """
        if self._value_ok(value):
            instance.__dict__[self.key] = value
            instance._dirty = True
        else:
            raise TypeError, "Expected %s, got %s" % (self.ptype, type(value))

    def _fastget(self, instance, database):
        return database.quote_tid_oid(instance.__dict__.get(self.key, self.default))

