"""Interface to SQLite databases

Requires PySQLite (http://sourceforge.net/projects/pysqlite), by Michael
Owens, and Gerhard Haring.
"""

import db_base, sql_db, _sqlite
import datetime, time

class sqlite_db(sql_db.sql_db):
    """Interface to an SQLite database"""

    def __init__(self, database, mode=0755):
        """Create the interface

        database and mode are passed as arguments to _sqlite.connect
        """
        self.connection = _sqlite.connect(database, mode)
        self.cache = {}
        self.tid_map = {}

    ###########################################################################
    # Utility methods

    def result_generator(self, cmnd):
        result = self.connection.execute(cmnd)
        for row in result.row_list:
            for item in row:
                yield item

    def fetchall(self, cmnd):
        return list(self.result_generator(cmnd))

    def execute(self, cmnd):
        result = self.connection.execute(cmnd)
        return self.connection.sqlite_last_insert_rowid()

    ###########################################################################
    # Class registration methods
    
    def table_exists(self, cls):
        table_name = cls.__name__.lower()
        cmnd = "SELECT oid FROM sqlite_master WHERE type='table' AND name='%s'" % table_name
        data = self.fetchall(cmnd)
        return bool(data)

    def get_tid(self, table_name):
        cmnd = "SELECT oid FROM sqlite_master WHERE type='table' AND name='%s'" % table_name
        data = self.fetchall(cmnd)
        if len(data) != 1:
            raise db_base.DBError, "Expected to find 1 TID"
        else:
            return data[0]

    #######################################################################
    # Methods called by property objects

    def create_ref_columns(self, prop):
        return '%s INTEGER, %s INTEGER' % (prop.sql_tid_name, prop.sql_oid_name)

    def create_int_column(self, prop):
        return "%s INTEGER" % prop.sql_name
        
    def create_float_column(self, prop):
        return "%s FLOAT" % prop.sql_name
                                         
    def create_str_column(self, prop):
        return "%s TEXT" % prop.sql_name

    def create_date_column(self, prop):
        return "%s DATE" % prop.sql_name

    def create_time_column(self, prop):
        return "%s TIME" % prop.sql_name

    def create_datetime_column(self, prop):
        return "%s TIMESTAMP" % prop.sql_name

    def quote_str(self, s):
        return "'%s'" % s.replace("'", "''")

    def quote_date(self, s):
        return s.strftime("'%Y-%m-%d'")

    def quote_time(self, s):
        return s.strftime("'%H:%M:%S'")

    def quote_datetime(self, s):
        return s.strftime("'%Y-%m-%d %H:%M:%S'")

    def quote_tid_oid(self, val):
        if val is None:
            return "NULL", "NULL"
        elif isinstance(val, tuple):
            return str(val[0]), str(val[1])
        else:
            return str(val.tid), str(val.oid)

    def unquote_date(self, s):
        t = time.strptime(s, "%Y-%m-%d")
        return datetime.date(t.tm_year, t.tm_mon, t.tm_mday)
    
    def unquote_time(self, s):
        t = time.strptime(s, "%H:%M:%S")
        return datetime.time(t.tm_hour, t.tm_min, t.tm_sec)

    def unquote_datetime(self, s):
        t = time.strptime(s, "%Y-%m-%d %H:%M:%S")        
        return datetime.datetime(t.tm_year, t.tm_mon, t.tm_mday, t.tm_hour, t.tm_min, t.tm_sec)

    ###########################################################################
