from __future__ import print_function

import math
import re
import textwrap

import numpy as np

import numba.unittest_support as unittest
from numba.compiler import compile_isolated
from numba import jit, types
from numba.errors import TypingError
from .support import TestCase


def what():
    pass

def foo():
    return what()

def bar(x):
    return x.a

def issue_868(a):
    return a.shape * 2

def impossible_return_type(x):
    if x > 0:
        return ()
    else:
        return 1j

def bad_hypot_usage():
    return math.hypot(1)

def imprecise_list():
    l = []
    return len(l)

def unknown_module():
    return numpyz.int32(0)

def nop(x, y, z):
    pass


class Foo(object):
    def __repr__(self):
        return "<Foo instance>"


class TestTypingError(unittest.TestCase):

    def test_unknown_function(self):
        try:
            compile_isolated(foo, ())
        except TypingError as e:
            self.assertTrue(e.msg.startswith("Untyped global name"), e.msg)
        else:
            self.fail("Should raise error")

    def test_unknown_attrs(self):
        try:
            compile_isolated(bar, (types.int32,))
        except TypingError as e:
            self.assertTrue(e.msg.startswith("Unknown attribute"), e.msg)
        else:
            self.fail("Should raise error")

    def test_unknown_module(self):
        # This used to print "'object' object has no attribute 'int32'"
        with self.assertRaises(TypingError) as raises:
            compile_isolated(unknown_module, ())
        self.assertIn("Untyped global name 'numpyz'", str(raises.exception))

    def test_issue_868(self):
        '''
        Summary: multiplying a scalar by a non-scalar would cause a crash in
        type inference because TimeDeltaMixOp always assumed at least one of
        its operands was an NPTimeDelta in its generic() method.
        '''
        with self.assertRaises(TypingError) as raises:
            compile_isolated(issue_868, (types.Array(types.int32, 1, 'C'),))

        expected = (
            "Invalid usage of * with parameters (({0} x 1), {0})"
            .format(str(types.intp)))
        self.assertIn(expected, str(raises.exception))
        self.assertIn("[1] During: typing of", str(raises.exception))

    def test_return_type_unification(self):
        with self.assertRaises(TypingError) as raises:
            compile_isolated(impossible_return_type, (types.int32,))
        self.assertIn("Can't unify return type from the following types: (), complex128",
                      str(raises.exception))

    def test_bad_hypot_usage(self):
        with self.assertRaises(TypingError) as raises:
            compile_isolated(bad_hypot_usage, ())

        errmsg = str(raises.exception)
        # Make sure it listed the known signatures.
        # This is sensitive to the formatting of the error message.
        self.assertIn(" * (float64, float64) -> float64", errmsg)

        # Check contextual msg
        last_two = errmsg.splitlines()[-2:]
        self.assertTrue(re.search(r'\[1\] During: resolving callee type: Function.*hypot', last_two[0]))
        self.assertTrue(re.search(r'\[2\] During: typing of call .*test_typingerror.py', last_two[1]))


    def test_imprecise_list(self):
        """
        Type inference should catch that a list type's remain imprecise,
        instead of letting lowering fail.
        """
        with self.assertRaises(TypingError) as raises:
            compile_isolated(imprecise_list, ())

        errmsg = str(raises.exception)
        self.assertIn("Can't infer type of variable 'l': list(undefined)",
                      errmsg)


class TestArgumentTypingError(unittest.TestCase):
    """
    Test diagnostics of typing errors caused by argument inference failure.
    """

    def test_unsupported_array_dtype(self):
        # See issue #1943
        cfunc = jit(nopython=True)(nop)
        a = np.ones(3)
        a = a.astype(a.dtype.newbyteorder())
        with self.assertRaises(TypingError) as raises:
            cfunc(1, a, a)
        expected = textwrap.dedent("""\
            This error may have been caused by the following argument(s):
            - argument 1: Unsupported array dtype: {0}
            - argument 2: Unsupported array dtype: {0}"""
            ).format(a.dtype)
        self.assertIn(expected, str(raises.exception))

    def test_unsupported_type(self):
        cfunc = jit(nopython=True)(nop)
        foo = Foo()
        with self.assertRaises(TypingError) as raises:
            cfunc(1, foo, 1)
        expected = textwrap.dedent("""\
            This error may have been caused by the following argument(s):
            - argument 1: cannot determine Numba type of value <Foo instance>"""
            )
        self.assertIn(expected, str(raises.exception))


if __name__ == '__main__':
    unittest.main()
