__version__   = "$Revision: 1.3 $"[11:-2]
__copyright__ = """Copyright (c) 2003 Not Another Corporation Incorporated
                   www.notanothercorporation.com"""
__license__ = """Licensed under the FSF GPL

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""
__doc__ = """* $Id: test_mongoose.py,v 1.3 2003/09/24 17:03:31 philiplindsay Exp $ *

    ** %s **  Ver %s
 
%s

%s

Test units for the C{mongoose} module.

""" % (__name__, __version__, __copyright__, __license__)

import sys
import unittest

import extunittest

import mongoose

try:
    True
except NameError:
    True = (1 == 1)
    False = not True

class ExceptionHookInstallUnitTests(unittest.TestCase):
    """
    Test units for the exception hook installation and removal.
    """

    def testStartAndStop(self):
        """
        Check our handler is installed and removed correctly.
        """
        initialExceptionHook = sys.excepthook
        
        mongoose.start()

        self.assertEqual(sys.excepthook, mongoose.mongoose._exceptionHook)

        mongoose.stop()

        # Note: This may fail if there was an exception handler that wasn't
        #       the default system one installed before ours.
        self.assertEqual(sys.excepthook, initialExceptionHook)



class RegisterReporterUnitTests(unittest.TestCase):
    """
    Test units for registering reporters.

    TODO: Re-import C{mongoose} before each test?
    """

    def testRegisteringIncreasesReporterCount(self):
        """
        Check that registering a reporter increases the number of
        reporters by one.
        """
        myReporter = mongoose.Reporter()

        mongoose.start()

        initialLen = len(mongoose.mongoose._reporters)

        mongoose.registerReporter(myReporter)

        self.assertEqual(len(mongoose.mongoose._reporters), initialLen + 1)

        mongoose.stop()
        
        
    def testRegisteredReporterListIsAppended(self):
        """
        Check that registering a reporter places it last in the list.
        """
        myReporter = mongoose.Reporter()
        anotherReporter = mongoose.Reporter()

        mongoose.start()

        mongoose.registerReporter(anotherReporter)
        mongoose.registerReporter(myReporter)

        self.assertEqual(mongoose.mongoose._reporters[-1], myReporter)

        mongoose.stop()


    def testStopDeletesGoodReporters(self):
        """
        Check that stopping the mongoose causes good reporters (i.e. ones that
        I{don't} propagate exceptions) to be deleted.
        """
        myReporter = mongoose.Reporter()

        mongoose.start()

        mongoose.registerReporter(myReporter)

        mongoose.stop()

        self.assertEqual(len(mongoose.mongoose._reporters), 0)


    class BadReporter(mongoose.Reporter):
        """
        A reporter that causes an exception when it's unloaded.
        """
        def unload(self):
            """
            An unload routine that causes an exception when it's called.
            """
            raise Exception()
        

    def testStopDeletesNonDebugBadReporters(self):
        """
        Check that stopping the mongoose causes bad reporters (i.e. ones that
        I{do} propagate exceptions) to be deleted when not in debug mode.

        TODO: Check during debug mode too? (Need to ensure stuff is reset...)
        """        
        myReporter = self.BadReporter()

        mongoose.setDebug(False)
        
        mongoose.start()

        mongoose.registerReporter(myReporter)

        mongoose.stop()

        self.assertEqual(len(mongoose.mongoose._reporters), 0)

        
class SetSessionInfoUnitTests(unittest.TestCase):
    """
    Tests related to setting session info values.
    """

    def testSetStringValue(self):
        """
        Check setting session info with a legal name and string value succeeds.
        """
        NAME = "foo"
        VALUE = "bah"
        
        mongoose.setSessionInfo(NAME, VALUE)
        
        self.assertEqual(mongoose.mongoose._sessionInfo[NAME], VALUE)

        
    def testSetNonStringValue(self):
        """
        Check setting session info with a legal name and non-string value
        succeeds (by stringizing the non-string value).
        """
        NAME = "bah"
        VALUE = TypeError
        
        mongoose.setSessionInfo(NAME, VALUE)
        
        self.assertEqual(mongoose.mongoose._sessionInfo[NAME], str(VALUE))


    def testIllegalNameNoDebug(self):
        """
        Check setting session info with an illegal name doesn't throw an
        exception when not debugging.
        """
        NAME = dict()
        VALUE = "bah"

        mongoose.setDebug(False)
        
        mongoose.setSessionInfo(NAME, VALUE) # Shouldn't fail.
        

    def testIllegalNameDebug(self):
        """
        Check setting session info with an illegal name I{does} throw an
        exception when debugging.
        """
        NAME = dict()
        VALUE = "bah"

        mongoose.setDebug(True)
        
        self.assertRaises(TypeError, \
                          lambda x=0 : \
                          mongoose.setSessionInfo(NAME, VALUE))


# --- Related to external tests... ---

class MongooseExternalTestSuite(extunittest.ExternalTestSuite):
    """
    """
    writeFailFiles = True
    normaliseExceptions = True
    ignoreLineNumChanges = True

# --- The external tests... ---


class BadReporter(mongoose.Reporter):
    """
    A reporter that causes an exception when it's called to process or
    unloaded.
    """
    def process(self, uhx, sessionInfo):
        """
        """
        raise Exception("Exception in 'process()'")

        
    def unload(self):
        """
        An unload routine that causes an exception when it's called.
        """
        raise Exception("Exception in 'unload()'")



class GoodReporter(mongoose.Reporter):
    """
    A really basic reporter.
    """
    def process(self, uhx, sessionInfo):
        """
        """
        print ">>%s<<" % type(uhx)
        print ">>%s<<" % type(sessionInfo)

    def unload(self):
        """
        Indicate that our unload routine has been called.
        """
        print "GoodReporter 'unload()' routine called."


def extTestNoReporterDebugOff():
    """
    Test exceptions are caught silently when no reporters are loaded.
    Debug is False.
    """
    mongoose.setDebug(False)

    mongoose.start()

    raise Exception("TestException")

    mongoose.stop()


def extTestNoReporterDebugOn():
    """
    Test operation when no reporters are loaded. Debug is True.
    """
    mongoose.setDebug(True)

    mongoose.start()

    raise Exception("Test exception")

    mongoose.stop()


def extTestGoodReporterDebugOff():
    """
    Test good reporter processing.
    
    Debug is False.
    """

    mongoose.setDebug(False)
    mongoose.registerReporter(GoodReporter())
    
    mongoose.start()

    raise Exception("Test exception")

    mongoose.stop()


def extTestGoodReporterDebugOn():
    """
    Test good reporter processing.
    
    Debug is True.
    """

    mongoose.setDebug(True)
    mongoose.registerReporter(GoodReporter())
    
    mongoose.start()

    raise Exception("Test exception")

    mongoose.stop()


def extTestBadReporterUnloadDebugOn():
    """
    Test bad reporter unloading.
    
    Debug is True.
    """
    mongoose.setDebug(True)
    
    mongoose.registerReporter(BadReporter())
    
    mongoose.start()

    mongoose.stop()


def extTestBadReporterUnloadDebugOff():
    """
    Test bad reporter unloading.
    
    Debug is False.
    """
    mongoose.setDebug(False)
    
    mongoose.registerReporter(BadReporter())
    
    mongoose.start()

    mongoose.stop()


def extTestBadReporterProcessingDebugOn():
    """
    Test bad reporter processing.
    
    Debug is True.
    """
    mongoose.setDebug(True)
    
    mongoose.registerReporter(BadReporter())
    
    mongoose.start()

    raise Exception("Test exception")

    mongoose.stop()


def extTestBadReporterProcessingDebugOff():
    """
    Test bad reporter processing.
    
    Debug is False.
    """
    mongoose.setDebug(False)
    
    mongoose.registerReporter(BadReporter())
    
    mongoose.start()

    raise Exception("Test exception")

    mongoose.stop()


def extTestBadReporterDoesNotStopGoodReporterProcessing():
    """
    Test that when an exception occurs and a bad reporter is invoked
    it doesn't stop the processing of other reporters.
    """
    
    mongoose.setDebug(False)
    
    mongoose.registerReporter(BadReporter())
    mongoose.registerReporter(GoodReporter())
    
    mongoose.start()

    raise Exception("Test exception")

    mongoose.stop()
    

def extTestReporterGetsSessionInfoValues():
    """
    Test that information is retrieved successfully from the session info
    object.
    """

    TEST_INFO = {'test_info1': '',
                 'test_info2': 'A string',
                 'test_info3': {'name': 'value'},
                 'test_info4': Exception}

    class TestReporter(mongoose.Reporter):
        """
        A reporter that checks it receives session info values it expects.
        """

        def process(self, uhx, info):
            """
            """

            for key in TEST_INFO:
                result = ("does not match",
                          "matches")[str(TEST_INFO[key]) == getattr(info, key)]
                print "%s... %s" % (key, result)

    mongoose.setDebug(False)
    
    mongoose.registerReporter(TestReporter())
    
    mongoose.start()

    for currName in TEST_INFO:
        mongoose.setSessionInfo(currName, TEST_INFO[currName])

    raise Exception("Test exception")

    mongoose.stop()
            
                
if __name__ == '__main__':
    sys.exit(extunittest.run(sys.argv))
