__version__   = "$Revision: 1.3 $"[11:-2]
__copyright__ = """Copyright (c) 2003 Not Another Corporation Incorporated
                   www.notanothercorporation.com"""
__license__   = """Licensed under the GNU LGPL

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
"""
__doc__ = """
** %(name)s **
File version %(version)s::
 
    %(copyright)s

    %(license)s

Part of the C{mongoose} package, provides the C{UnhandledException} object
which allows read-only retrieval of information about an unhandled exception.

I{* $Id: unhandledexception.py,v 1.3 2003/09/24 17:03:31 philiplindsay Exp $ *}
""" % {'name':__name__, 'version':__version__, 'copyright':__copyright__,
       'license':__license__}


import traceback

from safereadonlymixin import SafeReadOnlyMixin


try:
    True
except NameError:
    True = (1 == 1)
    False = not True
                 

INCIDENT_ID_MOD_VALUE = 100000
def calcIncidentId(processedTraceback):
    """
    Returns a somewhat unique I{Incident ID} string based on the
    function call path that resulted in the exception.

    @param processedTraceback: the result of calling C{traceback.extract_tb}
    """
    functionNames = [traceEntry[2]
                       for traceEntry in processedTraceback]

    #TODO: Add in version #? And line #?
    return str(hash("".join(functionNames)) % INCIDENT_ID_MOD_VALUE)


class UnhandledException(SafeReadOnlyMixin):
    """
    Stores information about an unhandled exception.
    """

    def __init__(self, exctype, value, tb, dateTime, DEBUG = False):
        """
        Initialise ourself with supplied values, if possible.

        @param exctype: Exception type. (From C{sys.exc_info})
        @param value: Exception value. (From C{sys.exc_info})
        @param tb: Instance of C{Traceback}. (From C{sys.exc_info})
        @param dateTime: String of the date and time exception occurred.
        @param DEBUG: If C{False} we don't propagate exceptions.
        
        """
        infoDict = {}

        infoDict['type'] = exctype
        infoDict['value'] = value
        infoDict['traceback'] = tb

        infoDict['when'] = dateTime

        try:
            infoDict['processedTB'] = \
                                    traceback.extract_tb(infoDict['traceback'])
        except:
            if DEBUG:
                raise

        try:
            infoDict['incidentId'] = calcIncidentId(infoDict['processedTB'])
        except:
            if DEBUG:
                raise            

        try:
            infoDict['formattedException'] = "".join(
                traceback.format_exception(exctype, value,tb))
        except:
            if DEBUG:
                raise            
        
        super(UnhandledException, self).__init__(infoDict)
