import { PromiseDelegate } from '@lumino/coreutils';
import { IRateLimiter } from './index';
import { Poll } from './poll';
/**
 * A base class to implement rate limiters with different invocation strategies.
 *
 * @typeparam T - The resolved type of the underlying function.
 *
 * @typeparam U - The rejected type of the underlying function.
 *
 * @typeparam V - Arguments for the underlying function.
 */
export declare abstract class RateLimiter<T, U, V extends any[]> implements IRateLimiter<T, U, V> {
    /**
     * Instantiate a rate limiter.
     *
     * @param fn - The function to rate limit.
     *
     * @param limit - The rate limit; defaults to 500ms.
     */
    constructor(fn: (...args: V) => T | Promise<T>, limit?: number);
    /**
     * Whether the rate limiter is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Disposes the rate limiter.
     */
    dispose(): void;
    /**
     * The rate limit in milliseconds.
     */
    readonly limit: number;
    /**
     * Invoke the rate limited function.
     */
    abstract invoke(...args: V): Promise<T>;
    /**
     * Stop the function if it is mid-flight.
     */
    stop(): Promise<void>;
    /**
     * Arguments for the underlying function.
     */
    protected args: V | undefined;
    /**
     * A promise that resolves on each successful invocation.
     */
    protected payload: PromiseDelegate<T> | null;
    /**
     * The underlying poll instance used by the rate limiter.
     */
    protected poll: Poll<T, U, 'invoked'>;
}
/**
 * Wraps and debounces a function that can be called multiple times and only
 * executes the underlying function one `interval` after the last invocation.
 *
 * @typeparam T - The resolved type of the underlying function. Defaults to any.
 *
 * @typeparam U - The rejected type of the underlying function. Defaults to any.
 *
 * @typeparam V - Arguments for the underlying function. Defaults to any[].
 */
export declare class Debouncer<T = any, U = any, V extends any[] = any[]> extends RateLimiter<T, U, V> {
    /**
     * Invokes the function and only executes after rate limit has elapsed.
     * Each invocation resets the timer.
     */
    invoke(...args: V): Promise<T>;
}
/**
 * Wraps and throttles a function that can be called multiple times and only
 * executes the underlying function once per `interval`.
 *
 * @typeparam T - The resolved type of the underlying function. Defaults to any.
 *
 * @typeparam U - The rejected type of the underlying function. Defaults to any.
 *
 * @typeparam V - Arguments for the underlying function. Defaults to any[].
 */
export declare class Throttler<T = any, U = any, V extends any[] = any[]> extends RateLimiter<T, U, V> {
    /**
     * Instantiate a throttler.
     *
     * @param fn - The function being throttled.
     *
     * @param options - Throttling configuration or throttling limit in ms.
     *
     * #### Notes
     * The `edge` defaults to `leading`; the `limit` defaults to `500`.
     */
    constructor(fn: (...args: V) => T | Promise<T>, options?: Throttler.IOptions | number);
    /**
     * Throttles function invocations if one is currently in flight.
     */
    invoke(...args: V): Promise<T>;
    private _interval;
    private _trailing;
}
/**
 * A namespace for `Throttler` interfaces.
 */
export declare namespace Throttler {
    /**
     * Instantiation options for a `Throttler`.
     */
    interface IOptions {
        /**
         * The throttling limit; defaults to 500ms.
         */
        limit?: number;
        /**
         * Whether to invoke at the leading or trailing edge of throttle cycle.
         * Defaults to `leading`.
         */
        edge?: 'leading' | 'trailing';
    }
}
