# client.py - watchman client
#
# Copyright 2013 Facebook, Inc.
#
# This software may be used and distributed according to the terms of the
# GNU General Public License version 2 or any later version.

import getpass
from mercurial import util
import pywatchman

class Unavailable(Exception):
    def __init__(self, msg, warn=True, invalidate=False):
        self.msg = msg
        self.warn = warn
        self.invalidate = invalidate
    def __str__(self):
        if self.warn:
            return 'warning: watchman unavailable: %s' % self.msg
        else:
            return 'watchman unavailable: %s' % self.msg

class WatchmanNoRoot(Unavailable):
    def __init__(self, root, msg):
        self.root = root
        super(WatchmanNoRoot, self).__init__(msg)

class client(object):
    def __init__(self, repo, timeout=1.0):
        err = None
        if not self._user:
            err = "couldn't get user"
            warn = True
        if self._user in repo.ui.configlist('watchman', 'blacklistusers'):
            err = 'user %s in blacklist' % self._user
            warn = False

        if err:
            raise Unavailable(err, warn)

        self._watchmanclient = pywatchman.client(timeout=timeout)
        self._root = repo.root
        self._ui = repo.ui
        self._available = True

    def getcurrentclock(self):
        result = self.command('clock')
        if 'clock' not in result:
            raise Unavailable('returned corrupt response for "clock" query -- '
                              'first 128 bytes of result are: %s' %
                              repr(result)[:128], invalidate=True)
        return result['clock']

    def available(self):
        return self._available

    @util.propertycache
    def _user(self):
        try:
            return getpass.getuser()
        except:
            # couldn't figure out our user
            return None

    def _command(self, *args):
        watchmanargs = (args[0], self._root) + args[1:]
        try:
            return self._watchmanclient.query(*watchmanargs)
        except pywatchman.CommandError, ex:
            if ex.msg.startswith('unable to resolve root'):
                raise WatchmanNoRoot(self._root, ex.msg)
            raise Unavailable(ex.msg)
        except pywatchman.WatchmanError, ex:
            raise Unavailable(str(ex))

    def command(self, *args):
        try:
            try:
                return self._command(*args)
            except WatchmanNoRoot:
                # this 'watch' command can also raise a WatchmanNoRoot if watchman
                # refuses to accept this root
                self._command('watch')
                return self._command(*args)
        except Unavailable:
            # this is in an outer scope to catch Unavailable form any of the
            # above _command calls
            self._available = False
            raise
