from gssapi.raw.cython_types cimport *
from gssapi.raw.oids cimport OID
GSSAPI="BASE"  # This ensures that a full module is generated by Cython

from gssapi.raw.cython_converters cimport c_make_oid

from gssapi.raw.named_tuples import InquireSASLNameResult
from gssapi.raw.misc import GSSError

cdef extern from "python_gssapi_ext.h":
    OM_uint32 gss_inquire_saslname_for_mech(
        OM_uint32 *min_stat,
        const gss_OID desired_mech,
        gss_buffer_t sasl_mech_name,
        gss_buffer_t mech_name,
        gss_buffer_t mech_description) nogil

    OM_uint32 gss_inquire_mech_for_saslname(
        OM_uint32 *min_stat,
        const gss_buffer_t sasl_mech_name,
        gss_OID *mech_type) nogil


def inquire_saslname_for_mech(OID mech not None):
    """
    inquire_saslname_for_mech(mech)
    Gets information about a specified mech, including the SASL name,
    the mech name, and the mech description.

    Args:
        mech (~gssapi.OID): Mechanism to inquire about

    Returns:
        InquireSASLNameResult: the results of inquiry; a mech's SASL name,
        name, and description.

    Raises:
        ~gssapi.exceptions.GSSError: an unknown failure occurred
    """
    cdef OM_uint32 maj_stat, min_stat
    cdef gss_buffer_desc sasl_mech_name
    cdef gss_buffer_desc mech_name
    cdef gss_buffer_desc mech_desc
    cdef gss_OID m = GSS_C_NO_OID

    m = &mech.raw_oid

    with nogil:
        maj_stat = gss_inquire_saslname_for_mech(&min_stat, m, &sasl_mech_name,
                                                 &mech_name, &mech_desc)

    if maj_stat == GSS_S_COMPLETE:
        out_smn = (<char*>sasl_mech_name.value)[:sasl_mech_name.length]
        out_mn = (<char*>mech_name.value)[:mech_name.length]
        out_md = (<char*>mech_desc.value)[:mech_desc.length]

        gss_release_buffer(&min_stat, &sasl_mech_name)
        gss_release_buffer(&min_stat, &mech_name)
        gss_release_buffer(&min_stat, &mech_desc)

        return InquireSASLNameResult(out_smn, out_mn, out_md)
    else:
        raise GSSError(maj_stat, min_stat)


def inquire_mech_for_saslname(bytes sasl_name not None):
    """
    inquire_mech_for_saslname(sasl_name)
    Gets the OID for the mech specified by SASL name.

    Args:
        sasl_name (bytes): SASL name of the mechanism

    Returns:
        ~gssapi.OID: the mechanism with corresponding SASL name.

    Raises:
        ~gssapi.exceptions.GSSError: An unknown failure occurred
    """
    cdef OM_uint32 maj_stat, min_stat
    cdef gss_buffer_desc sn
    cdef gss_OID m

    sn.length = len(sasl_name)
    sn.value = sasl_name

    with nogil:
        maj_stat = gss_inquire_mech_for_saslname(&min_stat, &sn, &m)

    if maj_stat == GSS_S_COMPLETE:
        return c_make_oid(m)
    else:
        raise GSSError(maj_stat, min_stat)
