
# getargs.py
# Provides CLParser, a command line parser class that is a more powerful and
# more flexible alternative to getopt
#
# Author: Vinod Vijayarajan <vinod.vijayarajan@hp.com>
# Dated 11 Jan, 2004
#
# Copyright (c) 2004-2005, Vinod Vijayarajan <vinod.vijayarajan@hp.com>
# All rights reserved
#
# Last Modified: 06 Feb 2004
# Revision: 2.14
# For version history see the CHANGELOG


'''Module: getargs
   Revision: 2.14

   Author: Vinod Vijayarajan <vinod.vijayarajan@hp.com>
   Copyright (c) 2004-2005, Vinod Vijayarajan
   
   This module implements a Command Line Parser class (CLParser) that 
   applications can use to parse command line arguments in sys.argv. 
   getargs is designed as a considerably more powerful and flexible alternative
   to the standard getopt module.

   The CLParser provides the following capabilities:
    - Each command line option can take an arbitrary number of arguments.
    - Distingushes between switches (options without arguments) and options
      with arguments.
    - Further classifies options as 'single-argument-options' or 
      'default-if-set' options.
    - Switches and options with arguments can be interleaved in the command
      line.
    - You can specify the maximum and minimum number of arguments an option
      can take. Use -1 if you don't want to specify an upper bound.
    - Specify default arguments to an option
    - Optionally typechecks arguments to an option
    - Returns arguments to an option in the specified type
    - Short options can be upto 2 letters in length
    - A given option may be invoked by multiple names
    - Recognizes by default the '--version'/'-h'/'--help' options
    - Arguments need not all be on the commandline. They can be specified
      in a file too.
    - Arguments may be specified using the '=' and ',' syntax also.
    - Methods isset() and getargs() to check if a switch (an option without
      arguments) is set (present) on the commandline and to obtain arguments
      to an option, respectively

   The CLParser expects as input:
    - The entire command line, including the script name (sys.argv)
    - A dictionary whose items describe each possible option in the following
      format:
         'long_option' : ('short_option', max_args, min_args, [defaults])

      See the CLParser class docstring for information on what each
      of the above represent as well as their significance in determining
      how each option on the command line is treated.

      You can also ask the module to read in arguments from a file.
      The file name must be prepended with a '@', thusly -
           
          test.py @test.opt          
          test.py @test.opt -v -bs128

   Usage:
     The CLParser class provides two methods to access options from the 
     parsed command line. These are:
      - isset(option)   : Used to check whether a switch 'option' is set 
                          (present) on the commandline.
                          Returns 1 if set, 0 if not.
      - getargs(option) : Used to obtain arguments to an option.
                          If 'option' is used in the commandline, returns
                          either a value (for single argument options) or the
                          list of arguments to 'option'. Even if 'option' is 
                          not used but has defaults specified, these are 
                          returned.
                          An empty list [] is returned otherwise.

   Format of the opt_file:
     The user can ask the module to read in options and arguments from a
     file using the 'prog_name @opt_file' directive. The opt_file must obey 
     the following syntax:

      - All lines beginning with '#' are considered comment lines
      - You can have comments inline, following whatever is valid on the line.
      - Arguments and options can be written just as you would on the
        command line *with* the '-' and '--'s. You can intersperse them, as
        you please, across lines.
      - Arguments that contain spaces in them must be written as they would
        be on the prompt with \'s, thusly:
            --file this\ file.py that\ file.py
      - If you want arguments you specify in the opt_file to be globbed (i.e
        they contain wildcards), remember to enclose them in quotes(''/"")
        thusly:
            --file '*.py' "file.*"   # Can be 'single' or "double" quotes
     
     Important: The options/arguments specified in the opt_file are considered
     in place on the command line. As such, if you wish arguments (for an 
     option) specified on the command line to override arguments specified in 
     the opt_file, remember to position the @opt_file directive before the said
     arguments on the command line.

   Variants of the above theme:
     You may also choose not to override, but to append to arguments specified
     in the opt_file by using the '++option arg' directive on the command line,
     rather than the usual '--option arg'
     Eg:
        test.py @test.opt ++file another.py
    
     In this case, if '--file a.py b.py' was given in test.opt, 'file' would
     have as arguments, ['a.py', 'b.py', 'another.py']

     Note: The '++' notation can only be used for options that have been 
     encountered atleast once previously by the parser. An UsageError is
     thrown if this is the first instance of the option.

   Example:
     Say sys.argv is 'test.py @test.opt -vb128 --file *.pyc hell.pyo -n 2 3'
    
     opt_f = { 'verbose' : ('v', 0), 'block-size' : ('bs', 1, 1), 
               'file' : ('f', -1, 1), 'nodes/machines' : ('n/m', 4, 1), 
               'log' : ('l', 1, 1, 'a default.log'), 'debug' : ('d', 1, 0, 1)
             }
     
     try:
        clparser = CLParser(opt_f, sys.argv)
     except UsageError, err:
        print 'error:', err
        usage()
        sys.exit(1)
     
     files = clparser.getargs('f')            # returns a list
     bsize = clparser.getargs('block-size')   # returns a value
     log_file = clparser.getargs('l')         # returns 'a default.log'
     debug_lvl = clparser.getargs('debug')    # debug is default-if-set option
     
     if clparser.isset('verbose'):
        print 'I'm going to talk an awful lot'

   Exceptions:
     Throws HelpQuery when a '--help' or '-h' is seen on the command line
     Throws VersionQuery when a '--version' is encountered on the command line
     Throws UsageError, KeyError or ValueError exceptions.
   '''


__author__    = 'Vinod Vijayarajan'
__revision__  = '2.14'


class UsageError(Exception):
    '''Defines an exception for use by the CLParser.
       Thrown whenever a CLParser instance encounters a parse error
       '''
    def __init__(self, err):
        Exception.__init__(self)
        self.args = err
        

class VersionQuery(Exception):
    '''Defines an exception for use by the CLParser.
       Thrown whenever a '--version' is encountered in sys.argv
       '''
    def __init__(self):
        Exception.__init__(self)


class HelpQuery(Exception):
    '''Defines an exception for use by the CLParser.
       Thrown whenever a '--help' or a '-h' is encountered in sys.argv
       '''
    def __init__(self):
        Exception.__init__(self)
        

class CLParser:
    '''Defines a Command Line Parser class
       '''
    def __init__(self, opt_format, argv, allow_alt = 0):
        '''The constructor. Parses the command line and builds a dictionary
           of command line options and their arguments (if any).
           Expects opt_format to be a dictionary whose items describes each 
           possible option in the following format:
                
           'long_opt' : ('short_opt', ['type'], max_args, min_args, [defaults])
           
           max_args/min_args are what they seem to be - the most/least
           number of arguments a option can/must have.

           The 'defaults' field is optional and is used to specify default
           arguments to an option. These will be assigned to the option if 
           it is *not* seen on (used in) the command line.
           Default values can be -
             - a value for options that require a single argument
             - a list for options with more than one possible argument.
           Default values are mandatory for 'default-if-set' options.

           If max_args is 0 (option is just a switch), min_args is ignored.
           If max_args is -1, then option can have any number of arguments 
           greater/equal to min_args

           If max_args == min_args == 1, the option is treated as a single
           argument option.

           If max_args >= 1 and min_args == 0, the option is treated as a
           'default-if-set' option. This implies that it is only if the
           option is encountered on the command line without any arguments,
           that it is assigned the 'defaults' value. (Note: defaults must 
           be specified for such options) 
           If absent from the command line altogether, the defaults are 
           *not* applied. If an argument 'arg' is specified on the command
           line, then 'arg' is assigned to the option.
           Thus -
                a --debug in the command line would cause debug = 1
                a --debug 2 in the command line would result in debug = 2
                if unused on the command line, debug = []

           Expects argv to be the entire command line (including the prog name)

           The allow_alt argument defaults to 0 and indicates whether the 
           parser will accept '='s and ','s on the command line. For eg. if
           this value is set to 1, the following will be accepted by the parser

            test.py --file=do.py,da.py, di.py --block-size=64 -i=2 --debug 3

           Parses and stores internally the built dictionary.
           Access is by use of the isset and getargs methods.
           
           Throws:
             - a HelpQuery exception on encountering a '--help' or a '-h'
             - a VersionQuery exception on encountering a '--version'
             - an UsageError("diagnostic message") exception on a parse error.
           '''
        
        if allow_alt:
            _argv = []
            for i in argv[1:]:
                _argv += i.replace('=', ' ').replace(',', ' ').split()
        else:
            _argv = argv[1:]

        self.opt_format = opt_format
        self.opt_dict = {}
        
        self.short_long = {}
        self.opt_alias = {}
        
        self.__prepareSuppDicts()

        self.opt_append = {}
        self.prev_opt_args = []
        self.prev_opt = None

        self.__parseInput(_argv)
        self.__populateDefaults()
        self.__verifyArgs()


    def __prepareSuppDicts(self):
        '''Builds the opt_alias dict that stores the various names an
           option may have. Populates the short_long mapping dict.
           '''
        for long_opt in self.opt_format.keys():
            if long_opt.find('/') == -1:
                option = long_opt
                self.opt_alias[option] = long_opt
            else:
                aliases = long_opt.split('/')
                option = aliases[0]
                for i in aliases:
                    self.opt_alias[i] = option
                self.opt_format[option] = self.opt_format[long_opt]
                del self.opt_format[long_opt]

            for j in self.opt_format[option][0].split('/'):
                self.short_long[j] = option


    def __parseInput(self, argv):
        '''Parses the command line
           '''
        i = 0
        while argv[i:]:
            token = argv[i]
            if token[0] == '-':
                self.__updateDict()
                
                if token[1] == '-':
                    option = token[2:]
                    if option == 'help':
                        raise HelpQuery()
                    if option == 'version':
                        raise VersionQuery()
                    try:
                        self.prev_opt = self.opt_alias[option]
                    except KeyError:
                        raise UsageError("unknown option - '%s'" % option)
                else:
                    j = 1
                    while j < len(token):
                        if token[j] == 'h':
                            raise HelpQuery()
                        try:
                            self.prev_opt = self.short_long[token[j]]
                            self.opt_dict[self.prev_opt] = []
                        except KeyError:
                            try:
                                self.prev_opt = self.short_long[token[j] + \
                                                                    token[j+1]]
                                self.opt_dict[self.prev_opt] = []
                                j = j + 1
                            except (KeyError, IndexError):
                                if j == 1:
                                    raise UsageError(
                                        "unknown option - '%s'" % token[j:])
                            
                                self.prev_opt_args = [token[j:]]
                                break
                        j = j + 1
            elif token[0] == '@':
                argv = self.__parseFile(token[1:]) + argv[i+1:]
                i = -1
            elif token[0] == '+':
                self.__updateDict()
                
                if token[1] == '+':
                    option = token[2:]
                    try:
                        option = self.opt_alias[option]
                    except KeyError:
                        raise UsageError("unknown option - '%s'" % option)
                        
                    if not self.opt_dict.has_key(option):
                        raise UsageError(
                        "'%s' not seen previously. use -- not ++" % option)
                    self.opt_append[option] = 1 
                    self.prev_opt = option
                else:
                    raise UsageError(
                        "the ++ notation can only be used with long options")
            else:
                self.prev_opt_args.append(token)
            i += 1

        self.__updateDict()
        del self.opt_dict[None]


    def __updateDict(self):
        '''Updates the option-arguments dictionary
           '''
        if self.opt_append.has_key(self.prev_opt):
            self.opt_dict[self.prev_opt] += self.prev_opt_args
            del self.opt_append[self.prev_opt]
        else:
            self.opt_dict[self.prev_opt] = self.prev_opt_args
        self.prev_opt_args = []


    def __verifyArgs(self):
        '''Routine to verify that the number of arguments given to an 
           option conforms to its specification in opt_format.
           '''
        for key, value in self.opt_dict.items():
            optf = self.opt_format[key]
            if optf[1] == 0:
                if value != []:
                    raise UsageError("option '%s' accepts no arguments" % key)
            else:
                if value == []:
                    raise UsageError("option '%s' needs arguments" % key)
                if isinstance(optf[1], type):
                    if not optf[3] <= len(value) <= optf[2] % 32768:
                        raise UsageError("invalid number of args to '%s'"% key)
                    self.opt_dict[key] = []
                    for i in value:
                        try:
                            self.opt_dict[key].append(optf[1](i))
                        except ValueError:
                            raise UsageError(
                    "invalid type of argument '%s' to option '%s'" % (i, key))
                    self.opt_format[key] = [optf[0]] + list(optf[2:])
                else:             
                    if not optf[2] <= len(value) <= optf[1] % 32768:
                        raise UsageError("invalid number of args to '%s'"% key)
                    
            
    def __populateDefaults(self):
        '''Assign default arguments, if specified, to options that were not
           seen on the command line
           '''
        for key, value in self.opt_format.items():
            if isinstance(value[1], type):
                value = [value[0]] + list(value[2:])
            try:
                try:
                    opt_arg = self.opt_dict[key]
                    if opt_arg == [] and value[1] and value[2] == 0:
                        try:
                            self.opt_dict[key] += value[3]
                        except IndexError:
                            raise UsageError(
                                        "'%s' needs default arguments" % key)
                except KeyError:
                    if len(value) > 3:
                        self.opt_dict[key] = [] + value[3]
            except TypeError:
                self.opt_dict[key] = [value[3]]
            
                        
    def __parseFile(self, opt_file):
        '''Routine to obtain arguments from a file. Throws an uncaught
           IOError exception on error (file not present etc.)
           '''
        import glob
        
        argv = []
        for line in open(opt_file, 'r'):
            line = line.strip().replace('\ ', '\\')
            if line and line[0] != '#':
                idx = line.find('#')
                if idx == -1:
                    for i in line.split():
                        if i[0] in ['\'', '\"']:
                            argv += glob.glob(i[1:-1])
                        else:
                            argv.append(i.replace('\\', ' '))
                else:
                    for i in line[:idx].split():
                        if i[0] in ['\'', '\"']:
                            argv += glob.glob(i[1:-1])
                        else:
                            argv.append(i.replace('\\', ' '))
        return argv
            
                
    def isset(self, option):
        '''Check to see if 'option' is set in the command line. 
           Returns 1 if set or 0 if not.
           Throws ValueError if 'option' has arguments (is not just a switch)
           Throws KeyError if 'option' is not recognized
           '''
        try:
            try:
                option = self.short_long[option]
            except KeyError:
                value = self.opt_dict[self.opt_alias[option]]
            else:
                value = self.opt_dict[option]
        except KeyError:
            if self.opt_alias.has_key(option) or \
                                    self.short_long.has_key(option):
                return 0
            else:
                raise KeyError("unknown option - '%s'" % option)
                
        if len(value):
            raise ValueError("option '%s' has arguments. use getargs" % option)
            
        return 1


    def getargs(self, option):
        '''Gets arguments to 'option' in the command line.
           If 'option' is used in the commandline or has defaults, returns 
           either a value (if 'option' accepts only a single argument) or a 
           list of arguments. Returns [] if 'option' is not used and has no
           defaults.
           Throws ValueError if 'option' has no arguments (is just a switch)
           Throws KeyError if 'option' is not recognized
           '''
        try:
            try:
                option = self.short_long[option]
            except KeyError:
                value = self.opt_dict[self.opt_alias[option]]
            else:
                value = self.opt_dict[option]
        except KeyError:
            if self.opt_alias.has_key(option) or \
                                       self.short_long.has_key(option):
                return []
            else:
                raise KeyError("unknown option - '%s'" % option)

        if not len(value):
            raise ValueError("option '%s' is a switch. use isset" % option)

        if self.opt_format[self.opt_alias[option]][1] == 1: 
            return value[0]
            
        return value
