class Element:
  def __init__(self):
    self.identifier = 'Unknown'
    self.attributes = []
    self.level = 0
    self.discoveryNumber = 0
    self.finishNumber = 0

  def getIdentifier(self):
    return self.identifier

  def setIdentifier(self, identifier):
    self.identifier = identifier
  
  def clone(self):
    '''
    Return an empty object of the same type
    '''
    return Element()
  
  def accept(self, visitor):
    '''
    Support the Visitor pattern for traversal
    '''
    visitor.visitElement(self)
  
  def getIdentifier(self):
    '''
    Retrieve the identifier associated with this element
    '''
    return self.identifier
  
  def setIdentifier(self, identifier):
    '''
    Assign the identifier associated with this element
    '''
    self.identifier = identifier
  
  def getAttributes(self):
    '''
    Get attribute list for this element
    '''
    return self.attributes
  
  def setAttributes(self, attributes):
    '''
    Set attribute list for this element
    '''
    self.attributes = attributes
  
  def hasAttribute(self, name):
    '''
    Determine whether an attribute is present on this node
    '''
    for attribute in self.attributes:
      if name == attribute.getName():
        return 1
    return 0
  
  def addAttribute(self, attribute):
    '''
    Add attribute to this node
    '''
    if not (self.hasAttribute(attribute.getName())):
      self.attributes.append(attribute)
  
  def deleteAttribute(self, name):
    '''
    Remove an attribute on this node
    '''
    for attribute in self.attributes:
      if name == attribute.getName():
        self.attributes.remove(attribute)
        return 
  
  def getAttribute(self, name):
    '''
    Get an attribute on this node
    '''
    for attribute in self.attributes:
      if name == attribute.getName():
        return attribute
  
  def getDiscoveryNumber(self):
    '''
    Retrieve the discovery number
    This level is used to identify order in a search
    '''
    return self.discoveryNumber
  
  def setDiscoveryNumber(self, discoveryNumber):
    '''
    Set the discovery number
    This level is used to identify order in a search
    '''
    self.discoveryNumber = discoveryNumber
  
  def getFinishNumber(self):
    '''
    Retrieve the finish number
    This level is used to identify order in a search
    '''
    return self.finishNumber
  
  def setFinishNumber(self, finishNumber):
    '''
    Set the finish number
    This level is used to identify order in a search
    '''
    self.finishNumber = finishNumber
  
  def getLevel(self):
    '''
    Retrieve the level
    This level is used to identify levels in a breadth-first search
    '''
    return self.level
  
  def setLevel(self, level):
    '''
    Set the level
    This level is used to identify levels in a breadth-first search
    '''
    self.level = level
  
  def incrementLevel(self):
    '''
    Increment the level
    This level is used to identify levels in a breadth-first search
    '''
    self.level += 1
  
  def decrementLevel(self):
    '''
    Decrement the level
    This level is used to identify levels in a breadth-first search
    '''
    self.level -= 1

class Vertex(Element):
  def __init__(self):
    self.inEdges = []
    self.outEdges = []
    self.discoveryNumber = 0
    self.finishNumber = 0
    self.level = 0
  
  def getInEdges(self):
    '''
    Retrieve the edge for which this vertex is the target
    '''
    return self.inEdges
  
  def setInEdges(self, edges):
    '''
    Set the edge for which this vertex is the target
    '''
    self.inEdges = [e for e in edges]
  
  def addInEdge(self, edge):
    '''
    Add an input edge
    '''
    if not (edge in self.inEdges):
      if not (edge.getTarget() == self):
        edge.setTarget(self)
      self.inEdges.append(edge)
  
  def removeInEdge(self, edge):
    '''
    Remove an input edge
    '''
    if edge in self.inEdges:
      self.inEdges.remove(edge)
  
  def getOutEdges(self):
    '''
    Retrieve the edge for which this vertex is the source
    '''
    return self.outEdges
  
  def setOutEdges(self, edges):
    '''
    Set the edge for which this vertex is the source
    '''
    self.outEdges = [e for e in edges]
  
  def addOutEdge(self, edge):
    '''
    Add an output edge
    '''
    if not (edge in self.outEdges):
      if not (edge.getSource() == self):
        edge.setSource(self)
      self.outEdges.append(edge)
  
  def removeOutEdge(self, edge):
    '''
    Remove an output edge
    '''
    if edge in self.outEdges:
      self.outEdges.remove(edge)
  
  def accept(self, visitor):
    visitor.visitVertex(self)
  
  def clone(self):
    return Vertex()

class Edge(Element):
  def __init__(self):
    self.source = None
    self.target = None
  
  def getSource(self):
    '''
    Retrieve the source vertex
    '''
    return self.source
  
  def setSource(self, source):
    '''
    Set the source vertex
    '''
    self.source = source
  
  def getTarget(self):
    '''
    Retrieve the target vertex
    '''
    return self.target
  
  def setTarget(self, target):
    '''
    Set the target vertex
    '''
    self.target = target
  
  def accept(self, visitor):
    visitor.visitEdge(self)
  
  def clone(self):
    return Edge()

class Graph:
  def __init__(self):
    self.vertices = []
    self.edges = []
    self.edgeType = 'graph.Edge'
  
  def clone(self):
    '''
    Return an empty object of the same type
    '''
    return Graph()
  
  def getVertices(self):
    '''
    Retrieve all graph vertices
    '''
    return self.vertices
  
  def setVertices(self, vertices):
    '''
    Set all graph vertices
    '''
    self.vertices = vertices
  
  def addVertex(self, vertex):
    '''
    Add a vertex to the graph
    '''
    if not (vertex in self.vertices):
      self.vertices.append(vertex)
  
  def removeVertex(self, vertex):
    '''
    Remove a vertex to the graph, and all associated edges
    '''
    if vertex in self.vertices:
      for edge in vertex.getInEdges():
        self.removeEdge(edge)
      for edge in vertex.getOutEdges():
        self.removeEdge(edge)
      self.vertices.remove(vertex)
  
  def replaceVertex(self, vertex, newVertex):
    '''
    Replace a vertex to the graph, and retarget all associated edges
    '''
    if vertex in self.vertices:
      self.vertices.insert(self.vertices.index(vertex), newVertex)
      self.vertices.remove(vertex)
      [newVertex.addInEdge(edge) for edge in vertex.getInEdges()]
      [newVertex.addOutEdge(edge) for edge in vertex.getOutEdges()]
  
  def getEdges(self):
    '''
    Retrieve all graph edges
    '''
    return self.edges
  
  def setEdges(self, edges):
    '''
    Retrieve all graph edges
    '''
    self.edges = edges
  
  def addEdge(self, edge):
    '''
    Add an edge to the graph, adding the source and target vertices if necessary
    '''
    if not (edge in self.edges):
      self.edges.append(edge)
      self.addVertex(edge.getSource())
      self.addVertex(edge.getTarget())
      edge.getSource().addOutEdge(edge)
      edge.getTarget().addInEdge(edge)
  
  def removeEdge(self, edge):
    '''
    Remove an edge to the graph
    '''
    if edge in self.edges:
      edge.getSource().removeOutEdge(edge)
      edge.getTarget().removeInEdge(edge)
      self.edges.remove(edge)

  def importModule(moduleName):
    '''STOLEN FROM script.py
       Import the named module, and return the module object
       - Works properly for fully qualified names'''
    module     = __import__(moduleName)
    components = moduleName.split('.')
    for comp in components[1:]:
      module = getattr(module, comp)
    return module
  importModule = staticmethod(importModule)

  def addEdges(self, vertex, inputs, outputs):
    '''
    Add edges to the graph connecting "vertex" as the target for "inputs" and the source for "outputs", adding vertex if necessary
    '''
    path = self.getEdgeType().split('.')
    cls = getattr(self.importModule('.'.join(path[:-1])), path[-1])
    for input in inputs:
      edge = cls()
      edge.setSource(input)
      edge.setTarget(vertex)
      self.addEdge(edge)
    for output in outputs:
      edge = cls()
      edge.setSource(vertex)
      edge.setTarget(output)
      self.addEdge(edge)
  
  def getEdgeType(self):
    '''
    Retrieve the SIDL type for edges which are added automatically
    '''
    return self.edgeType
  
  def setEdgeType(self, edgeType):
    '''
    Set the SIDL type for edges which are added automatically
    '''
    self.edgeType = edgeType
  
  def getRoots(self):
    '''
    Return all the sources in the graph (vertices without entering edges)
    '''
    return filter(lambda v: not (len(v.getInEdges())), self.getVertices())
  
  def getLeaves(self):
    '''
    Return all the sinks in the graph (nodes without exiting edges)
    '''
    return filter(lambda v: not (len(v.getOutEdges())), self.getVertices())
  
  def addSubgraph(self, graph):
    '''
    Add all graph elements from the input graph into this one
    '''
    map(self.addVertex, graph.getVertices())
    map(self.addEdge, graph.getEdges())
  
  def removeSubgraph(self, graph):
    '''
    Remove all graph elements from the input graph in this one, as well as connecting edges
    '''
    map(self.removeVertex, graph.getVertices())
    map(self.removeEdge, graph.getEdges())
  
  def appendSubgraph(self, graph):
    '''
    Join every leaf of this graph to every root of the input graph, leaving the result in this graph
    '''
    roots = graph.getRoots()
    leaves = self.getLeaves()
    self.addSubgraph(graph)
    map(lambda v: self.addEdges(v, [], outputs = roots), leaves)
  
  def prependSubgraph(self, graph):
    '''
    Join every leaf of the input graph to every root of this graph, leaving the result in this graph
    '''
    roots = self.getRoots()
    leaves = graph.getLeaves()
    self.addSubgraph(graph)
    map(lambda v: self.addEdges(v, [], outputs = roots), leaves)
  
  def getSubgraph(self, vertex):
    '''
    Return the subgraph rooted at a particular vertex
    '''
    subgraph = self.clone()
    subgraph.addVertex(vertex)
    vertex.setInEdges([])
    for outEdge in vertex.getOutEdges():
      subgraph.addSubgraph(self.getSubgraph(outEdge.getTarget()))
      subgraph.addEdge(outEdge)
    self.subgraph = subgraph
    return self.subgraph

  def _getValue(self, valueName, obj, default):
    if hasattr(obj, '_'+valueName):
      return getattr(obj, '_'+valueName)
    return default

  def _computeHeight(self, points):
    modifiedPoints = set();

    for point in points:
      # Compute the max height of the points in the support of p, and add 1
      h0 = self._getValue("height", point, -1)
      h1 = max(map(lambda p: self._getValue("height", p, -1), support(point))) + 1;

      if not h1 == h0:
        point._height = h1
        if h1 > self._maxHeight: self._maxHeight = h1
        modifiedPoints.add(point)

      if len(modifiedPoints):
        self._computeHeight(cone(modifiedPoints))
    return

  def computeHeights(self):
    self._maxHeight = -1
    self._computeHeight(self.getLeaves());
    return

  def _computeDepth(self, points):
    modifiedPoints = set();

    for point in points:
      # Compute the max depth of the points in the cone of p, and add 1
      d0 = self._getValue("depth", point, -1)
      d1 = max(map(lambda p: self._getValue("depth", p, -1), cone(point))) + 1;

      if not d1 == d0:
        point._depth = d1
        if d1 > self._maxDepth: self._maxDepth = d1
        modifiedPoints.add(point)

      if len(modifiedPoints):
        self._computeDepth(support(modifiedPoints))
    return

  def computeDepths(self):
    self._maxDepth = -1
    self._computeDepth(self.getRoots());
    return
