class DepthFirstVisitor:
  '''
  Traverses the AST in a depth first manner
  '''
  def __init__(self):
    self.repository = []
    self.seen = []
    self.reverseEdges = 0
    self.vertexNumber = 0
  
  def getRepository(self):
    '''
    Retrieve auxilliary ASTs
    '''
    return self.repository
  
  def setRepository(self, repository):
    '''
    Set auxilliary ASTs
    '''
    self.repository = repository
  
  def visitVertex(self, vertex):
    self.discoverVertex(vertex)
    self.traverseVertex(vertex)
    self.finishVertex(vertex)
  
  def visitSource(self, vertex):
    self.visitVertex(vertex)
  
  def visitHeader(self, vertex):
    self.visitVertex(vertex)
  
  def visitPythonModule(self, vertex):
    self.visitVertex(vertex)
  
  def visitInclude(self, vertex):
    self.visitVertex(vertex)
  
  def visitDefine(self, vertex):
    self.visitVertex(vertex)
  
  def visitMacro(self, vertex):
    self.visitVertex(vertex)
  
  def visitPreprocessorConditional(self, vertex):
    self.visitVertex(vertex)
  
  def visitType(self, vertex):
    self.visitVertex(vertex)
  
  def visitPointer(self, vertex):
    self.visitVertex(vertex)
  
  def visitStruct(self, vertex):
    self.visitVertex(vertex)
  
  def visitBitField(self, vertex):
    self.visitVertex(vertex)
  
  def visitEnumeration(self, vertex):
    self.visitVertex(vertex)
  
  def visitEnumerator(self, vertex):
    self.visitVertex(vertex)
  
  def visitTypedef(self, vertex):
    self.visitVertex(vertex)
  
  def visitGroup(self, vertex):
    self.visitVertex(vertex)
  
  def visitDeclarator(self, vertex):
    self.visitVertex(vertex)
  
  def visitDeclaratorGroup(self, vertex):
    self.visitVertex(vertex)
  
  def visitArray(self, vertex):
    self.visitVertex(vertex)
  
  def visitFunction(self, vertex):
    self.visitVertex(vertex)
  
  def visitParameter(self, vertex):
    self.visitVertex(vertex)
  
  def visitDeclaration(self, vertex):
    self.visitVertex(vertex)
  
  def visitStatement(self, vertex):
    self.visitVertex(vertex)
  
  def visitExpressionStatement(self, vertex):
    self.visitVertex(vertex)
  
  def visitCompoundStatement(self, vertex):
    self.visitVertex(vertex)
  
  def visitIf(self, vertex):
    self.visitVertex(vertex)
  
  def visitSwitch(self, vertex):
    self.visitVertex(vertex)
  
  def visitWhile(self, vertex):
    self.visitVertex(vertex)
  
  def visitFor(self, vertex):
    self.visitVertex(vertex)
  
  def visitGoto(self, vertex):
    self.visitVertex(vertex)
  
  def visitContinue(self, vertex):
    self.visitVertex(vertex)
  
  def visitBreak(self, vertex):
    self.visitVertex(vertex)
  
  def visitReturn(self, vertex):
    self.visitVertex(vertex)
  
  def visitInitializer(self, vertex):
    self.visitVertex(vertex)
  
  def visitExpression(self, vertex):
    self.visitVertex(vertex)
  
  def visitAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitInclusiveOrAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitExclusiveOrAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitAndAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitLeftShiftAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitRightShiftAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitAdditionAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitSubtractionAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitMultiplicationAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitDivisionAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitModuloAssignment(self, vertex):
    self.visitVertex(vertex)
  
  def visitLogicalOr(self, vertex):
    self.visitVertex(vertex)
  
  def visitLogicalAnd(self, vertex):
    self.visitVertex(vertex)
  
  def visitInclusiveOr(self, vertex):
    self.visitVertex(vertex)
  
  def visitExclusiveOr(self, vertex):
    self.visitVertex(vertex)
  
  def visitAnd(self, vertex):
    self.visitVertex(vertex)
  
  def visitEquality(self, vertex):
    self.visitVertex(vertex)
  
  def visitInequality(self, vertex):
    self.visitVertex(vertex)
  
  def visitLessThan(self, vertex):
    self.visitVertex(vertex)
  
  def visitGreaterThan(self, vertex):
    self.visitVertex(vertex)
  
  def visitLessThanOrEqual(self, vertex):
    self.visitVertex(vertex)
  
  def visitGreaterThanOrEqual(self, vertex):
    self.visitVertex(vertex)
  
  def visitLeftShift(self, vertex):
    self.visitVertex(vertex)
  
  def visitRightShift(self, vertex):
    self.visitVertex(vertex)
  
  def visitAddition(self, vertex):
    self.visitVertex(vertex)
  
  def visitSubtraction(self, vertex):
    self.visitVertex(vertex)
  
  def visitMultiplication(self, vertex):
    self.visitVertex(vertex)
  
  def visitDivision(self, vertex):
    self.visitVertex(vertex)
  
  def visitModulo(self, vertex):
    self.visitVertex(vertex)
  
  def visitCast(self, vertex):
    self.visitVertex(vertex)
  
  def visitSizeof(self, vertex):
    self.visitVertex(vertex)
  
  def visitAddress(self, vertex):
    self.visitVertex(vertex)
  
  def visitIndirection(self, vertex):
    self.visitVertex(vertex)
  
  def visitUnaryPlus(self, vertex):
    self.visitVertex(vertex)
  
  def visitUnaryMinus(self, vertex):
    self.visitVertex(vertex)
  
  def visitComplement(self, vertex):
    self.visitVertex(vertex)
  
  def visitNegation(self, vertex):
    self.visitVertex(vertex)
  
  def visitIncrement(self, vertex):
    self.visitVertex(vertex)
  
  def visitStructReference(self, vertex):
    self.visitVertex(vertex)
  
  def visitArrayReference(self, vertex):
    self.visitVertex(vertex)
  
  def visitFunctionCall(self, vertex):
    self.visitVertex(vertex)
  
  def visitVariable(self, vertex):
    self.visitVertex(vertex)
  
  def visitConstant(self, vertex):
    self.visitVertex(vertex)
  
  def getReverseEdges(self):
    '''
    Retrieve the flag for reversing the orientation of edges in the traversal
    '''
    return self.reverseEdges
  
  def setReverseEdges(self, reverseEdges):
    '''
    Set the flag for reversing the orientation of edges in the traversal
    '''
    self.reverseEdges = reverseEdges
  
  def traverseVertex(self, vertex):
    '''
    Traverse each edge of this vertex
    '''
    self.setVertexSeen(vertex, 1)
    if self.getReverseEdges():
      neighbors = [vertex.parent]
    else:
      neighbors = vertex.children
    for neighbor in neighbors:
      neighbor.accept(self)
  
  def resetVertexNumbering(self):
    '''
    Reset the vertex discovery and finish numbering
    '''
    self.vertexNumber = 0
  
  def discoverVertex(self, vertex):
    '''
    Mark a vertex as discovered and increment the count
    '''
    self.vertexNumber += 1
  
  def finishVertex(self, vertex):
    '''
    Mark a vertex as finished and increment the count
    '''
    self.vertexNumber += 1
  
  def getVerticesSeen(self):
    '''
    Retrieve the list of vertices discovered
    '''
    return self.seen
  
  def setVerticesSeen(self, seen):
    '''
    Set the list of vertices discovered
    '''
    self.seen = list(seen)
  
  def getVertexSeen(self, vertex):
    '''
    Retrieve the vertex discovery state
    '''
    return vertex in self.seen
  
  def setVertexSeen(self, vertex, state):
    '''
    Set the vertex discovery state
    '''
    if state:
      if not (vertex in self.seen):
        self.seen.append(vertex)
    else:
      if vertex in self.seen:
        self.seen.remove(vertex)

class Printer(DepthFirstVisitor):
  '''
  This class prints a Cxx AST to standard output as Cxx source
  '''
  def __init__(self):
    DepthFirstVisitor.__init__(self)
    import sys
    self.tab = '  '
    self.indentSize = 0
    self.output = sys.stdout
  
  def getTab(self):
    '''
    Retrieve the tab string
    '''
    return self.tab
  
  def setTab(self, tab):
    '''
    Set the tab string
    '''
    self.tab = tab
  
  def getIndentSize(self):
    '''
    Retrieve the number of tabs of the current indent
    '''
    return self.indentSize
  
  def setIndentSize(self, indentSize):
    '''
    Set the number of tabs of the current indent
    Throws an exception on a negative indent size
    '''
    if self.indentSize < 0:
      raise RuntimeError('Indent size must be positive: ' + str(self.indentSize))
    self.indentSize = indentSize
  
  def increaseIndent(self):
    '''
    Increase the index size by one
    '''
    self.indentSize += 1
  
  def decreaseIndent(self):
    '''
    Decrease the index size by one
    Throws an exception on a negative indent size
    '''
    self.indentSize -= 1
    if self.indentSize < 0:
      raise RuntimeError('Indent size decreased below zero: ' + str(self.indentSize))
  
  def indent(self):
    '''
    Output the current indent (the indent size in tabs)
    '''
    for i in range(self.getIndentSize()):
      self.write(self.getTab())
  
  def write(self, s):
    '''
    Output a string
    '''
    self.output.write(s)
  
  def println(self, s):
    '''
    Output a string, properly indented and followed by a newline
    '''
    self.indent()
    self.write(s)
    self.write('\n')
  
  def visitInclude(self, vertex):
    self.println('#include ' + vertex.identifier)
  
  def visitDefine(self, vertex):
    self.indent()
    if vertex.undef:
      self.write('#undef ')
    else:
      self.write('#define ')
    self.write(vertex.identifier)
    if len(vertex.argumentNames):
      self.write(' (')
      self.write(', '.join(vertex.argumentNames))
      self.write(')')
    if vertex.replacementText:
      self.write(' ' + vertex.replacementText)
    self.write('\n')
  
  def visitMacro(self, vertex):
    self.write(vertex.identifier)
  
  def visitPreprocessorConditional(self, vertex):
    self.println('#' + vertex.identifier + ' ' + vertex.getBranchExpression())
    if vertex.getValue():
      num = len(vertex.children)
      for (c, child) in enumerate(vertex.children):
        child.accept(self)
        if c < (num - 1):
          self.write('\n')
      if vertex.getInactiveCode():
        self.println('#else')
        self.write(vertex.getInactiveCode())
        self.write('\n')
    else:
      self.write(vertex.getInactiveCode())
      self.write('\n')
      if len(vertex.children):
        self.println('#else')
        num = len(vertex.children)
        for (c, child) in enumerate(vertex.children):
          child.accept(self)
          if c < (num - 1):
            self.write('\n')
    self.println('#endif')
  
  def visitType(self, vertex):
    self.writeTypeQualifier(vertex)
    self.write(vertex.identifier)
  
  def visitPointer(self, vertex):
    from Cxx import Pointer
    self.visitVertex(vertex)
    if (len(vertex.children) and not (isinstance(vertex.children[0], Pointer))):
      self.write(' ')
    self.write('*')
    self.writeTypeQualifier(vertex, 1)
  
  def visitStruct(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.writeTypeQualifier(vertex)
    if vertex.getUnion():
      self.write('union')
    else:
      self.write('struct')
    if vertex.identifier:
      self.write(' ' + vertex.identifier)
    if len(vertex.children) > 0:
      self.write(' {\n')
      self.increaseIndent()
      self.visitVertex(vertex)
      self.decreaseIndent()
      self.indent()
      self.write('}')
  
  def visitBitField(self, vertex):
    self.writeTypeQualifier(vertex)
  
  def visitEnumeration(self, vertex):
    self.writeTypeQualifier(vertex)
    if len(vertex.children) == 0:
      self.write('enum ' + vertex.identifier)
    else:
      self.indent()
      self.write('enum ')
      if vertex.identifier:
        self.write(vertex.identifier + ' ')
      self.write('{')
      num = len(vertex.children)
      for (i, enumerator) in enumerate(vertex.children):
        enumerator.accept(self)
        if i < (num - 1):
          self.write(', ')
      self.write('}')
  
  def visitEnumerator(self, vertex):
    self.write(vertex.identifier)
    if vertex.getInitialized():
      self.write(' = ')
      self.write(str(vertex.getValue()))
  
  def visitTypedef(self, vertex):
    self.writeTypeQualifier(vertex)
    self.write(vertex.identifier)
  
  def visitGroup(self, vertex):
    self.write('(')
    self.visitVertex(vertex)
    self.write(')')
  
  def visitDeclarator(self, vertex):
    from Cxx import Pointer
    self.writeStorageClassSpecifier(vertex)
    if not (vertex.type is None):
      vertex.type.accept(self)
    if vertex.identifier:
      if (not (vertex.type is None) and not (isinstance(vertex.type, Pointer))):
        self.write(' ')
      self.write(vertex.identifier)
    if vertex.initializer:
      self.write(' = ')
      vertex.initializer.accept(self)
  
  def visitDeclaratorGroup(self, vertex):
    self.write('(')
    self.visitVertex(vertex)
    self.write(')')
  
  def visitArray(self, vertex):
    from Cxx import Pointer
    self.writeStorageClassSpecifier(vertex)
    if not (vertex.type is None):
      vertex.type.accept(self)
      if not (isinstance(vertex.type, Pointer)):
        self.write(' ')
    self.visitVertex(vertex)
    self.write('[')
    if vertex.size:
      vertex.size.accept(self)
    self.write(']')
    if vertex.initializer:
      self.write(' = ')
      vertex.initializer.accept(self)
  
  def visitFunction(self, vertex):
    from Cxx import Pointer
    self.writeStorageClassSpecifier(vertex)
    vertex.type.accept(self)
    if not (isinstance(vertex.type, Pointer)):
      self.write(' ')
    self.visitVertex(vertex)
    num = len(vertex.parameters)
    if num:
      self.write('(')
      for (i, parameter) in enumerate(vertex.parameters):
        parameter.accept(self)
        if i < (num - 1):
          self.write(', ')
      self.write(')')
    else:
      self.write('(void)')
    if not (vertex.getBody() is None):
      self.write('\n')
      vertex.getBody().accept(self)
    else:
      if not (vertex.initializer is None):
        self.write(' = ')
        vertex.initializer.accept(self)
  
  def visitParameter(self, vertex):
    from Cxx import Pointer
    self.writeComments(vertex)
    vertex.type.accept(self)
    if vertex.identifier:
      if not (isinstance(vertex.type, Pointer)):
        self.write(' ')
      self.write(vertex.identifier)
  
  def visitDeclaration(self, vertex):
    self.writeComments(vertex, newline = 1)
    for child in vertex.children:
      self.indent()
      child.accept(self)
      self.write(';\n')
  
  def visitStatement(self, vertex):
    self.visitVertex(vertex)
  
  def visitExpressionStatement(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    if not (vertex.identifier is None):
      self.write(vertex.identifier + ': ')
    self.visitVertex(vertex)
    self.write(';\n')
  
  def visitCompoundStatement(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    if not (vertex.identifier is None):
      self.write(vertex.identifier + ': ')
    self.write('{\n')
    self.increaseIndent()
    if len(vertex.getDeclarations()):
      for decl in vertex.getDeclarations():
        decl.accept(self)
      self.println('')
    self.visitVertex(vertex)
    self.decreaseIndent()
    self.println('}')
  
  def visitIf(self, vertex):
    self.writeComments(vertex, newline = 1)
    if not (vertex.getTernary()):
      self.indent()
      self.write('if (')
      vertex.getBranch().accept(self)
      self.write(')\n')
      vertex.children[0].accept(self)
      if len(vertex.children) > 1:
        self.println('else')
        vertex.children[1].accept(self)
    else:
      vertex.getBranch().accept(self)
      self.write(' ? ')
      vertex.children[0].accept(self)
      self.write(' : ')
      vertex.children[1].accept(self)
  
  def visitSwitch(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    self.write('switch(')
    vertex.branch.accept(self)
    self.write(') {\n')
    self.increaseIndent()
    inCase = 0
    for statement in vertex.children[0].children:
      if not (statement.caseLabel is None):
        if inCase:
          self.decreaseIndent()
          inCase = 0
        self.indent()
        if statement.caseLabel.identifier == 'default':
          self.write('default')
        else:
          self.write('case ')
          statement.caseLabel.accept(self)
        self.write(':\n')
        if not (inCase):
          self.increaseIndent()
          inCase = 1
      statement.accept(self)
    if inCase:
      self.decreaseIndent()
    self.decreaseIndent()
    self.println('}')
  
  def visitWhile(self, vertex):
    self.writeComments(vertex, newline = 1)
    if vertex.getDoWhile():
      self.println('do')
      self.visitVertex(vertex)
      self.indent()
      self.write('while(')
      vertex.getBranch().accept(self)
      self.write(')\n')
    else:
      self.indent()
      self.write('while(')
      vertex.getBranch().accept(self)
      self.write(')\n')
      self.visitVertex(vertex)
  
  def visitFor(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    self.write('for(')
    if not (vertex.getInitialization() is None):
      vertex.getInitialization().accept(self)
    self.write(';')
    if not (vertex.getBranch() is None):
      self.write(' ')
      vertex.getBranch().accept(self)
    self.write(';')
    if not (vertex.getIncrement() is None):
      self.write(' ')
      vertex.getIncrement().accept(self)
    self.write(')\n')
    self.visitVertex(vertex)
  
  def visitGoto(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.println('goto ' + vertex.identifier + ';')
  
  def visitContinue(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    if not (vertex.identifier is None):
      self.write(vertex.identifier + ': ')
    self.write('continue;\n')
  
  def visitBreak(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    if not (vertex.identifier is None):
      self.write(vertex.identifier + ': ')
    self.write('break;\n')

  def visitReturn(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    if not (vertex.identifier is None):
      self.write(vertex.identifier + ': ')
    self.write('return')
    if len(vertex.children):
      self.write(' ')
      self.visitVertex(vertex)
    self.write(';\n')

  def visitThrow(self, vertex):
    self.writeComments(vertex, newline = 1)
    self.indent()
    if not (vertex.identifier is None):
      self.write(vertex.identifier + ': ')
    self.write('throw')
    if len(vertex.children):
      self.write(' ')
      self.visitVertex(vertex)
    self.write(';\n')

  def visitInitializer(self, vertex):
    from Cxx import Initializer
    from Cxx import Macro
    num = len(vertex.children)
    nested = (not (vertex.parent is None) and isinstance(vertex.parent, Initializer))
    if num > 1:
      self.write('{')
      self.increaseIndent()
      if not (nested):
        self.write('\n')
        self.indent()
      for (c, child) in enumerate(vertex.children):
        if not (nested):
          if c > 0:
            self.indent()
        child.accept(self)
        if nested:
          if c < (num - 1):
            if not (isinstance(child, Macro)):
              self.write(',')
            self.write(' ')
        else:
          if c < (num - 1):
            if not (isinstance(child, Macro)):
              self.write(',')
            self.write('\n')
      self.decreaseIndent()
      self.write('}')
    elif vertex.list:
      self.write('{')
      vertex.children[0].accept(self)
      self.write('}')
    else:
      self.visitVertex(vertex)
  
  def visitAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitInclusiveOrAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitExclusiveOrAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitAndAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitLeftShiftAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitRightShiftAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitAdditionAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitSubtractionAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitMultiplicationAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitDivisionAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitModuloAssignment(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitLogicalOr(self, vertex):
    self.visitNaryExpression(vertex)
  
  def visitLogicalAnd(self, vertex):
    self.visitNaryExpression(vertex)
  
  def visitInclusiveOr(self, vertex):
    self.visitNaryExpression(vertex)
  
  def visitExclusiveOr(self, vertex):
    self.visitNaryExpression(vertex)
  
  def visitAnd(self, vertex):
    self.visitNaryExpression(vertex)
  
  def visitEquality(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitInequality(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitLessThan(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitGreaterThan(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitLessThanOrEqual(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitGreaterThanOrEqual(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitLeftShift(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitRightShift(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitAddition(self, vertex):
    self.visitNaryExpression(vertex)
  
  def visitSubtraction(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitMultiplication(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitDivision(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitModulo(self, vertex):
    self.visitBinaryExpression(vertex)
  
  def visitCast(self, vertex):
    self.write('(')
    vertex.getTypeName().accept(self)
    self.write(') ')
    self.visitVertex(vertex)
  
  def visitSizeof(self, vertex):
    if len(vertex.children):
      self.write('sizeof ')
      self.visitVertex(vertex)
    else:
      self.write('sizeof(')
      vertex.getTypeName().accept(self)
      self.write(')')
  
  def visitAddress(self, vertex):
    self.visitUnaryExpression(vertex)
  
  def visitIndirection(self, vertex):
    self.visitUnaryExpression(vertex)
  
  def visitUnaryPlus(self, vertex):
    self.visitUnaryExpression(vertex)
  
  def visitUnaryMinus(self, vertex):
    self.visitUnaryExpression(vertex)
  
  def visitComplement(self, vertex):
    self.visitUnaryExpression(vertex)
  
  def visitNegation(self, vertex):
    self.visitUnaryExpression(vertex)
  
  def visitIncrement(self, vertex):
    if vertex.getDecrement():
      if vertex.getPostfix():
        self.visitVertex(vertex)
        self.write('--')
      else:
        self.write('--')
        self.visitVertex(vertex)
    else:
      if vertex.getPostfix():
        self.visitVertex(vertex)
        self.write('++')
      else:
        self.write('++')
        self.visitVertex(vertex)
  
  def visitStructReference(self, vertex):
    self.visitVertex(vertex)
    if vertex.getPointer():
      self.write('->')
    else:
      self.write('.')
    self.write(vertex.getMember())
  
  def visitArrayReference(self, vertex):
    self.visitVertex(vertex)
    self.write('[')
    vertex.getIndex().accept(self)
    self.write(']')
  
  def visitFunctionCall(self, vertex):
    self.visitVertex(vertex)
    self.write('(')
    num = len(vertex.arguments)
    for (a, argument) in enumerate(vertex.arguments):
      argument.accept(self)
      if a < (num - 1):
        self.write(', ')
    self.write(')')
  
  def visitVariable(self, vertex):
    self.write(vertex.identifier)
  
  def visitConstant(self, vertex):
    if not (vertex.identifier is None):
      self.write('"' + vertex.identifier + '"')
    else:
      self.write(str(vertex.value))
  
  def shitMethod1(self, x, y):
    '''
    Should be lambda x, y: x or y
    '''
    return (x or y)
  
  def getDeclAbstract(self, decl):
    '''
    Determine whether a declaration is abstract
       - The argument can be either a declaration or a type
    '''
    from Cxx import Array
    from Cxx import Function
    from Cxx import DeclaratorGroup
    if reduce(self.shitMethod1, [isinstance(decl, interface) for interface in [Array, Function, DeclaratorGroup]], 0):
      isAbstract = self.getDeclAbstract(decl.children[0])
    else:
      isAbstract = decl.identifier is None
    return isAbstract
  
  def visitUnaryExpression(self, vertex):
    children = vertex.children
    if not (len(children) == 1):
      raise RuntimeError('Unary expressions must have exactly one child')
    self.write(vertex.identifier)
    children[0].accept(self)
    return
  
  def visitBinaryExpression(self, vertex):
    children = vertex.children
    if not (len(children) == 2):
      raise RuntimeError('Binary expressions must have exactly two children')
    children[0].accept(self)
    self.write(' ' + vertex.identifier + ' ')
    children[1].accept(self)
    return
  
  def visitNaryExpression(self, vertex):
    children = vertex.children
    if len(children) < 2:
      raise RuntimeError('Nary expressions must have at least two children')
    children[0].accept(self)
    for child in children[1:]:
      self.write(' ' + vertex.identifier + ' ')
      child.accept(self)
    return
  
  def writeComments(self, vertex, newline = 0):
    for comment in vertex.comments:
      if newline:
        self.println('/* ' + comment.comment + ' */')
      else:
        self.write('/*' + comment.comment + '*/ ')
    return
  
  def writeTypeQualifier(self, type, isPointer = 0):
    if type.const:
      if isPointer:
        self.write(' ')
      self.write('const ')
    if type.volatile:
      if isPointer:
        self.write(' ')
      self.write('volatile ')
    return None
  
  def writeStorageClassSpecifier(self, declarator):
    if declarator.extern:
      self.write('extern ')
    if declarator.static:
      self.write('static ')
    if declarator.typedef:
      self.write('typedef ')
    return None

class Output(Printer):
  '''
  This class prints a Cxx AST to the file system as Cxx source
  '''
  def __init__(self):
    Printer.__init__(self)
    self.disableOutput = 0
    self.disableSource = 0
    self.roots = {}

    self.files = []
    self.f = None
    self.includes = []
    self.systemIncludes = []
    return

  def getDisableOutput(self):
    '''
    Retrieve the flag to disable writing to disk
    This is useful for generating the list of files which would have been output
    '''
    return self.disableOutput
  
  def setDisableOutput(self, disableOutput):
    '''
    Set the flag to disable writing to disk
    This is useful for generating the list of files which would have been output
    '''
    self.disableOutput = disableOutput
  
  def getDisableSource(self):
    '''
    Retrieve the flag to disable writing source
    Headers and other support files will still be output
    '''
    return self.disableSource
  
  def setDisableSource(self, disableSource):
    '''
    Set the flag to disable writing source
    Headers and other support files will still be output
    '''
    self.disableSource = disableSource
  
  def getRoot(self, purpose):
    '''
    Retrieve the directory root for files generated for the given purpose
    The purpose can be stub, cartilage, IOR, skeleton, or implementation
    '''
    return self.roots[purpose]
  
  def setRoot(self, purpose, root):
    '''
    Retrieve the directory root for files generated for the given purpose
    The purpose can be stub, cartilage, IOR, skeleton, or implementation
    '''
    self.roots[purpose] = root
  
  def getIncludeDirectories(self):
    '''
    Retrieve the list of include directories
    This seems general enough to include for all languages
    '''
    return self.includes
  
  def setIncludeDirectories(self, dirs):
    '''
    Set the list of include directories
    This seems general enough to include for all languages
    '''
    self.includes = list(dirs)
  
  def getSystemIncludeDirectories(self):
    '''
    Retrieve the list of system include directories
    This seems general enough to include for all languages
    '''
    return self.systemIncludes
  
  def setSystemIncludeDirectories(self, dirs):
    '''
    Set the list of system include directories
    This seems general enough to include for all languages
    '''
    self.systemIncludes = list(dirs)
  
  def getFiles(self):
    '''
    Retrieve the list of output files
    '''
    return self.files
  
  def setFiles(self, files):
    '''
    Set the list of output files
    '''
    self.files = list(files)
  
  def visitSource(self, vertex):
    if self.disableSource:
      return None
    from GenericCompiler import CodePurpose
    import os
    if not (os.path.splitext(vertex.filename)[1] == '.c'):
      raise RuntimeError('A source file must have a .c extension: ' + vertex.filename)
    filename = os.path.join(self.getRoot(vertex.purpose), vertex.filename)
    if not (os.path.exists(os.path.dirname(filename))):
      os.makedirs(os.path.dirname(filename))
    if (not (self.disableOutput) and os.path.isfile(filename) and vertex.purpose == CodePurpose.IMPLEMENTATION):
      from GenericCompiler import ParserException
      from Cxx import Parser
      try:
        print 'Parsing', filename
        parser = Parser()
        parser.systemIncludeDirectories = self.systemIncludeDirectories
        parser.includeDirectories = self.includeDirectories
        userModule = parser.parseFile(filename)
      except ParserException:
        pass
      children = list(vertex.children)
      userChildren = list(userModule.children)
      userStdIncludes = self.checkStandardIncludes(children, userChildren)
      userIncludes = self.checkIncludes(children, userChildren)
      userImplStruct = self.checkImplStruct(children, userChildren)
      userSetEpvDecl = self.checkSetEpvDecl(children, userChildren)
      userMethods = self.checkMethods(children, userChildren)
      vertex.setChildren(userStdIncludes + userIncludes + userImplStruct + userSetEpvDecl + userMethods)
    print 'Creating', filename
    self.files.append(filename)
    if self.disableOutput:
      if not (os.path.isfile(filename)):
        raise RuntimeError('SIDL file ' + filename + ' has not been generated')
    else:
      self.f = file(filename, 'w')
      num = len(vertex.children)
      for (c, child) in enumerate(vertex.children):
        child.accept(self)
        if c < (num - 1):
          self.write('\n')
      self.f.close()
  
  def visitHeader(self, vertex):
    import os
    if not (os.path.splitext(vertex.filename)[1] == '.h'):
      raise RuntimeError('A header file must have a .h extension: ' + vertex.filename)
    filename = os.path.join(self.getRoot(vertex.purpose), vertex.filename)
    if not (os.path.exists(os.path.dirname(filename))):
      os.makedirs(os.path.dirname(filename))
    print 'Creating', filename
    self.files.append(filename)
    if self.disableOutput:
      if not (os.path.isfile(filename)):
        raise RuntimeError('SIDL file ' + filename + ' has not been generated')
    else:
      self.f = file(filename, 'w')
      num = len(vertex.children)
      for (c, child) in enumerate(vertex.children):
        child.accept(self)
        if c < (num - 1):
          self.write('\n')
      self.f.close()
  
  def visitPythonModule(self, vertex):
    if self.disableSource:
      return None
    import ASE.Compiler.CodePurpose
    import os
    klass = vertex.getClass().split('.')
    filename = apply(os.path.join, [self.getRoot(ASE.Compiler.CodePurpose.STUB)] + klass[:-(1)] + [klass[-(1)] + '.c'])
    if not (os.path.exists(os.path.dirname(filename))):
      os.makedirs(os.path.dirname(filename))
    print 'Creating', filename
    self.files.append(filename)
    if self.disableOutput:
      if not (os.path.isfile(filename)):
        raise RuntimeError('SIDL file ' + filename + ' has not been generated')
    else:
      self.f = file(filename, 'w')
      num = len(vertex.children)
      for (c, child) in enumerate(vertex.children):
        child.accept(self)
        if c < (num - 1):
          self.write('\n')
      self.f.close()
  
  def write(self, s):
    '''
    This is overridden to redirect output to a file
    '''
    self.f.write(s)
  
  def getCodeSection(self, nodes, check):
    '''
    Returns a tuple of nodes preceeding a code block together with the code block
       - The block now consists of a single statement
       - The check function is called on each node, and should return true for termination
    '''
    prefixBlock = []
    codeBlock = []
    for node in nodes:
      if check(node):
        break
      prefixBlock.append(node)
    if len(prefixBlock) == len(nodes):
      raise RuntimeError('Missing code block')
    codeBlock.append(nodes[len(prefixBlock)])
    del nodes[:len(prefixBlock) + len(codeBlock)]
    return (prefixBlock, codeBlock)
  
  def checkStandardIncludes(self, genChildren, userChildren):
    '''
    The first section of generated children is a standard include
    '''
    
    def check(node):
      from Cxx import Include
      if (isinstance(node, Include) and node.identifier == '<stdlib.h>'):
        return 1
      return 0
    (genPrefix, genIncludes) = self.getCodeSection(genChildren, check)
    if len(genPrefix):
      raise RuntimeError('Invalid generated module: standard includes block cannot be preceeded by code')
    (userPrefix, userIncludes) = self.getCodeSection(userChildren, check)
    return userPrefix + genIncludes
  
  def checkIncludes(self, genChildren, userChildren):
    '''
    The second section of generated children is a guarded IOR include
    '''
    
    def check(node):
      from Cxx import Include
      from Cxx import PreprocessorConditional
      if (isinstance(node, PreprocessorConditional) and node.identifier == 'ifndef'):
        import ASE.Compiler.Cxx.PreprocessorConditional
        if (len(node.children) and isinstance(node.children[0], Include)):
          return 1
        if ASE.Compiler.Cxx.PreprocessorConditional.PreprocessorConditional(node).getInactiveCode().startswith('# include <'):
          return 1
      return 0
    (genPrefix, genIncludes) = self.getCodeSection(genChildren, check)
    if len(genPrefix):
      raise RuntimeError('Invalid generated module: includes block cannot be preceeded by code')
    (userPrefix, userIncludes) = self.getCodeSection(userChildren, check)
    return userPrefix + genIncludes
  
  def checkImplStruct(self, genChildren, userChildren):
    '''
    The third section of generated children is the declaration of the implementation structure
    '''
    import ASE.Compiler.Cxx.Function
    includes = []
    
    def checkInclude(node):
      from Cxx import Include
      from Cxx import PreprocessorConditional
      if (isinstance(node, PreprocessorConditional) and node.identifier == 'ifndef'):
        import ASE.Compiler.Cxx.PreprocessorConditional
        if (len(node.children) and isinstance(node.children[0], Include)):
          return 1
        if ASE.Compiler.Cxx.PreprocessorConditional.PreprocessorConditional(node).getInactiveCode().startswith('# include <'):
          return 1
      return 0
    
    def check(node):
      from Cxx import Declaration
      from Cxx import Struct
      if isinstance(node, Declaration):
        if len(node.children) == 1:
          struct = node.children[0]
          if isinstance(struct, Struct):
            if (len(struct.children) and isinstance(struct.children[0], Declaration)):
              decl = struct.children[0]
              if (len(decl.children) and decl.children[0].identifier == 'self'):
                return 1
      elif checkInclude(node):
        import ASE.Compiler.Cxx.PreprocessorConditional
        includes.append(ASE.Compiler.Cxx.PreprocessorConditional.PreprocessorConditional(node))
      return 0
    
    def checkBody(struct):
      if len(struct.children):
        if struct.children[0].isInstanceOf('ASE.Compiler.Cxx.Declaration'):
          decl = struct.children[0].children[0]
          if (decl.isInstanceOf('ASE.Compiler.Cxx.Declarator') and decl.identifier == 'self'):
            return 1
      return 0
    (genPrefix, genDecl) = self.getCodeSection(genChildren, check)
    genIncludes = includes[:]
    includes = []
    if len(genPrefix) != len(genIncludes):
      raise RuntimeError('Invalid generated module: implementation structure block cannot be preceeded by code')
    genStruct = genDecl[0].children[0]
    (userPrefix, userDecl) = self.getCodeSection(userChildren, check)
    userIncludes = includes[:]
    userStruct = userDecl[0].children[0]
    if not (checkBody(userStruct)):
      raise RuntimeError('User implementation struct must begin with the self pointer')
    if len(userStruct.children) > 1:
      genStruct.addChildren(userStruct.children[1:])
    
    def equalInclude(nodeA, nodeB):
      import ASE.Compiler.Cxx.Include
      filenames = []
      for node in [nodeA, nodeB]:
        if len(node.children):
          filenames.append(ASE.Compiler.Cxx.Include.Include(node.children[0]).identifier)
        else:
          filenames.append(node.getInactiveCode().replace(' ', '')[8:])
      return filenames[0] == filenames[1]
    for inc in genIncludes:
      found = 0
      for node in userIncludes:
        if equalInclude(inc, node):
          found = 1
          break
      if not (found):
        userPrefix.append(inc)
    return userPrefix + genDecl
  
  def checkSetEpvDecl(self, genChildren, userChildren):
    '''
    The fourth section of generated children is the declaration of set_epv()
    '''
    import ASE.Compiler.Cxx.Function
    
    def check(node):
      if node.isInstanceOf('ASE.Compiler.Cxx.Declaration'):
        func = node.children[0]
        if func.isInstanceOf('ASE.Compiler.Cxx.Function'):
          name = func.children[0]
          if (name.isInstanceOf('ASE.Compiler.Cxx.Declarator') and name.identifier.endswith('_set_epv') and ASE.Compiler.Cxx.Function.Function(func).getBody() is None):
            return 1
      return 0
    (genPrefix, genIncludes) = self.getCodeSection(genChildren, check)
    if len(genPrefix):
      raise RuntimeError('Invalid generated module: set_epv declaration block cannot be preceeded by code')
    (userPrefix, userIncludes) = self.getCodeSection(userChildren, check)
    return userPrefix + genIncludes
  
  def checkSetEpv(self, genChildren, userChildren):
    '''
    The fourth section of generated children is the definition of set_epv()
    '''
    import ASE.Compiler.Cxx.Function
    
    def check(node):
      if node.isInstanceOf('ASE.Compiler.Cxx.Function'):
        name = node.children[0]
        if (name.isInstanceOf('ASE.Compiler.Cxx.Declarator') and name.identifier.endswith('_set_epv') and not (ASE.Compiler.Cxx.Function.Function(node).getBody() is None)):
          return 1
      return 0
    (genPrefix, genIncludes) = self.getCodeSection(genChildren, check)
    if len(genPrefix):
      raise RuntimeError('Invalid generated module: set_epv block cannot be preceeded by code')
    (userPrefix, userIncludes) = self.getCodeSection(userChildren, check)
    return userPrefix + genIncludes
  
  def checkMethods(self, genChildren, userChildren):
    '''
    The fifth section of generated children is the definition of implemented methods
    '''
    import ASE.Compiler.Cxx.CompoundStatement
    import ASE.Compiler.Cxx.Declarator
    import ASE.Compiler.Cxx.Function
    
    def check(node):
      import ASE.Compiler.Cxx.Function
      if node.isInstanceOf('ASE.Compiler.Cxx.Function'):
        name = node.children[0]
        if (name.isInstanceOf('ASE.Compiler.Cxx.Declarator') and name.identifier and not (ASE.Compiler.Cxx.Function.Function(node).getBody() is None)):
          return 1
      return 0
    
    def checkBody(body):
      import ASE.Compiler.Cxx.Declarator
      import ASE.Compiler.Cxx.Pointer
      decls = body.getDeclarations()
      if len(decls):
        if decls[0].isInstanceOf('ASE.Compiler.Cxx.Declaration'):
          decl = ASE.Compiler.Cxx.Declarator.Declarator(decls[0].children[0])
          if (decl.isInstanceOf('ASE.Compiler.Cxx.Declarator') and decl.identifier == '_obj'):
            if decl.type.isInstanceOf('ASE.Compiler.Cxx.Pointer'):
              if ASE.Compiler.Cxx.Pointer.Pointer(decl.type).children[0].isInstanceOf('ASE.Compiler.Cxx.Struct'):
                return 1
      return 0
    userMethods = {}

    for child in userChildren:
      if not (check(child)):
        raise RuntimeError('Invalid user code: Implementation must end with SIDL method defintions')
      userMethods[child.children[0].identifier] = child
    del userChildren[:]
    for child in genChildren[:-(3)]:
      func = ASE.Compiler.Cxx.Function.Function(child)
      body = ASE.Compiler.Cxx.CompoundStatement.CompoundStatement(func.getBody())
      name = func.children[0].identifier
      if name in userMethods:
        userFunc = ASE.Compiler.Cxx.Function.Function(userMethods[name])
        userBody = ASE.Compiler.Cxx.CompoundStatement.CompoundStatement(userFunc.getBody())
        if not (checkBody(userBody)):
          raise RuntimeError('Implementation method ' + name + ' must start with the object declaration')
        body.addDeclarations(userBody.getDeclarations()[1:])
        body.addChildren(userBody.children)
    for child in genChildren[-(3):-(1)]:
      func = ASE.Compiler.Cxx.Function.Function(child)
      body = ASE.Compiler.Cxx.CompoundStatement.CompoundStatement(func.getBody())
      name = func.children[0].identifier
      if name in userMethods:
        userFunc = ASE.Compiler.Cxx.Function.Function(userMethods[name])
        userBody = ASE.Compiler.Cxx.CompoundStatement.CompoundStatement(userFunc.getBody())
        body.setDeclarations(userBody.getDeclarations())
        body.setChildren(userBody.children)
    return genChildren
