class CxxNode:
  def __init__(self):
    self.identifier = None
    self.lineNumber = -1
    self.comments   = []
    self.parent     = None
    self.children   = []
    return

  def getChildren(self):
    return self.children
  def setChildren(self, children):
    self.children = children
    for c in children:
      c.parent = self
    return
  children = property(getChildren, setChildren, doc = 'Children of this tree node')

  def copy(self):
    '''Deep copy of this node'''
    root = self.__class__()
    root.identifier = self.identifier
    root.lineNumber = self.lineNumber
    root.comments = self.comments
    root.children = [child.copy() for child in self.children]
    return root

  def accept(self, visitor):
    '''This method is called to act on this node with the given visitor
       - It normally dispatches to the correct visitor callback'''
    return

class Type(CxxNode):
  '''
  A C type
  Appendix A 8.2-4, K&R
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.baseType = False
    self.const = False
    self.volatile = False
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitType(self)

  def getBaseType(self):
    return self.baseType
  def setBaseType(self, baseType):
    self.baseType = baseType

  def getConst(self):
    return self.const
  def setConst(self, const):
    self.const = const

  def getVolatile(self):
    return self.volatile
  def setVolatile(self, volatile):
    self.volatile = volatile

class BitField(Type):
  '''
  A C bitfield
  This is only a member of a struct or union
  '''
  def __init__(self):
    Type.__init__(self)
    self.size = 0
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitBitField(self)

  def copy(self):
    '''
    Need to override this
    '''
    root = Type.copy(self)
    root.size = self.size
    return root

class Comment(CxxNode):
  '''
  A Cxx comment
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.comment = ''
    return

class Define(CxxNode):
  '''
  A preprocessor define
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.undef = False
    self.argumentNames = []
    self.replacementText = ''
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitDefine(self)

class Enumeration:
  '''
  A C enumeration
  The children are Enumerators
  '''
  def __init__(self):
    self.isInitialized = 0
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitEnumeration(self)

  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Enumeration.Enumeration()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    root.setBaseType(self.getBaseType())
    root.setConst(self.getConst())
    root.setVolatile(self.getVolatile())
    root.setInitialized(self.getInitialized())
    return root
  
  def getInitialized(self):
    '''
    Retrieve the flag indicating whether the enumeration values are specified
    '''
    return self.isInitialized
  
  def setInitialized(self, initialized):
    '''
    Set the flag indicating whether the enumeration values are specified
    '''
    self.isInitialized = initialized

class Enumerator:
  '''
  A component of a C enumeration
  '''
  def __init__(self):
    self.isInitialized = 0
    self.value = 0
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitEnumerator(self)

  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Enumerator.Enumerator()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    root.setInitialized(self.getInitialized())
    root.setValue(self.getValue())
    return root
  
  def getInitialized(self):
    '''
    Retrieve the flag indicating whether the entry had an initializer
    '''
    return self.isInitialized
  
  def setInitialized(self, initialized):
    '''
    Set the flag indicating whether the entry had an initializer
    '''
    self.isInitialized = initialized
  
  def getValue(self):
    '''
    Retrieve the entry value
    '''
    return self.value
  
  def setValue(self, value):
    '''
    Set the entry value
    '''
    self.value = value

class Source(CxxNode):
  '''
  A C source file
  The children are translation units
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.filename = None
    self.purpose = None
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitSource(self)

  def copy(self):
    '''
    Need to override this
    '''
    root = CxxNode.copy(self)
    root.filename = self.filename
    root.purpose = self.purpose
    return root

class Header(Source):
  '''
  A C header file, which must have a .h extension
  '''
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitHeader(self)

class Include(CxxNode):
  '''
  A preprocessor include
  '''

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitInclude(self)

class Macro:
  '''
  A C macro which will be expanded by some #define
  '''
  def __init__(self):
    pass
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitMacro(self)

class Parser:
  '''
  A Cxx parser
  '''
  def __init__(self):
    self.includeDirs = []
    self.systemIncludeDirs = []
    self.currentFile = 'Unknown'
    return

  def parse(self, buf):
    '''
    Parse the buffer and return an AST object
    '''
    from Cxx import CxxNode
    self.holder = CxxNode()
    expander = ASE.Compiler.Cxx.cIncludeExpansion.IncludeExpander()
    expander.includeDirs = self.includeDirs
    expander.systemIncludeDirs = self.systemIncludeDirs
    expander.currentFile = self.currentFile
    cparse.setup()
    cparse.lexer = expander
    try:
      children = cparse.parser.parse(buf, lexer = expander)
    except Exception, e:
      import sys
      import traceback
      print e
      print traceback.print_tb(sys.exc_info()[2])
      raise RuntimeError('Bad parse')
    self.holder.setChildren(children)
    cparse.typeDefs.clear()
    cparse.parser.restart()
    expander.typeDefs.clear()
    return self.holder
  
  def parseFile(self, filename):
    '''
    Parse the file and return an AST object
    '''
    f = file(filename)
    s = f.read()
    f.close()
    self.currentFile = filename
    self.holder = self.parse(s)
    self.currentFile = 'Unknown'
    if filename.endswith('.c'):
      from GenericCompiler import CodePurpose
      from Cxx import Source
      source = Source()
      source.filename = filename
      source.children = self.holder.getChildren()
      source.purpose  = CodePurpose.STUB
      self.holder = source
    elif filename.endswith('.h'):
      from GenericCompiler import CodePurpose
      from Cxx import Header
      header = Header()
      header.filename = filename
      header.children = self.holder.getChildren()
      header.purpose  = ASE.Compiler.CodePurpose.STUB
      self.holder = header
    return self.holder

class Pointer(Type):
  '''
  A C pointer
  The child is a type
  '''
  def __init__(self):
    Type.__init__(self)

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitPointer(self)

class PreprocessorConditional:
  '''
  This represents a #if, #ifdef, or #ifndef
  The identifier is the conditional name, e.g. ifdef
  The children are the active code trees
  '''
  def __init__(self):
    self.branchExpression = None
    self.value = 0
    self.inactiveCode = None
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitPreprocessorConditional(self)
  
  def getBranchExpression(self):
    '''
    Retrieve the branch expression text
    '''
    return self.branchExpression
  
  def setBranchExpression(self, branchExpression):
    '''
    Set the branch expression text
    '''
    self.branchExpression = branchExpression
  
  def getValue(self):
    '''
    Retrieve the value of the branch expression
    '''
    return self.value
  
  def setValue(self, value):
    '''
    Set the value of the branch expression
    '''
    self.value = value
  
  def getInactiveCode(self):
    '''
    Retrieve the code bypassed by the jump
    '''
    return self.inactiveCode
  
  def setInactiveCode(self, inactiveCode):
    '''
    Set the code bypassed by the jump
    '''
    self.inactiveCode = inactiveCode

class PythonModule:
  '''
  A Python module implemented in C
  '''
  def __init__(self):
    self.klass = None
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitPythonModule(self)
  
  def getClass(self):
    '''
    Retrieve the fully qualified Python class name
    '''
    return self.klass
  
  def setClass(self, klass):
    '''
    Set the fully qualified Python class name
    '''
    self.klass = klass

class Struct(Type):
  '''
  A C struct or union
  The children are structure entries if present
  '''
  def __init__(self):
    Type.__init__(self)
    self.union = False
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitStruct(self)

  def copy(self):
    '''
    Need to override this
    '''
    root = Type.copy(self)
    root.union = self.union
    return root

class Typedef:
  '''
  A C type which is valid due to a typedef statement
  The child is the original type
  Typedef statement themselves are handled by the include expander, not the parser
  '''
  def __init__(self):
    pass
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitTypedef(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Typedef.Typedef()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    root.setBaseType(self.getBaseType())
    root.setConst(self.getConst())
    root.setVolatile(self.getVolatile())
    return root

class Declarator(CxxNode):
  '''
  A C declarator
  Appendix A 8.5-7, K&R
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.type = None
    self.static = False
    self.extern = False
    self.typedef = False
    self.initializer = None
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitDeclarator(self)

  def getType(self):
    return self.type
  def setType(self, t):
    self.type = t

  def getStatic(self):
    return self.static
  def setStatic(self, static):
    self.static = static

  def getExtern(self):
    return self.extern
  def setExten(self, extern):
    self.extern = extern

  def getTypedef(self):
    return self.typedef
  def setTypedef(self, typedef):
    self.typedef = typedef

  def getInitializer(self):
    return self.initializer
  def setInitializer(self, initializer):
    self.initializer = initializer

class Array(Declarator):
  '''
  A C array
  The child is the type modifier, which is just an identifier for normal arrays (See K&R, A8.6.2)
  '''
  def __init__(self):
    Declarator.__init__(self)
    self.size = None
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitArray(self)

  def copy(self):
    '''
    Need to override this
    '''
    root = Declarator.copy(self)
    root.size = self.size
    return root

class Declaration(CxxNode):
  '''
  A C declaration
  The children are declarators or struct, union, and enum definitions
  '''

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitDeclaration(self)

class DeclaratorGroup(Declarator):
  '''
  A C declarator grouped using parentheses
  The child is the original declarator
  '''
  def __init__(self):
    Declarator.__init__(self)
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitDeclaratorGroup(self)

class Function(Declarator):
  '''
  A C function
  The return type is given by getType()
  The child is the type modifier, which is just an identifier for normal functions (See K&R, A8.6.3)
  '''
  def __init__(self):
    Declarator.__init__(self)
    self.parameters = []
    self.optionalArguments = 0
    self.body = None
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitFunction(self)

  def getOptionalArguments(self):
    '''
    Retrieve the flag for optional arguments (a trailing ...)
    '''
    return self.optionalArguments

  def setOptionalArguments(self, optionalArguments):
    '''
    Set the flag for optional arguments (a trailing ...)
    '''
    self.optionalArguments = optionalArguments

  def getBody(self):
    '''
    Retrieve the function body, which is absent for declarations
    '''
    return self.body

  def setBody(self, body):
    '''
    Set the function body, which is absent for declarations
    '''
    self.body = body

class Parameter(Declarator):
  '''
  A C function parameter
  '''
  def __init__(self):
    Declarator.__init__(self)
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitParameter(self)

class Expression(CxxNode):
  '''
  A C expression
  This is a general expression, although we will generally assume its a constant_expression
  '''
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitExpression(self)

class AdditionAssignment(Expression):
  '''
  A C assignment expression (+=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '+='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitAdditionAssignment(self)

class AndAssignment(Expression):
  '''
  A C assignment expression (&=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '&='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitAndAssignment(self)

class And(Expression):
  '''
  A C and expression (&)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '&'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitAnd(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.And.And()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class Assignment(Expression):
  '''
  A C assignment expression (=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitAssignment(self)

class DivisionAssignment(Expression):
  '''
  A C assignment expression (/=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.setIdentifier = '/='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitDivisionAssignment(self)

class Equality(Expression):
  '''
  A C equality expression (==)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '=='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitEquality(self)

class ExclusiveOrAssignment(Expression):
  '''
  A C assignment expression (^=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.setIdentifier = '^='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitExclusiveOrAssignment(self)

class ExclusiveOr(Expression):
  '''
  A C exclusive-or expression (^)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '^'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitExclusiveOr(self)

class GreaterThanOrEqual(Expression):
  '''
  A C greater-than-or-equal expression (>=)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '>='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitGreaterThanOrEqual(self)

class GreaterThan(Expression):
  '''
  A C greater-than expression (>)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '>'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitGreaterThan(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.GreaterThan.GreaterThan()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class Group(Expression):
  '''
  A C expression grouped using parentheses
  The child is the original expression
  '''
  def __init__(self):
    Expression.__init__(self)
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitGroup(self)

class InclusiveOrAssignment:
  '''
  A C assignment expression (|=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    self.identifier = '|='
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitInclusiveOrAssignment(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.InclusiveOrAssignment.InclusiveOrAssignment()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class InclusiveOr(Expression):
  '''
  A C inclusive-or expression (|)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '|'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitInclusiveOr(self)

class Inequality(Expression):
  '''
  A C inequality expression (!=)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '!='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitInequality(self)

class LeftShiftAssignment:
  '''
  A C assignment expression (<<=)
  The children are the lvalue and rvalue
  '''
  def __init__(self, IORself):
    self.identifier = '<<='
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitLeftShiftAssignment(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.LeftShiftAssignment.LeftShiftAssignment()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class LessThanOrEqual:
  '''
  A C less-than-or-equal expression (<=)
  The children are the two subexpressions
  '''
  def __init__(self):
    self.identifier = '<='
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitLessThanOrEqual(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.LessThanOrEqual.LessThanOrEqual()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class LessThan(Expression):
  '''
  A C less-than expression (<)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '<'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitLessThan(self)

class LogicalAnd:
  '''
  A C logical or expression (&&)
  The children are the two subexpressions
  '''
  def __init__(self):
    self.identifier = '&&'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitLogicalAnd(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.LogicalAnd.LogicalAnd()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class LogicalOr:
  '''
  A C logical or expression (||)
  The children are the two subexpressions
  '''
  def __init__(self):
    self.identifier = '||'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitLogicalOr(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.LogicalOr.LogicalOr()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class ModuloAssignment:
  '''
  A C assignment expression (%=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    self.identifier = '%='
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitModuloAssignment(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.ModuloAssignment.ModuloAssignment()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class MultiplicationAssignment:
  '''
  A C assignment expression (*=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    self.identifier = '*='
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitMultiplicationAssignment(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.MultiplicationAssignment.MultiplicationAssignment()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class RightShiftAssignment(Expression):
  '''
  A C assignment expression (>>=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '>>='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitRightShiftAssignment(self)

class SubtractionAssignment(Expression):
  '''
  A C assignment expression (-=)
  The children are the lvalue and rvalue
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '-='
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitSubtractionAssignment(self)

class Addition(Expression):
  '''
  A C addition expression (+)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '+'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitAddition(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Addition.Addition()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class Address(Expression):
  '''
  A C address expression (&)
  The child is the expression whose address is returned
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '&'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitAddress(self)

class ArrayReference(Expression):
  '''
  A C array reference expression
  The child is the array expression
  '''
  def __init__(self):
    Expression.__init__(self)
    self.index = None
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitArrayReference(self)
  
  def getIndex(self):
    '''
    Retrieve the index
    '''
    return self.index
  
  def setIndex(self, index):
    '''
    Set the index
    '''
    self.index = index

class Cast(Expression):
  '''
  A C cast expression
  The child is the expression being cast
  '''
  def __init__(self):
    Expression.__init__(self)
    self.typeName = None
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitCast(self)
  
  def getTypeName(self):
    '''
    Retrieve the type being cast to
    '''
    return self.typeName
  
  def setTypeName(self, typeName):
    '''
    Set the type being cast to
    '''
    self.typeName = typeName

class Complement:
  '''
  A C ones complement expression (~)
  The child is the expression whose complement is returned
  '''
  def __init__(self):
    self.identifier = '~'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitComplement(self)
  
  def copy(self):
    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Complement.Complement()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class Constant(Expression):
  '''
  A C constant
  This is a primary expression
  '''
  def __init__(self):
    Expression.__init__(self)
    self.value = None
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitConstant(self)

class Division(Expression):
  '''
  A C division expression (/)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '/'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitDivision(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Division.Division()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class FunctionCall(Expression):
  '''
  A C function call expression
  The child is the function expression
  '''
  def __init__(self):
    Expression.__init__(self)
    self.arguments = []
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitFunctionCall(self)

class Increment(Expression):
  '''
  A C increment expression (++, --)
  The child is the expression to be incremented
  '''
  def __init__(self):
    Expression.__init__(self)
    self.isDecrement = 0
    self.isPostfix = 0
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitIncrement(self)
  
  def getDecrement(self):
    '''
    Retrieve the flag for a decrement
    '''
    return self.isDecrement
  
  def setDecrement(self, isDecrement):
    '''
    Set the flag for a decrement
    '''
    self.isDecrement = isDecrement
  
  def getPostfix(self):
    '''
    Retrieve the flag for a postincrement
    '''
    return self.isPostfix
  
  def setPostfix(self, isPostfix):
    '''
    Set the flag for a decrement
    '''
    self.isPostfix = isPostfix

class Indirection(Expression):
  '''
  A C indirection expression (*)
  The child is the expression whose data is returned
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '*'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitIndirection(self)

class LeftShift(Expression):
  '''
  A C left shift expression (<<)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '<<'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitLeftShift(self)

class Modulo(Expression):
  '''
  A C modulo expression (%)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '%'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitModulo(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Modulo.Modulo()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class Multiplication(Expression):
  '''
  A C multiplication expression (*)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '*'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitMultiplication(self)

class Negation(Expression):
  '''
  A C logical negation expression (!)
  The child is the expression whose negation is returned
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '!'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitNegation(self)

class RightShift(Expression):
  '''
  A C right shift expression (>>)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '>>'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitRightShift(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.RightShift.RightShift()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class Sizeof(Expression):
  '''
  A C sizeof expression (sizeof)
  The child is the optional expression whose size is calculated
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = 'sizeof'
    self.typeName = None
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitSizeof(self)
  
  def getTypeName(self):
    '''
    Retrieve the type whose size is calculated
    '''
    return self.typeName
  
  def setTypeName(self, typeName):
    '''
    Set the type whose size is calculated
    '''
    self.typeName = typeName

class StructReference(Expression):
  '''
  A C structure reference expression
  The child is the structure being referenced
  '''
  def __init__(self):
    Expression.__init__(self)
    self.member = None
    self.isPointer = 0
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitStructReference(self)
  
  def getMember(self):
    '''
    Retrieve the structure member which is being referenced
    '''
    return self.member
  
  def setMember(self, member):
    '''
    Set the structure member which is being referenced
    '''
    self.member = member
  
  def getPointer(self):
    '''
    Retrieve the flag indicating the reference to a structure pointer
    '''
    return self.isPointer
  
  def setPointer(self, isPointer):
    '''
    Set the flag indicating the reference to a structure pointer
    '''
    self.isPointer = isPointer

class Subtraction(Expression):
  '''
  A C subtraction expression (-)
  The children are the two subexpressions
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '-'
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitSubtraction(self)

class UnaryMinus(Expression):
  '''
  A C unary minus expression (-)
  The child is the expression whose negative is returned
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '-'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitUnaryMinus(self)

class UnaryPlus(Expression):
  '''
  A C unary plus expression (+)
  The child is the expression whose value is returned
  '''
  def __init__(self):
    Expression.__init__(self)
    self.identifier = '+'
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitUnaryPlus(self)

class Variable(Expression):
  '''
  A C variable reference
  This is a primary expression
  '''
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitVariable(self)

class Statement(CxxNode):
  '''
  A C statement
  If the statement is labeled, the label is set as the identifier unless it is a case expression
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.caseLabel = None
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitStatement(self)

class Break(Statement):
  '''
  A C break statement
  '''
  def __init__(self):
    Statement.__init__(self)
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitBreak(self)

class CompoundStatement(Statement):
  '''
  A (possibly empty) set of C declarations and statements delimited by braces
  The enclosed statements are children
  '''
  def __init__(self):
    Statement.__init__(self)
    self.declarations = []
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitCompoundStatement(self)
  
  def getDeclarations(self):
    '''
    Retrieve the superclasses of this class
    '''
    return self.declarations
  
  def setDeclarations(self, declarations):
    '''
    Set the superclasses of this class
    '''
    self.declarations = list(declarations)
  
  def addDeclarations(self, declarations):
    '''
    Add superclasses to this class
    '''
    self.declarations.extend(declarations)
  
  def removeDeclarations(self, declarations):
    '''
    Remove superclasses from this class
    Throws an exception if a parent does not exist
    '''
    for declaration in declarations:
      if declaration in self.declarations:
        self.declarations.remove(declaration)
      else:
        e = ASE.Compiler.CompilerException()
        e.setMessage('Declaration does not exist: ' + declaration.getIdentifier())
        raise e

class Continue(Statement):
  '''
  A C continue statement
  '''
  def __init__(self):
    Statement.__init__(self)
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitContinue(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Continue.Continue()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class ExpressionStatement(Statement):
  '''
  A C expression statement
  The child is the expression
  '''
  def __init__(self):
    Statement.__init__(self)
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitExpressionStatement(self)

class For(Statement):
  '''
  A C for statement
  The child is the iteration statement
  '''
  def __init__(self):
    Statement.__init__(self)
    self.initialization = None
    self.branch = None
    self.increment = None
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitFor(self)
  
  def getInitialization(self):
    '''
    Retrieve the initialization expression
    '''
    return self.initialization
  
  def setInitialization(self, initialization):
    '''
    Set the initialization expression
    '''
    self.initialization = initialization
  
  def getBranch(self):
    '''
    Retrieve the branch expression
    '''
    return self.branch
  
  def setBranch(self, branch):
    '''
    Set the branch expression
    '''
    self.branch = branch
  
  def getIncrement(self):
    '''
    Retrieve the increment expression
    '''
    return self.increment
  
  def setIncrement(self, increment):
    '''
    Set the increment expression
    '''
    self.increment = increment

class Goto(Statement):
  '''
  A C goto statement
  The identifier is the label
  '''
  def __init__(self):
    Statement.__init__(self)
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitGoto(self)

    '''
    Need to override this
    '''
    root = ASE.Compiler.Cxx.Goto.Goto()
    root.setIdentifier(self.getIdentifier())
    root.setAttributes(self.getAttributes())
    root.setLineNumber(self.getLineNumber())
    root.setComments(self.getComments())
    root.setChildren([child.copy() for child in self.getChildren()])
    return root

class If(Statement):
  '''
  A C if statement
  The children are the true and false branches
  '''
  def __init__(self):
    Statement.__init__(self)
    self.branch = None
    self.isTernary = 0
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitIf(self)
  
  def getBranch(self):
    '''
    Retrieve the branch expression
    '''
    return self.branch
  
  def setBranch(self, branch):
    '''
    Set the branch expression
    '''
    self.branch = branch
  
  def getTernary(self):
    '''
    Retrieve the flag for a ternary operator form
    '''
    return self.isTernary
  
  def setTernary(self, isTernary):
    '''
    Set the flag for a ternary operator form
    '''
    self.isTernary = isTernary

class Initializer(CxxNode):
  '''
  A C variable initializer
  The children are the initializer expressions
  If there are multiple children, this is a structure initializer
  '''
  def __init__(self):
    CxxNode.__init__(self)
    self.list = False
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitInitializer(self)

  def getList(self):
    return self.list
  def setList(self, l):
    self.list = l

class Return(Statement):
  '''
  A C return statement
  The child is the return value
  '''
  def __init__(self):
    Statement.__init__(self)
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitReturn(self)

class Throw(Statement):
  '''
  A C++ throw statement
  The child is the exception
  '''
  def __init__(self):
    Statement.__init__(self)
    return
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitThrow(self)

class Switch(Statement):
  '''
  A C switch statement
  The child is a compound statement of case statements
  '''
  def __init__(self):
    Statement.__init__(self)
    self.branch = None
    return

  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitSwitch(self)

class While(Statement):
  '''
  A C while statement
  The child is the iteration statement
  '''
  def __init__(self):
    Statement.__init__(self)
    self.branch = None
    self.isDoWhile = 0
  
  def accept(self, visitor):
    '''
    Overridden to dispatch correctly
    '''
    return visitor.visitWhile(self)
  
  def getBranch(self):
    '''
    Retrieve the branch expression
    '''
    return self.branch
  
  def setBranch(self, branch):
    '''
    Set the branch expression
    '''
    self.branch = branch
  
  def getDoWhile(self):
    '''
    Retrieve the flag for a do-while statement
    '''
    return self.isDoWhile
  
  def setDoWhile(self, isDoWhile):
    '''
    Set the flag for a do-while statement
    '''
    self.isDoWhile = isDoWhile
