# -*- coding: utf-8 -*-
"""
Munsell Renotation System Dataset - All Munsell Colours
=======================================================

Defines the *Munsell Renotation System* *All* datasets. Those are *all*
published *Munsell* colours, including the extrapolated colors.

Notes
-----
-   The Munsell Renotation data commonly available within the all.dat,
    experimental.dat and real.dat files features *CIE xyY* colourspace values
    that are scaled by a :math:`1 / 0.975 \\simeq 1.02568` factor. If you are
    performing conversions using *Munsell* *Colorlab* specification,
    e.g. *2.5R 9/2*, according to *ASTM D1535-08e1* method, you should not
    scale the output :math:`Y` Luminance. However, if you use directly the
    *CIE xyY* colourspace values from the Munsell Renotation data data, you
    should scale the :math:`Y` Luminance before conversions by a :math:`0.975`
    factor.

    *ASTM D1535-08e1* states that::

        The coefficients of this equation are obtained from the 1943 equation
        by multiplying each coefficient by 0.975, the reflectance factor of
        magnesium oxide with respect to the perfect reflecting diffuser, and
        rounding to ve digits of precision.

References
----------
-   :cite:`MunsellColorSciencec` : Munsell Color Science. (n.d.). Munsell
    Colours Data. Retrieved August 20, 2014, from
    http://www.cis.rit.edu/research/mcsl2/online/munsell.php
"""

from __future__ import division, unicode_literals

import numpy as np

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2020 - Colour Developers'
__license__ = 'New BSD License - https://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-developers@colour-science.org'
__status__ = 'Production'

__all__ = ['MUNSELL_COLOURS_ALL']

# yapf: disable
MUNSELL_COLOURS_ALL = (
    (('2.5GY', 0.2, 2.0), np.array([0.7130, 1.4140, 0.2370])),
    (('5GY', 0.2, 2.0), np.array([0.4490, 1.1450, 0.2370])),
    (('7.5GY', 0.2, 2.0), np.array([0.2620, 0.8370, 0.2370])),
    (('7.5GY', 0.2, 4.0), np.array([-0.0780, 2.1600, 0.2370])),
    (('10GY', 0.2, 2.0), np.array([0.1850, 0.6760, 0.2370])),
    (('10GY', 0.2, 4.0), np.array([-0.2570, 1.2330, 0.2370])),
    (('2.5G', 0.2, 2.0), np.array([0.1440, 0.5840, 0.2370])),
    (('2.5G', 0.2, 4.0), np.array([-0.2350, 0.8910, 0.2370])),
    (('5G', 0.2, 2.0), np.array([0.1170, 0.5160, 0.2370])),
    (('5G', 0.2, 4.0), np.array([-0.2090, 0.7190, 0.2370])),
    (('7.5G', 0.2, 2.0), np.array([0.0970, 0.4580, 0.2370])),
    (('7.5G', 0.2, 4.0), np.array([-0.1810, 0.5750, 0.2370])),
    (('10G', 0.2, 2.0), np.array([0.0800, 0.3970, 0.2370])),
    (('10G', 0.2, 4.0), np.array([-0.1370, 0.4250, 0.2370])),
    (('2.5BG', 0.2, 2.0), np.array([0.0680, 0.3320, 0.2370])),
    (('2.5BG', 0.2, 4.0), np.array([-0.1000, 0.3250, 0.2370])),
    (('5BG', 0.2, 2.0), np.array([0.0660, 0.2610, 0.2370])),
    (('5BG', 0.2, 4.0), np.array([-0.0480, 0.2230, 0.2370])),
    (('7.5BG', 0.2, 2.0), np.array([0.0720, 0.2260, 0.2370])),
    (('7.5BG', 0.2, 4.0), np.array([-0.0160, 0.1810, 0.2370])),
    (('10BG', 0.2, 2.0), np.array([0.0850, 0.1950, 0.2370])),
    (('10BG', 0.2, 4.0), np.array([0.0150, 0.1480, 0.2370])),
    (('2.5B', 0.2, 2.0), np.array([0.0970, 0.1770, 0.2370])),
    (('2.5B', 0.2, 4.0), np.array([0.0390, 0.1280, 0.2370])),
    (('5B', 0.2, 2.0), np.array([0.1110, 0.1640, 0.2370])),
    (('5B', 0.2, 4.0), np.array([0.0630, 0.1140, 0.2370])),
    (('7.5B', 0.2, 2.0), np.array([0.1210, 0.1570, 0.2370])),
    (('7.5B', 0.2, 4.0), np.array([0.0740, 0.1090, 0.2370])),
    (('10B', 0.2, 2.0), np.array([0.1330, 0.1490, 0.2370])),
    (('10B', 0.2, 4.0), np.array([0.0880, 0.1020, 0.2370])),
    (('10B', 0.2, 6.0), np.array([0.0630, 0.0740, 0.2370])),
    (('2.5PB', 0.2, 2.0), np.array([0.1470, 0.1430, 0.2370])),
    (('2.5PB', 0.2, 4.0), np.array([0.1090, 0.0940, 0.2370])),
    (('2.5PB', 0.2, 6.0), np.array([0.0870, 0.0670, 0.2370])),
    (('5PB', 0.2, 2.0), np.array([0.1650, 0.1360, 0.2370])),
    (('5PB', 0.2, 4.0), np.array([0.1330, 0.0900, 0.2370])),
    (('5PB', 0.2, 6.0), np.array([0.1150, 0.0620, 0.2370])),
    (('5PB', 0.2, 8.0), np.array([0.1040, 0.0450, 0.2370])),
    (('5PB', 0.2, 10.0), np.array([0.0960, 0.0320, 0.2370])),
    (('5PB', 0.2, 12.0), np.array([0.0860, 0.0180, 0.2370])),
    (('7.5PB', 0.2, 2.0), np.array([0.1920, 0.1300, 0.2370])),
    (('7.5PB', 0.2, 4.0), np.array([0.1710, 0.0870, 0.2370])),
    (('7.5PB', 0.2, 6.0), np.array([0.1590, 0.0610, 0.2370])),
    (('7.5PB', 0.2, 8.0), np.array([0.1520, 0.0450, 0.2370])),
    (('7.5PB', 0.2, 10.0), np.array([0.1480, 0.0340, 0.2370])),
    (('7.5PB', 0.2, 12.0), np.array([0.1450, 0.0250, 0.2370])),
    (('7.5PB', 0.2, 14.0), np.array([0.1420, 0.0180, 0.2370])),
    (('7.5PB', 0.2, 16.0), np.array([0.1390, 0.0120, 0.2370])),
    (('7.5PB', 0.2, 18.0), np.array([0.1370, 0.0060, 0.2370])),
    (('7.5PB', 0.2, 20.0), np.array([0.1360, 0.0000, 0.2370])),
    (('10PB', 0.2, 2.0), np.array([0.2270, 0.1260, 0.2370])),
    (('10PB', 0.2, 4.0), np.array([0.2130, 0.0880, 0.2370])),
    (('10PB', 0.2, 6.0), np.array([0.2060, 0.0640, 0.2370])),
    (('10PB', 0.2, 8.0), np.array([0.2020, 0.0490, 0.2370])),
    (('10PB', 0.2, 10.0), np.array([0.2000, 0.0400, 0.2370])),
    (('10PB', 0.2, 12.0), np.array([0.1980, 0.0310, 0.2370])),
    (('10PB', 0.2, 14.0), np.array([0.1960, 0.0240, 0.2370])),
    (('10PB', 0.2, 16.0), np.array([0.1950, 0.0180, 0.2370])),
    (('10PB', 0.2, 18.0), np.array([0.1930, 0.0120, 0.2370])),
    (('10PB', 0.2, 20.0), np.array([0.1920, 0.0060, 0.2370])),
    (('2.5P', 0.2, 2.0), np.array([0.2500, 0.1270, 0.2370])),
    (('2.5P', 0.2, 4.0), np.array([0.2410, 0.0900, 0.2370])),
    (('2.5P', 0.2, 6.0), np.array([0.2360, 0.0670, 0.2370])),
    (('2.5P', 0.2, 8.0), np.array([0.2320, 0.0520, 0.2370])),
    (('2.5P', 0.2, 10.0), np.array([0.2310, 0.0430, 0.2370])),
    (('2.5P', 0.2, 12.0), np.array([0.2290, 0.0350, 0.2370])),
    (('2.5P', 0.2, 14.0), np.array([0.2280, 0.0280, 0.2370])),
    (('2.5P', 0.2, 16.0), np.array([0.2270, 0.0220, 0.2370])),
    (('2.5P', 0.2, 18.0), np.array([0.2260, 0.0160, 0.2370])),
    (('5P', 0.2, 2.0), np.array([0.2750, 0.1290, 0.2370])),
    (('5P', 0.2, 4.0), np.array([0.2690, 0.0930, 0.2370])),
    (('5P', 0.2, 6.0), np.array([0.2660, 0.0720, 0.2370])),
    (('5P', 0.2, 8.0), np.array([0.2640, 0.0560, 0.2370])),
    (('5P', 0.2, 10.0), np.array([0.2630, 0.0470, 0.2370])),
    (('5P', 0.2, 12.0), np.array([0.2630, 0.0390, 0.2370])),
    (('5P', 0.2, 14.0), np.array([0.2620, 0.0320, 0.2370])),
    (('7.5P', 0.2, 2.0), np.array([0.2880, 0.1310, 0.2370])),
    (('7.5P', 0.2, 4.0), np.array([0.2830, 0.0940, 0.2370])),
    (('7.5P', 0.2, 6.0), np.array([0.2800, 0.0740, 0.2370])),
    (('7.5P', 0.2, 8.0), np.array([0.2770, 0.0580, 0.2370])),
    (('7.5P', 0.2, 10.0), np.array([0.2760, 0.0490, 0.2370])),
    (('7.5P', 0.2, 12.0), np.array([0.2750, 0.0400, 0.2370])),
    (('10P', 0.2, 2.0), np.array([0.3000, 0.1340, 0.2370])),
    (('10P', 0.2, 4.0), np.array([0.2960, 0.0970, 0.2370])),
    (('10P', 0.2, 6.0), np.array([0.2930, 0.0750, 0.2370])),
    (('10P', 0.2, 8.0), np.array([0.2910, 0.0600, 0.2370])),
    (('10P', 0.2, 10.0), np.array([0.2900, 0.0510, 0.2370])),
    (('2.5RP', 0.2, 2.0), np.array([0.3150, 0.1370, 0.2370])),
    (('2.5RP', 0.2, 4.0), np.array([0.3130, 0.1000, 0.2370])),
    (('2.5RP', 0.2, 6.0), np.array([0.3120, 0.0780, 0.2370])),
    (('2.5RP', 0.2, 8.0), np.array([0.3100, 0.0640, 0.2370])),
    (('2.5RP', 0.2, 10.0), np.array([0.3100, 0.0530, 0.2370])),
    (('5RP', 0.2, 2.0), np.array([0.3370, 0.1430, 0.2370])),
    (('5RP', 0.2, 4.0), np.array([0.3410, 0.1060, 0.2370])),
    (('5RP', 0.2, 6.0), np.array([0.3420, 0.0840, 0.2370])),
    (('5RP', 0.2, 8.0), np.array([0.3420, 0.0690, 0.2370])),
    (('7.5RP', 0.2, 2.0), np.array([0.3700, 0.1520, 0.2370])),
    (('7.5RP', 0.2, 4.0), np.array([0.3810, 0.1150, 0.2370])),
    (('7.5RP', 0.2, 6.0), np.array([0.3880, 0.0930, 0.2370])),
    (('7.5RP', 0.2, 8.0), np.array([0.3930, 0.0770, 0.2370])),
    (('10RP', 0.2, 2.0), np.array([0.4040, 0.1640, 0.2370])),
    (('10RP', 0.2, 4.0), np.array([0.4240, 0.1250, 0.2370])),
    (('10RP', 0.2, 6.0), np.array([0.4350, 0.1030, 0.2370])),
    (('2.5R', 0.2, 2.0), np.array([0.4510, 0.1830, 0.2370])),
    (('2.5R', 0.2, 4.0), np.array([0.4880, 0.1420, 0.2370])),
    (('2.5R', 0.2, 6.0), np.array([0.5100, 0.1190, 0.2370])),
    (('5R', 0.2, 2.0), np.array([0.5010, 0.2040, 0.2370])),
    (('5R', 0.2, 4.0), np.array([0.5560, 0.1590, 0.2370])),
    (('7.5R', 0.2, 2.0), np.array([0.5430, 0.2240, 0.2370])),
    (('7.5R', 0.2, 4.0), np.array([0.6260, 0.1780, 0.2370])),
    (('10R', 0.2, 2.0), np.array([0.5920, 0.2460, 0.2370])),
    (('10R', 0.2, 4.0), np.array([0.6960, 0.1980, 0.2370])),
    (('2.5YR', 0.2, 2.0), np.array([0.6790, 0.2900, 0.2370])),
    (('2.5YR', 0.2, 4.0), np.array([0.8430, 0.2450, 0.2370])),
    (('5YR', 0.2, 2.0), np.array([0.8370, 0.3750, 0.2370])),
    (('5YR', 0.2, 4.0), np.array([1.1740, 0.3550, 0.2370])),
    (('7.5YR', 0.2, 2.0), np.array([1.0000, 0.4800, 0.2370])),
    (('10YR', 0.2, 2.0), np.array([1.2900, 0.7400, 0.2370])),
    (('2.5Y', 0.2, 2.0), np.array([1.4300, 0.9700, 0.2370])),
    (('5Y', 0.2, 2.0), np.array([1.4950, 1.2840, 0.2370])),
    (('7.5Y', 0.2, 2.0), np.array([1.4340, 1.4590, 0.2370])),
    (('10Y', 0.4, 2.0), np.array([0.5420, 0.6700, 0.4670])),
    (('2.5GY', 0.4, 2.0), np.array([0.4230, 0.5900, 0.4670])),
    (('2.5GY', 0.4, 4.0), np.array([0.7350, 1.8200, 0.4670])),
    (('5GY', 0.4, 2.0), np.array([0.3580, 0.5280, 0.4670])),
    (('5GY', 0.4, 4.0), np.array([0.3230, 1.4670, 0.4670])),
    (('7.5GY', 0.4, 2.0), np.array([0.3120, 0.4820, 0.4670])),
    (('7.5GY', 0.4, 4.0), np.array([0.1280, 1.1410, 0.4670])),
    (('10GY', 0.4, 2.0), np.array([0.2770, 0.4450, 0.4670])),
    (('10GY', 0.4, 4.0), np.array([0.0410, 0.9060, 0.4670])),
    (('2.5G', 0.4, 2.0), np.array([0.2580, 0.4230, 0.4670])),
    (('2.5G', 0.4, 4.0), np.array([0.0050, 0.7430, 0.4670])),
    (('5G', 0.4, 2.0), np.array([0.2390, 0.3990, 0.4670])),
    (('5G', 0.4, 4.0), np.array([-0.0110, 0.6040, 0.4670])),
    (('7.5G', 0.4, 2.0), np.array([0.2260, 0.3800, 0.4670])),
    (('7.5G', 0.4, 4.0), np.array([-0.0110, 0.5010, 0.4670])),
    (('10G', 0.4, 2.0), np.array([0.2130, 0.3610, 0.4670])),
    (('10G', 0.4, 4.0), np.array([-0.0040, 0.4050, 0.4670])),
    (('2.5BG', 0.4, 2.0), np.array([0.1960, 0.3320, 0.4670])),
    (('2.5BG', 0.4, 4.0), np.array([0.0080, 0.3340, 0.4670])),
    (('5BG', 0.4, 2.0), np.array([0.1800, 0.2980, 0.4670])),
    (('5BG', 0.4, 4.0), np.array([0.0310, 0.2590, 0.4670])),
    (('7.5BG', 0.4, 2.0), np.array([0.1730, 0.2750, 0.4670])),
    (('7.5BG', 0.4, 4.0), np.array([0.0510, 0.2200, 0.4670])),
    (('7.5BG', 0.4, 6.0), np.array([-0.0050, 0.1930, 0.4670])),
    (('10BG', 0.4, 2.0), np.array([0.1690, 0.2490, 0.4670])),
    (('10BG', 0.4, 4.0), np.array([0.0740, 0.1870, 0.4670])),
    (('10BG', 0.4, 6.0), np.array([0.0320, 0.1490, 0.4670])),
    (('2.5B', 0.4, 2.0), np.array([0.1690, 0.2360, 0.4670])),
    (('2.5B', 0.4, 4.0), np.array([0.0870, 0.1720, 0.4670])),
    (('2.5B', 0.4, 6.0), np.array([0.0480, 0.1340, 0.4670])),
    (('5B', 0.4, 2.0), np.array([0.1720, 0.2230, 0.4670])),
    (('5B', 0.4, 4.0), np.array([0.1020, 0.1590, 0.4670])),
    (('5B', 0.4, 6.0), np.array([0.0670, 0.1190, 0.4670])),
    (('7.5B', 0.4, 2.0), np.array([0.1760, 0.2130, 0.4670])),
    (('7.5B', 0.4, 4.0), np.array([0.1130, 0.1510, 0.4670])),
    (('7.5B', 0.4, 6.0), np.array([0.0810, 0.1110, 0.4670])),
    (('10B', 0.4, 2.0), np.array([0.1830, 0.2030, 0.4670])),
    (('10B', 0.4, 4.0), np.array([0.1260, 0.1450, 0.4670])),
    (('10B', 0.4, 6.0), np.array([0.0930, 0.1050, 0.4670])),
    (('10B', 0.4, 8.0), np.array([0.0670, 0.0670, 0.4670])),
    (('2.5PB', 0.4, 2.0), np.array([0.1900, 0.1960, 0.4670])),
    (('2.5PB', 0.4, 4.0), np.array([0.1410, 0.1390, 0.4670])),
    (('2.5PB', 0.4, 6.0), np.array([0.1130, 0.0980, 0.4670])),
    (('2.5PB', 0.4, 8.0), np.array([0.0930, 0.0650, 0.4670])),
    (('5PB', 0.4, 2.0), np.array([0.2020, 0.1880, 0.4670])),
    (('5PB', 0.4, 4.0), np.array([0.1610, 0.1340, 0.4670])),
    (('5PB', 0.4, 6.0), np.array([0.1350, 0.0950, 0.4670])),
    (('5PB', 0.4, 8.0), np.array([0.1160, 0.0660, 0.4670])),
    (('5PB', 0.4, 10.0), np.array([0.1070, 0.0530, 0.4670])),
    (('5PB', 0.4, 12.0), np.array([0.0990, 0.0420, 0.4670])),
    (('5PB', 0.4, 14.0), np.array([0.0960, 0.0360, 0.4670])),
    (('5PB', 0.4, 16.0), np.array([0.0940, 0.0300, 0.4670])),
    (('5PB', 0.4, 18.0), np.array([0.0910, 0.0250, 0.4670])),
    (('5PB', 0.4, 20.0), np.array([0.0860, 0.0180, 0.4670])),
    (('7.5PB', 0.4, 2.0), np.array([0.2200, 0.1800, 0.4670])),
    (('7.5PB', 0.4, 4.0), np.array([0.1920, 0.1300, 0.4670])),
    (('7.5PB', 0.4, 6.0), np.array([0.1750, 0.0950, 0.4670])),
    (('7.5PB', 0.4, 8.0), np.array([0.1650, 0.0720, 0.4670])),
    (('7.5PB', 0.4, 10.0), np.array([0.1600, 0.0580, 0.4670])),
    (('7.5PB', 0.4, 12.0), np.array([0.1550, 0.0460, 0.4670])),
    (('7.5PB', 0.4, 14.0), np.array([0.1520, 0.0400, 0.4670])),
    (('7.5PB', 0.4, 16.0), np.array([0.1500, 0.0320, 0.4670])),
    (('7.5PB', 0.4, 18.0), np.array([0.1470, 0.0260, 0.4670])),
    (('7.5PB', 0.4, 20.0), np.array([0.1450, 0.0200, 0.4670])),
    (('7.5PB', 0.4, 22.0), np.array([0.1430, 0.0160, 0.4670])),
    (('7.5PB', 0.4, 24.0), np.array([0.1410, 0.0100, 0.4670])),
    (('7.5PB', 0.4, 26.0), np.array([0.1390, 0.0050, 0.4670])),
    (('7.5PB', 0.4, 28.0), np.array([0.1370, 0.0010, 0.4670])),
    (('10PB', 0.4, 2.0), np.array([0.2410, 0.1760, 0.4670])),
    (('10PB', 0.4, 4.0), np.array([0.2230, 0.1310, 0.4670])),
    (('10PB', 0.4, 6.0), np.array([0.2120, 0.1000, 0.4670])),
    (('10PB', 0.4, 8.0), np.array([0.2060, 0.0780, 0.4670])),
    (('10PB', 0.4, 10.0), np.array([0.2020, 0.0620, 0.4670])),
    (('10PB', 0.4, 12.0), np.array([0.1990, 0.0500, 0.4670])),
    (('10PB', 0.4, 14.0), np.array([0.1970, 0.0430, 0.4670])),
    (('10PB', 0.4, 16.0), np.array([0.1950, 0.0360, 0.4670])),
    (('10PB', 0.4, 18.0), np.array([0.1930, 0.0290, 0.4670])),
    (('10PB', 0.4, 20.0), np.array([0.1920, 0.0240, 0.4670])),
    (('10PB', 0.4, 22.0), np.array([0.1910, 0.0180, 0.4670])),
    (('10PB', 0.4, 24.0), np.array([0.1900, 0.0130, 0.4670])),
    (('10PB', 0.4, 26.0), np.array([0.1880, 0.0080, 0.4670])),
    (('10PB', 0.4, 28.0), np.array([0.1870, 0.0040, 0.4670])),
    (('2.5P', 0.4, 2.0), np.array([0.2590, 0.1770, 0.4670])),
    (('2.5P', 0.4, 4.0), np.array([0.2460, 0.1340, 0.4670])),
    (('2.5P', 0.4, 6.0), np.array([0.2380, 0.1040, 0.4670])),
    (('2.5P', 0.4, 8.0), np.array([0.2330, 0.0820, 0.4670])),
    (('2.5P', 0.4, 10.0), np.array([0.2300, 0.0640, 0.4670])),
    (('2.5P', 0.4, 12.0), np.array([0.2270, 0.0540, 0.4670])),
    (('2.5P', 0.4, 14.0), np.array([0.2260, 0.0450, 0.4670])),
    (('2.5P', 0.4, 16.0), np.array([0.2240, 0.0380, 0.4670])),
    (('2.5P', 0.4, 18.0), np.array([0.2230, 0.0310, 0.4670])),
    (('2.5P', 0.4, 20.0), np.array([0.2220, 0.0260, 0.4670])),
    (('2.5P', 0.4, 22.0), np.array([0.2200, 0.0200, 0.4670])),
    (('2.5P', 0.4, 24.0), np.array([0.2190, 0.0160, 0.4670])),
    (('2.5P', 0.4, 26.0), np.array([0.2180, 0.0100, 0.4670])),
    (('5P', 0.4, 2.0), np.array([0.2810, 0.1820, 0.4670])),
    (('5P', 0.4, 4.0), np.array([0.2720, 0.1380, 0.4670])),
    (('5P', 0.4, 6.0), np.array([0.2650, 0.1100, 0.4670])),
    (('5P', 0.4, 8.0), np.array([0.2600, 0.0870, 0.4670])),
    (('5P', 0.4, 10.0), np.array([0.2570, 0.0690, 0.4670])),
    (('5P', 0.4, 12.0), np.array([0.2530, 0.0570, 0.4670])),
    (('5P', 0.4, 14.0), np.array([0.2520, 0.0480, 0.4670])),
    (('5P', 0.4, 16.0), np.array([0.2500, 0.0400, 0.4670])),
    (('5P', 0.4, 18.0), np.array([0.2480, 0.0340, 0.4670])),
    (('5P', 0.4, 20.0), np.array([0.2470, 0.0280, 0.4670])),
    (('7.5P', 0.4, 2.0), np.array([0.2960, 0.1850, 0.4670])),
    (('7.5P', 0.4, 4.0), np.array([0.2890, 0.1420, 0.4670])),
    (('7.5P', 0.4, 6.0), np.array([0.2840, 0.1140, 0.4670])),
    (('7.5P', 0.4, 8.0), np.array([0.2800, 0.0910, 0.4670])),
    (('7.5P', 0.4, 10.0), np.array([0.2760, 0.0720, 0.4670])),
    (('7.5P', 0.4, 12.0), np.array([0.2730, 0.0590, 0.4670])),
    (('7.5P', 0.4, 14.0), np.array([0.2720, 0.0490, 0.4670])),
    (('7.5P', 0.4, 16.0), np.array([0.2700, 0.0420, 0.4670])),
    (('7.5P', 0.4, 18.0), np.array([0.2690, 0.0340, 0.4670])),
    (('10P', 0.4, 2.0), np.array([0.3090, 0.1890, 0.4670])),
    (('10P', 0.4, 4.0), np.array([0.3040, 0.1460, 0.4670])),
    (('10P', 0.4, 6.0), np.array([0.3020, 0.1190, 0.4670])),
    (('10P', 0.4, 8.0), np.array([0.2980, 0.0950, 0.4670])),
    (('10P', 0.4, 10.0), np.array([0.2960, 0.0730, 0.4670])),
    (('10P', 0.4, 12.0), np.array([0.2940, 0.0610, 0.4670])),
    (('10P', 0.4, 14.0), np.array([0.2930, 0.0520, 0.4670])),
    (('10P', 0.4, 16.0), np.array([0.2910, 0.0430, 0.4670])),
    (('2.5RP', 0.4, 2.0), np.array([0.3200, 0.1930, 0.4670])),
    (('2.5RP', 0.4, 4.0), np.array([0.3200, 0.1510, 0.4670])),
    (('2.5RP', 0.4, 6.0), np.array([0.3200, 0.1230, 0.4670])),
    (('2.5RP', 0.4, 8.0), np.array([0.3200, 0.1000, 0.4670])),
    (('2.5RP', 0.4, 10.0), np.array([0.3200, 0.0780, 0.4670])),
    (('2.5RP', 0.4, 12.0), np.array([0.3200, 0.0650, 0.4670])),
    (('2.5RP', 0.4, 14.0), np.array([0.3200, 0.0540, 0.4670])),
    (('5RP', 0.4, 2.0), np.array([0.3370, 0.1990, 0.4670])),
    (('5RP', 0.4, 4.0), np.array([0.3440, 0.1580, 0.4670])),
    (('5RP', 0.4, 6.0), np.array([0.3480, 0.1310, 0.4670])),
    (('5RP', 0.4, 8.0), np.array([0.3500, 0.1060, 0.4670])),
    (('5RP', 0.4, 10.0), np.array([0.3530, 0.0820, 0.4670])),
    (('5RP', 0.4, 12.0), np.array([0.3530, 0.0700, 0.4670])),
    (('7.5RP', 0.4, 2.0), np.array([0.3600, 0.2090, 0.4670])),
    (('7.5RP', 0.4, 4.0), np.array([0.3740, 0.1690, 0.4670])),
    (('7.5RP', 0.4, 6.0), np.array([0.3840, 0.1410, 0.4670])),
    (('7.5RP', 0.4, 8.0), np.array([0.3910, 0.1170, 0.4670])),
    (('7.5RP', 0.4, 10.0), np.array([0.3980, 0.0900, 0.4670])),
    (('10RP', 0.4, 2.0), np.array([0.3810, 0.2200, 0.4670])),
    (('10RP', 0.4, 4.0), np.array([0.4060, 0.1810, 0.4670])),
    (('10RP', 0.4, 6.0), np.array([0.4230, 0.1530, 0.4670])),
    (('10RP', 0.4, 8.0), np.array([0.4370, 0.1280, 0.4670])),
    (('10RP', 0.4, 10.0), np.array([0.4540, 0.0980, 0.4670])),
    (('2.5R', 0.4, 2.0), np.array([0.4110, 0.2360, 0.4670])),
    (('2.5R', 0.4, 4.0), np.array([0.4500, 0.1980, 0.4670])),
    (('2.5R', 0.4, 6.0), np.array([0.4770, 0.1700, 0.4670])),
    (('2.5R', 0.4, 8.0), np.array([0.5010, 0.1430, 0.4670])),
    (('2.5R', 0.4, 10.0), np.array([0.5290, 0.1130, 0.4670])),
    (('5R', 0.4, 2.0), np.array([0.4410, 0.2550, 0.4670])),
    (('5R', 0.4, 4.0), np.array([0.4980, 0.2190, 0.4670])),
    (('5R', 0.4, 6.0), np.array([0.5370, 0.1900, 0.4670])),
    (('5R', 0.4, 8.0), np.array([0.5750, 0.1610, 0.4670])),
    (('7.5R', 0.4, 2.0), np.array([0.4660, 0.2720, 0.4670])),
    (('7.5R', 0.4, 4.0), np.array([0.5390, 0.2380, 0.4670])),
    (('7.5R', 0.4, 6.0), np.array([0.5880, 0.2080, 0.4670])),
    (('7.5R', 0.4, 8.0), np.array([0.6350, 0.1760, 0.4670])),
    (('10R', 0.4, 2.0), np.array([0.4900, 0.2890, 0.4670])),
    (('10R', 0.4, 4.0), np.array([0.5820, 0.2580, 0.4670])),
    (('10R', 0.4, 6.0), np.array([0.6490, 0.2290, 0.4670])),
    (('10R', 0.4, 8.0), np.array([0.7060, 0.1960, 0.4670])),
    (('2.5YR', 0.4, 2.0), np.array([0.5340, 0.3240, 0.4670])),
    (('2.5YR', 0.4, 4.0), np.array([0.6650, 0.2980, 0.4670])),
    (('2.5YR', 0.4, 6.0), np.array([0.7550, 0.2700, 0.4670])),
    (('2.5YR', 0.4, 8.0), np.array([0.8440, 0.2410, 0.4670])),
    (('5YR', 0.4, 2.0), np.array([0.5850, 0.3670, 0.4670])),
    (('5YR', 0.4, 4.0), np.array([0.7750, 0.3620, 0.4670])),
    (('5YR', 0.4, 6.0), np.array([0.8890, 0.3440, 0.4670])),
    (('7.5YR', 0.4, 2.0), np.array([0.6380, 0.4200, 0.4670])),
    (('7.5YR', 0.4, 4.0), np.array([0.9080, 0.4520, 0.4670])),
    (('10YR', 0.4, 2.0), np.array([0.6980, 0.4990, 0.4670])),
    (('2.5Y', 0.4, 2.0), np.array([0.7290, 0.5880, 0.4670])),
    (('5Y', 0.4, 2.0), np.array([0.7210, 0.6560, 0.4670])),
    (('7.5Y', 0.4, 2.0), np.array([0.6590, 0.7000, 0.4670])),
    (('10Y', 0.6, 2.0), np.array([0.4320, 0.5010, 0.6990])),
    (('10Y', 0.6, 4.0), np.array([0.9020, 1.2230, 0.6990])),
    (('2.5GY', 0.6, 2.0), np.array([0.3770, 0.4680, 0.6990])),
    (('2.5GY', 0.6, 4.0), np.array([0.6030, 1.1690, 0.6990])),
    (('5GY', 0.6, 2.0), np.array([0.3420, 0.4420, 0.6990])),
    (('5GY', 0.6, 4.0), np.array([0.4100, 0.9950, 0.6990])),
    (('7.5GY', 0.6, 2.0), np.array([0.3150, 0.4200, 0.6990])),
    (('7.5GY', 0.6, 4.0), np.array([0.2740, 0.7920, 0.6990])),
    (('7.5GY', 0.6, 6.0), np.array([-0.0450, 1.4590, 0.6990])),
    (('10GY', 0.6, 2.0), np.array([0.2920, 0.3990, 0.6990])),
    (('10GY', 0.6, 4.0), np.array([0.2080, 0.6520, 0.6990])),
    (('10GY', 0.6, 6.0), np.array([-0.1280, 1.1250, 0.6990])),
    (('2.5G', 0.6, 2.0), np.array([0.2810, 0.3880, 0.6990])),
    (('2.5G', 0.6, 4.0), np.array([0.1750, 0.5610, 0.6990])),
    (('2.5G', 0.6, 6.0), np.array([-0.1430, 0.8890, 0.6990])),
    (('5G', 0.6, 2.0), np.array([0.2700, 0.3760, 0.6990])),
    (('5G', 0.6, 4.0), np.array([0.1520, 0.4930, 0.6990])),
    (('5G', 0.6, 6.0), np.array([-0.1280, 0.6820, 0.6990])),
    (('7.5G', 0.6, 2.0), np.array([0.2590, 0.3630, 0.6990])),
    (('7.5G', 0.6, 4.0), np.array([0.1370, 0.4400, 0.6990])),
    (('7.5G', 0.6, 6.0), np.array([-0.1020, 0.5410, 0.6990])),
    (('10G', 0.6, 2.0), np.array([0.2470, 0.3490, 0.6990])),
    (('10G', 0.6, 4.0), np.array([0.1240, 0.3890, 0.6990])),
    (('10G', 0.6, 6.0), np.array([-0.0680, 0.4250, 0.6990])),
    (('2.5BG', 0.6, 2.0), np.array([0.2360, 0.3340, 0.6990])),
    (('2.5BG', 0.6, 4.0), np.array([0.1170, 0.3410, 0.6990])),
    (('2.5BG', 0.6, 6.0), np.array([-0.0360, 0.3340, 0.6990])),
    (('5BG', 0.6, 2.0), np.array([0.2210, 0.3110, 0.6990])),
    (('5BG', 0.6, 4.0), np.array([0.1120, 0.2840, 0.6990])),
    (('5BG', 0.6, 6.0), np.array([0.0090, 0.2430, 0.6990])),
    (('7.5BG', 0.6, 2.0), np.array([0.2130, 0.2950, 0.6990])),
    (('7.5BG', 0.6, 4.0), np.array([0.1130, 0.2540, 0.6990])),
    (('7.5BG', 0.6, 6.0), np.array([0.0300, 0.2110, 0.6990])),
    (('10BG', 0.6, 2.0), np.array([0.2060, 0.2760, 0.6990])),
    (('10BG', 0.6, 4.0), np.array([0.1160, 0.2210, 0.6990])),
    (('10BG', 0.6, 6.0), np.array([0.0520, 0.1760, 0.6990])),
    (('2.5B', 0.6, 2.0), np.array([0.2020, 0.2600, 0.6990])),
    (('2.5B', 0.6, 4.0), np.array([0.1230, 0.2020, 0.6990])),
    (('2.5B', 0.6, 6.0), np.array([0.0710, 0.1570, 0.6990])),
    (('2.5B', 0.6, 8.0), np.array([0.0170, 0.1110, 0.6990])),
    (('5B', 0.6, 2.0), np.array([0.2020, 0.2450, 0.6990])),
    (('5B', 0.6, 4.0), np.array([0.1340, 0.1870, 0.6990])),
    (('5B', 0.6, 6.0), np.array([0.0880, 0.1450, 0.6990])),
    (('5B', 0.6, 8.0), np.array([0.0440, 0.1000, 0.6990])),
    (('7.5B', 0.6, 2.0), np.array([0.2040, 0.2350, 0.6990])),
    (('7.5B', 0.6, 4.0), np.array([0.1430, 0.1780, 0.6990])),
    (('7.5B', 0.6, 6.0), np.array([0.0990, 0.1360, 0.6990])),
    (('7.5B', 0.6, 8.0), np.array([0.0600, 0.0960, 0.6990])),
    (('10B', 0.6, 2.0), np.array([0.2090, 0.2270, 0.6990])),
    (('10B', 0.6, 4.0), np.array([0.1530, 0.1720, 0.6990])),
    (('10B', 0.6, 6.0), np.array([0.1150, 0.1280, 0.6990])),
    (('10B', 0.6, 8.0), np.array([0.0840, 0.0910, 0.6990])),
    (('2.5PB', 0.6, 2.0), np.array([0.2150, 0.2210, 0.6990])),
    (('2.5PB', 0.6, 4.0), np.array([0.1660, 0.1650, 0.6990])),
    (('2.5PB', 0.6, 6.0), np.array([0.1310, 0.1220, 0.6990])),
    (('2.5PB', 0.6, 8.0), np.array([0.1060, 0.0900, 0.6990])),
    (('2.5PB', 0.6, 10.0), np.array([0.0890, 0.0680, 0.6990])),
    (('2.5PB', 0.6, 12.0), np.array([0.0790, 0.0550, 0.6990])),
    (('5PB', 0.6, 2.0), np.array([0.2230, 0.2150, 0.6990])),
    (('5PB', 0.6, 4.0), np.array([0.1820, 0.1600, 0.6990])),
    (('5PB', 0.6, 6.0), np.array([0.1520, 0.1180, 0.6990])),
    (('5PB', 0.6, 8.0), np.array([0.1310, 0.0880, 0.6990])),
    (('5PB', 0.6, 10.0), np.array([0.1180, 0.0690, 0.6990])),
    (('5PB', 0.6, 12.0), np.array([0.1100, 0.0570, 0.6990])),
    (('5PB', 0.6, 14.0), np.array([0.1040, 0.0480, 0.6990])),
    (('5PB', 0.6, 16.0), np.array([0.0990, 0.0400, 0.6990])),
    (('5PB', 0.6, 18.0), np.array([0.0930, 0.0320, 0.6990])),
    (('5PB', 0.6, 20.0), np.array([0.0880, 0.0240, 0.6990])),
    (('5PB', 0.6, 22.0), np.array([0.0840, 0.0180, 0.6990])),
    (('7.5PB', 0.6, 2.0), np.array([0.2390, 0.2080, 0.6990])),
    (('7.5PB', 0.6, 4.0), np.array([0.2080, 0.1550, 0.6990])),
    (('7.5PB', 0.6, 6.0), np.array([0.1880, 0.1170, 0.6990])),
    (('7.5PB', 0.6, 8.0), np.array([0.1760, 0.0920, 0.6990])),
    (('7.5PB', 0.6, 10.0), np.array([0.1680, 0.0740, 0.6990])),
    (('7.5PB', 0.6, 12.0), np.array([0.1630, 0.0620, 0.6990])),
    (('7.5PB', 0.6, 14.0), np.array([0.1590, 0.0530, 0.6990])),
    (('7.5PB', 0.6, 16.0), np.array([0.1560, 0.0450, 0.6990])),
    (('7.5PB', 0.6, 18.0), np.array([0.1530, 0.0370, 0.6990])),
    (('7.5PB', 0.6, 20.0), np.array([0.1500, 0.0300, 0.6990])),
    (('7.5PB', 0.6, 22.0), np.array([0.1480, 0.0250, 0.6990])),
    (('7.5PB', 0.6, 24.0), np.array([0.1460, 0.0200, 0.6990])),
    (('7.5PB', 0.6, 26.0), np.array([0.1440, 0.0140, 0.6990])),
    (('7.5PB', 0.6, 28.0), np.array([0.1420, 0.0080, 0.6990])),
    (('7.5PB', 0.6, 30.0), np.array([0.1400, 0.0040, 0.6990])),
    (('10PB', 0.6, 2.0), np.array([0.2570, 0.2040, 0.6990])),
    (('10PB', 0.6, 4.0), np.array([0.2370, 0.1570, 0.6990])),
    (('10PB', 0.6, 6.0), np.array([0.2250, 0.1240, 0.6990])),
    (('10PB', 0.6, 8.0), np.array([0.2160, 0.0980, 0.6990])),
    (('10PB', 0.6, 10.0), np.array([0.2110, 0.0820, 0.6990])),
    (('10PB', 0.6, 12.0), np.array([0.2080, 0.0700, 0.6990])),
    (('10PB', 0.6, 14.0), np.array([0.2040, 0.0580, 0.6990])),
    (('10PB', 0.6, 16.0), np.array([0.2020, 0.0500, 0.6990])),
    (('10PB', 0.6, 18.0), np.array([0.2000, 0.0420, 0.6990])),
    (('10PB', 0.6, 20.0), np.array([0.1980, 0.0350, 0.6990])),
    (('10PB', 0.6, 22.0), np.array([0.1960, 0.0290, 0.6990])),
    (('10PB', 0.6, 24.0), np.array([0.1950, 0.0240, 0.6990])),
    (('10PB', 0.6, 26.0), np.array([0.1940, 0.0180, 0.6990])),
    (('10PB', 0.6, 28.0), np.array([0.1920, 0.0120, 0.6990])),
    (('10PB', 0.6, 30.0), np.array([0.1900, 0.0060, 0.6990])),
    (('2.5P', 0.6, 2.0), np.array([0.2720, 0.2050, 0.6990])),
    (('2.5P', 0.6, 4.0), np.array([0.2580, 0.1600, 0.6990])),
    (('2.5P', 0.6, 6.0), np.array([0.2500, 0.1290, 0.6990])),
    (('2.5P', 0.6, 8.0), np.array([0.2440, 0.1040, 0.6990])),
    (('2.5P', 0.6, 10.0), np.array([0.2400, 0.0880, 0.6990])),
    (('2.5P', 0.6, 12.0), np.array([0.2380, 0.0750, 0.6990])),
    (('2.5P', 0.6, 14.0), np.array([0.2350, 0.0600, 0.6990])),
    (('2.5P', 0.6, 16.0), np.array([0.2330, 0.0530, 0.6990])),
    (('2.5P', 0.6, 18.0), np.array([0.2320, 0.0440, 0.6990])),
    (('2.5P', 0.6, 20.0), np.array([0.2300, 0.0380, 0.6990])),
    (('2.5P', 0.6, 22.0), np.array([0.2290, 0.0320, 0.6990])),
    (('2.5P', 0.6, 24.0), np.array([0.2280, 0.0270, 0.6990])),
    (('2.5P', 0.6, 26.0), np.array([0.2270, 0.0210, 0.6990])),
    (('2.5P', 0.6, 28.0), np.array([0.2260, 0.0140, 0.6990])),
    (('5P', 0.6, 2.0), np.array([0.2870, 0.2070, 0.6990])),
    (('5P', 0.6, 4.0), np.array([0.2800, 0.1660, 0.6990])),
    (('5P', 0.6, 6.0), np.array([0.2740, 0.1360, 0.6990])),
    (('5P', 0.6, 8.0), np.array([0.2700, 0.1100, 0.6990])),
    (('5P', 0.6, 10.0), np.array([0.2680, 0.0940, 0.6990])),
    (('5P', 0.6, 12.0), np.array([0.2660, 0.0800, 0.6990])),
    (('5P', 0.6, 14.0), np.array([0.2640, 0.0650, 0.6990])),
    (('5P', 0.6, 16.0), np.array([0.2630, 0.0560, 0.6990])),
    (('5P', 0.6, 18.0), np.array([0.2620, 0.0460, 0.6990])),
    (('5P', 0.6, 20.0), np.array([0.2610, 0.0410, 0.6990])),
    (('5P', 0.6, 22.0), np.array([0.2600, 0.0340, 0.6990])),
    (('5P', 0.6, 24.0), np.array([0.2600, 0.0300, 0.6990])),
    (('7.5P', 0.6, 2.0), np.array([0.3010, 0.2110, 0.6990])),
    (('7.5P', 0.6, 4.0), np.array([0.2960, 0.1700, 0.6990])),
    (('7.5P', 0.6, 6.0), np.array([0.2920, 0.1410, 0.6990])),
    (('7.5P', 0.6, 8.0), np.array([0.2880, 0.1150, 0.6990])),
    (('7.5P', 0.6, 10.0), np.array([0.2840, 0.0980, 0.6990])),
    (('7.5P', 0.6, 12.0), np.array([0.2810, 0.0830, 0.6990])),
    (('7.5P', 0.6, 14.0), np.array([0.2790, 0.0670, 0.6990])),
    (('7.5P', 0.6, 16.0), np.array([0.2770, 0.0560, 0.6990])),
    (('7.5P', 0.6, 18.0), np.array([0.2760, 0.0490, 0.6990])),
    (('7.5P', 0.6, 20.0), np.array([0.2750, 0.0420, 0.6990])),
    (('10P', 0.6, 2.0), np.array([0.3110, 0.2140, 0.6990])),
    (('10P', 0.6, 4.0), np.array([0.3080, 0.1740, 0.6990])),
    (('10P', 0.6, 6.0), np.array([0.3060, 0.1450, 0.6990])),
    (('10P', 0.6, 8.0), np.array([0.3040, 0.1190, 0.6990])),
    (('10P', 0.6, 10.0), np.array([0.3020, 0.1010, 0.6990])),
    (('10P', 0.6, 12.0), np.array([0.3010, 0.0870, 0.6990])),
    (('10P', 0.6, 14.0), np.array([0.2990, 0.0690, 0.6990])),
    (('10P', 0.6, 16.0), np.array([0.2980, 0.0600, 0.6990])),
    (('10P', 0.6, 18.0), np.array([0.2970, 0.0510, 0.6990])),
    (('2.5RP', 0.6, 2.0), np.array([0.3220, 0.2180, 0.6990])),
    (('2.5RP', 0.6, 4.0), np.array([0.3240, 0.1790, 0.6990])),
    (('2.5RP', 0.6, 6.0), np.array([0.3250, 0.1510, 0.6990])),
    (('2.5RP', 0.6, 8.0), np.array([0.3260, 0.1250, 0.6990])),
    (('2.5RP', 0.6, 10.0), np.array([0.3260, 0.1060, 0.6990])),
    (('2.5RP', 0.6, 12.0), np.array([0.3260, 0.0920, 0.6990])),
    (('2.5RP', 0.6, 14.0), np.array([0.3260, 0.0730, 0.6990])),
    (('2.5RP', 0.6, 16.0), np.array([0.3250, 0.0620, 0.6990])),
    (('5RP', 0.6, 2.0), np.array([0.3370, 0.2260, 0.6990])),
    (('5RP', 0.6, 4.0), np.array([0.3470, 0.1890, 0.6990])),
    (('5RP', 0.6, 6.0), np.array([0.3540, 0.1590, 0.6990])),
    (('5RP', 0.6, 8.0), np.array([0.3590, 0.1350, 0.6990])),
    (('5RP', 0.6, 10.0), np.array([0.3630, 0.1140, 0.6990])),
    (('5RP', 0.6, 12.0), np.array([0.3660, 0.0990, 0.6990])),
    (('5RP', 0.6, 14.0), np.array([0.3700, 0.0800, 0.6990])),
    (('5RP', 0.6, 16.0), np.array([0.3730, 0.0680, 0.6990])),
    (('7.5RP', 0.6, 2.0), np.array([0.3550, 0.2360, 0.6990])),
    (('7.5RP', 0.6, 4.0), np.array([0.3730, 0.2000, 0.6990])),
    (('7.5RP', 0.6, 6.0), np.array([0.3870, 0.1700, 0.6990])),
    (('7.5RP', 0.6, 8.0), np.array([0.3970, 0.1460, 0.6990])),
    (('7.5RP', 0.6, 10.0), np.array([0.4060, 0.1240, 0.6990])),
    (('7.5RP', 0.6, 12.0), np.array([0.4130, 0.1060, 0.6990])),
    (('7.5RP', 0.6, 14.0), np.array([0.4210, 0.0870, 0.6990])),
    (('10RP', 0.6, 2.0), np.array([0.3720, 0.2470, 0.6990])),
    (('10RP', 0.6, 4.0), np.array([0.3990, 0.2110, 0.6990])),
    (('10RP', 0.6, 6.0), np.array([0.4190, 0.1820, 0.6990])),
    (('10RP', 0.6, 8.0), np.array([0.4340, 0.1580, 0.6990])),
    (('10RP', 0.6, 10.0), np.array([0.4470, 0.1350, 0.6990])),
    (('10RP', 0.6, 12.0), np.array([0.4590, 0.1140, 0.6990])),
    (('2.5R', 0.6, 2.0), np.array([0.3910, 0.2600, 0.6990])),
    (('2.5R', 0.6, 4.0), np.array([0.4320, 0.2270, 0.6990])),
    (('2.5R', 0.6, 6.0), np.array([0.4640, 0.2000, 0.6990])),
    (('2.5R', 0.6, 8.0), np.array([0.4890, 0.1760, 0.6990])),
    (('2.5R', 0.6, 10.0), np.array([0.5110, 0.1540, 0.6990])),
    (('2.5R', 0.6, 12.0), np.array([0.5370, 0.1260, 0.6990])),
    (('5R', 0.6, 2.0), np.array([0.4110, 0.2740, 0.6990])),
    (('5R', 0.6, 4.0), np.array([0.4690, 0.2460, 0.6990])),
    (('5R', 0.6, 6.0), np.array([0.5140, 0.2210, 0.6990])),
    (('5R', 0.6, 8.0), np.array([0.5510, 0.1970, 0.6990])),
    (('5R', 0.6, 10.0), np.array([0.5860, 0.1760, 0.6990])),
    (('7.5R', 0.6, 2.0), np.array([0.4310, 0.2900, 0.6990])),
    (('7.5R', 0.6, 4.0), np.array([0.5020, 0.2640, 0.6990])),
    (('7.5R', 0.6, 6.0), np.array([0.5580, 0.2400, 0.6990])),
    (('7.5R', 0.6, 8.0), np.array([0.6040, 0.2140, 0.6990])),
    (('7.5R', 0.6, 10.0), np.array([0.6400, 0.1920, 0.6990])),
    (('10R', 0.6, 2.0), np.array([0.4470, 0.3050, 0.6990])),
    (('10R', 0.6, 4.0), np.array([0.5370, 0.2840, 0.6990])),
    (('10R', 0.6, 6.0), np.array([0.6050, 0.2610, 0.6990])),
    (('10R', 0.6, 8.0), np.array([0.6600, 0.2350, 0.6990])),
    (('10R', 0.6, 10.0), np.array([0.7040, 0.2140, 0.6990])),
    (('2.5YR', 0.6, 2.0), np.array([0.4740, 0.3320, 0.6990])),
    (('2.5YR', 0.6, 4.0), np.array([0.6030, 0.3220, 0.6990])),
    (('2.5YR', 0.6, 6.0), np.array([0.6930, 0.3030, 0.6990])),
    (('2.5YR', 0.6, 8.0), np.array([0.7870, 0.2820, 0.6990])),
    (('2.5YR', 0.6, 10.0), np.array([0.8460, 0.2660, 0.6990])),
    (('5YR', 0.6, 2.0), np.array([0.5050, 0.3670, 0.6990])),
    (('5YR', 0.6, 4.0), np.array([0.6730, 0.3710, 0.6990])),
    (('5YR', 0.6, 6.0), np.array([0.8050, 0.3620, 0.6990])),
    (('5YR', 0.6, 8.0), np.array([0.9300, 0.3470, 0.6990])),
    (('7.5YR', 0.6, 2.0), np.array([0.5260, 0.3970, 0.6990])),
    (('7.5YR', 0.6, 4.0), np.array([0.7640, 0.4410, 0.6990])),
    (('10YR', 0.6, 2.0), np.array([0.5510, 0.4440, 0.6990])),
    (('10YR', 0.6, 4.0), np.array([0.8990, 0.5870, 0.6990])),
    (('2.5Y', 0.6, 2.0), np.array([0.5630, 0.4910, 0.6990])),
    (('2.5Y', 0.6, 4.0), np.array([0.9950, 0.7340, 0.6990])),
    (('5Y', 0.6, 2.0), np.array([0.5480, 0.5260, 0.6990])),
    (('7.5Y', 0.6, 2.0), np.array([0.5020, 0.5310, 0.6990])),
    (('7.5Y', 0.6, 4.0), np.array([1.0440, 1.0740, 0.6990])),
    (('10Y', 0.8, 2.0), np.array([0.3970, 0.4480, 0.9430])),
    (('10Y', 0.8, 4.0), np.array([0.7610, 0.9980, 0.9430])),
    (('2.5GY', 0.8, 2.0), np.array([0.3630, 0.4250, 0.9430])),
    (('2.5GY', 0.8, 4.0), np.array([0.5560, 0.9070, 0.9430])),
    (('5GY', 0.8, 2.0), np.array([0.3360, 0.4100, 0.9430])),
    (('5GY', 0.8, 4.0), np.array([0.4000, 0.7380, 0.9430])),
    (('5GY', 0.8, 6.0), np.array([0.4250, 1.2660, 0.9430])),
    (('7.5GY', 0.8, 2.0), np.array([0.3140, 0.3940, 0.9430])),
    (('7.5GY', 0.8, 4.0), np.array([0.3050, 0.6130, 0.9430])),
    (('7.5GY', 0.8, 6.0), np.array([0.2370, 0.9690, 0.9430])),
    (('7.5GY', 0.8, 8.0), np.array([-0.1190, 1.6320, 0.9430])),
    (('10GY', 0.8, 2.0), np.array([0.2980, 0.3810, 0.9430])),
    (('10GY', 0.8, 4.0), np.array([0.2540, 0.5370, 0.9430])),
    (('10GY', 0.8, 6.0), np.array([0.1500, 0.7910, 0.9430])),
    (('10GY', 0.8, 8.0), np.array([-0.1560, 1.2200, 0.9430])),
    (('2.5G', 0.8, 2.0), np.array([0.2870, 0.3710, 0.9430])),
    (('2.5G', 0.8, 4.0), np.array([0.2250, 0.4880, 0.9430])),
    (('2.5G', 0.8, 6.0), np.array([0.1020, 0.6600, 0.9430])),
    (('2.5G', 0.8, 8.0), np.array([-0.2130, 0.9670, 0.9430])),
    (('5G', 0.8, 2.0), np.array([0.2800, 0.3630, 0.9430])),
    (('5G', 0.8, 4.0), np.array([0.2050, 0.4470, 0.9430])),
    (('5G', 0.8, 6.0), np.array([0.0820, 0.5530, 0.9430])),
    (('5G', 0.8, 8.0), np.array([-0.1850, 0.7370, 0.9430])),
    (('7.5G', 0.8, 2.0), np.array([0.2720, 0.3550, 0.9430])),
    (('7.5G', 0.8, 4.0), np.array([0.1910, 0.4140, 0.9430])),
    (('7.5G', 0.8, 6.0), np.array([0.0730, 0.4760, 0.9430])),
    (('7.5G', 0.8, 8.0), np.array([-0.1430, 0.5610, 0.9430])),
    (('10G', 0.8, 2.0), np.array([0.2650, 0.3460, 0.9430])),
    (('10G', 0.8, 4.0), np.array([0.1780, 0.3820, 0.9430])),
    (('10G', 0.8, 6.0), np.array([0.0700, 0.4080, 0.9430])),
    (('10G', 0.8, 8.0), np.array([-0.1000, 0.4280, 0.9430])),
    (('2.5BG', 0.8, 2.0), np.array([0.2530, 0.3320, 0.9430])),
    (('2.5BG', 0.8, 4.0), np.array([0.1630, 0.3420, 0.9430])),
    (('2.5BG', 0.8, 6.0), np.array([0.0700, 0.3410, 0.9430])),
    (('2.5BG', 0.8, 8.0), np.array([-0.0620, 0.3300, 0.9430])),
    (('5BG', 0.8, 2.0), np.array([0.2410, 0.3150, 0.9430])),
    (('5BG', 0.8, 4.0), np.array([0.1500, 0.2990, 0.9430])),
    (('5BG', 0.8, 6.0), np.array([0.0720, 0.2750, 0.9430])),
    (('5BG', 0.8, 8.0), np.array([-0.0180, 0.2390, 0.9430])),
    (('7.5BG', 0.8, 2.0), np.array([0.2300, 0.2960, 0.9430])),
    (('7.5BG', 0.8, 4.0), np.array([0.1450, 0.2640, 0.9430])),
    (('7.5BG', 0.8, 6.0), np.array([0.0770, 0.2330, 0.9430])),
    (('7.5BG', 0.8, 8.0), np.array([0.0080, 0.1980, 0.9430])),
    (('10BG', 0.8, 2.0), np.array([0.2230, 0.2800, 0.9430])),
    (('10BG', 0.8, 4.0), np.array([0.1460, 0.2370, 0.9430])),
    (('10BG', 0.8, 6.0), np.array([0.0860, 0.1990, 0.9430])),
    (('10BG', 0.8, 8.0), np.array([0.0330, 0.1640, 0.9430])),
    (('2.5B', 0.8, 2.0), np.array([0.2200, 0.2710, 0.9430])),
    (('2.5B', 0.8, 4.0), np.array([0.1490, 0.2220, 0.9430])),
    (('2.5B', 0.8, 6.0), np.array([0.0940, 0.1810, 0.9430])),
    (('2.5B', 0.8, 8.0), np.array([0.0480, 0.1470, 0.9430])),
    (('5B', 0.8, 2.0), np.array([0.2180, 0.2580, 0.9430])),
    (('5B', 0.8, 4.0), np.array([0.1540, 0.2070, 0.9430])),
    (('5B', 0.8, 6.0), np.array([0.1060, 0.1630, 0.9430])),
    (('5B', 0.8, 8.0), np.array([0.0690, 0.1270, 0.9430])),
    (('7.5B', 0.8, 2.0), np.array([0.2200, 0.2490, 0.9430])),
    (('7.5B', 0.8, 4.0), np.array([0.1600, 0.1960, 0.9430])),
    (('7.5B', 0.8, 6.0), np.array([0.1150, 0.1530, 0.9430])),
    (('7.5B', 0.8, 8.0), np.array([0.0820, 0.1200, 0.9430])),
    (('7.5B', 0.8, 10.0), np.array([0.0490, 0.0890, 0.9430])),
    (('10B', 0.8, 2.0), np.array([0.2220, 0.2410, 0.9430])),
    (('10B', 0.8, 4.0), np.array([0.1680, 0.1870, 0.9430])),
    (('10B', 0.8, 6.0), np.array([0.1280, 0.1450, 0.9430])),
    (('10B', 0.8, 8.0), np.array([0.0970, 0.1120, 0.9430])),
    (('10B', 0.8, 10.0), np.array([0.0710, 0.0830, 0.9430])),
    (('2.5PB', 0.8, 2.0), np.array([0.2250, 0.2340, 0.9430])),
    (('2.5PB', 0.8, 4.0), np.array([0.1780, 0.1810, 0.9430])),
    (('2.5PB', 0.8, 6.0), np.array([0.1420, 0.1380, 0.9430])),
    (('2.5PB', 0.8, 8.0), np.array([0.1170, 0.1050, 0.9430])),
    (('2.5PB', 0.8, 10.0), np.array([0.0980, 0.0810, 0.9430])),
    (('2.5PB', 0.8, 12.0), np.array([0.0820, 0.0580, 0.9430])),
    (('5PB', 0.8, 2.0), np.array([0.2340, 0.2260, 0.9430])),
    (('5PB', 0.8, 4.0), np.array([0.1920, 0.1740, 0.9430])),
    (('5PB', 0.8, 6.0), np.array([0.1600, 0.1320, 0.9430])),
    (('5PB', 0.8, 8.0), np.array([0.1390, 0.1020, 0.9430])),
    (('5PB', 0.8, 10.0), np.array([0.1230, 0.0830, 0.9430])),
    (('5PB', 0.8, 12.0), np.array([0.1090, 0.0600, 0.9430])),
    (('5PB', 0.8, 14.0), np.array([0.0990, 0.0450, 0.9430])),
    (('5PB', 0.8, 16.0), np.array([0.0920, 0.0340, 0.9430])),
    (('5PB', 0.8, 18.0), np.array([0.0870, 0.0260, 0.9430])),
    (('5PB', 0.8, 20.0), np.array([0.0810, 0.0180, 0.9430])),
    (('7.5PB', 0.8, 2.0), np.array([0.2470, 0.2210, 0.9430])),
    (('7.5PB', 0.8, 4.0), np.array([0.2160, 0.1700, 0.9430])),
    (('7.5PB', 0.8, 6.0), np.array([0.1940, 0.1310, 0.9430])),
    (('7.5PB', 0.8, 8.0), np.array([0.1790, 0.1040, 0.9430])),
    (('7.5PB', 0.8, 10.0), np.array([0.1700, 0.0890, 0.9430])),
    (('7.5PB', 0.8, 12.0), np.array([0.1620, 0.0720, 0.9430])),
    (('7.5PB', 0.8, 14.0), np.array([0.1550, 0.0560, 0.9430])),
    (('7.5PB', 0.8, 16.0), np.array([0.1500, 0.0460, 0.9430])),
    (('7.5PB', 0.8, 18.0), np.array([0.1470, 0.0390, 0.9430])),
    (('7.5PB', 0.8, 20.0), np.array([0.1440, 0.0320, 0.9430])),
    (('7.5PB', 0.8, 22.0), np.array([0.1400, 0.0250, 0.9430])),
    (('7.5PB', 0.8, 24.0), np.array([0.1370, 0.0180, 0.9430])),
    (('7.5PB', 0.8, 26.0), np.array([0.1350, 0.0130, 0.9430])),
    (('7.5PB', 0.8, 28.0), np.array([0.1330, 0.0090, 0.9430])),
    (('7.5PB', 0.8, 30.0), np.array([0.1310, 0.0050, 0.9430])),
    (('7.5PB', 0.8, 32.0), np.array([0.1300, 0.0020, 0.9430])),
    (('7.5PB', 0.8, 34.0), np.array([0.1290, -0.0010, 0.9430])),
    (('7.5PB', 0.8, 36.0), np.array([0.1280, -0.0040, 0.9430])),
    (('10PB', 0.8, 2.0), np.array([0.2630, 0.2190, 0.9430])),
    (('10PB', 0.8, 4.0), np.array([0.2420, 0.1700, 0.9430])),
    (('10PB', 0.8, 6.0), np.array([0.2290, 0.1370, 0.9430])),
    (('10PB', 0.8, 8.0), np.array([0.2200, 0.1120, 0.9430])),
    (('10PB', 0.8, 10.0), np.array([0.2150, 0.0970, 0.9430])),
    (('10PB', 0.8, 12.0), np.array([0.2100, 0.0810, 0.9430])),
    (('10PB', 0.8, 14.0), np.array([0.2060, 0.0680, 0.9430])),
    (('10PB', 0.8, 16.0), np.array([0.2030, 0.0570, 0.9430])),
    (('10PB', 0.8, 18.0), np.array([0.2000, 0.0500, 0.9430])),
    (('10PB', 0.8, 20.0), np.array([0.1990, 0.0430, 0.9430])),
    (('10PB', 0.8, 22.0), np.array([0.1970, 0.0360, 0.9430])),
    (('10PB', 0.8, 24.0), np.array([0.1960, 0.0300, 0.9430])),
    (('10PB', 0.8, 26.0), np.array([0.1940, 0.0260, 0.9430])),
    (('10PB', 0.8, 28.0), np.array([0.1930, 0.0210, 0.9430])),
    (('10PB', 0.8, 30.0), np.array([0.1920, 0.0180, 0.9430])),
    (('10PB', 0.8, 32.0), np.array([0.1910, 0.0140, 0.9430])),
    (('10PB', 0.8, 34.0), np.array([0.1900, 0.0100, 0.9430])),
    (('10PB', 0.8, 36.0), np.array([0.1890, 0.0080, 0.9430])),
    (('2.5P', 0.8, 2.0), np.array([0.2770, 0.2200, 0.9430])),
    (('2.5P', 0.8, 4.0), np.array([0.2640, 0.1740, 0.9430])),
    (('2.5P', 0.8, 6.0), np.array([0.2550, 0.1440, 0.9430])),
    (('2.5P', 0.8, 8.0), np.array([0.2480, 0.1200, 0.9430])),
    (('2.5P', 0.8, 10.0), np.array([0.2450, 0.1050, 0.9430])),
    (('2.5P', 0.8, 12.0), np.array([0.2420, 0.0900, 0.9430])),
    (('2.5P', 0.8, 14.0), np.array([0.2380, 0.0760, 0.9430])),
    (('2.5P', 0.8, 16.0), np.array([0.2360, 0.0640, 0.9430])),
    (('2.5P', 0.8, 18.0), np.array([0.2350, 0.0570, 0.9430])),
    (('2.5P', 0.8, 20.0), np.array([0.2330, 0.0500, 0.9430])),
    (('2.5P', 0.8, 22.0), np.array([0.2320, 0.0440, 0.9430])),
    (('2.5P', 0.8, 24.0), np.array([0.2310, 0.0380, 0.9430])),
    (('2.5P', 0.8, 26.0), np.array([0.2300, 0.0340, 0.9430])),
    (('2.5P', 0.8, 28.0), np.array([0.2280, 0.0280, 0.9430])),
    (('2.5P', 0.8, 30.0), np.array([0.2280, 0.0260, 0.9430])),
    (('2.5P', 0.8, 32.0), np.array([0.2270, 0.0220, 0.9430])),
    (('2.5P', 0.8, 34.0), np.array([0.2270, 0.0190, 0.9430])),
    (('2.5P', 0.8, 36.0), np.array([0.2270, 0.0150, 0.9430])),
    (('5P', 0.8, 2.0), np.array([0.2920, 0.2240, 0.9430])),
    (('5P', 0.8, 4.0), np.array([0.2830, 0.1790, 0.9430])),
    (('5P', 0.8, 6.0), np.array([0.2790, 0.1510, 0.9430])),
    (('5P', 0.8, 8.0), np.array([0.2750, 0.1270, 0.9430])),
    (('5P', 0.8, 10.0), np.array([0.2690, 0.1120, 0.9430])),
    (('5P', 0.8, 12.0), np.array([0.2660, 0.0950, 0.9430])),
    (('5P', 0.8, 14.0), np.array([0.2630, 0.0800, 0.9430])),
    (('5P', 0.8, 16.0), np.array([0.2620, 0.0690, 0.9430])),
    (('5P', 0.8, 18.0), np.array([0.2600, 0.0620, 0.9430])),
    (('5P', 0.8, 20.0), np.array([0.2580, 0.0560, 0.9430])),
    (('5P', 0.8, 22.0), np.array([0.2570, 0.0490, 0.9430])),
    (('5P', 0.8, 24.0), np.array([0.2560, 0.0440, 0.9430])),
    (('5P', 0.8, 26.0), np.array([0.2560, 0.0390, 0.9430])),
    (('5P', 0.8, 28.0), np.array([0.2550, 0.0350, 0.9430])),
    (('5P', 0.8, 30.0), np.array([0.2540, 0.0300, 0.9430])),
    (('7.5P', 0.8, 2.0), np.array([0.3040, 0.2280, 0.9430])),
    (('7.5P', 0.8, 4.0), np.array([0.2980, 0.1840, 0.9430])),
    (('7.5P', 0.8, 6.0), np.array([0.2940, 0.1560, 0.9430])),
    (('7.5P', 0.8, 8.0), np.array([0.2910, 0.1320, 0.9430])),
    (('7.5P', 0.8, 10.0), np.array([0.2900, 0.1180, 0.9430])),
    (('7.5P', 0.8, 12.0), np.array([0.2870, 0.1010, 0.9430])),
    (('7.5P', 0.8, 14.0), np.array([0.2840, 0.0840, 0.9430])),
    (('7.5P', 0.8, 16.0), np.array([0.2830, 0.0720, 0.9430])),
    (('7.5P', 0.8, 18.0), np.array([0.2810, 0.0660, 0.9430])),
    (('7.5P', 0.8, 20.0), np.array([0.2800, 0.0600, 0.9430])),
    (('7.5P', 0.8, 22.0), np.array([0.2790, 0.0540, 0.9430])),
    (('7.5P', 0.8, 24.0), np.array([0.2780, 0.0490, 0.9430])),
    (('7.5P', 0.8, 26.0), np.array([0.2780, 0.0440, 0.9430])),
    (('10P', 0.8, 2.0), np.array([0.3120, 0.2320, 0.9430])),
    (('10P', 0.8, 4.0), np.array([0.3100, 0.1890, 0.9430])),
    (('10P', 0.8, 6.0), np.array([0.3090, 0.1620, 0.9430])),
    (('10P', 0.8, 8.0), np.array([0.3080, 0.1370, 0.9430])),
    (('10P', 0.8, 10.0), np.array([0.3070, 0.1230, 0.9430])),
    (('10P', 0.8, 12.0), np.array([0.3060, 0.1060, 0.9430])),
    (('10P', 0.8, 14.0), np.array([0.3040, 0.0890, 0.9430])),
    (('10P', 0.8, 16.0), np.array([0.3030, 0.0780, 0.9430])),
    (('10P', 0.8, 18.0), np.array([0.3020, 0.0710, 0.9430])),
    (('10P', 0.8, 20.0), np.array([0.3020, 0.0650, 0.9430])),
    (('10P', 0.8, 22.0), np.array([0.3020, 0.0590, 0.9430])),
    (('10P', 0.8, 24.0), np.array([0.3020, 0.0530, 0.9430])),
    (('2.5RP', 0.8, 2.0), np.array([0.3220, 0.2360, 0.9430])),
    (('2.5RP', 0.8, 4.0), np.array([0.3260, 0.1950, 0.9430])),
    (('2.5RP', 0.8, 6.0), np.array([0.3280, 0.1680, 0.9430])),
    (('2.5RP', 0.8, 8.0), np.array([0.3290, 0.1440, 0.9430])),
    (('2.5RP', 0.8, 10.0), np.array([0.3290, 0.1290, 0.9430])),
    (('2.5RP', 0.8, 12.0), np.array([0.3300, 0.1110, 0.9430])),
    (('2.5RP', 0.8, 14.0), np.array([0.3300, 0.0930, 0.9430])),
    (('2.5RP', 0.8, 16.0), np.array([0.3310, 0.0840, 0.9430])),
    (('2.5RP', 0.8, 18.0), np.array([0.3310, 0.0770, 0.9430])),
    (('2.5RP', 0.8, 20.0), np.array([0.3310, 0.0700, 0.9430])),
    (('2.5RP', 0.8, 22.0), np.array([0.3310, 0.0650, 0.9430])),
    (('5RP', 0.8, 2.0), np.array([0.3360, 0.2430, 0.9430])),
    (('5RP', 0.8, 4.0), np.array([0.3470, 0.2060, 0.9430])),
    (('5RP', 0.8, 6.0), np.array([0.3550, 0.1790, 0.9430])),
    (('5RP', 0.8, 8.0), np.array([0.3620, 0.1540, 0.9430])),
    (('5RP', 0.8, 10.0), np.array([0.3660, 0.1390, 0.9430])),
    (('5RP', 0.8, 12.0), np.array([0.3700, 0.1200, 0.9430])),
    (('5RP', 0.8, 14.0), np.array([0.3730, 0.1050, 0.9430])),
    (('5RP', 0.8, 16.0), np.array([0.3760, 0.0940, 0.9430])),
    (('5RP', 0.8, 18.0), np.array([0.3780, 0.0860, 0.9430])),
    (('5RP', 0.8, 20.0), np.array([0.3790, 0.0810, 0.9430])),
    (('7.5RP', 0.8, 2.0), np.array([0.3500, 0.2510, 0.9430])),
    (('7.5RP', 0.8, 4.0), np.array([0.3690, 0.2170, 0.9430])),
    (('7.5RP', 0.8, 6.0), np.array([0.3840, 0.1900, 0.9430])),
    (('7.5RP', 0.8, 8.0), np.array([0.3970, 0.1650, 0.9430])),
    (('7.5RP', 0.8, 10.0), np.array([0.4050, 0.1490, 0.9430])),
    (('7.5RP', 0.8, 12.0), np.array([0.4130, 0.1320, 0.9430])),
    (('7.5RP', 0.8, 14.0), np.array([0.4210, 0.1150, 0.9430])),
    (('7.5RP', 0.8, 16.0), np.array([0.4270, 0.1040, 0.9430])),
    (('10RP', 0.8, 2.0), np.array([0.3650, 0.2610, 0.9430])),
    (('10RP', 0.8, 4.0), np.array([0.3930, 0.2300, 0.9430])),
    (('10RP', 0.8, 6.0), np.array([0.4150, 0.2030, 0.9430])),
    (('10RP', 0.8, 8.0), np.array([0.4350, 0.1770, 0.9430])),
    (('10RP', 0.8, 10.0), np.array([0.4470, 0.1600, 0.9430])),
    (('10RP', 0.8, 12.0), np.array([0.4590, 0.1430, 0.9430])),
    (('10RP', 0.8, 14.0), np.array([0.4700, 0.1270, 0.9430])),
    (('2.5R', 0.8, 2.0), np.array([0.3810, 0.2720, 0.9430])),
    (('2.5R', 0.8, 4.0), np.array([0.4210, 0.2450, 0.9430])),
    (('2.5R', 0.8, 6.0), np.array([0.4550, 0.2190, 0.9430])),
    (('2.5R', 0.8, 8.0), np.array([0.4830, 0.1950, 0.9430])),
    (('2.5R', 0.8, 10.0), np.array([0.5020, 0.1770, 0.9430])),
    (('2.5R', 0.8, 12.0), np.array([0.5210, 0.1590, 0.9430])),
    (('2.5R', 0.8, 14.0), np.array([0.5360, 0.1430, 0.9430])),
    (('5R', 0.8, 2.0), np.array([0.3990, 0.2860, 0.9430])),
    (('5R', 0.8, 4.0), np.array([0.4500, 0.2610, 0.9430])),
    (('5R', 0.8, 6.0), np.array([0.4960, 0.2370, 0.9430])),
    (('5R', 0.8, 8.0), np.array([0.5360, 0.2140, 0.9430])),
    (('5R', 0.8, 10.0), np.array([0.5650, 0.1970, 0.9430])),
    (('5R', 0.8, 12.0), np.array([0.5910, 0.1800, 0.9430])),
    (('7.5R', 0.8, 2.0), np.array([0.4110, 0.2970, 0.9430])),
    (('7.5R', 0.8, 4.0), np.array([0.4770, 0.2760, 0.9430])),
    (('7.5R', 0.8, 6.0), np.array([0.5340, 0.2550, 0.9430])),
    (('7.5R', 0.8, 8.0), np.array([0.5840, 0.2340, 0.9430])),
    (('7.5R', 0.8, 10.0), np.array([0.6240, 0.2160, 0.9430])),
    (('7.5R', 0.8, 12.0), np.array([0.6600, 0.1990, 0.9430])),
    (('10R', 0.8, 2.0), np.array([0.4230, 0.3090, 0.9430])),
    (('10R', 0.8, 4.0), np.array([0.5080, 0.2960, 0.9430])),
    (('10R', 0.8, 6.0), np.array([0.5780, 0.2800, 0.9430])),
    (('10R', 0.8, 8.0), np.array([0.6350, 0.2590, 0.9430])),
    (('10R', 0.8, 10.0), np.array([0.6850, 0.2400, 0.9430])),
    (('10R', 0.8, 12.0), np.array([0.7250, 0.2180, 0.9430])),
    (('2.5YR', 0.8, 2.0), np.array([0.4450, 0.3330, 0.9430])),
    (('2.5YR', 0.8, 4.0), np.array([0.5580, 0.3300, 0.9430])),
    (('2.5YR', 0.8, 6.0), np.array([0.6370, 0.3200, 0.9430])),
    (('2.5YR', 0.8, 8.0), np.array([0.7150, 0.3060, 0.9430])),
    (('2.5YR', 0.8, 10.0), np.array([0.8020, 0.2880, 0.9430])),
    (('2.5YR', 0.8, 12.0), np.array([0.8750, 0.2730, 0.9430])),
    (('5YR', 0.8, 2.0), np.array([0.4630, 0.3610, 0.9430])),
    (('5YR', 0.8, 4.0), np.array([0.6120, 0.3760, 0.9430])),
    (('5YR', 0.8, 6.0), np.array([0.7210, 0.3760, 0.9430])),
    (('5YR', 0.8, 8.0), np.array([0.8260, 0.3720, 0.9430])),
    (('7.5YR', 0.8, 2.0), np.array([0.4750, 0.3860, 0.9430])),
    (('7.5YR', 0.8, 4.0), np.array([0.6630, 0.4300, 0.9430])),
    (('7.5YR', 0.8, 6.0), np.array([0.8170, 0.4500, 0.9430])),
    (('10YR', 0.8, 2.0), np.array([0.4810, 0.4110, 0.9430])),
    (('10YR', 0.8, 4.0), np.array([0.7490, 0.5340, 0.9430])),
    (('2.5Y', 0.8, 2.0), np.array([0.4790, 0.4390, 0.9430])),
    (('2.5Y', 0.8, 4.0), np.array([0.8200, 0.6400, 0.9430])),
    (('5Y', 0.8, 2.0), np.array([0.4650, 0.4570, 0.9430])),
    (('5Y', 0.8, 4.0), np.array([0.8910, 0.7900, 0.9430])),
    (('7.5Y', 0.8, 2.0), np.array([0.4340, 0.4600, 0.9430])),
    (('7.5Y', 0.8, 4.0), np.array([0.8800, 0.9360, 0.9430])),
    (('10Y', 1.0, 2.0), np.array([0.3802, 0.4212, 1.2100])),
    (('10Y', 1.0, 4.0), np.array([0.5010, 0.6000, 1.2100])),
    (('2.5GY', 1.0, 2.0), np.array([0.3540, 0.4088, 1.2100])),
    (('2.5GY', 1.0, 4.0), np.array([0.4390, 0.6150, 1.2100])),
    (('2.5GY', 1.0, 6.0), np.array([0.4800, 0.7230, 1.2100])),
    (('5GY', 1.0, 2.0), np.array([0.3359, 0.3982, 1.2100])),
    (('5GY', 1.0, 4.0), np.array([0.3765, 0.5942, 1.2100])),
    (('5GY', 1.0, 6.0), np.array([0.3980, 0.7400, 1.2100])),
    (('7.5GY', 1.0, 2.0), np.array([0.3154, 0.3840, 1.2100])),
    (('7.5GY', 1.0, 4.0), np.array([0.3133, 0.5380, 1.2100])),
    (('7.5GY', 1.0, 6.0), np.array([0.2900, 0.7060, 1.2100])),
    (('7.5GY', 1.0, 8.0), np.array([0.2350, 0.9440, 1.2100])),
    (('10GY', 1.0, 2.0), np.array([0.3006, 0.3720, 1.2100])),
    (('10GY', 1.0, 4.0), np.array([0.2722, 0.4903, 1.2100])),
    (('10GY', 1.0, 6.0), np.array([0.2232, 0.6392, 1.2100])),
    (('10GY', 1.0, 8.0), np.array([0.1100, 0.8830, 1.2100])),
    (('10GY', 1.0, 10.0), np.array([-0.0130, 1.0650, 1.2100])),
    (('2.5G', 1.0, 2.0), np.array([0.2910, 0.3634, 1.2100])),
    (('2.5G', 1.0, 4.0), np.array([0.2454, 0.4489, 1.2100])),
    (('2.5G', 1.0, 6.0), np.array([0.1711, 0.5619, 1.2100])),
    (('2.5G', 1.0, 8.0), np.array([0.0620, 0.6896, 1.2100])),
    (('2.5G', 1.0, 10.0), np.array([-0.0220, 0.7760, 1.2100])),
    (('5G', 1.0, 2.0), np.array([0.2833, 0.3564, 1.2100])),
    (('5G', 1.0, 4.0), np.array([0.2290, 0.4218, 1.2100])),
    (('5G', 1.0, 6.0), np.array([0.1468, 0.4996, 1.2100])),
    (('5G', 1.0, 8.0), np.array([0.0559, 0.5710, 1.2100])),
    (('5G', 1.0, 10.0), np.array([-0.0200, 0.6200, 1.2100])),
    (('7.5G', 1.0, 2.0), np.array([0.2758, 0.3484, 1.2100])),
    (('7.5G', 1.0, 4.0), np.array([0.2159, 0.3967, 1.2100])),
    (('7.5G', 1.0, 6.0), np.array([0.1344, 0.4505, 1.2100])),
    (('7.5G', 1.0, 8.0), np.array([0.0530, 0.4943, 1.2100])),
    (('7.5G', 1.0, 10.0), np.array([-0.0200, 0.5280, 1.2100])),
    (('10G', 1.0, 2.0), np.array([0.2689, 0.3407, 1.2100])),
    (('10G', 1.0, 4.0), np.array([0.2040, 0.3724, 1.2100])),
    (('10G', 1.0, 6.0), np.array([0.1249, 0.4019, 1.2100])),
    (('10G', 1.0, 8.0), np.array([0.0511, 0.4158, 1.2100])),
    (('10G', 1.0, 10.0), np.array([-0.0180, 0.4240, 1.2100])),
    (('2.5BG', 1.0, 2.0), np.array([0.2600, 0.3289, 1.2100])),
    (('2.5BG', 1.0, 4.0), np.array([0.1883, 0.3406, 1.2100])),
    (('2.5BG', 1.0, 6.0), np.array([0.1169, 0.3452, 1.2100])),
    (('2.5BG', 1.0, 8.0), np.array([0.0476, 0.3458, 1.2100])),
    (('2.5BG', 1.0, 10.0), np.array([-0.0170, 0.3440, 1.2100])),
    (('5BG', 1.0, 2.0), np.array([0.2500, 0.3141, 1.2100])),
    (('5BG', 1.0, 4.0), np.array([0.1753, 0.3021, 1.2100])),
    (('5BG', 1.0, 6.0), np.array([0.1093, 0.2860, 1.2100])),
    (('5BG', 1.0, 8.0), np.array([0.0460, 0.2640, 1.2100])),
    (('5BG', 1.0, 10.0), np.array([-0.0140, 0.2370, 1.2100])),
    (('7.5BG', 1.0, 2.0), np.array([0.2430, 0.3023, 1.2100])),
    (('7.5BG', 1.0, 4.0), np.array([0.1702, 0.2768, 1.2100])),
    (('7.5BG', 1.0, 6.0), np.array([0.1059, 0.2485, 1.2100])),
    (('7.5BG', 1.0, 8.0), np.array([0.0500, 0.2180, 1.2100])),
    (('10BG', 1.0, 2.0), np.array([0.2362, 0.2882, 1.2100])),
    (('10BG', 1.0, 4.0), np.array([0.1658, 0.2496, 1.2100])),
    (('10BG', 1.0, 6.0), np.array([0.1074, 0.2129, 1.2100])),
    (('10BG', 1.0, 8.0), np.array([0.0600, 0.1800, 1.2100])),
    (('2.5B', 1.0, 2.0), np.array([0.2322, 0.2781, 1.2100])),
    (('2.5B', 1.0, 4.0), np.array([0.1649, 0.2324, 1.2100])),
    (('2.5B', 1.0, 6.0), np.array([0.1118, 0.1908, 1.2100])),
    (('2.5B', 1.0, 8.0), np.array([0.0710, 0.1550, 1.2100])),
    (('5B', 1.0, 2.0), np.array([0.2291, 0.2677, 1.2100])),
    (('5B', 1.0, 4.0), np.array([0.1667, 0.2168, 1.2100])),
    (('5B', 1.0, 6.0), np.array([0.1212, 0.1745, 1.2100])),
    (('5B', 1.0, 8.0), np.array([0.0840, 0.1390, 1.2100])),
    (('5B', 1.0, 10.0), np.array([0.0550, 0.1030, 1.2100])),
    (('7.5B', 1.0, 2.0), np.array([0.2291, 0.2579, 1.2100])),
    (('7.5B', 1.0, 4.0), np.array([0.1716, 0.2048, 1.2100])),
    (('7.5B', 1.0, 6.0), np.array([0.1303, 0.1639, 1.2100])),
    (('7.5B', 1.0, 8.0), np.array([0.0968, 0.1280, 1.2100])),
    (('7.5B', 1.0, 10.0), np.array([0.0710, 0.0970, 1.2100])),
    (('10B', 1.0, 2.0), np.array([0.2309, 0.2491, 1.2100])),
    (('10B', 1.0, 4.0), np.array([0.1783, 0.1974, 1.2100])),
    (('10B', 1.0, 6.0), np.array([0.1392, 0.1563, 1.2100])),
    (('10B', 1.0, 8.0), np.array([0.1077, 0.1218, 1.2100])),
    (('10B', 1.0, 10.0), np.array([0.0840, 0.0940, 1.2100])),
    (('2.5PB', 1.0, 2.0), np.array([0.2360, 0.2420, 1.2100])),
    (('2.5PB', 1.0, 4.0), np.array([0.1895, 0.1911, 1.2100])),
    (('2.5PB', 1.0, 6.0), np.array([0.1539, 0.1491, 1.2100])),
    (('2.5PB', 1.0, 8.0), np.array([0.1273, 0.1157, 1.2100])),
    (('2.5PB', 1.0, 10.0), np.array([0.1060, 0.0900, 1.2100])),
    (('2.5PB', 1.0, 12.0), np.array([0.0910, 0.0680, 1.2100])),
    (('2.5PB', 1.0, 14.0), np.array([0.0780, 0.0560, 1.2100])),
    (('5PB', 1.0, 2.0), np.array([0.2427, 0.2368, 1.2100])),
    (('5PB', 1.0, 4.0), np.array([0.2012, 0.1867, 1.2100])),
    (('5PB', 1.0, 6.0), np.array([0.1678, 0.1447, 1.2100])),
    (('5PB', 1.0, 8.0), np.array([0.1447, 0.1124, 1.2100])),
    (('5PB', 1.0, 10.0), np.array([0.1285, 0.0870, 1.2100])),
    (('5PB', 1.0, 12.0), np.array([0.1170, 0.0700, 1.2100])),
    (('5PB', 1.0, 14.0), np.array([0.1120, 0.0600, 1.2100])),
    (('5PB', 1.0, 16.0), np.array([0.1060, 0.0500, 1.2100])),
    (('5PB', 1.0, 18.0), np.array([0.1040, 0.0440, 1.2100])),
    (('5PB', 1.0, 20.0), np.array([0.1000, 0.0380, 1.2100])),
    (('5PB', 1.0, 22.0), np.array([0.0980, 0.0340, 1.2100])),
    (('5PB', 1.0, 24.0), np.array([0.0960, 0.0300, 1.2100])),
    (('5PB', 1.0, 26.0), np.array([0.0950, 0.0270, 1.2100])),
    (('5PB', 1.0, 28.0), np.array([0.0940, 0.0240, 1.2100])),
    (('5PB', 1.0, 30.0), np.array([0.0930, 0.0220, 1.2100])),
    (('5PB', 1.0, 32.0), np.array([0.0920, 0.0200, 1.2100])),
    (('5PB', 1.0, 34.0), np.array([0.0910, 0.0180, 1.2100])),
    (('5PB', 1.0, 36.0), np.array([0.0900, 0.0160, 1.2100])),
    (('5PB', 1.0, 38.0), np.array([0.0890, 0.0140, 1.2100])),
    (('5PB', 1.0, 40.0), np.array([0.0880, 0.0120, 1.2100])),
    (('5PB', 1.0, 42.0), np.array([0.0880, 0.0100, 1.2100])),
    (('5PB', 1.0, 44.0), np.array([0.0870, 0.0080, 1.2100])),
    (('7.5PB', 1.0, 2.0), np.array([0.2547, 0.2310, 1.2100])),
    (('7.5PB', 1.0, 4.0), np.array([0.2232, 0.1821, 1.2100])),
    (('7.5PB', 1.0, 6.0), np.array([0.2000, 0.1422, 1.2100])),
    (('7.5PB', 1.0, 8.0), np.array([0.1872, 0.1141, 1.2100])),
    (('7.5PB', 1.0, 10.0), np.array([0.1804, 0.0950, 1.2100])),
    (('7.5PB', 1.0, 12.0), np.array([0.1763, 0.0804, 1.2100])),
    (('7.5PB', 1.0, 14.0), np.array([0.1738, 0.0688, 1.2100])),
    (('7.5PB', 1.0, 16.0), np.array([0.1720, 0.0583, 1.2100])),
    (('7.5PB', 1.0, 18.0), np.array([0.1709, 0.0518, 1.2100])),
    (('7.5PB', 1.0, 20.0), np.array([0.1701, 0.0454, 1.2100])),
    (('7.5PB', 1.0, 22.0), np.array([0.1696, 0.0402, 1.2100])),
    (('7.5PB', 1.0, 24.0), np.array([0.1691, 0.0352, 1.2100])),
    (('7.5PB', 1.0, 26.0), np.array([0.1689, 0.0309, 1.2100])),
    (('7.5PB', 1.0, 28.0), np.array([0.1686, 0.0270, 1.2100])),
    (('7.5PB', 1.0, 30.0), np.array([0.1684, 0.0234, 1.2100])),
    (('7.5PB', 1.0, 32.0), np.array([0.1682, 0.0202, 1.2100])),
    (('7.5PB', 1.0, 34.0), np.array([0.1682, 0.0180, 1.2100])),
    (('7.5PB', 1.0, 36.0), np.array([0.1681, 0.0160, 1.2100])),
    (('7.5PB', 1.0, 38.0), np.array([0.1680, 0.0140, 1.2100])),
    (('7.5PB', 1.0, 40.0), np.array([0.1680, 0.0120, 1.2100])),
    (('7.5PB', 1.0, 42.0), np.array([0.1680, 0.0100, 1.2100])),
    (('7.5PB', 1.0, 44.0), np.array([0.1680, 0.0080, 1.2100])),
    (('7.5PB', 1.0, 46.0), np.array([0.1680, 0.0060, 1.2100])),
    (('7.5PB', 1.0, 48.0), np.array([0.1680, 0.0040, 1.2100])),
    (('10PB', 1.0, 2.0), np.array([0.2677, 0.2280, 1.2100])),
    (('10PB', 1.0, 4.0), np.array([0.2459, 0.1828, 1.2100])),
    (('10PB', 1.0, 6.0), np.array([0.2290, 0.1470, 1.2100])),
    (('10PB', 1.0, 8.0), np.array([0.2190, 0.1228, 1.2100])),
    (('10PB', 1.0, 10.0), np.array([0.2120, 0.1029, 1.2100])),
    (('10PB', 1.0, 12.0), np.array([0.2070, 0.0869, 1.2100])),
    (('10PB', 1.0, 14.0), np.array([0.2038, 0.0745, 1.2100])),
    (('10PB', 1.0, 16.0), np.array([0.2008, 0.0638, 1.2100])),
    (('10PB', 1.0, 18.0), np.array([0.1991, 0.0564, 1.2100])),
    (('10PB', 1.0, 20.0), np.array([0.1976, 0.0493, 1.2100])),
    (('10PB', 1.0, 22.0), np.array([0.1965, 0.0436, 1.2100])),
    (('10PB', 1.0, 24.0), np.array([0.1952, 0.0380, 1.2100])),
    (('10PB', 1.0, 26.0), np.array([0.1942, 0.0326, 1.2100])),
    (('10PB', 1.0, 28.0), np.array([0.1936, 0.0281, 1.2100])),
    (('10PB', 1.0, 30.0), np.array([0.1928, 0.0240, 1.2100])),
    (('10PB', 1.0, 32.0), np.array([0.1930, 0.0210, 1.2100])),
    (('10PB', 1.0, 34.0), np.array([0.1930, 0.0190, 1.2100])),
    (('10PB', 1.0, 36.0), np.array([0.1930, 0.0160, 1.2100])),
    (('10PB', 1.0, 38.0), np.array([0.1930, 0.0140, 1.2100])),
    (('10PB', 1.0, 40.0), np.array([0.1930, 0.0130, 1.2100])),
    (('10PB', 1.0, 42.0), np.array([0.1920, 0.0110, 1.2100])),
    (('10PB', 1.0, 44.0), np.array([0.1920, 0.0090, 1.2100])),
    (('10PB', 1.0, 46.0), np.array([0.1920, 0.0070, 1.2100])),
    (('10PB', 1.0, 48.0), np.array([0.1920, 0.0050, 1.2100])),
    (('2.5P', 1.0, 2.0), np.array([0.2808, 0.2296, 1.2100])),
    (('2.5P', 1.0, 4.0), np.array([0.2668, 0.1874, 1.2100])),
    (('2.5P', 1.0, 6.0), np.array([0.2570, 0.1559, 1.2100])),
    (('2.5P', 1.0, 8.0), np.array([0.2496, 0.1303, 1.2100])),
    (('2.5P', 1.0, 10.0), np.array([0.2441, 0.1112, 1.2100])),
    (('2.5P', 1.0, 12.0), np.array([0.2394, 0.0940, 1.2100])),
    (('2.5P', 1.0, 14.0), np.array([0.2361, 0.0810, 1.2100])),
    (('2.5P', 1.0, 16.0), np.array([0.2331, 0.0696, 1.2100])),
    (('2.5P', 1.0, 18.0), np.array([0.2312, 0.0618, 1.2100])),
    (('2.5P', 1.0, 20.0), np.array([0.2295, 0.0542, 1.2100])),
    (('2.5P', 1.0, 22.0), np.array([0.2279, 0.0473, 1.2100])),
    (('2.5P', 1.0, 24.0), np.array([0.2266, 0.0418, 1.2100])),
    (('2.5P', 1.0, 26.0), np.array([0.2251, 0.0355, 1.2100])),
    (('2.5P', 1.0, 28.0), np.array([0.2250, 0.0310, 1.2100])),
    (('2.5P', 1.0, 30.0), np.array([0.2240, 0.0260, 1.2100])),
    (('2.5P', 1.0, 32.0), np.array([0.2240, 0.0220, 1.2100])),
    (('2.5P', 1.0, 34.0), np.array([0.2230, 0.0200, 1.2100])),
    (('2.5P', 1.0, 36.0), np.array([0.2220, 0.0180, 1.2100])),
    (('2.5P', 1.0, 38.0), np.array([0.2220, 0.0150, 1.2100])),
    (('2.5P', 1.0, 40.0), np.array([0.2220, 0.0130, 1.2100])),
    (('2.5P', 1.0, 42.0), np.array([0.2220, 0.0110, 1.2100])),
    (('5P', 1.0, 2.0), np.array([0.2936, 0.2330, 1.2100])),
    (('5P', 1.0, 4.0), np.array([0.2854, 0.1927, 1.2100])),
    (('5P', 1.0, 6.0), np.array([0.2794, 0.1628, 1.2100])),
    (('5P', 1.0, 8.0), np.array([0.2742, 0.1375, 1.2100])),
    (('5P', 1.0, 10.0), np.array([0.2701, 0.1178, 1.2100])),
    (('5P', 1.0, 12.0), np.array([0.2670, 0.1006, 1.2100])),
    (('5P', 1.0, 14.0), np.array([0.2645, 0.0863, 1.2100])),
    (('5P', 1.0, 16.0), np.array([0.2625, 0.0746, 1.2100])),
    (('5P', 1.0, 18.0), np.array([0.2612, 0.0667, 1.2100])),
    (('5P', 1.0, 20.0), np.array([0.2601, 0.0586, 1.2100])),
    (('5P', 1.0, 22.0), np.array([0.2590, 0.0509, 1.2100])),
    (('5P', 1.0, 24.0), np.array([0.2580, 0.0460, 1.2100])),
    (('5P', 1.0, 26.0), np.array([0.2580, 0.0390, 1.2100])),
    (('5P', 1.0, 28.0), np.array([0.2570, 0.0340, 1.2100])),
    (('5P', 1.0, 30.0), np.array([0.2560, 0.0280, 1.2100])),
    (('5P', 1.0, 32.0), np.array([0.2560, 0.0240, 1.2100])),
    (('7.5P', 1.0, 2.0), np.array([0.3030, 0.2361, 1.2100])),
    (('7.5P', 1.0, 4.0), np.array([0.2991, 0.1974, 1.2100])),
    (('7.5P', 1.0, 6.0), np.array([0.2960, 0.1682, 1.2100])),
    (('7.5P', 1.0, 8.0), np.array([0.2932, 0.1429, 1.2100])),
    (('7.5P', 1.0, 10.0), np.array([0.2905, 0.1229, 1.2100])),
    (('7.5P', 1.0, 12.0), np.array([0.2884, 0.1059, 1.2100])),
    (('7.5P', 1.0, 14.0), np.array([0.2868, 0.0903, 1.2100])),
    (('7.5P', 1.0, 16.0), np.array([0.2852, 0.0790, 1.2100])),
    (('7.5P', 1.0, 18.0), np.array([0.2841, 0.0706, 1.2100])),
    (('7.5P', 1.0, 20.0), np.array([0.2831, 0.0625, 1.2100])),
    (('7.5P', 1.0, 22.0), np.array([0.2820, 0.0550, 1.2100])),
    (('7.5P', 1.0, 24.0), np.array([0.2820, 0.0490, 1.2100])),
    (('7.5P', 1.0, 26.0), np.array([0.2810, 0.0420, 1.2100])),
    (('10P', 1.0, 2.0), np.array([0.3132, 0.2404, 1.2100])),
    (('10P', 1.0, 4.0), np.array([0.3132, 0.2032, 1.2100])),
    (('10P', 1.0, 6.0), np.array([0.3126, 0.1737, 1.2100])),
    (('10P', 1.0, 8.0), np.array([0.3114, 0.1481, 1.2100])),
    (('10P', 1.0, 10.0), np.array([0.3102, 0.1282, 1.2100])),
    (('10P', 1.0, 12.0), np.array([0.3094, 0.1110, 1.2100])),
    (('10P', 1.0, 14.0), np.array([0.3084, 0.0952, 1.2100])),
    (('10P', 1.0, 16.0), np.array([0.3078, 0.0839, 1.2100])),
    (('10P', 1.0, 18.0), np.array([0.3069, 0.0748, 1.2100])),
    (('10P', 1.0, 20.0), np.array([0.3070, 0.0660, 1.2100])),
    (('10P', 1.0, 22.0), np.array([0.3060, 0.0580, 1.2100])),
    (('10P', 1.0, 24.0), np.array([0.3050, 0.0510, 1.2100])),
    (('2.5RP', 1.0, 2.0), np.array([0.3240, 0.2459, 1.2100])),
    (('2.5RP', 1.0, 4.0), np.array([0.3290, 0.2095, 1.2100])),
    (('2.5RP', 1.0, 6.0), np.array([0.3321, 0.1811, 1.2100])),
    (('2.5RP', 1.0, 8.0), np.array([0.3342, 0.1551, 1.2100])),
    (('2.5RP', 1.0, 10.0), np.array([0.3354, 0.1351, 1.2100])),
    (('2.5RP', 1.0, 12.0), np.array([0.3361, 0.1181, 1.2100])),
    (('2.5RP', 1.0, 14.0), np.array([0.3368, 0.1020, 1.2100])),
    (('2.5RP', 1.0, 16.0), np.array([0.3368, 0.0902, 1.2100])),
    (('2.5RP', 1.0, 18.0), np.array([0.3370, 0.0800, 1.2100])),
    (('2.5RP', 1.0, 20.0), np.array([0.3360, 0.0700, 1.2100])),
    (('2.5RP', 1.0, 22.0), np.array([0.3360, 0.0610, 1.2100])),
    (('5RP', 1.0, 2.0), np.array([0.3378, 0.2542, 1.2100])),
    (('5RP', 1.0, 4.0), np.array([0.3503, 0.2196, 1.2100])),
    (('5RP', 1.0, 6.0), np.array([0.3588, 0.1920, 1.2100])),
    (('5RP', 1.0, 8.0), np.array([0.3660, 0.1662, 1.2100])),
    (('5RP', 1.0, 10.0), np.array([0.3727, 0.1458, 1.2100])),
    (('5RP', 1.0, 12.0), np.array([0.3772, 0.1283, 1.2100])),
    (('5RP', 1.0, 14.0), np.array([0.3811, 0.1138, 1.2100])),
    (('5RP', 1.0, 16.0), np.array([0.3840, 0.1020, 1.2100])),
    (('5RP', 1.0, 18.0), np.array([0.3860, 0.0920, 1.2100])),
    (('5RP', 1.0, 20.0), np.array([0.3890, 0.0790, 1.2100])),
    (('7.5RP', 1.0, 2.0), np.array([0.3498, 0.2617, 1.2100])),
    (('7.5RP', 1.0, 4.0), np.array([0.3705, 0.2300, 1.2100])),
    (('7.5RP', 1.0, 6.0), np.array([0.3865, 0.2036, 1.2100])),
    (('7.5RP', 1.0, 8.0), np.array([0.4005, 0.1793, 1.2100])),
    (('7.5RP', 1.0, 10.0), np.array([0.4132, 0.1580, 1.2100])),
    (('7.5RP', 1.0, 12.0), np.array([0.4240, 0.1400, 1.2100])),
    (('7.5RP', 1.0, 14.0), np.array([0.4330, 0.1250, 1.2100])),
    (('7.5RP', 1.0, 16.0), np.array([0.4400, 0.1130, 1.2100])),
    (('10RP', 1.0, 2.0), np.array([0.3629, 0.2710, 1.2100])),
    (('10RP', 1.0, 4.0), np.array([0.3920, 0.2423, 1.2100])),
    (('10RP', 1.0, 6.0), np.array([0.4151, 0.2169, 1.2100])),
    (('10RP', 1.0, 8.0), np.array([0.4357, 0.1921, 1.2100])),
    (('10RP', 1.0, 10.0), np.array([0.4521, 0.1710, 1.2100])),
    (('10RP', 1.0, 12.0), np.array([0.4668, 0.1514, 1.2100])),
    (('10RP', 1.0, 14.0), np.array([0.4790, 0.1360, 1.2100])),
    (('10RP', 1.0, 16.0), np.array([0.4880, 0.1240, 1.2100])),
    (('2.5R', 1.0, 2.0), np.array([0.3768, 0.2816, 1.2100])),
    (('2.5R', 1.0, 4.0), np.array([0.4166, 0.2569, 1.2100])),
    (('2.5R', 1.0, 6.0), np.array([0.4515, 0.2329, 1.2100])),
    (('2.5R', 1.0, 8.0), np.array([0.4812, 0.2103, 1.2100])),
    (('2.5R', 1.0, 10.0), np.array([0.5058, 0.1900, 1.2100])),
    (('2.5R', 1.0, 12.0), np.array([0.5330, 0.1690, 1.2100])),
    (('2.5R', 1.0, 14.0), np.array([0.5480, 0.1560, 1.2100])),
    (('5R', 1.0, 2.0), np.array([0.3908, 0.2929, 1.2100])),
    (('5R', 1.0, 4.0), np.array([0.4420, 0.2728, 1.2100])),
    (('5R', 1.0, 6.0), np.array([0.4885, 0.2515, 1.2100])),
    (('5R', 1.0, 8.0), np.array([0.5282, 0.2297, 1.2100])),
    (('5R', 1.0, 10.0), np.array([0.5604, 0.2100, 1.2100])),
    (('5R', 1.0, 12.0), np.array([0.5950, 0.1870, 1.2100])),
    (('7.5R', 1.0, 2.0), np.array([0.4020, 0.3034, 1.2100])),
    (('7.5R', 1.0, 4.0), np.array([0.4660, 0.2888, 1.2100])),
    (('7.5R', 1.0, 6.0), np.array([0.5235, 0.2698, 1.2100])),
    (('7.5R', 1.0, 8.0), np.array([0.5722, 0.2487, 1.2100])),
    (('7.5R', 1.0, 10.0), np.array([0.6111, 0.2290, 1.2100])),
    (('7.5R', 1.0, 12.0), np.array([0.6540, 0.2040, 1.2100])),
    (('10R', 1.0, 2.0), np.array([0.4128, 0.3154, 1.2100])),
    (('10R', 1.0, 4.0), np.array([0.4933, 0.3068, 1.2100])),
    (('10R', 1.0, 6.0), np.array([0.5584, 0.2921, 1.2100])),
    (('10R', 1.0, 8.0), np.array([0.6178, 0.2713, 1.2100])),
    (('10R', 1.0, 10.0), np.array([0.6661, 0.2499, 1.2100])),
    (('10R', 1.0, 12.0), np.array([0.7110, 0.2270, 1.2100])),
    (('2.5YR', 1.0, 2.0), np.array([0.4258, 0.3344, 1.2100])),
    (('2.5YR', 1.0, 4.0), np.array([0.5311, 0.3371, 1.2100])),
    (('2.5YR', 1.0, 6.0), np.array([0.6048, 0.3270, 1.2100])),
    (('2.5YR', 1.0, 8.0), np.array([0.6721, 0.3058, 1.2100])),
    (('2.5YR', 1.0, 10.0), np.array([0.7270, 0.2790, 1.2100])),
    (('2.5YR', 1.0, 12.0), np.array([0.7780, 0.2480, 1.2100])),
    (('5YR', 1.0, 2.0), np.array([0.4377, 0.3580, 1.2100])),
    (('5YR', 1.0, 4.0), np.array([0.5660, 0.3795, 1.2100])),
    (('5YR', 1.0, 6.0), np.array([0.6560, 0.3840, 1.2100])),
    (('5YR', 1.0, 8.0), np.array([0.7360, 0.3850, 1.2100])),
    (('5YR', 1.0, 10.0), np.array([0.8290, 0.3850, 1.2100])),
    (('7.5YR', 1.0, 2.0), np.array([0.4430, 0.3775, 1.2100])),
    (('7.5YR', 1.0, 4.0), np.array([0.5850, 0.4220, 1.2100])),
    (('7.5YR', 1.0, 6.0), np.array([0.6850, 0.4420, 1.2100])),
    (('10YR', 1.0, 2.0), np.array([0.4446, 0.3982, 1.2100])),
    (('10YR', 1.0, 4.0), np.array([0.5910, 0.4640, 1.2100])),
    (('2.5Y', 1.0, 2.0), np.array([0.4362, 0.4177, 1.2100])),
    (('2.5Y', 1.0, 4.0), np.array([0.5810, 0.5050, 1.2100])),
    (('5Y', 1.0, 2.0), np.array([0.4230, 0.4265, 1.2100])),
    (('5Y', 1.0, 4.0), np.array([0.5650, 0.5430, 1.2100])),
    (('7.5Y', 1.0, 2.0), np.array([0.4042, 0.4287, 1.2100])),
    (('7.5Y', 1.0, 4.0), np.array([0.5430, 0.5700, 1.2100])),
    (('10Y', 2.0, 2.0), np.array([0.3556, 0.3848, 3.1260])),
    (('10Y', 2.0, 4.0), np.array([0.4188, 0.4789, 3.1260])),
    (('10Y', 2.0, 6.0), np.array([0.4820, 0.5760, 3.1260])),
    (('2.5GY', 2.0, 2.0), np.array([0.3421, 0.3803, 3.1260])),
    (('2.5GY', 2.0, 4.0), np.array([0.3881, 0.4752, 3.1260])),
    (('2.5GY', 2.0, 6.0), np.array([0.4340, 0.5900, 3.1260])),
    (('2.5GY', 2.0, 8.0), np.array([0.4760, 0.7160, 3.1260])),
    (('5GY', 2.0, 2.0), np.array([0.3309, 0.3743, 3.1260])),
    (('5GY', 2.0, 4.0), np.array([0.3582, 0.4650, 3.1260])),
    (('5GY', 2.0, 6.0), np.array([0.3839, 0.5748, 3.1260])),
    (('5GY', 2.0, 8.0), np.array([0.4090, 0.7100, 3.1260])),
    (('5GY', 2.0, 10.0), np.array([0.4180, 0.8250, 3.1260])),
    (('7.5GY', 2.0, 2.0), np.array([0.3165, 0.3650, 3.1260])),
    (('7.5GY', 2.0, 4.0), np.array([0.3248, 0.4457, 3.1260])),
    (('7.5GY', 2.0, 6.0), np.array([0.3260, 0.5379, 3.1260])),
    (('7.5GY', 2.0, 8.0), np.array([0.3160, 0.6509, 3.1260])),
    (('7.5GY', 2.0, 10.0), np.array([0.2970, 0.7680, 3.1260])),
    (('7.5GY', 2.0, 12.0), np.array([0.2740, 0.8790, 3.1260])),
    (('7.5GY', 2.0, 14.0), np.array([0.2430, 1.0100, 3.1260])),
    (('10GY', 2.0, 2.0), np.array([0.3069, 0.3580, 3.1260])),
    (('10GY', 2.0, 4.0), np.array([0.2986, 0.4240, 3.1260])),
    (('10GY', 2.0, 6.0), np.array([0.2852, 0.4972, 3.1260])),
    (('10GY', 2.0, 8.0), np.array([0.2628, 0.5837, 3.1260])),
    (('10GY', 2.0, 10.0), np.array([0.2307, 0.6814, 3.1260])),
    (('10GY', 2.0, 12.0), np.array([0.1907, 0.7798, 3.1260])),
    (('10GY', 2.0, 14.0), np.array([0.1410, 0.8760, 3.1260])),
    (('10GY', 2.0, 16.0), np.array([0.0610, 0.9980, 3.1260])),
    (('10GY', 2.0, 18.0), np.array([-0.0300, 1.1100, 3.1260])),
    (('2.5G', 2.0, 2.0), np.array([0.2978, 0.3507, 3.1260])),
    (('2.5G', 2.0, 4.0), np.array([0.2763, 0.3998, 3.1260])),
    (('2.5G', 2.0, 6.0), np.array([0.2493, 0.4522, 3.1260])),
    (('2.5G', 2.0, 8.0), np.array([0.2192, 0.5042, 3.1260])),
    (('2.5G', 2.0, 10.0), np.array([0.1773, 0.5698, 3.1260])),
    (('2.5G', 2.0, 12.0), np.array([0.1307, 0.6308, 3.1260])),
    (('2.5G', 2.0, 14.0), np.array([0.0820, 0.6860, 3.1260])),
    (('2.5G', 2.0, 16.0), np.array([0.0329, 0.7358, 3.1260])),
    (('2.5G', 2.0, 18.0), np.array([-0.0200, 0.7880, 3.1260])),
    (('5G', 2.0, 2.0), np.array([0.2918, 0.3450, 3.1260])),
    (('5G', 2.0, 4.0), np.array([0.2640, 0.3845, 3.1260])),
    (('5G', 2.0, 6.0), np.array([0.2318, 0.4231, 3.1260])),
    (('5G', 2.0, 8.0), np.array([0.1979, 0.4583, 3.1260])),
    (('5G', 2.0, 10.0), np.array([0.1560, 0.4981, 3.1260])),
    (('5G', 2.0, 12.0), np.array([0.1120, 0.5358, 3.1260])),
    (('5G', 2.0, 14.0), np.array([0.0688, 0.5691, 3.1260])),
    (('5G', 2.0, 16.0), np.array([0.0277, 0.5986, 3.1260])),
    (('5G', 2.0, 18.0), np.array([-0.0150, 0.6250, 3.1260])),
    (('7.5G', 2.0, 2.0), np.array([0.2869, 0.3400, 3.1260])),
    (('7.5G', 2.0, 4.0), np.array([0.2540, 0.3705, 3.1260])),
    (('7.5G', 2.0, 6.0), np.array([0.2200, 0.3983, 3.1260])),
    (('7.5G', 2.0, 8.0), np.array([0.1842, 0.4244, 3.1260])),
    (('7.5G', 2.0, 10.0), np.array([0.1442, 0.4505, 3.1260])),
    (('7.5G', 2.0, 12.0), np.array([0.1022, 0.4759, 3.1260])),
    (('7.5G', 2.0, 14.0), np.array([0.0629, 0.4973, 3.1260])),
    (('7.5G', 2.0, 16.0), np.array([0.0276, 0.5153, 3.1260])),
    (('7.5G', 2.0, 18.0), np.array([-0.0100, 0.5320, 3.1260])),
    (('10G', 2.0, 2.0), np.array([0.2820, 0.3341, 3.1260])),
    (('10G', 2.0, 4.0), np.array([0.2442, 0.3559, 3.1260])),
    (('10G', 2.0, 6.0), np.array([0.2092, 0.3739, 3.1260])),
    (('10G', 2.0, 8.0), np.array([0.1705, 0.3911, 3.1260])),
    (('10G', 2.0, 10.0), np.array([0.1321, 0.4059, 3.1260])),
    (('10G', 2.0, 12.0), np.array([0.0934, 0.4183, 3.1260])),
    (('10G', 2.0, 14.0), np.array([0.0599, 0.4270, 3.1260])),
    (('10G', 2.0, 16.0), np.array([0.0285, 0.4327, 3.1260])),
    (('10G', 2.0, 18.0), np.array([-0.0070, 0.4410, 3.1260])),
    (('2.5BG', 2.0, 2.0), np.array([0.2765, 0.3271, 3.1260])),
    (('2.5BG', 2.0, 4.0), np.array([0.2343, 0.3378, 3.1260])),
    (('2.5BG', 2.0, 6.0), np.array([0.1971, 0.3452, 3.1260])),
    (('2.5BG', 2.0, 8.0), np.array([0.1557, 0.3517, 3.1260])),
    (('2.5BG', 2.0, 10.0), np.array([0.1190, 0.3551, 3.1260])),
    (('2.5BG', 2.0, 12.0), np.array([0.0851, 0.3576, 3.1260])),
    (('2.5BG', 2.0, 14.0), np.array([0.0555, 0.3588, 3.1260])),
    (('2.5BG', 2.0, 16.0), np.array([0.0290, 0.3590, 3.1260])),
    (('2.5BG', 2.0, 18.0), np.array([-0.0020, 0.3600, 3.1260])),
    (('5BG', 2.0, 2.0), np.array([0.2697, 0.3175, 3.1260])),
    (('5BG', 2.0, 4.0), np.array([0.2234, 0.3150, 3.1260])),
    (('5BG', 2.0, 6.0), np.array([0.1843, 0.3110, 3.1260])),
    (('5BG', 2.0, 8.0), np.array([0.1405, 0.3037, 3.1260])),
    (('5BG', 2.0, 10.0), np.array([0.1050, 0.2956, 3.1260])),
    (('5BG', 2.0, 12.0), np.array([0.0769, 0.2880, 3.1260])),
    (('5BG', 2.0, 14.0), np.array([0.0510, 0.2800, 3.1260])),
    (('5BG', 2.0, 16.0), np.array([0.0280, 0.2740, 3.1260])),
    (('7.5BG', 2.0, 2.0), np.array([0.2651, 0.3098, 3.1260])),
    (('7.5BG', 2.0, 4.0), np.array([0.2162, 0.2981, 3.1260])),
    (('7.5BG', 2.0, 6.0), np.array([0.1747, 0.2853, 3.1260])),
    (('7.5BG', 2.0, 8.0), np.array([0.1325, 0.2710, 3.1260])),
    (('7.5BG', 2.0, 10.0), np.array([0.0991, 0.2582, 3.1260])),
    (('7.5BG', 2.0, 12.0), np.array([0.0724, 0.2478, 3.1260])),
    (('7.5BG', 2.0, 14.0), np.array([0.0500, 0.2370, 3.1260])),
    (('7.5BG', 2.0, 16.0), np.array([0.0280, 0.2280, 3.1260])),
    (('10BG', 2.0, 2.0), np.array([0.2606, 0.3010, 3.1260])),
    (('10BG', 2.0, 4.0), np.array([0.2096, 0.2790, 3.1260])),
    (('10BG', 2.0, 6.0), np.array([0.1669, 0.2570, 3.1260])),
    (('10BG', 2.0, 8.0), np.array([0.1258, 0.2331, 3.1260])),
    (('10BG', 2.0, 10.0), np.array([0.0929, 0.2133, 3.1260])),
    (('10BG', 2.0, 12.0), np.array([0.0680, 0.1940, 3.1260])),
    (('10BG', 2.0, 14.0), np.array([0.0500, 0.1810, 3.1260])),
    (('2.5B', 2.0, 2.0), np.array([0.2578, 0.2940, 3.1260])),
    (('2.5B', 2.0, 4.0), np.array([0.2060, 0.2649, 3.1260])),
    (('2.5B', 2.0, 6.0), np.array([0.1621, 0.2358, 3.1260])),
    (('2.5B', 2.0, 8.0), np.array([0.1230, 0.2076, 3.1260])),
    (('2.5B', 2.0, 10.0), np.array([0.0911, 0.1828, 3.1260])),
    (('2.5B', 2.0, 12.0), np.array([0.0670, 0.1650, 3.1260])),
    (('5B', 2.0, 2.0), np.array([0.2559, 0.2874, 3.1260])),
    (('5B', 2.0, 4.0), np.array([0.2048, 0.2518, 3.1260])),
    (('5B', 2.0, 6.0), np.array([0.1617, 0.2162, 3.1260])),
    (('5B', 2.0, 8.0), np.array([0.1245, 0.1827, 3.1260])),
    (('5B', 2.0, 10.0), np.array([0.0965, 0.1558, 3.1260])),
    (('5B', 2.0, 12.0), np.array([0.0770, 0.1360, 3.1260])),
    (('7.5B', 2.0, 2.0), np.array([0.2545, 0.2799, 3.1260])),
    (('7.5B', 2.0, 4.0), np.array([0.2063, 0.2400, 3.1260])),
    (('7.5B', 2.0, 6.0), np.array([0.1658, 0.2026, 3.1260])),
    (('7.5B', 2.0, 8.0), np.array([0.1313, 0.1692, 3.1260])),
    (('7.5B', 2.0, 10.0), np.array([0.1051, 0.1422, 3.1260])),
    (('7.5B', 2.0, 12.0), np.array([0.0880, 0.1240, 3.1260])),
    (('7.5B', 2.0, 14.0), np.array([0.0700, 0.1060, 3.1260])),
    (('10B', 2.0, 2.0), np.array([0.2558, 0.2725, 3.1260])),
    (('10B', 2.0, 4.0), np.array([0.2102, 0.2313, 3.1260])),
    (('10B', 2.0, 6.0), np.array([0.1716, 0.1937, 3.1260])),
    (('10B', 2.0, 8.0), np.array([0.1396, 0.1603, 3.1260])),
    (('10B', 2.0, 10.0), np.array([0.1157, 0.1346, 3.1260])),
    (('10B', 2.0, 12.0), np.array([0.0990, 0.1150, 3.1260])),
    (('10B', 2.0, 14.0), np.array([0.0830, 0.0970, 3.1260])),
    (('2.5PB', 2.0, 2.0), np.array([0.2592, 0.2675, 3.1260])),
    (('2.5PB', 2.0, 4.0), np.array([0.2175, 0.2245, 3.1260])),
    (('2.5PB', 2.0, 6.0), np.array([0.1825, 0.1857, 3.1260])),
    (('2.5PB', 2.0, 8.0), np.array([0.1540, 0.1530, 3.1260])),
    (('2.5PB', 2.0, 10.0), np.array([0.1332, 0.1278, 3.1260])),
    (('2.5PB', 2.0, 12.0), np.array([0.1166, 0.1076, 3.1260])),
    (('2.5PB', 2.0, 14.0), np.array([0.1020, 0.0890, 3.1260])),
    (('2.5PB', 2.0, 16.0), np.array([0.0920, 0.0760, 3.1260])),
    (('2.5PB', 2.0, 18.0), np.array([0.0840, 0.0640, 3.1260])),
    (('5PB', 2.0, 2.0), np.array([0.2638, 0.2624, 3.1260])),
    (('5PB', 2.0, 4.0), np.array([0.2263, 0.2192, 3.1260])),
    (('5PB', 2.0, 6.0), np.array([0.1942, 0.1811, 3.1260])),
    (('5PB', 2.0, 8.0), np.array([0.1685, 0.1491, 3.1260])),
    (('5PB', 2.0, 10.0), np.array([0.1500, 0.1240, 3.1260])),
    (('5PB', 2.0, 12.0), np.array([0.1363, 0.1048, 3.1260])),
    (('5PB', 2.0, 14.0), np.array([0.1253, 0.0873, 3.1260])),
    (('5PB', 2.0, 16.0), np.array([0.1170, 0.0740, 3.1260])),
    (('5PB', 2.0, 18.0), np.array([0.1110, 0.0640, 3.1260])),
    (('5PB', 2.0, 20.0), np.array([0.1080, 0.0580, 3.1260])),
    (('5PB', 2.0, 22.0), np.array([0.1040, 0.0520, 3.1260])),
    (('5PB', 2.0, 24.0), np.array([0.1000, 0.0460, 3.1260])),
    (('5PB', 2.0, 26.0), np.array([0.0980, 0.0420, 3.1260])),
    (('5PB', 2.0, 28.0), np.array([0.0970, 0.0400, 3.1260])),
    (('5PB', 2.0, 30.0), np.array([0.0960, 0.0380, 3.1260])),
    (('5PB', 2.0, 32.0), np.array([0.0950, 0.0360, 3.1260])),
    (('5PB', 2.0, 34.0), np.array([0.0930, 0.0330, 3.1260])),
    (('5PB', 2.0, 36.0), np.array([0.0920, 0.0310, 3.1260])),
    (('5PB', 2.0, 38.0), np.array([0.0900, 0.0280, 3.1260])),
    (('5PB', 2.0, 40.0), np.array([0.0890, 0.0250, 3.1260])),
    (('5PB', 2.0, 42.0), np.array([0.0870, 0.0220, 3.1260])),
    (('5PB', 2.0, 44.0), np.array([0.0860, 0.0190, 3.1260])),
    (('5PB', 2.0, 46.0), np.array([0.0840, 0.0160, 3.1260])),
    (('7.5PB', 2.0, 2.0), np.array([0.2712, 0.2582, 3.1260])),
    (('7.5PB', 2.0, 4.0), np.array([0.2420, 0.2148, 3.1260])),
    (('7.5PB', 2.0, 6.0), np.array([0.2189, 0.1790, 3.1260])),
    (('7.5PB', 2.0, 8.0), np.array([0.2005, 0.1495, 3.1260])),
    (('7.5PB', 2.0, 10.0), np.array([0.1882, 0.1258, 3.1260])),
    (('7.5PB', 2.0, 12.0), np.array([0.1813, 0.1094, 3.1260])),
    (('7.5PB', 2.0, 14.0), np.array([0.1762, 0.0955, 3.1260])),
    (('7.5PB', 2.0, 16.0), np.array([0.1728, 0.0839, 3.1260])),
    (('7.5PB', 2.0, 18.0), np.array([0.1701, 0.0742, 3.1260])),
    (('7.5PB', 2.0, 20.0), np.array([0.1685, 0.0666, 3.1260])),
    (('7.5PB', 2.0, 22.0), np.array([0.1670, 0.0594, 3.1260])),
    (('7.5PB', 2.0, 24.0), np.array([0.1660, 0.0538, 3.1260])),
    (('7.5PB', 2.0, 26.0), np.array([0.1653, 0.0492, 3.1260])),
    (('7.5PB', 2.0, 28.0), np.array([0.1647, 0.0451, 3.1260])),
    (('7.5PB', 2.0, 30.0), np.array([0.1640, 0.0409, 3.1260])),
    (('7.5PB', 2.0, 32.0), np.array([0.1635, 0.0373, 3.1260])),
    (('7.5PB', 2.0, 34.0), np.array([0.1630, 0.0340, 3.1260])),
    (('7.5PB', 2.0, 36.0), np.array([0.1628, 0.0310, 3.1260])),
    (('7.5PB', 2.0, 38.0), np.array([0.1623, 0.0280, 3.1260])),
    (('7.5PB', 2.0, 40.0), np.array([0.1620, 0.0250, 3.1260])),
    (('7.5PB', 2.0, 42.0), np.array([0.1620, 0.0220, 3.1260])),
    (('7.5PB', 2.0, 44.0), np.array([0.1620, 0.0190, 3.1260])),
    (('7.5PB', 2.0, 46.0), np.array([0.1610, 0.0160, 3.1260])),
    (('7.5PB', 2.0, 48.0), np.array([0.1610, 0.0130, 3.1260])),
    (('7.5PB', 2.0, 50.0), np.array([0.1600, 0.0100, 3.1260])),
    (('10PB', 2.0, 2.0), np.array([0.2803, 0.2567, 3.1260])),
    (('10PB', 2.0, 4.0), np.array([0.2600, 0.2162, 3.1260])),
    (('10PB', 2.0, 6.0), np.array([0.2440, 0.1840, 3.1260])),
    (('10PB', 2.0, 8.0), np.array([0.2294, 0.1551, 3.1260])),
    (('10PB', 2.0, 10.0), np.array([0.2200, 0.1330, 3.1260])),
    (('10PB', 2.0, 12.0), np.array([0.2139, 0.1170, 3.1260])),
    (('10PB', 2.0, 14.0), np.array([0.2087, 0.1026, 3.1260])),
    (('10PB', 2.0, 16.0), np.array([0.2052, 0.0910, 3.1260])),
    (('10PB', 2.0, 18.0), np.array([0.2021, 0.0808, 3.1260])),
    (('10PB', 2.0, 20.0), np.array([0.1998, 0.0718, 3.1260])),
    (('10PB', 2.0, 22.0), np.array([0.1978, 0.0643, 3.1260])),
    (('10PB', 2.0, 24.0), np.array([0.1962, 0.0578, 3.1260])),
    (('10PB', 2.0, 26.0), np.array([0.1949, 0.0520, 3.1260])),
    (('10PB', 2.0, 28.0), np.array([0.1937, 0.0471, 3.1260])),
    (('10PB', 2.0, 30.0), np.array([0.1925, 0.0420, 3.1260])),
    (('10PB', 2.0, 32.0), np.array([0.1918, 0.0379, 3.1260])),
    (('10PB', 2.0, 34.0), np.array([0.1911, 0.0344, 3.1260])),
    (('10PB', 2.0, 36.0), np.array([0.1900, 0.0310, 3.1260])),
    (('10PB', 2.0, 38.0), np.array([0.1900, 0.0280, 3.1260])),
    (('10PB', 2.0, 40.0), np.array([0.1900, 0.0250, 3.1260])),
    (('10PB', 2.0, 42.0), np.array([0.1890, 0.0220, 3.1260])),
    (('10PB', 2.0, 44.0), np.array([0.1890, 0.0190, 3.1260])),
    (('10PB', 2.0, 46.0), np.array([0.1890, 0.0160, 3.1260])),
    (('10PB', 2.0, 48.0), np.array([0.1880, 0.0130, 3.1260])),
    (('10PB', 2.0, 50.0), np.array([0.1880, 0.0100, 3.1260])),
    (('2.5P', 2.0, 2.0), np.array([0.2892, 0.2583, 3.1260])),
    (('2.5P', 2.0, 4.0), np.array([0.2758, 0.2208, 3.1260])),
    (('2.5P', 2.0, 6.0), np.array([0.2661, 0.1921, 3.1260])),
    (('2.5P', 2.0, 8.0), np.array([0.2570, 0.1635, 3.1260])),
    (('2.5P', 2.0, 10.0), np.array([0.2501, 0.1422, 3.1260])),
    (('2.5P', 2.0, 12.0), np.array([0.2449, 0.1245, 3.1260])),
    (('2.5P', 2.0, 14.0), np.array([0.2406, 0.1100, 3.1260])),
    (('2.5P', 2.0, 16.0), np.array([0.2372, 0.0980, 3.1260])),
    (('2.5P', 2.0, 18.0), np.array([0.2345, 0.0873, 3.1260])),
    (('2.5P', 2.0, 20.0), np.array([0.2320, 0.0779, 3.1260])),
    (('2.5P', 2.0, 22.0), np.array([0.2298, 0.0696, 3.1260])),
    (('2.5P', 2.0, 24.0), np.array([0.2277, 0.0621, 3.1260])),
    (('2.5P', 2.0, 26.0), np.array([0.2260, 0.0555, 3.1260])),
    (('2.5P', 2.0, 28.0), np.array([0.2245, 0.0491, 3.1260])),
    (('2.5P', 2.0, 30.0), np.array([0.2231, 0.0432, 3.1260])),
    (('2.5P', 2.0, 32.0), np.array([0.2220, 0.0380, 3.1260])),
    (('2.5P', 2.0, 34.0), np.array([0.2210, 0.0350, 3.1260])),
    (('2.5P', 2.0, 36.0), np.array([0.2200, 0.0310, 3.1260])),
    (('2.5P', 2.0, 38.0), np.array([0.2190, 0.0280, 3.1260])),
    (('2.5P', 2.0, 40.0), np.array([0.2190, 0.0250, 3.1260])),
    (('2.5P', 2.0, 42.0), np.array([0.2180, 0.0220, 3.1260])),
    (('2.5P', 2.0, 44.0), np.array([0.2170, 0.0190, 3.1260])),
    (('5P', 2.0, 2.0), np.array([0.2984, 0.2612, 3.1260])),
    (('5P', 2.0, 4.0), np.array([0.2908, 0.2261, 3.1260])),
    (('5P', 2.0, 6.0), np.array([0.2850, 0.1992, 3.1260])),
    (('5P', 2.0, 8.0), np.array([0.2791, 0.1707, 3.1260])),
    (('5P', 2.0, 10.0), np.array([0.2748, 0.1500, 3.1260])),
    (('5P', 2.0, 12.0), np.array([0.2709, 0.1320, 3.1260])),
    (('5P', 2.0, 14.0), np.array([0.2676, 0.1163, 3.1260])),
    (('5P', 2.0, 16.0), np.array([0.2652, 0.1045, 3.1260])),
    (('5P', 2.0, 18.0), np.array([0.2632, 0.0935, 3.1260])),
    (('5P', 2.0, 20.0), np.array([0.2612, 0.0838, 3.1260])),
    (('5P', 2.0, 22.0), np.array([0.2597, 0.0750, 3.1260])),
    (('5P', 2.0, 24.0), np.array([0.2582, 0.0669, 3.1260])),
    (('5P', 2.0, 26.0), np.array([0.2569, 0.0594, 3.1260])),
    (('5P', 2.0, 28.0), np.array([0.2559, 0.0525, 3.1260])),
    (('5P', 2.0, 30.0), np.array([0.2550, 0.0450, 3.1260])),
    (('5P', 2.0, 32.0), np.array([0.2540, 0.0390, 3.1260])),
    (('5P', 2.0, 34.0), np.array([0.2530, 0.0350, 3.1260])),
    (('5P', 2.0, 36.0), np.array([0.2520, 0.0310, 3.1260])),
    (('7.5P', 2.0, 2.0), np.array([0.3071, 0.2647, 3.1260])),
    (('7.5P', 2.0, 4.0), np.array([0.3048, 0.2321, 3.1260])),
    (('7.5P', 2.0, 6.0), np.array([0.3025, 0.2058, 3.1260])),
    (('7.5P', 2.0, 8.0), np.array([0.3000, 0.1781, 3.1260])),
    (('7.5P', 2.0, 10.0), np.array([0.2979, 0.1569, 3.1260])),
    (('7.5P', 2.0, 12.0), np.array([0.2956, 0.1392, 3.1260])),
    (('7.5P', 2.0, 14.0), np.array([0.2938, 0.1235, 3.1260])),
    (('7.5P', 2.0, 16.0), np.array([0.2922, 0.1106, 3.1260])),
    (('7.5P', 2.0, 18.0), np.array([0.2912, 0.0995, 3.1260])),
    (('7.5P', 2.0, 20.0), np.array([0.2902, 0.0901, 3.1260])),
    (('7.5P', 2.0, 22.0), np.array([0.2890, 0.0799, 3.1260])),
    (('7.5P', 2.0, 24.0), np.array([0.2882, 0.0719, 3.1260])),
    (('7.5P', 2.0, 26.0), np.array([0.2870, 0.0640, 3.1260])),
    (('7.5P', 2.0, 28.0), np.array([0.2860, 0.0550, 3.1260])),
    (('7.5P', 2.0, 30.0), np.array([0.2840, 0.0460, 3.1260])),
    (('10P', 2.0, 2.0), np.array([0.3161, 0.2691, 3.1260])),
    (('10P', 2.0, 4.0), np.array([0.3189, 0.2390, 3.1260])),
    (('10P', 2.0, 6.0), np.array([0.3207, 0.2132, 3.1260])),
    (('10P', 2.0, 8.0), np.array([0.3219, 0.1862, 3.1260])),
    (('10P', 2.0, 10.0), np.array([0.3230, 0.1659, 3.1260])),
    (('10P', 2.0, 12.0), np.array([0.3233, 0.1477, 3.1260])),
    (('10P', 2.0, 14.0), np.array([0.3235, 0.1317, 3.1260])),
    (('10P', 2.0, 16.0), np.array([0.3235, 0.1181, 3.1260])),
    (('10P', 2.0, 18.0), np.array([0.3233, 0.1063, 3.1260])),
    (('10P', 2.0, 20.0), np.array([0.3231, 0.0962, 3.1260])),
    (('10P', 2.0, 22.0), np.array([0.3230, 0.0861, 3.1260])),
    (('10P', 2.0, 24.0), np.array([0.3220, 0.0760, 3.1260])),
    (('10P', 2.0, 26.0), np.array([0.3220, 0.0680, 3.1260])),
    (('10P', 2.0, 28.0), np.array([0.3220, 0.0580, 3.1260])),
    (('2.5RP', 2.0, 2.0), np.array([0.3279, 0.2754, 3.1260])),
    (('2.5RP', 2.0, 4.0), np.array([0.3382, 0.2496, 3.1260])),
    (('2.5RP', 2.0, 6.0), np.array([0.3470, 0.2259, 3.1260])),
    (('2.5RP', 2.0, 8.0), np.array([0.3555, 0.2003, 3.1260])),
    (('2.5RP', 2.0, 10.0), np.array([0.3617, 0.1800, 3.1260])),
    (('2.5RP', 2.0, 12.0), np.array([0.3668, 0.1618, 3.1260])),
    (('2.5RP', 2.0, 14.0), np.array([0.3711, 0.1449, 3.1260])),
    (('2.5RP', 2.0, 16.0), np.array([0.3748, 0.1310, 3.1260])),
    (('2.5RP', 2.0, 18.0), np.array([0.3778, 0.1188, 3.1260])),
    (('2.5RP', 2.0, 20.0), np.array([0.3802, 0.1080, 3.1260])),
    (('2.5RP', 2.0, 22.0), np.array([0.3830, 0.0960, 3.1260])),
    (('2.5RP', 2.0, 24.0), np.array([0.3860, 0.0860, 3.1260])),
    (('2.5RP', 2.0, 26.0), np.array([0.3880, 0.0760, 3.1260])),
    (('5RP', 2.0, 2.0), np.array([0.3383, 0.2829, 3.1260])),
    (('5RP', 2.0, 4.0), np.array([0.3558, 0.2597, 3.1260])),
    (('5RP', 2.0, 6.0), np.array([0.3708, 0.2380, 3.1260])),
    (('5RP', 2.0, 8.0), np.array([0.3858, 0.2140, 3.1260])),
    (('5RP', 2.0, 10.0), np.array([0.3971, 0.1939, 3.1260])),
    (('5RP', 2.0, 12.0), np.array([0.4080, 0.1764, 3.1260])),
    (('5RP', 2.0, 14.0), np.array([0.4180, 0.1598, 3.1260])),
    (('5RP', 2.0, 16.0), np.array([0.4269, 0.1454, 3.1260])),
    (('5RP', 2.0, 18.0), np.array([0.4338, 0.1340, 3.1260])),
    (('5RP', 2.0, 20.0), np.array([0.4420, 0.1210, 3.1260])),
    (('5RP', 2.0, 22.0), np.array([0.4480, 0.1100, 3.1260])),
    (('7.5RP', 2.0, 2.0), np.array([0.3459, 0.2892, 3.1260])),
    (('7.5RP', 2.0, 4.0), np.array([0.3702, 0.2683, 3.1260])),
    (('7.5RP', 2.0, 6.0), np.array([0.3918, 0.2490, 3.1260])),
    (('7.5RP', 2.0, 8.0), np.array([0.4137, 0.2276, 3.1260])),
    (('7.5RP', 2.0, 10.0), np.array([0.4321, 0.2082, 3.1260])),
    (('7.5RP', 2.0, 12.0), np.array([0.4481, 0.1903, 3.1260])),
    (('7.5RP', 2.0, 14.0), np.array([0.4624, 0.1737, 3.1260])),
    (('7.5RP', 2.0, 16.0), np.array([0.4744, 0.1595, 3.1260])),
    (('7.5RP', 2.0, 18.0), np.array([0.4850, 0.1460, 3.1260])),
    (('7.5RP', 2.0, 20.0), np.array([0.4970, 0.1320, 3.1260])),
    (('10RP', 2.0, 2.0), np.array([0.3532, 0.2957, 3.1260])),
    (('10RP', 2.0, 4.0), np.array([0.3850, 0.2778, 3.1260])),
    (('10RP', 2.0, 6.0), np.array([0.4139, 0.2608, 3.1260])),
    (('10RP', 2.0, 8.0), np.array([0.4428, 0.2419, 3.1260])),
    (('10RP', 2.0, 10.0), np.array([0.4678, 0.2237, 3.1260])),
    (('10RP', 2.0, 12.0), np.array([0.4911, 0.2060, 3.1260])),
    (('10RP', 2.0, 14.0), np.array([0.5129, 0.1888, 3.1260])),
    (('10RP', 2.0, 16.0), np.array([0.5310, 0.1740, 3.1260])),
    (('10RP', 2.0, 18.0), np.array([0.5460, 0.1610, 3.1260])),
    (('2.5R', 2.0, 2.0), np.array([0.3614, 0.3033, 3.1260])),
    (('2.5R', 2.0, 4.0), np.array([0.4021, 0.2900, 3.1260])),
    (('2.5R', 2.0, 6.0), np.array([0.4390, 0.2760, 3.1260])),
    (('2.5R', 2.0, 8.0), np.array([0.4776, 0.2593, 3.1260])),
    (('2.5R', 2.0, 10.0), np.array([0.5122, 0.2428, 3.1260])),
    (('2.5R', 2.0, 12.0), np.array([0.5438, 0.2254, 3.1260])),
    (('2.5R', 2.0, 14.0), np.array([0.5734, 0.2083, 3.1260])),
    (('2.5R', 2.0, 16.0), np.array([0.6010, 0.1920, 3.1260])),
    (('5R', 2.0, 2.0), np.array([0.3692, 0.3111, 3.1260])),
    (('5R', 2.0, 4.0), np.array([0.4184, 0.3032, 3.1260])),
    (('5R', 2.0, 6.0), np.array([0.4642, 0.2934, 3.1260])),
    (('5R', 2.0, 8.0), np.array([0.5143, 0.2800, 3.1260])),
    (('5R', 2.0, 10.0), np.array([0.5557, 0.2633, 3.1260])),
    (('5R', 2.0, 12.0), np.array([0.5930, 0.2465, 3.1260])),
    (('5R', 2.0, 14.0), np.array([0.6302, 0.2287, 3.1260])),
    (('5R', 2.0, 16.0), np.array([0.6590, 0.2120, 3.1260])),
    (('7.5R', 2.0, 2.0), np.array([0.3751, 0.3181, 3.1260])),
    (('7.5R', 2.0, 4.0), np.array([0.4335, 0.3169, 3.1260])),
    (('7.5R', 2.0, 6.0), np.array([0.4875, 0.3123, 3.1260])),
    (('7.5R', 2.0, 8.0), np.array([0.5433, 0.3027, 3.1260])),
    (('7.5R', 2.0, 10.0), np.array([0.5952, 0.2874, 3.1260])),
    (('7.5R', 2.0, 12.0), np.array([0.6392, 0.2704, 3.1260])),
    (('7.5R', 2.0, 14.0), np.array([0.6791, 0.2520, 3.1260])),
    (('7.5R', 2.0, 16.0), np.array([0.7140, 0.2340, 3.1260])),
    (('10R', 2.0, 2.0), np.array([0.3811, 0.3274, 3.1260])),
    (('10R', 2.0, 4.0), np.array([0.4481, 0.3330, 3.1260])),
    (('10R', 2.0, 6.0), np.array([0.5095, 0.3331, 3.1260])),
    (('10R', 2.0, 8.0), np.array([0.5713, 0.3259, 3.1260])),
    (('10R', 2.0, 10.0), np.array([0.6247, 0.3120, 3.1260])),
    (('10R', 2.0, 12.0), np.array([0.6732, 0.2937, 3.1260])),
    (('10R', 2.0, 14.0), np.array([0.7165, 0.2734, 3.1260])),
    (('10R', 2.0, 16.0), np.array([0.7520, 0.2540, 3.1260])),
    (('2.5YR', 2.0, 2.0), np.array([0.3852, 0.3365, 3.1260])),
    (('2.5YR', 2.0, 4.0), np.array([0.4598, 0.3508, 3.1260])),
    (('2.5YR', 2.0, 6.0), np.array([0.5280, 0.3581, 3.1260])),
    (('2.5YR', 2.0, 8.0), np.array([0.5995, 0.3590, 3.1260])),
    (('2.5YR', 2.0, 10.0), np.array([0.6590, 0.3500, 3.1260])),
    (('2.5YR', 2.0, 12.0), np.array([0.7180, 0.3400, 3.1260])),
    (('2.5YR', 2.0, 14.0), np.array([0.7790, 0.3230, 3.1260])),
    (('2.5YR', 2.0, 16.0), np.array([0.8240, 0.3090, 3.1260])),
    (('5YR', 2.0, 2.0), np.array([0.3880, 0.3476, 3.1260])),
    (('5YR', 2.0, 4.0), np.array([0.4674, 0.3738, 3.1260])),
    (('5YR', 2.0, 6.0), np.array([0.5426, 0.3925, 3.1260])),
    (('5YR', 2.0, 8.0), np.array([0.6200, 0.4060, 3.1260])),
    (('5YR', 2.0, 10.0), np.array([0.6840, 0.4150, 3.1260])),
    (('7.5YR', 2.0, 2.0), np.array([0.3889, 0.3590, 3.1260])),
    (('7.5YR', 2.0, 4.0), np.array([0.4690, 0.3964, 3.1260])),
    (('7.5YR', 2.0, 6.0), np.array([0.5475, 0.4271, 3.1260])),
    (('7.5YR', 2.0, 8.0), np.array([0.6200, 0.4560, 3.1260])),
    (('10YR', 2.0, 2.0), np.array([0.3872, 0.3688, 3.1260])),
    (('10YR', 2.0, 4.0), np.array([0.4676, 0.4168, 3.1260])),
    (('10YR', 2.0, 6.0), np.array([0.5450, 0.4580, 3.1260])),
    (('10YR', 2.0, 8.0), np.array([0.6120, 0.4930, 3.1260])),
    (('2.5Y', 2.0, 2.0), np.array([0.3825, 0.3785, 3.1260])),
    (('2.5Y', 2.0, 4.0), np.array([0.4627, 0.4392, 3.1260])),
    (('2.5Y', 2.0, 6.0), np.array([0.5380, 0.4860, 3.1260])),
    (('2.5Y', 2.0, 8.0), np.array([0.6040, 0.5260, 3.1260])),
    (('5Y', 2.0, 2.0), np.array([0.3757, 0.3839, 3.1260])),
    (('5Y', 2.0, 4.0), np.array([0.4543, 0.4573, 3.1260])),
    (('5Y', 2.0, 6.0), np.array([0.5260, 0.5190, 3.1260])),
    (('7.5Y', 2.0, 2.0), np.array([0.3660, 0.3858, 3.1260])),
    (('7.5Y', 2.0, 4.0), np.array([0.4401, 0.4723, 3.1260])),
    (('7.5Y', 2.0, 6.0), np.array([0.5100, 0.5470, 3.1260])),
    (('10Y', 3.0, 2.0), np.array([0.3513, 0.3789, 6.5500])),
    (('10Y', 3.0, 4.0), np.array([0.3961, 0.4452, 6.5500])),
    (('10Y', 3.0, 6.0), np.array([0.4345, 0.5026, 6.5500])),
    (('10Y', 3.0, 8.0), np.array([0.4700, 0.5550, 6.5500])),
    (('2.5GY', 3.0, 2.0), np.array([0.3412, 0.3768, 6.5500])),
    (('2.5GY', 3.0, 4.0), np.array([0.3772, 0.4484, 6.5500])),
    (('2.5GY', 3.0, 6.0), np.array([0.4069, 0.5110, 6.5500])),
    (('2.5GY', 3.0, 8.0), np.array([0.4320, 0.5760, 6.5500])),
    (('2.5GY', 3.0, 10.0), np.array([0.4550, 0.6380, 6.5500])),
    (('5GY', 3.0, 2.0), np.array([0.3319, 0.3729, 6.5500])),
    (('5GY', 3.0, 4.0), np.array([0.3554, 0.4429, 6.5500])),
    (('5GY', 3.0, 6.0), np.array([0.3750, 0.5109, 6.5500])),
    (('5GY', 3.0, 8.0), np.array([0.3924, 0.5832, 6.5500])),
    (('5GY', 3.0, 10.0), np.array([0.4040, 0.6500, 6.5500])),
    (('5GY', 3.0, 12.0), np.array([0.4130, 0.7180, 6.5500])),
    (('7.5GY', 3.0, 2.0), np.array([0.3180, 0.3644, 6.5500])),
    (('7.5GY', 3.0, 4.0), np.array([0.3270, 0.4288, 6.5500])),
    (('7.5GY', 3.0, 6.0), np.array([0.3333, 0.4967, 6.5500])),
    (('7.5GY', 3.0, 8.0), np.array([0.3341, 0.5700, 6.5500])),
    (('7.5GY', 3.0, 10.0), np.array([0.3266, 0.6448, 6.5500])),
    (('7.5GY', 3.0, 12.0), np.array([0.3150, 0.7200, 6.5500])),
    (('7.5GY', 3.0, 14.0), np.array([0.3000, 0.7930, 6.5500])),
    (('10GY', 3.0, 2.0), np.array([0.3088, 0.3578, 6.5500])),
    (('10GY', 3.0, 4.0), np.array([0.3053, 0.4123, 6.5500])),
    (('10GY', 3.0, 6.0), np.array([0.2992, 0.4717, 6.5500])),
    (('10GY', 3.0, 8.0), np.array([0.2887, 0.5361, 6.5500])),
    (('10GY', 3.0, 10.0), np.array([0.2724, 0.6026, 6.5500])),
    (('10GY', 3.0, 12.0), np.array([0.2531, 0.6700, 6.5500])),
    (('10GY', 3.0, 14.0), np.array([0.2283, 0.7423, 6.5500])),
    (('10GY', 3.0, 16.0), np.array([0.2020, 0.8070, 6.5500])),
    (('10GY', 3.0, 18.0), np.array([0.1680, 0.8800, 6.5500])),
    (('10GY', 3.0, 20.0), np.array([0.1300, 0.9480, 6.5500])),
    (('10GY', 3.0, 22.0), np.array([0.0890, 1.0140, 6.5500])),
    (('10GY', 3.0, 24.0), np.array([0.0460, 1.0780, 6.5500])),
    (('2.5G', 3.0, 2.0), np.array([0.2999, 0.3500, 6.5500])),
    (('2.5G', 3.0, 4.0), np.array([0.2836, 0.3915, 6.5500])),
    (('2.5G', 3.0, 6.0), np.array([0.2642, 0.4342, 6.5500])),
    (('2.5G', 3.0, 8.0), np.array([0.2435, 0.4752, 6.5500])),
    (('2.5G', 3.0, 10.0), np.array([0.2170, 0.5211, 6.5500])),
    (('2.5G', 3.0, 12.0), np.array([0.1902, 0.5642, 6.5500])),
    (('2.5G', 3.0, 14.0), np.array([0.1626, 0.6052, 6.5500])),
    (('2.5G', 3.0, 16.0), np.array([0.1341, 0.6420, 6.5500])),
    (('2.5G', 3.0, 18.0), np.array([0.1049, 0.6766, 6.5500])),
    (('2.5G', 3.0, 20.0), np.array([0.0720, 0.7127, 6.5500])),
    (('2.5G', 3.0, 22.0), np.array([0.0390, 0.7468, 6.5500])),
    (('2.5G', 3.0, 24.0), np.array([0.0090, 0.7760, 6.5500])),
    (('5G', 3.0, 2.0), np.array([0.2935, 0.3439, 6.5500])),
    (('5G', 3.0, 4.0), np.array([0.2711, 0.3780, 6.5500])),
    (('5G', 3.0, 6.0), np.array([0.2471, 0.4100, 6.5500])),
    (('5G', 3.0, 8.0), np.array([0.2228, 0.4380, 6.5500])),
    (('5G', 3.0, 10.0), np.array([0.1935, 0.4682, 6.5500])),
    (('5G', 3.0, 12.0), np.array([0.1660, 0.4948, 6.5500])),
    (('5G', 3.0, 14.0), np.array([0.1382, 0.5197, 6.5500])),
    (('5G', 3.0, 16.0), np.array([0.1120, 0.5414, 6.5500])),
    (('5G', 3.0, 18.0), np.array([0.0882, 0.5605, 6.5500])),
    (('5G', 3.0, 20.0), np.array([0.0620, 0.5802, 6.5500])),
    (('5G', 3.0, 22.0), np.array([0.0340, 0.6011, 6.5500])),
    (('5G', 3.0, 24.0), np.array([0.0040, 0.6220, 6.5500])),
    (('7.5G', 3.0, 2.0), np.array([0.2890, 0.3391, 6.5500])),
    (('7.5G', 3.0, 4.0), np.array([0.2618, 0.3667, 6.5500])),
    (('7.5G', 3.0, 6.0), np.array([0.2346, 0.3901, 6.5500])),
    (('7.5G', 3.0, 8.0), np.array([0.2088, 0.4101, 6.5500])),
    (('7.5G', 3.0, 10.0), np.array([0.1800, 0.4310, 6.5500])),
    (('7.5G', 3.0, 12.0), np.array([0.1516, 0.4505, 6.5500])),
    (('7.5G', 3.0, 14.0), np.array([0.1262, 0.4667, 6.5500])),
    (('7.5G', 3.0, 16.0), np.array([0.1023, 0.4818, 6.5500])),
    (('7.5G', 3.0, 18.0), np.array([0.0798, 0.4954, 6.5500])),
    (('7.5G', 3.0, 20.0), np.array([0.0568, 0.5082, 6.5500])),
    (('7.5G', 3.0, 22.0), np.array([0.0332, 0.5206, 6.5500])),
    (('7.5G', 3.0, 24.0), np.array([0.0060, 0.5340, 6.5500])),
    (('10G', 3.0, 2.0), np.array([0.2844, 0.3337, 6.5500])),
    (('10G', 3.0, 4.0), np.array([0.2525, 0.3537, 6.5500])),
    (('10G', 3.0, 6.0), np.array([0.2240, 0.3699, 6.5500])),
    (('10G', 3.0, 8.0), np.array([0.1970, 0.3841, 6.5500])),
    (('10G', 3.0, 10.0), np.array([0.1688, 0.3974, 6.5500])),
    (('10G', 3.0, 12.0), np.array([0.1411, 0.4095, 6.5500])),
    (('10G', 3.0, 14.0), np.array([0.1161, 0.4192, 6.5500])),
    (('10G', 3.0, 16.0), np.array([0.0925, 0.4275, 6.5500])),
    (('10G', 3.0, 18.0), np.array([0.0718, 0.4340, 6.5500])),
    (('10G', 3.0, 20.0), np.array([0.0528, 0.4393, 6.5500])),
    (('10G', 3.0, 22.0), np.array([0.0333, 0.4444, 6.5500])),
    (('10G', 3.0, 24.0), np.array([0.0090, 0.4500, 6.5500])),
    (('2.5BG', 3.0, 2.0), np.array([0.2799, 0.3271, 6.5500])),
    (('2.5BG', 3.0, 4.0), np.array([0.2437, 0.3386, 6.5500])),
    (('2.5BG', 3.0, 6.0), np.array([0.2132, 0.3468, 6.5500])),
    (('2.5BG', 3.0, 8.0), np.array([0.1845, 0.3531, 6.5500])),
    (('2.5BG', 3.0, 10.0), np.array([0.1552, 0.3580, 6.5500])),
    (('2.5BG', 3.0, 12.0), np.array([0.1288, 0.3620, 6.5500])),
    (('2.5BG', 3.0, 14.0), np.array([0.1051, 0.3648, 6.5500])),
    (('2.5BG', 3.0, 16.0), np.array([0.0843, 0.3667, 6.5500])),
    (('2.5BG', 3.0, 18.0), np.array([0.0648, 0.3682, 6.5500])),
    (('2.5BG', 3.0, 20.0), np.array([0.0482, 0.3695, 6.5500])),
    (('2.5BG', 3.0, 22.0), np.array([0.0320, 0.3700, 6.5500])),
    (('2.5BG', 3.0, 24.0), np.array([0.0120, 0.3710, 6.5500])),
    (('5BG', 3.0, 2.0), np.array([0.2742, 0.3192, 6.5500])),
    (('5BG', 3.0, 4.0), np.array([0.2343, 0.3200, 6.5500])),
    (('5BG', 3.0, 6.0), np.array([0.2020, 0.3188, 6.5500])),
    (('5BG', 3.0, 8.0), np.array([0.1703, 0.3159, 6.5500])),
    (('5BG', 3.0, 10.0), np.array([0.1410, 0.3118, 6.5500])),
    (('5BG', 3.0, 12.0), np.array([0.1158, 0.3071, 6.5500])),
    (('5BG', 3.0, 14.0), np.array([0.0940, 0.3027, 6.5500])),
    (('5BG', 3.0, 16.0), np.array([0.0735, 0.2979, 6.5500])),
    (('5BG', 3.0, 18.0), np.array([0.0580, 0.2940, 6.5500])),
    (('5BG', 3.0, 20.0), np.array([0.0430, 0.2910, 6.5500])),
    (('5BG', 3.0, 22.0), np.array([0.0280, 0.2860, 6.5500])),
    (('7.5BG', 3.0, 2.0), np.array([0.2699, 0.3120, 6.5500])),
    (('7.5BG', 3.0, 4.0), np.array([0.2272, 0.3041, 6.5500])),
    (('7.5BG', 3.0, 6.0), np.array([0.1928, 0.2958, 6.5500])),
    (('7.5BG', 3.0, 8.0), np.array([0.1620, 0.2872, 6.5500])),
    (('7.5BG', 3.0, 10.0), np.array([0.1326, 0.2784, 6.5500])),
    (('7.5BG', 3.0, 12.0), np.array([0.1086, 0.2706, 6.5500])),
    (('7.5BG', 3.0, 14.0), np.array([0.0874, 0.2627, 6.5500])),
    (('7.5BG', 3.0, 16.0), np.array([0.0691, 0.2559, 6.5500])),
    (('7.5BG', 3.0, 18.0), np.array([0.0530, 0.2490, 6.5500])),
    (('7.5BG', 3.0, 20.0), np.array([0.0390, 0.2420, 6.5500])),
    (('10BG', 3.0, 2.0), np.array([0.2660, 0.3050, 6.5500])),
    (('10BG', 3.0, 4.0), np.array([0.2221, 0.2886, 6.5500])),
    (('10BG', 3.0, 6.0), np.array([0.1861, 0.2722, 6.5500])),
    (('10BG', 3.0, 8.0), np.array([0.1551, 0.2571, 6.5500])),
    (('10BG', 3.0, 10.0), np.array([0.1250, 0.2411, 6.5500])),
    (('10BG', 3.0, 12.0), np.array([0.1018, 0.2281, 6.5500])),
    (('10BG', 3.0, 14.0), np.array([0.0798, 0.2151, 6.5500])),
    (('10BG', 3.0, 16.0), np.array([0.0650, 0.2060, 6.5500])),
    (('10BG', 3.0, 18.0), np.array([0.0530, 0.1990, 6.5500])),
    (('2.5B', 3.0, 2.0), np.array([0.2636, 0.2983, 6.5500])),
    (('2.5B', 3.0, 4.0), np.array([0.2183, 0.2748, 6.5500])),
    (('2.5B', 3.0, 6.0), np.array([0.1826, 0.2536, 6.5500])),
    (('2.5B', 3.0, 8.0), np.array([0.1511, 0.2331, 6.5500])),
    (('2.5B', 3.0, 10.0), np.array([0.1220, 0.2132, 6.5500])),
    (('2.5B', 3.0, 12.0), np.array([0.0989, 0.1963, 6.5500])),
    (('2.5B', 3.0, 14.0), np.array([0.0800, 0.1800, 6.5500])),
    (('2.5B', 3.0, 16.0), np.array([0.0650, 0.1700, 6.5500])),
    (('5B', 3.0, 2.0), np.array([0.2617, 0.2921, 6.5500])),
    (('5B', 3.0, 4.0), np.array([0.2176, 0.2632, 6.5500])),
    (('5B', 3.0, 6.0), np.array([0.1835, 0.2375, 6.5500])),
    (('5B', 3.0, 8.0), np.array([0.1527, 0.2119, 6.5500])),
    (('5B', 3.0, 10.0), np.array([0.1259, 0.1879, 6.5500])),
    (('5B', 3.0, 12.0), np.array([0.1042, 0.1681, 6.5500])),
    (('5B', 3.0, 14.0), np.array([0.0860, 0.1500, 6.5500])),
    (('5B', 3.0, 16.0), np.array([0.0710, 0.1370, 6.5500])),
    (('7.5B', 3.0, 2.0), np.array([0.2616, 0.2857, 6.5500])),
    (('7.5B', 3.0, 4.0), np.array([0.2200, 0.2536, 6.5500])),
    (('7.5B', 3.0, 6.0), np.array([0.1875, 0.2258, 6.5500])),
    (('7.5B', 3.0, 8.0), np.array([0.1583, 0.1987, 6.5500])),
    (('7.5B', 3.0, 10.0), np.array([0.1343, 0.1756, 6.5500])),
    (('7.5B', 3.0, 12.0), np.array([0.1131, 0.1542, 6.5500])),
    (('7.5B', 3.0, 14.0), np.array([0.0950, 0.1360, 6.5500])),
    (('7.5B', 3.0, 16.0), np.array([0.0830, 0.1230, 6.5500])),
    (('10B', 3.0, 2.0), np.array([0.2631, 0.2801, 6.5500])),
    (('10B', 3.0, 4.0), np.array([0.2246, 0.2467, 6.5500])),
    (('10B', 3.0, 6.0), np.array([0.1933, 0.2173, 6.5500])),
    (('10B', 3.0, 8.0), np.array([0.1658, 0.1905, 6.5500])),
    (('10B', 3.0, 10.0), np.array([0.1432, 0.1675, 6.5500])),
    (('10B', 3.0, 12.0), np.array([0.1228, 0.1460, 6.5500])),
    (('10B', 3.0, 14.0), np.array([0.1065, 0.1285, 6.5500])),
    (('10B', 3.0, 16.0), np.array([0.0950, 0.1150, 6.5500])),
    (('10B', 3.0, 18.0), np.array([0.0840, 0.1000, 6.5500])),
    (('2.5PB', 3.0, 2.0), np.array([0.2663, 0.2756, 6.5500])),
    (('2.5PB', 3.0, 4.0), np.array([0.2312, 0.2405, 6.5500])),
    (('2.5PB', 3.0, 6.0), np.array([0.2022, 0.2101, 6.5500])),
    (('2.5PB', 3.0, 8.0), np.array([0.1780, 0.1833, 6.5500])),
    (('2.5PB', 3.0, 10.0), np.array([0.1576, 0.1600, 6.5500])),
    (('2.5PB', 3.0, 12.0), np.array([0.1398, 0.1395, 6.5500])),
    (('2.5PB', 3.0, 14.0), np.array([0.1251, 0.1218, 6.5500])),
    (('2.5PB', 3.0, 16.0), np.array([0.1130, 0.1070, 6.5500])),
    (('2.5PB', 3.0, 18.0), np.array([0.1020, 0.0930, 6.5500])),
    (('2.5PB', 3.0, 20.0), np.array([0.0950, 0.0830, 6.5500])),
    (('2.5PB', 3.0, 22.0), np.array([0.0880, 0.0730, 6.5500])),
    (('5PB', 3.0, 2.0), np.array([0.2708, 0.2719, 6.5500])),
    (('5PB', 3.0, 4.0), np.array([0.2393, 0.2361, 6.5500])),
    (('5PB', 3.0, 6.0), np.array([0.2122, 0.2052, 6.5500])),
    (('5PB', 3.0, 8.0), np.array([0.1908, 0.1799, 6.5500])),
    (('5PB', 3.0, 10.0), np.array([0.1718, 0.1562, 6.5500])),
    (('5PB', 3.0, 12.0), np.array([0.1557, 0.1356, 6.5500])),
    (('5PB', 3.0, 14.0), np.array([0.1431, 0.1184, 6.5500])),
    (('5PB', 3.0, 16.0), np.array([0.1318, 0.1024, 6.5500])),
    (('5PB', 3.0, 18.0), np.array([0.1228, 0.0895, 6.5500])),
    (('5PB', 3.0, 20.0), np.array([0.1170, 0.0800, 6.5500])),
    (('5PB', 3.0, 22.0), np.array([0.1120, 0.0720, 6.5500])),
    (('5PB', 3.0, 24.0), np.array([0.1070, 0.0640, 6.5500])),
    (('5PB', 3.0, 26.0), np.array([0.1040, 0.0590, 6.5500])),
    (('5PB', 3.0, 28.0), np.array([0.1020, 0.0540, 6.5500])),
    (('5PB', 3.0, 30.0), np.array([0.0990, 0.0500, 6.5500])),
    (('5PB', 3.0, 32.0), np.array([0.0970, 0.0450, 6.5500])),
    (('5PB', 3.0, 34.0), np.array([0.0950, 0.0420, 6.5500])),
    (('5PB', 3.0, 36.0), np.array([0.0940, 0.0400, 6.5500])),
    (('5PB', 3.0, 38.0), np.array([0.0920, 0.0370, 6.5500])),
    (('5PB', 3.0, 40.0), np.array([0.0910, 0.0340, 6.5500])),
    (('5PB', 3.0, 42.0), np.array([0.0900, 0.0320, 6.5500])),
    (('5PB', 3.0, 44.0), np.array([0.0890, 0.0300, 6.5500])),
    (('5PB', 3.0, 46.0), np.array([0.0880, 0.0280, 6.5500])),
    (('5PB', 3.0, 48.0), np.array([0.0870, 0.0250, 6.5500])),
    (('5PB', 3.0, 50.0), np.array([0.0860, 0.0220, 6.5500])),
    (('7.5PB', 3.0, 2.0), np.array([0.2777, 0.2687, 6.5500])),
    (('7.5PB', 3.0, 4.0), np.array([0.2520, 0.2319, 6.5500])),
    (('7.5PB', 3.0, 6.0), np.array([0.2311, 0.2010, 6.5500])),
    (('7.5PB', 3.0, 8.0), np.array([0.2149, 0.1761, 6.5500])),
    (('7.5PB', 3.0, 10.0), np.array([0.2005, 0.1536, 6.5500])),
    (('7.5PB', 3.0, 12.0), np.array([0.1903, 0.1353, 6.5500])),
    (('7.5PB', 3.0, 14.0), np.array([0.1824, 0.1188, 6.5500])),
    (('7.5PB', 3.0, 16.0), np.array([0.1765, 0.1048, 6.5500])),
    (('7.5PB', 3.0, 18.0), np.array([0.1730, 0.0948, 6.5500])),
    (('7.5PB', 3.0, 20.0), np.array([0.1702, 0.0867, 6.5500])),
    (('7.5PB', 3.0, 22.0), np.array([0.1677, 0.0782, 6.5500])),
    (('7.5PB', 3.0, 24.0), np.array([0.1658, 0.0711, 6.5500])),
    (('7.5PB', 3.0, 26.0), np.array([0.1642, 0.0655, 6.5500])),
    (('7.5PB', 3.0, 28.0), np.array([0.1632, 0.0609, 6.5500])),
    (('7.5PB', 3.0, 30.0), np.array([0.1621, 0.0556, 6.5500])),
    (('7.5PB', 3.0, 32.0), np.array([0.1612, 0.0511, 6.5500])),
    (('7.5PB', 3.0, 34.0), np.array([0.1608, 0.0480, 6.5500])),
    (('7.5PB', 3.0, 36.0), np.array([0.1590, 0.0440, 6.5500])),
    (('7.5PB', 3.0, 38.0), np.array([0.1580, 0.0400, 6.5500])),
    (('7.5PB', 3.0, 40.0), np.array([0.1580, 0.0370, 6.5500])),
    (('7.5PB', 3.0, 42.0), np.array([0.1570, 0.0340, 6.5500])),
    (('7.5PB', 3.0, 44.0), np.array([0.1570, 0.0310, 6.5500])),
    (('7.5PB', 3.0, 46.0), np.array([0.1570, 0.0280, 6.5500])),
    (('7.5PB', 3.0, 48.0), np.array([0.1560, 0.0250, 6.5500])),
    (('7.5PB', 3.0, 50.0), np.array([0.1560, 0.0220, 6.5500])),
    (('10PB', 3.0, 2.0), np.array([0.2847, 0.2670, 6.5500])),
    (('10PB', 3.0, 4.0), np.array([0.2660, 0.2319, 6.5500])),
    (('10PB', 3.0, 6.0), np.array([0.2511, 0.2031, 6.5500])),
    (('10PB', 3.0, 8.0), np.array([0.2387, 0.1786, 6.5500])),
    (('10PB', 3.0, 10.0), np.array([0.2278, 0.1565, 6.5500])),
    (('10PB', 3.0, 12.0), np.array([0.2206, 0.1407, 6.5500])),
    (('10PB', 3.0, 14.0), np.array([0.2142, 0.1250, 6.5500])),
    (('10PB', 3.0, 16.0), np.array([0.2092, 0.1118, 6.5500])),
    (('10PB', 3.0, 18.0), np.array([0.2060, 0.1020, 6.5500])),
    (('10PB', 3.0, 20.0), np.array([0.2030, 0.0930, 6.5500])),
    (('10PB', 3.0, 22.0), np.array([0.2004, 0.0847, 6.5500])),
    (('10PB', 3.0, 24.0), np.array([0.1982, 0.0772, 6.5500])),
    (('10PB', 3.0, 26.0), np.array([0.1963, 0.0708, 6.5500])),
    (('10PB', 3.0, 28.0), np.array([0.1950, 0.0650, 6.5500])),
    (('10PB', 3.0, 30.0), np.array([0.1938, 0.0599, 6.5500])),
    (('10PB', 3.0, 32.0), np.array([0.1926, 0.0542, 6.5500])),
    (('10PB', 3.0, 34.0), np.array([0.1918, 0.0503, 6.5500])),
    (('10PB', 3.0, 36.0), np.array([0.1900, 0.0460, 6.5500])),
    (('10PB', 3.0, 38.0), np.array([0.1900, 0.0420, 6.5500])),
    (('10PB', 3.0, 40.0), np.array([0.1890, 0.0380, 6.5500])),
    (('10PB', 3.0, 42.0), np.array([0.1890, 0.0340, 6.5500])),
    (('10PB', 3.0, 44.0), np.array([0.1880, 0.0310, 6.5500])),
    (('10PB', 3.0, 46.0), np.array([0.1880, 0.0280, 6.5500])),
    (('10PB', 3.0, 48.0), np.array([0.1880, 0.0250, 6.5500])),
    (('10PB', 3.0, 50.0), np.array([0.1880, 0.0220, 6.5500])),
    (('2.5P', 3.0, 2.0), np.array([0.2922, 0.2680, 6.5500])),
    (('2.5P', 3.0, 4.0), np.array([0.2792, 0.2342, 6.5500])),
    (('2.5P', 3.0, 6.0), np.array([0.2691, 0.2072, 6.5500])),
    (('2.5P', 3.0, 8.0), np.array([0.2615, 0.1845, 6.5500])),
    (('2.5P', 3.0, 10.0), np.array([0.2548, 0.1638, 6.5500])),
    (('2.5P', 3.0, 12.0), np.array([0.2498, 0.1480, 6.5500])),
    (('2.5P', 3.0, 14.0), np.array([0.2449, 0.1325, 6.5500])),
    (('2.5P', 3.0, 16.0), np.array([0.2410, 0.1198, 6.5500])),
    (('2.5P', 3.0, 18.0), np.array([0.2380, 0.1094, 6.5500])),
    (('2.5P', 3.0, 20.0), np.array([0.2354, 0.1003, 6.5500])),
    (('2.5P', 3.0, 22.0), np.array([0.2329, 0.0911, 6.5500])),
    (('2.5P', 3.0, 24.0), np.array([0.2305, 0.0832, 6.5500])),
    (('2.5P', 3.0, 26.0), np.array([0.2286, 0.0765, 6.5500])),
    (('2.5P', 3.0, 28.0), np.array([0.2268, 0.0698, 6.5500])),
    (('2.5P', 3.0, 30.0), np.array([0.2252, 0.0638, 6.5500])),
    (('2.5P', 3.0, 32.0), np.array([0.2242, 0.0587, 6.5500])),
    (('2.5P', 3.0, 34.0), np.array([0.2230, 0.0543, 6.5500])),
    (('2.5P', 3.0, 36.0), np.array([0.2220, 0.0480, 6.5500])),
    (('2.5P', 3.0, 38.0), np.array([0.2210, 0.0440, 6.5500])),
    (('2.5P', 3.0, 40.0), np.array([0.2200, 0.0400, 6.5500])),
    (('2.5P', 3.0, 42.0), np.array([0.2200, 0.0350, 6.5500])),
    (('2.5P', 3.0, 44.0), np.array([0.2190, 0.0320, 6.5500])),
    (('2.5P', 3.0, 46.0), np.array([0.2180, 0.0280, 6.5500])),
    (('5P', 3.0, 2.0), np.array([0.2997, 0.2700, 6.5500])),
    (('5P', 3.0, 4.0), np.array([0.2928, 0.2386, 6.5500])),
    (('5P', 3.0, 6.0), np.array([0.2870, 0.2135, 6.5500])),
    (('5P', 3.0, 8.0), np.array([0.2819, 0.1910, 6.5500])),
    (('5P', 3.0, 10.0), np.array([0.2772, 0.1707, 6.5500])),
    (('5P', 3.0, 12.0), np.array([0.2739, 0.1539, 6.5500])),
    (('5P', 3.0, 14.0), np.array([0.2707, 0.1397, 6.5500])),
    (('5P', 3.0, 16.0), np.array([0.2680, 0.1272, 6.5500])),
    (('5P', 3.0, 18.0), np.array([0.2657, 0.1163, 6.5500])),
    (('5P', 3.0, 20.0), np.array([0.2639, 0.1074, 6.5500])),
    (('5P', 3.0, 22.0), np.array([0.2620, 0.0978, 6.5500])),
    (('5P', 3.0, 24.0), np.array([0.2602, 0.0891, 6.5500])),
    (('5P', 3.0, 26.0), np.array([0.2590, 0.0822, 6.5500])),
    (('5P', 3.0, 28.0), np.array([0.2579, 0.0750, 6.5500])),
    (('5P', 3.0, 30.0), np.array([0.2568, 0.0690, 6.5500])),
    (('5P', 3.0, 32.0), np.array([0.2557, 0.0630, 6.5500])),
    (('5P', 3.0, 34.0), np.array([0.2550, 0.0560, 6.5500])),
    (('5P', 3.0, 36.0), np.array([0.2540, 0.0510, 6.5500])),
    (('5P', 3.0, 38.0), np.array([0.2530, 0.0460, 6.5500])),
    (('5P', 3.0, 40.0), np.array([0.2520, 0.0410, 6.5500])),
    (('5P', 3.0, 42.0), np.array([0.2510, 0.0360, 6.5500])),
    (('7.5P', 3.0, 2.0), np.array([0.3088, 0.2740, 6.5500])),
    (('7.5P', 3.0, 4.0), np.array([0.3072, 0.2448, 6.5500])),
    (('7.5P', 3.0, 6.0), np.array([0.3057, 0.2208, 6.5500])),
    (('7.5P', 3.0, 8.0), np.array([0.3037, 0.1981, 6.5500])),
    (('7.5P', 3.0, 10.0), np.array([0.3020, 0.1794, 6.5500])),
    (('7.5P', 3.0, 12.0), np.array([0.3003, 0.1618, 6.5500])),
    (('7.5P', 3.0, 14.0), np.array([0.2992, 0.1475, 6.5500])),
    (('7.5P', 3.0, 16.0), np.array([0.2981, 0.1356, 6.5500])),
    (('7.5P', 3.0, 18.0), np.array([0.2969, 0.1239, 6.5500])),
    (('7.5P', 3.0, 20.0), np.array([0.2961, 0.1151, 6.5500])),
    (('7.5P', 3.0, 22.0), np.array([0.2953, 0.1057, 6.5500])),
    (('7.5P', 3.0, 24.0), np.array([0.2944, 0.0967, 6.5500])),
    (('7.5P', 3.0, 26.0), np.array([0.2938, 0.0892, 6.5500])),
    (('7.5P', 3.0, 28.0), np.array([0.2930, 0.0812, 6.5500])),
    (('7.5P', 3.0, 30.0), np.array([0.2922, 0.0750, 6.5500])),
    (('7.5P', 3.0, 32.0), np.array([0.2920, 0.0670, 6.5500])),
    (('7.5P', 3.0, 34.0), np.array([0.2910, 0.0600, 6.5500])),
    (('7.5P', 3.0, 36.0), np.array([0.2900, 0.0540, 6.5500])),
    (('7.5P', 3.0, 38.0), np.array([0.2890, 0.0480, 6.5500])),
    (('10P', 3.0, 2.0), np.array([0.3170, 0.2790, 6.5500])),
    (('10P', 3.0, 4.0), np.array([0.3214, 0.2517, 6.5500])),
    (('10P', 3.0, 6.0), np.array([0.3243, 0.2293, 6.5500])),
    (('10P', 3.0, 8.0), np.array([0.3269, 0.2075, 6.5500])),
    (('10P', 3.0, 10.0), np.array([0.3286, 0.1889, 6.5500])),
    (('10P', 3.0, 12.0), np.array([0.3301, 0.1715, 6.5500])),
    (('10P', 3.0, 14.0), np.array([0.3309, 0.1572, 6.5500])),
    (('10P', 3.0, 16.0), np.array([0.3320, 0.1456, 6.5500])),
    (('10P', 3.0, 18.0), np.array([0.3329, 0.1332, 6.5500])),
    (('10P', 3.0, 20.0), np.array([0.3332, 0.1240, 6.5500])),
    (('10P', 3.0, 22.0), np.array([0.3340, 0.1146, 6.5500])),
    (('10P', 3.0, 24.0), np.array([0.3341, 0.1055, 6.5500])),
    (('10P', 3.0, 26.0), np.array([0.3343, 0.0978, 6.5500])),
    (('10P', 3.0, 28.0), np.array([0.3350, 0.0880, 6.5500])),
    (('10P', 3.0, 30.0), np.array([0.3350, 0.0810, 6.5500])),
    (('10P', 3.0, 32.0), np.array([0.3350, 0.0720, 6.5500])),
    (('10P', 3.0, 34.0), np.array([0.3350, 0.0630, 6.5500])),
    (('2.5RP', 3.0, 2.0), np.array([0.3272, 0.2861, 6.5500])),
    (('2.5RP', 3.0, 4.0), np.array([0.3400, 0.2624, 6.5500])),
    (('2.5RP', 3.0, 6.0), np.array([0.3501, 0.2425, 6.5500])),
    (('2.5RP', 3.0, 8.0), np.array([0.3598, 0.2233, 6.5500])),
    (('2.5RP', 3.0, 10.0), np.array([0.3681, 0.2054, 6.5500])),
    (('2.5RP', 3.0, 12.0), np.array([0.3754, 0.1898, 6.5500])),
    (('2.5RP', 3.0, 14.0), np.array([0.3818, 0.1758, 6.5500])),
    (('2.5RP', 3.0, 16.0), np.array([0.3876, 0.1629, 6.5500])),
    (('2.5RP', 3.0, 18.0), np.array([0.3929, 0.1506, 6.5500])),
    (('2.5RP', 3.0, 20.0), np.array([0.3969, 0.1413, 6.5500])),
    (('2.5RP', 3.0, 22.0), np.array([0.4018, 0.1304, 6.5500])),
    (('2.5RP', 3.0, 24.0), np.array([0.4050, 0.1220, 6.5500])),
    (('2.5RP', 3.0, 26.0), np.array([0.4080, 0.1140, 6.5500])),
    (('2.5RP', 3.0, 28.0), np.array([0.4140, 0.1020, 6.5500])),
    (('2.5RP', 3.0, 30.0), np.array([0.4170, 0.0940, 6.5500])),
    (('5RP', 3.0, 2.0), np.array([0.3370, 0.2940, 6.5500])),
    (('5RP', 3.0, 4.0), np.array([0.3586, 0.2742, 6.5500])),
    (('5RP', 3.0, 6.0), np.array([0.3765, 0.2569, 6.5500])),
    (('5RP', 3.0, 8.0), np.array([0.3930, 0.2395, 6.5500])),
    (('5RP', 3.0, 10.0), np.array([0.4073, 0.2235, 6.5500])),
    (('5RP', 3.0, 12.0), np.array([0.4199, 0.2089, 6.5500])),
    (('5RP', 3.0, 14.0), np.array([0.4313, 0.1944, 6.5500])),
    (('5RP', 3.0, 16.0), np.array([0.4418, 0.1809, 6.5500])),
    (('5RP', 3.0, 18.0), np.array([0.4503, 0.1695, 6.5500])),
    (('5RP', 3.0, 20.0), np.array([0.4577, 0.1593, 6.5500])),
    (('5RP', 3.0, 22.0), np.array([0.4670, 0.1450, 6.5500])),
    (('5RP', 3.0, 24.0), np.array([0.4720, 0.1360, 6.5500])),
    (('5RP', 3.0, 26.0), np.array([0.4790, 0.1270, 6.5500])),
    (('7.5RP', 3.0, 2.0), np.array([0.3450, 0.3001, 6.5500])),
    (('7.5RP', 3.0, 4.0), np.array([0.3739, 0.2851, 6.5500])),
    (('7.5RP', 3.0, 6.0), np.array([0.3990, 0.2708, 6.5500])),
    (('7.5RP', 3.0, 8.0), np.array([0.4234, 0.2556, 6.5500])),
    (('7.5RP', 3.0, 10.0), np.array([0.4445, 0.2419, 6.5500])),
    (('7.5RP', 3.0, 12.0), np.array([0.4654, 0.2273, 6.5500])),
    (('7.5RP', 3.0, 14.0), np.array([0.4831, 0.2140, 6.5500])),
    (('7.5RP', 3.0, 16.0), np.array([0.4991, 0.2011, 6.5500])),
    (('7.5RP', 3.0, 18.0), np.array([0.5130, 0.1893, 6.5500])),
    (('7.5RP', 3.0, 20.0), np.array([0.5280, 0.1780, 6.5500])),
    (('7.5RP', 3.0, 22.0), np.array([0.5420, 0.1650, 6.5500])),
    (('10RP', 3.0, 2.0), np.array([0.3526, 0.3068, 6.5500])),
    (('10RP', 3.0, 4.0), np.array([0.3889, 0.2969, 6.5500])),
    (('10RP', 3.0, 6.0), np.array([0.4218, 0.2864, 6.5500])),
    (('10RP', 3.0, 8.0), np.array([0.4552, 0.2741, 6.5500])),
    (('10RP', 3.0, 10.0), np.array([0.4851, 0.2618, 6.5500])),
    (('10RP', 3.0, 12.0), np.array([0.5139, 0.2489, 6.5500])),
    (('10RP', 3.0, 14.0), np.array([0.5380, 0.2369, 6.5500])),
    (('10RP', 3.0, 16.0), np.array([0.5628, 0.2241, 6.5500])),
    (('10RP', 3.0, 18.0), np.array([0.5840, 0.2120, 6.5500])),
    (('10RP', 3.0, 20.0), np.array([0.6020, 0.2000, 6.5500])),
    (('2.5R', 3.0, 2.0), np.array([0.3591, 0.3130, 6.5500])),
    (('2.5R', 3.0, 4.0), np.array([0.4021, 0.3076, 6.5500])),
    (('2.5R', 3.0, 6.0), np.array([0.4409, 0.3009, 6.5500])),
    (('2.5R', 3.0, 8.0), np.array([0.4821, 0.2918, 6.5500])),
    (('2.5R', 3.0, 10.0), np.array([0.5191, 0.2811, 6.5500])),
    (('2.5R', 3.0, 12.0), np.array([0.5536, 0.2691, 6.5500])),
    (('2.5R', 3.0, 14.0), np.array([0.5828, 0.2579, 6.5500])),
    (('2.5R', 3.0, 16.0), np.array([0.6116, 0.2456, 6.5500])),
    (('2.5R', 3.0, 18.0), np.array([0.6400, 0.2320, 6.5500])),
    (('2.5R', 3.0, 20.0), np.array([0.6670, 0.2170, 6.5500])),
    (('5R', 3.0, 2.0), np.array([0.3645, 0.3190, 6.5500])),
    (('5R', 3.0, 4.0), np.array([0.4148, 0.3190, 6.5500])),
    (('5R', 3.0, 6.0), np.array([0.4592, 0.3168, 6.5500])),
    (('5R', 3.0, 8.0), np.array([0.5064, 0.3114, 6.5500])),
    (('5R', 3.0, 10.0), np.array([0.5500, 0.3024, 6.5500])),
    (('5R', 3.0, 12.0), np.array([0.5884, 0.2904, 6.5500])),
    (('5R', 3.0, 14.0), np.array([0.6204, 0.2789, 6.5500])),
    (('5R', 3.0, 16.0), np.array([0.6520, 0.2660, 6.5500])),
    (('5R', 3.0, 18.0), np.array([0.6820, 0.2510, 6.5500])),
    (('5R', 3.0, 20.0), np.array([0.7100, 0.2340, 6.5500])),
    (('7.5R', 3.0, 2.0), np.array([0.3690, 0.3248, 6.5500])),
    (('7.5R', 3.0, 4.0), np.array([0.4240, 0.3302, 6.5500])),
    (('7.5R', 3.0, 6.0), np.array([0.4738, 0.3316, 6.5500])),
    (('7.5R', 3.0, 8.0), np.array([0.5251, 0.3297, 6.5500])),
    (('7.5R', 3.0, 10.0), np.array([0.5730, 0.3240, 6.5500])),
    (('7.5R', 3.0, 12.0), np.array([0.6158, 0.3129, 6.5500])),
    (('7.5R', 3.0, 14.0), np.array([0.6492, 0.3012, 6.5500])),
    (('7.5R', 3.0, 16.0), np.array([0.6817, 0.2872, 6.5500])),
    (('7.5R', 3.0, 18.0), np.array([0.7140, 0.2710, 6.5500])),
    (('7.5R', 3.0, 20.0), np.array([0.7470, 0.2510, 6.5500])),
    (('10R', 3.0, 2.0), np.array([0.3728, 0.3314, 6.5500])),
    (('10R', 3.0, 4.0), np.array([0.4308, 0.3412, 6.5500])),
    (('10R', 3.0, 6.0), np.array([0.4854, 0.3467, 6.5500])),
    (('10R', 3.0, 8.0), np.array([0.5393, 0.3477, 6.5500])),
    (('10R', 3.0, 10.0), np.array([0.5871, 0.3440, 6.5500])),
    (('10R', 3.0, 12.0), np.array([0.6322, 0.3361, 6.5500])),
    (('10R', 3.0, 14.0), np.array([0.6703, 0.3249, 6.5500])),
    (('10R', 3.0, 16.0), np.array([0.7030, 0.3140, 6.5500])),
    (('10R', 3.0, 18.0), np.array([0.7390, 0.3020, 6.5500])),
    (('10R', 3.0, 20.0), np.array([0.7780, 0.2860, 6.5500])),
    (('2.5YR', 3.0, 2.0), np.array([0.3757, 0.3391, 6.5500])),
    (('2.5YR', 3.0, 4.0), np.array([0.4360, 0.3563, 6.5500])),
    (('2.5YR', 3.0, 6.0), np.array([0.4954, 0.3692, 6.5500])),
    (('2.5YR', 3.0, 8.0), np.array([0.5475, 0.3771, 6.5500])),
    (('2.5YR', 3.0, 10.0), np.array([0.5941, 0.3818, 6.5500])),
    (('2.5YR', 3.0, 12.0), np.array([0.6370, 0.3810, 6.5500])),
    (('2.5YR', 3.0, 14.0), np.array([0.6740, 0.3790, 6.5500])),
    (('2.5YR', 3.0, 16.0), np.array([0.7080, 0.3740, 6.5500])),
    (('5YR', 3.0, 2.0), np.array([0.3771, 0.3476, 6.5500])),
    (('5YR', 3.0, 4.0), np.array([0.4376, 0.3715, 6.5500])),
    (('5YR', 3.0, 6.0), np.array([0.4966, 0.3908, 6.5500])),
    (('5YR', 3.0, 8.0), np.array([0.5456, 0.4040, 6.5500])),
    (('5YR', 3.0, 10.0), np.array([0.5900, 0.4140, 6.5500])),
    (('5YR', 3.0, 12.0), np.array([0.6290, 0.4230, 6.5500])),
    (('7.5YR', 3.0, 2.0), np.array([0.3771, 0.3549, 6.5500])),
    (('7.5YR', 3.0, 4.0), np.array([0.4378, 0.3865, 6.5500])),
    (('7.5YR', 3.0, 6.0), np.array([0.4930, 0.4116, 6.5500])),
    (('7.5YR', 3.0, 8.0), np.array([0.5390, 0.4306, 6.5500])),
    (('7.5YR', 3.0, 10.0), np.array([0.5810, 0.4480, 6.5500])),
    (('10YR', 3.0, 2.0), np.array([0.3747, 0.3630, 6.5500])),
    (('10YR', 3.0, 4.0), np.array([0.4341, 0.4018, 6.5500])),
    (('10YR', 3.0, 6.0), np.array([0.4872, 0.4326, 6.5500])),
    (('10YR', 3.0, 8.0), np.array([0.5305, 0.4559, 6.5500])),
    (('10YR', 3.0, 10.0), np.array([0.5720, 0.4750, 6.5500])),
    (('2.5Y', 3.0, 2.0), np.array([0.3703, 0.3700, 6.5500])),
    (('2.5Y', 3.0, 4.0), np.array([0.4277, 0.4166, 6.5500])),
    (('2.5Y', 3.0, 6.0), np.array([0.4784, 0.4531, 6.5500])),
    (('2.5Y', 3.0, 8.0), np.array([0.5210, 0.4820, 6.5500])),
    (('2.5Y', 3.0, 10.0), np.array([0.5600, 0.5050, 6.5500])),
    (('5Y', 3.0, 2.0), np.array([0.3646, 0.3748, 6.5500])),
    (('5Y', 3.0, 4.0), np.array([0.4191, 0.4283, 6.5500])),
    (('5Y', 3.0, 6.0), np.array([0.4670, 0.4711, 6.5500])),
    (('5Y', 3.0, 8.0), np.array([0.5090, 0.5090, 6.5500])),
    (('7.5Y', 3.0, 2.0), np.array([0.3589, 0.3778, 6.5500])),
    (('7.5Y', 3.0, 4.0), np.array([0.4086, 0.4379, 6.5500])),
    (('7.5Y', 3.0, 6.0), np.array([0.4526, 0.4889, 6.5500])),
    (('7.5Y', 3.0, 8.0), np.array([0.4920, 0.5350, 6.5500])),
    (('10Y', 4.0, 2.0), np.array([0.3476, 0.3732, 12.0000])),
    (('10Y', 4.0, 4.0), np.array([0.3871, 0.4321, 12.0000])),
    (('10Y', 4.0, 6.0), np.array([0.4190, 0.4795, 12.0000])),
    (('10Y', 4.0, 8.0), np.array([0.4430, 0.5153, 12.0000])),
    (('10Y', 4.0, 10.0), np.array([0.4620, 0.5430, 12.0000])),
    (('10Y', 4.0, 12.0), np.array([0.4730, 0.5620, 12.0000])),
    (('2.5GY', 4.0, 2.0), np.array([0.3382, 0.3706, 12.0000])),
    (('2.5GY', 4.0, 4.0), np.array([0.3708, 0.4329, 12.0000])),
    (('2.5GY', 4.0, 6.0), np.array([0.3968, 0.4857, 12.0000])),
    (('2.5GY', 4.0, 8.0), np.array([0.4174, 0.5300, 12.0000])),
    (('2.5GY', 4.0, 10.0), np.array([0.4330, 0.5680, 12.0000])),
    (('2.5GY', 4.0, 12.0), np.array([0.4430, 0.5940, 12.0000])),
    (('5GY', 4.0, 2.0), np.array([0.3312, 0.3678, 12.0000])),
    (('5GY', 4.0, 4.0), np.array([0.3538, 0.4284, 12.0000])),
    (('5GY', 4.0, 6.0), np.array([0.3718, 0.4852, 12.0000])),
    (('5GY', 4.0, 8.0), np.array([0.3868, 0.5384, 12.0000])),
    (('5GY', 4.0, 10.0), np.array([0.3983, 0.5850, 12.0000])),
    (('5GY', 4.0, 12.0), np.array([0.4070, 0.6190, 12.0000])),
    (('5GY', 4.0, 14.0), np.array([0.4150, 0.6590, 12.0000])),
    (('7.5GY', 4.0, 2.0), np.array([0.3185, 0.3604, 12.0000])),
    (('7.5GY', 4.0, 4.0), np.array([0.3281, 0.4157, 12.0000])),
    (('7.5GY', 4.0, 6.0), np.array([0.3355, 0.4739, 12.0000])),
    (('7.5GY', 4.0, 8.0), np.array([0.3400, 0.5348, 12.0000])),
    (('7.5GY', 4.0, 10.0), np.array([0.3395, 0.5913, 12.0000])),
    (('7.5GY', 4.0, 12.0), np.array([0.3348, 0.6468, 12.0000])),
    (('7.5GY', 4.0, 14.0), np.array([0.3270, 0.6980, 12.0000])),
    (('7.5GY', 4.0, 16.0), np.array([0.3150, 0.7570, 12.0000])),
    (('7.5GY', 4.0, 18.0), np.array([0.3030, 0.8090, 12.0000])),
    (('10GY', 4.0, 2.0), np.array([0.3109, 0.3550, 12.0000])),
    (('10GY', 4.0, 4.0), np.array([0.3100, 0.4018, 12.0000])),
    (('10GY', 4.0, 6.0), np.array([0.3069, 0.4550, 12.0000])),
    (('10GY', 4.0, 8.0), np.array([0.3008, 0.5095, 12.0000])),
    (('10GY', 4.0, 10.0), np.array([0.2908, 0.5672, 12.0000])),
    (('10GY', 4.0, 12.0), np.array([0.2758, 0.6282, 12.0000])),
    (('10GY', 4.0, 14.0), np.array([0.2590, 0.6858, 12.0000])),
    (('10GY', 4.0, 16.0), np.array([0.2422, 0.7360, 12.0000])),
    (('10GY', 4.0, 18.0), np.array([0.2210, 0.7930, 12.0000])),
    (('10GY', 4.0, 20.0), np.array([0.1920, 0.8600, 12.0000])),
    (('10GY', 4.0, 22.0), np.array([0.1650, 0.9170, 12.0000])),
    (('10GY', 4.0, 24.0), np.array([0.1420, 0.9640, 12.0000])),
    (('10GY', 4.0, 26.0), np.array([0.1060, 1.0280, 12.0000])),
    (('10GY', 4.0, 28.0), np.array([0.0580, 1.1000, 12.0000])),
    (('2.5G', 4.0, 2.0), np.array([0.3012, 0.3470, 12.0000])),
    (('2.5G', 4.0, 4.0), np.array([0.2891, 0.3821, 12.0000])),
    (('2.5G', 4.0, 6.0), np.array([0.2735, 0.4215, 12.0000])),
    (('2.5G', 4.0, 8.0), np.array([0.2561, 0.4597, 12.0000])),
    (('2.5G', 4.0, 10.0), np.array([0.2355, 0.5006, 12.0000])),
    (('2.5G', 4.0, 12.0), np.array([0.2128, 0.5425, 12.0000])),
    (('2.5G', 4.0, 14.0), np.array([0.1909, 0.5779, 12.0000])),
    (('2.5G', 4.0, 16.0), np.array([0.1682, 0.6111, 12.0000])),
    (('2.5G', 4.0, 18.0), np.array([0.1446, 0.6431, 12.0000])),
    (('2.5G', 4.0, 20.0), np.array([0.1230, 0.6706, 12.0000])),
    (('2.5G', 4.0, 22.0), np.array([0.1009, 0.6975, 12.0000])),
    (('2.5G', 4.0, 24.0), np.array([0.0760, 0.7250, 12.0000])),
    (('2.5G', 4.0, 26.0), np.array([0.0528, 0.7502, 12.0000])),
    (('2.5G', 4.0, 28.0), np.array([0.0280, 0.7800, 12.0000])),
    (('2.5G', 4.0, 30.0), np.array([-0.0050, 0.8160, 12.0000])),
    (('5G', 4.0, 2.0), np.array([0.2959, 0.3417, 12.0000])),
    (('5G', 4.0, 4.0), np.array([0.2781, 0.3704, 12.0000])),
    (('5G', 4.0, 6.0), np.array([0.2581, 0.3992, 12.0000])),
    (('5G', 4.0, 8.0), np.array([0.2359, 0.4266, 12.0000])),
    (('5G', 4.0, 10.0), np.array([0.2115, 0.4532, 12.0000])),
    (('5G', 4.0, 12.0), np.array([0.1843, 0.4807, 12.0000])),
    (('5G', 4.0, 14.0), np.array([0.1627, 0.5015, 12.0000])),
    (('5G', 4.0, 16.0), np.array([0.1402, 0.5214, 12.0000])),
    (('5G', 4.0, 18.0), np.array([0.1188, 0.5400, 12.0000])),
    (('5G', 4.0, 20.0), np.array([0.1018, 0.5543, 12.0000])),
    (('5G', 4.0, 22.0), np.array([0.0841, 0.5684, 12.0000])),
    (('5G', 4.0, 24.0), np.array([0.0614, 0.5857, 12.0000])),
    (('5G', 4.0, 26.0), np.array([0.0407, 0.6010, 12.0000])),
    (('5G', 4.0, 28.0), np.array([0.0200, 0.6180, 12.0000])),
    (('5G', 4.0, 30.0), np.array([-0.0030, 0.6320, 12.0000])),
    (('7.5G', 4.0, 2.0), np.array([0.2919, 0.3371, 12.0000])),
    (('7.5G', 4.0, 4.0), np.array([0.2702, 0.3602, 12.0000])),
    (('7.5G', 4.0, 6.0), np.array([0.2467, 0.3822, 12.0000])),
    (('7.5G', 4.0, 8.0), np.array([0.2232, 0.4022, 12.0000])),
    (('7.5G', 4.0, 10.0), np.array([0.1989, 0.4219, 12.0000])),
    (('7.5G', 4.0, 12.0), np.array([0.1706, 0.4419, 12.0000])),
    (('7.5G', 4.0, 14.0), np.array([0.1500, 0.4562, 12.0000])),
    (('7.5G', 4.0, 16.0), np.array([0.1293, 0.4703, 12.0000])),
    (('7.5G', 4.0, 18.0), np.array([0.1086, 0.4842, 12.0000])),
    (('7.5G', 4.0, 20.0), np.array([0.0928, 0.4942, 12.0000])),
    (('7.5G', 4.0, 22.0), np.array([0.0770, 0.5040, 12.0000])),
    (('7.5G', 4.0, 24.0), np.array([0.0581, 0.5151, 12.0000])),
    (('7.5G', 4.0, 26.0), np.array([0.0392, 0.5258, 12.0000])),
    (('7.5G', 4.0, 28.0), np.array([0.0200, 0.5360, 12.0000])),
    (('7.5G', 4.0, 30.0), np.array([0.0020, 0.5460, 12.0000])),
    (('10G', 4.0, 2.0), np.array([0.2880, 0.3327, 12.0000])),
    (('10G', 4.0, 4.0), np.array([0.2628, 0.3498, 12.0000])),
    (('10G', 4.0, 6.0), np.array([0.2374, 0.3655, 12.0000])),
    (('10G', 4.0, 8.0), np.array([0.2124, 0.3799, 12.0000])),
    (('10G', 4.0, 10.0), np.array([0.1876, 0.3933, 12.0000])),
    (('10G', 4.0, 12.0), np.array([0.1602, 0.4070, 12.0000])),
    (('10G', 4.0, 14.0), np.array([0.1398, 0.4168, 12.0000])),
    (('10G', 4.0, 16.0), np.array([0.1212, 0.4245, 12.0000])),
    (('10G', 4.0, 18.0), np.array([0.1006, 0.4330, 12.0000])),
    (('10G', 4.0, 20.0), np.array([0.0850, 0.4388, 12.0000])),
    (('10G', 4.0, 22.0), np.array([0.0702, 0.4440, 12.0000])),
    (('10G', 4.0, 24.0), np.array([0.0553, 0.4492, 12.0000])),
    (('10G', 4.0, 26.0), np.array([0.0400, 0.4545, 12.0000])),
    (('10G', 4.0, 28.0), np.array([0.0230, 0.4600, 12.0000])),
    (('10G', 4.0, 30.0), np.array([0.0080, 0.4650, 12.0000])),
    (('2.5BG', 4.0, 2.0), np.array([0.2840, 0.3270, 12.0000])),
    (('2.5BG', 4.0, 4.0), np.array([0.2552, 0.3375, 12.0000])),
    (('2.5BG', 4.0, 6.0), np.array([0.2278, 0.3463, 12.0000])),
    (('2.5BG', 4.0, 8.0), np.array([0.2006, 0.3540, 12.0000])),
    (('2.5BG', 4.0, 10.0), np.array([0.1738, 0.3600, 12.0000])),
    (('2.5BG', 4.0, 12.0), np.array([0.1492, 0.3649, 12.0000])),
    (('2.5BG', 4.0, 14.0), np.array([0.1283, 0.3688, 12.0000])),
    (('2.5BG', 4.0, 16.0), np.array([0.1102, 0.3720, 12.0000])),
    (('2.5BG', 4.0, 18.0), np.array([0.0915, 0.3754, 12.0000])),
    (('2.5BG', 4.0, 20.0), np.array([0.0768, 0.3773, 12.0000])),
    (('2.5BG', 4.0, 22.0), np.array([0.0636, 0.3788, 12.0000])),
    (('2.5BG', 4.0, 24.0), np.array([0.0510, 0.3800, 12.0000])),
    (('2.5BG', 4.0, 26.0), np.array([0.0380, 0.3820, 12.0000])),
    (('2.5BG', 4.0, 28.0), np.array([0.0250, 0.3830, 12.0000])),
    (('5BG', 4.0, 2.0), np.array([0.2799, 0.3208, 12.0000])),
    (('5BG', 4.0, 4.0), np.array([0.2480, 0.3232, 12.0000])),
    (('5BG', 4.0, 6.0), np.array([0.2182, 0.3240, 12.0000])),
    (('5BG', 4.0, 8.0), np.array([0.1890, 0.3234, 12.0000])),
    (('5BG', 4.0, 10.0), np.array([0.1618, 0.3219, 12.0000])),
    (('5BG', 4.0, 12.0), np.array([0.1379, 0.3198, 12.0000])),
    (('5BG', 4.0, 14.0), np.array([0.1170, 0.3170, 12.0000])),
    (('5BG', 4.0, 16.0), np.array([0.0992, 0.3141, 12.0000])),
    (('5BG', 4.0, 18.0), np.array([0.0828, 0.3108, 12.0000])),
    (('5BG', 4.0, 20.0), np.array([0.0675, 0.3075, 12.0000])),
    (('5BG', 4.0, 22.0), np.array([0.0560, 0.3050, 12.0000])),
    (('5BG', 4.0, 24.0), np.array([0.0470, 0.3040, 12.0000])),
    (('5BG', 4.0, 26.0), np.array([0.0360, 0.3030, 12.0000])),
    (('7.5BG', 4.0, 2.0), np.array([0.2764, 0.3148, 12.0000])),
    (('7.5BG', 4.0, 4.0), np.array([0.2429, 0.3108, 12.0000])),
    (('7.5BG', 4.0, 6.0), np.array([0.2113, 0.3052, 12.0000])),
    (('7.5BG', 4.0, 8.0), np.array([0.1815, 0.2985, 12.0000])),
    (('7.5BG', 4.0, 10.0), np.array([0.1540, 0.2910, 12.0000])),
    (('7.5BG', 4.0, 12.0), np.array([0.1298, 0.2840, 12.0000])),
    (('7.5BG', 4.0, 14.0), np.array([0.1092, 0.2774, 12.0000])),
    (('7.5BG', 4.0, 16.0), np.array([0.0922, 0.2718, 12.0000])),
    (('7.5BG', 4.0, 18.0), np.array([0.0768, 0.2667, 12.0000])),
    (('7.5BG', 4.0, 20.0), np.array([0.0650, 0.2620, 12.0000])),
    (('7.5BG', 4.0, 22.0), np.array([0.0540, 0.2580, 12.0000])),
    (('7.5BG', 4.0, 24.0), np.array([0.0450, 0.2550, 12.0000])),
    (('10BG', 4.0, 2.0), np.array([0.2740, 0.3091, 12.0000])),
    (('10BG', 4.0, 4.0), np.array([0.2384, 0.2984, 12.0000])),
    (('10BG', 4.0, 6.0), np.array([0.2065, 0.2863, 12.0000])),
    (('10BG', 4.0, 8.0), np.array([0.1760, 0.2730, 12.0000])),
    (('10BG', 4.0, 10.0), np.array([0.1480, 0.2600, 12.0000])),
    (('10BG', 4.0, 12.0), np.array([0.1248, 0.2484, 12.0000])),
    (('10BG', 4.0, 14.0), np.array([0.1033, 0.2376, 12.0000])),
    (('10BG', 4.0, 16.0), np.array([0.0888, 0.2298, 12.0000])),
    (('10BG', 4.0, 18.0), np.array([0.0730, 0.2210, 12.0000])),
    (('10BG', 4.0, 20.0), np.array([0.0620, 0.2140, 12.0000])),
    (('10BG', 4.0, 22.0), np.array([0.0510, 0.2070, 12.0000])),
    (('2.5B', 4.0, 2.0), np.array([0.2727, 0.3038, 12.0000])),
    (('2.5B', 4.0, 4.0), np.array([0.2360, 0.2872, 12.0000])),
    (('2.5B', 4.0, 6.0), np.array([0.2048, 0.2708, 12.0000])),
    (('2.5B', 4.0, 8.0), np.array([0.1737, 0.2524, 12.0000])),
    (('2.5B', 4.0, 10.0), np.array([0.1463, 0.2354, 12.0000])),
    (('2.5B', 4.0, 12.0), np.array([0.1247, 0.2209, 12.0000])),
    (('2.5B', 4.0, 14.0), np.array([0.1027, 0.2057, 12.0000])),
    (('2.5B', 4.0, 16.0), np.array([0.0900, 0.1973, 12.0000])),
    (('2.5B', 4.0, 18.0), np.array([0.0730, 0.1840, 12.0000])),
    (('2.5B', 4.0, 20.0), np.array([0.0620, 0.1770, 12.0000])),
    (('5B', 4.0, 2.0), np.array([0.2723, 0.2992, 12.0000])),
    (('5B', 4.0, 4.0), np.array([0.2363, 0.2782, 12.0000])),
    (('5B', 4.0, 6.0), np.array([0.2060, 0.2572, 12.0000])),
    (('5B', 4.0, 8.0), np.array([0.1759, 0.2345, 12.0000])),
    (('5B', 4.0, 10.0), np.array([0.1512, 0.2148, 12.0000])),
    (('5B', 4.0, 12.0), np.array([0.1299, 0.1963, 12.0000])),
    (('5B', 4.0, 14.0), np.array([0.1098, 0.1785, 12.0000])),
    (('5B', 4.0, 16.0), np.array([0.0940, 0.1630, 12.0000])),
    (('5B', 4.0, 18.0), np.array([0.0790, 0.1500, 12.0000])),
    (('7.5B', 4.0, 2.0), np.array([0.2733, 0.2947, 12.0000])),
    (('7.5B', 4.0, 4.0), np.array([0.2388, 0.2704, 12.0000])),
    (('7.5B', 4.0, 6.0), np.array([0.2102, 0.2470, 12.0000])),
    (('7.5B', 4.0, 8.0), np.array([0.1821, 0.2232, 12.0000])),
    (('7.5B', 4.0, 10.0), np.array([0.1601, 0.2028, 12.0000])),
    (('7.5B', 4.0, 12.0), np.array([0.1393, 0.1837, 12.0000])),
    (('7.5B', 4.0, 14.0), np.array([0.1204, 0.1655, 12.0000])),
    (('7.5B', 4.0, 16.0), np.array([0.1020, 0.1490, 12.0000])),
    (('7.5B', 4.0, 18.0), np.array([0.0910, 0.1380, 12.0000])),
    (('10B', 4.0, 2.0), np.array([0.2753, 0.2910, 12.0000])),
    (('10B', 4.0, 4.0), np.array([0.2429, 0.2648, 12.0000])),
    (('10B', 4.0, 6.0), np.array([0.2157, 0.2407, 12.0000])),
    (('10B', 4.0, 8.0), np.array([0.1893, 0.2160, 12.0000])),
    (('10B', 4.0, 10.0), np.array([0.1681, 0.1954, 12.0000])),
    (('10B', 4.0, 12.0), np.array([0.1487, 0.1760, 12.0000])),
    (('10B', 4.0, 14.0), np.array([0.1310, 0.1580, 12.0000])),
    (('10B', 4.0, 16.0), np.array([0.1155, 0.1416, 12.0000])),
    (('10B', 4.0, 18.0), np.array([0.1030, 0.1280, 12.0000])),
    (('10B', 4.0, 20.0), np.array([0.0920, 0.1170, 12.0000])),
    (('2.5PB', 4.0, 2.0), np.array([0.2782, 0.2876, 12.0000])),
    (('2.5PB', 4.0, 4.0), np.array([0.2487, 0.2597, 12.0000])),
    (('2.5PB', 4.0, 6.0), np.array([0.2235, 0.2343, 12.0000])),
    (('2.5PB', 4.0, 8.0), np.array([0.1995, 0.2094, 12.0000])),
    (('2.5PB', 4.0, 10.0), np.array([0.1805, 0.1888, 12.0000])),
    (('2.5PB', 4.0, 12.0), np.array([0.1634, 0.1698, 12.0000])),
    (('2.5PB', 4.0, 14.0), np.array([0.1473, 0.1513, 12.0000])),
    (('2.5PB', 4.0, 16.0), np.array([0.1336, 0.1349, 12.0000])),
    (('2.5PB', 4.0, 18.0), np.array([0.1218, 0.1208, 12.0000])),
    (('2.5PB', 4.0, 20.0), np.array([0.1120, 0.1080, 12.0000])),
    (('2.5PB', 4.0, 22.0), np.array([0.1040, 0.0960, 12.0000])),
    (('2.5PB', 4.0, 24.0), np.array([0.0980, 0.0890, 12.0000])),
    (('5PB', 4.0, 2.0), np.array([0.2816, 0.2842, 12.0000])),
    (('5PB', 4.0, 4.0), np.array([0.2562, 0.2560, 12.0000])),
    (('5PB', 4.0, 6.0), np.array([0.2325, 0.2300, 12.0000])),
    (('5PB', 4.0, 8.0), np.array([0.2103, 0.2050, 12.0000])),
    (('5PB', 4.0, 10.0), np.array([0.1925, 0.1843, 12.0000])),
    (('5PB', 4.0, 12.0), np.array([0.1773, 0.1659, 12.0000])),
    (('5PB', 4.0, 14.0), np.array([0.1627, 0.1479, 12.0000])),
    (('5PB', 4.0, 16.0), np.array([0.1504, 0.1317, 12.0000])),
    (('5PB', 4.0, 18.0), np.array([0.1392, 0.1167, 12.0000])),
    (('5PB', 4.0, 20.0), np.array([0.1288, 0.1027, 12.0000])),
    (('5PB', 4.0, 22.0), np.array([0.1220, 0.0920, 12.0000])),
    (('5PB', 4.0, 24.0), np.array([0.1180, 0.0840, 12.0000])),
    (('5PB', 4.0, 26.0), np.array([0.1150, 0.0780, 12.0000])),
    (('5PB', 4.0, 28.0), np.array([0.1110, 0.0730, 12.0000])),
    (('5PB', 4.0, 30.0), np.array([0.1080, 0.0680, 12.0000])),
    (('5PB', 4.0, 32.0), np.array([0.1060, 0.0640, 12.0000])),
    (('5PB', 4.0, 34.0), np.array([0.1040, 0.0600, 12.0000])),
    (('5PB', 4.0, 36.0), np.array([0.1020, 0.0550, 12.0000])),
    (('5PB', 4.0, 38.0), np.array([0.0980, 0.0480, 12.0000])),
    (('5PB', 4.0, 40.0), np.array([0.0960, 0.0440, 12.0000])),
    (('7.5PB', 4.0, 2.0), np.array([0.2861, 0.2819, 12.0000])),
    (('7.5PB', 4.0, 4.0), np.array([0.2657, 0.2528, 12.0000])),
    (('7.5PB', 4.0, 6.0), np.array([0.2471, 0.2266, 12.0000])),
    (('7.5PB', 4.0, 8.0), np.array([0.2304, 0.2023, 12.0000])),
    (('7.5PB', 4.0, 10.0), np.array([0.2158, 0.1811, 12.0000])),
    (('7.5PB', 4.0, 12.0), np.array([0.2037, 0.1629, 12.0000])),
    (('7.5PB', 4.0, 14.0), np.array([0.1941, 0.1468, 12.0000])),
    (('7.5PB', 4.0, 16.0), np.array([0.1861, 0.1316, 12.0000])),
    (('7.5PB', 4.0, 18.0), np.array([0.1798, 0.1185, 12.0000])),
    (('7.5PB', 4.0, 20.0), np.array([0.1742, 0.1058, 12.0000])),
    (('7.5PB', 4.0, 22.0), np.array([0.1713, 0.0980, 12.0000])),
    (('7.5PB', 4.0, 24.0), np.array([0.1684, 0.0899, 12.0000])),
    (('7.5PB', 4.0, 26.0), np.array([0.1659, 0.0825, 12.0000])),
    (('7.5PB', 4.0, 28.0), np.array([0.1640, 0.0770, 12.0000])),
    (('7.5PB', 4.0, 30.0), np.array([0.1620, 0.0720, 12.0000])),
    (('7.5PB', 4.0, 32.0), np.array([0.1600, 0.0660, 12.0000])),
    (('7.5PB', 4.0, 34.0), np.array([0.1580, 0.0600, 12.0000])),
    (('7.5PB', 4.0, 36.0), np.array([0.1570, 0.0540, 12.0000])),
    (('7.5PB', 4.0, 38.0), np.array([0.1550, 0.0490, 12.0000])),
    (('7.5PB', 4.0, 40.0), np.array([0.1530, 0.0440, 12.0000])),
    (('10PB', 4.0, 2.0), np.array([0.2911, 0.2804, 12.0000])),
    (('10PB', 4.0, 4.0), np.array([0.2759, 0.2522, 12.0000])),
    (('10PB', 4.0, 6.0), np.array([0.2618, 0.2263, 12.0000])),
    (('10PB', 4.0, 8.0), np.array([0.2497, 0.2038, 12.0000])),
    (('10PB', 4.0, 10.0), np.array([0.2388, 0.1837, 12.0000])),
    (('10PB', 4.0, 12.0), np.array([0.2298, 0.1659, 12.0000])),
    (('10PB', 4.0, 14.0), np.array([0.2220, 0.1503, 12.0000])),
    (('10PB', 4.0, 16.0), np.array([0.2170, 0.1373, 12.0000])),
    (('10PB', 4.0, 18.0), np.array([0.2120, 0.1256, 12.0000])),
    (('10PB', 4.0, 20.0), np.array([0.2075, 0.1140, 12.0000])),
    (('10PB', 4.0, 22.0), np.array([0.2048, 0.1064, 12.0000])),
    (('10PB', 4.0, 24.0), np.array([0.2020, 0.0985, 12.0000])),
    (('10PB', 4.0, 26.0), np.array([0.1994, 0.0904, 12.0000])),
    (('10PB', 4.0, 28.0), np.array([0.1971, 0.0840, 12.0000])),
    (('10PB', 4.0, 30.0), np.array([0.1952, 0.0778, 12.0000])),
    (('10PB', 4.0, 32.0), np.array([0.1920, 0.0710, 12.0000])),
    (('10PB', 4.0, 34.0), np.array([0.1910, 0.0640, 12.0000])),
    (('10PB', 4.0, 36.0), np.array([0.1890, 0.0580, 12.0000])),
    (('10PB', 4.0, 38.0), np.array([0.1870, 0.0520, 12.0000])),
    (('10PB', 4.0, 40.0), np.array([0.1840, 0.0460, 12.0000])),
    (('2.5P', 4.0, 2.0), np.array([0.2962, 0.2807, 12.0000])),
    (('2.5P', 4.0, 4.0), np.array([0.2855, 0.2531, 12.0000])),
    (('2.5P', 4.0, 6.0), np.array([0.2763, 0.2300, 12.0000])),
    (('2.5P', 4.0, 8.0), np.array([0.2685, 0.2089, 12.0000])),
    (('2.5P', 4.0, 10.0), np.array([0.2619, 0.1903, 12.0000])),
    (('2.5P', 4.0, 12.0), np.array([0.2559, 0.1730, 12.0000])),
    (('2.5P', 4.0, 14.0), np.array([0.2509, 0.1585, 12.0000])),
    (('2.5P', 4.0, 16.0), np.array([0.2467, 0.1452, 12.0000])),
    (('2.5P', 4.0, 18.0), np.array([0.2430, 0.1332, 12.0000])),
    (('2.5P', 4.0, 20.0), np.array([0.2394, 0.1221, 12.0000])),
    (('2.5P', 4.0, 22.0), np.array([0.2371, 0.1143, 12.0000])),
    (('2.5P', 4.0, 24.0), np.array([0.2348, 0.1062, 12.0000])),
    (('2.5P', 4.0, 26.0), np.array([0.2322, 0.0978, 12.0000])),
    (('2.5P', 4.0, 28.0), np.array([0.2302, 0.0909, 12.0000])),
    (('2.5P', 4.0, 30.0), np.array([0.2285, 0.0847, 12.0000])),
    (('2.5P', 4.0, 32.0), np.array([0.2265, 0.0774, 12.0000])),
    (('2.5P', 4.0, 34.0), np.array([0.2240, 0.0700, 12.0000])),
    (('2.5P', 4.0, 36.0), np.array([0.2220, 0.0640, 12.0000])),
    (('2.5P', 4.0, 38.0), np.array([0.2200, 0.0570, 12.0000])),
    (('2.5P', 4.0, 40.0), np.array([0.2180, 0.0510, 12.0000])),
    (('5P', 4.0, 2.0), np.array([0.3022, 0.2825, 12.0000])),
    (('5P', 4.0, 4.0), np.array([0.2958, 0.2565, 12.0000])),
    (('5P', 4.0, 6.0), np.array([0.2903, 0.2347, 12.0000])),
    (('5P', 4.0, 8.0), np.array([0.2855, 0.2150, 12.0000])),
    (('5P', 4.0, 10.0), np.array([0.2814, 0.1967, 12.0000])),
    (('5P', 4.0, 12.0), np.array([0.2778, 0.1808, 12.0000])),
    (('5P', 4.0, 14.0), np.array([0.2747, 0.1660, 12.0000])),
    (('5P', 4.0, 16.0), np.array([0.2718, 0.1520, 12.0000])),
    (('5P', 4.0, 18.0), np.array([0.2693, 0.1408, 12.0000])),
    (('5P', 4.0, 20.0), np.array([0.2670, 0.1300, 12.0000])),
    (('5P', 4.0, 22.0), np.array([0.2652, 0.1218, 12.0000])),
    (('5P', 4.0, 24.0), np.array([0.2635, 0.1132, 12.0000])),
    (('5P', 4.0, 26.0), np.array([0.2618, 0.1052, 12.0000])),
    (('5P', 4.0, 28.0), np.array([0.2600, 0.0971, 12.0000])),
    (('5P', 4.0, 30.0), np.array([0.2588, 0.0907, 12.0000])),
    (('5P', 4.0, 32.0), np.array([0.2574, 0.0833, 12.0000])),
    (('5P', 4.0, 34.0), np.array([0.2560, 0.0740, 12.0000])),
    (('5P', 4.0, 36.0), np.array([0.2550, 0.0690, 12.0000])),
    (('5P', 4.0, 38.0), np.array([0.2540, 0.0630, 12.0000])),
    (('5P', 4.0, 40.0), np.array([0.2540, 0.0560, 12.0000])),
    (('7.5P', 4.0, 2.0), np.array([0.3093, 0.2859, 12.0000])),
    (('7.5P', 4.0, 4.0), np.array([0.3084, 0.2622, 12.0000])),
    (('7.5P', 4.0, 6.0), np.array([0.3076, 0.2416, 12.0000])),
    (('7.5P', 4.0, 8.0), np.array([0.3066, 0.2228, 12.0000])),
    (('7.5P', 4.0, 10.0), np.array([0.3056, 0.2060, 12.0000])),
    (('7.5P', 4.0, 12.0), np.array([0.3045, 0.1905, 12.0000])),
    (('7.5P', 4.0, 14.0), np.array([0.3035, 0.1755, 12.0000])),
    (('7.5P', 4.0, 16.0), np.array([0.3028, 0.1621, 12.0000])),
    (('7.5P', 4.0, 18.0), np.array([0.3016, 0.1500, 12.0000])),
    (('7.5P', 4.0, 20.0), np.array([0.3010, 0.1396, 12.0000])),
    (('7.5P', 4.0, 22.0), np.array([0.3001, 0.1306, 12.0000])),
    (('7.5P', 4.0, 24.0), np.array([0.2993, 0.1225, 12.0000])),
    (('7.5P', 4.0, 26.0), np.array([0.2986, 0.1135, 12.0000])),
    (('7.5P', 4.0, 28.0), np.array([0.2979, 0.1062, 12.0000])),
    (('7.5P', 4.0, 30.0), np.array([0.2969, 0.0979, 12.0000])),
    (('7.5P', 4.0, 32.0), np.array([0.2962, 0.0906, 12.0000])),
    (('7.5P', 4.0, 34.0), np.array([0.2950, 0.0820, 12.0000])),
    (('7.5P', 4.0, 36.0), np.array([0.2950, 0.0750, 12.0000])),
    (('7.5P', 4.0, 38.0), np.array([0.2940, 0.0690, 12.0000])),
    (('7.5P', 4.0, 40.0), np.array([0.2930, 0.0620, 12.0000])),
    (('10P', 4.0, 2.0), np.array([0.3162, 0.2902, 12.0000])),
    (('10P', 4.0, 4.0), np.array([0.3210, 0.2686, 12.0000])),
    (('10P', 4.0, 6.0), np.array([0.3248, 0.2493, 12.0000])),
    (('10P', 4.0, 8.0), np.array([0.3280, 0.2318, 12.0000])),
    (('10P', 4.0, 10.0), np.array([0.3306, 0.2162, 12.0000])),
    (('10P', 4.0, 12.0), np.array([0.3331, 0.2014, 12.0000])),
    (('10P', 4.0, 14.0), np.array([0.3351, 0.1875, 12.0000])),
    (('10P', 4.0, 16.0), np.array([0.3370, 0.1756, 12.0000])),
    (('10P', 4.0, 18.0), np.array([0.3386, 0.1626, 12.0000])),
    (('10P', 4.0, 20.0), np.array([0.3400, 0.1500, 12.0000])),
    (('10P', 4.0, 22.0), np.array([0.3411, 0.1424, 12.0000])),
    (('10P', 4.0, 24.0), np.array([0.3421, 0.1337, 12.0000])),
    (('10P', 4.0, 26.0), np.array([0.3428, 0.1248, 12.0000])),
    (('10P', 4.0, 28.0), np.array([0.3432, 0.1172, 12.0000])),
    (('10P', 4.0, 30.0), np.array([0.3440, 0.1080, 12.0000])),
    (('10P', 4.0, 32.0), np.array([0.3450, 0.1000, 12.0000])),
    (('10P', 4.0, 34.0), np.array([0.3460, 0.0930, 12.0000])),
    (('10P', 4.0, 36.0), np.array([0.3460, 0.0850, 12.0000])),
    (('10P', 4.0, 38.0), np.array([0.3470, 0.0780, 12.0000])),
    (('2.5RP', 4.0, 2.0), np.array([0.3231, 0.2951, 12.0000])),
    (('2.5RP', 4.0, 4.0), np.array([0.3340, 0.2770, 12.0000])),
    (('2.5RP', 4.0, 6.0), np.array([0.3442, 0.2595, 12.0000])),
    (('2.5RP', 4.0, 8.0), np.array([0.3533, 0.2438, 12.0000])),
    (('2.5RP', 4.0, 10.0), np.array([0.3608, 0.2301, 12.0000])),
    (('2.5RP', 4.0, 12.0), np.array([0.3683, 0.2162, 12.0000])),
    (('2.5RP', 4.0, 14.0), np.array([0.3748, 0.2039, 12.0000])),
    (('2.5RP', 4.0, 16.0), np.array([0.3807, 0.1923, 12.0000])),
    (('2.5RP', 4.0, 18.0), np.array([0.3865, 0.1802, 12.0000])),
    (('2.5RP', 4.0, 20.0), np.array([0.3926, 0.1679, 12.0000])),
    (('2.5RP', 4.0, 22.0), np.array([0.3967, 0.1593, 12.0000])),
    (('2.5RP', 4.0, 24.0), np.array([0.4011, 0.1504, 12.0000])),
    (('2.5RP', 4.0, 26.0), np.array([0.4048, 0.1428, 12.0000])),
    (('2.5RP', 4.0, 28.0), np.array([0.4090, 0.1340, 12.0000])),
    (('2.5RP', 4.0, 30.0), np.array([0.4140, 0.1220, 12.0000])),
    (('2.5RP', 4.0, 32.0), np.array([0.4170, 0.1150, 12.0000])),
    (('2.5RP', 4.0, 34.0), np.array([0.4190, 0.1080, 12.0000])),
    (('5RP', 4.0, 2.0), np.array([0.3310, 0.3010, 12.0000])),
    (('5RP', 4.0, 4.0), np.array([0.3491, 0.2872, 12.0000])),
    (('5RP', 4.0, 6.0), np.array([0.3671, 0.2733, 12.0000])),
    (('5RP', 4.0, 8.0), np.array([0.3833, 0.2600, 12.0000])),
    (('5RP', 4.0, 10.0), np.array([0.3960, 0.2489, 12.0000])),
    (('5RP', 4.0, 12.0), np.array([0.4104, 0.2361, 12.0000])),
    (('5RP', 4.0, 14.0), np.array([0.4225, 0.2249, 12.0000])),
    (('5RP', 4.0, 16.0), np.array([0.4339, 0.2139, 12.0000])),
    (('5RP', 4.0, 18.0), np.array([0.4455, 0.2023, 12.0000])),
    (('5RP', 4.0, 20.0), np.array([0.4571, 0.1906, 12.0000])),
    (('5RP', 4.0, 22.0), np.array([0.4656, 0.1821, 12.0000])),
    (('5RP', 4.0, 24.0), np.array([0.4730, 0.1720, 12.0000])),
    (('5RP', 4.0, 26.0), np.array([0.4820, 0.1620, 12.0000])),
    (('5RP', 4.0, 28.0), np.array([0.4890, 0.1540, 12.0000])),
    (('5RP', 4.0, 30.0), np.array([0.4990, 0.1430, 12.0000])),
    (('7.5RP', 4.0, 2.0), np.array([0.3371, 0.3061, 12.0000])),
    (('7.5RP', 4.0, 4.0), np.array([0.3612, 0.2963, 12.0000])),
    (('7.5RP', 4.0, 6.0), np.array([0.3850, 0.2859, 12.0000])),
    (('7.5RP', 4.0, 8.0), np.array([0.4072, 0.2750, 12.0000])),
    (('7.5RP', 4.0, 10.0), np.array([0.4259, 0.2651, 12.0000])),
    (('7.5RP', 4.0, 12.0), np.array([0.4450, 0.2541, 12.0000])),
    (('7.5RP', 4.0, 14.0), np.array([0.4629, 0.2437, 12.0000])),
    (('7.5RP', 4.0, 16.0), np.array([0.4799, 0.2329, 12.0000])),
    (('7.5RP', 4.0, 18.0), np.array([0.4965, 0.2217, 12.0000])),
    (('7.5RP', 4.0, 20.0), np.array([0.5130, 0.2101, 12.0000])),
    (('7.5RP', 4.0, 22.0), np.array([0.5230, 0.2020, 12.0000])),
    (('7.5RP', 4.0, 24.0), np.array([0.5360, 0.1920, 12.0000])),
    (('7.5RP', 4.0, 26.0), np.array([0.5490, 0.1810, 12.0000])),
    (('10RP', 4.0, 2.0), np.array([0.3417, 0.3106, 12.0000])),
    (('10RP', 4.0, 4.0), np.array([0.3715, 0.3042, 12.0000])),
    (('10RP', 4.0, 6.0), np.array([0.3999, 0.2972, 12.0000])),
    (('10RP', 4.0, 8.0), np.array([0.4282, 0.2890, 12.0000])),
    (('10RP', 4.0, 10.0), np.array([0.4528, 0.2811, 12.0000])),
    (('10RP', 4.0, 12.0), np.array([0.4789, 0.2717, 12.0000])),
    (('10RP', 4.0, 14.0), np.array([0.5020, 0.2623, 12.0000])),
    (('10RP', 4.0, 16.0), np.array([0.5234, 0.2530, 12.0000])),
    (('10RP', 4.0, 18.0), np.array([0.5466, 0.2424, 12.0000])),
    (('10RP', 4.0, 20.0), np.array([0.5674, 0.2319, 12.0000])),
    (('10RP', 4.0, 22.0), np.array([0.5820, 0.2240, 12.0000])),
    (('10RP', 4.0, 24.0), np.array([0.5980, 0.2140, 12.0000])),
    (('2.5R', 4.0, 2.0), np.array([0.3461, 0.3150, 12.0000])),
    (('2.5R', 4.0, 4.0), np.array([0.3806, 0.3125, 12.0000])),
    (('2.5R', 4.0, 6.0), np.array([0.4141, 0.3085, 12.0000])),
    (('2.5R', 4.0, 8.0), np.array([0.4472, 0.3031, 12.0000])),
    (('2.5R', 4.0, 10.0), np.array([0.4774, 0.2969, 12.0000])),
    (('2.5R', 4.0, 12.0), np.array([0.5072, 0.2897, 12.0000])),
    (('2.5R', 4.0, 14.0), np.array([0.5369, 0.2810, 12.0000])),
    (('2.5R', 4.0, 16.0), np.array([0.5620, 0.2724, 12.0000])),
    (('2.5R', 4.0, 18.0), np.array([0.5898, 0.2622, 12.0000])),
    (('2.5R', 4.0, 20.0), np.array([0.6150, 0.2530, 12.0000])),
    (('2.5R', 4.0, 22.0), np.array([0.6330, 0.2440, 12.0000])),
    (('2.5R', 4.0, 24.0), np.array([0.6540, 0.2360, 12.0000])),
    (('5R', 4.0, 2.0), np.array([0.3508, 0.3200, 12.0000])),
    (('5R', 4.0, 4.0), np.array([0.3916, 0.3223, 12.0000])),
    (('5R', 4.0, 6.0), np.array([0.4299, 0.3226, 12.0000])),
    (('5R', 4.0, 8.0), np.array([0.4690, 0.3209, 12.0000])),
    (('5R', 4.0, 10.0), np.array([0.5043, 0.3176, 12.0000])),
    (('5R', 4.0, 12.0), np.array([0.5385, 0.3129, 12.0000])),
    (('5R', 4.0, 14.0), np.array([0.5734, 0.3057, 12.0000])),
    (('5R', 4.0, 16.0), np.array([0.6039, 0.2978, 12.0000])),
    (('5R', 4.0, 18.0), np.array([0.6329, 0.2881, 12.0000])),
    (('5R', 4.0, 20.0), np.array([0.6580, 0.2780, 12.0000])),
    (('5R', 4.0, 22.0), np.array([0.6780, 0.2700, 12.0000])),
    (('5R', 4.0, 24.0), np.array([0.6990, 0.2600, 12.0000])),
    (('7.5R', 4.0, 2.0), np.array([0.3538, 0.3236, 12.0000])),
    (('7.5R', 4.0, 4.0), np.array([0.3990, 0.3300, 12.0000])),
    (('7.5R', 4.0, 6.0), np.array([0.4415, 0.3340, 12.0000])),
    (('7.5R', 4.0, 8.0), np.array([0.4850, 0.3359, 12.0000])),
    (('7.5R', 4.0, 10.0), np.array([0.5235, 0.3351, 12.0000])),
    (('7.5R', 4.0, 12.0), np.array([0.5603, 0.3321, 12.0000])),
    (('7.5R', 4.0, 14.0), np.array([0.5959, 0.3269, 12.0000])),
    (('7.5R', 4.0, 16.0), np.array([0.6260, 0.3192, 12.0000])),
    (('7.5R', 4.0, 18.0), np.array([0.6538, 0.3100, 12.0000])),
    (('7.5R', 4.0, 20.0), np.array([0.6806, 0.2988, 12.0000])),
    (('7.5R', 4.0, 22.0), np.array([0.7030, 0.2910, 12.0000])),
    (('7.5R', 4.0, 24.0), np.array([0.7260, 0.2800, 12.0000])),
    (('10R', 4.0, 2.0), np.array([0.3582, 0.3294, 12.0000])),
    (('10R', 4.0, 4.0), np.array([0.4078, 0.3412, 12.0000])),
    (('10R', 4.0, 6.0), np.array([0.4535, 0.3500, 12.0000])),
    (('10R', 4.0, 8.0), np.array([0.4995, 0.3557, 12.0000])),
    (('10R', 4.0, 10.0), np.array([0.5418, 0.3580, 12.0000])),
    (('10R', 4.0, 12.0), np.array([0.5801, 0.3588, 12.0000])),
    (('10R', 4.0, 14.0), np.array([0.6154, 0.3568, 12.0000])),
    (('10R', 4.0, 16.0), np.array([0.6409, 0.3533, 12.0000])),
    (('10R', 4.0, 18.0), np.array([0.6710, 0.3480, 12.0000])),
    (('10R', 4.0, 20.0), np.array([0.6980, 0.3440, 12.0000])),
    (('10R', 4.0, 22.0), np.array([0.7260, 0.3380, 12.0000])),
    (('10R', 4.0, 24.0), np.array([0.7560, 0.3320, 12.0000])),
    (('2.5YR', 4.0, 2.0), np.array([0.3624, 0.3367, 12.0000])),
    (('2.5YR', 4.0, 4.0), np.array([0.4141, 0.3539, 12.0000])),
    (('2.5YR', 4.0, 6.0), np.array([0.4612, 0.3674, 12.0000])),
    (('2.5YR', 4.0, 8.0), np.array([0.5071, 0.3777, 12.0000])),
    (('2.5YR', 4.0, 10.0), np.array([0.5475, 0.3856, 12.0000])),
    (('2.5YR', 4.0, 12.0), np.array([0.5809, 0.3910, 12.0000])),
    (('2.5YR', 4.0, 14.0), np.array([0.6140, 0.3960, 12.0000])),
    (('2.5YR', 4.0, 16.0), np.array([0.6390, 0.4000, 12.0000])),
    (('2.5YR', 4.0, 18.0), np.array([0.6690, 0.4040, 12.0000])),
    (('5YR', 4.0, 2.0), np.array([0.3651, 0.3442, 12.0000])),
    (('5YR', 4.0, 4.0), np.array([0.4187, 0.3679, 12.0000])),
    (('5YR', 4.0, 6.0), np.array([0.4651, 0.3859, 12.0000])),
    (('5YR', 4.0, 8.0), np.array([0.5070, 0.3994, 12.0000])),
    (('5YR', 4.0, 10.0), np.array([0.5432, 0.4097, 12.0000])),
    (('5YR', 4.0, 12.0), np.array([0.5729, 0.4169, 12.0000])),
    (('5YR', 4.0, 14.0), np.array([0.6030, 0.4230, 12.0000])),
    (('7.5YR', 4.0, 2.0), np.array([0.3662, 0.3504, 12.0000])),
    (('7.5YR', 4.0, 4.0), np.array([0.4208, 0.3809, 12.0000])),
    (('7.5YR', 4.0, 6.0), np.array([0.4655, 0.4029, 12.0000])),
    (('7.5YR', 4.0, 8.0), np.array([0.5038, 0.4204, 12.0000])),
    (('7.5YR', 4.0, 10.0), np.array([0.5356, 0.4342, 12.0000])),
    (('7.5YR', 4.0, 12.0), np.array([0.5590, 0.4450, 12.0000])),
    (('7.5YR', 4.0, 14.0), np.array([0.5870, 0.4560, 12.0000])),
    (('10YR', 4.0, 2.0), np.array([0.3660, 0.3590, 12.0000])),
    (('10YR', 4.0, 4.0), np.array([0.4189, 0.3948, 12.0000])),
    (('10YR', 4.0, 6.0), np.array([0.4618, 0.4213, 12.0000])),
    (('10YR', 4.0, 8.0), np.array([0.4965, 0.4414, 12.0000])),
    (('10YR', 4.0, 10.0), np.array([0.5250, 0.4573, 12.0000])),
    (('10YR', 4.0, 12.0), np.array([0.5450, 0.4680, 12.0000])),
    (('2.5Y', 4.0, 2.0), np.array([0.3633, 0.3654, 12.0000])),
    (('2.5Y', 4.0, 4.0), np.array([0.4138, 0.4076, 12.0000])),
    (('2.5Y', 4.0, 6.0), np.array([0.4542, 0.4391, 12.0000])),
    (('2.5Y', 4.0, 8.0), np.array([0.4865, 0.4625, 12.0000])),
    (('2.5Y', 4.0, 10.0), np.array([0.5120, 0.4800, 12.0000])),
    (('2.5Y', 4.0, 12.0), np.array([0.5290, 0.4920, 12.0000])),
    (('5Y', 4.0, 2.0), np.array([0.3590, 0.3701, 12.0000])),
    (('5Y', 4.0, 4.0), np.array([0.4069, 0.4188, 12.0000])),
    (('5Y', 4.0, 6.0), np.array([0.4451, 0.4550, 12.0000])),
    (('5Y', 4.0, 8.0), np.array([0.4745, 0.4810, 12.0000])),
    (('5Y', 4.0, 10.0), np.array([0.4960, 0.5030, 12.0000])),
    (('5Y', 4.0, 12.0), np.array([0.5120, 0.5160, 12.0000])),
    (('7.5Y', 4.0, 2.0), np.array([0.3542, 0.3727, 12.0000])),
    (('7.5Y', 4.0, 4.0), np.array([0.3982, 0.4272, 12.0000])),
    (('7.5Y', 4.0, 6.0), np.array([0.4331, 0.4688, 12.0000])),
    (('7.5Y', 4.0, 8.0), np.array([0.4595, 0.4990, 12.0000])),
    (('7.5Y', 4.0, 10.0), np.array([0.4810, 0.5230, 12.0000])),
    (('7.5Y', 4.0, 12.0), np.array([0.4940, 0.5380, 12.0000])),
    (('10Y', 5.0, 2.0), np.array([0.3422, 0.3648, 19.7700])),
    (('10Y', 5.0, 4.0), np.array([0.3762, 0.4158, 19.7700])),
    (('10Y', 5.0, 6.0), np.array([0.4072, 0.4621, 19.7700])),
    (('10Y', 5.0, 8.0), np.array([0.4307, 0.4967, 19.7700])),
    (('10Y', 5.0, 10.0), np.array([0.4468, 0.5209, 19.7700])),
    (('10Y', 5.0, 12.0), np.array([0.4590, 0.5390, 19.7700])),
    (('10Y', 5.0, 14.0), np.array([0.4690, 0.5540, 19.7700])),
    (('2.5GY', 5.0, 2.0), np.array([0.3352, 0.3636, 19.7700])),
    (('2.5GY', 5.0, 4.0), np.array([0.3621, 0.4143, 19.7700])),
    (('2.5GY', 5.0, 6.0), np.array([0.3879, 0.4646, 19.7700])),
    (('2.5GY', 5.0, 8.0), np.array([0.4088, 0.5068, 19.7700])),
    (('2.5GY', 5.0, 10.0), np.array([0.4224, 0.5369, 19.7700])),
    (('2.5GY', 5.0, 12.0), np.array([0.4333, 0.5602, 19.7700])),
    (('2.5GY', 5.0, 14.0), np.array([0.4400, 0.5800, 19.7700])),
    (('5GY', 5.0, 2.0), np.array([0.3289, 0.3612, 19.7700])),
    (('5GY', 5.0, 4.0), np.array([0.3482, 0.4097, 19.7700])),
    (('5GY', 5.0, 6.0), np.array([0.3663, 0.4614, 19.7700])),
    (('5GY', 5.0, 8.0), np.array([0.3815, 0.5093, 19.7700])),
    (('5GY', 5.0, 10.0), np.array([0.3928, 0.5485, 19.7700])),
    (('5GY', 5.0, 12.0), np.array([0.4011, 0.5802, 19.7700])),
    (('5GY', 5.0, 14.0), np.array([0.4070, 0.6040, 19.7700])),
    (('5GY', 5.0, 16.0), np.array([0.4100, 0.6260, 19.7700])),
    (('7.5GY', 5.0, 2.0), np.array([0.3188, 0.3560, 19.7700])),
    (('7.5GY', 5.0, 4.0), np.array([0.3274, 0.3994, 19.7700])),
    (('7.5GY', 5.0, 6.0), np.array([0.3354, 0.4483, 19.7700])),
    (('7.5GY', 5.0, 8.0), np.array([0.3412, 0.4976, 19.7700])),
    (('7.5GY', 5.0, 10.0), np.array([0.3451, 0.5490, 19.7700])),
    (('7.5GY', 5.0, 12.0), np.array([0.3450, 0.5949, 19.7700])),
    (('7.5GY', 5.0, 14.0), np.array([0.3429, 0.6335, 19.7700])),
    (('7.5GY', 5.0, 16.0), np.array([0.3410, 0.6660, 19.7700])),
    (('7.5GY', 5.0, 18.0), np.array([0.3390, 0.6970, 19.7700])),
    (('7.5GY', 5.0, 20.0), np.array([0.3330, 0.7330, 19.7700])),
    (('10GY', 5.0, 2.0), np.array([0.3110, 0.3508, 19.7700])),
    (('10GY', 5.0, 4.0), np.array([0.3111, 0.3881, 19.7700])),
    (('10GY', 5.0, 6.0), np.array([0.3108, 0.4301, 19.7700])),
    (('10GY', 5.0, 8.0), np.array([0.3080, 0.4759, 19.7700])),
    (('10GY', 5.0, 10.0), np.array([0.3028, 0.5237, 19.7700])),
    (('10GY', 5.0, 12.0), np.array([0.2940, 0.5751, 19.7700])),
    (('10GY', 5.0, 14.0), np.array([0.2838, 0.6208, 19.7700])),
    (('10GY', 5.0, 16.0), np.array([0.2702, 0.6700, 19.7700])),
    (('10GY', 5.0, 18.0), np.array([0.2549, 0.7179, 19.7700])),
    (('10GY', 5.0, 20.0), np.array([0.2370, 0.7670, 19.7700])),
    (('10GY', 5.0, 22.0), np.array([0.2210, 0.8080, 19.7700])),
    (('10GY', 5.0, 24.0), np.array([0.2020, 0.8520, 19.7700])),
    (('10GY', 5.0, 26.0), np.array([0.1790, 0.8980, 19.7700])),
    (('10GY', 5.0, 28.0), np.array([0.1540, 0.9490, 19.7700])),
    (('10GY', 5.0, 30.0), np.array([0.1250, 1.0000, 19.7700])),
    (('10GY', 5.0, 32.0), np.array([0.0970, 1.0480, 19.7700])),
    (('2.5G', 5.0, 2.0), np.array([0.3030, 0.3445, 19.7700])),
    (('2.5G', 5.0, 4.0), np.array([0.2943, 0.3735, 19.7700])),
    (('2.5G', 5.0, 6.0), np.array([0.2841, 0.4045, 19.7700])),
    (('2.5G', 5.0, 8.0), np.array([0.2710, 0.4380, 19.7700])),
    (('2.5G', 5.0, 10.0), np.array([0.2565, 0.4705, 19.7700])),
    (('2.5G', 5.0, 12.0), np.array([0.2385, 0.5071, 19.7700])),
    (('2.5G', 5.0, 14.0), np.array([0.2211, 0.5411, 19.7700])),
    (('2.5G', 5.0, 16.0), np.array([0.2005, 0.5759, 19.7700])),
    (('2.5G', 5.0, 18.0), np.array([0.1782, 0.6095, 19.7700])),
    (('2.5G', 5.0, 20.0), np.array([0.1579, 0.6392, 19.7700])),
    (('2.5G', 5.0, 22.0), np.array([0.1377, 0.6674, 19.7700])),
    (('2.5G', 5.0, 24.0), np.array([0.1188, 0.6918, 19.7700])),
    (('2.5G', 5.0, 26.0), np.array([0.0992, 0.7155, 19.7700])),
    (('2.5G', 5.0, 28.0), np.array([0.0794, 0.7385, 19.7700])),
    (('2.5G', 5.0, 30.0), np.array([0.0550, 0.7660, 19.7700])),
    (('2.5G', 5.0, 32.0), np.array([0.0340, 0.7890, 19.7700])),
    (('5G', 5.0, 2.0), np.array([0.2978, 0.3392, 19.7700])),
    (('5G', 5.0, 4.0), np.array([0.2841, 0.3628, 19.7700])),
    (('5G', 5.0, 6.0), np.array([0.2690, 0.3860, 19.7700])),
    (('5G', 5.0, 8.0), np.array([0.2511, 0.4107, 19.7700])),
    (('5G', 5.0, 10.0), np.array([0.2329, 0.4331, 19.7700])),
    (('5G', 5.0, 12.0), np.array([0.2104, 0.4578, 19.7700])),
    (('5G', 5.0, 14.0), np.array([0.1912, 0.4773, 19.7700])),
    (('5G', 5.0, 16.0), np.array([0.1695, 0.4981, 19.7700])),
    (('5G', 5.0, 18.0), np.array([0.1489, 0.5171, 19.7700])),
    (('5G', 5.0, 20.0), np.array([0.1318, 0.5321, 19.7700])),
    (('5G', 5.0, 22.0), np.array([0.1144, 0.5463, 19.7700])),
    (('5G', 5.0, 24.0), np.array([0.0953, 0.5628, 19.7700])),
    (('5G', 5.0, 26.0), np.array([0.0784, 0.5761, 19.7700])),
    (('5G', 5.0, 28.0), np.array([0.0609, 0.5898, 19.7700])),
    (('5G', 5.0, 30.0), np.array([0.0400, 0.6050, 19.7700])),
    (('5G', 5.0, 32.0), np.array([0.0200, 0.6220, 19.7700])),
    (('7.5G', 5.0, 2.0), np.array([0.2945, 0.3355, 19.7700])),
    (('7.5G', 5.0, 4.0), np.array([0.2775, 0.3545, 19.7700])),
    (('7.5G', 5.0, 6.0), np.array([0.2598, 0.3724, 19.7700])),
    (('7.5G', 5.0, 8.0), np.array([0.2395, 0.3915, 19.7700])),
    (('7.5G', 5.0, 10.0), np.array([0.2200, 0.4082, 19.7700])),
    (('7.5G', 5.0, 12.0), np.array([0.1964, 0.4271, 19.7700])),
    (('7.5G', 5.0, 14.0), np.array([0.1776, 0.4415, 19.7700])),
    (('7.5G', 5.0, 16.0), np.array([0.1571, 0.4561, 19.7700])),
    (('7.5G', 5.0, 18.0), np.array([0.1372, 0.4705, 19.7700])),
    (('7.5G', 5.0, 20.0), np.array([0.1212, 0.4817, 19.7700])),
    (('7.5G', 5.0, 22.0), np.array([0.1050, 0.4927, 19.7700])),
    (('7.5G', 5.0, 24.0), np.array([0.0878, 0.5039, 19.7700])),
    (('7.5G', 5.0, 26.0), np.array([0.0730, 0.5131, 19.7700])),
    (('7.5G', 5.0, 28.0), np.array([0.0585, 0.5224, 19.7700])),
    (('7.5G', 5.0, 30.0), np.array([0.0410, 0.5320, 19.7700])),
    (('7.5G', 5.0, 32.0), np.array([0.0210, 0.5420, 19.7700])),
    (('10G', 5.0, 2.0), np.array([0.2910, 0.3310, 19.7700])),
    (('10G', 5.0, 4.0), np.array([0.2711, 0.3455, 19.7700])),
    (('10G', 5.0, 6.0), np.array([0.2519, 0.3587, 19.7700])),
    (('10G', 5.0, 8.0), np.array([0.2297, 0.3730, 19.7700])),
    (('10G', 5.0, 10.0), np.array([0.2095, 0.3853, 19.7700])),
    (('10G', 5.0, 12.0), np.array([0.1852, 0.3992, 19.7700])),
    (('10G', 5.0, 14.0), np.array([0.1671, 0.4089, 19.7700])),
    (('10G', 5.0, 16.0), np.array([0.1469, 0.4192, 19.7700])),
    (('10G', 5.0, 18.0), np.array([0.1275, 0.4288, 19.7700])),
    (('10G', 5.0, 20.0), np.array([0.1120, 0.4360, 19.7700])),
    (('10G', 5.0, 22.0), np.array([0.0958, 0.4428, 19.7700])),
    (('10G', 5.0, 24.0), np.array([0.0811, 0.4491, 19.7700])),
    (('10G', 5.0, 26.0), np.array([0.0690, 0.4542, 19.7700])),
    (('10G', 5.0, 28.0), np.array([0.0572, 0.4590, 19.7700])),
    (('10G', 5.0, 30.0), np.array([0.0410, 0.4660, 19.7700])),
    (('10G', 5.0, 32.0), np.array([0.0230, 0.4740, 19.7700])),
    (('2.5BG', 5.0, 2.0), np.array([0.2880, 0.3270, 19.7700])),
    (('2.5BG', 5.0, 4.0), np.array([0.2659, 0.3369, 19.7700])),
    (('2.5BG', 5.0, 6.0), np.array([0.2448, 0.3452, 19.7700])),
    (('2.5BG', 5.0, 8.0), np.array([0.2205, 0.3537, 19.7700])),
    (('2.5BG', 5.0, 10.0), np.array([0.1980, 0.3606, 19.7700])),
    (('2.5BG', 5.0, 12.0), np.array([0.1735, 0.3668, 19.7700])),
    (('2.5BG', 5.0, 14.0), np.array([0.1559, 0.3708, 19.7700])),
    (('2.5BG', 5.0, 16.0), np.array([0.1348, 0.3750, 19.7700])),
    (('2.5BG', 5.0, 18.0), np.array([0.1165, 0.3785, 19.7700])),
    (('2.5BG', 5.0, 20.0), np.array([0.1005, 0.3814, 19.7700])),
    (('2.5BG', 5.0, 22.0), np.array([0.0861, 0.3832, 19.7700])),
    (('2.5BG', 5.0, 24.0), np.array([0.0738, 0.3851, 19.7700])),
    (('2.5BG', 5.0, 26.0), np.array([0.0640, 0.3860, 19.7700])),
    (('2.5BG', 5.0, 28.0), np.array([0.0530, 0.3880, 19.7700])),
    (('2.5BG', 5.0, 30.0), np.array([0.0420, 0.3890, 19.7700])),
    (('2.5BG', 5.0, 32.0), np.array([0.0270, 0.3910, 19.7700])),
    (('5BG', 5.0, 2.0), np.array([0.2841, 0.3210, 19.7700])),
    (('5BG', 5.0, 4.0), np.array([0.2591, 0.3246, 19.7700])),
    (('5BG', 5.0, 6.0), np.array([0.2360, 0.3270, 19.7700])),
    (('5BG', 5.0, 8.0), np.array([0.2100, 0.3280, 19.7700])),
    (('5BG', 5.0, 10.0), np.array([0.1850, 0.3280, 19.7700])),
    (('5BG', 5.0, 12.0), np.array([0.1614, 0.3280, 19.7700])),
    (('5BG', 5.0, 14.0), np.array([0.1448, 0.3275, 19.7700])),
    (('5BG', 5.0, 16.0), np.array([0.1243, 0.3261, 19.7700])),
    (('5BG', 5.0, 18.0), np.array([0.1046, 0.3244, 19.7700])),
    (('5BG', 5.0, 20.0), np.array([0.0904, 0.3231, 19.7700])),
    (('5BG', 5.0, 22.0), np.array([0.0781, 0.3211, 19.7700])),
    (('5BG', 5.0, 24.0), np.array([0.0670, 0.3200, 19.7700])),
    (('5BG', 5.0, 26.0), np.array([0.0580, 0.3180, 19.7700])),
    (('5BG', 5.0, 28.0), np.array([0.0500, 0.3170, 19.7700])),
    (('5BG', 5.0, 30.0), np.array([0.0420, 0.3160, 19.7700])),
    (('7.5BG', 5.0, 2.0), np.array([0.2812, 0.3161, 19.7700])),
    (('7.5BG', 5.0, 4.0), np.array([0.2550, 0.3150, 19.7700])),
    (('7.5BG', 5.0, 6.0), np.array([0.2292, 0.3125, 19.7700])),
    (('7.5BG', 5.0, 8.0), np.array([0.2030, 0.3082, 19.7700])),
    (('7.5BG', 5.0, 10.0), np.array([0.1776, 0.3032, 19.7700])),
    (('7.5BG', 5.0, 12.0), np.array([0.1537, 0.2976, 19.7700])),
    (('7.5BG', 5.0, 14.0), np.array([0.1364, 0.2932, 19.7700])),
    (('7.5BG', 5.0, 16.0), np.array([0.1167, 0.2880, 19.7700])),
    (('7.5BG', 5.0, 18.0), np.array([0.0982, 0.2828, 19.7700])),
    (('7.5BG', 5.0, 20.0), np.array([0.0830, 0.2780, 19.7700])),
    (('7.5BG', 5.0, 22.0), np.array([0.0710, 0.2740, 19.7700])),
    (('7.5BG', 5.0, 24.0), np.array([0.0620, 0.2700, 19.7700])),
    (('7.5BG', 5.0, 26.0), np.array([0.0550, 0.2680, 19.7700])),
    (('10BG', 5.0, 2.0), np.array([0.2796, 0.3111, 19.7700])),
    (('10BG', 5.0, 4.0), np.array([0.2512, 0.3040, 19.7700])),
    (('10BG', 5.0, 6.0), np.array([0.2234, 0.2952, 19.7700])),
    (('10BG', 5.0, 8.0), np.array([0.1970, 0.2860, 19.7700])),
    (('10BG', 5.0, 10.0), np.array([0.1716, 0.2760, 19.7700])),
    (('10BG', 5.0, 12.0), np.array([0.1485, 0.2662, 19.7700])),
    (('10BG', 5.0, 14.0), np.array([0.1308, 0.2582, 19.7700])),
    (('10BG', 5.0, 16.0), np.array([0.1108, 0.2489, 19.7700])),
    (('10BG', 5.0, 18.0), np.array([0.0930, 0.2400, 19.7700])),
    (('10BG', 5.0, 20.0), np.array([0.0790, 0.2320, 19.7700])),
    (('10BG', 5.0, 22.0), np.array([0.0660, 0.2260, 19.7700])),
    (('10BG', 5.0, 24.0), np.array([0.0580, 0.2210, 19.7700])),
    (('2.5B', 5.0, 2.0), np.array([0.2791, 0.3071, 19.7700])),
    (('2.5B', 5.0, 4.0), np.array([0.2492, 0.2954, 19.7700])),
    (('2.5B', 5.0, 6.0), np.array([0.2210, 0.2823, 19.7700])),
    (('2.5B', 5.0, 8.0), np.array([0.1947, 0.2687, 19.7700])),
    (('2.5B', 5.0, 10.0), np.array([0.1697, 0.2549, 19.7700])),
    (('2.5B', 5.0, 12.0), np.array([0.1461, 0.2406, 19.7700])),
    (('2.5B', 5.0, 14.0), np.array([0.1283, 0.2292, 19.7700])),
    (('2.5B', 5.0, 16.0), np.array([0.1090, 0.2166, 19.7700])),
    (('2.5B', 5.0, 18.0), np.array([0.0940, 0.2060, 19.7700])),
    (('2.5B', 5.0, 20.0), np.array([0.0770, 0.1940, 19.7700])),
    (('5B', 5.0, 2.0), np.array([0.2794, 0.3032, 19.7700])),
    (('5B', 5.0, 4.0), np.array([0.2493, 0.2879, 19.7700])),
    (('5B', 5.0, 6.0), np.array([0.2215, 0.2701, 19.7700])),
    (('5B', 5.0, 8.0), np.array([0.1958, 0.2519, 19.7700])),
    (('5B', 5.0, 10.0), np.array([0.1729, 0.2347, 19.7700])),
    (('5B', 5.0, 12.0), np.array([0.1505, 0.2172, 19.7700])),
    (('5B', 5.0, 14.0), np.array([0.1320, 0.2021, 19.7700])),
    (('5B', 5.0, 16.0), np.array([0.1132, 0.1863, 19.7700])),
    (('5B', 5.0, 18.0), np.array([0.0980, 0.1720, 19.7700])),
    (('5B', 5.0, 20.0), np.array([0.0860, 0.1600, 19.7700])),
    (('7.5B', 5.0, 2.0), np.array([0.2803, 0.3000, 19.7700])),
    (('7.5B', 5.0, 4.0), np.array([0.2511, 0.2808, 19.7700])),
    (('7.5B', 5.0, 6.0), np.array([0.2248, 0.2612, 19.7700])),
    (('7.5B', 5.0, 8.0), np.array([0.2007, 0.2417, 19.7700])),
    (('7.5B', 5.0, 10.0), np.array([0.1792, 0.2230, 19.7700])),
    (('7.5B', 5.0, 12.0), np.array([0.1584, 0.2042, 19.7700])),
    (('7.5B', 5.0, 14.0), np.array([0.1404, 0.1878, 19.7700])),
    (('7.5B', 5.0, 16.0), np.array([0.1230, 0.1711, 19.7700])),
    (('7.5B', 5.0, 18.0), np.array([0.1090, 0.1570, 19.7700])),
    (('7.5B', 5.0, 20.0), np.array([0.0980, 0.1460, 19.7700])),
    (('10B', 5.0, 2.0), np.array([0.2821, 0.2966, 19.7700])),
    (('10B', 5.0, 4.0), np.array([0.2547, 0.2757, 19.7700])),
    (('10B', 5.0, 6.0), np.array([0.2299, 0.2548, 19.7700])),
    (('10B', 5.0, 8.0), np.array([0.2067, 0.2344, 19.7700])),
    (('10B', 5.0, 10.0), np.array([0.1860, 0.2149, 19.7700])),
    (('10B', 5.0, 12.0), np.array([0.1666, 0.1964, 19.7700])),
    (('10B', 5.0, 14.0), np.array([0.1492, 0.1797, 19.7700])),
    (('10B', 5.0, 16.0), np.array([0.1326, 0.1632, 19.7700])),
    (('10B', 5.0, 18.0), np.array([0.1203, 0.1505, 19.7700])),
    (('10B', 5.0, 20.0), np.array([0.1080, 0.1370, 19.7700])),
    (('10B', 5.0, 22.0), np.array([0.0960, 0.1260, 19.7700])),
    (('2.5PB', 5.0, 2.0), np.array([0.2847, 0.2942, 19.7700])),
    (('2.5PB', 5.0, 4.0), np.array([0.2600, 0.2720, 19.7700])),
    (('2.5PB', 5.0, 6.0), np.array([0.2365, 0.2488, 19.7700])),
    (('2.5PB', 5.0, 8.0), np.array([0.2157, 0.2278, 19.7700])),
    (('2.5PB', 5.0, 10.0), np.array([0.1968, 0.2078, 19.7700])),
    (('2.5PB', 5.0, 12.0), np.array([0.1793, 0.1894, 19.7700])),
    (('2.5PB', 5.0, 14.0), np.array([0.1642, 0.1728, 19.7700])),
    (('2.5PB', 5.0, 16.0), np.array([0.1495, 0.1559, 19.7700])),
    (('2.5PB', 5.0, 18.0), np.array([0.1363, 0.1410, 19.7700])),
    (('2.5PB', 5.0, 20.0), np.array([0.1240, 0.1280, 19.7700])),
    (('2.5PB', 5.0, 22.0), np.array([0.1140, 0.1160, 19.7700])),
    (('2.5PB', 5.0, 24.0), np.array([0.1030, 0.1040, 19.7700])),
    (('5PB', 5.0, 2.0), np.array([0.2882, 0.2923, 19.7700])),
    (('5PB', 5.0, 4.0), np.array([0.2662, 0.2687, 19.7700])),
    (('5PB', 5.0, 6.0), np.array([0.2447, 0.2449, 19.7700])),
    (('5PB', 5.0, 8.0), np.array([0.2255, 0.2239, 19.7700])),
    (('5PB', 5.0, 10.0), np.array([0.2080, 0.2041, 19.7700])),
    (('5PB', 5.0, 12.0), np.array([0.1918, 0.1858, 19.7700])),
    (('5PB', 5.0, 14.0), np.array([0.1773, 0.1689, 19.7700])),
    (('5PB', 5.0, 16.0), np.array([0.1638, 0.1521, 19.7700])),
    (('5PB', 5.0, 18.0), np.array([0.1518, 0.1365, 19.7700])),
    (('5PB', 5.0, 20.0), np.array([0.1420, 0.1230, 19.7700])),
    (('5PB', 5.0, 22.0), np.array([0.1340, 0.1130, 19.7700])),
    (('5PB', 5.0, 24.0), np.array([0.1240, 0.1000, 19.7700])),
    (('5PB', 5.0, 26.0), np.array([0.1160, 0.0890, 19.7700])),
    (('5PB', 5.0, 28.0), np.array([0.1090, 0.0790, 19.7700])),
    (('5PB', 5.0, 30.0), np.array([0.1040, 0.0700, 19.7700])),
    (('5PB', 5.0, 32.0), np.array([0.0970, 0.0620, 19.7700])),
    (('7.5PB', 5.0, 2.0), np.array([0.2918, 0.2908, 19.7700])),
    (('7.5PB', 5.0, 4.0), np.array([0.2739, 0.2666, 19.7700])),
    (('7.5PB', 5.0, 6.0), np.array([0.2563, 0.2417, 19.7700])),
    (('7.5PB', 5.0, 8.0), np.array([0.2417, 0.2204, 19.7700])),
    (('7.5PB', 5.0, 10.0), np.array([0.2285, 0.2020, 19.7700])),
    (('7.5PB', 5.0, 12.0), np.array([0.2157, 0.1830, 19.7700])),
    (('7.5PB', 5.0, 14.0), np.array([0.2042, 0.1661, 19.7700])),
    (('7.5PB', 5.0, 16.0), np.array([0.1945, 0.1511, 19.7700])),
    (('7.5PB', 5.0, 18.0), np.array([0.1862, 0.1365, 19.7700])),
    (('7.5PB', 5.0, 20.0), np.array([0.1794, 0.1239, 19.7700])),
    (('7.5PB', 5.0, 22.0), np.array([0.1750, 0.1140, 19.7700])),
    (('7.5PB', 5.0, 24.0), np.array([0.1700, 0.1030, 19.7700])),
    (('7.5PB', 5.0, 26.0), np.array([0.1650, 0.0940, 19.7700])),
    (('7.5PB', 5.0, 28.0), np.array([0.1620, 0.0860, 19.7700])),
    (('7.5PB', 5.0, 30.0), np.array([0.1580, 0.0780, 19.7700])),
    (('7.5PB', 5.0, 32.0), np.array([0.1540, 0.0700, 19.7700])),
    (('7.5PB', 5.0, 34.0), np.array([0.1500, 0.0620, 19.7700])),
    (('10PB', 5.0, 2.0), np.array([0.2959, 0.2905, 19.7700])),
    (('10PB', 5.0, 4.0), np.array([0.2821, 0.2659, 19.7700])),
    (('10PB', 5.0, 6.0), np.array([0.2686, 0.2412, 19.7700])),
    (('10PB', 5.0, 8.0), np.array([0.2572, 0.2211, 19.7700])),
    (('10PB', 5.0, 10.0), np.array([0.2478, 0.2030, 19.7700])),
    (('10PB', 5.0, 12.0), np.array([0.2384, 0.1857, 19.7700])),
    (('10PB', 5.0, 14.0), np.array([0.2299, 0.1698, 19.7700])),
    (('10PB', 5.0, 16.0), np.array([0.2224, 0.1555, 19.7700])),
    (('10PB', 5.0, 18.0), np.array([0.2174, 0.1444, 19.7700])),
    (('10PB', 5.0, 20.0), np.array([0.2121, 0.1329, 19.7700])),
    (('10PB', 5.0, 22.0), np.array([0.2082, 0.1225, 19.7700])),
    (('10PB', 5.0, 24.0), np.array([0.2040, 0.1130, 19.7700])),
    (('10PB', 5.0, 26.0), np.array([0.2000, 0.1030, 19.7700])),
    (('10PB', 5.0, 28.0), np.array([0.1970, 0.0950, 19.7700])),
    (('10PB', 5.0, 30.0), np.array([0.1930, 0.0860, 19.7700])),
    (('10PB', 5.0, 32.0), np.array([0.1890, 0.0780, 19.7700])),
    (('10PB', 5.0, 34.0), np.array([0.1860, 0.0710, 19.7700])),
    (('10PB', 5.0, 36.0), np.array([0.1830, 0.0630, 19.7700])),
    (('2.5P', 5.0, 2.0), np.array([0.3000, 0.2912, 19.7700])),
    (('2.5P', 5.0, 4.0), np.array([0.2898, 0.2667, 19.7700])),
    (('2.5P', 5.0, 6.0), np.array([0.2806, 0.2444, 19.7700])),
    (('2.5P', 5.0, 8.0), np.array([0.2728, 0.2240, 19.7700])),
    (('2.5P', 5.0, 10.0), np.array([0.2665, 0.2075, 19.7700])),
    (('2.5P', 5.0, 12.0), np.array([0.2608, 0.1913, 19.7700])),
    (('2.5P', 5.0, 14.0), np.array([0.2560, 0.1774, 19.7700])),
    (('2.5P', 5.0, 16.0), np.array([0.2515, 0.1644, 19.7700])),
    (('2.5P', 5.0, 18.0), np.array([0.2476, 0.1532, 19.7700])),
    (('2.5P', 5.0, 20.0), np.array([0.2438, 0.1419, 19.7700])),
    (('2.5P', 5.0, 22.0), np.array([0.2402, 0.1315, 19.7700])),
    (('2.5P', 5.0, 24.0), np.array([0.2372, 0.1223, 19.7700])),
    (('2.5P', 5.0, 26.0), np.array([0.2348, 0.1140, 19.7700])),
    (('2.5P', 5.0, 28.0), np.array([0.2310, 0.1040, 19.7700])),
    (('2.5P', 5.0, 30.0), np.array([0.2290, 0.0970, 19.7700])),
    (('2.5P', 5.0, 32.0), np.array([0.2260, 0.0880, 19.7700])),
    (('2.5P', 5.0, 34.0), np.array([0.2230, 0.0800, 19.7700])),
    (('2.5P', 5.0, 36.0), np.array([0.2210, 0.0720, 19.7700])),
    (('2.5P', 5.0, 38.0), np.array([0.2190, 0.0660, 19.7700])),
    (('5P', 5.0, 2.0), np.array([0.3045, 0.2928, 19.7700])),
    (('5P', 5.0, 4.0), np.array([0.2986, 0.2699, 19.7700])),
    (('5P', 5.0, 6.0), np.array([0.2932, 0.2487, 19.7700])),
    (('5P', 5.0, 8.0), np.array([0.2885, 0.2296, 19.7700])),
    (('5P', 5.0, 10.0), np.array([0.2845, 0.2137, 19.7700])),
    (('5P', 5.0, 12.0), np.array([0.2806, 0.1977, 19.7700])),
    (('5P', 5.0, 14.0), np.array([0.2775, 0.1847, 19.7700])),
    (('5P', 5.0, 16.0), np.array([0.2744, 0.1718, 19.7700])),
    (('5P', 5.0, 18.0), np.array([0.2718, 0.1604, 19.7700])),
    (('5P', 5.0, 20.0), np.array([0.2694, 0.1499, 19.7700])),
    (('5P', 5.0, 22.0), np.array([0.2673, 0.1398, 19.7700])),
    (('5P', 5.0, 24.0), np.array([0.2652, 0.1304, 19.7700])),
    (('5P', 5.0, 26.0), np.array([0.2635, 0.1224, 19.7700])),
    (('5P', 5.0, 28.0), np.array([0.2618, 0.1135, 19.7700])),
    (('5P', 5.0, 30.0), np.array([0.2590, 0.1050, 19.7700])),
    (('5P', 5.0, 32.0), np.array([0.2560, 0.0960, 19.7700])),
    (('5P', 5.0, 34.0), np.array([0.2550, 0.0880, 19.7700])),
    (('5P', 5.0, 36.0), np.array([0.2530, 0.0810, 19.7700])),
    (('5P', 5.0, 38.0), np.array([0.2520, 0.0750, 19.7700])),
    (('5P', 5.0, 40.0), np.array([0.2510, 0.0690, 19.7700])),
    (('7.5P', 5.0, 2.0), np.array([0.3103, 0.2959, 19.7700])),
    (('7.5P', 5.0, 4.0), np.array([0.3100, 0.2750, 19.7700])),
    (('7.5P', 5.0, 6.0), np.array([0.3093, 0.2555, 19.7700])),
    (('7.5P', 5.0, 8.0), np.array([0.3087, 0.2375, 19.7700])),
    (('7.5P', 5.0, 10.0), np.array([0.3080, 0.2230, 19.7700])),
    (('7.5P', 5.0, 12.0), np.array([0.3071, 0.2080, 19.7700])),
    (('7.5P', 5.0, 14.0), np.array([0.3068, 0.1951, 19.7700])),
    (('7.5P', 5.0, 16.0), np.array([0.3060, 0.1830, 19.7700])),
    (('7.5P', 5.0, 18.0), np.array([0.3052, 0.1711, 19.7700])),
    (('7.5P', 5.0, 20.0), np.array([0.3042, 0.1606, 19.7700])),
    (('7.5P', 5.0, 22.0), np.array([0.3038, 0.1500, 19.7700])),
    (('7.5P', 5.0, 24.0), np.array([0.3030, 0.1423, 19.7700])),
    (('7.5P', 5.0, 26.0), np.array([0.3022, 0.1331, 19.7700])),
    (('7.5P', 5.0, 28.0), np.array([0.3018, 0.1253, 19.7700])),
    (('7.5P', 5.0, 30.0), np.array([0.3010, 0.1170, 19.7700])),
    (('7.5P', 5.0, 32.0), np.array([0.3000, 0.1080, 19.7700])),
    (('7.5P', 5.0, 34.0), np.array([0.3000, 0.1000, 19.7700])),
    (('7.5P', 5.0, 36.0), np.array([0.2990, 0.0930, 19.7700])),
    (('7.5P', 5.0, 38.0), np.array([0.2980, 0.0880, 19.7700])),
    (('7.5P', 5.0, 40.0), np.array([0.2980, 0.0810, 19.7700])),
    (('10P', 5.0, 2.0), np.array([0.3148, 0.2986, 19.7700])),
    (('10P', 5.0, 4.0), np.array([0.3198, 0.2807, 19.7700])),
    (('10P', 5.0, 6.0), np.array([0.3243, 0.2630, 19.7700])),
    (('10P', 5.0, 8.0), np.array([0.3280, 0.2464, 19.7700])),
    (('10P', 5.0, 10.0), np.array([0.3308, 0.2328, 19.7700])),
    (('10P', 5.0, 12.0), np.array([0.3335, 0.2187, 19.7700])),
    (('10P', 5.0, 14.0), np.array([0.3360, 0.2066, 19.7700])),
    (('10P', 5.0, 16.0), np.array([0.3382, 0.1951, 19.7700])),
    (('10P', 5.0, 18.0), np.array([0.3401, 0.1840, 19.7700])),
    (('10P', 5.0, 20.0), np.array([0.3422, 0.1735, 19.7700])),
    (('10P', 5.0, 22.0), np.array([0.3437, 0.1644, 19.7700])),
    (('10P', 5.0, 24.0), np.array([0.3450, 0.1555, 19.7700])),
    (('10P', 5.0, 26.0), np.array([0.3468, 0.1460, 19.7700])),
    (('10P', 5.0, 28.0), np.array([0.3478, 0.1388, 19.7700])),
    (('10P', 5.0, 30.0), np.array([0.3490, 0.1308, 19.7700])),
    (('10P', 5.0, 32.0), np.array([0.3500, 0.1240, 19.7700])),
    (('10P', 5.0, 34.0), np.array([0.3510, 0.1150, 19.7700])),
    (('10P', 5.0, 36.0), np.array([0.3520, 0.1070, 19.7700])),
    (('10P', 5.0, 38.0), np.array([0.3530, 0.1020, 19.7700])),
    (('10P', 5.0, 40.0), np.array([0.3540, 0.0960, 19.7700])),
    (('2.5RP', 5.0, 2.0), np.array([0.3199, 0.3019, 19.7700])),
    (('2.5RP', 5.0, 4.0), np.array([0.3298, 0.2869, 19.7700])),
    (('2.5RP', 5.0, 6.0), np.array([0.3396, 0.2718, 19.7700])),
    (('2.5RP', 5.0, 8.0), np.array([0.3490, 0.2570, 19.7700])),
    (('2.5RP', 5.0, 10.0), np.array([0.3560, 0.2452, 19.7700])),
    (('2.5RP', 5.0, 12.0), np.array([0.3635, 0.2325, 19.7700])),
    (('2.5RP', 5.0, 14.0), np.array([0.3703, 0.2211, 19.7700])),
    (('2.5RP', 5.0, 16.0), np.array([0.3763, 0.2108, 19.7700])),
    (('2.5RP', 5.0, 18.0), np.array([0.3821, 0.2007, 19.7700])),
    (('2.5RP', 5.0, 20.0), np.array([0.3873, 0.1909, 19.7700])),
    (('2.5RP', 5.0, 22.0), np.array([0.3924, 0.1814, 19.7700])),
    (('2.5RP', 5.0, 24.0), np.array([0.3965, 0.1738, 19.7700])),
    (('2.5RP', 5.0, 26.0), np.array([0.4011, 0.1652, 19.7700])),
    (('2.5RP', 5.0, 28.0), np.array([0.4050, 0.1570, 19.7700])),
    (('2.5RP', 5.0, 30.0), np.array([0.4090, 0.1490, 19.7700])),
    (('2.5RP', 5.0, 32.0), np.array([0.4130, 0.1420, 19.7700])),
    (('2.5RP', 5.0, 34.0), np.array([0.4160, 0.1350, 19.7700])),
    (('2.5RP', 5.0, 36.0), np.array([0.4210, 0.1260, 19.7700])),
    (('5RP', 5.0, 2.0), np.array([0.3256, 0.3065, 19.7700])),
    (('5RP', 5.0, 4.0), np.array([0.3421, 0.2954, 19.7700])),
    (('5RP', 5.0, 6.0), np.array([0.3585, 0.2842, 19.7700])),
    (('5RP', 5.0, 8.0), np.array([0.3748, 0.2729, 19.7700])),
    (('5RP', 5.0, 10.0), np.array([0.3880, 0.2630, 19.7700])),
    (('5RP', 5.0, 12.0), np.array([0.4022, 0.2523, 19.7700])),
    (('5RP', 5.0, 14.0), np.array([0.4142, 0.2428, 19.7700])),
    (('5RP', 5.0, 16.0), np.array([0.4261, 0.2331, 19.7700])),
    (('5RP', 5.0, 18.0), np.array([0.4372, 0.2242, 19.7700])),
    (('5RP', 5.0, 20.0), np.array([0.4484, 0.2150, 19.7700])),
    (('5RP', 5.0, 22.0), np.array([0.4581, 0.2068, 19.7700])),
    (('5RP', 5.0, 24.0), np.array([0.4683, 0.1978, 19.7700])),
    (('5RP', 5.0, 26.0), np.array([0.4760, 0.1900, 19.7700])),
    (('5RP', 5.0, 28.0), np.array([0.4830, 0.1830, 19.7700])),
    (('5RP', 5.0, 30.0), np.array([0.4910, 0.1760, 19.7700])),
    (('5RP', 5.0, 32.0), np.array([0.4980, 0.1690, 19.7700])),
    (('7.5RP', 5.0, 2.0), np.array([0.3296, 0.3098, 19.7700])),
    (('7.5RP', 5.0, 4.0), np.array([0.3515, 0.3024, 19.7700])),
    (('7.5RP', 5.0, 6.0), np.array([0.3726, 0.2941, 19.7700])),
    (('7.5RP', 5.0, 8.0), np.array([0.3932, 0.2852, 19.7700])),
    (('7.5RP', 5.0, 10.0), np.array([0.4108, 0.2773, 19.7700])),
    (('7.5RP', 5.0, 12.0), np.array([0.4303, 0.2675, 19.7700])),
    (('7.5RP', 5.0, 14.0), np.array([0.4454, 0.2596, 19.7700])),
    (('7.5RP', 5.0, 16.0), np.array([0.4617, 0.2506, 19.7700])),
    (('7.5RP', 5.0, 18.0), np.array([0.4761, 0.2421, 19.7700])),
    (('7.5RP', 5.0, 20.0), np.array([0.4915, 0.2330, 19.7700])),
    (('7.5RP', 5.0, 22.0), np.array([0.5045, 0.2248, 19.7700])),
    (('7.5RP', 5.0, 24.0), np.array([0.5170, 0.2170, 19.7700])),
    (('7.5RP', 5.0, 26.0), np.array([0.5260, 0.2100, 19.7700])),
    (('7.5RP', 5.0, 28.0), np.array([0.5360, 0.2040, 19.7700])),
    (('7.5RP', 5.0, 30.0), np.array([0.5460, 0.1970, 19.7700])),
    (('10RP', 5.0, 2.0), np.array([0.3332, 0.3131, 19.7700])),
    (('10RP', 5.0, 4.0), np.array([0.3594, 0.3090, 19.7700])),
    (('10RP', 5.0, 6.0), np.array([0.3851, 0.3039, 19.7700])),
    (('10RP', 5.0, 8.0), np.array([0.4105, 0.2980, 19.7700])),
    (('10RP', 5.0, 10.0), np.array([0.4332, 0.2918, 19.7700])),
    (('10RP', 5.0, 12.0), np.array([0.4579, 0.2841, 19.7700])),
    (('10RP', 5.0, 14.0), np.array([0.4767, 0.2776, 19.7700])),
    (('10RP', 5.0, 16.0), np.array([0.4986, 0.2695, 19.7700])),
    (('10RP', 5.0, 18.0), np.array([0.5185, 0.2620, 19.7700])),
    (('10RP', 5.0, 20.0), np.array([0.5396, 0.2535, 19.7700])),
    (('10RP', 5.0, 22.0), np.array([0.5540, 0.2460, 19.7700])),
    (('10RP', 5.0, 24.0), np.array([0.5700, 0.2400, 19.7700])),
    (('10RP', 5.0, 26.0), np.array([0.5820, 0.2340, 19.7700])),
    (('10RP', 5.0, 28.0), np.array([0.5930, 0.2290, 19.7700])),
    (('2.5R', 5.0, 2.0), np.array([0.3360, 0.3158, 19.7700])),
    (('2.5R', 5.0, 4.0), np.array([0.3660, 0.3148, 19.7700])),
    (('2.5R', 5.0, 6.0), np.array([0.3960, 0.3130, 19.7700])),
    (('2.5R', 5.0, 8.0), np.array([0.4252, 0.3101, 19.7700])),
    (('2.5R', 5.0, 10.0), np.array([0.4533, 0.3058, 19.7700])),
    (('2.5R', 5.0, 12.0), np.array([0.4820, 0.3002, 19.7700])),
    (('2.5R', 5.0, 14.0), np.array([0.5047, 0.2950, 19.7700])),
    (('2.5R', 5.0, 16.0), np.array([0.5300, 0.2880, 19.7700])),
    (('2.5R', 5.0, 18.0), np.array([0.5540, 0.2804, 19.7700])),
    (('2.5R', 5.0, 20.0), np.array([0.5784, 0.2719, 19.7700])),
    (('2.5R', 5.0, 22.0), np.array([0.5930, 0.2650, 19.7700])),
    (('2.5R', 5.0, 24.0), np.array([0.6070, 0.2590, 19.7700])),
    (('2.5R', 5.0, 26.0), np.array([0.6200, 0.2530, 19.7700])),
    (('2.5R', 5.0, 28.0), np.array([0.6310, 0.2480, 19.7700])),
    (('5R', 5.0, 2.0), np.array([0.3392, 0.3192, 19.7700])),
    (('5R', 5.0, 4.0), np.array([0.3740, 0.3220, 19.7700])),
    (('5R', 5.0, 6.0), np.array([0.4078, 0.3238, 19.7700])),
    (('5R', 5.0, 8.0), np.array([0.4413, 0.3240, 19.7700])),
    (('5R', 5.0, 10.0), np.array([0.4747, 0.3227, 19.7700])),
    (('5R', 5.0, 12.0), np.array([0.5071, 0.3194, 19.7700])),
    (('5R', 5.0, 14.0), np.array([0.5341, 0.3158, 19.7700])),
    (('5R', 5.0, 16.0), np.array([0.5637, 0.3102, 19.7700])),
    (('5R', 5.0, 18.0), np.array([0.5918, 0.3038, 19.7700])),
    (('5R', 5.0, 20.0), np.array([0.6142, 0.2970, 19.7700])),
    (('5R', 5.0, 22.0), np.array([0.6340, 0.2900, 19.7700])),
    (('5R', 5.0, 24.0), np.array([0.6480, 0.2840, 19.7700])),
    (('5R', 5.0, 26.0), np.array([0.6620, 0.2790, 19.7700])),
    (('5R', 5.0, 28.0), np.array([0.6740, 0.2740, 19.7700])),
    (('7.5R', 5.0, 2.0), np.array([0.3425, 0.3229, 19.7700])),
    (('7.5R', 5.0, 4.0), np.array([0.3806, 0.3294, 19.7700])),
    (('7.5R', 5.0, 6.0), np.array([0.4180, 0.3348, 19.7700])),
    (('7.5R', 5.0, 8.0), np.array([0.4563, 0.3387, 19.7700])),
    (('7.5R', 5.0, 10.0), np.array([0.4927, 0.3399, 19.7700])),
    (('7.5R', 5.0, 12.0), np.array([0.5280, 0.3389, 19.7700])),
    (('7.5R', 5.0, 14.0), np.array([0.5590, 0.3370, 19.7700])),
    (('7.5R', 5.0, 16.0), np.array([0.5901, 0.3331, 19.7700])),
    (('7.5R', 5.0, 18.0), np.array([0.6161, 0.3277, 19.7700])),
    (('7.5R', 5.0, 20.0), np.array([0.6388, 0.3216, 19.7700])),
    (('7.5R', 5.0, 22.0), np.array([0.6610, 0.3170, 19.7700])),
    (('7.5R', 5.0, 24.0), np.array([0.6800, 0.3120, 19.7700])),
    (('7.5R', 5.0, 26.0), np.array([0.6920, 0.3070, 19.7700])),
    (('7.5R', 5.0, 28.0), np.array([0.7080, 0.3020, 19.7700])),
    (('10R', 5.0, 2.0), np.array([0.3465, 0.3278, 19.7700])),
    (('10R', 5.0, 4.0), np.array([0.3879, 0.3398, 19.7700])),
    (('10R', 5.0, 6.0), np.array([0.4299, 0.3499, 19.7700])),
    (('10R', 5.0, 8.0), np.array([0.4713, 0.3575, 19.7700])),
    (('10R', 5.0, 10.0), np.array([0.5113, 0.3630, 19.7700])),
    (('10R', 5.0, 12.0), np.array([0.5481, 0.3660, 19.7700])),
    (('10R', 5.0, 14.0), np.array([0.5771, 0.3664, 19.7700])),
    (('10R', 5.0, 16.0), np.array([0.6037, 0.3657, 19.7700])),
    (('10R', 5.0, 18.0), np.array([0.6297, 0.3642, 19.7700])),
    (('10R', 5.0, 20.0), np.array([0.6550, 0.3610, 19.7700])),
    (('10R', 5.0, 22.0), np.array([0.6800, 0.3570, 19.7700])),
    (('10R', 5.0, 24.0), np.array([0.7020, 0.3540, 19.7700])),
    (('10R', 5.0, 26.0), np.array([0.7210, 0.3500, 19.7700])),
    (('2.5YR', 5.0, 2.0), np.array([0.3506, 0.3337, 19.7700])),
    (('2.5YR', 5.0, 4.0), np.array([0.3925, 0.3494, 19.7700])),
    (('2.5YR', 5.0, 6.0), np.array([0.4365, 0.3640, 19.7700])),
    (('2.5YR', 5.0, 8.0), np.array([0.4795, 0.3758, 19.7700])),
    (('2.5YR', 5.0, 10.0), np.array([0.5175, 0.3844, 19.7700])),
    (('2.5YR', 5.0, 12.0), np.array([0.5482, 0.3909, 19.7700])),
    (('2.5YR', 5.0, 14.0), np.array([0.5731, 0.3953, 19.7700])),
    (('2.5YR', 5.0, 16.0), np.array([0.5933, 0.3989, 19.7700])),
    (('2.5YR', 5.0, 18.0), np.array([0.6170, 0.4020, 19.7700])),
    (('2.5YR', 5.0, 20.0), np.array([0.6360, 0.4060, 19.7700])),
    (('5YR', 5.0, 2.0), np.array([0.3530, 0.3395, 19.7700])),
    (('5YR', 5.0, 4.0), np.array([0.3968, 0.3614, 19.7700])),
    (('5YR', 5.0, 6.0), np.array([0.4420, 0.3808, 19.7700])),
    (('5YR', 5.0, 8.0), np.array([0.4830, 0.3960, 19.7700])),
    (('5YR', 5.0, 10.0), np.array([0.5161, 0.4064, 19.7700])),
    (('5YR', 5.0, 12.0), np.array([0.5422, 0.4141, 19.7700])),
    (('5YR', 5.0, 14.0), np.array([0.5642, 0.4201, 19.7700])),
    (('5YR', 5.0, 16.0), np.array([0.5840, 0.4250, 19.7700])),
    (('5YR', 5.0, 18.0), np.array([0.6010, 0.4290, 19.7700])),
    (('7.5YR', 5.0, 2.0), np.array([0.3540, 0.3445, 19.7700])),
    (('7.5YR', 5.0, 4.0), np.array([0.3991, 0.3714, 19.7700])),
    (('7.5YR', 5.0, 6.0), np.array([0.4440, 0.3954, 19.7700])),
    (('7.5YR', 5.0, 8.0), np.array([0.4820, 0.4141, 19.7700])),
    (('7.5YR', 5.0, 10.0), np.array([0.5108, 0.4276, 19.7700])),
    (('7.5YR', 5.0, 12.0), np.array([0.5335, 0.4378, 19.7700])),
    (('7.5YR', 5.0, 14.0), np.array([0.5506, 0.4450, 19.7700])),
    (('7.5YR', 5.0, 16.0), np.array([0.5680, 0.4530, 19.7700])),
    (('10YR', 5.0, 2.0), np.array([0.3546, 0.3514, 19.7700])),
    (('10YR', 5.0, 4.0), np.array([0.3995, 0.3840, 19.7700])),
    (('10YR', 5.0, 6.0), np.array([0.4428, 0.4128, 19.7700])),
    (('10YR', 5.0, 8.0), np.array([0.4770, 0.4338, 19.7700])),
    (('10YR', 5.0, 10.0), np.array([0.5025, 0.4489, 19.7700])),
    (('10YR', 5.0, 12.0), np.array([0.5211, 0.4600, 19.7700])),
    (('10YR', 5.0, 14.0), np.array([0.5390, 0.4680, 19.7700])),
    (('10YR', 5.0, 16.0), np.array([0.5520, 0.4770, 19.7700])),
    (('2.5Y', 5.0, 2.0), np.array([0.3534, 0.3570, 19.7700])),
    (('2.5Y', 5.0, 4.0), np.array([0.3968, 0.3954, 19.7700])),
    (('2.5Y', 5.0, 6.0), np.array([0.4380, 0.4292, 19.7700])),
    (('2.5Y', 5.0, 8.0), np.array([0.4685, 0.4524, 19.7700])),
    (('2.5Y', 5.0, 10.0), np.array([0.4905, 0.4683, 19.7700])),
    (('2.5Y', 5.0, 12.0), np.array([0.5082, 0.4812, 19.7700])),
    (('2.5Y', 5.0, 14.0), np.array([0.5230, 0.4910, 19.7700])),
    (('5Y', 5.0, 2.0), np.array([0.3500, 0.3620, 19.7700])),
    (('5Y', 5.0, 4.0), np.array([0.3915, 0.4057, 19.7700])),
    (('5Y', 5.0, 6.0), np.array([0.4302, 0.4435, 19.7700])),
    (('5Y', 5.0, 8.0), np.array([0.4579, 0.4692, 19.7700])),
    (('5Y', 5.0, 10.0), np.array([0.4777, 0.4876, 19.7700])),
    (('5Y', 5.0, 12.0), np.array([0.4932, 0.5019, 19.7700])),
    (('5Y', 5.0, 14.0), np.array([0.5070, 0.5120, 19.7700])),
    (('7.5Y', 5.0, 2.0), np.array([0.3470, 0.3640, 19.7700])),
    (('7.5Y', 5.0, 4.0), np.array([0.3850, 0.4120, 19.7700])),
    (('7.5Y', 5.0, 6.0), np.array([0.4199, 0.4551, 19.7700])),
    (('7.5Y', 5.0, 8.0), np.array([0.4450, 0.4850, 19.7700])),
    (('7.5Y', 5.0, 10.0), np.array([0.4632, 0.5057, 19.7700])),
    (('7.5Y', 5.0, 12.0), np.array([0.4767, 0.5208, 19.7700])),
    (('7.5Y', 5.0, 14.0), np.array([0.4890, 0.5350, 19.7700])),
    (('10Y', 6.0, 2.0), np.array([0.3398, 0.3611, 30.0300])),
    (('10Y', 6.0, 4.0), np.array([0.3679, 0.4033, 30.0300])),
    (('10Y', 6.0, 6.0), np.array([0.3960, 0.4452, 30.0300])),
    (('10Y', 6.0, 8.0), np.array([0.4201, 0.4812, 30.0300])),
    (('10Y', 6.0, 10.0), np.array([0.4372, 0.5068, 30.0300])),
    (('10Y', 6.0, 12.0), np.array([0.4488, 0.5237, 30.0300])),
    (('10Y', 6.0, 14.0), np.array([0.4593, 0.5392, 30.0300])),
    (('10Y', 6.0, 16.0), np.array([0.4650, 0.5460, 30.0300])),
    (('2.5GY', 6.0, 2.0), np.array([0.3342, 0.3607, 30.0300])),
    (('2.5GY', 6.0, 4.0), np.array([0.3572, 0.4038, 30.0300])),
    (('2.5GY', 6.0, 6.0), np.array([0.3799, 0.4470, 30.0300])),
    (('2.5GY', 6.0, 8.0), np.array([0.4006, 0.4885, 30.0300])),
    (('2.5GY', 6.0, 10.0), np.array([0.4159, 0.5190, 30.0300])),
    (('2.5GY', 6.0, 12.0), np.array([0.4269, 0.5414, 30.0300])),
    (('2.5GY', 6.0, 14.0), np.array([0.4354, 0.5594, 30.0300])),
    (('2.5GY', 6.0, 16.0), np.array([0.4390, 0.5680, 30.0300])),
    (('5GY', 6.0, 2.0), np.array([0.3288, 0.3592, 30.0300])),
    (('5GY', 6.0, 4.0), np.array([0.3461, 0.4008, 30.0300])),
    (('5GY', 6.0, 6.0), np.array([0.3622, 0.4438, 30.0300])),
    (('5GY', 6.0, 8.0), np.array([0.3772, 0.4880, 30.0300])),
    (('5GY', 6.0, 10.0), np.array([0.3891, 0.5264, 30.0300])),
    (('5GY', 6.0, 12.0), np.array([0.3980, 0.5564, 30.0300])),
    (('5GY', 6.0, 14.0), np.array([0.4042, 0.5788, 30.0300])),
    (('5GY', 6.0, 16.0), np.array([0.4090, 0.5940, 30.0300])),
    (('5GY', 6.0, 18.0), np.array([0.4130, 0.6070, 30.0300])),
    (('7.5GY', 6.0, 2.0), np.array([0.3193, 0.3550, 30.0300])),
    (('7.5GY', 6.0, 4.0), np.array([0.3275, 0.3922, 30.0300])),
    (('7.5GY', 6.0, 6.0), np.array([0.3351, 0.4321, 30.0300])),
    (('7.5GY', 6.0, 8.0), np.array([0.3418, 0.4768, 30.0300])),
    (('7.5GY', 6.0, 10.0), np.array([0.3463, 0.5196, 30.0300])),
    (('7.5GY', 6.0, 12.0), np.array([0.3488, 0.5596, 30.0300])),
    (('7.5GY', 6.0, 14.0), np.array([0.3498, 0.5985, 30.0300])),
    (('7.5GY', 6.0, 16.0), np.array([0.3498, 0.6282, 30.0300])),
    (('7.5GY', 6.0, 18.0), np.array([0.3490, 0.6500, 30.0300])),
    (('7.5GY', 6.0, 20.0), np.array([0.3480, 0.6670, 30.0300])),
    (('7.5GY', 6.0, 22.0), np.array([0.3470, 0.6880, 30.0300])),
    (('10GY', 6.0, 2.0), np.array([0.3112, 0.3496, 30.0300])),
    (('10GY', 6.0, 4.0), np.array([0.3124, 0.3822, 30.0300])),
    (('10GY', 6.0, 6.0), np.array([0.3128, 0.4175, 30.0300])),
    (('10GY', 6.0, 8.0), np.array([0.3116, 0.4563, 30.0300])),
    (('10GY', 6.0, 10.0), np.array([0.3086, 0.4949, 30.0300])),
    (('10GY', 6.0, 12.0), np.array([0.3037, 0.5358, 30.0300])),
    (('10GY', 6.0, 14.0), np.array([0.2962, 0.5802, 30.0300])),
    (('10GY', 6.0, 16.0), np.array([0.2872, 0.6199, 30.0300])),
    (('10GY', 6.0, 18.0), np.array([0.2763, 0.6616, 30.0300])),
    (('10GY', 6.0, 20.0), np.array([0.2648, 0.7004, 30.0300])),
    (('10GY', 6.0, 22.0), np.array([0.2530, 0.7380, 30.0300])),
    (('10GY', 6.0, 24.0), np.array([0.2380, 0.7820, 30.0300])),
    (('10GY', 6.0, 26.0), np.array([0.2250, 0.8200, 30.0300])),
    (('10GY', 6.0, 28.0), np.array([0.2080, 0.8630, 30.0300])),
    (('10GY', 6.0, 30.0), np.array([0.1880, 0.9050, 30.0300])),
    (('10GY', 6.0, 32.0), np.array([0.1710, 0.9390, 30.0300])),
    (('2.5G', 6.0, 2.0), np.array([0.3039, 0.3437, 30.0300])),
    (('2.5G', 6.0, 4.0), np.array([0.2967, 0.3695, 30.0300])),
    (('2.5G', 6.0, 6.0), np.array([0.2892, 0.3963, 30.0300])),
    (('2.5G', 6.0, 8.0), np.array([0.2799, 0.4239, 30.0300])),
    (('2.5G', 6.0, 10.0), np.array([0.2690, 0.4530, 30.0300])),
    (('2.5G', 6.0, 12.0), np.array([0.2574, 0.4814, 30.0300])),
    (('2.5G', 6.0, 14.0), np.array([0.2426, 0.5133, 30.0300])),
    (('2.5G', 6.0, 16.0), np.array([0.2278, 0.5430, 30.0300])),
    (('2.5G', 6.0, 18.0), np.array([0.2102, 0.5737, 30.0300])),
    (('2.5G', 6.0, 20.0), np.array([0.1922, 0.6035, 30.0300])),
    (('2.5G', 6.0, 22.0), np.array([0.1739, 0.6318, 30.0300])),
    (('2.5G', 6.0, 24.0), np.array([0.1536, 0.6605, 30.0300])),
    (('2.5G', 6.0, 26.0), np.array([0.1340, 0.6871, 30.0300])),
    (('2.5G', 6.0, 28.0), np.array([0.1145, 0.7122, 30.0300])),
    (('2.5G', 6.0, 30.0), np.array([0.0920, 0.7380, 30.0300])),
    (('2.5G', 6.0, 32.0), np.array([0.0690, 0.7640, 30.0300])),
    (('2.5G', 6.0, 34.0), np.array([0.0470, 0.7890, 30.0300])),
    (('5G', 6.0, 2.0), np.array([0.2988, 0.3382, 30.0300])),
    (('5G', 6.0, 4.0), np.array([0.2868, 0.3595, 30.0300])),
    (('5G', 6.0, 6.0), np.array([0.2748, 0.3795, 30.0300])),
    (('5G', 6.0, 8.0), np.array([0.2612, 0.3990, 30.0300])),
    (('5G', 6.0, 10.0), np.array([0.2466, 0.4181, 30.0300])),
    (('5G', 6.0, 12.0), np.array([0.2293, 0.4390, 30.0300])),
    (('5G', 6.0, 14.0), np.array([0.2130, 0.4571, 30.0300])),
    (('5G', 6.0, 16.0), np.array([0.1960, 0.4751, 30.0300])),
    (('5G', 6.0, 18.0), np.array([0.1785, 0.4924, 30.0300])),
    (('5G', 6.0, 20.0), np.array([0.1609, 0.5091, 30.0300])),
    (('5G', 6.0, 22.0), np.array([0.1432, 0.5252, 30.0300])),
    (('5G', 6.0, 24.0), np.array([0.1252, 0.5408, 30.0300])),
    (('5G', 6.0, 26.0), np.array([0.1079, 0.5560, 30.0300])),
    (('5G', 6.0, 28.0), np.array([0.0908, 0.5695, 30.0300])),
    (('5G', 6.0, 30.0), np.array([0.0710, 0.5860, 30.0300])),
    (('5G', 6.0, 32.0), np.array([0.0530, 0.5990, 30.0300])),
    (('5G', 6.0, 34.0), np.array([0.0310, 0.6140, 30.0300])),
    (('7.5G', 6.0, 2.0), np.array([0.2958, 0.3344, 30.0300])),
    (('7.5G', 6.0, 4.0), np.array([0.2807, 0.3522, 30.0300])),
    (('7.5G', 6.0, 6.0), np.array([0.2662, 0.3672, 30.0300])),
    (('7.5G', 6.0, 8.0), np.array([0.2510, 0.3829, 30.0300])),
    (('7.5G', 6.0, 10.0), np.array([0.2350, 0.3979, 30.0300])),
    (('7.5G', 6.0, 12.0), np.array([0.2171, 0.4138, 30.0300])),
    (('7.5G', 6.0, 14.0), np.array([0.2001, 0.4278, 30.0300])),
    (('7.5G', 6.0, 16.0), np.array([0.1832, 0.4414, 30.0300])),
    (('7.5G', 6.0, 18.0), np.array([0.1654, 0.4551, 30.0300])),
    (('7.5G', 6.0, 20.0), np.array([0.1485, 0.4677, 30.0300])),
    (('7.5G', 6.0, 22.0), np.array([0.1325, 0.4795, 30.0300])),
    (('7.5G', 6.0, 24.0), np.array([0.1159, 0.4910, 30.0300])),
    (('7.5G', 6.0, 26.0), np.array([0.1010, 0.5018, 30.0300])),
    (('7.5G', 6.0, 28.0), np.array([0.0858, 0.5127, 30.0300])),
    (('7.5G', 6.0, 30.0), np.array([0.0680, 0.5240, 30.0300])),
    (('7.5G', 6.0, 32.0), np.array([0.0520, 0.5350, 30.0300])),
    (('7.5G', 6.0, 34.0), np.array([0.0340, 0.5460, 30.0300])),
    (('10G', 6.0, 2.0), np.array([0.2929, 0.3303, 30.0300])),
    (('10G', 6.0, 4.0), np.array([0.2749, 0.3443, 30.0300])),
    (('10G', 6.0, 6.0), np.array([0.2591, 0.3558, 30.0300])),
    (('10G', 6.0, 8.0), np.array([0.2420, 0.3679, 30.0300])),
    (('10G', 6.0, 10.0), np.array([0.2247, 0.3796, 30.0300])),
    (('10G', 6.0, 12.0), np.array([0.2060, 0.3914, 30.0300])),
    (('10G', 6.0, 14.0), np.array([0.1895, 0.4015, 30.0300])),
    (('10G', 6.0, 16.0), np.array([0.1722, 0.4113, 30.0300])),
    (('10G', 6.0, 18.0), np.array([0.1551, 0.4208, 30.0300])),
    (('10G', 6.0, 20.0), np.array([0.1382, 0.4299, 30.0300])),
    (('10G', 6.0, 22.0), np.array([0.1230, 0.4378, 30.0300])),
    (('10G', 6.0, 24.0), np.array([0.1070, 0.4458, 30.0300])),
    (('10G', 6.0, 26.0), np.array([0.0941, 0.4520, 30.0300])),
    (('10G', 6.0, 28.0), np.array([0.0810, 0.4580, 30.0300])),
    (('10G', 6.0, 30.0), np.array([0.0660, 0.4650, 30.0300])),
    (('10G', 6.0, 32.0), np.array([0.0520, 0.4710, 30.0300])),
    (('10G', 6.0, 34.0), np.array([0.0350, 0.4800, 30.0300])),
    (('2.5BG', 6.0, 2.0), np.array([0.2902, 0.3268, 30.0300])),
    (('2.5BG', 6.0, 4.0), np.array([0.2702, 0.3369, 30.0300])),
    (('2.5BG', 6.0, 6.0), np.array([0.2526, 0.3448, 30.0300])),
    (('2.5BG', 6.0, 8.0), np.array([0.2332, 0.3522, 30.0300])),
    (('2.5BG', 6.0, 10.0), np.array([0.2148, 0.3584, 30.0300])),
    (('2.5BG', 6.0, 12.0), np.array([0.1954, 0.3645, 30.0300])),
    (('2.5BG', 6.0, 14.0), np.array([0.1779, 0.3699, 30.0300])),
    (('2.5BG', 6.0, 16.0), np.array([0.1600, 0.3748, 30.0300])),
    (('2.5BG', 6.0, 18.0), np.array([0.1428, 0.3790, 30.0300])),
    (('2.5BG', 6.0, 20.0), np.array([0.1269, 0.3829, 30.0300])),
    (('2.5BG', 6.0, 22.0), np.array([0.1120, 0.3860, 30.0300])),
    (('2.5BG', 6.0, 24.0), np.array([0.0960, 0.3890, 30.0300])),
    (('2.5BG', 6.0, 26.0), np.array([0.0830, 0.3920, 30.0300])),
    (('2.5BG', 6.0, 28.0), np.array([0.0740, 0.3940, 30.0300])),
    (('2.5BG', 6.0, 30.0), np.array([0.0620, 0.3970, 30.0300])),
    (('2.5BG', 6.0, 32.0), np.array([0.0500, 0.3980, 30.0300])),
    (('5BG', 6.0, 2.0), np.array([0.2872, 0.3219, 30.0300])),
    (('5BG', 6.0, 4.0), np.array([0.2648, 0.3262, 30.0300])),
    (('5BG', 6.0, 6.0), np.array([0.2441, 0.3290, 30.0300])),
    (('5BG', 6.0, 8.0), np.array([0.2236, 0.3311, 30.0300])),
    (('5BG', 6.0, 10.0), np.array([0.2037, 0.3329, 30.0300])),
    (('5BG', 6.0, 12.0), np.array([0.1844, 0.3337, 30.0300])),
    (('5BG', 6.0, 14.0), np.array([0.1662, 0.3343, 30.0300])),
    (('5BG', 6.0, 16.0), np.array([0.1491, 0.3345, 30.0300])),
    (('5BG', 6.0, 18.0), np.array([0.1325, 0.3345, 30.0300])),
    (('5BG', 6.0, 20.0), np.array([0.1168, 0.3344, 30.0300])),
    (('5BG', 6.0, 22.0), np.array([0.1010, 0.3320, 30.0300])),
    (('5BG', 6.0, 24.0), np.array([0.0860, 0.3320, 30.0300])),
    (('5BG', 6.0, 26.0), np.array([0.0750, 0.3310, 30.0300])),
    (('5BG', 6.0, 28.0), np.array([0.0680, 0.3300, 30.0300])),
    (('5BG', 6.0, 30.0), np.array([0.0600, 0.3300, 30.0300])),
    (('7.5BG', 6.0, 2.0), np.array([0.2849, 0.3172, 30.0300])),
    (('7.5BG', 6.0, 4.0), np.array([0.2604, 0.3169, 30.0300])),
    (('7.5BG', 6.0, 6.0), np.array([0.2384, 0.3155, 30.0300])),
    (('7.5BG', 6.0, 8.0), np.array([0.2171, 0.3138, 30.0300])),
    (('7.5BG', 6.0, 10.0), np.array([0.1961, 0.3110, 30.0300])),
    (('7.5BG', 6.0, 12.0), np.array([0.1762, 0.3081, 30.0300])),
    (('7.5BG', 6.0, 14.0), np.array([0.1585, 0.3052, 30.0300])),
    (('7.5BG', 6.0, 16.0), np.array([0.1408, 0.3017, 30.0300])),
    (('7.5BG', 6.0, 18.0), np.array([0.1248, 0.2981, 30.0300])),
    (('7.5BG', 6.0, 20.0), np.array([0.1080, 0.2940, 30.0300])),
    (('7.5BG', 6.0, 22.0), np.array([0.0920, 0.2900, 30.0300])),
    (('7.5BG', 6.0, 24.0), np.array([0.0780, 0.2870, 30.0300])),
    (('7.5BG', 6.0, 26.0), np.array([0.0680, 0.2830, 30.0300])),
    (('10BG', 6.0, 2.0), np.array([0.2837, 0.3132, 30.0300])),
    (('10BG', 6.0, 4.0), np.array([0.2578, 0.3078, 30.0300])),
    (('10BG', 6.0, 6.0), np.array([0.2335, 0.3015, 30.0300])),
    (('10BG', 6.0, 8.0), np.array([0.2116, 0.2950, 30.0300])),
    (('10BG', 6.0, 10.0), np.array([0.1909, 0.2881, 30.0300])),
    (('10BG', 6.0, 12.0), np.array([0.1698, 0.2802, 30.0300])),
    (('10BG', 6.0, 14.0), np.array([0.1518, 0.2729, 30.0300])),
    (('10BG', 6.0, 16.0), np.array([0.1337, 0.2651, 30.0300])),
    (('10BG', 6.0, 18.0), np.array([0.1181, 0.2581, 30.0300])),
    (('10BG', 6.0, 20.0), np.array([0.1010, 0.2510, 30.0300])),
    (('10BG', 6.0, 22.0), np.array([0.0860, 0.2440, 30.0300])),
    (('10BG', 6.0, 24.0), np.array([0.0720, 0.2370, 30.0300])),
    (('2.5B', 6.0, 2.0), np.array([0.2835, 0.3097, 30.0300])),
    (('2.5B', 6.0, 4.0), np.array([0.2571, 0.3008, 30.0300])),
    (('2.5B', 6.0, 6.0), np.array([0.2312, 0.2899, 30.0300])),
    (('2.5B', 6.0, 8.0), np.array([0.2080, 0.2789, 30.0300])),
    (('2.5B', 6.0, 10.0), np.array([0.1879, 0.2682, 30.0300])),
    (('2.5B', 6.0, 12.0), np.array([0.1660, 0.2561, 30.0300])),
    (('2.5B', 6.0, 14.0), np.array([0.1480, 0.2459, 30.0300])),
    (('2.5B', 6.0, 16.0), np.array([0.1294, 0.2348, 30.0300])),
    (('2.5B', 6.0, 18.0), np.array([0.1140, 0.2260, 30.0300])),
    (('2.5B', 6.0, 20.0), np.array([0.0990, 0.2160, 30.0300])),
    (('2.5B', 6.0, 22.0), np.array([0.0850, 0.2050, 30.0300])),
    (('5B', 6.0, 2.0), np.array([0.2842, 0.3063, 30.0300])),
    (('5B', 6.0, 4.0), np.array([0.2579, 0.2938, 30.0300])),
    (('5B', 6.0, 6.0), np.array([0.2320, 0.2789, 30.0300])),
    (('5B', 6.0, 8.0), np.array([0.2088, 0.2635, 30.0300])),
    (('5B', 6.0, 10.0), np.array([0.1883, 0.2487, 30.0300])),
    (('5B', 6.0, 12.0), np.array([0.1685, 0.2339, 30.0300])),
    (('5B', 6.0, 14.0), np.array([0.1496, 0.2193, 30.0300])),
    (('5B', 6.0, 16.0), np.array([0.1310, 0.2048, 30.0300])),
    (('5B', 6.0, 18.0), np.array([0.1170, 0.1920, 30.0300])),
    (('5B', 6.0, 20.0), np.array([0.1020, 0.1810, 30.0300])),
    (('7.5B', 6.0, 2.0), np.array([0.2854, 0.3037, 30.0300])),
    (('7.5B', 6.0, 4.0), np.array([0.2602, 0.2881, 30.0300])),
    (('7.5B', 6.0, 6.0), np.array([0.2352, 0.2708, 30.0300])),
    (('7.5B', 6.0, 8.0), np.array([0.2132, 0.2537, 30.0300])),
    (('7.5B', 6.0, 10.0), np.array([0.1934, 0.2374, 30.0300])),
    (('7.5B', 6.0, 12.0), np.array([0.1734, 0.2203, 30.0300])),
    (('7.5B', 6.0, 14.0), np.array([0.1556, 0.2043, 30.0300])),
    (('7.5B', 6.0, 16.0), np.array([0.1376, 0.1879, 30.0300])),
    (('7.5B', 6.0, 18.0), np.array([0.1230, 0.1740, 30.0300])),
    (('7.5B', 6.0, 20.0), np.array([0.1110, 0.1620, 30.0300])),
    (('7.5B', 6.0, 22.0), np.array([0.1000, 0.1510, 30.0300])),
    (('10B', 6.0, 2.0), np.array([0.2871, 0.3012, 30.0300])),
    (('10B', 6.0, 4.0), np.array([0.2637, 0.2840, 30.0300])),
    (('10B', 6.0, 6.0), np.array([0.2399, 0.2650, 30.0300])),
    (('10B', 6.0, 8.0), np.array([0.2189, 0.2468, 30.0300])),
    (('10B', 6.0, 10.0), np.array([0.2000, 0.2298, 30.0300])),
    (('10B', 6.0, 12.0), np.array([0.1803, 0.2114, 30.0300])),
    (('10B', 6.0, 14.0), np.array([0.1629, 0.1947, 30.0300])),
    (('10B', 6.0, 16.0), np.array([0.1454, 0.1778, 30.0300])),
    (('10B', 6.0, 18.0), np.array([0.1310, 0.1640, 30.0300])),
    (('10B', 6.0, 20.0), np.array([0.1200, 0.1530, 30.0300])),
    (('10B', 6.0, 22.0), np.array([0.1110, 0.1440, 30.0300])),
    (('10B', 6.0, 24.0), np.array([0.0990, 0.1330, 30.0300])),
    (('2.5PB', 6.0, 2.0), np.array([0.2897, 0.2991, 30.0300])),
    (('2.5PB', 6.0, 4.0), np.array([0.2684, 0.2804, 30.0300])),
    (('2.5PB', 6.0, 6.0), np.array([0.2465, 0.2599, 30.0300])),
    (('2.5PB', 6.0, 8.0), np.array([0.2274, 0.2406, 30.0300])),
    (('2.5PB', 6.0, 10.0), np.array([0.2095, 0.2225, 30.0300])),
    (('2.5PB', 6.0, 12.0), np.array([0.1913, 0.2038, 30.0300])),
    (('2.5PB', 6.0, 14.0), np.array([0.1754, 0.1868, 30.0300])),
    (('2.5PB', 6.0, 16.0), np.array([0.1590, 0.1690, 30.0300])),
    (('2.5PB', 6.0, 18.0), np.array([0.1480, 0.1560, 30.0300])),
    (('2.5PB', 6.0, 20.0), np.array([0.1370, 0.1440, 30.0300])),
    (('2.5PB', 6.0, 22.0), np.array([0.1270, 0.1340, 30.0300])),
    (('2.5PB', 6.0, 24.0), np.array([0.1170, 0.1230, 30.0300])),
    (('5PB', 6.0, 2.0), np.array([0.2923, 0.2978, 30.0300])),
    (('5PB', 6.0, 4.0), np.array([0.2734, 0.2778, 30.0300])),
    (('5PB', 6.0, 6.0), np.array([0.2533, 0.2558, 30.0300])),
    (('5PB', 6.0, 8.0), np.array([0.2360, 0.2365, 30.0300])),
    (('5PB', 6.0, 10.0), np.array([0.2197, 0.2188, 30.0300])),
    (('5PB', 6.0, 12.0), np.array([0.2026, 0.1999, 30.0300])),
    (('5PB', 6.0, 14.0), np.array([0.1873, 0.1822, 30.0300])),
    (('5PB', 6.0, 16.0), np.array([0.1710, 0.1640, 30.0300])),
    (('5PB', 6.0, 18.0), np.array([0.1620, 0.1530, 30.0300])),
    (('5PB', 6.0, 20.0), np.array([0.1510, 0.1400, 30.0300])),
    (('5PB', 6.0, 22.0), np.array([0.1430, 0.1290, 30.0300])),
    (('5PB', 6.0, 24.0), np.array([0.1350, 0.1180, 30.0300])),
    (('5PB', 6.0, 26.0), np.array([0.1260, 0.1070, 30.0300])),
    (('7.5PB', 6.0, 2.0), np.array([0.2955, 0.2963, 30.0300])),
    (('7.5PB', 6.0, 4.0), np.array([0.2798, 0.2752, 30.0300])),
    (('7.5PB', 6.0, 6.0), np.array([0.2638, 0.2531, 30.0300])),
    (('7.5PB', 6.0, 8.0), np.array([0.2505, 0.2347, 30.0300])),
    (('7.5PB', 6.0, 10.0), np.array([0.2378, 0.2168, 30.0300])),
    (('7.5PB', 6.0, 12.0), np.array([0.2241, 0.1975, 30.0300])),
    (('7.5PB', 6.0, 14.0), np.array([0.2119, 0.1799, 30.0300])),
    (('7.5PB', 6.0, 16.0), np.array([0.1990, 0.1630, 30.0300])),
    (('7.5PB', 6.0, 18.0), np.array([0.1920, 0.1510, 30.0300])),
    (('7.5PB', 6.0, 20.0), np.array([0.1850, 0.1390, 30.0300])),
    (('7.5PB', 6.0, 22.0), np.array([0.1790, 0.1280, 30.0300])),
    (('7.5PB', 6.0, 24.0), np.array([0.1730, 0.1180, 30.0300])),
    (('7.5PB', 6.0, 26.0), np.array([0.1660, 0.1050, 30.0300])),
    (('10PB', 6.0, 2.0), np.array([0.2988, 0.2961, 30.0300])),
    (('10PB', 6.0, 4.0), np.array([0.2863, 0.2747, 30.0300])),
    (('10PB', 6.0, 6.0), np.array([0.2740, 0.2533, 30.0300])),
    (('10PB', 6.0, 8.0), np.array([0.2637, 0.2352, 30.0300])),
    (('10PB', 6.0, 10.0), np.array([0.2540, 0.2176, 30.0300])),
    (('10PB', 6.0, 12.0), np.array([0.2440, 0.1998, 30.0300])),
    (('10PB', 6.0, 14.0), np.array([0.2352, 0.1839, 30.0300])),
    (('10PB', 6.0, 16.0), np.array([0.2265, 0.1671, 30.0300])),
    (('10PB', 6.0, 18.0), np.array([0.2210, 0.1560, 30.0300])),
    (('10PB', 6.0, 20.0), np.array([0.2140, 0.1430, 30.0300])),
    (('10PB', 6.0, 22.0), np.array([0.2080, 0.1320, 30.0300])),
    (('10PB', 6.0, 24.0), np.array([0.2040, 0.1230, 30.0300])),
    (('10PB', 6.0, 26.0), np.array([0.1990, 0.1110, 30.0300])),
    (('10PB', 6.0, 28.0), np.array([0.1940, 0.1010, 30.0300])),
    (('2.5P', 6.0, 2.0), np.array([0.3016, 0.2960, 30.0300])),
    (('2.5P', 6.0, 4.0), np.array([0.2932, 0.2759, 30.0300])),
    (('2.5P', 6.0, 6.0), np.array([0.2842, 0.2550, 30.0300])),
    (('2.5P', 6.0, 8.0), np.array([0.2770, 0.2372, 30.0300])),
    (('2.5P', 6.0, 10.0), np.array([0.2703, 0.2204, 30.0300])),
    (('2.5P', 6.0, 12.0), np.array([0.2647, 0.2052, 30.0300])),
    (('2.5P', 6.0, 14.0), np.array([0.2593, 0.1909, 30.0300])),
    (('2.5P', 6.0, 16.0), np.array([0.2548, 0.1768, 30.0300])),
    (('2.5P', 6.0, 18.0), np.array([0.2504, 0.1658, 30.0300])),
    (('2.5P', 6.0, 20.0), np.array([0.2460, 0.1530, 30.0300])),
    (('2.5P', 6.0, 22.0), np.array([0.2420, 0.1420, 30.0300])),
    (('2.5P', 6.0, 24.0), np.array([0.2390, 0.1320, 30.0300])),
    (('2.5P', 6.0, 26.0), np.array([0.2350, 0.1210, 30.0300])),
    (('2.5P', 6.0, 28.0), np.array([0.2310, 0.1120, 30.0300])),
    (('2.5P', 6.0, 30.0), np.array([0.2280, 0.1020, 30.0300])),
    (('2.5P', 6.0, 32.0), np.array([0.2250, 0.0930, 30.0300])),
    (('5P', 6.0, 2.0), np.array([0.3050, 0.2967, 30.0300])),
    (('5P', 6.0, 4.0), np.array([0.3001, 0.2778, 30.0300])),
    (('5P', 6.0, 6.0), np.array([0.2950, 0.2585, 30.0300])),
    (('5P', 6.0, 8.0), np.array([0.2905, 0.2421, 30.0300])),
    (('5P', 6.0, 10.0), np.array([0.2862, 0.2260, 30.0300])),
    (('5P', 6.0, 12.0), np.array([0.2829, 0.2121, 30.0300])),
    (('5P', 6.0, 14.0), np.array([0.2794, 0.1979, 30.0300])),
    (('5P', 6.0, 16.0), np.array([0.2761, 0.1852, 30.0300])),
    (('5P', 6.0, 18.0), np.array([0.2731, 0.1738, 30.0300])),
    (('5P', 6.0, 20.0), np.array([0.2702, 0.1621, 30.0300])),
    (('5P', 6.0, 22.0), np.array([0.2670, 0.1490, 30.0300])),
    (('5P', 6.0, 24.0), np.array([0.2650, 0.1400, 30.0300])),
    (('5P', 6.0, 26.0), np.array([0.2630, 0.1300, 30.0300])),
    (('5P', 6.0, 28.0), np.array([0.2600, 0.1200, 30.0300])),
    (('5P', 6.0, 30.0), np.array([0.2580, 0.1110, 30.0300])),
    (('5P', 6.0, 32.0), np.array([0.2560, 0.1030, 30.0300])),
    (('5P', 6.0, 34.0), np.array([0.2540, 0.0950, 30.0300])),
    (('5P', 6.0, 36.0), np.array([0.2520, 0.0890, 30.0300])),
    (('7.5P', 6.0, 2.0), np.array([0.3107, 0.2993, 30.0300])),
    (('7.5P', 6.0, 4.0), np.array([0.3107, 0.2831, 30.0300])),
    (('7.5P', 6.0, 6.0), np.array([0.3101, 0.2650, 30.0300])),
    (('7.5P', 6.0, 8.0), np.array([0.3099, 0.2502, 30.0300])),
    (('7.5P', 6.0, 10.0), np.array([0.3092, 0.2350, 30.0300])),
    (('7.5P', 6.0, 12.0), np.array([0.3090, 0.2222, 30.0300])),
    (('7.5P', 6.0, 14.0), np.array([0.3084, 0.2095, 30.0300])),
    (('7.5P', 6.0, 16.0), np.array([0.3080, 0.1976, 30.0300])),
    (('7.5P', 6.0, 18.0), np.array([0.3075, 0.1870, 30.0300])),
    (('7.5P', 6.0, 20.0), np.array([0.3069, 0.1745, 30.0300])),
    (('7.5P', 6.0, 22.0), np.array([0.3062, 0.1638, 30.0300])),
    (('7.5P', 6.0, 24.0), np.array([0.3058, 0.1547, 30.0300])),
    (('7.5P', 6.0, 26.0), np.array([0.3050, 0.1440, 30.0300])),
    (('7.5P', 6.0, 28.0), np.array([0.3040, 0.1360, 30.0300])),
    (('7.5P', 6.0, 30.0), np.array([0.3030, 0.1260, 30.0300])),
    (('7.5P', 6.0, 32.0), np.array([0.3020, 0.1180, 30.0300])),
    (('7.5P', 6.0, 34.0), np.array([0.3010, 0.1110, 30.0300])),
    (('7.5P', 6.0, 36.0), np.array([0.3010, 0.1050, 30.0300])),
    (('10P', 6.0, 2.0), np.array([0.3146, 0.3018, 30.0300])),
    (('10P', 6.0, 4.0), np.array([0.3181, 0.2871, 30.0300])),
    (('10P', 6.0, 6.0), np.array([0.3226, 0.2716, 30.0300])),
    (('10P', 6.0, 8.0), np.array([0.3259, 0.2584, 30.0300])),
    (('10P', 6.0, 10.0), np.array([0.3293, 0.2450, 30.0300])),
    (('10P', 6.0, 12.0), np.array([0.3321, 0.2329, 30.0300])),
    (('10P', 6.0, 14.0), np.array([0.3349, 0.2203, 30.0300])),
    (('10P', 6.0, 16.0), np.array([0.3370, 0.2095, 30.0300])),
    (('10P', 6.0, 18.0), np.array([0.3388, 0.1995, 30.0300])),
    (('10P', 6.0, 20.0), np.array([0.3409, 0.1882, 30.0300])),
    (('10P', 6.0, 22.0), np.array([0.3426, 0.1785, 30.0300])),
    (('10P', 6.0, 24.0), np.array([0.3441, 0.1698, 30.0300])),
    (('10P', 6.0, 26.0), np.array([0.3457, 0.1604, 30.0300])),
    (('10P', 6.0, 28.0), np.array([0.3470, 0.1510, 30.0300])),
    (('10P', 6.0, 30.0), np.array([0.3490, 0.1440, 30.0300])),
    (('10P', 6.0, 32.0), np.array([0.3500, 0.1360, 30.0300])),
    (('10P', 6.0, 34.0), np.array([0.3510, 0.1290, 30.0300])),
    (('10P', 6.0, 36.0), np.array([0.3520, 0.1230, 30.0300])),
    (('2.5RP', 6.0, 2.0), np.array([0.3188, 0.3048, 30.0300])),
    (('2.5RP', 6.0, 4.0), np.array([0.3272, 0.2929, 30.0300])),
    (('2.5RP', 6.0, 6.0), np.array([0.3362, 0.2799, 30.0300])),
    (('2.5RP', 6.0, 8.0), np.array([0.3437, 0.2688, 30.0300])),
    (('2.5RP', 6.0, 10.0), np.array([0.3509, 0.2578, 30.0300])),
    (('2.5RP', 6.0, 12.0), np.array([0.3582, 0.2462, 30.0300])),
    (('2.5RP', 6.0, 14.0), np.array([0.3652, 0.2355, 30.0300])),
    (('2.5RP', 6.0, 16.0), np.array([0.3718, 0.2251, 30.0300])),
    (('2.5RP', 6.0, 18.0), np.array([0.3773, 0.2158, 30.0300])),
    (('2.5RP', 6.0, 20.0), np.array([0.3833, 0.2056, 30.0300])),
    (('2.5RP', 6.0, 22.0), np.array([0.3877, 0.1978, 30.0300])),
    (('2.5RP', 6.0, 24.0), np.array([0.3927, 0.1892, 30.0300])),
    (('2.5RP', 6.0, 26.0), np.array([0.3970, 0.1810, 30.0300])),
    (('2.5RP', 6.0, 28.0), np.array([0.4010, 0.1740, 30.0300])),
    (('2.5RP', 6.0, 30.0), np.array([0.4060, 0.1660, 30.0300])),
    (('2.5RP', 6.0, 32.0), np.array([0.4100, 0.1590, 30.0300])),
    (('2.5RP', 6.0, 34.0), np.array([0.4130, 0.1530, 30.0300])),
    (('2.5RP', 6.0, 36.0), np.array([0.4160, 0.1480, 30.0300])),
    (('5RP', 6.0, 2.0), np.array([0.3232, 0.3085, 30.0300])),
    (('5RP', 6.0, 4.0), np.array([0.3371, 0.3001, 30.0300])),
    (('5RP', 6.0, 6.0), np.array([0.3520, 0.2904, 30.0300])),
    (('5RP', 6.0, 8.0), np.array([0.3648, 0.2820, 30.0300])),
    (('5RP', 6.0, 10.0), np.array([0.3769, 0.2738, 30.0300])),
    (('5RP', 6.0, 12.0), np.array([0.3900, 0.2646, 30.0300])),
    (('5RP', 6.0, 14.0), np.array([0.4023, 0.2552, 30.0300])),
    (('5RP', 6.0, 16.0), np.array([0.4136, 0.2467, 30.0300])),
    (('5RP', 6.0, 18.0), np.array([0.4245, 0.2382, 30.0300])),
    (('5RP', 6.0, 20.0), np.array([0.4368, 0.2283, 30.0300])),
    (('5RP', 6.0, 22.0), np.array([0.4449, 0.2219, 30.0300])),
    (('5RP', 6.0, 24.0), np.array([0.4520, 0.2150, 30.0300])),
    (('5RP', 6.0, 26.0), np.array([0.4620, 0.2070, 30.0300])),
    (('5RP', 6.0, 28.0), np.array([0.4680, 0.2010, 30.0300])),
    (('5RP', 6.0, 30.0), np.array([0.4740, 0.1960, 30.0300])),
    (('5RP', 6.0, 32.0), np.array([0.4810, 0.1890, 30.0300])),
    (('7.5RP', 6.0, 2.0), np.array([0.3261, 0.3113, 30.0300])),
    (('7.5RP', 6.0, 4.0), np.array([0.3439, 0.3056, 30.0300])),
    (('7.5RP', 6.0, 6.0), np.array([0.3635, 0.2987, 30.0300])),
    (('7.5RP', 6.0, 8.0), np.array([0.3791, 0.2929, 30.0300])),
    (('7.5RP', 6.0, 10.0), np.array([0.3960, 0.2860, 30.0300])),
    (('7.5RP', 6.0, 12.0), np.array([0.4125, 0.2784, 30.0300])),
    (('7.5RP', 6.0, 14.0), np.array([0.4285, 0.2705, 30.0300])),
    (('7.5RP', 6.0, 16.0), np.array([0.4448, 0.2622, 30.0300])),
    (('7.5RP', 6.0, 18.0), np.array([0.4581, 0.2549, 30.0300])),
    (('7.5RP', 6.0, 20.0), np.array([0.4735, 0.2464, 30.0300])),
    (('7.5RP', 6.0, 22.0), np.array([0.4860, 0.2400, 30.0300])),
    (('7.5RP', 6.0, 24.0), np.array([0.4960, 0.2330, 30.0300])),
    (('7.5RP', 6.0, 26.0), np.array([0.5050, 0.2270, 30.0300])),
    (('7.5RP', 6.0, 28.0), np.array([0.5130, 0.2210, 30.0300])),
    (('7.5RP', 6.0, 30.0), np.array([0.5190, 0.2160, 30.0300])),
    (('10RP', 6.0, 2.0), np.array([0.3292, 0.3141, 30.0300])),
    (('10RP', 6.0, 4.0), np.array([0.3508, 0.3112, 30.0300])),
    (('10RP', 6.0, 6.0), np.array([0.3740, 0.3074, 30.0300])),
    (('10RP', 6.0, 8.0), np.array([0.3930, 0.3038, 30.0300])),
    (('10RP', 6.0, 10.0), np.array([0.4150, 0.2989, 30.0300])),
    (('10RP', 6.0, 12.0), np.array([0.4360, 0.2936, 30.0300])),
    (('10RP', 6.0, 14.0), np.array([0.4552, 0.2881, 30.0300])),
    (('10RP', 6.0, 16.0), np.array([0.4781, 0.2812, 30.0300])),
    (('10RP', 6.0, 18.0), np.array([0.4961, 0.2751, 30.0300])),
    (('10RP', 6.0, 20.0), np.array([0.5130, 0.2680, 30.0300])),
    (('10RP', 6.0, 22.0), np.array([0.5270, 0.2620, 30.0300])),
    (('10RP', 6.0, 24.0), np.array([0.5410, 0.2560, 30.0300])),
    (('10RP', 6.0, 26.0), np.array([0.5520, 0.2500, 30.0300])),
    (('10RP', 6.0, 28.0), np.array([0.5630, 0.2460, 30.0300])),
    (('2.5R', 6.0, 2.0), np.array([0.3318, 0.3166, 30.0300])),
    (('2.5R', 6.0, 4.0), np.array([0.3566, 0.3163, 30.0300])),
    (('2.5R', 6.0, 6.0), np.array([0.3832, 0.3158, 30.0300])),
    (('2.5R', 6.0, 8.0), np.array([0.4065, 0.3144, 30.0300])),
    (('2.5R', 6.0, 10.0), np.array([0.4320, 0.3118, 30.0300])),
    (('2.5R', 6.0, 12.0), np.array([0.4568, 0.3082, 30.0300])),
    (('2.5R', 6.0, 14.0), np.array([0.4790, 0.3041, 30.0300])),
    (('2.5R', 6.0, 16.0), np.array([0.5041, 0.2983, 30.0300])),
    (('2.5R', 6.0, 18.0), np.array([0.5262, 0.2928, 30.0300])),
    (('2.5R', 6.0, 20.0), np.array([0.5450, 0.2880, 30.0300])),
    (('2.5R', 6.0, 22.0), np.array([0.5610, 0.2820, 30.0300])),
    (('2.5R', 6.0, 24.0), np.array([0.5740, 0.2760, 30.0300])),
    (('2.5R', 6.0, 26.0), np.array([0.5880, 0.2710, 30.0300])),
    (('2.5R', 6.0, 28.0), np.array([0.5970, 0.2650, 30.0300])),
    (('5R', 6.0, 2.0), np.array([0.3343, 0.3190, 30.0300])),
    (('5R', 6.0, 4.0), np.array([0.3628, 0.3221, 30.0300])),
    (('5R', 6.0, 6.0), np.array([0.3921, 0.3244, 30.0300])),
    (('5R', 6.0, 8.0), np.array([0.4187, 0.3251, 30.0300])),
    (('5R', 6.0, 10.0), np.array([0.4480, 0.3250, 30.0300])),
    (('5R', 6.0, 12.0), np.array([0.4760, 0.3234, 30.0300])),
    (('5R', 6.0, 14.0), np.array([0.5020, 0.3212, 30.0300])),
    (('5R', 6.0, 16.0), np.array([0.5297, 0.3179, 30.0300])),
    (('5R', 6.0, 18.0), np.array([0.5552, 0.3138, 30.0300])),
    (('5R', 6.0, 20.0), np.array([0.5750, 0.3090, 30.0300])),
    (('5R', 6.0, 22.0), np.array([0.5930, 0.3040, 30.0300])),
    (('5R', 6.0, 24.0), np.array([0.6070, 0.2990, 30.0300])),
    (('5R', 6.0, 26.0), np.array([0.6180, 0.2950, 30.0300])),
    (('5R', 6.0, 28.0), np.array([0.6320, 0.2890, 30.0300])),
    (('5R', 6.0, 30.0), np.array([0.6430, 0.2830, 30.0300])),
    (('7.5R', 6.0, 2.0), np.array([0.3381, 0.3228, 30.0300])),
    (('7.5R', 6.0, 4.0), np.array([0.3692, 0.3291, 30.0300])),
    (('7.5R', 6.0, 6.0), np.array([0.4000, 0.3340, 30.0300])),
    (('7.5R', 6.0, 8.0), np.array([0.4318, 0.3383, 30.0300])),
    (('7.5R', 6.0, 10.0), np.array([0.4655, 0.3412, 30.0300])),
    (('7.5R', 6.0, 12.0), np.array([0.4961, 0.3428, 30.0300])),
    (('7.5R', 6.0, 14.0), np.array([0.5265, 0.3431, 30.0300])),
    (('7.5R', 6.0, 16.0), np.array([0.5560, 0.3420, 30.0300])),
    (('7.5R', 6.0, 18.0), np.array([0.5829, 0.3396, 30.0300])),
    (('7.5R', 6.0, 20.0), np.array([0.6030, 0.3360, 30.0300])),
    (('7.5R', 6.0, 22.0), np.array([0.6250, 0.3330, 30.0300])),
    (('7.5R', 6.0, 24.0), np.array([0.6390, 0.3300, 30.0300])),
    (('7.5R', 6.0, 26.0), np.array([0.6520, 0.3260, 30.0300])),
    (('7.5R', 6.0, 28.0), np.array([0.6670, 0.3210, 30.0300])),
    (('7.5R', 6.0, 30.0), np.array([0.6790, 0.3160, 30.0300])),
    (('10R', 6.0, 2.0), np.array([0.3417, 0.3268, 30.0300])),
    (('10R', 6.0, 4.0), np.array([0.3768, 0.3381, 30.0300])),
    (('10R', 6.0, 6.0), np.array([0.4103, 0.3473, 30.0300])),
    (('10R', 6.0, 8.0), np.array([0.4449, 0.3550, 30.0300])),
    (('10R', 6.0, 10.0), np.array([0.4812, 0.3619, 30.0300])),
    (('10R', 6.0, 12.0), np.array([0.5150, 0.3667, 30.0300])),
    (('10R', 6.0, 14.0), np.array([0.5468, 0.3697, 30.0300])),
    (('10R', 6.0, 16.0), np.array([0.5741, 0.3713, 30.0300])),
    (('10R', 6.0, 18.0), np.array([0.6009, 0.3720, 30.0300])),
    (('10R', 6.0, 20.0), np.array([0.6240, 0.3710, 30.0300])),
    (('10R', 6.0, 22.0), np.array([0.6460, 0.3700, 30.0300])),
    (('10R', 6.0, 24.0), np.array([0.6640, 0.3700, 30.0300])),
    (('10R', 6.0, 26.0), np.array([0.6790, 0.3690, 30.0300])),
    (('10R', 6.0, 28.0), np.array([0.6960, 0.3680, 30.0300])),
    (('10R', 6.0, 30.0), np.array([0.7140, 0.3650, 30.0300])),
    (('2.5YR', 6.0, 2.0), np.array([0.3453, 0.3321, 30.0300])),
    (('2.5YR', 6.0, 4.0), np.array([0.3806, 0.3467, 30.0300])),
    (('2.5YR', 6.0, 6.0), np.array([0.4180, 0.3600, 30.0300])),
    (('2.5YR', 6.0, 8.0), np.array([0.4533, 0.3708, 30.0300])),
    (('2.5YR', 6.0, 10.0), np.array([0.4891, 0.3806, 30.0300])),
    (('2.5YR', 6.0, 12.0), np.array([0.5215, 0.3887, 30.0300])),
    (('2.5YR', 6.0, 14.0), np.array([0.5488, 0.3947, 30.0300])),
    (('2.5YR', 6.0, 16.0), np.array([0.5698, 0.3990, 30.0300])),
    (('2.5YR', 6.0, 18.0), np.array([0.5879, 0.4021, 30.0300])),
    (('2.5YR', 6.0, 20.0), np.array([0.6040, 0.4030, 30.0300])),
    (('2.5YR', 6.0, 22.0), np.array([0.6140, 0.4060, 30.0300])),
    (('5YR', 6.0, 2.0), np.array([0.3474, 0.3373, 30.0300])),
    (('5YR', 6.0, 4.0), np.array([0.3840, 0.3564, 30.0300])),
    (('5YR', 6.0, 6.0), np.array([0.4229, 0.3750, 30.0300])),
    (('5YR', 6.0, 8.0), np.array([0.4592, 0.3900, 30.0300])),
    (('5YR', 6.0, 10.0), np.array([0.4921, 0.4022, 30.0300])),
    (('5YR', 6.0, 12.0), np.array([0.5199, 0.4119, 30.0300])),
    (('5YR', 6.0, 14.0), np.array([0.5423, 0.4188, 30.0300])),
    (('5YR', 6.0, 16.0), np.array([0.5597, 0.4239, 30.0300])),
    (('5YR', 6.0, 18.0), np.array([0.5715, 0.4270, 30.0300])),
    (('5YR', 6.0, 20.0), np.array([0.5850, 0.4310, 30.0300])),
    (('7.5YR', 6.0, 2.0), np.array([0.3487, 0.3421, 30.0300])),
    (('7.5YR', 6.0, 4.0), np.array([0.3860, 0.3652, 30.0300])),
    (('7.5YR', 6.0, 6.0), np.array([0.4242, 0.3876, 30.0300])),
    (('7.5YR', 6.0, 8.0), np.array([0.4596, 0.4064, 30.0300])),
    (('7.5YR', 6.0, 10.0), np.array([0.4904, 0.4220, 30.0300])),
    (('7.5YR', 6.0, 12.0), np.array([0.5145, 0.4331, 30.0300])),
    (('7.5YR', 6.0, 14.0), np.array([0.5320, 0.4412, 30.0300])),
    (('7.5YR', 6.0, 16.0), np.array([0.5468, 0.4478, 30.0300])),
    (('7.5YR', 6.0, 18.0), np.array([0.5550, 0.4520, 30.0300])),
    (('7.5YR', 6.0, 20.0), np.array([0.5640, 0.4550, 30.0300])),
    (('10YR', 6.0, 2.0), np.array([0.3491, 0.3483, 30.0300])),
    (('10YR', 6.0, 4.0), np.array([0.3861, 0.3767, 30.0300])),
    (('10YR', 6.0, 6.0), np.array([0.4240, 0.4030, 30.0300])),
    (('10YR', 6.0, 8.0), np.array([0.4570, 0.4249, 30.0300])),
    (('10YR', 6.0, 10.0), np.array([0.4843, 0.4416, 30.0300])),
    (('10YR', 6.0, 12.0), np.array([0.5050, 0.4536, 30.0300])),
    (('10YR', 6.0, 14.0), np.array([0.5200, 0.4623, 30.0300])),
    (('10YR', 6.0, 16.0), np.array([0.5310, 0.4690, 30.0300])),
    (('10YR', 6.0, 18.0), np.array([0.5390, 0.4720, 30.0300])),
    (('2.5Y', 6.0, 2.0), np.array([0.3480, 0.3540, 30.0300])),
    (('2.5Y', 6.0, 4.0), np.array([0.3840, 0.3867, 30.0300])),
    (('2.5Y', 6.0, 6.0), np.array([0.4203, 0.4176, 30.0300])),
    (('2.5Y', 6.0, 8.0), np.array([0.4517, 0.4421, 30.0300])),
    (('2.5Y', 6.0, 10.0), np.array([0.4760, 0.4607, 30.0300])),
    (('2.5Y', 6.0, 12.0), np.array([0.4928, 0.4730, 30.0300])),
    (('2.5Y', 6.0, 14.0), np.array([0.5061, 0.4829, 30.0300])),
    (('2.5Y', 6.0, 16.0), np.array([0.5160, 0.4890, 30.0300])),
    (('2.5Y', 6.0, 18.0), np.array([0.5230, 0.4940, 30.0300])),
    (('5Y', 6.0, 2.0), np.array([0.3457, 0.3580, 30.0300])),
    (('5Y', 6.0, 4.0), np.array([0.3794, 0.3955, 30.0300])),
    (('5Y', 6.0, 6.0), np.array([0.4140, 0.4305, 30.0300])),
    (('5Y', 6.0, 8.0), np.array([0.4426, 0.4588, 30.0300])),
    (('5Y', 6.0, 10.0), np.array([0.4639, 0.4790, 30.0300])),
    (('5Y', 6.0, 12.0), np.array([0.4780, 0.4920, 30.0300])),
    (('5Y', 6.0, 14.0), np.array([0.4905, 0.5038, 30.0300])),
    (('5Y', 6.0, 16.0), np.array([0.4990, 0.5100, 30.0300])),
    (('7.5Y', 6.0, 2.0), np.array([0.3431, 0.3601, 30.0300])),
    (('7.5Y', 6.0, 4.0), np.array([0.3745, 0.4004, 30.0300])),
    (('7.5Y', 6.0, 6.0), np.array([0.4060, 0.4400, 30.0300])),
    (('7.5Y', 6.0, 8.0), np.array([0.4321, 0.4719, 30.0300])),
    (('7.5Y', 6.0, 10.0), np.array([0.4512, 0.4943, 30.0300])),
    (('7.5Y', 6.0, 12.0), np.array([0.4638, 0.5087, 30.0300])),
    (('7.5Y', 6.0, 14.0), np.array([0.4754, 0.5220, 30.0300])),
    (('7.5Y', 6.0, 16.0), np.array([0.4820, 0.5280, 30.0300])),
    (('10Y', 7.0, 2.0), np.array([0.3369, 0.3569, 43.0600])),
    (('10Y', 7.0, 4.0), np.array([0.3624, 0.3951, 43.0600])),
    (('10Y', 7.0, 6.0), np.array([0.3864, 0.4305, 43.0600])),
    (('10Y', 7.0, 8.0), np.array([0.4090, 0.4641, 43.0600])),
    (('10Y', 7.0, 10.0), np.array([0.4289, 0.4937, 43.0600])),
    (('10Y', 7.0, 12.0), np.array([0.4420, 0.5131, 43.0600])),
    (('10Y', 7.0, 14.0), np.array([0.4516, 0.5277, 43.0600])),
    (('10Y', 7.0, 16.0), np.array([0.4582, 0.5375, 43.0600])),
    (('10Y', 7.0, 18.0), np.array([0.4620, 0.5430, 43.0600])),
    (('2.5GY', 7.0, 2.0), np.array([0.3328, 0.3569, 43.0600])),
    (('2.5GY', 7.0, 4.0), np.array([0.3534, 0.3953, 43.0600])),
    (('2.5GY', 7.0, 6.0), np.array([0.3728, 0.4316, 43.0600])),
    (('2.5GY', 7.0, 8.0), np.array([0.3919, 0.4684, 43.0600])),
    (('2.5GY', 7.0, 10.0), np.array([0.4091, 0.5030, 43.0600])),
    (('2.5GY', 7.0, 12.0), np.array([0.4213, 0.5270, 43.0600])),
    (('2.5GY', 7.0, 14.0), np.array([0.4309, 0.5459, 43.0600])),
    (('2.5GY', 7.0, 16.0), np.array([0.4366, 0.5578, 43.0600])),
    (('2.5GY', 7.0, 18.0), np.array([0.4390, 0.5640, 43.0600])),
    (('5GY', 7.0, 2.0), np.array([0.3284, 0.3559, 43.0600])),
    (('5GY', 7.0, 4.0), np.array([0.3437, 0.3929, 43.0600])),
    (('5GY', 7.0, 6.0), np.array([0.3581, 0.4291, 43.0600])),
    (('5GY', 7.0, 8.0), np.array([0.3722, 0.4669, 43.0600])),
    (('5GY', 7.0, 10.0), np.array([0.3852, 0.5051, 43.0600])),
    (('5GY', 7.0, 12.0), np.array([0.3949, 0.5367, 43.0600])),
    (('5GY', 7.0, 14.0), np.array([0.4027, 0.5615, 43.0600])),
    (('5GY', 7.0, 16.0), np.array([0.4076, 0.5783, 43.0600])),
    (('5GY', 7.0, 18.0), np.array([0.4100, 0.5890, 43.0600])),
    (('5GY', 7.0, 20.0), np.array([0.4130, 0.6010, 43.0600])),
    (('7.5GY', 7.0, 2.0), np.array([0.3190, 0.3516, 43.0600])),
    (('7.5GY', 7.0, 4.0), np.array([0.3267, 0.3848, 43.0600])),
    (('7.5GY', 7.0, 6.0), np.array([0.3341, 0.4191, 43.0600])),
    (('7.5GY', 7.0, 8.0), np.array([0.3406, 0.4558, 43.0600])),
    (('7.5GY', 7.0, 10.0), np.array([0.3461, 0.4950, 43.0600])),
    (('7.5GY', 7.0, 12.0), np.array([0.3502, 0.5328, 43.0600])),
    (('7.5GY', 7.0, 14.0), np.array([0.3532, 0.5700, 43.0600])),
    (('7.5GY', 7.0, 16.0), np.array([0.3549, 0.6000, 43.0600])),
    (('7.5GY', 7.0, 18.0), np.array([0.3555, 0.6242, 43.0600])),
    (('7.5GY', 7.0, 20.0), np.array([0.3560, 0.6420, 43.0600])),
    (('7.5GY', 7.0, 22.0), np.array([0.3560, 0.6610, 43.0600])),
    (('7.5GY', 7.0, 24.0), np.array([0.3560, 0.6760, 43.0600])),
    (('10GY', 7.0, 2.0), np.array([0.3117, 0.3469, 43.0600])),
    (('10GY', 7.0, 4.0), np.array([0.3133, 0.3764, 43.0600])),
    (('10GY', 7.0, 6.0), np.array([0.3142, 0.4058, 43.0600])),
    (('10GY', 7.0, 8.0), np.array([0.3140, 0.4387, 43.0600])),
    (('10GY', 7.0, 10.0), np.array([0.3123, 0.4732, 43.0600])),
    (('10GY', 7.0, 12.0), np.array([0.3092, 0.5095, 43.0600])),
    (('10GY', 7.0, 14.0), np.array([0.3047, 0.5458, 43.0600])),
    (('10GY', 7.0, 16.0), np.array([0.2981, 0.5835, 43.0600])),
    (('10GY', 7.0, 18.0), np.array([0.2905, 0.6186, 43.0600])),
    (('10GY', 7.0, 20.0), np.array([0.2816, 0.6563, 43.0600])),
    (('10GY', 7.0, 22.0), np.array([0.2728, 0.6893, 43.0600])),
    (('10GY', 7.0, 24.0), np.array([0.2620, 0.7270, 43.0600])),
    (('10GY', 7.0, 26.0), np.array([0.2520, 0.7580, 43.0600])),
    (('10GY', 7.0, 28.0), np.array([0.2420, 0.7900, 43.0600])),
    (('10GY', 7.0, 30.0), np.array([0.2330, 0.8190, 43.0600])),
    (('10GY', 7.0, 32.0), np.array([0.2240, 0.8510, 43.0600])),
    (('2.5G', 7.0, 2.0), np.array([0.3047, 0.3413, 43.0600])),
    (('2.5G', 7.0, 4.0), np.array([0.2992, 0.3644, 43.0600])),
    (('2.5G', 7.0, 6.0), np.array([0.2933, 0.3873, 43.0600])),
    (('2.5G', 7.0, 8.0), np.array([0.2861, 0.4129, 43.0600])),
    (('2.5G', 7.0, 10.0), np.array([0.2775, 0.4395, 43.0600])),
    (('2.5G', 7.0, 12.0), np.array([0.2672, 0.4667, 43.0600])),
    (('2.5G', 7.0, 14.0), np.array([0.2568, 0.4931, 43.0600])),
    (('2.5G', 7.0, 16.0), np.array([0.2448, 0.5203, 43.0600])),
    (('2.5G', 7.0, 18.0), np.array([0.2328, 0.5467, 43.0600])),
    (('2.5G', 7.0, 20.0), np.array([0.2181, 0.5744, 43.0600])),
    (('2.5G', 7.0, 22.0), np.array([0.2029, 0.6017, 43.0600])),
    (('2.5G', 7.0, 24.0), np.array([0.1875, 0.6265, 43.0600])),
    (('2.5G', 7.0, 26.0), np.array([0.1689, 0.6549, 43.0600])),
    (('2.5G', 7.0, 28.0), np.array([0.1490, 0.6810, 43.0600])),
    (('2.5G', 7.0, 30.0), np.array([0.1260, 0.7090, 43.0600])),
    (('2.5G', 7.0, 32.0), np.array([0.1060, 0.7330, 43.0600])),
    (('2.5G', 7.0, 34.0), np.array([0.0810, 0.7590, 43.0600])),
    (('5G', 7.0, 2.0), np.array([0.3001, 0.3366, 43.0600])),
    (('5G', 7.0, 4.0), np.array([0.2902, 0.3548, 43.0600])),
    (('5G', 7.0, 6.0), np.array([0.2801, 0.3721, 43.0600])),
    (('5G', 7.0, 8.0), np.array([0.2687, 0.3901, 43.0600])),
    (('5G', 7.0, 10.0), np.array([0.2554, 0.4087, 43.0600])),
    (('5G', 7.0, 12.0), np.array([0.2416, 0.4267, 43.0600])),
    (('5G', 7.0, 14.0), np.array([0.2262, 0.4450, 43.0600])),
    (('5G', 7.0, 16.0), np.array([0.2111, 0.4616, 43.0600])),
    (('5G', 7.0, 18.0), np.array([0.1967, 0.4771, 43.0600])),
    (('5G', 7.0, 20.0), np.array([0.1805, 0.4933, 43.0600])),
    (('5G', 7.0, 22.0), np.array([0.1659, 0.5074, 43.0600])),
    (('5G', 7.0, 24.0), np.array([0.1521, 0.5200, 43.0600])),
    (('5G', 7.0, 26.0), np.array([0.1397, 0.5312, 43.0600])),
    (('5G', 7.0, 28.0), np.array([0.1230, 0.5460, 43.0600])),
    (('5G', 7.0, 30.0), np.array([0.1050, 0.5600, 43.0600])),
    (('5G', 7.0, 32.0), np.array([0.0880, 0.5730, 43.0600])),
    (('5G', 7.0, 34.0), np.array([0.0690, 0.5870, 43.0600])),
    (('7.5G', 7.0, 2.0), np.array([0.2972, 0.3333, 43.0600])),
    (('7.5G', 7.0, 4.0), np.array([0.2850, 0.3482, 43.0600])),
    (('7.5G', 7.0, 6.0), np.array([0.2728, 0.3622, 43.0600])),
    (('7.5G', 7.0, 8.0), np.array([0.2595, 0.3764, 43.0600])),
    (('7.5G', 7.0, 10.0), np.array([0.2445, 0.3914, 43.0600])),
    (('7.5G', 7.0, 12.0), np.array([0.2295, 0.4058, 43.0600])),
    (('7.5G', 7.0, 14.0), np.array([0.2139, 0.4199, 43.0600])),
    (('7.5G', 7.0, 16.0), np.array([0.1982, 0.4330, 43.0600])),
    (('7.5G', 7.0, 18.0), np.array([0.1841, 0.4448, 43.0600])),
    (('7.5G', 7.0, 20.0), np.array([0.1688, 0.4570, 43.0600])),
    (('7.5G', 7.0, 22.0), np.array([0.1539, 0.4683, 43.0600])),
    (('7.5G', 7.0, 24.0), np.array([0.1415, 0.4778, 43.0600])),
    (('7.5G', 7.0, 26.0), np.array([0.1303, 0.4858, 43.0600])),
    (('7.5G', 7.0, 28.0), np.array([0.1150, 0.4970, 43.0600])),
    (('7.5G', 7.0, 30.0), np.array([0.0990, 0.5070, 43.0600])),
    (('7.5G', 7.0, 32.0), np.array([0.0840, 0.5170, 43.0600])),
    (('7.5G', 7.0, 34.0), np.array([0.0690, 0.5270, 43.0600])),
    (('10G', 7.0, 2.0), np.array([0.2945, 0.3297, 43.0600])),
    (('10G', 7.0, 4.0), np.array([0.2803, 0.3415, 43.0600])),
    (('10G', 7.0, 6.0), np.array([0.2662, 0.3526, 43.0600])),
    (('10G', 7.0, 8.0), np.array([0.2513, 0.3635, 43.0600])),
    (('10G', 7.0, 10.0), np.array([0.2352, 0.3748, 43.0600])),
    (('10G', 7.0, 12.0), np.array([0.2195, 0.3854, 43.0600])),
    (('10G', 7.0, 14.0), np.array([0.2033, 0.3956, 43.0600])),
    (('10G', 7.0, 16.0), np.array([0.1881, 0.4049, 43.0600])),
    (('10G', 7.0, 18.0), np.array([0.1734, 0.4135, 43.0600])),
    (('10G', 7.0, 20.0), np.array([0.1589, 0.4220, 43.0600])),
    (('10G', 7.0, 22.0), np.array([0.1434, 0.4306, 43.0600])),
    (('10G', 7.0, 24.0), np.array([0.1310, 0.4377, 43.0600])),
    (('10G', 7.0, 26.0), np.array([0.1210, 0.4430, 43.0600])),
    (('10G', 7.0, 28.0), np.array([0.1080, 0.4490, 43.0600])),
    (('10G', 7.0, 30.0), np.array([0.0940, 0.4570, 43.0600])),
    (('10G', 7.0, 32.0), np.array([0.0830, 0.4630, 43.0600])),
    (('2.5BG', 7.0, 2.0), np.array([0.2927, 0.3269, 43.0600])),
    (('2.5BG', 7.0, 4.0), np.array([0.2764, 0.3354, 43.0600])),
    (('2.5BG', 7.0, 6.0), np.array([0.2608, 0.3430, 43.0600])),
    (('2.5BG', 7.0, 8.0), np.array([0.2439, 0.3508, 43.0600])),
    (('2.5BG', 7.0, 10.0), np.array([0.2264, 0.3576, 43.0600])),
    (('2.5BG', 7.0, 12.0), np.array([0.2102, 0.3636, 43.0600])),
    (('2.5BG', 7.0, 14.0), np.array([0.1932, 0.3694, 43.0600])),
    (('2.5BG', 7.0, 16.0), np.array([0.1788, 0.3739, 43.0600])),
    (('2.5BG', 7.0, 18.0), np.array([0.1626, 0.3788, 43.0600])),
    (('2.5BG', 7.0, 20.0), np.array([0.1490, 0.3827, 43.0600])),
    (('2.5BG', 7.0, 22.0), np.array([0.1334, 0.3870, 43.0600])),
    (('2.5BG', 7.0, 24.0), np.array([0.1220, 0.3900, 43.0600])),
    (('2.5BG', 7.0, 26.0), np.array([0.1110, 0.3920, 43.0600])),
    (('2.5BG', 7.0, 28.0), np.array([0.0990, 0.3940, 43.0600])),
    (('2.5BG', 7.0, 30.0), np.array([0.0880, 0.3970, 43.0600])),
    (('2.5BG', 7.0, 32.0), np.array([0.0790, 0.3980, 43.0600])),
    (('5BG', 7.0, 2.0), np.array([0.2898, 0.3225, 43.0600])),
    (('5BG', 7.0, 4.0), np.array([0.2712, 0.3269, 43.0600])),
    (('5BG', 7.0, 6.0), np.array([0.2543, 0.3302, 43.0600])),
    (('5BG', 7.0, 8.0), np.array([0.2354, 0.3335, 43.0600])),
    (('5BG', 7.0, 10.0), np.array([0.2163, 0.3361, 43.0600])),
    (('5BG', 7.0, 12.0), np.array([0.1997, 0.3379, 43.0600])),
    (('5BG', 7.0, 14.0), np.array([0.1838, 0.3390, 43.0600])),
    (('5BG', 7.0, 16.0), np.array([0.1675, 0.3401, 43.0600])),
    (('5BG', 7.0, 18.0), np.array([0.1515, 0.3410, 43.0600])),
    (('5BG', 7.0, 20.0), np.array([0.1380, 0.3412, 43.0600])),
    (('5BG', 7.0, 22.0), np.array([0.1220, 0.3400, 43.0600])),
    (('5BG', 7.0, 24.0), np.array([0.1100, 0.3400, 43.0600])),
    (('5BG', 7.0, 26.0), np.array([0.1010, 0.3400, 43.0600])),
    (('5BG', 7.0, 28.0), np.array([0.0920, 0.3400, 43.0600])),
    (('5BG', 7.0, 30.0), np.array([0.0830, 0.3390, 43.0600])),
    (('7.5BG', 7.0, 2.0), np.array([0.2878, 0.3182, 43.0600])),
    (('7.5BG', 7.0, 4.0), np.array([0.2671, 0.3189, 43.0600])),
    (('7.5BG', 7.0, 6.0), np.array([0.2490, 0.3186, 43.0600])),
    (('7.5BG', 7.0, 8.0), np.array([0.2292, 0.3178, 43.0600])),
    (('7.5BG', 7.0, 10.0), np.array([0.2094, 0.3165, 43.0600])),
    (('7.5BG', 7.0, 12.0), np.array([0.1914, 0.3148, 43.0600])),
    (('7.5BG', 7.0, 14.0), np.array([0.1751, 0.3129, 43.0600])),
    (('7.5BG', 7.0, 16.0), np.array([0.1584, 0.3101, 43.0600])),
    (('7.5BG', 7.0, 18.0), np.array([0.1427, 0.3076, 43.0600])),
    (('7.5BG', 7.0, 20.0), np.array([0.1300, 0.3050, 43.0600])),
    (('7.5BG', 7.0, 22.0), np.array([0.1140, 0.3020, 43.0600])),
    (('7.5BG', 7.0, 24.0), np.array([0.1020, 0.3000, 43.0600])),
    (('7.5BG', 7.0, 26.0), np.array([0.0940, 0.2980, 43.0600])),
    (('10BG', 7.0, 2.0), np.array([0.2869, 0.3143, 43.0600])),
    (('10BG', 7.0, 4.0), np.array([0.2642, 0.3109, 43.0600])),
    (('10BG', 7.0, 6.0), np.array([0.2448, 0.3069, 43.0600])),
    (('10BG', 7.0, 8.0), np.array([0.2235, 0.3014, 43.0600])),
    (('10BG', 7.0, 10.0), np.array([0.2035, 0.2956, 43.0600])),
    (('10BG', 7.0, 12.0), np.array([0.1841, 0.2892, 43.0600])),
    (('10BG', 7.0, 14.0), np.array([0.1671, 0.2832, 43.0600])),
    (('10BG', 7.0, 16.0), np.array([0.1489, 0.2768, 43.0600])),
    (('10BG', 7.0, 18.0), np.array([0.1340, 0.2710, 43.0600])),
    (('10BG', 7.0, 20.0), np.array([0.1220, 0.2660, 43.0600])),
    (('10BG', 7.0, 22.0), np.array([0.1080, 0.2600, 43.0600])),
    (('10BG', 7.0, 24.0), np.array([0.0940, 0.2530, 43.0600])),
    (('2.5B', 7.0, 2.0), np.array([0.2867, 0.3110, 43.0600])),
    (('2.5B', 7.0, 4.0), np.array([0.2629, 0.3038, 43.0600])),
    (('2.5B', 7.0, 6.0), np.array([0.2418, 0.2960, 43.0600])),
    (('2.5B', 7.0, 8.0), np.array([0.2208, 0.2871, 43.0600])),
    (('2.5B', 7.0, 10.0), np.array([0.1994, 0.2775, 43.0600])),
    (('2.5B', 7.0, 12.0), np.array([0.1797, 0.2672, 43.0600])),
    (('2.5B', 7.0, 14.0), np.array([0.1624, 0.2581, 43.0600])),
    (('2.5B', 7.0, 16.0), np.array([0.1435, 0.2472, 43.0600])),
    (('2.5B', 7.0, 18.0), np.array([0.1300, 0.2390, 43.0600])),
    (('2.5B', 7.0, 20.0), np.array([0.1170, 0.2310, 43.0600])),
    (('2.5B', 7.0, 22.0), np.array([0.1040, 0.2220, 43.0600])),
    (('5B', 7.0, 2.0), np.array([0.2875, 0.3078, 43.0600])),
    (('5B', 7.0, 4.0), np.array([0.2633, 0.2972, 43.0600])),
    (('5B', 7.0, 6.0), np.array([0.2410, 0.2854, 43.0600])),
    (('5B', 7.0, 8.0), np.array([0.2204, 0.2729, 43.0600])),
    (('5B', 7.0, 10.0), np.array([0.1986, 0.2579, 43.0600])),
    (('5B', 7.0, 12.0), np.array([0.1778, 0.2430, 43.0600])),
    (('5B', 7.0, 14.0), np.array([0.1615, 0.2307, 43.0600])),
    (('5B', 7.0, 16.0), np.array([0.1450, 0.2190, 43.0600])),
    (('5B', 7.0, 18.0), np.array([0.1280, 0.2060, 43.0600])),
    (('5B', 7.0, 20.0), np.array([0.1140, 0.1950, 43.0600])),
    (('7.5B', 7.0, 2.0), np.array([0.2888, 0.3058, 43.0600])),
    (('7.5B', 7.0, 4.0), np.array([0.2651, 0.2927, 43.0600])),
    (('7.5B', 7.0, 6.0), np.array([0.2436, 0.2787, 43.0600])),
    (('7.5B', 7.0, 8.0), np.array([0.2225, 0.2631, 43.0600])),
    (('7.5B', 7.0, 10.0), np.array([0.2016, 0.2466, 43.0600])),
    (('7.5B', 7.0, 12.0), np.array([0.1818, 0.2303, 43.0600])),
    (('7.5B', 7.0, 14.0), np.array([0.1650, 0.2160, 43.0600])),
    (('7.5B', 7.0, 16.0), np.array([0.1490, 0.2020, 43.0600])),
    (('7.5B', 7.0, 18.0), np.array([0.1320, 0.1870, 43.0600])),
    (('7.5B', 7.0, 20.0), np.array([0.1210, 0.1750, 43.0600])),
    (('10B', 7.0, 2.0), np.array([0.2908, 0.3039, 43.0600])),
    (('10B', 7.0, 4.0), np.array([0.2685, 0.2886, 43.0600])),
    (('10B', 7.0, 6.0), np.array([0.2478, 0.2728, 43.0600])),
    (('10B', 7.0, 8.0), np.array([0.2277, 0.2559, 43.0600])),
    (('10B', 7.0, 10.0), np.array([0.2078, 0.2382, 43.0600])),
    (('10B', 7.0, 12.0), np.array([0.1883, 0.2203, 43.0600])),
    (('10B', 7.0, 14.0), np.array([0.1720, 0.2060, 43.0600])),
    (('10B', 7.0, 16.0), np.array([0.1560, 0.1900, 43.0600])),
    (('10B', 7.0, 18.0), np.array([0.1410, 0.1760, 43.0600])),
    (('10B', 7.0, 20.0), np.array([0.1280, 0.1620, 43.0600])),
    (('2.5PB', 7.0, 2.0), np.array([0.2932, 0.3025, 43.0600])),
    (('2.5PB', 7.0, 4.0), np.array([0.2729, 0.2848, 43.0600])),
    (('2.5PB', 7.0, 6.0), np.array([0.2538, 0.2677, 43.0600])),
    (('2.5PB', 7.0, 8.0), np.array([0.2352, 0.2498, 43.0600])),
    (('2.5PB', 7.0, 10.0), np.array([0.2162, 0.2309, 43.0600])),
    (('2.5PB', 7.0, 12.0), np.array([0.1990, 0.2130, 43.0600])),
    (('2.5PB', 7.0, 14.0), np.array([0.1830, 0.1950, 43.0600])),
    (('2.5PB', 7.0, 16.0), np.array([0.1680, 0.1790, 43.0600])),
    (('2.5PB', 7.0, 18.0), np.array([0.1560, 0.1650, 43.0600])),
    (('5PB', 7.0, 2.0), np.array([0.2952, 0.3011, 43.0600])),
    (('5PB', 7.0, 4.0), np.array([0.2773, 0.2828, 43.0600])),
    (('5PB', 7.0, 6.0), np.array([0.2596, 0.2643, 43.0600])),
    (('5PB', 7.0, 8.0), np.array([0.2427, 0.2458, 43.0600])),
    (('5PB', 7.0, 10.0), np.array([0.2254, 0.2267, 43.0600])),
    (('5PB', 7.0, 12.0), np.array([0.2080, 0.2080, 43.0600])),
    (('5PB', 7.0, 14.0), np.array([0.1940, 0.1910, 43.0600])),
    (('5PB', 7.0, 16.0), np.array([0.1810, 0.1750, 43.0600])),
    (('5PB', 7.0, 18.0), np.array([0.1700, 0.1610, 43.0600])),
    (('7.5PB', 7.0, 2.0), np.array([0.2982, 0.3003, 43.0600])),
    (('7.5PB', 7.0, 4.0), np.array([0.2833, 0.2809, 43.0600])),
    (('7.5PB', 7.0, 6.0), np.array([0.2687, 0.2612, 43.0600])),
    (('7.5PB', 7.0, 8.0), np.array([0.2546, 0.2418, 43.0600])),
    (('7.5PB', 7.0, 10.0), np.array([0.2410, 0.2224, 43.0600])),
    (('7.5PB', 7.0, 12.0), np.array([0.2280, 0.2040, 43.0600])),
    (('7.5PB', 7.0, 14.0), np.array([0.2170, 0.1890, 43.0600])),
    (('7.5PB', 7.0, 16.0), np.array([0.2070, 0.1730, 43.0600])),
    (('7.5PB', 7.0, 18.0), np.array([0.1980, 0.1600, 43.0600])),
    (('7.5PB', 7.0, 20.0), np.array([0.1880, 0.1460, 43.0600])),
    (('10PB', 7.0, 2.0), np.array([0.3005, 0.3000, 43.0600])),
    (('10PB', 7.0, 4.0), np.array([0.2886, 0.2801, 43.0600])),
    (('10PB', 7.0, 6.0), np.array([0.2776, 0.2612, 43.0600])),
    (('10PB', 7.0, 8.0), np.array([0.2670, 0.2425, 43.0600])),
    (('10PB', 7.0, 10.0), np.array([0.2563, 0.2240, 43.0600])),
    (('10PB', 7.0, 12.0), np.array([0.2465, 0.2058, 43.0600])),
    (('10PB', 7.0, 14.0), np.array([0.2390, 0.1910, 43.0600])),
    (('10PB', 7.0, 16.0), np.array([0.2320, 0.1780, 43.0600])),
    (('10PB', 7.0, 18.0), np.array([0.2250, 0.1660, 43.0600])),
    (('10PB', 7.0, 20.0), np.array([0.2190, 0.1520, 43.0600])),
    (('10PB', 7.0, 22.0), np.array([0.2120, 0.1400, 43.0600])),
    (('2.5P', 7.0, 2.0), np.array([0.3031, 0.3000, 43.0600])),
    (('2.5P', 7.0, 4.0), np.array([0.2950, 0.2810, 43.0600])),
    (('2.5P', 7.0, 6.0), np.array([0.2873, 0.2633, 43.0600])),
    (('2.5P', 7.0, 8.0), np.array([0.2799, 0.2459, 43.0600])),
    (('2.5P', 7.0, 10.0), np.array([0.2729, 0.2289, 43.0600])),
    (('2.5P', 7.0, 12.0), np.array([0.2664, 0.2127, 43.0600])),
    (('2.5P', 7.0, 14.0), np.array([0.2610, 0.1980, 43.0600])),
    (('2.5P', 7.0, 16.0), np.array([0.2560, 0.1850, 43.0600])),
    (('2.5P', 7.0, 18.0), np.array([0.2520, 0.1760, 43.0600])),
    (('2.5P', 7.0, 20.0), np.array([0.2480, 0.1620, 43.0600])),
    (('2.5P', 7.0, 22.0), np.array([0.2440, 0.1500, 43.0600])),
    (('2.5P', 7.0, 24.0), np.array([0.2400, 0.1400, 43.0600])),
    (('5P', 7.0, 2.0), np.array([0.3059, 0.3010, 43.0600])),
    (('5P', 7.0, 4.0), np.array([0.3009, 0.2831, 43.0600])),
    (('5P', 7.0, 6.0), np.array([0.2961, 0.2663, 43.0600])),
    (('5P', 7.0, 8.0), np.array([0.2918, 0.2504, 43.0600])),
    (('5P', 7.0, 10.0), np.array([0.2872, 0.2343, 43.0600])),
    (('5P', 7.0, 12.0), np.array([0.2833, 0.2197, 43.0600])),
    (('5P', 7.0, 14.0), np.array([0.2801, 0.2068, 43.0600])),
    (('5P', 7.0, 16.0), np.array([0.2770, 0.1940, 43.0600])),
    (('5P', 7.0, 18.0), np.array([0.2740, 0.1820, 43.0600])),
    (('5P', 7.0, 20.0), np.array([0.2710, 0.1700, 43.0600])),
    (('5P', 7.0, 22.0), np.array([0.2680, 0.1580, 43.0600])),
    (('5P', 7.0, 24.0), np.array([0.2650, 0.1480, 43.0600])),
    (('5P', 7.0, 26.0), np.array([0.2620, 0.1370, 43.0600])),
    (('5P', 7.0, 28.0), np.array([0.2600, 0.1280, 43.0600])),
    (('7.5P', 7.0, 2.0), np.array([0.3109, 0.3037, 43.0600])),
    (('7.5P', 7.0, 4.0), np.array([0.3111, 0.2880, 43.0600])),
    (('7.5P', 7.0, 6.0), np.array([0.3111, 0.2730, 43.0600])),
    (('7.5P', 7.0, 8.0), np.array([0.3109, 0.2584, 43.0600])),
    (('7.5P', 7.0, 10.0), np.array([0.3108, 0.2442, 43.0600])),
    (('7.5P', 7.0, 12.0), np.array([0.3104, 0.2320, 43.0600])),
    (('7.5P', 7.0, 14.0), np.array([0.3101, 0.2192, 43.0600])),
    (('7.5P', 7.0, 16.0), np.array([0.3099, 0.2074, 43.0600])),
    (('7.5P', 7.0, 18.0), np.array([0.3093, 0.1962, 43.0600])),
    (('7.5P', 7.0, 20.0), np.array([0.3080, 0.1850, 43.0600])),
    (('7.5P', 7.0, 22.0), np.array([0.3080, 0.1740, 43.0600])),
    (('7.5P', 7.0, 24.0), np.array([0.3070, 0.1640, 43.0600])),
    (('7.5P', 7.0, 26.0), np.array([0.3060, 0.1540, 43.0600])),
    (('7.5P', 7.0, 28.0), np.array([0.3050, 0.1450, 43.0600])),
    (('7.5P', 7.0, 30.0), np.array([0.3040, 0.1360, 43.0600])),
    (('10P', 7.0, 2.0), np.array([0.3138, 0.3054, 43.0600])),
    (('10P', 7.0, 4.0), np.array([0.3181, 0.2920, 43.0600])),
    (('10P', 7.0, 6.0), np.array([0.3221, 0.2786, 43.0600])),
    (('10P', 7.0, 8.0), np.array([0.3256, 0.2654, 43.0600])),
    (('10P', 7.0, 10.0), np.array([0.3288, 0.2531, 43.0600])),
    (('10P', 7.0, 12.0), np.array([0.3314, 0.2423, 43.0600])),
    (('10P', 7.0, 14.0), np.array([0.3341, 0.2308, 43.0600])),
    (('10P', 7.0, 16.0), np.array([0.3368, 0.2192, 43.0600])),
    (('10P', 7.0, 18.0), np.array([0.3391, 0.2088, 43.0600])),
    (('10P', 7.0, 20.0), np.array([0.3410, 0.1988, 43.0600])),
    (('10P', 7.0, 22.0), np.array([0.3430, 0.1883, 43.0600])),
    (('10P', 7.0, 24.0), np.array([0.3440, 0.1790, 43.0600])),
    (('10P', 7.0, 26.0), np.array([0.3440, 0.1700, 43.0600])),
    (('10P', 7.0, 28.0), np.array([0.3450, 0.1620, 43.0600])),
    (('10P', 7.0, 30.0), np.array([0.3460, 0.1540, 43.0600])),
    (('2.5RP', 7.0, 2.0), np.array([0.3170, 0.3076, 43.0600])),
    (('2.5RP', 7.0, 4.0), np.array([0.3254, 0.2971, 43.0600])),
    (('2.5RP', 7.0, 6.0), np.array([0.3338, 0.2854, 43.0600])),
    (('2.5RP', 7.0, 8.0), np.array([0.3417, 0.2745, 43.0600])),
    (('2.5RP', 7.0, 10.0), np.array([0.3487, 0.2648, 43.0600])),
    (('2.5RP', 7.0, 12.0), np.array([0.3555, 0.2545, 43.0600])),
    (('2.5RP', 7.0, 14.0), np.array([0.3620, 0.2448, 43.0600])),
    (('2.5RP', 7.0, 16.0), np.array([0.3688, 0.2342, 43.0600])),
    (('2.5RP', 7.0, 18.0), np.array([0.3751, 0.2241, 43.0600])),
    (('2.5RP', 7.0, 20.0), np.array([0.3811, 0.2143, 43.0600])),
    (('2.5RP', 7.0, 22.0), np.array([0.3850, 0.2060, 43.0600])),
    (('2.5RP', 7.0, 24.0), np.array([0.3890, 0.2000, 43.0600])),
    (('2.5RP', 7.0, 26.0), np.array([0.3940, 0.1920, 43.0600])),
    (('2.5RP', 7.0, 28.0), np.array([0.3980, 0.1840, 43.0600])),
    (('2.5RP', 7.0, 30.0), np.array([0.4020, 0.1780, 43.0600])),
    (('5RP', 7.0, 2.0), np.array([0.3206, 0.3104, 43.0600])),
    (('5RP', 7.0, 4.0), np.array([0.3332, 0.3032, 43.0600])),
    (('5RP', 7.0, 6.0), np.array([0.3470, 0.2949, 43.0600])),
    (('5RP', 7.0, 8.0), np.array([0.3603, 0.2869, 43.0600])),
    (('5RP', 7.0, 10.0), np.array([0.3713, 0.2798, 43.0600])),
    (('5RP', 7.0, 12.0), np.array([0.3841, 0.2710, 43.0600])),
    (('5RP', 7.0, 14.0), np.array([0.3958, 0.2628, 43.0600])),
    (('5RP', 7.0, 16.0), np.array([0.4076, 0.2540, 43.0600])),
    (('5RP', 7.0, 18.0), np.array([0.4186, 0.2459, 43.0600])),
    (('5RP', 7.0, 20.0), np.array([0.4260, 0.2390, 43.0600])),
    (('5RP', 7.0, 22.0), np.array([0.4360, 0.2320, 43.0600])),
    (('5RP', 7.0, 24.0), np.array([0.4430, 0.2260, 43.0600])),
    (('5RP', 7.0, 26.0), np.array([0.4520, 0.2170, 43.0600])),
    (('5RP', 7.0, 28.0), np.array([0.4600, 0.2110, 43.0600])),
    (('7.5RP', 7.0, 2.0), np.array([0.3232, 0.3125, 43.0600])),
    (('7.5RP', 7.0, 4.0), np.array([0.3389, 0.3079, 43.0600])),
    (('7.5RP', 7.0, 6.0), np.array([0.3562, 0.3022, 43.0600])),
    (('7.5RP', 7.0, 8.0), np.array([0.3722, 0.2963, 43.0600])),
    (('7.5RP', 7.0, 10.0), np.array([0.3871, 0.2906, 43.0600])),
    (('7.5RP', 7.0, 12.0), np.array([0.4040, 0.2834, 43.0600])),
    (('7.5RP', 7.0, 14.0), np.array([0.4195, 0.2762, 43.0600])),
    (('7.5RP', 7.0, 16.0), np.array([0.4346, 0.2689, 43.0600])),
    (('7.5RP', 7.0, 18.0), np.array([0.4480, 0.2610, 43.0600])),
    (('7.5RP', 7.0, 20.0), np.array([0.4600, 0.2550, 43.0600])),
    (('7.5RP', 7.0, 22.0), np.array([0.4710, 0.2490, 43.0600])),
    (('7.5RP', 7.0, 24.0), np.array([0.4810, 0.2430, 43.0600])),
    (('7.5RP', 7.0, 26.0), np.array([0.4910, 0.2360, 43.0600])),
    (('10RP', 7.0, 2.0), np.array([0.3258, 0.3148, 43.0600])),
    (('10RP', 7.0, 4.0), np.array([0.3446, 0.3125, 43.0600])),
    (('10RP', 7.0, 6.0), np.array([0.3648, 0.3098, 43.0600])),
    (('10RP', 7.0, 8.0), np.array([0.3851, 0.3067, 43.0600])),
    (('10RP', 7.0, 10.0), np.array([0.4040, 0.3030, 43.0600])),
    (('10RP', 7.0, 12.0), np.array([0.4260, 0.2980, 43.0600])),
    (('10RP', 7.0, 14.0), np.array([0.4456, 0.2931, 43.0600])),
    (('10RP', 7.0, 16.0), np.array([0.4648, 0.2878, 43.0600])),
    (('10RP', 7.0, 18.0), np.array([0.4830, 0.2830, 43.0600])),
    (('10RP', 7.0, 20.0), np.array([0.4980, 0.2790, 43.0600])),
    (('10RP', 7.0, 22.0), np.array([0.5110, 0.2740, 43.0600])),
    (('10RP', 7.0, 24.0), np.array([0.5260, 0.2670, 43.0600])),
    (('2.5R', 7.0, 2.0), np.array([0.3284, 0.3170, 43.0600])),
    (('2.5R', 7.0, 4.0), np.array([0.3499, 0.3171, 43.0600])),
    (('2.5R', 7.0, 6.0), np.array([0.3728, 0.3170, 43.0600])),
    (('2.5R', 7.0, 8.0), np.array([0.3961, 0.3160, 43.0600])),
    (('2.5R', 7.0, 10.0), np.array([0.4183, 0.3144, 43.0600])),
    (('2.5R', 7.0, 12.0), np.array([0.4435, 0.3119, 43.0600])),
    (('2.5R', 7.0, 14.0), np.array([0.4660, 0.3082, 43.0600])),
    (('2.5R', 7.0, 16.0), np.array([0.4885, 0.3039, 43.0600])),
    (('2.5R', 7.0, 18.0), np.array([0.5070, 0.3000, 43.0600])),
    (('2.5R', 7.0, 20.0), np.array([0.5230, 0.2960, 43.0600])),
    (('2.5R', 7.0, 22.0), np.array([0.5360, 0.2910, 43.0600])),
    (('2.5R', 7.0, 24.0), np.array([0.5530, 0.2870, 43.0600])),
    (('5R', 7.0, 2.0), np.array([0.3306, 0.3190, 43.0600])),
    (('5R', 7.0, 4.0), np.array([0.3552, 0.3222, 43.0600])),
    (('5R', 7.0, 6.0), np.array([0.3805, 0.3244, 43.0600])),
    (('5R', 7.0, 8.0), np.array([0.4067, 0.3256, 43.0600])),
    (('5R', 7.0, 10.0), np.array([0.4320, 0.3260, 43.0600])),
    (('5R', 7.0, 12.0), np.array([0.4595, 0.3252, 43.0600])),
    (('5R', 7.0, 14.0), np.array([0.4848, 0.3238, 43.0600])),
    (('5R', 7.0, 16.0), np.array([0.5100, 0.3210, 43.0600])),
    (('5R', 7.0, 18.0), np.array([0.5300, 0.3180, 43.0600])),
    (('5R', 7.0, 20.0), np.array([0.5470, 0.3150, 43.0600])),
    (('5R', 7.0, 22.0), np.array([0.5630, 0.3110, 43.0600])),
    (('5R', 7.0, 24.0), np.array([0.5800, 0.3060, 43.0600])),
    (('7.5R', 7.0, 2.0), np.array([0.3335, 0.3220, 43.0600])),
    (('7.5R', 7.0, 4.0), np.array([0.3611, 0.3282, 43.0600])),
    (('7.5R', 7.0, 6.0), np.array([0.3888, 0.3336, 43.0600])),
    (('7.5R', 7.0, 8.0), np.array([0.4196, 0.3382, 43.0600])),
    (('7.5R', 7.0, 10.0), np.array([0.4470, 0.3413, 43.0600])),
    (('7.5R', 7.0, 12.0), np.array([0.4777, 0.3435, 43.0600])),
    (('7.5R', 7.0, 14.0), np.array([0.5059, 0.3450, 43.0600])),
    (('7.5R', 7.0, 16.0), np.array([0.5341, 0.3452, 43.0600])),
    (('7.5R', 7.0, 18.0), np.array([0.5540, 0.3430, 43.0600])),
    (('7.5R', 7.0, 20.0), np.array([0.5730, 0.3410, 43.0600])),
    (('7.5R', 7.0, 22.0), np.array([0.5920, 0.3380, 43.0600])),
    (('7.5R', 7.0, 24.0), np.array([0.6070, 0.3350, 43.0600])),
    (('7.5R', 7.0, 26.0), np.array([0.6220, 0.3320, 43.0600])),
    (('10R', 7.0, 2.0), np.array([0.3360, 0.3253, 43.0600])),
    (('10R', 7.0, 4.0), np.array([0.3671, 0.3360, 43.0600])),
    (('10R', 7.0, 6.0), np.array([0.3984, 0.3452, 43.0600])),
    (('10R', 7.0, 8.0), np.array([0.4308, 0.3533, 43.0600])),
    (('10R', 7.0, 10.0), np.array([0.4600, 0.3596, 43.0600])),
    (('10R', 7.0, 12.0), np.array([0.4930, 0.3659, 43.0600])),
    (('10R', 7.0, 14.0), np.array([0.5234, 0.3700, 43.0600])),
    (('10R', 7.0, 16.0), np.array([0.5519, 0.3729, 43.0600])),
    (('10R', 7.0, 18.0), np.array([0.5710, 0.3730, 43.0600])),
    (('10R', 7.0, 20.0), np.array([0.5920, 0.3740, 43.0600])),
    (('10R', 7.0, 22.0), np.array([0.6140, 0.3740, 43.0600])),
    (('10R', 7.0, 24.0), np.array([0.6300, 0.3720, 43.0600])),
    (('10R', 7.0, 26.0), np.array([0.6470, 0.3690, 43.0600])),
    (('2.5YR', 7.0, 2.0), np.array([0.3392, 0.3298, 43.0600])),
    (('2.5YR', 7.0, 4.0), np.array([0.3715, 0.3439, 43.0600])),
    (('2.5YR', 7.0, 6.0), np.array([0.4053, 0.3570, 43.0600])),
    (('2.5YR', 7.0, 8.0), np.array([0.4371, 0.3679, 43.0600])),
    (('2.5YR', 7.0, 10.0), np.array([0.4671, 0.3768, 43.0600])),
    (('2.5YR', 7.0, 12.0), np.array([0.5001, 0.3861, 43.0600])),
    (('2.5YR', 7.0, 14.0), np.array([0.5297, 0.3938, 43.0600])),
    (('2.5YR', 7.0, 16.0), np.array([0.5522, 0.3989, 43.0600])),
    (('2.5YR', 7.0, 18.0), np.array([0.5695, 0.4024, 43.0600])),
    (('2.5YR', 7.0, 20.0), np.array([0.5824, 0.4046, 43.0600])),
    (('2.5YR', 7.0, 22.0), np.array([0.5940, 0.4070, 43.0600])),
    (('2.5YR', 7.0, 24.0), np.array([0.6020, 0.4090, 43.0600])),
    (('5YR', 7.0, 2.0), np.array([0.3421, 0.3349, 43.0600])),
    (('5YR', 7.0, 4.0), np.array([0.3750, 0.3530, 43.0600])),
    (('5YR', 7.0, 6.0), np.array([0.4091, 0.3701, 43.0600])),
    (('5YR', 7.0, 8.0), np.array([0.4402, 0.3842, 43.0600])),
    (('5YR', 7.0, 10.0), np.array([0.4711, 0.3972, 43.0600])),
    (('5YR', 7.0, 12.0), np.array([0.5007, 0.4081, 43.0600])),
    (('5YR', 7.0, 14.0), np.array([0.5252, 0.4168, 43.0600])),
    (('5YR', 7.0, 16.0), np.array([0.5437, 0.4228, 43.0600])),
    (('5YR', 7.0, 18.0), np.array([0.5564, 0.4267, 43.0600])),
    (('5YR', 7.0, 20.0), np.array([0.5657, 0.4298, 43.0600])),
    (('5YR', 7.0, 22.0), np.array([0.5750, 0.4340, 43.0600])),
    (('7.5YR', 7.0, 2.0), np.array([0.3437, 0.3397, 43.0600])),
    (('7.5YR', 7.0, 4.0), np.array([0.3772, 0.3613, 43.0600])),
    (('7.5YR', 7.0, 6.0), np.array([0.4107, 0.3820, 43.0600])),
    (('7.5YR', 7.0, 8.0), np.array([0.4415, 0.3996, 43.0600])),
    (('7.5YR', 7.0, 10.0), np.array([0.4704, 0.4151, 43.0600])),
    (('7.5YR', 7.0, 12.0), np.array([0.4970, 0.4282, 43.0600])),
    (('7.5YR', 7.0, 14.0), np.array([0.5174, 0.4381, 43.0600])),
    (('7.5YR', 7.0, 16.0), np.array([0.5319, 0.4449, 43.0600])),
    (('7.5YR', 7.0, 18.0), np.array([0.5417, 0.4492, 43.0600])),
    (('7.5YR', 7.0, 20.0), np.array([0.5480, 0.4530, 43.0600])),
    (('7.5YR', 7.0, 22.0), np.array([0.5560, 0.4560, 43.0600])),
    (('10YR', 7.0, 2.0), np.array([0.3443, 0.3454, 43.0600])),
    (('10YR', 7.0, 4.0), np.array([0.3778, 0.3719, 43.0600])),
    (('10YR', 7.0, 6.0), np.array([0.4102, 0.3960, 43.0600])),
    (('10YR', 7.0, 8.0), np.array([0.4399, 0.4164, 43.0600])),
    (('10YR', 7.0, 10.0), np.array([0.4667, 0.4335, 43.0600])),
    (('10YR', 7.0, 12.0), np.array([0.4900, 0.4480, 43.0600])),
    (('10YR', 7.0, 14.0), np.array([0.5074, 0.4581, 43.0600])),
    (('10YR', 7.0, 16.0), np.array([0.5188, 0.4650, 43.0600])),
    (('10YR', 7.0, 18.0), np.array([0.5276, 0.4700, 43.0600])),
    (('10YR', 7.0, 20.0), np.array([0.5320, 0.4740, 43.0600])),
    (('2.5Y', 7.0, 2.0), np.array([0.3436, 0.3507, 43.0600])),
    (('2.5Y', 7.0, 4.0), np.array([0.3761, 0.3800, 43.0600])),
    (('2.5Y', 7.0, 6.0), np.array([0.4073, 0.4073, 43.0600])),
    (('2.5Y', 7.0, 8.0), np.array([0.4353, 0.4312, 43.0600])),
    (('2.5Y', 7.0, 10.0), np.array([0.4606, 0.4516, 43.0600])),
    (('2.5Y', 7.0, 12.0), np.array([0.4806, 0.4666, 43.0600])),
    (('2.5Y', 7.0, 14.0), np.array([0.4950, 0.4773, 43.0600])),
    (('2.5Y', 7.0, 16.0), np.array([0.5049, 0.4843, 43.0600])),
    (('2.5Y', 7.0, 18.0), np.array([0.5110, 0.4880, 43.0600])),
    (('2.5Y', 7.0, 20.0), np.array([0.5160, 0.4920, 43.0600])),
    (('5Y', 7.0, 2.0), np.array([0.3419, 0.3540, 43.0600])),
    (('5Y', 7.0, 4.0), np.array([0.3718, 0.3885, 43.0600])),
    (('5Y', 7.0, 6.0), np.array([0.4009, 0.4198, 43.0600])),
    (('5Y', 7.0, 8.0), np.array([0.4271, 0.4462, 43.0600])),
    (('5Y', 7.0, 10.0), np.array([0.4509, 0.4696, 43.0600])),
    (('5Y', 7.0, 12.0), np.array([0.4677, 0.4857, 43.0600])),
    (('5Y', 7.0, 14.0), np.array([0.4791, 0.4965, 43.0600])),
    (('5Y', 7.0, 16.0), np.array([0.4875, 0.5047, 43.0600])),
    (('5Y', 7.0, 18.0), np.array([0.4930, 0.5090, 43.0600])),
    (('5Y', 7.0, 20.0), np.array([0.4980, 0.5130, 43.0600])),
    (('7.5Y', 7.0, 2.0), np.array([0.3396, 0.3558, 43.0600])),
    (('7.5Y', 7.0, 4.0), np.array([0.3677, 0.3925, 43.0600])),
    (('7.5Y', 7.0, 6.0), np.array([0.3943, 0.4264, 43.0600])),
    (('7.5Y', 7.0, 8.0), np.array([0.4184, 0.4568, 43.0600])),
    (('7.5Y', 7.0, 10.0), np.array([0.4400, 0.4830, 43.0600])),
    (('7.5Y', 7.0, 12.0), np.array([0.4547, 0.5005, 43.0600])),
    (('7.5Y', 7.0, 14.0), np.array([0.4652, 0.5128, 43.0600])),
    (('7.5Y', 7.0, 16.0), np.array([0.4728, 0.5215, 43.0600])),
    (('7.5Y', 7.0, 18.0), np.array([0.4770, 0.5270, 43.0600])),
    (('10Y', 8.0, 2.0), np.array([0.3359, 0.3552, 59.1000])),
    (('10Y', 8.0, 4.0), np.array([0.3581, 0.3883, 59.1000])),
    (('10Y', 8.0, 6.0), np.array([0.3803, 0.4216, 59.1000])),
    (('10Y', 8.0, 8.0), np.array([0.4008, 0.4520, 59.1000])),
    (('10Y', 8.0, 10.0), np.array([0.4190, 0.4791, 59.1000])),
    (('10Y', 8.0, 12.0), np.array([0.4341, 0.5020, 59.1000])),
    (('10Y', 8.0, 14.0), np.array([0.4450, 0.5181, 59.1000])),
    (('10Y', 8.0, 16.0), np.array([0.4525, 0.5295, 59.1000])),
    (('10Y', 8.0, 18.0), np.array([0.4570, 0.5366, 59.1000])),
    (('10Y', 8.0, 20.0), np.array([0.4610, 0.5420, 59.1000])),
    (('2.5GY', 8.0, 2.0), np.array([0.3327, 0.3555, 59.1000])),
    (('2.5GY', 8.0, 4.0), np.array([0.3504, 0.3887, 59.1000])),
    (('2.5GY', 8.0, 6.0), np.array([0.3690, 0.4230, 59.1000])),
    (('2.5GY', 8.0, 8.0), np.array([0.3858, 0.4550, 59.1000])),
    (('2.5GY', 8.0, 10.0), np.array([0.4021, 0.4869, 59.1000])),
    (('2.5GY', 8.0, 12.0), np.array([0.4154, 0.5133, 59.1000])),
    (('2.5GY', 8.0, 14.0), np.array([0.4261, 0.5344, 59.1000])),
    (('2.5GY', 8.0, 16.0), np.array([0.4327, 0.5475, 59.1000])),
    (('2.5GY', 8.0, 18.0), np.array([0.4371, 0.5557, 59.1000])),
    (('2.5GY', 8.0, 20.0), np.array([0.4400, 0.5620, 59.1000])),
    (('2.5GY', 8.0, 22.0), np.array([0.4430, 0.5680, 59.1000])),
    (('5GY', 8.0, 2.0), np.array([0.3284, 0.3542, 59.1000])),
    (('5GY', 8.0, 4.0), np.array([0.3433, 0.3872, 59.1000])),
    (('5GY', 8.0, 6.0), np.array([0.3573, 0.4214, 59.1000])),
    (('5GY', 8.0, 8.0), np.array([0.3696, 0.4542, 59.1000])),
    (('5GY', 8.0, 10.0), np.array([0.3816, 0.4879, 59.1000])),
    (('5GY', 8.0, 12.0), np.array([0.3924, 0.5199, 59.1000])),
    (('5GY', 8.0, 14.0), np.array([0.4011, 0.5468, 59.1000])),
    (('5GY', 8.0, 16.0), np.array([0.4061, 0.5641, 59.1000])),
    (('5GY', 8.0, 18.0), np.array([0.4104, 0.5785, 59.1000])),
    (('5GY', 8.0, 20.0), np.array([0.4127, 0.5855, 59.1000])),
    (('5GY', 8.0, 22.0), np.array([0.4150, 0.5950, 59.1000])),
    (('7.5GY', 8.0, 2.0), np.array([0.3194, 0.3502, 59.1000])),
    (('7.5GY', 8.0, 4.0), np.array([0.3266, 0.3809, 59.1000])),
    (('7.5GY', 8.0, 6.0), np.array([0.3339, 0.4129, 59.1000])),
    (('7.5GY', 8.0, 8.0), np.array([0.3408, 0.4452, 59.1000])),
    (('7.5GY', 8.0, 10.0), np.array([0.3463, 0.4791, 59.1000])),
    (('7.5GY', 8.0, 12.0), np.array([0.3511, 0.5144, 59.1000])),
    (('7.5GY', 8.0, 14.0), np.array([0.3546, 0.5490, 59.1000])),
    (('7.5GY', 8.0, 16.0), np.array([0.3569, 0.5798, 59.1000])),
    (('7.5GY', 8.0, 18.0), np.array([0.3585, 0.6063, 59.1000])),
    (('7.5GY', 8.0, 20.0), np.array([0.3592, 0.6235, 59.1000])),
    (('7.5GY', 8.0, 22.0), np.array([0.3600, 0.6450, 59.1000])),
    (('7.5GY', 8.0, 24.0), np.array([0.3600, 0.6600, 59.1000])),
    (('7.5GY', 8.0, 26.0), np.array([0.3610, 0.6790, 59.1000])),
    (('7.5GY', 8.0, 28.0), np.array([0.3610, 0.6960, 59.1000])),
    (('10GY', 8.0, 2.0), np.array([0.3121, 0.3459, 59.1000])),
    (('10GY', 8.0, 4.0), np.array([0.3140, 0.3727, 59.1000])),
    (('10GY', 8.0, 6.0), np.array([0.3150, 0.4014, 59.1000])),
    (('10GY', 8.0, 8.0), np.array([0.3149, 0.4284, 59.1000])),
    (('10GY', 8.0, 10.0), np.array([0.3140, 0.4601, 59.1000])),
    (('10GY', 8.0, 12.0), np.array([0.3124, 0.4926, 59.1000])),
    (('10GY', 8.0, 14.0), np.array([0.3091, 0.5247, 59.1000])),
    (('10GY', 8.0, 16.0), np.array([0.3043, 0.5578, 59.1000])),
    (('10GY', 8.0, 18.0), np.array([0.2987, 0.5919, 59.1000])),
    (('10GY', 8.0, 20.0), np.array([0.2918, 0.6255, 59.1000])),
    (('10GY', 8.0, 22.0), np.array([0.2846, 0.6564, 59.1000])),
    (('10GY', 8.0, 24.0), np.array([0.2781, 0.6840, 59.1000])),
    (('10GY', 8.0, 26.0), np.array([0.2710, 0.7020, 59.1000])),
    (('10GY', 8.0, 28.0), np.array([0.2620, 0.7260, 59.1000])),
    (('10GY', 8.0, 30.0), np.array([0.2520, 0.7520, 59.1000])),
    (('10GY', 8.0, 32.0), np.array([0.2400, 0.7840, 59.1000])),
    (('2.5G', 8.0, 2.0), np.array([0.3053, 0.3404, 59.1000])),
    (('2.5G', 8.0, 4.0), np.array([0.3009, 0.3614, 59.1000])),
    (('2.5G', 8.0, 6.0), np.array([0.2952, 0.3851, 59.1000])),
    (('2.5G', 8.0, 8.0), np.array([0.2896, 0.4065, 59.1000])),
    (('2.5G', 8.0, 10.0), np.array([0.2829, 0.4301, 59.1000])),
    (('2.5G', 8.0, 12.0), np.array([0.2743, 0.4554, 59.1000])),
    (('2.5G', 8.0, 14.0), np.array([0.2661, 0.4780, 59.1000])),
    (('2.5G', 8.0, 16.0), np.array([0.2563, 0.5045, 59.1000])),
    (('2.5G', 8.0, 18.0), np.array([0.2451, 0.5309, 59.1000])),
    (('2.5G', 8.0, 20.0), np.array([0.2339, 0.5561, 59.1000])),
    (('2.5G', 8.0, 22.0), np.array([0.2221, 0.5799, 59.1000])),
    (('2.5G', 8.0, 24.0), np.array([0.2091, 0.6033, 59.1000])),
    (('2.5G', 8.0, 26.0), np.array([0.1960, 0.6220, 59.1000])),
    (('2.5G', 8.0, 28.0), np.array([0.1800, 0.6470, 59.1000])),
    (('2.5G', 8.0, 30.0), np.array([0.1630, 0.6700, 59.1000])),
    (('2.5G', 8.0, 32.0), np.array([0.1440, 0.6920, 59.1000])),
    (('5G', 8.0, 2.0), np.array([0.3009, 0.3359, 59.1000])),
    (('5G', 8.0, 4.0), np.array([0.2924, 0.3523, 59.1000])),
    (('5G', 8.0, 6.0), np.array([0.2822, 0.3702, 59.1000])),
    (('5G', 8.0, 8.0), np.array([0.2723, 0.3865, 59.1000])),
    (('5G', 8.0, 10.0), np.array([0.2613, 0.4026, 59.1000])),
    (('5G', 8.0, 12.0), np.array([0.2489, 0.4191, 59.1000])),
    (('5G', 8.0, 14.0), np.array([0.2368, 0.4348, 59.1000])),
    (('5G', 8.0, 16.0), np.array([0.2240, 0.4500, 59.1000])),
    (('5G', 8.0, 18.0), np.array([0.2103, 0.4652, 59.1000])),
    (('5G', 8.0, 20.0), np.array([0.1956, 0.4806, 59.1000])),
    (('5G', 8.0, 22.0), np.array([0.1821, 0.4940, 59.1000])),
    (('5G', 8.0, 24.0), np.array([0.1680, 0.5070, 59.1000])),
    (('5G', 8.0, 26.0), np.array([0.1510, 0.5210, 59.1000])),
    (('5G', 8.0, 28.0), np.array([0.1340, 0.5370, 59.1000])),
    (('5G', 8.0, 30.0), np.array([0.1150, 0.5530, 59.1000])),
    (('5G', 8.0, 32.0), np.array([0.0960, 0.5640, 59.1000])),
    (('7.5G', 8.0, 2.0), np.array([0.2981, 0.3326, 59.1000])),
    (('7.5G', 8.0, 4.0), np.array([0.2874, 0.3464, 59.1000])),
    (('7.5G', 8.0, 6.0), np.array([0.2754, 0.3608, 59.1000])),
    (('7.5G', 8.0, 8.0), np.array([0.2639, 0.3733, 59.1000])),
    (('7.5G', 8.0, 10.0), np.array([0.2515, 0.3867, 59.1000])),
    (('7.5G', 8.0, 12.0), np.array([0.2380, 0.4002, 59.1000])),
    (('7.5G', 8.0, 14.0), np.array([0.2254, 0.4125, 59.1000])),
    (('7.5G', 8.0, 16.0), np.array([0.2120, 0.4252, 59.1000])),
    (('7.5G', 8.0, 18.0), np.array([0.1980, 0.4372, 59.1000])),
    (('7.5G', 8.0, 20.0), np.array([0.1845, 0.4492, 59.1000])),
    (('7.5G', 8.0, 22.0), np.array([0.1700, 0.4590, 59.1000])),
    (('7.5G', 8.0, 24.0), np.array([0.1550, 0.4710, 59.1000])),
    (('7.5G', 8.0, 26.0), np.array([0.1390, 0.4830, 59.1000])),
    (('7.5G', 8.0, 28.0), np.array([0.1230, 0.4940, 59.1000])),
    (('7.5G', 8.0, 30.0), np.array([0.1050, 0.5070, 59.1000])),
    (('10G', 8.0, 2.0), np.array([0.2957, 0.3293, 59.1000])),
    (('10G', 8.0, 4.0), np.array([0.2828, 0.3403, 59.1000])),
    (('10G', 8.0, 6.0), np.array([0.2693, 0.3512, 59.1000])),
    (('10G', 8.0, 8.0), np.array([0.2564, 0.3611, 59.1000])),
    (('10G', 8.0, 10.0), np.array([0.2430, 0.3710, 59.1000])),
    (('10G', 8.0, 12.0), np.array([0.2282, 0.3811, 59.1000])),
    (('10G', 8.0, 14.0), np.array([0.2148, 0.3903, 59.1000])),
    (('10G', 8.0, 16.0), np.array([0.2012, 0.3992, 59.1000])),
    (('10G', 8.0, 18.0), np.array([0.1866, 0.4086, 59.1000])),
    (('10G', 8.0, 20.0), np.array([0.1734, 0.4164, 59.1000])),
    (('10G', 8.0, 22.0), np.array([0.1590, 0.4240, 59.1000])),
    (('10G', 8.0, 24.0), np.array([0.1420, 0.4330, 59.1000])),
    (('10G', 8.0, 26.0), np.array([0.1270, 0.4410, 59.1000])),
    (('10G', 8.0, 28.0), np.array([0.1120, 0.4480, 59.1000])),
    (('2.5BG', 8.0, 2.0), np.array([0.2940, 0.3268, 59.1000])),
    (('2.5BG', 8.0, 4.0), np.array([0.2791, 0.3351, 59.1000])),
    (('2.5BG', 8.0, 6.0), np.array([0.2647, 0.3429, 59.1000])),
    (('2.5BG', 8.0, 8.0), np.array([0.2500, 0.3500, 59.1000])),
    (('2.5BG', 8.0, 10.0), np.array([0.2352, 0.3566, 59.1000])),
    (('2.5BG', 8.0, 12.0), np.array([0.2196, 0.3630, 59.1000])),
    (('2.5BG', 8.0, 14.0), np.array([0.2057, 0.3681, 59.1000])),
    (('2.5BG', 8.0, 16.0), np.array([0.1915, 0.3732, 59.1000])),
    (('2.5BG', 8.0, 18.0), np.array([0.1759, 0.3782, 59.1000])),
    (('2.5BG', 8.0, 20.0), np.array([0.1620, 0.3830, 59.1000])),
    (('2.5BG', 8.0, 22.0), np.array([0.1480, 0.3870, 59.1000])),
    (('2.5BG', 8.0, 24.0), np.array([0.1320, 0.3920, 59.1000])),
    (('2.5BG', 8.0, 26.0), np.array([0.1160, 0.3960, 59.1000])),
    (('5BG', 8.0, 2.0), np.array([0.2919, 0.3228, 59.1000])),
    (('5BG', 8.0, 4.0), np.array([0.2752, 0.3278, 59.1000])),
    (('5BG', 8.0, 6.0), np.array([0.2588, 0.3318, 59.1000])),
    (('5BG', 8.0, 8.0), np.array([0.2419, 0.3352, 59.1000])),
    (('5BG', 8.0, 10.0), np.array([0.2264, 0.3383, 59.1000])),
    (('5BG', 8.0, 12.0), np.array([0.2101, 0.3412, 59.1000])),
    (('5BG', 8.0, 14.0), np.array([0.1958, 0.3432, 59.1000])),
    (('5BG', 8.0, 16.0), np.array([0.1814, 0.3450, 59.1000])),
    (('5BG', 8.0, 18.0), np.array([0.1650, 0.3460, 59.1000])),
    (('5BG', 8.0, 20.0), np.array([0.1510, 0.3480, 59.1000])),
    (('5BG', 8.0, 22.0), np.array([0.1380, 0.3490, 59.1000])),
    (('5BG', 8.0, 24.0), np.array([0.1220, 0.3510, 59.1000])),
    (('5BG', 8.0, 26.0), np.array([0.1050, 0.3520, 59.1000])),
    (('7.5BG', 8.0, 2.0), np.array([0.2900, 0.3183, 59.1000])),
    (('7.5BG', 8.0, 4.0), np.array([0.2718, 0.3200, 59.1000])),
    (('7.5BG', 8.0, 6.0), np.array([0.2525, 0.3198, 59.1000])),
    (('7.5BG', 8.0, 8.0), np.array([0.2352, 0.3198, 59.1000])),
    (('7.5BG', 8.0, 10.0), np.array([0.2184, 0.3196, 59.1000])),
    (('7.5BG', 8.0, 12.0), np.array([0.2010, 0.3188, 59.1000])),
    (('7.5BG', 8.0, 14.0), np.array([0.1868, 0.3179, 59.1000])),
    (('7.5BG', 8.0, 16.0), np.array([0.1721, 0.3168, 59.1000])),
    (('7.5BG', 8.0, 18.0), np.array([0.1550, 0.3140, 59.1000])),
    (('7.5BG', 8.0, 20.0), np.array([0.1420, 0.3110, 59.1000])),
    (('7.5BG', 8.0, 22.0), np.array([0.1270, 0.3090, 59.1000])),
    (('7.5BG', 8.0, 24.0), np.array([0.1110, 0.3070, 59.1000])),
    (('10BG', 8.0, 2.0), np.array([0.2894, 0.3152, 59.1000])),
    (('10BG', 8.0, 4.0), np.array([0.2686, 0.3130, 59.1000])),
    (('10BG', 8.0, 6.0), np.array([0.2489, 0.3099, 59.1000])),
    (('10BG', 8.0, 8.0), np.array([0.2302, 0.3063, 59.1000])),
    (('10BG', 8.0, 10.0), np.array([0.2120, 0.3025, 59.1000])),
    (('10BG', 8.0, 12.0), np.array([0.1937, 0.2978, 59.1000])),
    (('10BG', 8.0, 14.0), np.array([0.1788, 0.2936, 59.1000])),
    (('10BG', 8.0, 16.0), np.array([0.1610, 0.2880, 59.1000])),
    (('10BG', 8.0, 18.0), np.array([0.1450, 0.2820, 59.1000])),
    (('10BG', 8.0, 20.0), np.array([0.1330, 0.2780, 59.1000])),
    (('10BG', 8.0, 22.0), np.array([0.1180, 0.2730, 59.1000])),
    (('2.5B', 8.0, 2.0), np.array([0.2897, 0.3124, 59.1000])),
    (('2.5B', 8.0, 4.0), np.array([0.2668, 0.3067, 59.1000])),
    (('2.5B', 8.0, 6.0), np.array([0.2462, 0.3000, 59.1000])),
    (('2.5B', 8.0, 8.0), np.array([0.2264, 0.2923, 59.1000])),
    (('2.5B', 8.0, 10.0), np.array([0.2066, 0.2839, 59.1000])),
    (('2.5B', 8.0, 12.0), np.array([0.1877, 0.2752, 59.1000])),
    (('2.5B', 8.0, 14.0), np.array([0.1720, 0.2660, 59.1000])),
    (('2.5B', 8.0, 16.0), np.array([0.1520, 0.2560, 59.1000])),
    (('2.5B', 8.0, 18.0), np.array([0.1370, 0.2450, 59.1000])),
    (('2.5B', 8.0, 20.0), np.array([0.1230, 0.2350, 59.1000])),
    (('5B', 8.0, 2.0), np.array([0.2908, 0.3096, 59.1000])),
    (('5B', 8.0, 4.0), np.array([0.2671, 0.2998, 59.1000])),
    (('5B', 8.0, 6.0), np.array([0.2457, 0.2888, 59.1000])),
    (('5B', 8.0, 8.0), np.array([0.2237, 0.2761, 59.1000])),
    (('5B', 8.0, 10.0), np.array([0.2040, 0.2630, 59.1000])),
    (('5B', 8.0, 12.0), np.array([0.1820, 0.2470, 59.1000])),
    (('5B', 8.0, 14.0), np.array([0.1660, 0.2360, 59.1000])),
    (('5B', 8.0, 16.0), np.array([0.1480, 0.2210, 59.1000])),
    (('5B', 8.0, 18.0), np.array([0.1320, 0.2070, 59.1000])),
    (('7.5B', 8.0, 2.0), np.array([0.2922, 0.3077, 59.1000])),
    (('7.5B', 8.0, 4.0), np.array([0.2688, 0.2956, 59.1000])),
    (('7.5B', 8.0, 6.0), np.array([0.2472, 0.2821, 59.1000])),
    (('7.5B', 8.0, 8.0), np.array([0.2252, 0.2668, 59.1000])),
    (('7.5B', 8.0, 10.0), np.array([0.2050, 0.2500, 59.1000])),
    (('7.5B', 8.0, 12.0), np.array([0.1840, 0.2330, 59.1000])),
    (('7.5B', 8.0, 14.0), np.array([0.1680, 0.2190, 59.1000])),
    (('7.5B', 8.0, 16.0), np.array([0.1510, 0.2040, 59.1000])),
    (('10B', 8.0, 2.0), np.array([0.2935, 0.3062, 59.1000])),
    (('10B', 8.0, 4.0), np.array([0.2718, 0.2911, 59.1000])),
    (('10B', 8.0, 6.0), np.array([0.2512, 0.2760, 59.1000])),
    (('10B', 8.0, 8.0), np.array([0.2294, 0.2587, 59.1000])),
    (('10B', 8.0, 10.0), np.array([0.2100, 0.2420, 59.1000])),
    (('10B', 8.0, 12.0), np.array([0.1900, 0.2240, 59.1000])),
    (('10B', 8.0, 14.0), np.array([0.1740, 0.2060, 59.1000])),
    (('2.5PB', 8.0, 2.0), np.array([0.2957, 0.3047, 59.1000])),
    (('2.5PB', 8.0, 4.0), np.array([0.2758, 0.2879, 59.1000])),
    (('2.5PB', 8.0, 6.0), np.array([0.2562, 0.2709, 59.1000])),
    (('2.5PB', 8.0, 8.0), np.array([0.2370, 0.2530, 59.1000])),
    (('2.5PB', 8.0, 10.0), np.array([0.2180, 0.2350, 59.1000])),
    (('2.5PB', 8.0, 12.0), np.array([0.2020, 0.2170, 59.1000])),
    (('2.5PB', 8.0, 14.0), np.array([0.1850, 0.1990, 59.1000])),
    (('5PB', 8.0, 2.0), np.array([0.2974, 0.3039, 59.1000])),
    (('5PB', 8.0, 4.0), np.array([0.2798, 0.2861, 59.1000])),
    (('5PB', 8.0, 6.0), np.array([0.2614, 0.2670, 59.1000])),
    (('5PB', 8.0, 8.0), np.array([0.2440, 0.2490, 59.1000])),
    (('5PB', 8.0, 10.0), np.array([0.2270, 0.2290, 59.1000])),
    (('5PB', 8.0, 12.0), np.array([0.2140, 0.2140, 59.1000])),
    (('5PB', 8.0, 14.0), np.array([0.2000, 0.1940, 59.1000])),
    (('7.5PB', 8.0, 2.0), np.array([0.3003, 0.3034, 59.1000])),
    (('7.5PB', 8.0, 4.0), np.array([0.2856, 0.2846, 59.1000])),
    (('7.5PB', 8.0, 6.0), np.array([0.2702, 0.2648, 59.1000])),
    (('7.5PB', 8.0, 8.0), np.array([0.2550, 0.2440, 59.1000])),
    (('7.5PB', 8.0, 10.0), np.array([0.2410, 0.2240, 59.1000])),
    (('7.5PB', 8.0, 12.0), np.array([0.2320, 0.2100, 59.1000])),
    (('7.5PB', 8.0, 14.0), np.array([0.2210, 0.1930, 59.1000])),
    (('10PB', 8.0, 2.0), np.array([0.3027, 0.3035, 59.1000])),
    (('10PB', 8.0, 4.0), np.array([0.2911, 0.2848, 59.1000])),
    (('10PB', 8.0, 6.0), np.array([0.2792, 0.2649, 59.1000])),
    (('10PB', 8.0, 8.0), np.array([0.2677, 0.2443, 59.1000])),
    (('10PB', 8.0, 10.0), np.array([0.2570, 0.2250, 59.1000])),
    (('10PB', 8.0, 12.0), np.array([0.2480, 0.2110, 59.1000])),
    (('10PB', 8.0, 14.0), np.array([0.2400, 0.1960, 59.1000])),
    (('10PB', 8.0, 16.0), np.array([0.2330, 0.1850, 59.1000])),
    (('2.5P', 8.0, 2.0), np.array([0.3048, 0.3040, 59.1000])),
    (('2.5P', 8.0, 4.0), np.array([0.2962, 0.2850, 59.1000])),
    (('2.5P', 8.0, 6.0), np.array([0.2881, 0.2671, 59.1000])),
    (('2.5P', 8.0, 8.0), np.array([0.2800, 0.2488, 59.1000])),
    (('2.5P', 8.0, 10.0), np.array([0.2740, 0.2310, 59.1000])),
    (('2.5P', 8.0, 12.0), np.array([0.2680, 0.2180, 59.1000])),
    (('2.5P', 8.0, 14.0), np.array([0.2630, 0.2040, 59.1000])),
    (('2.5P', 8.0, 16.0), np.array([0.2590, 0.1940, 59.1000])),
    (('2.5P', 8.0, 18.0), np.array([0.2550, 0.1830, 59.1000])),
    (('5P', 8.0, 2.0), np.array([0.3065, 0.3047, 59.1000])),
    (('5P', 8.0, 4.0), np.array([0.3012, 0.2868, 59.1000])),
    (('5P', 8.0, 6.0), np.array([0.2963, 0.2704, 59.1000])),
    (('5P', 8.0, 8.0), np.array([0.2914, 0.2534, 59.1000])),
    (('5P', 8.0, 10.0), np.array([0.2870, 0.2380, 59.1000])),
    (('5P', 8.0, 12.0), np.array([0.2830, 0.2240, 59.1000])),
    (('5P', 8.0, 14.0), np.array([0.2800, 0.2110, 59.1000])),
    (('5P', 8.0, 16.0), np.array([0.2780, 0.2010, 59.1000])),
    (('5P', 8.0, 18.0), np.array([0.2760, 0.1910, 59.1000])),
    (('5P', 8.0, 20.0), np.array([0.2730, 0.1790, 59.1000])),
    (('5P', 8.0, 22.0), np.array([0.2700, 0.1680, 59.1000])),
    (('7.5P', 8.0, 2.0), np.array([0.3107, 0.3070, 59.1000])),
    (('7.5P', 8.0, 4.0), np.array([0.3114, 0.2915, 59.1000])),
    (('7.5P', 8.0, 6.0), np.array([0.3114, 0.2785, 59.1000])),
    (('7.5P', 8.0, 8.0), np.array([0.3116, 0.2626, 59.1000])),
    (('7.5P', 8.0, 10.0), np.array([0.3116, 0.2497, 59.1000])),
    (('7.5P', 8.0, 12.0), np.array([0.3117, 0.2370, 59.1000])),
    (('7.5P', 8.0, 14.0), np.array([0.3110, 0.2240, 59.1000])),
    (('7.5P', 8.0, 16.0), np.array([0.3110, 0.2140, 59.1000])),
    (('7.5P', 8.0, 18.0), np.array([0.3110, 0.2040, 59.1000])),
    (('7.5P', 8.0, 20.0), np.array([0.3110, 0.1940, 59.1000])),
    (('7.5P', 8.0, 22.0), np.array([0.3110, 0.1840, 59.1000])),
    (('7.5P', 8.0, 24.0), np.array([0.3110, 0.1730, 59.1000])),
    (('7.5P', 8.0, 26.0), np.array([0.3110, 0.1620, 59.1000])),
    (('10P', 8.0, 2.0), np.array([0.3131, 0.3084, 59.1000])),
    (('10P', 8.0, 4.0), np.array([0.3175, 0.2955, 59.1000])),
    (('10P', 8.0, 6.0), np.array([0.3213, 0.2829, 59.1000])),
    (('10P', 8.0, 8.0), np.array([0.3250, 0.2700, 59.1000])),
    (('10P', 8.0, 10.0), np.array([0.3282, 0.2582, 59.1000])),
    (('10P', 8.0, 12.0), np.array([0.3312, 0.2470, 59.1000])),
    (('10P', 8.0, 14.0), np.array([0.3342, 0.2349, 59.1000])),
    (('10P', 8.0, 16.0), np.array([0.3370, 0.2250, 59.1000])),
    (('10P', 8.0, 18.0), np.array([0.3390, 0.2170, 59.1000])),
    (('10P', 8.0, 20.0), np.array([0.3410, 0.2070, 59.1000])),
    (('10P', 8.0, 22.0), np.array([0.3440, 0.1960, 59.1000])),
    (('10P', 8.0, 24.0), np.array([0.3460, 0.1860, 59.1000])),
    (('10P', 8.0, 26.0), np.array([0.3480, 0.1760, 59.1000])),
    (('2.5RP', 8.0, 2.0), np.array([0.3154, 0.3100, 59.1000])),
    (('2.5RP', 8.0, 4.0), np.array([0.3239, 0.3000, 59.1000])),
    (('2.5RP', 8.0, 6.0), np.array([0.3327, 0.2898, 59.1000])),
    (('2.5RP', 8.0, 8.0), np.array([0.3406, 0.2793, 59.1000])),
    (('2.5RP', 8.0, 10.0), np.array([0.3479, 0.2699, 59.1000])),
    (('2.5RP', 8.0, 12.0), np.array([0.3552, 0.2594, 59.1000])),
    (('2.5RP', 8.0, 14.0), np.array([0.3621, 0.2496, 59.1000])),
    (('2.5RP', 8.0, 16.0), np.array([0.3690, 0.2400, 59.1000])),
    (('2.5RP', 8.0, 18.0), np.array([0.3730, 0.2330, 59.1000])),
    (('2.5RP', 8.0, 20.0), np.array([0.3790, 0.2240, 59.1000])),
    (('2.5RP', 8.0, 22.0), np.array([0.3850, 0.2150, 59.1000])),
    (('2.5RP', 8.0, 24.0), np.array([0.3890, 0.2060, 59.1000])),
    (('2.5RP', 8.0, 26.0), np.array([0.3940, 0.1970, 59.1000])),
    (('5RP', 8.0, 2.0), np.array([0.3180, 0.3120, 59.1000])),
    (('5RP', 8.0, 4.0), np.array([0.3308, 0.3052, 59.1000])),
    (('5RP', 8.0, 6.0), np.array([0.3440, 0.2978, 59.1000])),
    (('5RP', 8.0, 8.0), np.array([0.3570, 0.2900, 59.1000])),
    (('5RP', 8.0, 10.0), np.array([0.3685, 0.2828, 59.1000])),
    (('5RP', 8.0, 12.0), np.array([0.3818, 0.2742, 59.1000])),
    (('5RP', 8.0, 14.0), np.array([0.3930, 0.2670, 59.1000])),
    (('5RP', 8.0, 16.0), np.array([0.4020, 0.2600, 59.1000])),
    (('5RP', 8.0, 18.0), np.array([0.4110, 0.2530, 59.1000])),
    (('5RP', 8.0, 20.0), np.array([0.4200, 0.2450, 59.1000])),
    (('5RP', 8.0, 22.0), np.array([0.4290, 0.2370, 59.1000])),
    (('5RP', 8.0, 24.0), np.array([0.4370, 0.2300, 59.1000])),
    (('7.5RP', 8.0, 2.0), np.array([0.3200, 0.3136, 59.1000])),
    (('7.5RP', 8.0, 4.0), np.array([0.3360, 0.3092, 59.1000])),
    (('7.5RP', 8.0, 6.0), np.array([0.3521, 0.3042, 59.1000])),
    (('7.5RP', 8.0, 8.0), np.array([0.3682, 0.2983, 59.1000])),
    (('7.5RP', 8.0, 10.0), np.array([0.3830, 0.2930, 59.1000])),
    (('7.5RP', 8.0, 12.0), np.array([0.4002, 0.2859, 59.1000])),
    (('7.5RP', 8.0, 14.0), np.array([0.4140, 0.2800, 59.1000])),
    (('7.5RP', 8.0, 16.0), np.array([0.4260, 0.2740, 59.1000])),
    (('7.5RP', 8.0, 18.0), np.array([0.4380, 0.2670, 59.1000])),
    (('7.5RP', 8.0, 20.0), np.array([0.4490, 0.2610, 59.1000])),
    (('7.5RP', 8.0, 22.0), np.array([0.4600, 0.2550, 59.1000])),
    (('10RP', 8.0, 2.0), np.array([0.3218, 0.3152, 59.1000])),
    (('10RP', 8.0, 4.0), np.array([0.3412, 0.3135, 59.1000])),
    (('10RP', 8.0, 6.0), np.array([0.3600, 0.3112, 59.1000])),
    (('10RP', 8.0, 8.0), np.array([0.3800, 0.3082, 59.1000])),
    (('10RP', 8.0, 10.0), np.array([0.3983, 0.3049, 59.1000])),
    (('10RP', 8.0, 12.0), np.array([0.4220, 0.3000, 59.1000])),
    (('10RP', 8.0, 14.0), np.array([0.4390, 0.2970, 59.1000])),
    (('10RP', 8.0, 16.0), np.array([0.4530, 0.2930, 59.1000])),
    (('10RP', 8.0, 18.0), np.array([0.4690, 0.2890, 59.1000])),
    (('10RP', 8.0, 20.0), np.array([0.4840, 0.2840, 59.1000])),
    (('2.5R', 8.0, 2.0), np.array([0.3236, 0.3169, 59.1000])),
    (('2.5R', 8.0, 4.0), np.array([0.3460, 0.3177, 59.1000])),
    (('2.5R', 8.0, 6.0), np.array([0.3671, 0.3175, 59.1000])),
    (('2.5R', 8.0, 8.0), np.array([0.3900, 0.3171, 59.1000])),
    (('2.5R', 8.0, 10.0), np.array([0.4125, 0.3160, 59.1000])),
    (('2.5R', 8.0, 12.0), np.array([0.4390, 0.3140, 59.1000])),
    (('2.5R', 8.0, 14.0), np.array([0.4600, 0.3110, 59.1000])),
    (('2.5R', 8.0, 16.0), np.array([0.4760, 0.3080, 59.1000])),
    (('2.5R', 8.0, 18.0), np.array([0.4930, 0.3060, 59.1000])),
    (('2.5R', 8.0, 20.0), np.array([0.5100, 0.3030, 59.1000])),
    (('5R', 8.0, 2.0), np.array([0.3254, 0.3186, 59.1000])),
    (('5R', 8.0, 4.0), np.array([0.3510, 0.3224, 59.1000])),
    (('5R', 8.0, 6.0), np.array([0.3743, 0.3248, 59.1000])),
    (('5R', 8.0, 8.0), np.array([0.4001, 0.3263, 59.1000])),
    (('5R', 8.0, 10.0), np.array([0.4249, 0.3270, 59.1000])),
    (('5R', 8.0, 12.0), np.array([0.4540, 0.3260, 59.1000])),
    (('5R', 8.0, 14.0), np.array([0.4760, 0.3250, 59.1000])),
    (('5R', 8.0, 16.0), np.array([0.4940, 0.3240, 59.1000])),
    (('5R', 8.0, 18.0), np.array([0.5130, 0.3220, 59.1000])),
    (('7.5R', 8.0, 2.0), np.array([0.3277, 0.3211, 59.1000])),
    (('7.5R', 8.0, 4.0), np.array([0.3564, 0.3279, 59.1000])),
    (('7.5R', 8.0, 6.0), np.array([0.3830, 0.3335, 59.1000])),
    (('7.5R', 8.0, 8.0), np.array([0.4118, 0.3385, 59.1000])),
    (('7.5R', 8.0, 10.0), np.array([0.4388, 0.3419, 59.1000])),
    (('7.5R', 8.0, 12.0), np.array([0.4700, 0.3450, 59.1000])),
    (('7.5R', 8.0, 14.0), np.array([0.4920, 0.3440, 59.1000])),
    (('7.5R', 8.0, 16.0), np.array([0.5140, 0.3440, 59.1000])),
    (('7.5R', 8.0, 18.0), np.array([0.5360, 0.3430, 59.1000])),
    (('7.5R', 8.0, 20.0), np.array([0.5530, 0.3430, 59.1000])),
    (('10R', 8.0, 2.0), np.array([0.3301, 0.3237, 59.1000])),
    (('10R', 8.0, 4.0), np.array([0.3621, 0.3349, 59.1000])),
    (('10R', 8.0, 6.0), np.array([0.3910, 0.3442, 59.1000])),
    (('10R', 8.0, 8.0), np.array([0.4212, 0.3526, 59.1000])),
    (('10R', 8.0, 10.0), np.array([0.4490, 0.3589, 59.1000])),
    (('10R', 8.0, 12.0), np.array([0.4810, 0.3650, 59.1000])),
    (('10R', 8.0, 14.0), np.array([0.5070, 0.3690, 59.1000])),
    (('10R', 8.0, 16.0), np.array([0.5290, 0.3710, 59.1000])),
    (('10R', 8.0, 18.0), np.array([0.5510, 0.3740, 59.1000])),
    (('10R', 8.0, 20.0), np.array([0.5690, 0.3740, 59.1000])),
    (('10R', 8.0, 22.0), np.array([0.5880, 0.3750, 59.1000])),
    (('10R', 8.0, 24.0), np.array([0.6070, 0.3750, 59.1000])),
    (('10R', 8.0, 26.0), np.array([0.6250, 0.3750, 59.1000])),
    (('2.5YR', 8.0, 2.0), np.array([0.3334, 0.3276, 59.1000])),
    (('2.5YR', 8.0, 4.0), np.array([0.3667, 0.3429, 59.1000])),
    (('2.5YR', 8.0, 6.0), np.array([0.3960, 0.3547, 59.1000])),
    (('2.5YR', 8.0, 8.0), np.array([0.4275, 0.3662, 59.1000])),
    (('2.5YR', 8.0, 10.0), np.array([0.4552, 0.3761, 59.1000])),
    (('2.5YR', 8.0, 12.0), np.array([0.4852, 0.3847, 59.1000])),
    (('2.5YR', 8.0, 14.0), np.array([0.5130, 0.3910, 59.1000])),
    (('2.5YR', 8.0, 16.0), np.array([0.5330, 0.3960, 59.1000])),
    (('2.5YR', 8.0, 18.0), np.array([0.5510, 0.3990, 59.1000])),
    (('2.5YR', 8.0, 20.0), np.array([0.5660, 0.4020, 59.1000])),
    (('2.5YR', 8.0, 22.0), np.array([0.5800, 0.4030, 59.1000])),
    (('2.5YR', 8.0, 24.0), np.array([0.5920, 0.4040, 59.1000])),
    (('2.5YR', 8.0, 26.0), np.array([0.6020, 0.4050, 59.1000])),
    (('5YR', 8.0, 2.0), np.array([0.3373, 0.3330, 59.1000])),
    (('5YR', 8.0, 4.0), np.array([0.3690, 0.3510, 59.1000])),
    (('5YR', 8.0, 6.0), np.array([0.3988, 0.3663, 59.1000])),
    (('5YR', 8.0, 8.0), np.array([0.4310, 0.3820, 59.1000])),
    (('5YR', 8.0, 10.0), np.array([0.4576, 0.3938, 59.1000])),
    (('5YR', 8.0, 12.0), np.array([0.4849, 0.4050, 59.1000])),
    (('5YR', 8.0, 14.0), np.array([0.5088, 0.4145, 59.1000])),
    (('5YR', 8.0, 16.0), np.array([0.5300, 0.4210, 59.1000])),
    (('5YR', 8.0, 18.0), np.array([0.5420, 0.4250, 59.1000])),
    (('5YR', 8.0, 20.0), np.array([0.5530, 0.4270, 59.1000])),
    (('5YR', 8.0, 22.0), np.array([0.5610, 0.4310, 59.1000])),
    (('5YR', 8.0, 24.0), np.array([0.5710, 0.4330, 59.1000])),
    (('5YR', 8.0, 26.0), np.array([0.5780, 0.4350, 59.1000])),
    (('7.5YR', 8.0, 2.0), np.array([0.3395, 0.3379, 59.1000])),
    (('7.5YR', 8.0, 4.0), np.array([0.3699, 0.3586, 59.1000])),
    (('7.5YR', 8.0, 6.0), np.array([0.4000, 0.3770, 59.1000])),
    (('7.5YR', 8.0, 8.0), np.array([0.4306, 0.3952, 59.1000])),
    (('7.5YR', 8.0, 10.0), np.array([0.4568, 0.4100, 59.1000])),
    (('7.5YR', 8.0, 12.0), np.array([0.4816, 0.4232, 59.1000])),
    (('7.5YR', 8.0, 14.0), np.array([0.5025, 0.4338, 59.1000])),
    (('7.5YR', 8.0, 16.0), np.array([0.5195, 0.4424, 59.1000])),
    (('7.5YR', 8.0, 18.0), np.array([0.5316, 0.4480, 59.1000])),
    (('7.5YR', 8.0, 20.0), np.array([0.5391, 0.4518, 59.1000])),
    (('7.5YR', 8.0, 22.0), np.array([0.5460, 0.4540, 59.1000])),
    (('7.5YR', 8.0, 24.0), np.array([0.5530, 0.4580, 59.1000])),
    (('10YR', 8.0, 2.0), np.array([0.3407, 0.3434, 59.1000])),
    (('10YR', 8.0, 4.0), np.array([0.3701, 0.3674, 59.1000])),
    (('10YR', 8.0, 6.0), np.array([0.3994, 0.3896, 59.1000])),
    (('10YR', 8.0, 8.0), np.array([0.4280, 0.4102, 59.1000])),
    (('10YR', 8.0, 10.0), np.array([0.4527, 0.4268, 59.1000])),
    (('10YR', 8.0, 12.0), np.array([0.4753, 0.4414, 59.1000])),
    (('10YR', 8.0, 14.0), np.array([0.4940, 0.4530, 59.1000])),
    (('10YR', 8.0, 16.0), np.array([0.5079, 0.4613, 59.1000])),
    (('10YR', 8.0, 18.0), np.array([0.5179, 0.4670, 59.1000])),
    (('10YR', 8.0, 20.0), np.array([0.5245, 0.4709, 59.1000])),
    (('10YR', 8.0, 22.0), np.array([0.5300, 0.4740, 59.1000])),
    (('2.5Y', 8.0, 2.0), np.array([0.3406, 0.3484, 59.1000])),
    (('2.5Y', 8.0, 4.0), np.array([0.3684, 0.3751, 59.1000])),
    (('2.5Y', 8.0, 6.0), np.array([0.3969, 0.4009, 59.1000])),
    (('2.5Y', 8.0, 8.0), np.array([0.4231, 0.4231, 59.1000])),
    (('2.5Y', 8.0, 10.0), np.array([0.4469, 0.4423, 59.1000])),
    (('2.5Y', 8.0, 12.0), np.array([0.4678, 0.4589, 59.1000])),
    (('2.5Y', 8.0, 14.0), np.array([0.4842, 0.4712, 59.1000])),
    (('2.5Y', 8.0, 16.0), np.array([0.4957, 0.4800, 59.1000])),
    (('2.5Y', 8.0, 18.0), np.array([0.5033, 0.4855, 59.1000])),
    (('2.5Y', 8.0, 20.0), np.array([0.5091, 0.4900, 59.1000])),
    (('2.5Y', 8.0, 22.0), np.array([0.5140, 0.4940, 59.1000])),
    (('5Y', 8.0, 2.0), np.array([0.3394, 0.3518, 59.1000])),
    (('5Y', 8.0, 4.0), np.array([0.3650, 0.3826, 59.1000])),
    (('5Y', 8.0, 6.0), np.array([0.3913, 0.4117, 59.1000])),
    (('5Y', 8.0, 8.0), np.array([0.4158, 0.4378, 59.1000])),
    (('5Y', 8.0, 10.0), np.array([0.4376, 0.4601, 59.1000])),
    (('5Y', 8.0, 12.0), np.array([0.4562, 0.4788, 59.1000])),
    (('5Y', 8.0, 14.0), np.array([0.4699, 0.4920, 59.1000])),
    (('5Y', 8.0, 16.0), np.array([0.4791, 0.5012, 59.1000])),
    (('5Y', 8.0, 18.0), np.array([0.4847, 0.5069, 59.1000])),
    (('5Y', 8.0, 20.0), np.array([0.4900, 0.5110, 59.1000])),
    (('5Y', 8.0, 22.0), np.array([0.4930, 0.5150, 59.1000])),
    (('7.5Y', 8.0, 2.0), np.array([0.3379, 0.3540, 59.1000])),
    (('7.5Y', 8.0, 4.0), np.array([0.3622, 0.3861, 59.1000])),
    (('7.5Y', 8.0, 6.0), np.array([0.3862, 0.4175, 59.1000])),
    (('7.5Y', 8.0, 8.0), np.array([0.4088, 0.4466, 59.1000])),
    (('7.5Y', 8.0, 10.0), np.array([0.4283, 0.4712, 59.1000])),
    (('7.5Y', 8.0, 12.0), np.array([0.4455, 0.4917, 59.1000])),
    (('7.5Y', 8.0, 14.0), np.array([0.4574, 0.5062, 59.1000])),
    (('7.5Y', 8.0, 16.0), np.array([0.4658, 0.5158, 59.1000])),
    (('7.5Y', 8.0, 18.0), np.array([0.4709, 0.5220, 59.1000])),
    (('7.5Y', 8.0, 20.0), np.array([0.4750, 0.5260, 59.1000])),
    (('10Y', 9.0, 2.0), np.array([0.3349, 0.3537, 78.6600])),
    (('10Y', 9.0, 4.0), np.array([0.3558, 0.3852, 78.6600])),
    (('10Y', 9.0, 6.0), np.array([0.3761, 0.4155, 78.6600])),
    (('10Y', 9.0, 8.0), np.array([0.3957, 0.4450, 78.6600])),
    (('10Y', 9.0, 10.0), np.array([0.4120, 0.4694, 78.6600])),
    (('10Y', 9.0, 12.0), np.array([0.4271, 0.4920, 78.6600])),
    (('10Y', 9.0, 14.0), np.array([0.4393, 0.5101, 78.6600])),
    (('10Y', 9.0, 16.0), np.array([0.4477, 0.5225, 78.6600])),
    (('10Y', 9.0, 18.0), np.array([0.4540, 0.5320, 78.6600])),
    (('10Y', 9.0, 20.0), np.array([0.4570, 0.5370, 78.6600])),
    (('10Y', 9.0, 22.0), np.array([0.4620, 0.5440, 78.6600])),
    (('2.5GY', 9.0, 2.0), np.array([0.3321, 0.3539, 78.6600])),
    (('2.5GY', 9.0, 4.0), np.array([0.3499, 0.3866, 78.6600])),
    (('2.5GY', 9.0, 6.0), np.array([0.3670, 0.4178, 78.6600])),
    (('2.5GY', 9.0, 8.0), np.array([0.3834, 0.4490, 78.6600])),
    (('2.5GY', 9.0, 10.0), np.array([0.3973, 0.4761, 78.6600])),
    (('2.5GY', 9.0, 12.0), np.array([0.4108, 0.5028, 78.6600])),
    (('2.5GY', 9.0, 14.0), np.array([0.4212, 0.5237, 78.6600])),
    (('2.5GY', 9.0, 16.0), np.array([0.4288, 0.5383, 78.6600])),
    (('2.5GY', 9.0, 18.0), np.array([0.4354, 0.5508, 78.6600])),
    (('2.5GY', 9.0, 20.0), np.array([0.4380, 0.5560, 78.6600])),
    (('2.5GY', 9.0, 22.0), np.array([0.4420, 0.5640, 78.6600])),
    (('5GY', 9.0, 2.0), np.array([0.3284, 0.3534, 78.6600])),
    (('5GY', 9.0, 4.0), np.array([0.3437, 0.3861, 78.6600])),
    (('5GY', 9.0, 6.0), np.array([0.3572, 0.4179, 78.6600])),
    (('5GY', 9.0, 8.0), np.array([0.3698, 0.4497, 78.6600])),
    (('5GY', 9.0, 10.0), np.array([0.3810, 0.4791, 78.6600])),
    (('5GY', 9.0, 12.0), np.array([0.3911, 0.5082, 78.6600])),
    (('5GY', 9.0, 14.0), np.array([0.3993, 0.5329, 78.6600])),
    (('5GY', 9.0, 16.0), np.array([0.4058, 0.5541, 78.6600])),
    (('5GY', 9.0, 18.0), np.array([0.4108, 0.5699, 78.6600])),
    (('5GY', 9.0, 20.0), np.array([0.4140, 0.5800, 78.6600])),
    (('5GY', 9.0, 22.0), np.array([0.4170, 0.5900, 78.6600])),
    (('7.5GY', 9.0, 2.0), np.array([0.3198, 0.3500, 78.6600])),
    (('7.5GY', 9.0, 4.0), np.array([0.3274, 0.3793, 78.6600])),
    (('7.5GY', 9.0, 6.0), np.array([0.3351, 0.4111, 78.6600])),
    (('7.5GY', 9.0, 8.0), np.array([0.3414, 0.4415, 78.6600])),
    (('7.5GY', 9.0, 10.0), np.array([0.3471, 0.4735, 78.6600])),
    (('7.5GY', 9.0, 12.0), np.array([0.3518, 0.5042, 78.6600])),
    (('7.5GY', 9.0, 14.0), np.array([0.3551, 0.5339, 78.6600])),
    (('7.5GY', 9.0, 16.0), np.array([0.3581, 0.5654, 78.6600])),
    (('7.5GY', 9.0, 18.0), np.array([0.3602, 0.5920, 78.6600])),
    (('7.5GY', 9.0, 20.0), np.array([0.3610, 0.6140, 78.6600])),
    (('7.5GY', 9.0, 22.0), np.array([0.3620, 0.6380, 78.6600])),
    (('7.5GY', 9.0, 24.0), np.array([0.3600, 0.6570, 78.6600])),
    (('7.5GY', 9.0, 26.0), np.array([0.3600, 0.6820, 78.6600])),
    (('7.5GY', 9.0, 28.0), np.array([0.3600, 0.7070, 78.6600])),
    (('10GY', 9.0, 2.0), np.array([0.3124, 0.3454, 78.6600])),
    (('10GY', 9.0, 4.0), np.array([0.3144, 0.3711, 78.6600])),
    (('10GY', 9.0, 6.0), np.array([0.3153, 0.4008, 78.6600])),
    (('10GY', 9.0, 8.0), np.array([0.3157, 0.4259, 78.6600])),
    (('10GY', 9.0, 10.0), np.array([0.3155, 0.4558, 78.6600])),
    (('10GY', 9.0, 12.0), np.array([0.3139, 0.4829, 78.6600])),
    (('10GY', 9.0, 14.0), np.array([0.3115, 0.5129, 78.6600])),
    (('10GY', 9.0, 16.0), np.array([0.3079, 0.5440, 78.6600])),
    (('10GY', 9.0, 18.0), np.array([0.3032, 0.5748, 78.6600])),
    (('10GY', 9.0, 20.0), np.array([0.2980, 0.6020, 78.6600])),
    (('10GY', 9.0, 22.0), np.array([0.2920, 0.6270, 78.6600])),
    (('10GY', 9.0, 24.0), np.array([0.2840, 0.6530, 78.6600])),
    (('10GY', 9.0, 26.0), np.array([0.2770, 0.6810, 78.6600])),
    (('10GY', 9.0, 28.0), np.array([0.2680, 0.7070, 78.6600])),
    (('2.5G', 9.0, 2.0), np.array([0.3058, 0.3400, 78.6600])),
    (('2.5G', 9.0, 4.0), np.array([0.3018, 0.3606, 78.6600])),
    (('2.5G', 9.0, 6.0), np.array([0.2966, 0.3846, 78.6600])),
    (('2.5G', 9.0, 8.0), np.array([0.2912, 0.4054, 78.6600])),
    (('2.5G', 9.0, 10.0), np.array([0.2851, 0.4275, 78.6600])),
    (('2.5G', 9.0, 12.0), np.array([0.2786, 0.4491, 78.6600])),
    (('2.5G', 9.0, 14.0), np.array([0.2711, 0.4726, 78.6600])),
    (('2.5G', 9.0, 16.0), np.array([0.2630, 0.4966, 78.6600])),
    (('2.5G', 9.0, 18.0), np.array([0.2530, 0.5250, 78.6600])),
    (('2.5G', 9.0, 20.0), np.array([0.2420, 0.5520, 78.6600])),
    (('2.5G', 9.0, 22.0), np.array([0.2300, 0.5760, 78.6600])),
    (('2.5G', 9.0, 24.0), np.array([0.2190, 0.5990, 78.6600])),
    (('2.5G', 9.0, 26.0), np.array([0.2060, 0.6190, 78.6600])),
    (('2.5G', 9.0, 28.0), np.array([0.1920, 0.6430, 78.6600])),
    (('5G', 9.0, 2.0), np.array([0.3017, 0.3357, 78.6600])),
    (('5G', 9.0, 4.0), np.array([0.2933, 0.3519, 78.6600])),
    (('5G', 9.0, 6.0), np.array([0.2832, 0.3697, 78.6600])),
    (('5G', 9.0, 8.0), np.array([0.2735, 0.3854, 78.6600])),
    (('5G', 9.0, 10.0), np.array([0.2639, 0.4001, 78.6600])),
    (('5G', 9.0, 12.0), np.array([0.2528, 0.4160, 78.6600])),
    (('5G', 9.0, 14.0), np.array([0.2420, 0.4300, 78.6600])),
    (('5G', 9.0, 16.0), np.array([0.2290, 0.4460, 78.6600])),
    (('5G', 9.0, 18.0), np.array([0.2140, 0.4610, 78.6600])),
    (('5G', 9.0, 20.0), np.array([0.2000, 0.4740, 78.6600])),
    (('5G', 9.0, 22.0), np.array([0.1850, 0.4870, 78.6600])),
    (('5G', 9.0, 24.0), np.array([0.1700, 0.4990, 78.6600])),
    (('5G', 9.0, 26.0), np.array([0.1570, 0.5090, 78.6600])),
    (('5G', 9.0, 28.0), np.array([0.1430, 0.5190, 78.6600])),
    (('7.5G', 9.0, 2.0), np.array([0.2987, 0.3323, 78.6600])),
    (('7.5G', 9.0, 4.0), np.array([0.2882, 0.3461, 78.6600])),
    (('7.5G', 9.0, 6.0), np.array([0.2763, 0.3607, 78.6600])),
    (('7.5G', 9.0, 8.0), np.array([0.2652, 0.3738, 78.6600])),
    (('7.5G', 9.0, 10.0), np.array([0.2545, 0.3855, 78.6600])),
    (('7.5G', 9.0, 12.0), np.array([0.2419, 0.3985, 78.6600])),
    (('7.5G', 9.0, 14.0), np.array([0.2290, 0.4100, 78.6600])),
    (('7.5G', 9.0, 16.0), np.array([0.2160, 0.4230, 78.6600])),
    (('7.5G', 9.0, 18.0), np.array([0.2010, 0.4360, 78.6600])),
    (('7.5G', 9.0, 20.0), np.array([0.1870, 0.4460, 78.6600])),
    (('7.5G', 9.0, 22.0), np.array([0.1740, 0.4570, 78.6600])),
    (('7.5G', 9.0, 24.0), np.array([0.1600, 0.4670, 78.6600])),
    (('10G', 9.0, 2.0), np.array([0.2965, 0.3293, 78.6600])),
    (('10G', 9.0, 4.0), np.array([0.2840, 0.3402, 78.6600])),
    (('10G', 9.0, 6.0), np.array([0.2703, 0.3513, 78.6600])),
    (('10G', 9.0, 8.0), np.array([0.2574, 0.3618, 78.6600])),
    (('10G', 9.0, 10.0), np.array([0.2457, 0.3702, 78.6600])),
    (('10G', 9.0, 12.0), np.array([0.2325, 0.3796, 78.6600])),
    (('10G', 9.0, 14.0), np.array([0.2170, 0.3880, 78.6600])),
    (('10G', 9.0, 16.0), np.array([0.2030, 0.3960, 78.6600])),
    (('10G', 9.0, 18.0), np.array([0.1880, 0.4050, 78.6600])),
    (('10G', 9.0, 20.0), np.array([0.1750, 0.4110, 78.6600])),
    (('10G', 9.0, 22.0), np.array([0.1610, 0.4190, 78.6600])),
    (('10G', 9.0, 24.0), np.array([0.1470, 0.4250, 78.6600])),
    (('2.5BG', 9.0, 2.0), np.array([0.2947, 0.3267, 78.6600])),
    (('2.5BG', 9.0, 4.0), np.array([0.2805, 0.3349, 78.6600])),
    (('2.5BG', 9.0, 6.0), np.array([0.2652, 0.3433, 78.6600])),
    (('2.5BG', 9.0, 8.0), np.array([0.2509, 0.3507, 78.6600])),
    (('2.5BG', 9.0, 10.0), np.array([0.2382, 0.3568, 78.6600])),
    (('2.5BG', 9.0, 12.0), np.array([0.2220, 0.3640, 78.6600])),
    (('2.5BG', 9.0, 14.0), np.array([0.2080, 0.3700, 78.6600])),
    (('2.5BG', 9.0, 16.0), np.array([0.1940, 0.3760, 78.6600])),
    (('2.5BG', 9.0, 18.0), np.array([0.1780, 0.3820, 78.6600])),
    (('2.5BG', 9.0, 20.0), np.array([0.1660, 0.3860, 78.6600])),
    (('2.5BG', 9.0, 22.0), np.array([0.1520, 0.3910, 78.6600])),
    (('5BG', 9.0, 2.0), np.array([0.2930, 0.3232, 78.6600])),
    (('5BG', 9.0, 4.0), np.array([0.2768, 0.3287, 78.6600])),
    (('5BG', 9.0, 6.0), np.array([0.2599, 0.3338, 78.6600])),
    (('5BG', 9.0, 8.0), np.array([0.2437, 0.3378, 78.6600])),
    (('5BG', 9.0, 10.0), np.array([0.2301, 0.3405, 78.6600])),
    (('5BG', 9.0, 12.0), np.array([0.2120, 0.3440, 78.6600])),
    (('5BG', 9.0, 14.0), np.array([0.1960, 0.3460, 78.6600])),
    (('5BG', 9.0, 16.0), np.array([0.1800, 0.3480, 78.6600])),
    (('5BG', 9.0, 18.0), np.array([0.1640, 0.3490, 78.6600])),
    (('5BG', 9.0, 20.0), np.array([0.1500, 0.3500, 78.6600])),
    (('7.5BG', 9.0, 2.0), np.array([0.2911, 0.3188, 78.6600])),
    (('7.5BG', 9.0, 4.0), np.array([0.2728, 0.3208, 78.6600])),
    (('7.5BG', 9.0, 6.0), np.array([0.2543, 0.3220, 78.6600])),
    (('7.5BG', 9.0, 8.0), np.array([0.2361, 0.3225, 78.6600])),
    (('7.5BG', 9.0, 10.0), np.array([0.2215, 0.3226, 78.6600])),
    (('7.5BG', 9.0, 12.0), np.array([0.2030, 0.3230, 78.6600])),
    (('7.5BG', 9.0, 14.0), np.array([0.1870, 0.3230, 78.6600])),
    (('7.5BG', 9.0, 16.0), np.array([0.1700, 0.3230, 78.6600])),
    (('7.5BG', 9.0, 18.0), np.array([0.1560, 0.3220, 78.6600])),
    (('10BG', 9.0, 2.0), np.array([0.2907, 0.3159, 78.6600])),
    (('10BG', 9.0, 4.0), np.array([0.2700, 0.3140, 78.6600])),
    (('10BG', 9.0, 6.0), np.array([0.2501, 0.3118, 78.6600])),
    (('10BG', 9.0, 8.0), np.array([0.2320, 0.3100, 78.6600])),
    (('10BG', 9.0, 10.0), np.array([0.2140, 0.3060, 78.6600])),
    (('10BG', 9.0, 12.0), np.array([0.1960, 0.3040, 78.6600])),
    (('10BG', 9.0, 14.0), np.array([0.1800, 0.3010, 78.6600])),
    (('10BG', 9.0, 16.0), np.array([0.1620, 0.2980, 78.6600])),
    (('10BG', 9.0, 18.0), np.array([0.1470, 0.2940, 78.6600])),
    (('2.5B', 9.0, 2.0), np.array([0.2909, 0.3125, 78.6600])),
    (('2.5B', 9.0, 4.0), np.array([0.2680, 0.3073, 78.6600])),
    (('2.5B', 9.0, 6.0), np.array([0.2470, 0.3020, 78.6600])),
    (('2.5B', 9.0, 8.0), np.array([0.2280, 0.2960, 78.6600])),
    (('2.5B', 9.0, 10.0), np.array([0.2110, 0.2900, 78.6600])),
    (('2.5B', 9.0, 12.0), np.array([0.1930, 0.2840, 78.6600])),
    (('2.5B', 9.0, 14.0), np.array([0.1760, 0.2780, 78.6600])),
    (('2.5B', 9.0, 16.0), np.array([0.1580, 0.2710, 78.6600])),
    (('5B', 9.0, 2.0), np.array([0.2919, 0.3102, 78.6600])),
    (('5B', 9.0, 4.0), np.array([0.2675, 0.3005, 78.6600])),
    (('5B', 9.0, 6.0), np.array([0.2460, 0.2900, 78.6600])),
    (('5B', 9.0, 8.0), np.array([0.2280, 0.2810, 78.6600])),
    (('5B', 9.0, 10.0), np.array([0.2100, 0.2720, 78.6600])),
    (('5B', 9.0, 12.0), np.array([0.1910, 0.2600, 78.6600])),
    (('5B', 9.0, 14.0), np.array([0.1750, 0.2510, 78.6600])),
    (('5B', 9.0, 16.0), np.array([0.1570, 0.2410, 78.6600])),
    (('7.5B', 9.0, 2.0), np.array([0.2937, 0.3087, 78.6600])),
    (('7.5B', 9.0, 4.0), np.array([0.2688, 0.2961, 78.6600])),
    (('7.5B', 9.0, 6.0), np.array([0.2470, 0.2850, 78.6600])),
    (('7.5B', 9.0, 8.0), np.array([0.2280, 0.2730, 78.6600])),
    (('7.5B', 9.0, 10.0), np.array([0.2110, 0.2620, 78.6600])),
    (('7.5B', 9.0, 12.0), np.array([0.1940, 0.2520, 78.6600])),
    (('10B', 9.0, 2.0), np.array([0.2949, 0.3076, 78.6600])),
    (('10B', 9.0, 4.0), np.array([0.2712, 0.2924, 78.6600])),
    (('10B', 9.0, 6.0), np.array([0.2500, 0.2780, 78.6600])),
    (('10B', 9.0, 8.0), np.array([0.2330, 0.2650, 78.6600])),
    (('10B', 9.0, 10.0), np.array([0.2160, 0.2500, 78.6600])),
    (('2.5PB', 9.0, 2.0), np.array([0.2975, 0.3063, 78.6600])),
    (('2.5PB', 9.0, 4.0), np.array([0.2760, 0.2890, 78.6600])),
    (('2.5PB', 9.0, 6.0), np.array([0.2560, 0.2720, 78.6600])),
    (('2.5PB', 9.0, 8.0), np.array([0.2390, 0.2570, 78.6600])),
    (('2.5PB', 9.0, 10.0), np.array([0.2240, 0.2430, 78.6600])),
    (('5PB', 9.0, 2.0), np.array([0.2991, 0.3057, 78.6600])),
    (('5PB', 9.0, 4.0), np.array([0.2810, 0.2870, 78.6600])),
    (('5PB', 9.0, 6.0), np.array([0.2620, 0.2680, 78.6600])),
    (('5PB', 9.0, 8.0), np.array([0.2470, 0.2520, 78.6600])),
    (('5PB', 9.0, 10.0), np.array([0.2320, 0.2350, 78.6600])),
    (('7.5PB', 9.0, 2.0), np.array([0.3015, 0.3052, 78.6600])),
    (('7.5PB', 9.0, 4.0), np.array([0.2860, 0.2850, 78.6600])),
    (('7.5PB', 9.0, 6.0), np.array([0.2710, 0.2660, 78.6600])),
    (('7.5PB', 9.0, 8.0), np.array([0.2570, 0.2450, 78.6600])),
    (('7.5PB', 9.0, 10.0), np.array([0.2470, 0.2270, 78.6600])),
    (('10PB', 9.0, 2.0), np.array([0.3038, 0.3054, 78.6600])),
    (('10PB', 9.0, 4.0), np.array([0.2910, 0.2850, 78.6600])),
    (('10PB', 9.0, 6.0), np.array([0.2780, 0.2660, 78.6600])),
    (('10PB', 9.0, 8.0), np.array([0.2670, 0.2450, 78.6600])),
    (('10PB', 9.0, 10.0), np.array([0.2560, 0.2260, 78.6600])),
    (('2.5P', 9.0, 2.0), np.array([0.3050, 0.3051, 78.6600])),
    (('2.5P', 9.0, 4.0), np.array([0.2963, 0.2865, 78.6600])),
    (('2.5P', 9.0, 6.0), np.array([0.2880, 0.2680, 78.6600])),
    (('2.5P', 9.0, 8.0), np.array([0.2800, 0.2490, 78.6600])),
    (('2.5P', 9.0, 10.0), np.array([0.2730, 0.2300, 78.6600])),
    (('5P', 9.0, 2.0), np.array([0.3067, 0.3060, 78.6600])),
    (('5P', 9.0, 4.0), np.array([0.3003, 0.2870, 78.6600])),
    (('5P', 9.0, 6.0), np.array([0.2960, 0.2710, 78.6600])),
    (('5P', 9.0, 8.0), np.array([0.2920, 0.2530, 78.6600])),
    (('5P', 9.0, 10.0), np.array([0.2860, 0.2350, 78.6600])),
    (('5P', 9.0, 12.0), np.array([0.2840, 0.2210, 78.6600])),
    (('5P', 9.0, 14.0), np.array([0.2810, 0.2080, 78.6600])),
    (('7.5P', 9.0, 2.0), np.array([0.3107, 0.3081, 78.6600])),
    (('7.5P', 9.0, 4.0), np.array([0.3117, 0.2928, 78.6600])),
    (('7.5P', 9.0, 6.0), np.array([0.3120, 0.2788, 78.6600])),
    (('7.5P', 9.0, 8.0), np.array([0.3110, 0.2620, 78.6600])),
    (('7.5P', 9.0, 10.0), np.array([0.3100, 0.2470, 78.6600])),
    (('7.5P', 9.0, 12.0), np.array([0.3080, 0.2330, 78.6600])),
    (('7.5P', 9.0, 14.0), np.array([0.3060, 0.2200, 78.6600])),
    (('7.5P', 9.0, 16.0), np.array([0.3050, 0.2080, 78.6600])),
    (('10P', 9.0, 2.0), np.array([0.3128, 0.3094, 78.6600])),
    (('10P', 9.0, 4.0), np.array([0.3176, 0.2966, 78.6600])),
    (('10P', 9.0, 6.0), np.array([0.3218, 0.2845, 78.6600])),
    (('10P', 9.0, 8.0), np.array([0.3260, 0.2700, 78.6600])),
    (('10P', 9.0, 10.0), np.array([0.3300, 0.2570, 78.6600])),
    (('10P', 9.0, 12.0), np.array([0.3340, 0.2460, 78.6600])),
    (('10P', 9.0, 14.0), np.array([0.3360, 0.2350, 78.6600])),
    (('10P', 9.0, 16.0), np.array([0.3390, 0.2240, 78.6600])),
    (('2.5RP', 9.0, 2.0), np.array([0.3149, 0.3108, 78.6600])),
    (('2.5RP', 9.0, 4.0), np.array([0.3234, 0.3010, 78.6600])),
    (('2.5RP', 9.0, 6.0), np.array([0.3322, 0.2910, 78.6600])),
    (('2.5RP', 9.0, 8.0), np.array([0.3400, 0.2800, 78.6600])),
    (('2.5RP', 9.0, 10.0), np.array([0.3490, 0.2710, 78.6600])),
    (('2.5RP', 9.0, 12.0), np.array([0.3560, 0.2610, 78.6600])),
    (('2.5RP', 9.0, 14.0), np.array([0.3640, 0.2510, 78.6600])),
    (('2.5RP', 9.0, 16.0), np.array([0.3710, 0.2430, 78.6600])),
    (('5RP', 9.0, 2.0), np.array([0.3172, 0.3126, 78.6600])),
    (('5RP', 9.0, 4.0), np.array([0.3301, 0.3060, 78.6600])),
    (('5RP', 9.0, 6.0), np.array([0.3431, 0.2988, 78.6600])),
    (('5RP', 9.0, 8.0), np.array([0.3550, 0.2920, 78.6600])),
    (('5RP', 9.0, 10.0), np.array([0.3680, 0.2840, 78.6600])),
    (('5RP', 9.0, 12.0), np.array([0.3800, 0.2760, 78.6600])),
    (('5RP', 9.0, 14.0), np.array([0.3890, 0.2690, 78.6600])),
    (('5RP', 9.0, 16.0), np.array([0.4000, 0.2630, 78.6600])),
    (('7.5RP', 9.0, 2.0), np.array([0.3190, 0.3141, 78.6600])),
    (('7.5RP', 9.0, 4.0), np.array([0.3350, 0.3099, 78.6600])),
    (('7.5RP', 9.0, 6.0), np.array([0.3512, 0.3052, 78.6600])),
    (('7.5RP', 9.0, 8.0), np.array([0.3640, 0.3000, 78.6600])),
    (('7.5RP', 9.0, 10.0), np.array([0.3820, 0.2950, 78.6600])),
    (('7.5RP', 9.0, 12.0), np.array([0.3970, 0.2890, 78.6600])),
    (('7.5RP', 9.0, 14.0), np.array([0.4100, 0.2840, 78.6600])),
    (('10RP', 9.0, 2.0), np.array([0.3205, 0.3155, 78.6600])),
    (('10RP', 9.0, 4.0), np.array([0.3400, 0.3140, 78.6600])),
    (('10RP', 9.0, 6.0), np.array([0.3590, 0.3118, 78.6600])),
    (('10RP', 9.0, 8.0), np.array([0.3760, 0.3100, 78.6600])),
    (('10RP', 9.0, 10.0), np.array([0.3940, 0.3060, 78.6600])),
    (('10RP', 9.0, 12.0), np.array([0.4140, 0.3020, 78.6600])),
    (('10RP', 9.0, 14.0), np.array([0.4310, 0.2980, 78.6600])),
    (('2.5R', 9.0, 2.0), np.array([0.3220, 0.3168, 78.6600])),
    (('2.5R', 9.0, 4.0), np.array([0.3445, 0.3179, 78.6600])),
    (('2.5R', 9.0, 6.0), np.array([0.3665, 0.3183, 78.6600])),
    (('2.5R', 9.0, 8.0), np.array([0.3840, 0.3180, 78.6600])),
    (('2.5R', 9.0, 10.0), np.array([0.4060, 0.3160, 78.6600])),
    (('2.5R', 9.0, 12.0), np.array([0.4290, 0.3140, 78.6600])),
    (('2.5R', 9.0, 14.0), np.array([0.4480, 0.3110, 78.6600])),
    (('5R', 9.0, 2.0), np.array([0.3240, 0.3188, 78.6600])),
    (('5R', 9.0, 4.0), np.array([0.3495, 0.3226, 78.6600])),
    (('5R', 9.0, 6.0), np.array([0.3734, 0.3256, 78.6600])),
    (('5R', 9.0, 8.0), np.array([0.3950, 0.3260, 78.6600])),
    (('5R', 9.0, 10.0), np.array([0.4180, 0.3260, 78.6600])),
    (('5R', 9.0, 12.0), np.array([0.4440, 0.3250, 78.6600])),
    (('7.5R', 9.0, 2.0), np.array([0.3263, 0.3210, 78.6600])),
    (('7.5R', 9.0, 4.0), np.array([0.3551, 0.3283, 78.6600])),
    (('7.5R', 9.0, 6.0), np.array([0.3812, 0.3348, 78.6600])),
    (('7.5R', 9.0, 8.0), np.array([0.4070, 0.3380, 78.6600])),
    (('7.5R', 9.0, 10.0), np.array([0.4310, 0.3400, 78.6600])),
    (('7.5R', 9.0, 12.0), np.array([0.4570, 0.3400, 78.6600])),
    (('7.5R', 9.0, 14.0), np.array([0.4800, 0.3400, 78.6600])),
    (('10R', 9.0, 2.0), np.array([0.3284, 0.3233, 78.6600])),
    (('10R', 9.0, 4.0), np.array([0.3600, 0.3348, 78.6600])),
    (('10R', 9.0, 6.0), np.array([0.3880, 0.3439, 78.6600])),
    (('10R', 9.0, 8.0), np.array([0.4160, 0.3500, 78.6600])),
    (('10R', 9.0, 10.0), np.array([0.4410, 0.3560, 78.6600])),
    (('10R', 9.0, 12.0), np.array([0.4650, 0.3610, 78.6600])),
    (('10R', 9.0, 14.0), np.array([0.4900, 0.3640, 78.6600])),
    (('10R', 9.0, 16.0), np.array([0.5100, 0.3680, 78.6600])),
    (('2.5YR', 9.0, 2.0), np.array([0.3320, 0.3273, 78.6600])),
    (('2.5YR', 9.0, 4.0), np.array([0.3641, 0.3422, 78.6600])),
    (('2.5YR', 9.0, 6.0), np.array([0.3927, 0.3550, 78.6600])),
    (('2.5YR', 9.0, 8.0), np.array([0.4220, 0.3650, 78.6600])),
    (('2.5YR', 9.0, 10.0), np.array([0.4460, 0.3730, 78.6600])),
    (('2.5YR', 9.0, 12.0), np.array([0.4690, 0.3800, 78.6600])),
    (('2.5YR', 9.0, 14.0), np.array([0.4920, 0.3870, 78.6600])),
    (('2.5YR', 9.0, 16.0), np.array([0.5130, 0.3920, 78.6600])),
    (('2.5YR', 9.0, 18.0), np.array([0.5320, 0.3960, 78.6600])),
    (('5YR', 9.0, 2.0), np.array([0.3353, 0.3325, 78.6600])),
    (('5YR', 9.0, 4.0), np.array([0.3668, 0.3509, 78.6600])),
    (('5YR', 9.0, 6.0), np.array([0.3948, 0.3659, 78.6600])),
    (('5YR', 9.0, 8.0), np.array([0.4220, 0.3790, 78.6600])),
    (('5YR', 9.0, 10.0), np.array([0.4480, 0.3900, 78.6600])),
    (('5YR', 9.0, 12.0), np.array([0.4700, 0.4000, 78.6600])),
    (('5YR', 9.0, 14.0), np.array([0.4910, 0.4080, 78.6600])),
    (('5YR', 9.0, 16.0), np.array([0.5100, 0.4160, 78.6600])),
    (('5YR', 9.0, 18.0), np.array([0.5250, 0.4210, 78.6600])),
    (('5YR', 9.0, 20.0), np.array([0.5410, 0.4260, 78.6600])),
    (('5YR', 9.0, 22.0), np.array([0.5500, 0.4300, 78.6600])),
    (('5YR', 9.0, 24.0), np.array([0.5590, 0.4330, 78.6600])),
    (('5YR', 9.0, 26.0), np.array([0.5670, 0.4350, 78.6600])),
    (('7.5YR', 9.0, 2.0), np.array([0.3380, 0.3377, 78.6600])),
    (('7.5YR', 9.0, 4.0), np.array([0.3679, 0.3585, 78.6600])),
    (('7.5YR', 9.0, 6.0), np.array([0.3950, 0.3763, 78.6600])),
    (('7.5YR', 9.0, 8.0), np.array([0.4220, 0.3930, 78.6600])),
    (('7.5YR', 9.0, 10.0), np.array([0.4470, 0.4070, 78.6600])),
    (('7.5YR', 9.0, 12.0), np.array([0.4680, 0.4180, 78.6600])),
    (('7.5YR', 9.0, 14.0), np.array([0.4860, 0.4280, 78.6600])),
    (('7.5YR', 9.0, 16.0), np.array([0.5030, 0.4360, 78.6600])),
    (('7.5YR', 9.0, 18.0), np.array([0.5170, 0.4440, 78.6600])),
    (('7.5YR', 9.0, 20.0), np.array([0.5290, 0.4500, 78.6600])),
    (('7.5YR', 9.0, 22.0), np.array([0.5370, 0.4540, 78.6600])),
    (('7.5YR', 9.0, 24.0), np.array([0.5450, 0.4590, 78.6600])),
    (('7.5YR', 9.0, 26.0), np.array([0.5530, 0.4630, 78.6600])),
    (('10YR', 9.0, 2.0), np.array([0.3392, 0.3430, 78.6600])),
    (('10YR', 9.0, 4.0), np.array([0.3677, 0.3668, 78.6600])),
    (('10YR', 9.0, 6.0), np.array([0.3941, 0.3877, 78.6600])),
    (('10YR', 9.0, 8.0), np.array([0.4199, 0.4069, 78.6600])),
    (('10YR', 9.0, 10.0), np.array([0.4440, 0.4240, 78.6600])),
    (('10YR', 9.0, 12.0), np.array([0.4630, 0.4360, 78.6600])),
    (('10YR', 9.0, 14.0), np.array([0.4800, 0.4470, 78.6600])),
    (('10YR', 9.0, 16.0), np.array([0.4950, 0.4550, 78.6600])),
    (('10YR', 9.0, 18.0), np.array([0.5060, 0.4620, 78.6600])),
    (('10YR', 9.0, 20.0), np.array([0.5160, 0.4680, 78.6600])),
    (('10YR', 9.0, 22.0), np.array([0.5240, 0.4730, 78.6600])),
    (('10YR', 9.0, 24.0), np.array([0.5320, 0.4780, 78.6600])),
    (('2.5Y', 9.0, 2.0), np.array([0.3390, 0.3472, 78.6600])),
    (('2.5Y', 9.0, 4.0), np.array([0.3655, 0.3738, 78.6600])),
    (('2.5Y', 9.0, 6.0), np.array([0.3910, 0.3972, 78.6600])),
    (('2.5Y', 9.0, 8.0), np.array([0.4154, 0.4186, 78.6600])),
    (('2.5Y', 9.0, 10.0), np.array([0.4370, 0.4369, 78.6600])),
    (('2.5Y', 9.0, 12.0), np.array([0.4569, 0.4527, 78.6600])),
    (('2.5Y', 9.0, 14.0), np.array([0.4730, 0.4650, 78.6600])),
    (('2.5Y', 9.0, 16.0), np.array([0.4860, 0.4760, 78.6600])),
    (('2.5Y', 9.0, 18.0), np.array([0.4940, 0.4830, 78.6600])),
    (('2.5Y', 9.0, 20.0), np.array([0.5010, 0.4880, 78.6600])),
    (('2.5Y', 9.0, 22.0), np.array([0.5090, 0.4940, 78.6600])),
    (('2.5Y', 9.0, 24.0), np.array([0.5150, 0.4990, 78.6600])),
    (('5Y', 9.0, 2.0), np.array([0.3378, 0.3504, 78.6600])),
    (('5Y', 9.0, 4.0), np.array([0.3621, 0.3799, 78.6600])),
    (('5Y', 9.0, 6.0), np.array([0.3858, 0.4071, 78.6600])),
    (('5Y', 9.0, 8.0), np.array([0.4080, 0.4319, 78.6600])),
    (('5Y', 9.0, 10.0), np.array([0.4275, 0.4529, 78.6600])),
    (('5Y', 9.0, 12.0), np.array([0.4455, 0.4719, 78.6600])),
    (('5Y', 9.0, 14.0), np.array([0.4602, 0.4869, 78.6600])),
    (('5Y', 9.0, 16.0), np.array([0.4711, 0.4977, 78.6600])),
    (('5Y', 9.0, 18.0), np.array([0.4782, 0.5049, 78.6600])),
    (('5Y', 9.0, 20.0), np.array([0.4830, 0.5092, 78.6600])),
    (('5Y', 9.0, 22.0), np.array([0.4890, 0.5150, 78.6600])),
    (('5Y', 9.0, 24.0), np.array([0.4950, 0.5220, 78.6600])),
    (('7.5Y', 9.0, 2.0), np.array([0.3365, 0.3527, 78.6600])),
    (('7.5Y', 9.0, 4.0), np.array([0.3591, 0.3832, 78.6600])),
    (('7.5Y', 9.0, 6.0), np.array([0.3811, 0.4123, 78.6600])),
    (('7.5Y', 9.0, 8.0), np.array([0.4019, 0.4392, 78.6600])),
    (('7.5Y', 9.0, 10.0), np.array([0.4201, 0.4622, 78.6600])),
    (('7.5Y', 9.0, 12.0), np.array([0.4369, 0.4829, 78.6600])),
    (('7.5Y', 9.0, 14.0), np.array([0.4503, 0.4993, 78.6600])),
    (('7.5Y', 9.0, 16.0), np.array([0.4595, 0.5104, 78.6600])),
    (('7.5Y', 9.0, 18.0), np.array([0.4663, 0.5188, 78.6600])),
    (('7.5Y', 9.0, 20.0), np.array([0.4710, 0.5230, 78.6600])),
    (('7.5Y', 9.0, 22.0), np.array([0.4760, 0.5280, 78.6600])),
    (('10Y', 10.0, 2.0), np.array([0.3340, 0.3520, 102.5700])),
    (('10Y', 10.0, 4.0), np.array([0.3550, 0.3820, 102.5700])),
    (('10Y', 10.0, 6.0), np.array([0.3710, 0.4090, 102.5700])),
    (('10Y', 10.0, 8.0), np.array([0.3900, 0.4370, 102.5700])),
    (('10Y', 10.0, 10.0), np.array([0.4050, 0.4600, 102.5700])),
    (('10Y', 10.0, 12.0), np.array([0.4200, 0.4810, 102.5700])),
    (('10Y', 10.0, 14.0), np.array([0.4330, 0.5020, 102.5700])),
    (('10Y', 10.0, 16.0), np.array([0.4420, 0.5150, 102.5700])),
    (('10Y', 10.0, 18.0), np.array([0.4500, 0.5270, 102.5700])),
    (('10Y', 10.0, 20.0), np.array([0.4550, 0.5340, 102.5700])),
    (('10Y', 10.0, 22.0), np.array([0.4600, 0.5410, 102.5700])),
    (('2.5GY', 10.0, 2.0), np.array([0.3320, 0.3540, 102.5700])),
    (('2.5GY', 10.0, 4.0), np.array([0.3480, 0.3840, 102.5700])),
    (('2.5GY', 10.0, 6.0), np.array([0.3650, 0.4130, 102.5700])),
    (('2.5GY', 10.0, 8.0), np.array([0.3800, 0.4420, 102.5700])),
    (('2.5GY', 10.0, 10.0), np.array([0.3930, 0.4660, 102.5700])),
    (('2.5GY', 10.0, 12.0), np.array([0.4060, 0.4920, 102.5700])),
    (('2.5GY', 10.0, 14.0), np.array([0.4160, 0.5140, 102.5700])),
    (('2.5GY', 10.0, 16.0), np.array([0.4240, 0.5290, 102.5700])),
    (('2.5GY', 10.0, 18.0), np.array([0.4320, 0.5460, 102.5700])),
    (('2.5GY', 10.0, 20.0), np.array([0.4360, 0.5520, 102.5700])),
    (('2.5GY', 10.0, 22.0), np.array([0.4400, 0.5600, 102.5700])),
    (('5GY', 10.0, 2.0), np.array([0.3280, 0.3540, 102.5700])),
    (('5GY', 10.0, 4.0), np.array([0.3430, 0.3840, 102.5700])),
    (('5GY', 10.0, 6.0), np.array([0.3560, 0.4140, 102.5700])),
    (('5GY', 10.0, 8.0), np.array([0.3700, 0.4460, 102.5700])),
    (('5GY', 10.0, 10.0), np.array([0.3800, 0.4700, 102.5700])),
    (('5GY', 10.0, 12.0), np.array([0.3910, 0.4970, 102.5700])),
    (('5GY', 10.0, 14.0), np.array([0.3980, 0.5180, 102.5700])),
    (('5GY', 10.0, 16.0), np.array([0.4050, 0.5380, 102.5700])),
    (('5GY', 10.0, 18.0), np.array([0.4100, 0.5600, 102.5700])),
    (('5GY', 10.0, 20.0), np.array([0.4130, 0.5720, 102.5700])),
    (('5GY', 10.0, 22.0), np.array([0.4160, 0.5800, 102.5700])),
    (('7.5GY', 10.0, 2.0), np.array([0.3200, 0.3510, 102.5700])),
    (('7.5GY', 10.0, 4.0), np.array([0.3270, 0.3780, 102.5700])),
    (('7.5GY', 10.0, 6.0), np.array([0.3340, 0.4100, 102.5700])),
    (('7.5GY', 10.0, 8.0), np.array([0.3420, 0.4390, 102.5700])),
    (('7.5GY', 10.0, 10.0), np.array([0.3470, 0.4700, 102.5700])),
    (('7.5GY', 10.0, 12.0), np.array([0.3510, 0.4930, 102.5700])),
    (('7.5GY', 10.0, 14.0), np.array([0.3540, 0.5180, 102.5700])),
    (('7.5GY', 10.0, 16.0), np.array([0.3570, 0.5450, 102.5700])),
    (('7.5GY', 10.0, 18.0), np.array([0.3600, 0.5710, 102.5700])),
    (('7.5GY', 10.0, 20.0), np.array([0.3610, 0.5920, 102.5700])),
    (('7.5GY', 10.0, 22.0), np.array([0.3600, 0.6080, 102.5700])),
    (('10GY', 10.0, 2.0), np.array([0.3120, 0.3460, 102.5700])),
    (('10GY', 10.0, 4.0), np.array([0.3140, 0.3700, 102.5700])),
    (('10GY', 10.0, 6.0), np.array([0.3140, 0.4000, 102.5700])),
    (('10GY', 10.0, 8.0), np.array([0.3140, 0.4230, 102.5700])),
    (('10GY', 10.0, 10.0), np.array([0.3140, 0.4540, 102.5700])),
    (('10GY', 10.0, 12.0), np.array([0.3140, 0.4740, 102.5700])),
    (('10GY', 10.0, 14.0), np.array([0.3130, 0.5000, 102.5700])),
    (('10GY', 10.0, 16.0), np.array([0.3100, 0.5270, 102.5700])),
    (('10GY', 10.0, 18.0), np.array([0.3070, 0.5560, 102.5700])),
    (('10GY', 10.0, 20.0), np.array([0.3030, 0.5800, 102.5700])),
    (('10GY', 10.0, 22.0), np.array([0.2990, 0.6000, 102.5700])),
    (('2.5G', 10.0, 2.0), np.array([0.3060, 0.3420, 102.5700])),
    (('2.5G', 10.0, 4.0), np.array([0.3030, 0.3610, 102.5700])),
    (('2.5G', 10.0, 6.0), np.array([0.2980, 0.3850, 102.5700])),
    (('2.5G', 10.0, 8.0), np.array([0.2930, 0.4040, 102.5700])),
    (('2.5G', 10.0, 10.0), np.array([0.2860, 0.4270, 102.5700])),
    (('2.5G', 10.0, 12.0), np.array([0.2820, 0.4440, 102.5700])),
    (('2.5G', 10.0, 14.0), np.array([0.2760, 0.4670, 102.5700])),
    (('2.5G', 10.0, 16.0), np.array([0.2700, 0.4900, 102.5700])),
    (('2.5G', 10.0, 18.0), np.array([0.2620, 0.5170, 102.5700])),
    (('2.5G', 10.0, 20.0), np.array([0.2540, 0.5430, 102.5700])),
    (('2.5G', 10.0, 22.0), np.array([0.2480, 0.5630, 102.5700])),
    (('5G', 10.0, 2.0), np.array([0.3010, 0.3370, 102.5700])),
    (('5G', 10.0, 4.0), np.array([0.2930, 0.3530, 102.5700])),
    (('5G', 10.0, 6.0), np.array([0.2830, 0.3690, 102.5700])),
    (('5G', 10.0, 8.0), np.array([0.2740, 0.3840, 102.5700])),
    (('5G', 10.0, 10.0), np.array([0.2650, 0.3980, 102.5700])),
    (('5G', 10.0, 12.0), np.array([0.2560, 0.4120, 102.5700])),
    (('5G', 10.0, 14.0), np.array([0.2450, 0.4260, 102.5700])),
    (('5G', 10.0, 16.0), np.array([0.2350, 0.4400, 102.5700])),
    (('5G', 10.0, 18.0), np.array([0.2220, 0.4570, 102.5700])),
    (('7.5G', 10.0, 2.0), np.array([0.2980, 0.3330, 102.5700])),
    (('7.5G', 10.0, 4.0), np.array([0.2880, 0.3460, 102.5700])),
    (('7.5G', 10.0, 6.0), np.array([0.2770, 0.3600, 102.5700])),
    (('7.5G', 10.0, 8.0), np.array([0.2660, 0.3740, 102.5700])),
    (('7.5G', 10.0, 10.0), np.array([0.2560, 0.3850, 102.5700])),
    (('7.5G', 10.0, 12.0), np.array([0.2460, 0.3970, 102.5700])),
    (('7.5G', 10.0, 14.0), np.array([0.2350, 0.4090, 102.5700])),
    (('7.5G', 10.0, 16.0), np.array([0.2240, 0.4200, 102.5700])),
    (('10G', 10.0, 2.0), np.array([0.2960, 0.3310, 102.5700])),
    (('10G', 10.0, 4.0), np.array([0.2840, 0.3410, 102.5700])),
    (('10G', 10.0, 6.0), np.array([0.2710, 0.3510, 102.5700])),
    (('10G', 10.0, 8.0), np.array([0.2580, 0.3610, 102.5700])),
    (('10G', 10.0, 10.0), np.array([0.2480, 0.3690, 102.5700])),
    (('10G', 10.0, 12.0), np.array([0.2350, 0.3780, 102.5700])),
    (('10G', 10.0, 14.0), np.array([0.2230, 0.3870, 102.5700])),
    (('2.5BG', 10.0, 2.0), np.array([0.2940, 0.3270, 102.5700])),
    (('2.5BG', 10.0, 4.0), np.array([0.2800, 0.3350, 102.5700])),
    (('2.5BG', 10.0, 6.0), np.array([0.2650, 0.3430, 102.5700])),
    (('2.5BG', 10.0, 8.0), np.array([0.2510, 0.3500, 102.5700])),
    (('2.5BG', 10.0, 10.0), np.array([0.2410, 0.3570, 102.5700])),
    (('2.5BG', 10.0, 12.0), np.array([0.2270, 0.3620, 102.5700])),
    (('2.5BG', 10.0, 14.0), np.array([0.2130, 0.3680, 102.5700])),
    (('5BG', 10.0, 2.0), np.array([0.2930, 0.3230, 102.5700])),
    (('5BG', 10.0, 4.0), np.array([0.2770, 0.3300, 102.5700])),
    (('5BG', 10.0, 6.0), np.array([0.2620, 0.3350, 102.5700])),
    (('5BG', 10.0, 8.0), np.array([0.2460, 0.3400, 102.5700])),
    (('5BG', 10.0, 10.0), np.array([0.2330, 0.3420, 102.5700])),
    (('5BG', 10.0, 12.0), np.array([0.2200, 0.3450, 102.5700])),
    (('5BG', 10.0, 14.0), np.array([0.2030, 0.3480, 102.5700])),
    (('7.5BG', 10.0, 2.0), np.array([0.2930, 0.3180, 102.5700])),
    (('7.5BG', 10.0, 4.0), np.array([0.2730, 0.3200, 102.5700])),
    (('7.5BG', 10.0, 6.0), np.array([0.2560, 0.3210, 102.5700])),
    (('7.5BG', 10.0, 8.0), np.array([0.2380, 0.3220, 102.5700])),
    (('7.5BG', 10.0, 10.0), np.array([0.2240, 0.3220, 102.5700])),
    (('7.5BG', 10.0, 12.0), np.array([0.2090, 0.3220, 102.5700])),
    (('10BG', 10.0, 2.0), np.array([0.2930, 0.3150, 102.5700])),
    (('10BG', 10.0, 4.0), np.array([0.2700, 0.3130, 102.5700])),
    (('10BG', 10.0, 6.0), np.array([0.2510, 0.3110, 102.5700])),
    (('10BG', 10.0, 8.0), np.array([0.2340, 0.3100, 102.5700])),
    (('10BG', 10.0, 10.0), np.array([0.2190, 0.3080, 102.5700])),
    (('10BG', 10.0, 12.0), np.array([0.2030, 0.3050, 102.5700])),
    (('2.5B', 10.0, 2.0), np.array([0.2930, 0.3130, 102.5700])),
    (('2.5B', 10.0, 4.0), np.array([0.2700, 0.3080, 102.5700])),
    (('2.5B', 10.0, 6.0), np.array([0.2490, 0.3040, 102.5700])),
    (('2.5B', 10.0, 8.0), np.array([0.2320, 0.2990, 102.5700])),
    (('2.5B', 10.0, 10.0), np.array([0.2150, 0.2930, 102.5700])),
    (('5B', 10.0, 2.0), np.array([0.2940, 0.3120, 102.5700])),
    (('5B', 10.0, 4.0), np.array([0.2680, 0.3020, 102.5700])),
    (('5B', 10.0, 6.0), np.array([0.2480, 0.2960, 102.5700])),
    (('5B', 10.0, 8.0), np.array([0.2300, 0.2880, 102.5700])),
    (('7.5B', 10.0, 2.0), np.array([0.2950, 0.3100, 102.5700])),
    (('7.5B', 10.0, 4.0), np.array([0.2690, 0.2980, 102.5700])),
    (('7.5B', 10.0, 6.0), np.array([0.2490, 0.2870, 102.5700])),
    (('10B', 10.0, 2.0), np.array([0.2960, 0.3090, 102.5700])),
    (('10B', 10.0, 4.0), np.array([0.2720, 0.2940, 102.5700])),
    (('10B', 10.0, 6.0), np.array([0.2510, 0.2800, 102.5700])),
    (('2.5PB', 10.0, 2.0), np.array([0.2980, 0.3070, 102.5700])),
    (('2.5PB', 10.0, 4.0), np.array([0.2780, 0.2900, 102.5700])),
    (('2.5PB', 10.0, 6.0), np.array([0.2600, 0.2740, 102.5700])),
    (('5PB', 10.0, 2.0), np.array([0.2990, 0.3070, 102.5700])),
    (('5PB', 10.0, 4.0), np.array([0.2820, 0.2880, 102.5700])),
    (('5PB', 10.0, 6.0), np.array([0.2650, 0.2700, 102.5700])),
    (('7.5PB', 10.0, 2.0), np.array([0.3020, 0.3070, 102.5700])),
    (('7.5PB', 10.0, 4.0), np.array([0.2860, 0.2860, 102.5700])),
    (('7.5PB', 10.0, 6.0), np.array([0.2730, 0.2680, 102.5700])),
    (('10PB', 10.0, 2.0), np.array([0.3030, 0.3070, 102.5700])),
    (('10PB', 10.0, 4.0), np.array([0.2910, 0.2860, 102.5700])),
    (('10PB', 10.0, 6.0), np.array([0.2800, 0.2680, 102.5700])),
    (('2.5P', 10.0, 2.0), np.array([0.3050, 0.3070, 102.5700])),
    (('2.5P', 10.0, 4.0), np.array([0.2960, 0.2880, 102.5700])),
    (('2.5P', 10.0, 6.0), np.array([0.2890, 0.2700, 102.5700])),
    (('5P', 10.0, 2.0), np.array([0.3070, 0.3070, 102.5700])),
    (('5P', 10.0, 4.0), np.array([0.3020, 0.2890, 102.5700])),
    (('5P', 10.0, 6.0), np.array([0.2980, 0.2740, 102.5700])),
    (('5P', 10.0, 8.0), np.array([0.2940, 0.2580, 102.5700])),
    (('7.5P', 10.0, 2.0), np.array([0.3100, 0.3080, 102.5700])),
    (('7.5P', 10.0, 4.0), np.array([0.3110, 0.2940, 102.5700])),
    (('7.5P', 10.0, 6.0), np.array([0.3110, 0.2800, 102.5700])),
    (('7.5P', 10.0, 8.0), np.array([0.3110, 0.2660, 102.5700])),
    (('10P', 10.0, 2.0), np.array([0.3130, 0.3090, 102.5700])),
    (('10P', 10.0, 4.0), np.array([0.3170, 0.2970, 102.5700])),
    (('10P', 10.0, 6.0), np.array([0.3210, 0.2840, 102.5700])),
    (('10P', 10.0, 8.0), np.array([0.3260, 0.2730, 102.5700])),
    (('10P', 10.0, 10.0), np.array([0.3280, 0.2640, 102.5700])),
    (('2.5RP', 10.0, 2.0), np.array([0.3150, 0.3110, 102.5700])),
    (('2.5RP', 10.0, 4.0), np.array([0.3230, 0.3010, 102.5700])),
    (('2.5RP', 10.0, 6.0), np.array([0.3320, 0.2910, 102.5700])),
    (('2.5RP', 10.0, 8.0), np.array([0.3390, 0.2820, 102.5700])),
    (('2.5RP', 10.0, 10.0), np.array([0.3440, 0.2750, 102.5700])),
    (('5RP', 10.0, 2.0), np.array([0.3180, 0.3130, 102.5700])),
    (('5RP', 10.0, 4.0), np.array([0.3320, 0.3070, 102.5700])),
    (('5RP', 10.0, 6.0), np.array([0.3450, 0.3000, 102.5700])),
    (('5RP', 10.0, 8.0), np.array([0.3550, 0.2930, 102.5700])),
    (('5RP', 10.0, 10.0), np.array([0.3630, 0.2880, 102.5700])),
    (('7.5RP', 10.0, 2.0), np.array([0.3200, 0.3140, 102.5700])),
    (('7.5RP', 10.0, 4.0), np.array([0.3350, 0.3100, 102.5700])),
    (('7.5RP', 10.0, 6.0), np.array([0.3510, 0.3050, 102.5700])),
    (('7.5RP', 10.0, 8.0), np.array([0.3650, 0.3010, 102.5700])),
    (('10RP', 10.0, 2.0), np.array([0.3220, 0.3160, 102.5700])),
    (('10RP', 10.0, 4.0), np.array([0.3420, 0.3150, 102.5700])),
    (('10RP', 10.0, 6.0), np.array([0.3610, 0.3140, 102.5700])),
    (('10RP', 10.0, 8.0), np.array([0.3780, 0.3110, 102.5700])),
    (('2.5R', 10.0, 2.0), np.array([0.3240, 0.3180, 102.5700])),
    (('2.5R', 10.0, 4.0), np.array([0.3450, 0.3180, 102.5700])),
    (('2.5R', 10.0, 6.0), np.array([0.3660, 0.3180, 102.5700])),
    (('2.5R', 10.0, 8.0), np.array([0.3850, 0.3180, 102.5700])),
    (('5R', 10.0, 2.0), np.array([0.3260, 0.3200, 102.5700])),
    (('5R', 10.0, 4.0), np.array([0.3480, 0.3220, 102.5700])),
    (('5R', 10.0, 6.0), np.array([0.3720, 0.3250, 102.5700])),
    (('5R', 10.0, 8.0), np.array([0.3920, 0.3260, 102.5700])),
    (('7.5R', 10.0, 2.0), np.array([0.3290, 0.3220, 102.5700])),
    (('7.5R', 10.0, 4.0), np.array([0.3540, 0.3290, 102.5700])),
    (('7.5R', 10.0, 6.0), np.array([0.3800, 0.3340, 102.5700])),
    (('7.5R', 10.0, 8.0), np.array([0.4020, 0.3370, 102.5700])),
    (('10R', 10.0, 2.0), np.array([0.3320, 0.3260, 102.5700])),
    (('10R', 10.0, 4.0), np.array([0.3590, 0.3360, 102.5700])),
    (('10R', 10.0, 6.0), np.array([0.3860, 0.3440, 102.5700])),
    (('10R', 10.0, 8.0), np.array([0.4110, 0.3490, 102.5700])),
    (('2.5YR', 10.0, 2.0), np.array([0.3340, 0.3290, 102.5700])),
    (('2.5YR', 10.0, 4.0), np.array([0.3620, 0.3420, 102.5700])),
    (('2.5YR', 10.0, 6.0), np.array([0.3890, 0.3540, 102.5700])),
    (('2.5YR', 10.0, 8.0), np.array([0.4150, 0.3630, 102.5700])),
    (('5YR', 10.0, 2.0), np.array([0.3360, 0.3330, 102.5700])),
    (('5YR', 10.0, 4.0), np.array([0.3640, 0.3500, 102.5700])),
    (('5YR', 10.0, 6.0), np.array([0.3910, 0.3650, 102.5700])),
    (('5YR', 10.0, 8.0), np.array([0.4160, 0.3780, 102.5700])),
    (('5YR', 10.0, 10.0), np.array([0.4400, 0.3880, 102.5700])),
    (('7.5YR', 10.0, 2.0), np.array([0.3370, 0.3380, 102.5700])),
    (('7.5YR', 10.0, 4.0), np.array([0.3660, 0.3570, 102.5700])),
    (('7.5YR', 10.0, 6.0), np.array([0.3910, 0.3750, 102.5700])),
    (('7.5YR', 10.0, 8.0), np.array([0.4150, 0.3910, 102.5700])),
    (('7.5YR', 10.0, 10.0), np.array([0.4390, 0.4050, 102.5700])),
    (('7.5YR', 10.0, 12.0), np.array([0.4550, 0.4160, 102.5700])),
    (('10YR', 10.0, 2.0), np.array([0.3380, 0.3430, 102.5700])),
    (('10YR', 10.0, 4.0), np.array([0.3660, 0.3670, 102.5700])),
    (('10YR', 10.0, 6.0), np.array([0.3890, 0.3850, 102.5700])),
    (('10YR', 10.0, 8.0), np.array([0.4120, 0.4020, 102.5700])),
    (('10YR', 10.0, 10.0), np.array([0.4350, 0.4200, 102.5700])),
    (('10YR', 10.0, 12.0), np.array([0.4520, 0.4310, 102.5700])),
    (('10YR', 10.0, 14.0), np.array([0.4670, 0.4420, 102.5700])),
    (('10YR', 10.0, 16.0), np.array([0.4830, 0.4530, 102.5700])),
    (('2.5Y', 10.0, 2.0), np.array([0.3370, 0.3470, 102.5700])),
    (('2.5Y', 10.0, 4.0), np.array([0.3630, 0.3730, 102.5700])),
    (('2.5Y', 10.0, 6.0), np.array([0.3850, 0.3950, 102.5700])),
    (('2.5Y', 10.0, 8.0), np.array([0.4070, 0.4130, 102.5700])),
    (('2.5Y', 10.0, 10.0), np.array([0.4280, 0.4310, 102.5700])),
    (('2.5Y', 10.0, 12.0), np.array([0.4460, 0.4460, 102.5700])),
    (('2.5Y', 10.0, 14.0), np.array([0.4630, 0.4600, 102.5700])),
    (('2.5Y', 10.0, 16.0), np.array([0.4760, 0.4700, 102.5700])),
    (('2.5Y', 10.0, 18.0), np.array([0.4870, 0.4770, 102.5700])),
    (('2.5Y', 10.0, 20.0), np.array([0.4960, 0.4860, 102.5700])),
    (('2.5Y', 10.0, 22.0), np.array([0.5030, 0.4900, 102.5700])),
    (('5Y', 10.0, 2.0), np.array([0.3360, 0.3480, 102.5700])),
    (('5Y', 10.0, 4.0), np.array([0.3600, 0.3770, 102.5700])),
    (('5Y', 10.0, 6.0), np.array([0.3810, 0.4020, 102.5700])),
    (('5Y', 10.0, 8.0), np.array([0.4000, 0.4250, 102.5700])),
    (('5Y', 10.0, 10.0), np.array([0.4180, 0.4460, 102.5700])),
    (('5Y', 10.0, 12.0), np.array([0.4350, 0.4650, 102.5700])),
    (('5Y', 10.0, 14.0), np.array([0.4500, 0.4810, 102.5700])),
    (('5Y', 10.0, 16.0), np.array([0.4620, 0.4940, 102.5700])),
    (('5Y', 10.0, 18.0), np.array([0.4720, 0.5020, 102.5700])),
    (('5Y', 10.0, 20.0), np.array([0.4780, 0.5080, 102.5700])),
    (('5Y', 10.0, 22.0), np.array([0.4840, 0.5140, 102.5700])),
    (('7.5Y', 10.0, 2.0), np.array([0.3350, 0.3510, 102.5700])),
    (('7.5Y', 10.0, 4.0), np.array([0.3570, 0.3800, 102.5700])),
    (('7.5Y', 10.0, 6.0), np.array([0.3760, 0.4070, 102.5700])),
    (('7.5Y', 10.0, 8.0), np.array([0.3950, 0.4320, 102.5700])),
    (('7.5Y', 10.0, 10.0), np.array([0.4120, 0.4540, 102.5700])),
    (('7.5Y', 10.0, 12.0), np.array([0.4270, 0.4730, 102.5700])),
    (('7.5Y', 10.0, 14.0), np.array([0.4420, 0.4920, 102.5700])),
    (('7.5Y', 10.0, 16.0), np.array([0.4530, 0.5050, 102.5700])),
    (('7.5Y', 10.0, 18.0), np.array([0.4620, 0.5150, 102.5700])),
    (('7.5Y', 10.0, 20.0), np.array([0.4670, 0.5210, 102.5700])),
    (('7.5Y', 10.0, 22.0), np.array([0.4720, 0.5280, 102.5700])))
# yapf: enable
"""
*All* published *Munsell* colours, including the extrapolated colors.

References
----------
:cite:`MunsellColorSciencec`

MUNSELL_COLOURS_ALL : tuple
   (('hue', 'value', 'chroma'), np.array(['x', 'y', 'Y']))
"""
