# -*- coding: utf-8 -*-
"""
Colour Matching Functions Spectral Distributions
================================================

Defines various colour matching functions involved in spectral computations.

The colour matching functions data is in the form of a *dict* of
:class:`colour.colorimetry.MultiSpectralDistribution` classes as follows::

    {'name': MultiSpectralDistribution,
    ...,
    'name': MultiSpectralDistribution}

The following colour matching functions are available:

-   Stockman & Sharpe 2 Degree Cone Fundamentals
-   Stockman & Sharpe 10 Degree Cone Fundamentals
-   Smith & Pokorny 1975 Normal Trichromats
-   Wright & Guild 1931 2 Degree RGB CMFs
-   Stiles & Burch 1955 2 Degree RGB CMFs
-   Stiles & Burch 1959 10 Degree RGB CMFs
-   CIE 1931 2 Degree Standard Observer
-   CIE 1964 10 Degree Standard Observer
-   CIE 2012 10 Degree Standard Observer
-   CIE 2012 2 Degree Standard Observer

See Also
--------
`Colour Matching Functions Jupyter Notebook
<http://nbviewer.jupyter.org/github/colour-science/colour-notebooks/\
blob/master/notebooks/colorimetry/cmfs.ipynb>`_

Notes
-----
-   *Stockman & Sharpe* *LMS* colour matching functions are provided at 1 nm
    interval.
-   *Smith & Pokorny* normal trichromats colour matching functions are provided
    at 5 nm steps.
-   *Wright & Guild* and *Stiles & Burch* *CIE RGB* colour matching functions
    are provided at 5 nm interval.
-   *CIE* Standard Observers *XYZ* colour matching functions are provided at 5
    nm interval.

References
----------
-   :cite:`Broadbent2009a` : Broadbent, A. D. (2009). Calculation from the
    original experimental data of the CIE 1931 RGB standard observer spectral
    chromaticity co-ordinates and color matching functions. Retrieved June 12,
    2014, from http://www.cis.rit.edu/mcsl/research/1931.php
-   :cite:`CVRLr` : CVRL. (n.d.). New CIE XYZ functions transformed from the
    CIE (2006) LMS functions. Retrieved February 24, 2014, from
    http://cvrl.ioo.ucl.ac.uk/ciexyzpr.htm
-   :cite:`CVRLs` : CVRL. (n.d.). Older CIE Standards. Retrieved February 24,
    2014, from http://cvrl.ioo.ucl.ac.uk/cie.htm
-   :cite:`CVRLt` : CVRL. (n.d.). Stiles & Burch individual 10-deg colour
    matching data. Retrieved February 24, 2014, from
    http://www.cvrl.org/stilesburch10_ind.htm
-   :cite:`CVRLu` : Stockman, A., & Sharpe, L. T. (2000). Cone Fundamentals.
    Retrieved June 23, 2014, from http://www.cvrl.org/cones.htm
-   :cite:`CVRLw` : CVRL. (n.d.). Stiles & Burch individual 2-deg colour
    matching data. Retrieved February 24, 2014, from
    http://www.cvrl.org/stilesburch2_ind.htm
-   :cite:`Machado2010a` : Machado, G. M. (2010). A model for simulation of
    color vision deficiency and a color contrast enhancement technique for
    dichromats. Retrieved from http://www.lume.ufrgs.br/handle/10183/26950
"""

from __future__ import division, unicode_literals

from colour.colorimetry import (LMS_ConeFundamentals,
                                RGB_ColourMatchingFunctions,
                                XYZ_ColourMatchingFunctions)
from colour.utilities import CaseInsensitiveMapping

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2019 - Colour Developers'
__license__ = 'New BSD License - https://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = [
    'LMS_CMFS_DATA', 'LMS_CMFS', 'RGB_CMFS_DATA', 'RGB_CMFS',
    'STANDARD_OBSERVERS_CMFS_DATA', 'STANDARD_OBSERVERS_CMFS', 'CMFS'
]

# *S-cone* spectral sensitivity data wasn't measurable after 615 nm and has
# been set to zero.
LMS_CMFS_DATA = {
    'Stockman & Sharpe 2 Degree Cone Fundamentals': {
        390: (4.15003e-04, 3.68349e-04, 9.54729e-03),
        391: (5.02650e-04, 4.48015e-04, 1.14794e-02),
        392: (6.07367e-04, 5.43965e-04, 1.37986e-02),
        393: (7.31850e-04, 6.58983e-04, 1.65746e-02),
        394: (8.79012e-04, 7.96121e-04, 1.98869e-02),
        395: (1.05192e-03, 9.58658e-04, 2.38250e-02),
        396: (1.25373e-03, 1.15002e-03, 2.84877e-02),
        397: (1.48756e-03, 1.37367e-03, 3.39832e-02),
        398: (1.75633e-03, 1.63296e-03, 4.04274e-02),
        399: (2.06261e-03, 1.93089e-03, 4.79417e-02),
        400: (2.40836e-03, 2.26991e-03, 5.66498e-02),
        401: (2.79522e-03, 2.65210e-03, 6.66757e-02),
        402: (3.22640e-03, 3.08110e-03, 7.81479e-02),
        403: (3.70617e-03, 3.56156e-03, 9.11925e-02),
        404: (4.23972e-03, 4.09900e-03, 1.05926e-01),
        405: (4.83339e-03, 4.70010e-03, 1.22451e-01),
        406: (5.49335e-03, 5.37186e-03, 1.40844e-01),
        407: (6.21933e-03, 6.11757e-03, 1.61140e-01),
        408: (7.00631e-03, 6.93795e-03, 1.83325e-01),
        409: (7.84503e-03, 7.83144e-03, 2.07327e-01),
        410: (8.72127e-03, 8.79369e-03, 2.33008e-01),
        411: (9.61879e-03, 9.81865e-03, 2.60183e-01),
        412: (1.05324e-02, 1.09044e-02, 2.88723e-01),
        413: (1.14620e-02, 1.20509e-02, 3.18512e-01),
        414: (1.24105e-02, 1.32582e-02, 3.49431e-01),
        415: (1.33837e-02, 1.45277e-02, 3.81363e-01),
        416: (1.43870e-02, 1.58600e-02, 4.14141e-01),
        417: (1.54116e-02, 1.72496e-02, 4.47350e-01),
        418: (1.64424e-02, 1.86878e-02, 4.80439e-01),
        419: (1.74614e-02, 2.01638e-02, 5.12767e-01),
        420: (1.84480e-02, 2.16649e-02, 5.43618e-01),
        421: (1.93852e-02, 2.31801e-02, 5.72399e-01),
        422: (2.02811e-02, 2.47139e-02, 5.99284e-01),
        423: (2.11545e-02, 2.62785e-02, 6.24786e-01),
        424: (2.20286e-02, 2.78905e-02, 6.49576e-01),
        425: (2.29317e-02, 2.95714e-02, 6.74474e-01),
        426: (2.38896e-02, 3.13438e-02, 7.00186e-01),
        427: (2.49026e-02, 3.32151e-02, 7.26460e-01),
        428: (2.59631e-02, 3.51889e-02, 7.52726e-01),
        429: (2.70619e-02, 3.72683e-02, 7.78333e-01),
        430: (2.81877e-02, 3.94566e-02, 8.02555e-01),
        431: (2.93303e-02, 4.17546e-02, 8.24818e-01),
        432: (3.04898e-02, 4.41544e-02, 8.45422e-01),
        433: (3.16694e-02, 4.66432e-02, 8.64961e-01),
        434: (3.28731e-02, 4.92050e-02, 8.84100e-01),
        435: (3.41054e-02, 5.18199e-02, 9.03573e-01),
        436: (3.53670e-02, 5.44645e-02, 9.23844e-01),
        437: (3.66400e-02, 5.71131e-02, 9.44055e-01),
        438: (3.78988e-02, 5.97369e-02, 9.62920e-01),
        439: (3.91148e-02, 6.23038e-02, 9.79057e-01),
        440: (4.02563e-02, 6.47782e-02, 9.91020e-01),
        441: (4.12989e-02, 6.71329e-02, 9.97765e-01),
        442: (4.22582e-02, 6.93844e-02, 9.99982e-01),
        443: (4.31627e-02, 7.15658e-02, 9.98861e-01),
        444: (4.40444e-02, 7.37164e-02, 9.95628e-01),
        445: (4.49380e-02, 7.58812e-02, 9.91515e-01),
        446: (4.58729e-02, 7.80980e-02, 9.87377e-01),
        447: (4.68459e-02, 8.03538e-02, 9.82619e-01),
        448: (4.78446e-02, 8.26198e-02, 9.76301e-01),
        449: (4.88555e-02, 8.48644e-02, 9.67513e-01),
        450: (4.98639e-02, 8.70524e-02, 9.55393e-01),
        451: (5.08618e-02, 8.91640e-02, 9.39499e-01),
        452: (5.18731e-02, 9.12523e-02, 9.20807e-01),
        453: (5.29317e-02, 9.33950e-02, 9.00592e-01),
        454: (5.40746e-02, 9.56775e-02, 8.80040e-01),
        455: (5.53418e-02, 9.81934e-02, 8.60240e-01),
        456: (5.67734e-02, 1.01031e-01, 8.42031e-01),
        457: (5.83973e-02, 1.04229e-01, 8.25572e-01),
        458: (6.02409e-02, 1.07814e-01, 8.10860e-01),
        459: (6.23353e-02, 1.11817e-01, 7.97902e-01),
        460: (6.47164e-02, 1.16272e-01, 7.86704e-01),
        461: (6.74131e-02, 1.21204e-01, 7.77119e-01),
        462: (7.04040e-02, 1.26573e-01, 7.68365e-01),
        463: (7.36489e-02, 1.32311e-01, 7.59538e-01),
        464: (7.70978e-02, 1.38334e-01, 7.49777e-01),
        465: (8.06894e-02, 1.44541e-01, 7.38268e-01),
        466: (8.43613e-02, 1.50828e-01, 7.24389e-01),
        467: (8.80904e-02, 1.57146e-01, 7.08113e-01),
        468: (9.18630e-02, 1.63457e-01, 6.89572e-01),
        469: (9.56637e-02, 1.69721e-01, 6.68927e-01),
        470: (9.94755e-02, 1.75893e-01, 6.46359e-01),
        471: (1.03286e-01, 1.81938e-01, 6.22112e-01),
        472: (1.07103e-01, 1.87872e-01, 5.96591e-01),
        473: (1.10947e-01, 1.93731e-01, 5.70216e-01),
        474: (1.14838e-01, 1.99557e-01, 5.43373e-01),
        475: (1.18802e-01, 2.05398e-01, 5.16411e-01),
        476: (1.22863e-01, 2.11302e-01, 4.89647e-01),
        477: (1.27026e-01, 2.17283e-01, 4.63406e-01),
        478: (1.31293e-01, 2.23347e-01, 4.37965e-01),
        479: (1.35666e-01, 2.29501e-01, 4.13549e-01),
        480: (1.40145e-01, 2.35754e-01, 3.90333e-01),
        481: (1.44731e-01, 2.42108e-01, 3.68394e-01),
        482: (1.49415e-01, 2.48545e-01, 3.47582e-01),
        483: (1.54189e-01, 2.55037e-01, 3.27724e-01),
        484: (1.59038e-01, 2.61554e-01, 3.08676e-01),
        485: (1.63952e-01, 2.68063e-01, 2.90322e-01),
        486: (1.68932e-01, 2.74561e-01, 2.72626e-01),
        487: (1.74063e-01, 2.81180e-01, 2.55772e-01),
        488: (1.79457e-01, 2.88097e-01, 2.39945e-01),
        489: (1.85241e-01, 2.95508e-01, 2.25280e-01),
        490: (1.91556e-01, 3.03630e-01, 2.11867e-01),
        491: (1.98532e-01, 3.12649e-01, 1.99719e-01),
        492: (2.06182e-01, 3.22565e-01, 1.88674e-01),
        493: (2.14485e-01, 3.33319e-01, 1.78557e-01),
        494: (2.23411e-01, 3.44844e-01, 1.69217e-01),
        495: (2.32926e-01, 3.57061e-01, 1.60526e-01),
        496: (2.42992e-01, 3.69895e-01, 1.52368e-01),
        497: (2.53616e-01, 3.83355e-01, 1.44620e-01),
        498: (2.64810e-01, 3.97467e-01, 1.37173e-01),
        499: (2.76587e-01, 4.12260e-01, 1.29937e-01),
        500: (2.88959e-01, 4.27764e-01, 1.22839e-01),
        501: (3.01934e-01, 4.44001e-01, 1.15834e-01),
        502: (3.15508e-01, 4.60947e-01, 1.08922e-01),
        503: (3.29673e-01, 4.78562e-01, 1.02118e-01),
        504: (3.44416e-01, 4.96795e-01, 9.54374e-02),
        505: (3.59716e-01, 5.15587e-01, 8.88965e-02),
        506: (3.75550e-01, 5.34868e-01, 8.25343e-02),
        507: (3.91895e-01, 5.54575e-01, 7.64613e-02),
        508: (4.08722e-01, 5.74640e-01, 7.07769e-02),
        509: (4.25998e-01, 5.94984e-01, 6.55491e-02),
        510: (4.43683e-01, 6.15520e-01, 6.08210e-02),
        511: (4.61731e-01, 6.36162e-01, 5.65919e-02),
        512: (4.80094e-01, 6.56872e-01, 5.27659e-02),
        513: (4.98717e-01, 6.77624e-01, 4.92440e-02),
        514: (5.17539e-01, 6.98393e-01, 4.59470e-02),
        515: (5.36494e-01, 7.19154e-01, 4.28123e-02),
        516: (5.55493e-01, 7.39844e-01, 3.98014e-02),
        517: (5.74386e-01, 7.60235e-01, 3.69227e-02),
        518: (5.92995e-01, 7.80039e-01, 3.41909e-02),
        519: (6.11124e-01, 7.98941e-01, 3.16158e-02),
        520: (6.28561e-01, 8.16610e-01, 2.92033e-02),
        521: (6.45139e-01, 8.32783e-01, 2.69536e-02),
        522: (6.60907e-01, 8.47550e-01, 2.48575e-02),
        523: (6.75993e-01, 8.61113e-01, 2.29047e-02),
        524: (6.90542e-01, 8.73698e-01, 2.10855e-02),
        525: (7.04720e-01, 8.85550e-01, 1.93912e-02),
        526: (7.18662e-01, 8.96873e-01, 1.78145e-02),
        527: (7.32323e-01, 9.07638e-01, 1.63514e-02),
        528: (7.45606e-01, 9.17757e-01, 1.49980e-02),
        529: (7.58410e-01, 9.27137e-01, 1.37497e-02),
        530: (7.70630e-01, 9.35687e-01, 1.26013e-02),
        531: (7.82208e-01, 9.43362e-01, 1.15466e-02),
        532: (7.93290e-01, 9.50309e-01, 1.05766e-02),
        533: (8.04084e-01, 9.56733e-01, 9.68268e-03),
        534: (8.14813e-01, 9.62844e-01, 8.85746e-03),
        535: (8.25711e-01, 9.68858e-01, 8.09453e-03),
        536: (8.36943e-01, 9.74919e-01, 7.38890e-03),
        537: (8.48349e-01, 9.80850e-01, 6.73799e-03),
        538: (8.59676e-01, 9.86388e-01, 6.13947e-03),
        539: (8.70656e-01, 9.91267e-01, 5.59073e-03),
        540: (8.81011e-01, 9.95217e-01, 5.08900e-03),
        541: (8.90496e-01, 9.98007e-01, 4.63120e-03),
        542: (8.99052e-01, 9.99595e-01, 4.21366e-03),
        543: (9.06669e-01, 9.99982e-01, 3.83286e-03),
        544: (9.13341e-01, 9.99177e-01, 3.48559e-03),
        545: (9.19067e-01, 9.97193e-01, 3.16893e-03),
        546: (9.23898e-01, 9.94102e-01, 2.88021e-03),
        547: (9.28099e-01, 9.90204e-01, 2.61698e-03),
        548: (9.31995e-01, 9.85855e-01, 2.37701e-03),
        549: (9.35916e-01, 9.81404e-01, 2.15829e-03),
        550: (9.40198e-01, 9.77193e-01, 1.95896e-03),
        551: (9.45076e-01, 9.73441e-01, 1.77737e-03),
        552: (9.50367e-01, 9.69881e-01, 1.61218e-03),
        553: (9.55775e-01, 9.66133e-01, 1.46215e-03),
        554: (9.61000e-01, 9.61823e-01, 1.32606e-03),
        555: (9.65733e-01, 9.56583e-01, 1.20277e-03),
        556: (9.69744e-01, 9.50167e-01, 1.09118e-03),
        557: (9.73133e-01, 9.42773e-01, 9.90135e-04),
        558: (9.76086e-01, 9.34709e-01, 8.98564e-04),
        559: (9.78793e-01, 9.26271e-01, 8.15524e-04),
        560: (9.81445e-01, 9.17750e-01, 7.40174e-04),
        561: (9.84186e-01, 9.09339e-01, 6.71772e-04),
        562: (9.86965e-01, 9.00895e-01, 6.09693e-04),
        563: (9.89678e-01, 8.92197e-01, 5.53372e-04),
        564: (9.92220e-01, 8.83035e-01, 5.02292e-04),
        565: (9.94486e-01, 8.73205e-01, 4.55979e-04),
        566: (9.96386e-01, 8.62565e-01, 4.13997e-04),
        567: (9.97895e-01, 8.51173e-01, 3.75940e-04),
        568: (9.99004e-01, 8.39132e-01, 3.41439e-04),
        569: (9.99706e-01, 8.26545e-01, 3.10160e-04),
        570: (9.99993e-01, 8.13509e-01, 2.81800e-04),
        571: (9.99837e-01, 8.00082e-01, 2.56084e-04),
        572: (9.99123e-01, 7.86166e-01, 2.32764e-04),
        573: (9.97719e-01, 7.71635e-01, 2.11616e-04),
        574: (9.95491e-01, 7.56376e-01, 1.92436e-04),
        575: (9.92310e-01, 7.40291e-01, 1.75039e-04),
        576: (9.88146e-01, 7.23369e-01, 1.59257e-04),
        577: (9.83345e-01, 7.05890e-01, 1.44940e-04),
        578: (9.78342e-01, 6.88184e-01, 1.31948e-04),
        579: (9.73564e-01, 6.70554e-01, 1.20157e-04),
        580: (9.69429e-01, 6.53274e-01, 1.09454e-04),
        581: (9.66211e-01, 6.36524e-01, 9.97367e-05),
        582: (9.63630e-01, 6.20217e-01, 9.09125e-05),
        583: (9.61273e-01, 6.04211e-01, 8.28976e-05),
        584: (9.58732e-01, 5.88375e-01, 7.56160e-05),
        585: (9.55602e-01, 5.72597e-01, 6.89991e-05),
        586: (9.51569e-01, 5.56783e-01, 6.29846e-05),
        587: (9.46654e-01, 5.40896e-01, 5.75163e-05),
        588: (9.40962e-01, 5.24912e-01, 5.25432e-05),
        589: (9.34600e-01, 5.08817e-01, 4.80192e-05),
        590: (9.27673e-01, 4.92599e-01, 4.39024e-05),
        591: (9.20264e-01, 4.76268e-01, 4.01551e-05),
        592: (9.12391e-01, 4.59893e-01, 3.67431e-05),
        593: (9.04050e-01, 4.43551e-01, 3.36353e-05),
        594: (8.95243e-01, 4.27314e-01, 3.08037e-05),
        595: (8.85969e-01, 4.11246e-01, 2.82228e-05),
        596: (8.76242e-01, 3.95396e-01, 2.58697e-05),
        597: (8.66117e-01, 3.79782e-01, 2.37235e-05),
        598: (8.55658e-01, 3.64410e-01, 2.17653e-05),
        599: (8.44926e-01, 3.49289e-01, 1.99779e-05),
        600: (8.33982e-01, 3.34429e-01, 1.83459e-05),
        601: (8.22859e-01, 3.19843e-01, 1.68551e-05),
        602: (8.11491e-01, 3.05564e-01, 1.54929e-05),
        603: (7.99794e-01, 2.91625e-01, 1.42476e-05),
        604: (7.87689e-01, 2.78053e-01, 1.31087e-05),
        605: (7.75103e-01, 2.64872e-01, 1.20667e-05),
        606: (7.61996e-01, 2.52099e-01, 1.11131e-05),
        607: (7.48425e-01, 2.39747e-01, 1.02399e-05),
        608: (7.34470e-01, 2.27822e-01, 9.43999e-06),
        609: (7.20208e-01, 2.16330e-01, 8.70695e-06),
        610: (7.05713e-01, 2.05273e-01, 8.03488e-06),
        611: (6.91044e-01, 1.94650e-01, 7.41844e-06),
        612: (6.76212e-01, 1.84448e-01, 6.85279e-06),
        613: (6.61220e-01, 1.74654e-01, 6.33352e-06),
        614: (6.46072e-01, 1.65256e-01, 5.85662e-06),
        615: (6.30773e-01, 1.56243e-01, 5.41843e-06),
        616: (6.15349e-01, 1.47602e-01, 0.000000000),
        617: (5.99888e-01, 1.39329e-01, 0.000000000),
        618: (5.84489e-01, 1.31416e-01, 0.000000000),
        619: (5.69240e-01, 1.23856e-01, 0.000000000),
        620: (5.54224e-01, 1.16641e-01, 0.000000000),
        621: (5.39469e-01, 1.09766e-01, 0.000000000),
        622: (5.24827e-01, 1.03226e-01, 0.000000000),
        623: (5.10124e-01, 9.70205e-02, 0.000000000),
        624: (4.95206e-01, 9.11430e-02, 0.000000000),
        625: (4.79941e-01, 8.55872e-02, 0.000000000),
        626: (4.64270e-01, 8.03428e-02, 0.000000000),
        627: (4.48338e-01, 7.53913e-02, 0.000000000),
        628: (4.32329e-01, 7.07136e-02, 0.000000000),
        629: (4.16406e-01, 6.62924e-02, 0.000000000),
        630: (4.00711e-01, 6.21120e-02, 0.000000000),
        631: (3.85355e-01, 5.81595e-02, 0.000000000),
        632: (3.70377e-01, 5.44276e-02, 0.000000000),
        633: (3.55793e-01, 5.09098e-02, 0.000000000),
        634: (3.41618e-01, 4.75991e-02, 0.000000000),
        635: (3.27864e-01, 4.44879e-02, 0.000000000),
        636: (3.14541e-01, 4.15659e-02, 0.000000000),
        637: (3.01662e-01, 3.88152e-02, 0.000000000),
        638: (2.89239e-01, 3.62181e-02, 0.000000000),
        639: (2.77278e-01, 3.37599e-02, 0.000000000),
        640: (2.65784e-01, 3.14282e-02, 0.000000000),
        641: (2.54740e-01, 2.92176e-02, 0.000000000),
        642: (2.44054e-01, 2.71403e-02, 0.000000000),
        643: (2.33634e-01, 2.52084e-02, 0.000000000),
        644: (2.23399e-01, 2.34286e-02, 0.000000000),
        645: (2.13284e-01, 2.18037e-02, 0.000000000),
        646: (2.03257e-01, 2.03285e-02, 0.000000000),
        647: (1.93370e-01, 1.89779e-02, 0.000000000),
        648: (1.83688e-01, 1.77272e-02, 0.000000000),
        649: (1.74263e-01, 1.65560e-02, 0.000000000),
        650: (1.65141e-01, 1.54480e-02, 0.000000000),
        651: (1.56354e-01, 1.43924e-02, 0.000000000),
        652: (1.47916e-01, 1.33896e-02, 0.000000000),
        653: (1.39834e-01, 1.24414e-02, 0.000000000),
        654: (1.32111e-01, 1.15488e-02, 0.000000000),
        655: (1.24749e-01, 1.07120e-02, 0.000000000),
        656: (1.17744e-01, 9.92994e-03, 0.000000000),
        657: (1.11081e-01, 9.20057e-03, 0.000000000),
        658: (1.04747e-01, 8.52126e-03, 0.000000000),
        659: (9.87277e-02, 7.88945e-03, 0.000000000),
        660: (9.30085e-02, 7.30255e-03, 0.000000000),
        661: (8.75769e-02, 6.75820e-03, 0.000000000),
        662: (8.24219e-02, 6.25474e-03, 0.000000000),
        663: (7.75328e-02, 5.79045e-03, 0.000000000),
        664: (7.28989e-02, 5.36347e-03, 0.000000000),
        665: (6.85100e-02, 4.97179e-03, 0.000000000),
        666: (6.43554e-02, 4.61300e-03, 0.000000000),
        667: (6.04242e-02, 4.28339e-03, 0.000000000),
        668: (5.67057e-02, 3.97940e-03, 0.000000000),
        669: (5.31896e-02, 3.69803e-03, 0.000000000),
        670: (4.98661e-02, 3.43667e-03, 0.000000000),
        671: (4.67258e-02, 3.19326e-03, 0.000000000),
        672: (4.37598e-02, 2.96653e-03, 0.000000000),
        673: (4.09594e-02, 2.75543e-03, 0.000000000),
        674: (3.83165e-02, 2.55896e-03, 0.000000000),
        675: (3.58233e-02, 2.37617e-03, 0.000000000),
        676: (3.34725e-02, 2.20618e-03, 0.000000000),
        677: (3.12583e-02, 2.04809e-03, 0.000000000),
        678: (2.91751e-02, 1.90110e-03, 0.000000000),
        679: (2.72173e-02, 1.76442e-03, 0.000000000),
        680: (2.53790e-02, 1.63734e-03, 0.000000000),
        681: (2.36541e-02, 1.51916e-03, 0.000000000),
        682: (2.20336e-02, 1.40913e-03, 0.000000000),
        683: (2.05092e-02, 1.30655e-03, 0.000000000),
        684: (1.90735e-02, 1.21078e-03, 0.000000000),
        685: (1.77201e-02, 1.12128e-03, 0.000000000),
        686: (1.64451e-02, 1.03766e-03, 0.000000000),
        687: (1.52510e-02, 9.59892e-04, 0.000000000),
        688: (1.41404e-02, 8.87951e-04, 0.000000000),
        689: (1.31137e-02, 8.21729e-04, 0.000000000),
        690: (1.21701e-02, 7.61051e-04, 0.000000000),
        691: (1.13063e-02, 7.05623e-04, 0.000000000),
        692: (1.05131e-02, 6.54874e-04, 0.000000000),
        693: (9.78127e-03, 6.08240e-04, 0.000000000),
        694: (9.10300e-03, 5.65240e-04, 0.000000000),
        695: (8.47170e-03, 5.25457e-04, 0.000000000),
        696: (7.88227e-03, 4.88550e-04, 0.000000000),
        697: (7.33213e-03, 4.54277e-04, 0.000000000),
        698: (6.81928e-03, 4.22436e-04, 0.000000000),
        699: (6.34173e-03, 3.92839e-04, 0.000000000),
        700: (5.89749e-03, 3.65317e-04, 0.000000000),
        701: (5.48444e-03, 3.39710e-04, 0.000000000),
        702: (5.09978e-03, 3.15856e-04, 0.000000000),
        703: (4.74086e-03, 2.93607e-04, 0.000000000),
        704: (4.40538e-03, 2.72834e-04, 0.000000000),
        705: (4.09129e-03, 2.53417e-04, 0.000000000),
        706: (3.79703e-03, 2.35265e-04, 0.000000000),
        707: (3.52187e-03, 2.18331e-04, 0.000000000),
        708: (3.26520e-03, 2.02574e-04, 0.000000000),
        709: (3.02632e-03, 1.87948e-04, 0.000000000),
        710: (2.80447e-03, 1.74402e-04, 0.000000000),
        711: (2.59882e-03, 1.61878e-04, 0.000000000),
        712: (2.40849e-03, 1.50305e-04, 0.000000000),
        713: (2.23259e-03, 1.39612e-04, 0.000000000),
        714: (2.07024e-03, 1.29734e-04, 0.000000000),
        715: (1.92058e-03, 1.20608e-04, 0.000000000),
        716: (1.78269e-03, 1.12176e-04, 0.000000000),
        717: (1.65540e-03, 1.04373e-04, 0.000000000),
        718: (1.53762e-03, 9.71383e-05, 0.000000000),
        719: (1.42839e-03, 9.04202e-05, 0.000000000),
        720: (1.32687e-03, 8.41716e-05, 0.000000000),
        721: (1.23238e-03, 7.83538e-05, 0.000000000),
        722: (1.14456e-03, 7.29425e-05, 0.000000000),
        723: (1.06308e-03, 6.79164e-05, 0.000000000),
        724: (9.87592e-04, 6.32542e-05, 0.000000000),
        725: (9.17777e-04, 5.89349e-05, 0.000000000),
        726: (8.53264e-04, 5.49360e-05, 0.000000000),
        727: (7.93589e-04, 5.12291e-05, 0.000000000),
        728: (7.38306e-04, 4.77872e-05, 0.000000000),
        729: (6.87018e-04, 4.45862e-05, 0.000000000),
        730: (6.39373e-04, 4.16049e-05, 0.000000000),
        731: (5.95057e-04, 3.88245e-05, 0.000000000),
        732: (5.53797e-04, 3.62302e-05, 0.000000000),
        733: (5.15350e-04, 3.38086e-05, 0.000000000),
        734: (4.79496e-04, 3.15474e-05, 0.000000000),
        735: (4.46035e-04, 2.94354e-05, 0.000000000),
        736: (4.14809e-04, 2.74634e-05, 0.000000000),
        737: (3.85749e-04, 2.56268e-05, 0.000000000),
        738: (3.58792e-04, 2.39217e-05, 0.000000000),
        739: (3.33861e-04, 2.23432e-05, 0.000000000),
        740: (3.10869e-04, 2.08860e-05, 0.000000000),
        741: (2.89699e-04, 1.95424e-05, 0.000000000),
        742: (2.70145e-04, 1.82990e-05, 0.000000000),
        743: (2.52007e-04, 1.71423e-05, 0.000000000),
        744: (2.35117e-04, 1.60611e-05, 0.000000000),
        745: (2.19329e-04, 1.50458e-05, 0.000000000),
        746: (2.04535e-04, 1.40893e-05, 0.000000000),
        747: (1.90692e-04, 1.31899e-05, 0.000000000),
        748: (1.77771e-04, 1.23462e-05, 0.000000000),
        749: (1.65736e-04, 1.15569e-05, 0.000000000),
        750: (1.54549e-04, 1.08200e-05, 0.000000000),
        751: (1.44167e-04, 1.01334e-05, 0.000000000),
        752: (1.34528e-04, 9.49367e-06, 0.000000000),
        753: (1.25574e-04, 8.89736e-06, 0.000000000),
        754: (1.17251e-04, 8.34135e-06, 0.000000000),
        755: (1.09508e-04, 7.82271e-06, 0.000000000),
        756: (1.02300e-04, 7.33865e-06, 0.000000000),
        757: (9.55828e-05, 6.88612e-06, 0.000000000),
        758: (8.93161e-05, 6.46228e-06, 0.000000000),
        759: (8.34631e-05, 6.06462e-06, 0.000000000),
        760: (7.79912e-05, 5.69093e-06, 0.000000000),
        761: (7.28730e-05, 5.33942e-06, 0.000000000),
        762: (6.80921e-05, 5.00929e-06, 0.000000000),
        763: (6.36342e-05, 4.69984e-06, 0.000000000),
        764: (5.94841e-05, 4.41034e-06, 0.000000000),
        765: (5.56264e-05, 4.13998e-06, 0.000000000),
        766: (5.20430e-05, 3.88772e-06, 0.000000000),
        767: (4.87063e-05, 3.65186e-06, 0.000000000),
        768: (4.55900e-05, 3.43070e-06, 0.000000000),
        769: (4.26710e-05, 3.22278e-06, 0.000000000),
        770: (3.99295e-05, 3.02683e-06, 0.000000000),
        771: (3.73509e-05, 2.84192e-06, 0.000000000),
        772: (3.49326e-05, 2.66795e-06, 0.000000000),
        773: (3.26730e-05, 2.50491e-06, 0.000000000),
        774: (3.05690e-05, 2.35267e-06, 0.000000000),
        775: (2.86163e-05, 2.21100e-06, 0.000000000),
        776: (2.68077e-05, 2.07946e-06, 0.000000000),
        777: (2.51286e-05, 1.95700e-06, 0.000000000),
        778: (2.35645e-05, 1.84258e-06, 0.000000000),
        779: (2.21026e-05, 1.73528e-06, 0.000000000),
        780: (2.07321e-05, 1.63433e-06, 0.000000000),
        781: (1.94444e-05, 1.53910e-06, 0.000000000),
        782: (1.82351e-05, 1.44932e-06, 0.000000000),
        783: (1.71008e-05, 1.36478e-06, 0.000000000),
        784: (1.60380e-05, 1.28526e-06, 0.000000000),
        785: (1.50432e-05, 1.21054e-06, 0.000000000),
        786: (1.41127e-05, 1.14038e-06, 0.000000000),
        787: (1.32420e-05, 1.07446e-06, 0.000000000),
        788: (1.24263e-05, 1.01247e-06, 0.000000000),
        789: (1.16618e-05, 9.54130e-07, 0.000000000),
        790: (1.09446e-05, 8.99170e-07, 0.000000000),
        791: (1.02716e-05, 8.47379e-07, 0.000000000),
        792: (9.64036e-06, 7.98635e-07, 0.000000000),
        793: (9.04897e-06, 7.52832e-07, 0.000000000),
        794: (8.49535e-06, 7.09857e-07, 0.000000000),
        795: (7.97750e-06, 6.69594e-07, 0.000000000),
        796: (7.49341e-06, 6.31907e-07, 0.000000000),
        797: (7.04079e-06, 5.96604e-07, 0.000000000),
        798: (6.61744e-06, 5.63495e-07, 0.000000000),
        799: (6.22133e-06, 5.32408e-07, 0.000000000),
        800: (5.85057e-06, 5.03187e-07, 0.000000000),
        801: (5.50333e-06, 4.75686e-07, 0.000000000),
        802: (5.17756e-06, 4.49753e-07, 0.000000000),
        803: (4.87135e-06, 4.25249e-07, 0.000000000),
        804: (4.58300e-06, 4.02050e-07, 0.000000000),
        805: (4.31102e-06, 3.80046e-07, 0.000000000),
        806: (4.05422e-06, 3.59156e-07, 0.000000000),
        807: (3.81213e-06, 3.39356e-07, 0.000000000),
        808: (3.58438e-06, 3.20633e-07, 0.000000000),
        809: (3.37053e-06, 3.02966e-07, 0.000000000),
        810: (3.17009e-06, 2.86329e-07, 0.000000000),
        811: (2.98248e-06, 2.70687e-07, 0.000000000),
        812: (2.80691e-06, 2.55980e-07, 0.000000000),
        813: (2.64257e-06, 2.42147e-07, 0.000000000),
        814: (2.48873e-06, 2.29130e-07, 0.000000000),
        815: (2.34468e-06, 2.16878e-07, 0.000000000),
        816: (2.20974e-06, 2.05338e-07, 0.000000000),
        817: (2.08315e-06, 1.94449e-07, 0.000000000),
        818: (1.96419e-06, 1.84155e-07, 0.000000000),
        819: (1.85222e-06, 1.74407e-07, 0.000000000),
        820: (1.74666e-06, 1.65158e-07, 0.000000000),
        821: (1.64705e-06, 1.56373e-07, 0.000000000),
        822: (1.55307e-06, 1.48031e-07, 0.000000000),
        823: (1.46448e-06, 1.40117e-07, 0.000000000),
        824: (1.38100e-06, 1.32615e-07, 0.000000000),
        825: (1.30241e-06, 1.25508e-07, 0.000000000),
        826: (1.22844e-06, 1.18781e-07, 0.000000000),
        827: (1.15888e-06, 1.12416e-07, 0.000000000),
        828: (1.09348e-06, 1.06398e-07, 0.000000000),
        829: (1.03203e-06, 1.00711e-07, 0.000000000),
        830: (9.74306e-07, 9.53411e-08, 0.000000000)
    },
    'Stockman & Sharpe 10 Degree Cone Fundamentals': {
        390: (4.07619e-04, 3.58227e-04, 6.14265e-03),
        391: (4.97068e-04, 4.38660e-04, 7.44280e-03),
        392: (6.04713e-04, 5.36230e-04, 9.01661e-03),
        393: (7.33640e-04, 6.54061e-04, 1.09170e-02),
        394: (8.87247e-04, 7.95649e-04, 1.32053e-02),
        395: (1.06921e-03, 9.64828e-04, 1.59515e-02),
        396: (1.28340e-03, 1.16572e-03, 1.92347e-02),
        397: (1.53382e-03, 1.40263e-03, 2.31436e-02),
        398: (1.82443e-03, 1.67992e-03, 2.77750e-02),
        399: (2.15896e-03, 2.00180e-03, 3.32339e-02),
        400: (2.54073e-03, 2.37208e-03, 3.96308e-02),
        401: (2.97282e-03, 2.79433e-03, 4.70801e-02),
        402: (3.45993e-03, 3.27374e-03, 5.57012e-02),
        403: (4.00793e-03, 3.81660e-03, 6.56137e-02),
        404: (4.62370e-03, 4.43021e-03, 7.69323e-02),
        405: (5.31546e-03, 5.12316e-03, 8.97612e-02),
        406: (6.09138e-03, 5.90458e-03, 1.04188e-01),
        407: (6.95291e-03, 6.78005e-03, 1.20273e-01),
        408: (7.89634e-03, 7.75260e-03, 1.38044e-01),
        409: (8.91300e-03, 8.82286e-03, 1.57485e-01),
        410: (9.98835e-03, 9.98841e-03, 1.78530e-01),
        411: (1.11054e-02, 1.12452e-02, 2.01077e-01),
        412: (1.22607e-02, 1.25949e-02, 2.25091e-01),
        413: (1.34578e-02, 1.40425e-02, 2.50566e-01),
        414: (1.47044e-02, 1.55944e-02, 2.77507e-01),
        415: (1.60130e-02, 1.72596e-02, 3.05941e-01),
        416: (1.73958e-02, 1.90467e-02, 3.35858e-01),
        417: (1.88452e-02, 2.09545e-02, 3.66981e-01),
        418: (2.03440e-02, 2.29763e-02, 3.98876e-01),
        419: (2.18700e-02, 2.51017e-02, 4.30998e-01),
        420: (2.33957e-02, 2.73163e-02, 4.62692e-01),
        421: (2.48961e-02, 2.96062e-02, 4.93357e-01),
        422: (2.63761e-02, 3.19746e-02, 5.23006e-01),
        423: (2.78541e-02, 3.44330e-02, 5.51939e-01),
        424: (2.93551e-02, 3.69983e-02, 5.80599e-01),
        425: (3.09104e-02, 3.96928e-02, 6.09570e-01),
        426: (3.25498e-02, 4.25402e-02, 6.39359e-01),
        427: (3.42714e-02, 4.55474e-02, 6.69651e-01),
        428: (3.60620e-02, 4.87161e-02, 6.99829e-01),
        429: (3.79052e-02, 5.20467e-02, 7.29177e-01),
        430: (3.97810e-02, 5.55384e-02, 7.56885e-01),
        431: (4.16705e-02, 5.91876e-02, 7.82292e-01),
        432: (4.35728e-02, 6.29819e-02, 8.05666e-01),
        433: (4.54932e-02, 6.69031e-02, 8.27599e-01),
        434: (4.74385e-02, 7.09285e-02, 8.48780e-01),
        435: (4.94172e-02, 7.50299e-02, 8.69984e-01),
        436: (5.14343e-02, 7.91769e-02, 8.91761e-01),
        437: (5.34735e-02, 8.33463e-02, 9.13444e-01),
        438: (5.55102e-02, 8.75160e-02, 9.33977e-01),
        439: (5.75166e-02, 9.16625e-02, 9.52218e-01),
        440: (5.94619e-02, 9.57612e-02, 9.66960e-01),
        441: (6.13240e-02, 9.97977e-02, 9.77340e-01),
        442: (6.31288e-02, 1.03804e-01, 9.84028e-01),
        443: (6.49189e-02, 1.07834e-01, 9.88144e-01),
        444: (6.67425e-02, 1.11948e-01, 9.90851e-01),
        445: (6.86538e-02, 1.16220e-01, 9.93336e-01),
        446: (7.06963e-02, 1.20706e-01, 9.96373e-01),
        447: (7.28508e-02, 1.25363e-01, 9.99038e-01),
        448: (7.50778e-02, 1.30111e-01, 9.99978e-01),
        449: (7.73325e-02, 1.34856e-01, 9.97844e-01),
        450: (7.95647e-02, 1.39493e-01, 9.91329e-01),
        451: (8.17368e-02, 1.43943e-01, 9.79657e-01),
        452: (8.38826e-02, 1.48281e-01, 9.63905e-01),
        453: (8.60598e-02, 1.52637e-01, 9.45573e-01),
        454: (8.83322e-02, 1.57157e-01, 9.26078e-01),
        455: (9.07704e-02, 1.62006e-01, 9.06735e-01),
        456: (9.34397e-02, 1.67331e-01, 8.88509e-01),
        457: (9.63577e-02, 1.73144e-01, 8.71353e-01),
        458: (9.95304e-02, 1.79417e-01, 8.54998e-01),
        459: (1.02964e-01, 1.86117e-01, 8.39198e-01),
        460: (1.06663e-01, 1.93202e-01, 8.23726e-01),
        461: (1.10629e-01, 2.00621e-01, 8.08307e-01),
        462: (1.14828e-01, 2.08315e-01, 7.92431e-01),
        463: (1.19217e-01, 2.16212e-01, 7.75569e-01),
        464: (1.23740e-01, 2.24231e-01, 7.57244e-01),
        465: (1.28336e-01, 2.32275e-01, 7.37043e-01),
        466: (1.32947e-01, 2.40257e-01, 7.14730e-01),
        467: (1.37570e-01, 2.48160e-01, 6.90558e-01),
        468: (1.42218e-01, 2.55987e-01, 6.64890e-01),
        469: (1.46905e-01, 2.63744e-01, 6.38078e-01),
        470: (1.51651e-01, 2.71441e-01, 6.10456e-01),
        471: (1.56475e-01, 2.79097e-01, 5.82346e-01),
        472: (1.61404e-01, 2.86755e-01, 5.54065e-01),
        473: (1.66464e-01, 2.94476e-01, 5.25903e-01),
        474: (1.71690e-01, 3.02323e-01, 4.98108e-01),
        475: (1.77116e-01, 3.10372e-01, 4.70894e-01),
        476: (1.82777e-01, 3.18692e-01, 4.44450e-01),
        477: (1.88685e-01, 3.27307e-01, 4.18992e-01),
        478: (1.94845e-01, 3.36232e-01, 3.94699e-01),
        479: (2.01261e-01, 3.45480e-01, 3.71707e-01),
        480: (2.07940e-01, 3.55066e-01, 3.50108e-01),
        481: (2.14875e-01, 3.64985e-01, 3.29904e-01),
        482: (2.22022e-01, 3.75142e-01, 3.10864e-01),
        483: (2.29316e-01, 3.85410e-01, 2.92741e-01),
        484: (2.36684e-01, 3.95646e-01, 2.75338e-01),
        485: (2.44046e-01, 4.05688e-01, 2.58497e-01),
        486: (2.51347e-01, 4.15435e-01, 2.42158e-01),
        487: (2.58695e-01, 4.25064e-01, 2.26504e-01),
        488: (2.66252e-01, 4.34851e-01, 2.11726e-01),
        489: (2.74202e-01, 4.45097e-01, 1.97960e-01),
        490: (2.82752e-01, 4.56137e-01, 1.85297e-01),
        491: (2.92071e-01, 4.68236e-01, 1.73751e-01),
        492: (3.02094e-01, 4.81255e-01, 1.63149e-01),
        493: (3.12673e-01, 4.94927e-01, 1.53311e-01),
        494: (3.23637e-01, 5.08947e-01, 1.44086e-01),
        495: (3.34786e-01, 5.22970e-01, 1.35351e-01),
        496: (3.45943e-01, 5.36701e-01, 1.27011e-01),
        497: (3.57126e-01, 5.50195e-01, 1.19017e-01),
        498: (3.68416e-01, 5.63616e-01, 1.11333e-01),
        499: (3.79907e-01, 5.77150e-01, 1.03934e-01),
        500: (3.91705e-01, 5.91003e-01, 9.67990e-02),
        501: (4.03905e-01, 6.05350e-01, 8.99169e-02),
        502: (4.16500e-01, 6.20160e-01, 8.32878e-02),
        503: (4.29453e-01, 6.35343e-01, 7.69157e-02),
        504: (4.42720e-01, 6.50796e-01, 7.08052e-02),
        505: (4.56252e-01, 6.66404e-01, 6.49614e-02),
        506: (4.69997e-01, 6.82055e-01, 5.94049e-02),
        507: (4.83926e-01, 6.97672e-01, 5.42076e-02),
        508: (4.98012e-01, 7.13186e-01, 4.94281e-02),
        509: (5.12227e-01, 7.28526e-01, 4.50993e-02),
        510: (5.26538e-01, 7.43612e-01, 4.12337e-02),
        511: (5.40923e-01, 7.58396e-01, 3.78142e-02),
        512: (5.55406e-01, 7.72966e-01, 3.47627e-02),
        513: (5.70025e-01, 7.87457e-01, 3.20029e-02),
        514: (5.84828e-01, 8.02017e-01, 2.94746e-02),
        515: (5.99867e-01, 8.16808e-01, 2.71300e-02),
        516: (6.15162e-01, 8.31919e-01, 2.49376e-02),
        517: (6.30569e-01, 8.47096e-01, 2.28931e-02),
        518: (6.45883e-01, 8.61972e-01, 2.09956e-02),
        519: (6.60880e-01, 8.76152e-01, 1.92427e-02),
        520: (6.75313e-01, 8.89214e-01, 1.76298e-02),
        521: (6.88975e-01, 9.00812e-01, 1.61501e-02),
        522: (7.01886e-01, 9.11006e-01, 1.47911e-02),
        523: (7.14140e-01, 9.19968e-01, 1.35407e-02),
        524: (7.25841e-01, 9.27890e-01, 1.23884e-02),
        525: (7.37108e-01, 9.34977e-01, 1.13252e-02),
        526: (7.48046e-01, 9.41414e-01, 1.03437e-02),
        527: (7.58684e-01, 9.47279e-01, 9.44092e-03),
        528: (7.69032e-01, 9.52622e-01, 8.61365e-03),
        529: (7.79099e-01, 9.57498e-01, 7.85831e-03),
        530: (7.88900e-01, 9.61962e-01, 7.17089e-03),
        531: (7.98472e-01, 9.66080e-01, 6.54648e-03),
        532: (8.07945e-01, 9.69966e-01, 5.97777e-03),
        533: (8.17478e-01, 9.73743e-01, 5.45794e-03),
        534: (8.27239e-01, 9.77539e-01, 4.98125e-03),
        535: (8.37403e-01, 9.81481e-01, 4.54287e-03),
        536: (8.48078e-01, 9.85629e-01, 4.13908e-03),
        537: (8.59064e-01, 9.89754e-01, 3.76794e-03),
        538: (8.70068e-01, 9.93551e-01, 3.42783e-03),
        539: (8.80779e-01, 9.96714e-01, 3.11696e-03),
        540: (8.90871e-01, 9.98931e-01, 2.83352e-03),
        541: (9.00057e-01, 9.99942e-01, 2.57556e-03),
        542: (9.08253e-01, 9.99692e-01, 2.34084e-03),
        543: (9.15433e-01, 9.98178e-01, 2.12722e-03),
        544: (9.21575e-01, 9.95405e-01, 1.93276e-03),
        545: (9.26660e-01, 9.91383e-01, 1.75573e-03),
        546: (9.30744e-01, 9.86199e-01, 1.59455e-03),
        547: (9.34160e-01, 9.80229e-01, 1.44783e-03),
        548: (9.37318e-01, 9.73906e-01, 1.31430e-03),
        549: (9.40634e-01, 9.67653e-01, 1.19281e-03),
        550: (9.44527e-01, 9.61876e-01, 1.08230e-03),
        551: (9.49291e-01, 9.56820e-01, 9.81819e-04),
        552: (9.54680e-01, 9.52151e-01, 8.90533e-04),
        553: (9.60309e-01, 9.47398e-01, 8.07687e-04),
        554: (9.65785e-01, 9.42105e-01, 7.32568e-04),
        555: (9.70703e-01, 9.35829e-01, 6.64512e-04),
        556: (9.74756e-01, 9.28274e-01, 6.02887e-04),
        557: (9.78055e-01, 9.19667e-01, 5.47062e-04),
        558: (9.80819e-01, 9.10356e-01, 4.96461e-04),
        559: (9.83271e-01, 9.00677e-01, 4.50571e-04),
        560: (9.85636e-01, 8.90949e-01, 4.08931e-04),
        561: (9.88085e-01, 8.81386e-01, 3.71136e-04),
        562: (9.90560e-01, 8.71834e-01, 3.36838e-04),
        563: (9.92945e-01, 8.62059e-01, 3.05723e-04),
        564: (9.95124e-01, 8.51840e-01, 2.77504e-04),
        565: (9.96979e-01, 8.40969e-01, 2.51918e-04),
        566: (9.98411e-01, 8.29303e-01, 2.28725e-04),
        567: (9.99391e-01, 8.16911e-01, 2.07699e-04),
        568: (9.99912e-01, 8.03910e-01, 1.88638e-04),
        569: (9.99965e-01, 7.90413e-01, 1.71357e-04),
        570: (9.99543e-01, 7.76526e-01, 1.55688e-04),
        571: (9.98615e-01, 7.62311e-01, 1.41480e-04),
        572: (9.97051e-01, 7.47669e-01, 1.28596e-04),
        573: (9.94701e-01, 7.32476e-01, 1.16912e-04),
        574: (9.91416e-01, 7.16622e-01, 1.06316e-04),
        575: (9.87057e-01, 7.00013e-01, 9.67045e-05),
        576: (9.81599e-01, 6.82647e-01, 8.79858e-05),
        577: (9.75451e-01, 6.64817e-01, 8.00756e-05),
        578: (9.69120e-01, 6.46858e-01, 7.28979e-05),
        579: (9.63093e-01, 6.29072e-01, 6.63837e-05),
        580: (9.57841e-01, 6.11728e-01, 6.04705e-05),
        581: (9.53664e-01, 5.94998e-01, 5.51020e-05),
        582: (9.50236e-01, 5.78783e-01, 5.02269e-05),
        583: (9.47086e-01, 5.62935e-01, 4.57988e-05),
        584: (9.43752e-01, 5.47321e-01, 4.17759e-05),
        585: (9.39781e-01, 5.31825e-01, 3.81202e-05),
        586: (9.34826e-01, 5.16354e-01, 3.47974e-05),
        587: (9.28917e-01, 5.00870e-01, 3.17763e-05),
        588: (9.22177e-01, 4.85350e-01, 2.90288e-05),
        589: (9.14729e-01, 4.69777e-01, 2.65294e-05),
        590: (9.06693e-01, 4.54142e-01, 2.42549e-05),
        591: (8.98170e-01, 4.38454e-01, 2.21847e-05),
        592: (8.89188e-01, 4.22778e-01, 2.02996e-05),
        593: (8.79759e-01, 4.07188e-01, 1.85826e-05),
        594: (8.69894e-01, 3.91752e-01, 1.70182e-05),
        595: (8.59605e-01, 3.76527e-01, 1.55924e-05),
        596: (8.48912e-01, 3.61559e-01, 1.42924e-05),
        597: (8.37865e-01, 3.46856e-01, 1.31066e-05),
        598: (8.26522e-01, 3.32422e-01, 1.20248e-05),
        599: (8.14940e-01, 3.18261e-01, 1.10373e-05),
        600: (8.03173e-01, 3.04378e-01, 1.01356e-05),
        601: (7.91253e-01, 2.90784e-01, 9.31202e-06),
        602: (7.79118e-01, 2.77506e-01, 8.55941e-06),
        603: (7.66691e-01, 2.64575e-01, 7.87141e-06),
        604: (7.53900e-01, 2.52012e-01, 7.24221e-06),
        605: (7.40680e-01, 2.39837e-01, 6.66657e-06),
        606: (7.26995e-01, 2.28065e-01, 6.13970e-06),
        607: (7.12905e-01, 2.16703e-01, 5.65727e-06),
        608: (6.98491e-01, 2.05754e-01, 5.21535e-06),
        609: (6.83829e-01, 1.95221e-01, 4.81036e-06),
        610: (6.68991e-01, 1.85104e-01, 4.43906e-06),
        611: (6.54037e-01, 1.75398e-01, 4.09850e-06),
        612: (6.38979e-01, 1.66091e-01, 3.78599e-06),
        613: (6.23825e-01, 1.57169e-01, 3.49911e-06),
        614: (6.08579e-01, 1.48620e-01, 3.23563e-06),
        615: (5.93248e-01, 1.40431e-01, 2.99354e-06),
        616: (5.77857e-01, 1.32591e-01, 0.000000000),
        617: (5.62493e-01, 1.25092e-01, 0.000000000),
        618: (5.47248e-01, 1.17928e-01, 0.000000000),
        619: (5.32209e-01, 1.11091e-01, 0.000000000),
        620: (5.17449e-01, 1.04573e-01, 0.000000000),
        621: (5.02993e-01, 9.83663e-02, 0.000000000),
        622: (4.88692e-01, 9.24685e-02, 0.000000000),
        623: (4.74376e-01, 8.68759e-02, 0.000000000),
        624: (4.59896e-01, 8.15834e-02, 0.000000000),
        625: (4.45125e-01, 7.65841e-02, 0.000000000),
        626: (4.30007e-01, 7.18683e-02, 0.000000000),
        627: (4.14687e-01, 6.74186e-02, 0.000000000),
        628: (3.99340e-01, 6.32176e-02, 0.000000000),
        629: (3.84122e-01, 5.92492e-02, 0.000000000),
        630: (3.69168e-01, 5.54990e-02, 0.000000000),
        631: (3.54580e-01, 5.19550e-02, 0.000000000),
        632: (3.40389e-01, 4.86103e-02, 0.000000000),
        633: (3.26609e-01, 4.54591e-02, 0.000000000),
        634: (3.13249e-01, 4.24945e-02, 0.000000000),
        635: (3.00316e-01, 3.97097e-02, 0.000000000),
        636: (2.87817e-01, 3.70952e-02, 0.000000000),
        637: (2.75762e-01, 3.46347e-02, 0.000000000),
        638: (2.64158e-01, 3.23125e-02, 0.000000000),
        639: (2.53009e-01, 3.01151e-02, 0.000000000),
        640: (2.42316e-01, 2.80314e-02, 0.000000000),
        641: (2.32061e-01, 2.60564e-02, 0.000000000),
        642: (2.22158e-01, 2.42011e-02, 0.000000000),
        643: (2.12516e-01, 2.24760e-02, 0.000000000),
        644: (2.03060e-01, 2.08870e-02, 0.000000000),
        645: (1.93730e-01, 1.94366e-02, 0.000000000),
        646: (1.84495e-01, 1.81200e-02, 0.000000000),
        647: (1.75402e-01, 1.69149e-02, 0.000000000),
        648: (1.66509e-01, 1.57991e-02, 0.000000000),
        649: (1.57865e-01, 1.47543e-02, 0.000000000),
        650: (1.49509e-01, 1.37660e-02, 0.000000000),
        651: (1.41470e-01, 1.28246e-02, 0.000000000),
        652: (1.33760e-01, 1.19304e-02, 0.000000000),
        653: (1.26383e-01, 1.10850e-02, 0.000000000),
        654: (1.19343e-01, 1.02892e-02, 0.000000000),
        655: (1.12638e-01, 9.54315e-03, 0.000000000),
        656: (1.06264e-01, 8.84609e-03, 0.000000000),
        657: (1.00208e-01, 8.19600e-03, 0.000000000),
        658: (9.44558e-02, 7.59059e-03, 0.000000000),
        659: (8.89934e-02, 7.02754e-03, 0.000000000),
        660: (8.38077e-02, 6.50455e-03, 0.000000000),
        661: (7.88865e-02, 6.01952e-03, 0.000000000),
        662: (7.42191e-02, 5.57093e-03, 0.000000000),
        663: (6.97952e-02, 5.15728e-03, 0.000000000),
        664: (6.56050e-02, 4.77687e-03, 0.000000000),
        665: (6.16384e-02, 4.42794e-03, 0.000000000),
        666: (5.78857e-02, 4.10832e-03, 0.000000000),
        667: (5.43366e-02, 3.81470e-03, 0.000000000),
        668: (5.09811e-02, 3.54392e-03, 0.000000000),
        669: (4.78096e-02, 3.29329e-03, 0.000000000),
        670: (4.48132e-02, 3.06050e-03, 0.000000000),
        671: (4.19831e-02, 2.84369e-03, 0.000000000),
        672: (3.93111e-02, 2.64175e-03, 0.000000000),
        673: (3.67892e-02, 2.45374e-03, 0.000000000),
        674: (3.44098e-02, 2.27875e-03, 0.000000000),
        675: (3.21660e-02, 2.11596e-03, 0.000000000),
        676: (3.00509e-02, 1.96456e-03, 0.000000000),
        677: (2.80594e-02, 1.82378e-03, 0.000000000),
        678: (2.61861e-02, 1.69287e-03, 0.000000000),
        679: (2.44260e-02, 1.57115e-03, 0.000000000),
        680: (2.27738e-02, 1.45798e-03, 0.000000000),
        681: (2.12238e-02, 1.35274e-03, 0.000000000),
        682: (1.97679e-02, 1.25476e-03, 0.000000000),
        683: (1.83986e-02, 1.16340e-03, 0.000000000),
        684: (1.71092e-02, 1.07812e-03, 0.000000000),
        685: (1.58939e-02, 9.98424e-04, 0.000000000),
        686: (1.47492e-02, 9.23962e-04, 0.000000000),
        687: (1.36773e-02, 8.54713e-04, 0.000000000),
        688: (1.26804e-02, 7.90652e-04, 0.000000000),
        689: (1.17590e-02, 7.31684e-04, 0.000000000),
        690: (1.09123e-02, 6.77653e-04, 0.000000000),
        691: (1.01373e-02, 6.28297e-04, 0.000000000),
        692: (9.42568e-03, 5.83108e-04, 0.000000000),
        693: (8.76917e-03, 5.41584e-04, 0.000000000),
        694: (8.16076e-03, 5.03294e-04, 0.000000000),
        695: (7.59453e-03, 4.67870e-04, 0.000000000),
        696: (7.06588e-03, 4.35007e-04, 0.000000000),
        697: (6.57252e-03, 4.04490e-04, 0.000000000),
        698: (6.11262e-03, 3.76138e-04, 0.000000000),
        699: (5.68440e-03, 3.49784e-04, 0.000000000),
        700: (5.28607e-03, 3.25278e-04, 0.000000000),
        701: (4.91573e-03, 3.02477e-04, 0.000000000),
        702: (4.57086e-03, 2.81237e-04, 0.000000000),
        703: (4.24908e-03, 2.61427e-04, 0.000000000),
        704: (3.94832e-03, 2.42930e-04, 0.000000000),
        705: (3.66675e-03, 2.25641e-04, 0.000000000),
        706: (3.40297e-03, 2.09478e-04, 0.000000000),
        707: (3.15632e-03, 1.94400e-04, 0.000000000),
        708: (2.92624e-03, 1.80370e-04, 0.000000000),
        709: (2.71213e-03, 1.67347e-04, 0.000000000),
        710: (2.51327e-03, 1.55286e-04, 0.000000000),
        711: (2.32895e-03, 1.44135e-04, 0.000000000),
        712: (2.15836e-03, 1.33830e-04, 0.000000000),
        713: (2.00071e-03, 1.24309e-04, 0.000000000),
        714: (1.85521e-03, 1.15513e-04, 0.000000000),
        715: (1.72108e-03, 1.07388e-04, 0.000000000),
        716: (1.59750e-03, 9.98800e-05, 0.000000000),
        717: (1.48342e-03, 9.29320e-05, 0.000000000),
        718: (1.37787e-03, 8.64907e-05, 0.000000000),
        719: (1.27998e-03, 8.05090e-05, 0.000000000),
        720: (1.18900e-03, 7.49453e-05, 0.000000000),
        721: (1.10433e-03, 6.97652e-05, 0.000000000),
        722: (1.02563e-03, 6.49470e-05, 0.000000000),
        723: (9.52602e-04, 6.04718e-05, 0.000000000),
        724: (8.84959e-04, 5.63207e-05, 0.000000000),
        725: (8.22396e-04, 5.24748e-05, 0.000000000),
        726: (7.64585e-04, 4.89142e-05, 0.000000000),
        727: (7.11110e-04, 4.56137e-05, 0.000000000),
        728: (6.61570e-04, 4.25490e-05, 0.000000000),
        729: (6.15612e-04, 3.96989e-05, 0.000000000),
        730: (5.72917e-04, 3.70443e-05, 0.000000000),
        731: (5.33206e-04, 3.45688e-05, 0.000000000),
        732: (4.96234e-04, 3.22588e-05, 0.000000000),
        733: (4.61782e-04, 3.01026e-05, 0.000000000),
        734: (4.29654e-04, 2.80893e-05, 0.000000000),
        735: (3.99670e-04, 2.62088e-05, 0.000000000),
        736: (3.71690e-04, 2.44529e-05, 0.000000000),
        737: (3.45650e-04, 2.28177e-05, 0.000000000),
        738: (3.21494e-04, 2.12995e-05, 0.000000000),
        739: (2.99155e-04, 1.98941e-05, 0.000000000),
        740: (2.78553e-04, 1.85965e-05, 0.000000000),
        741: (2.59583e-04, 1.74003e-05, 0.000000000),
        742: (2.42061e-04, 1.62931e-05, 0.000000000),
        743: (2.25809e-04, 1.52632e-05, 0.000000000),
        744: (2.10674e-04, 1.43005e-05, 0.000000000),
        745: (1.96528e-04, 1.33965e-05, 0.000000000),
        746: (1.83271e-04, 1.25449e-05, 0.000000000),
        747: (1.70868e-04, 1.17441e-05, 0.000000000),
        748: (1.59290e-04, 1.09929e-05, 0.000000000),
        749: (1.48506e-04, 1.02900e-05, 0.000000000),
        750: (1.38482e-04, 9.63397e-06, 0.000000000),
        751: (1.29179e-04, 9.02263e-06, 0.000000000),
        752: (1.20542e-04, 8.45301e-06, 0.000000000),
        753: (1.12519e-04, 7.92207e-06, 0.000000000),
        754: (1.05061e-04, 7.42700e-06, 0.000000000),
        755: (9.81226e-05, 6.96522e-06, 0.000000000),
        756: (9.16641e-05, 6.53422e-06, 0.000000000),
        757: (8.56455e-05, 6.13129e-06, 0.000000000),
        758: (8.00302e-05, 5.75391e-06, 0.000000000),
        759: (7.47858e-05, 5.39984e-06, 0.000000000),
        760: (6.98827e-05, 5.06711e-06, 0.000000000),
        761: (6.52966e-05, 4.75413e-06, 0.000000000),
        762: (6.10128e-05, 4.46019e-06, 0.000000000),
        763: (5.70183e-05, 4.18466e-06, 0.000000000),
        764: (5.32996e-05, 3.92690e-06, 0.000000000),
        765: (4.98430e-05, 3.68617e-06, 0.000000000),
        766: (4.66322e-05, 3.46157e-06, 0.000000000),
        767: (4.36424e-05, 3.25155e-06, 0.000000000),
        768: (4.08501e-05, 3.05464e-06, 0.000000000),
        769: (3.82346e-05, 2.86951e-06, 0.000000000),
        770: (3.57781e-05, 2.69504e-06, 0.000000000),
        771: (3.34676e-05, 2.53040e-06, 0.000000000),
        772: (3.13007e-05, 2.37550e-06, 0.000000000),
        773: (2.92761e-05, 2.23033e-06, 0.000000000),
        774: (2.73908e-05, 2.09478e-06, 0.000000000),
        775: (2.56411e-05, 1.96864e-06, 0.000000000),
        776: (2.40205e-05, 1.85152e-06, 0.000000000),
        777: (2.25160e-05, 1.74248e-06, 0.000000000),
        778: (2.11145e-05, 1.64060e-06, 0.000000000),
        779: (1.98046e-05, 1.54507e-06, 0.000000000),
        780: (1.85766e-05, 1.45518e-06, 0.000000000),
        781: (1.74228e-05, 1.37039e-06, 0.000000000),
        782: (1.63392e-05, 1.29045e-06, 0.000000000),
        783: (1.53228e-05, 1.21518e-06, 0.000000000),
        784: (1.43705e-05, 1.14437e-06, 0.000000000),
        785: (1.34792e-05, 1.07784e-06, 0.000000000),
        786: (1.26454e-05, 1.01538e-06, 0.000000000),
        787: (1.18652e-05, 9.56684e-07, 0.000000000),
        788: (1.11344e-05, 9.01491e-07, 0.000000000),
        789: (1.04493e-05, 8.49542e-07, 0.000000000),
        790: (9.80671e-06, 8.00606e-07, 0.000000000),
        791: (9.20363e-06, 7.54492e-07, 0.000000000),
        792: (8.63806e-06, 7.11091e-07, 0.000000000),
        793: (8.10815e-06, 6.70309e-07, 0.000000000),
        794: (7.61209e-06, 6.32045e-07, 0.000000000),
        795: (7.14808e-06, 5.96195e-07, 0.000000000),
        796: (6.71432e-06, 5.62639e-07, 0.000000000),
        797: (6.30876e-06, 5.31206e-07, 0.000000000),
        798: (5.92942e-06, 5.01726e-07, 0.000000000),
        799: (5.57450e-06, 4.74047e-07, 0.000000000),
        800: (5.24229e-06, 4.48030e-07, 0.000000000),
        801: (4.93115e-06, 4.23543e-07, 0.000000000),
        802: (4.63925e-06, 4.00453e-07, 0.000000000),
        803: (4.36488e-06, 3.78634e-07, 0.000000000),
        804: (4.10651e-06, 3.57978e-07, 0.000000000),
        805: (3.86280e-06, 3.38387e-07, 0.000000000),
        806: (3.63270e-06, 3.19787e-07, 0.000000000),
        807: (3.41579e-06, 3.02157e-07, 0.000000000),
        808: (3.21172e-06, 2.85486e-07, 0.000000000),
        809: (3.02010e-06, 2.69756e-07, 0.000000000),
        810: (2.84049e-06, 2.54942e-07, 0.000000000),
        811: (2.67239e-06, 2.41015e-07, 0.000000000),
        812: (2.51507e-06, 2.27920e-07, 0.000000000),
        813: (2.36783e-06, 2.15603e-07, 0.000000000),
        814: (2.22998e-06, 2.04014e-07, 0.000000000),
        815: (2.10091e-06, 1.93105e-07, 0.000000000),
        816: (1.98000e-06, 1.82829e-07, 0.000000000),
        817: (1.86657e-06, 1.73134e-07, 0.000000000),
        818: (1.75997e-06, 1.63969e-07, 0.000000000),
        819: (1.65964e-06, 1.55289e-07, 0.000000000),
        820: (1.56506e-06, 1.47054e-07, 0.000000000),
        821: (1.47580e-06, 1.39231e-07, 0.000000000),
        822: (1.39160e-06, 1.31804e-07, 0.000000000),
        823: (1.31221e-06, 1.24758e-07, 0.000000000),
        824: (1.23742e-06, 1.18078e-07, 0.000000000),
        825: (1.16700e-06, 1.11751e-07, 0.000000000),
        826: (1.10072e-06, 1.05760e-07, 0.000000000),
        827: (1.03839e-06, 1.00093e-07, 0.000000000),
        828: (9.79788e-07, 9.47349e-08, 0.000000000),
        829: (9.24725e-07, 8.96718e-08, 0.000000000),
        830: (8.73008e-07, 8.48902e-08, 0.000000000)
    },
    'Smith & Pokorny 1975 Normal Trichromats': {
        380: (0.0000, 0.0000, 0.0000),
        385: (0.0000, 0.0000, 0.0000),
        390: (0.0000, 0.0000, 0.0000),
        395: (0.0000, 0.0000, 0.0000),
        400: (0.0027, 0.0028, 0.1080),
        405: (0.0044, 0.0047, 0.1790),
        410: (0.0069, 0.0077, 0.2850),
        415: (0.0108, 0.0124, 0.4530),
        420: (0.0158, 0.0189, 0.6590),
        425: (0.0200, 0.0254, 0.8130),
        430: (0.0233, 0.0317, 0.9080),
        435: (0.0268, 0.0395, 0.9770),
        440: (0.0301, 0.0477, 1.0000),
        445: (0.0324, 0.0555, 0.9700),
        450: (0.0343, 0.0635, 0.9100),
        455: (0.0368, 0.0731, 0.8500),
        460: (0.0412, 0.0860, 0.7990),
        465: (0.0502, 0.1070, 0.7750),
        470: (0.0627, 0.1300, 0.6890),
        475: (0.0798, 0.1570, 0.5820),
        480: (0.1020, 0.1890, 0.4680),
        485: (0.1280, 0.2240, 0.3620),
        490: (0.1620, 0.2670, 0.2760),
        495: (0.2060, 0.3240, 0.2120),
        500: (0.2630, 0.3960, 0.1640),
        505: (0.3370, 0.4910, 0.1280),
        510: (0.4230, 0.5950, 0.0956),
        515: (0.5200, 0.7060, 0.0676),
        520: (0.6170, 0.8080, 0.0474),
        525: (0.7000, 0.8840, 0.0347),
        530: (0.7730, 0.9410, 0.0256),
        535: (0.8340, 0.9780, 0.0182),
        540: (0.8830, 0.9970, 0.0124),
        545: (0.9230, 0.9990, 0.0083),
        550: (0.9540, 0.9870, 0.0055),
        555: (0.9770, 0.9610, 0.0037),
        560: (0.9930, 0.9220, 0.0025),
        565: (1.0000, 0.8700, 0.0018),
        570: (0.9970, 0.8060, 0.0014),
        575: (0.9860, 0.7320, 0.0013),
        580: (0.9650, 0.6510, 0.0012),
        585: (0.9340, 0.5640, 0.0010),
        590: (0.8940, 0.4770, 0.0008),
        595: (0.8480, 0.3930, 0.0007),
        600: (0.7950, 0.3180, 0.0006),
        605: (0.7350, 0.2500, 0.0005),
        610: (0.6700, 0.1930, 0.0003),
        615: (0.6020, 0.1470, 0.0002),
        620: (0.5300, 0.1100, 0.0002),
        625: (0.4540, 0.0808, 0.0001),
        630: (0.3800, 0.0583, 0.0001),
        635: (0.3150, 0.0418, 0.0001),
        640: (0.2560, 0.0296, 0.0001),
        645: (0.2040, 0.0207, 0.0000),
        650: (0.1590, 0.0144, 0.0000),
        655: (0.1220, 0.0101, 0.0000),
        660: (0.0914, 0.0070, 0.0000),
        665: (0.0670, 0.0049, 0.0000),
        670: (0.0482, 0.0033, 0.0000),
        675: (0.0350, 0.0023, 0.0000),
        680: (0.0257, 0.0016, 0.0000),
        685: (0.0180, 0.0011, 0.0000),
        690: (0.0124, 0.0008, 0.0000),
        695: (0.0087, 0.0005, 0.0000),
        700: (0.0062, 0.0004, 0.0000),
        705: (0.0000, 0.0000, 0.0000),
        710: (0.0000, 0.0000, 0.0000),
        715: (0.0000, 0.0000, 0.0000),
        720: (0.0000, 0.0000, 0.0000),
        725: (0.0000, 0.0000, 0.0000),
        730: (0.0000, 0.0000, 0.0000),
        735: (0.0000, 0.0000, 0.0000),
        740: (0.0000, 0.0000, 0.0000),
        745: (0.0000, 0.0000, 0.0000),
        750: (0.0000, 0.0000, 0.0000),
        755: (0.0000, 0.0000, 0.0000),
        760: (0.0000, 0.0000, 0.0000),
        765: (0.0000, 0.0000, 0.0000),
        770: (0.0000, 0.0000, 0.0000),
        775: (0.0000, 0.0000, 0.0000),
        780: (0.0000, 0.0000, 0.0000)
    }
}

LMS_CMFS = CaseInsensitiveMapping({
    'Stockman & Sharpe 2 Degree Cone Fundamentals':
        LMS_ConeFundamentals(
            LMS_CMFS_DATA['Stockman & Sharpe 2 Degree Cone Fundamentals'],
            name='Stockman & Sharpe 2 Degree Cone Fundamentals',
            strict_name='Stockman & Sharpe 2$^\\circ$ Cone Fundamentals'),
    'Stockman & Sharpe 10 Degree Cone Fundamentals':
        LMS_ConeFundamentals(
            LMS_CMFS_DATA['Stockman & Sharpe 10 Degree Cone Fundamentals'],
            name='Stockman & Sharpe 10 Degree Cone Fundamentals',
            strict_name='Stockman & Sharpe 10$^\\circ$ Cone Fundamentals'),
    'Smith & Pokorny 1975 Normal Trichromats':
        LMS_ConeFundamentals(
            LMS_CMFS_DATA['Smith & Pokorny 1975 Normal Trichromats'],
            name='Smith & Pokorny 1975 Normal Trichromats',
            strict_name='Smith & Pokorny (1975) Normal Trichromats')
})
LMS_CMFS.__doc__ = """
*LMS* colour matching functions.

References
----------
:cite:`CVRLu`, :cite:`Machado2010a`

LMS_CMFS : CaseInsensitiveMapping
    {'Stockman & Sharpe 2 Degree Cone Fundamentals',
    'Stockman & Sharpe 10 Degree Cone Fundamentals',
    'Smith & Pokorny 1975 Normal Trichromats'}
"""

RGB_CMFS_DATA = {
    'Wright & Guild 1931 2 Degree RGB CMFs': {
        380: (0.00003, -0.00001, 0.00117),
        385: (0.00005, -0.00002, 0.00189),
        390: (0.00010, -0.00004, 0.00359),
        395: (0.00017, -0.00007, 0.00647),
        400: (0.00030, -0.00014, 0.01214),
        405: (0.00047, -0.00022, 0.01969),
        410: (0.00084, -0.00041, 0.03707),
        415: (0.00139, -0.00070, 0.06637),
        420: (0.00211, -0.00110, 0.11541),
        425: (0.00266, -0.00143, 0.18575),
        430: (0.00218, -0.00119, 0.24769),
        435: (0.00036, -0.00021, 0.29012),
        440: (-0.00261, 0.00149, 0.31228),
        445: (-0.00673, 0.00379, 0.31860),
        450: (-0.01213, 0.00678, 0.31670),
        455: (-0.01874, 0.01046, 0.31166),
        460: (-0.02608, 0.01485, 0.29821),
        465: (-0.03324, 0.01977, 0.27295),
        470: (-0.03933, 0.02538, 0.22991),
        475: (-0.04471, 0.03183, 0.18592),
        480: (-0.04939, 0.03914, 0.14494),
        485: (-0.05364, 0.04713, 0.10968),
        490: (-0.05814, 0.05689, 0.08257),
        495: (-0.06414, 0.06948, 0.06246),
        500: (-0.07137, 0.08536, 0.04776),
        505: (-0.08120, 0.10593, 0.03688),
        510: (-0.08901, 0.12860, 0.02698),
        515: (-0.09356, 0.15262, 0.01842),
        520: (-0.09264, 0.17468, 0.01221),
        525: (-0.08473, 0.19113, 0.00830),
        530: (-0.07101, 0.20317, 0.00549),
        535: (-0.05316, 0.21083, 0.00320),
        540: (-0.03152, 0.21466, 0.00146),
        545: (-0.00613, 0.21487, 0.00023),
        550: (0.02279, 0.21178, -0.00058),
        555: (0.05514, 0.20588, -0.00105),
        560: (0.09060, 0.19702, -0.00130),
        565: (0.12840, 0.18522, -0.00138),
        570: (0.16768, 0.17087, -0.00135),
        575: (0.20715, 0.15429, -0.00123),
        580: (0.24562, 0.13610, -0.00108),
        585: (0.27989, 0.11686, -0.00093),
        590: (0.30928, 0.09754, -0.00079),
        595: (0.33184, 0.07909, -0.00063),
        600: (0.34429, 0.06246, -0.00049),
        605: (0.34756, 0.04776, -0.00038),
        610: (0.33971, 0.03557, -0.00030),
        615: (0.32265, 0.02583, -0.00022),
        620: (0.29708, 0.01828, -0.00015),
        625: (0.26348, 0.01253, -0.00011),
        630: (0.22677, 0.00833, -0.00008),
        635: (0.19233, 0.00537, -0.00005),
        640: (0.15968, 0.00334, -0.00003),
        645: (0.12905, 0.00199, -0.00002),
        650: (0.10167, 0.00116, -0.00001),
        655: (0.07857, 0.00066, -0.00001),
        660: (0.05932, 0.00037, 0.00000),
        665: (0.04366, 0.00021, 0.00000),
        670: (0.03149, 0.00011, 0.00000),
        675: (0.02294, 0.00006, 0.00000),
        680: (0.01687, 0.00003, 0.00000),
        685: (0.01187, 0.00001, 0.00000),
        690: (0.00819, 0.00000, 0.00000),
        695: (0.00572, 0.00000, 0.00000),
        700: (0.00410, 0.00000, 0.00000),
        705: (0.00291, 0.00000, 0.00000),
        710: (0.00210, 0.00000, 0.00000),
        715: (0.00148, 0.00000, 0.00000),
        720: (0.00105, 0.00000, 0.00000),
        725: (0.00074, 0.00000, 0.00000),
        730: (0.00052, 0.00000, 0.00000),
        735: (0.00036, 0.00000, 0.00000),
        740: (0.00025, 0.00000, 0.00000),
        745: (0.00017, 0.00000, 0.00000),
        750: (0.00012, 0.00000, 0.00000),
        755: (0.00008, 0.00000, 0.00000),
        760: (0.00006, 0.00000, 0.00000),
        765: (0.00004, 0.00000, 0.00000),
        770: (0.00003, 0.00000, 0.00000),
        775: (0.00001, 0.00000, 0.00000),
        780: (0.00000, 0.00000, 0.00000)
    },
    'Stiles & Burch 1955 2 Degree RGB CMFs': {
        390: (1.83970e-003, -4.53930e-004, 1.21520e-002),
        395: (4.61530e-003, -1.04640e-003, 3.11100e-002),
        400: (9.62640e-003, -2.16890e-003, 6.23710e-002),
        405: (1.89790e-002, -4.43040e-003, 1.31610e-001),
        410: (3.08030e-002, -7.20480e-003, 2.27500e-001),
        415: (4.24590e-002, -1.25790e-002, 3.58970e-001),
        420: (5.16620e-002, -1.66510e-002, 5.23960e-001),
        425: (5.28370e-002, -2.12400e-002, 6.85860e-001),
        430: (4.42870e-002, -1.99360e-002, 7.96040e-001),
        435: (3.22200e-002, -1.60970e-002, 8.94590e-001),
        440: (1.47630e-002, -7.34570e-003, 9.63950e-001),
        445: (-2.33920e-003, 1.36900e-003, 9.98140e-001),
        450: (-2.91300e-002, 1.96100e-002, 9.18750e-001),
        455: (-6.06770e-002, 4.34640e-002, 8.24870e-001),
        460: (-9.62240e-002, 7.09540e-002, 7.85540e-001),
        465: (-1.37590e-001, 1.10220e-001, 6.67230e-001),
        470: (-1.74860e-001, 1.50880e-001, 6.10980e-001),
        475: (-2.12600e-001, 1.97940e-001, 4.88290e-001),
        480: (-2.37800e-001, 2.40420e-001, 3.61950e-001),
        485: (-2.56740e-001, 2.79930e-001, 2.66340e-001),
        490: (-2.77270e-001, 3.33530e-001, 1.95930e-001),
        495: (-2.91250e-001, 4.05210e-001, 1.47300e-001),
        500: (-2.95000e-001, 4.90600e-001, 1.07490e-001),
        505: (-2.97060e-001, 5.96730e-001, 7.67140e-002),
        510: (-2.67590e-001, 7.01840e-001, 5.02480e-002),
        515: (-2.17250e-001, 8.08520e-001, 2.87810e-002),
        520: (-1.47680e-001, 9.10760e-001, 1.33090e-002),
        525: (-3.51840e-002, 9.84820e-001, 2.11700e-003),
        530: (1.06140e-001, 1.03390e+000, -4.15740e-003),
        535: (2.59810e-001, 1.05380e+000, -8.30320e-003),
        540: (4.19760e-001, 1.05120e+000, -1.21910e-002),
        545: (5.92590e-001, 1.04980e+000, -1.40390e-002),
        550: (7.90040e-001, 1.03680e+000, -1.46810e-002),
        555: (1.00780e+000, 9.98260e-001, -1.49470e-002),
        560: (1.22830e+000, 9.37830e-001, -1.46130e-002),
        565: (1.47270e+000, 8.80390e-001, -1.37820e-002),
        570: (1.74760e+000, 8.28350e-001, -1.26500e-002),
        575: (2.02140e+000, 7.46860e-001, -1.13560e-002),
        580: (2.27240e+000, 6.49300e-001, -9.93170e-003),
        585: (2.48960e+000, 5.63170e-001, -8.41480e-003),
        590: (2.67250e+000, 4.76750e-001, -7.02100e-003),
        595: (2.80930e+000, 3.84840e-001, -5.74370e-003),
        600: (2.87170e+000, 3.00690e-001, -4.27430e-003),
        605: (2.85250e+000, 2.28530e-001, -2.91320e-003),
        610: (2.76010e+000, 1.65750e-001, -2.26930e-003),
        615: (2.59890e+000, 1.13730e-001, -1.99660e-003),
        620: (2.37430e+000, 7.46820e-002, -1.50690e-003),
        625: (2.10540e+000, 4.65040e-002, -9.38220e-004),
        630: (1.81450e+000, 2.63330e-002, -5.53160e-004),
        635: (1.52470e+000, 1.27240e-002, -3.16680e-004),
        640: (1.25430e+000, 4.50330e-003, -1.43190e-004),
        645: (1.00760e+000, 9.66110e-005, -4.08310e-006),
        650: (7.86420e-001, -1.96450e-003, 1.10810e-004),
        655: (5.96590e-001, -2.63270e-003, 1.91750e-004),
        660: (4.43200e-001, -2.62620e-003, 2.26560e-004),
        665: (3.24100e-001, -2.30270e-003, 2.15200e-004),
        670: (2.34550e-001, -1.87000e-003, 1.63610e-004),
        675: (1.68840e-001, -1.44240e-003, 9.71640e-005),
        680: (1.20860e-001, -1.07550e-003, 5.10330e-005),
        685: (8.58110e-002, -7.90040e-004, 3.52710e-005),
        690: (6.02600e-002, -5.67650e-004, 3.12110e-005),
        695: (4.14800e-002, -3.92740e-004, 2.45080e-005),
        700: (2.81140e-002, -2.62310e-004, 1.65210e-005),
        705: (1.91170e-002, -1.75120e-004, 1.11240e-005),
        710: (1.33050e-002, -1.21400e-004, 8.69650e-006),
        715: (9.40920e-003, -8.57600e-005, 7.43510e-006),
        720: (6.51770e-003, -5.76770e-005, 6.10570e-006),
        725: (4.53770e-003, -3.90030e-005, 5.02770e-006),
        730: (3.17420e-003, -2.65110e-005, 4.12510e-006)
    },
    'Stiles & Burch 1959 10 Degree RGB CMFs': {
        390: (1.5000e-03, -4.0000e-04, 6.2000e-03),
        395: (3.8000e-03, -1.0000e-03, 1.6100e-02),
        400: (8.9000e-03, -2.5000e-03, 4.0000e-02),
        405: (1.8800e-02, -5.9000e-03, 9.0600e-02),
        410: (3.5000e-02, -1.1900e-02, 1.8020e-01),
        415: (5.3100e-02, -2.0100e-02, 3.0880e-01),
        420: (7.0200e-02, -2.8900e-02, 4.6700e-01),
        425: (7.6300e-02, -3.3800e-02, 6.1520e-01),
        430: (7.4500e-02, -3.4900e-02, 7.6380e-01),
        435: (5.6100e-02, -2.7600e-02, 8.7780e-01),
        440: (3.2300e-02, -1.6900e-02, 9.7550e-01),
        445: (-4.4000e-03, 2.4000e-03, 1.0019e+00),
        450: (-4.7800e-02, 2.8300e-02, 9.9960e-01),
        455: (-9.7000e-02, 6.3600e-02, 9.1390e-01),
        460: (-1.5860e-01, 1.0820e-01, 8.2970e-01),
        465: (-2.2350e-01, 1.6170e-01, 7.4170e-01),
        470: (-2.8480e-01, 2.2010e-01, 6.1340e-01),
        475: (-3.3460e-01, 2.7960e-01, 4.7200e-01),
        480: (-3.7760e-01, 3.4280e-01, 3.4950e-01),
        485: (-4.1360e-01, 4.0860e-01, 2.5640e-01),
        490: (-4.3170e-01, 4.7160e-01, 1.8190e-01),
        495: (-4.4520e-01, 5.4910e-01, 1.3070e-01),
        500: (-4.3500e-01, 6.2600e-01, 9.1000e-02),
        505: (-4.1400e-01, 7.0970e-01, 5.8000e-02),
        510: (-3.6730e-01, 7.9350e-01, 3.5700e-02),
        515: (-2.8450e-01, 8.7150e-01, 2.0000e-02),
        520: (-1.8550e-01, 9.4770e-01, 9.5000e-03),
        525: (-4.3500e-02, 9.9450e-01, 7.0000e-04),
        530: (1.2700e-01, 1.0203e+00, -4.3000e-03),
        535: (3.1290e-01, 1.0375e+00, -6.4000e-03),
        540: (5.3620e-01, 1.0517e+00, -8.2000e-03),
        545: (7.7220e-01, 1.0390e+00, -9.4000e-03),
        550: (1.0059e+00, 1.0029e+00, -9.7000e-03),
        555: (1.2710e+00, 9.6980e-01, -9.7000e-03),
        560: (1.5574e+00, 9.1620e-01, -9.3000e-03),
        565: (1.8465e+00, 8.5710e-01, -8.7000e-03),
        570: (2.1511e+00, 7.8230e-01, -8.0000e-03),
        575: (2.4250e+00, 6.9530e-01, -7.3000e-03),
        580: (2.6574e+00, 5.9660e-01, -6.3000e-03),
        585: (2.9151e+00, 5.0630e-01, -5.3700e-03),
        590: (3.0779e+00, 4.2030e-01, -4.4500e-03),
        595: (3.1613e+00, 3.3600e-01, -3.5700e-03),
        600: (3.1673e+00, 2.5910e-01, -2.7700e-03),
        605: (3.1048e+00, 1.9170e-01, -2.0800e-03),
        610: (2.9462e+00, 1.3670e-01, -1.5000e-03),
        615: (2.7194e+00, 9.3800e-02, -1.0300e-03),
        620: (2.4526e+00, 6.1100e-02, -6.8000e-04),
        625: (2.1700e+00, 3.7100e-02, -4.4200e-04),
        630: (1.8358e+00, 2.1500e-02, -2.7200e-04),
        635: (1.5179e+00, 1.1200e-02, -1.4100e-04),
        640: (1.2428e+00, 4.4000e-03, -5.4900e-05),
        645: (1.0070e+00, 7.8000e-05, -2.2000e-06),
        650: (7.8270e-01, -1.3680e-03, 2.3700e-05),
        655: (5.9340e-01, -1.9880e-03, 2.8600e-05),
        660: (4.4420e-01, -2.1680e-03, 2.6100e-05),
        665: (3.2830e-01, -2.0060e-03, 2.2500e-05),
        670: (2.3940e-01, -1.6420e-03, 1.8200e-05),
        675: (1.7220e-01, -1.2720e-03, 1.3900e-05),
        680: (1.2210e-01, -9.4700e-04, 1.0300e-05),
        685: (8.5300e-02, -6.8300e-04, 7.3800e-06),
        690: (5.8600e-02, -4.7800e-04, 5.2200e-06),
        695: (4.0800e-02, -3.3700e-04, 3.6700e-06),
        700: (2.8400e-02, -2.3500e-04, 2.5600e-06),
        705: (1.9700e-02, -1.6300e-04, 1.7600e-06),
        710: (1.3500e-02, -1.1100e-04, 1.2000e-06),
        715: (9.2400e-03, -7.4800e-05, 8.1700e-07),
        720: (6.3800e-03, -5.0800e-05, 5.5500e-07),
        725: (4.4100e-03, -3.4400e-05, 3.7500e-07),
        730: (3.0700e-03, -2.3400e-05, 2.5400e-07),
        735: (2.1400e-03, -1.5900e-05, 1.7100e-07),
        740: (1.4900e-03, -1.0700e-05, 1.1600e-07),
        745: (1.0500e-03, -7.2300e-06, 7.8500e-08),
        750: (7.3900e-04, -4.8700e-06, 5.3100e-08),
        755: (5.2300e-04, -3.2900e-06, 3.6000e-08),
        760: (3.7200e-04, -2.2200e-06, 2.4400e-08),
        765: (2.6500e-04, -1.5000e-06, 1.6500e-08),
        770: (1.9000e-04, -1.0200e-06, 1.1200e-08),
        775: (1.3600e-04, -6.8800e-07, 7.5300e-09),
        780: (9.8400e-05, -4.6500e-07, 5.0700e-09),
        785: (7.1300e-05, -3.1200e-07, 3.4000e-09),
        790: (5.1800e-05, -2.0800e-07, 2.2700e-09),
        795: (3.7700e-05, -1.3700e-07, 1.5000e-09),
        800: (2.7600e-05, -8.8000e-08, 9.8600e-10),
        805: (2.0300e-05, -5.5300e-08, 6.3900e-10),
        810: (1.4900e-05, -3.3600e-08, 4.0700e-10),
        815: (1.1000e-05, -1.9600e-08, 2.5300e-10),
        820: (8.1800e-06, -1.0900e-08, 1.5200e-10),
        825: (6.0900e-06, -5.7000e-09, 8.6400e-11),
        830: (4.5500e-06, -2.7700e-09, 4.4200e-11)
    }
}

RGB_CMFS = CaseInsensitiveMapping({
    'Wright & Guild 1931 2 Degree RGB CMFs':
        RGB_ColourMatchingFunctions(
            RGB_CMFS_DATA['Wright & Guild 1931 2 Degree RGB CMFs'],
            name='Wright & Guild 1931 2 Degree RGB CMFs',
            strict_name='Wright & Guild (1931) 2$^\\circ$ RGB CMFs',
        ),
    'Stiles & Burch 1955 2 Degree RGB CMFs':
        RGB_ColourMatchingFunctions(
            RGB_CMFS_DATA['Stiles & Burch 1955 2 Degree RGB CMFs'],
            name='Stiles & Burch 1955 2 Degree RGB CMFs',
            strict_name='Stiles & Burch (1955) 2$^\\circ$ RGB CMFs'),
    'Stiles & Burch 1959 10 Degree RGB CMFs':
        RGB_ColourMatchingFunctions(
            RGB_CMFS_DATA['Stiles & Burch 1959 10 Degree RGB CMFs'],
            name='Stiles & Burch 1959 10 Degree RGB CMFs',
            strict_name='Stiles & Burch (1959) 10$^\\circ$ RGB CMFs')
})
RGB_CMFS.__doc__ = """
*CIE RGB* colour matching functions.

References
----------
:cite:`Broadbent2009a`, :cite:`CVRLt`, :cite:`CVRLw`

RGB_CMFS : CaseInsensitiveMapping
    **{'Wright & Guild 1931 2 Degree RGB CMFs',
    'Stiles & Burch 1955 2 Degree RGB CMFs',
    'Stiles & Burch 1959 10 Degree RGB CMFs'}**
"""

STANDARD_OBSERVERS_CMFS_DATA = {
    'CIE 1931 2 Degree Standard Observer': {
        360: (0.000129900000, 0.000003917000, 0.000606100000),
        361: (0.000145847000, 0.000004393581, 0.000680879200),
        362: (0.000163802100, 0.000004929604, 0.000765145600),
        363: (0.000184003700, 0.000005532136, 0.000860012400),
        364: (0.000206690200, 0.000006208245, 0.000966592800),
        365: (0.000232100000, 0.000006965000, 0.001086000000),
        366: (0.000260728000, 0.000007813219, 0.001220586000),
        367: (0.000293075000, 0.000008767336, 0.001372729000),
        368: (0.000329388000, 0.000009839844, 0.001543579000),
        369: (0.000369914000, 0.000011043230, 0.001734286000),
        370: (0.000414900000, 0.000012390000, 0.001946000000),
        371: (0.000464158700, 0.000013886410, 0.002177777000),
        372: (0.000518986000, 0.000015557280, 0.002435809000),
        373: (0.000581854000, 0.000017442960, 0.002731953000),
        374: (0.000655234700, 0.000019583750, 0.003078064000),
        375: (0.000741600000, 0.000022020000, 0.003486000000),
        376: (0.000845029600, 0.000024839650, 0.003975227000),
        377: (0.000964526800, 0.000028041260, 0.004540880000),
        378: (0.001094949000, 0.000031531040, 0.005158320000),
        379: (0.001231154000, 0.000035215210, 0.005802907000),
        380: (0.001368000000, 0.000039000000, 0.006450001000),
        381: (0.001502050000, 0.000042826400, 0.007083216000),
        382: (0.001642328000, 0.000046914600, 0.007745488000),
        383: (0.001802382000, 0.000051589600, 0.008501152000),
        384: (0.001995757000, 0.000057176400, 0.009414544000),
        385: (0.002236000000, 0.000064000000, 0.010549990000),
        386: (0.002535385000, 0.000072344210, 0.011965800000),
        387: (0.002892603000, 0.000082212240, 0.013655870000),
        388: (0.003300829000, 0.000093508160, 0.015588050000),
        389: (0.003753236000, 0.000106136100, 0.017730150000),
        390: (0.004243000000, 0.000120000000, 0.020050010000),
        391: (0.004762389000, 0.000134984000, 0.022511360000),
        392: (0.005330048000, 0.000151492000, 0.025202880000),
        393: (0.005978712000, 0.000170208000, 0.028279720000),
        394: (0.006741117000, 0.000191816000, 0.031897040000),
        395: (0.007650000000, 0.000217000000, 0.036210000000),
        396: (0.008751373000, 0.000246906700, 0.041437710000),
        397: (0.010028880000, 0.000281240000, 0.047503720000),
        398: (0.011421700000, 0.000318520000, 0.054119880000),
        399: (0.012869010000, 0.000357266700, 0.060998030000),
        400: (0.014310000000, 0.000396000000, 0.067850010000),
        401: (0.015704430000, 0.000433714700, 0.074486320000),
        402: (0.017147440000, 0.000473024000, 0.081361560000),
        403: (0.018781220000, 0.000517876000, 0.089153640000),
        404: (0.020748010000, 0.000572218700, 0.098540480000),
        405: (0.023190000000, 0.000640000000, 0.110200000000),
        406: (0.026207360000, 0.000724560000, 0.124613300000),
        407: (0.029782480000, 0.000825500000, 0.141701700000),
        408: (0.033880920000, 0.000941160000, 0.161303500000),
        409: (0.038468240000, 0.001069880000, 0.183256800000),
        410: (0.043510000000, 0.001210000000, 0.207400000000),
        411: (0.048995600000, 0.001362091000, 0.233692100000),
        412: (0.055022600000, 0.001530752000, 0.262611400000),
        413: (0.061718800000, 0.001720368000, 0.294774600000),
        414: (0.069212000000, 0.001935323000, 0.330798500000),
        415: (0.077630000000, 0.002180000000, 0.371300000000),
        416: (0.086958110000, 0.002454800000, 0.416209100000),
        417: (0.097176720000, 0.002764000000, 0.465464200000),
        418: (0.108406300000, 0.003117800000, 0.519694800000),
        419: (0.120767200000, 0.003526400000, 0.579530300000),
        420: (0.134380000000, 0.004000000000, 0.645600000000),
        421: (0.149358200000, 0.004546240000, 0.718483800000),
        422: (0.165395700000, 0.005159320000, 0.796713300000),
        423: (0.181983100000, 0.005829280000, 0.877845900000),
        424: (0.198611000000, 0.006546160000, 0.959439000000),
        425: (0.214770000000, 0.007300000000, 1.039050100000),
        426: (0.230186800000, 0.008086507000, 1.115367300000),
        427: (0.244879700000, 0.008908720000, 1.188497100000),
        428: (0.258777300000, 0.009767680000, 1.258123300000),
        429: (0.271807900000, 0.010664430000, 1.323929600000),
        430: (0.283900000000, 0.011600000000, 1.385600000000),
        431: (0.294943800000, 0.012573170000, 1.442635200000),
        432: (0.304896500000, 0.013582720000, 1.494803500000),
        433: (0.313787300000, 0.014629680000, 1.542190300000),
        434: (0.321645400000, 0.015715090000, 1.584880700000),
        435: (0.328500000000, 0.016840000000, 1.622960000000),
        436: (0.334351300000, 0.018007360000, 1.656404800000),
        437: (0.339210100000, 0.019214480000, 1.685295900000),
        438: (0.343121300000, 0.020453920000, 1.709874500000),
        439: (0.346129600000, 0.021718240000, 1.730382100000),
        440: (0.348280000000, 0.023000000000, 1.747060000000),
        441: (0.349599900000, 0.024294610000, 1.760044600000),
        442: (0.350147400000, 0.025610240000, 1.769623300000),
        443: (0.350013000000, 0.026958570000, 1.776263700000),
        444: (0.349287000000, 0.028351250000, 1.780433400000),
        445: (0.348060000000, 0.029800000000, 1.782600000000),
        446: (0.346373300000, 0.031310830000, 1.782968200000),
        447: (0.344262400000, 0.032883680000, 1.781699800000),
        448: (0.341808800000, 0.034521120000, 1.779198200000),
        449: (0.339094100000, 0.036225710000, 1.775867100000),
        450: (0.336200000000, 0.038000000000, 1.772110000000),
        451: (0.333197700000, 0.039846670000, 1.768258900000),
        452: (0.330041100000, 0.041768000000, 1.764039000000),
        453: (0.326635700000, 0.043766000000, 1.758943800000),
        454: (0.322886800000, 0.045842670000, 1.752466300000),
        455: (0.318700000000, 0.048000000000, 1.744100000000),
        456: (0.314025100000, 0.050243680000, 1.733559500000),
        457: (0.308884000000, 0.052573040000, 1.720858100000),
        458: (0.303290400000, 0.054980560000, 1.705936900000),
        459: (0.297257900000, 0.057458720000, 1.688737200000),
        460: (0.290800000000, 0.060000000000, 1.669200000000),
        461: (0.283970100000, 0.062601970000, 1.647528700000),
        462: (0.276721400000, 0.065277520000, 1.623412700000),
        463: (0.268917800000, 0.068042080000, 1.596022300000),
        464: (0.260422700000, 0.070911090000, 1.564528000000),
        465: (0.251100000000, 0.073900000000, 1.528100000000),
        466: (0.240847500000, 0.077016000000, 1.486111400000),
        467: (0.229851200000, 0.080266400000, 1.439521500000),
        468: (0.218407200000, 0.083666800000, 1.389879900000),
        469: (0.206811500000, 0.087232800000, 1.338736200000),
        470: (0.195360000000, 0.090980000000, 1.287640000000),
        471: (0.184213600000, 0.094917550000, 1.237422300000),
        472: (0.173327300000, 0.099045840000, 1.187824300000),
        473: (0.162688100000, 0.103367400000, 1.138761100000),
        474: (0.152283300000, 0.107884600000, 1.090148000000),
        475: (0.142100000000, 0.112600000000, 1.041900000000),
        476: (0.132178600000, 0.117532000000, 0.994197600000),
        477: (0.122569600000, 0.122674400000, 0.947347300000),
        478: (0.113275200000, 0.127992800000, 0.901453100000),
        479: (0.104297900000, 0.133452800000, 0.856619300000),
        480: (0.095640000000, 0.139020000000, 0.812950100000),
        481: (0.087299550000, 0.144676400000, 0.770517300000),
        482: (0.079308040000, 0.150469300000, 0.729444800000),
        483: (0.071717760000, 0.156461900000, 0.689913600000),
        484: (0.064580990000, 0.162717700000, 0.652104900000),
        485: (0.057950010000, 0.169300000000, 0.616200000000),
        486: (0.051862110000, 0.176243100000, 0.582328600000),
        487: (0.046281520000, 0.183558100000, 0.550416200000),
        488: (0.041150880000, 0.191273500000, 0.520337600000),
        489: (0.036412830000, 0.199418000000, 0.491967300000),
        490: (0.032010000000, 0.208020000000, 0.465180000000),
        491: (0.027917200000, 0.217119900000, 0.439924600000),
        492: (0.024144400000, 0.226734500000, 0.416183600000),
        493: (0.020687000000, 0.236857100000, 0.393882200000),
        494: (0.017540400000, 0.247481200000, 0.372945900000),
        495: (0.014700000000, 0.258600000000, 0.353300000000),
        496: (0.012161790000, 0.270184900000, 0.334857800000),
        497: (0.009919960000, 0.282293900000, 0.317552100000),
        498: (0.007967240000, 0.295050500000, 0.301337500000),
        499: (0.006296346000, 0.308578000000, 0.286168600000),
        500: (0.004900000000, 0.323000000000, 0.272000000000),
        501: (0.003777173000, 0.338402100000, 0.258817100000),
        502: (0.002945320000, 0.354685800000, 0.246483800000),
        503: (0.002424880000, 0.371698600000, 0.234771800000),
        504: (0.002236293000, 0.389287500000, 0.223453300000),
        505: (0.002400000000, 0.407300000000, 0.212300000000),
        506: (0.002925520000, 0.425629900000, 0.201169200000),
        507: (0.003836560000, 0.444309600000, 0.190119600000),
        508: (0.005174840000, 0.463394400000, 0.179225400000),
        509: (0.006982080000, 0.482939500000, 0.168560800000),
        510: (0.009300000000, 0.503000000000, 0.158200000000),
        511: (0.012149490000, 0.523569300000, 0.148138300000),
        512: (0.015535880000, 0.544512000000, 0.138375800000),
        513: (0.019477520000, 0.565690000000, 0.128994200000),
        514: (0.023992770000, 0.586965300000, 0.120075100000),
        515: (0.029100000000, 0.608200000000, 0.111700000000),
        516: (0.034814850000, 0.629345600000, 0.103904800000),
        517: (0.041120160000, 0.650306800000, 0.096667480000),
        518: (0.047985040000, 0.670875200000, 0.089982720000),
        519: (0.055378610000, 0.690842400000, 0.083845310000),
        520: (0.063270000000, 0.710000000000, 0.078249990000),
        521: (0.071635010000, 0.728185200000, 0.073208990000),
        522: (0.080462240000, 0.745463600000, 0.068678160000),
        523: (0.089739960000, 0.761969400000, 0.064567840000),
        524: (0.099456450000, 0.777836800000, 0.060788350000),
        525: (0.109600000000, 0.793200000000, 0.057250010000),
        526: (0.120167400000, 0.808110400000, 0.053904350000),
        527: (0.131114500000, 0.822496200000, 0.050746640000),
        528: (0.142367900000, 0.836306800000, 0.047752760000),
        529: (0.153854200000, 0.849491600000, 0.044898590000),
        530: (0.165500000000, 0.862000000000, 0.042160000000),
        531: (0.177257100000, 0.873810800000, 0.039507280000),
        532: (0.189140000000, 0.884962400000, 0.036935640000),
        533: (0.201169400000, 0.895493600000, 0.034458360000),
        534: (0.213365800000, 0.905443200000, 0.032088720000),
        535: (0.225749900000, 0.914850100000, 0.029840000000),
        536: (0.238320900000, 0.923734800000, 0.027711810000),
        537: (0.251066800000, 0.932092400000, 0.025694440000),
        538: (0.263992200000, 0.939922600000, 0.023787160000),
        539: (0.277101700000, 0.947225200000, 0.021989250000),
        540: (0.290400000000, 0.954000000000, 0.020300000000),
        541: (0.303891200000, 0.960256100000, 0.018718050000),
        542: (0.317572600000, 0.966007400000, 0.017240360000),
        543: (0.331438400000, 0.971260600000, 0.015863640000),
        544: (0.345482800000, 0.976022500000, 0.014584610000),
        545: (0.359700000000, 0.980300000000, 0.013400000000),
        546: (0.374083900000, 0.984092400000, 0.012307230000),
        547: (0.388639600000, 0.987418200000, 0.011301880000),
        548: (0.403378400000, 0.990312800000, 0.010377920000),
        549: (0.418311500000, 0.992811600000, 0.009529306000),
        550: (0.433449900000, 0.994950100000, 0.008749999000),
        551: (0.448795300000, 0.996710800000, 0.008035200000),
        552: (0.464336000000, 0.998098300000, 0.007381600000),
        553: (0.480064000000, 0.999112000000, 0.006785400000),
        554: (0.495971300000, 0.999748200000, 0.006242800000),
        555: (0.512050100000, 1.000000000000, 0.005749999000),
        556: (0.528295900000, 0.999856700000, 0.005303600000),
        557: (0.544691600000, 0.999304600000, 0.004899800000),
        558: (0.561209400000, 0.998325500000, 0.004534200000),
        559: (0.577821500000, 0.996898700000, 0.004202400000),
        560: (0.594500000000, 0.995000000000, 0.003900000000),
        561: (0.611220900000, 0.992600500000, 0.003623200000),
        562: (0.627975800000, 0.989742600000, 0.003370600000),
        563: (0.644760200000, 0.986444400000, 0.003141400000),
        564: (0.661569700000, 0.982724100000, 0.002934800000),
        565: (0.678400000000, 0.978600000000, 0.002749999000),
        566: (0.695239200000, 0.974083700000, 0.002585200000),
        567: (0.712058600000, 0.969171200000, 0.002438600000),
        568: (0.728828400000, 0.963856800000, 0.002309400000),
        569: (0.745518800000, 0.958134900000, 0.002196800000),
        570: (0.762100000000, 0.952000000000, 0.002100000000),
        571: (0.778543200000, 0.945450400000, 0.002017733000),
        572: (0.794825600000, 0.938499200000, 0.001948200000),
        573: (0.810926400000, 0.931162800000, 0.001889800000),
        574: (0.826824800000, 0.923457600000, 0.001840933000),
        575: (0.842500000000, 0.915400000000, 0.001800000000),
        576: (0.857932500000, 0.907006400000, 0.001766267000),
        577: (0.873081600000, 0.898277200000, 0.001737800000),
        578: (0.887894400000, 0.889204800000, 0.001711200000),
        579: (0.902318100000, 0.879781600000, 0.001683067000),
        580: (0.916300000000, 0.870000000000, 0.001650001000),
        581: (0.929799500000, 0.859861300000, 0.001610133000),
        582: (0.942798400000, 0.849392000000, 0.001564400000),
        583: (0.955277600000, 0.838622000000, 0.001513600000),
        584: (0.967217900000, 0.827581300000, 0.001458533000),
        585: (0.978600000000, 0.816300000000, 0.001400000000),
        586: (0.989385600000, 0.804794700000, 0.001336667000),
        587: (0.999548800000, 0.793082000000, 0.001270000000),
        588: (1.009089200000, 0.781192000000, 0.001205000000),
        589: (1.018006400000, 0.769154700000, 0.001146667000),
        590: (1.026300000000, 0.757000000000, 0.001100000000),
        591: (1.033982700000, 0.744754100000, 0.001068800000),
        592: (1.040986000000, 0.732422400000, 0.001049400000),
        593: (1.047188000000, 0.720003600000, 0.001035600000),
        594: (1.052466700000, 0.707496500000, 0.001021200000),
        595: (1.056700000000, 0.694900000000, 0.001000000000),
        596: (1.059794400000, 0.682219200000, 0.000968640000),
        597: (1.061799200000, 0.669471600000, 0.000929920000),
        598: (1.062806800000, 0.656674400000, 0.000886880000),
        599: (1.062909600000, 0.643844800000, 0.000842560000),
        600: (1.062200000000, 0.631000000000, 0.000800000000),
        601: (1.060735200000, 0.618155500000, 0.000760960000),
        602: (1.058443600000, 0.605314400000, 0.000723680000),
        603: (1.055224400000, 0.592475600000, 0.000685920000),
        604: (1.050976800000, 0.579637900000, 0.000645440000),
        605: (1.045600000000, 0.566800000000, 0.000600000000),
        606: (1.039036900000, 0.553961100000, 0.000547866700),
        607: (1.031360800000, 0.541137200000, 0.000491600000),
        608: (1.022666200000, 0.528352800000, 0.000435400000),
        609: (1.013047700000, 0.515632300000, 0.000383466700),
        610: (1.002600000000, 0.503000000000, 0.000340000000),
        611: (0.991367500000, 0.490468800000, 0.000307253300),
        612: (0.979331400000, 0.478030400000, 0.000283160000),
        613: (0.966491600000, 0.465677600000, 0.000265440000),
        614: (0.952847900000, 0.453403200000, 0.000251813300),
        615: (0.938400000000, 0.441200000000, 0.000240000000),
        616: (0.923194000000, 0.429080000000, 0.000229546700),
        617: (0.907244000000, 0.417036000000, 0.000220640000),
        618: (0.890502000000, 0.405032000000, 0.000211960000),
        619: (0.872920000000, 0.393032000000, 0.000202186700),
        620: (0.854449900000, 0.381000000000, 0.000190000000),
        621: (0.835084000000, 0.368918400000, 0.000174213300),
        622: (0.814946000000, 0.356827200000, 0.000155640000),
        623: (0.794186000000, 0.344776800000, 0.000135960000),
        624: (0.772954000000, 0.332817600000, 0.000116853300),
        625: (0.751400000000, 0.321000000000, 0.000100000000),
        626: (0.729583600000, 0.309338100000, 0.000086133330),
        627: (0.707588800000, 0.297850400000, 0.000074600000),
        628: (0.685602200000, 0.286593600000, 0.000065000000),
        629: (0.663810400000, 0.275624500000, 0.000056933330),
        630: (0.642400000000, 0.265000000000, 0.000049999990),
        631: (0.621514900000, 0.254763200000, 0.000044160000),
        632: (0.601113800000, 0.244889600000, 0.000039480000),
        633: (0.581105200000, 0.235334400000, 0.000035720000),
        634: (0.561397700000, 0.226052800000, 0.000032640000),
        635: (0.541900000000, 0.217000000000, 0.000030000000),
        636: (0.522599500000, 0.208161600000, 0.000027653330),
        637: (0.503546400000, 0.199548800000, 0.000025560000),
        638: (0.484743600000, 0.191155200000, 0.000023640000),
        639: (0.466193900000, 0.182974400000, 0.000021813330),
        640: (0.447900000000, 0.175000000000, 0.000020000000),
        641: (0.429861300000, 0.167223500000, 0.000018133330),
        642: (0.412098000000, 0.159646400000, 0.000016200000),
        643: (0.394644000000, 0.152277600000, 0.000014200000),
        644: (0.377533300000, 0.145125900000, 0.000012133330),
        645: (0.360800000000, 0.138200000000, 0.000010000000),
        646: (0.344456300000, 0.131500300000, 0.000007733333),
        647: (0.328516800000, 0.125024800000, 0.000005400000),
        648: (0.313019200000, 0.118779200000, 0.000003200000),
        649: (0.298001100000, 0.112769100000, 0.000001333333),
        650: (0.283500000000, 0.107000000000, 0.000000000000),
        651: (0.269544800000, 0.101476200000, 0.000000000000),
        652: (0.256118400000, 0.096188640000, 0.000000000000),
        653: (0.243189600000, 0.091122960000, 0.000000000000),
        654: (0.230727200000, 0.086264850000, 0.000000000000),
        655: (0.218700000000, 0.081600000000, 0.000000000000),
        656: (0.207097100000, 0.077120640000, 0.000000000000),
        657: (0.195923200000, 0.072825520000, 0.000000000000),
        658: (0.185170800000, 0.068710080000, 0.000000000000),
        659: (0.174832300000, 0.064769760000, 0.000000000000),
        660: (0.164900000000, 0.061000000000, 0.000000000000),
        661: (0.155366700000, 0.057396210000, 0.000000000000),
        662: (0.146230000000, 0.053955040000, 0.000000000000),
        663: (0.137490000000, 0.050673760000, 0.000000000000),
        664: (0.129146700000, 0.047549650000, 0.000000000000),
        665: (0.121200000000, 0.044580000000, 0.000000000000),
        666: (0.113639700000, 0.041758720000, 0.000000000000),
        667: (0.106465000000, 0.039084960000, 0.000000000000),
        668: (0.099690440000, 0.036563840000, 0.000000000000),
        669: (0.093330610000, 0.034200480000, 0.000000000000),
        670: (0.087400000000, 0.032000000000, 0.000000000000),
        671: (0.081900960000, 0.029962610000, 0.000000000000),
        672: (0.076804280000, 0.028076640000, 0.000000000000),
        673: (0.072077120000, 0.026329360000, 0.000000000000),
        674: (0.067686640000, 0.024708050000, 0.000000000000),
        675: (0.063600000000, 0.023200000000, 0.000000000000),
        676: (0.059806850000, 0.021800770000, 0.000000000000),
        677: (0.056282160000, 0.020501120000, 0.000000000000),
        678: (0.052971040000, 0.019281080000, 0.000000000000),
        679: (0.049818610000, 0.018120690000, 0.000000000000),
        680: (0.046770000000, 0.017000000000, 0.000000000000),
        681: (0.043784050000, 0.015903790000, 0.000000000000),
        682: (0.040875360000, 0.014837180000, 0.000000000000),
        683: (0.038072640000, 0.013810680000, 0.000000000000),
        684: (0.035404610000, 0.012834780000, 0.000000000000),
        685: (0.032900000000, 0.011920000000, 0.000000000000),
        686: (0.030564190000, 0.011068310000, 0.000000000000),
        687: (0.028380560000, 0.010273390000, 0.000000000000),
        688: (0.026344840000, 0.009533311000, 0.000000000000),
        689: (0.024452750000, 0.008846157000, 0.000000000000),
        690: (0.022700000000, 0.008210000000, 0.000000000000),
        691: (0.021084290000, 0.007623781000, 0.000000000000),
        692: (0.019599880000, 0.007085424000, 0.000000000000),
        693: (0.018237320000, 0.006591476000, 0.000000000000),
        694: (0.016987170000, 0.006138485000, 0.000000000000),
        695: (0.015840000000, 0.005723000000, 0.000000000000),
        696: (0.014790640000, 0.005343059000, 0.000000000000),
        697: (0.013831320000, 0.004995796000, 0.000000000000),
        698: (0.012948680000, 0.004676404000, 0.000000000000),
        699: (0.012129200000, 0.004380075000, 0.000000000000),
        700: (0.011359160000, 0.004102000000, 0.000000000000),
        701: (0.010629350000, 0.003838453000, 0.000000000000),
        702: (0.009938846000, 0.003589099000, 0.000000000000),
        703: (0.009288422000, 0.003354219000, 0.000000000000),
        704: (0.008678854000, 0.003134093000, 0.000000000000),
        705: (0.008110916000, 0.002929000000, 0.000000000000),
        706: (0.007582388000, 0.002738139000, 0.000000000000),
        707: (0.007088746000, 0.002559876000, 0.000000000000),
        708: (0.006627313000, 0.002393244000, 0.000000000000),
        709: (0.006195408000, 0.002237275000, 0.000000000000),
        710: (0.005790346000, 0.002091000000, 0.000000000000),
        711: (0.005409826000, 0.001953587000, 0.000000000000),
        712: (0.005052583000, 0.001824580000, 0.000000000000),
        713: (0.004717512000, 0.001703580000, 0.000000000000),
        714: (0.004403507000, 0.001590187000, 0.000000000000),
        715: (0.004109457000, 0.001484000000, 0.000000000000),
        716: (0.003833913000, 0.001384496000, 0.000000000000),
        717: (0.003575748000, 0.001291268000, 0.000000000000),
        718: (0.003334342000, 0.001204092000, 0.000000000000),
        719: (0.003109075000, 0.001122744000, 0.000000000000),
        720: (0.002899327000, 0.001047000000, 0.000000000000),
        721: (0.002704348000, 0.000976589600, 0.000000000000),
        722: (0.002523020000, 0.000911108800, 0.000000000000),
        723: (0.002354168000, 0.000850133200, 0.000000000000),
        724: (0.002196616000, 0.000793238400, 0.000000000000),
        725: (0.002049190000, 0.000740000000, 0.000000000000),
        726: (0.001910960000, 0.000690082700, 0.000000000000),
        727: (0.001781438000, 0.000643310000, 0.000000000000),
        728: (0.001660110000, 0.000599496000, 0.000000000000),
        729: (0.001546459000, 0.000558454700, 0.000000000000),
        730: (0.001439971000, 0.000520000000, 0.000000000000),
        731: (0.001340042000, 0.000483913600, 0.000000000000),
        732: (0.001246275000, 0.000450052800, 0.000000000000),
        733: (0.001158471000, 0.000418345200, 0.000000000000),
        734: (0.001076430000, 0.000388718400, 0.000000000000),
        735: (0.000999949300, 0.000361100000, 0.000000000000),
        736: (0.000928735800, 0.000335383500, 0.000000000000),
        737: (0.000862433200, 0.000311440400, 0.000000000000),
        738: (0.000800750300, 0.000289165600, 0.000000000000),
        739: (0.000743396000, 0.000268453900, 0.000000000000),
        740: (0.000690078600, 0.000249200000, 0.000000000000),
        741: (0.000640515600, 0.000231301900, 0.000000000000),
        742: (0.000594502100, 0.000214685600, 0.000000000000),
        743: (0.000551864600, 0.000199288400, 0.000000000000),
        744: (0.000512429000, 0.000185047500, 0.000000000000),
        745: (0.000476021300, 0.000171900000, 0.000000000000),
        746: (0.000442453600, 0.000159778100, 0.000000000000),
        747: (0.000411511700, 0.000148604400, 0.000000000000),
        748: (0.000382981400, 0.000138301600, 0.000000000000),
        749: (0.000356649100, 0.000128792500, 0.000000000000),
        750: (0.000332301100, 0.000120000000, 0.000000000000),
        751: (0.000309758600, 0.000111859500, 0.000000000000),
        752: (0.000288887100, 0.000104322400, 0.000000000000),
        753: (0.000269539400, 0.000097335600, 0.000000000000),
        754: (0.000251568200, 0.000090845870, 0.000000000000),
        755: (0.000234826100, 0.000084800000, 0.000000000000),
        756: (0.000219171000, 0.000079146670, 0.000000000000),
        757: (0.000204525800, 0.000073858000, 0.000000000000),
        758: (0.000190840500, 0.000068916000, 0.000000000000),
        759: (0.000178065400, 0.000064302670, 0.000000000000),
        760: (0.000166150500, 0.000060000000, 0.000000000000),
        761: (0.000155023600, 0.000055981870, 0.000000000000),
        762: (0.000144621900, 0.000052225600, 0.000000000000),
        763: (0.000134909800, 0.000048718400, 0.000000000000),
        764: (0.000125852000, 0.000045447470, 0.000000000000),
        765: (0.000117413000, 0.000042400000, 0.000000000000),
        766: (0.000109551500, 0.000039561040, 0.000000000000),
        767: (0.000102224500, 0.000036915120, 0.000000000000),
        768: (0.000095394450, 0.000034448680, 0.000000000000),
        769: (0.000089023900, 0.000032148160, 0.000000000000),
        770: (0.000083075270, 0.000030000000, 0.000000000000),
        771: (0.000077512690, 0.000027991250, 0.000000000000),
        772: (0.000072313040, 0.000026113560, 0.000000000000),
        773: (0.000067457780, 0.000024360240, 0.000000000000),
        774: (0.000062928440, 0.000022724610, 0.000000000000),
        775: (0.000058706520, 0.000021200000, 0.000000000000),
        776: (0.000054770280, 0.000019778550, 0.000000000000),
        777: (0.000051099180, 0.000018452850, 0.000000000000),
        778: (0.000047676540, 0.000017216870, 0.000000000000),
        779: (0.000044485670, 0.000016064590, 0.000000000000),
        780: (0.000041509940, 0.000014990000, 0.000000000000),
        781: (0.000038733240, 0.000013987280, 0.000000000000),
        782: (0.000036142030, 0.000013051550, 0.000000000000),
        783: (0.000033723520, 0.000012178180, 0.000000000000),
        784: (0.000031464870, 0.000011362540, 0.000000000000),
        785: (0.000029353260, 0.000010600000, 0.000000000000),
        786: (0.000027375730, 0.000009885877, 0.000000000000),
        787: (0.000025524330, 0.000009217304, 0.000000000000),
        788: (0.000023793760, 0.000008592362, 0.000000000000),
        789: (0.000022178700, 0.000008009133, 0.000000000000),
        790: (0.000020673830, 0.000007465700, 0.000000000000),
        791: (0.000019272260, 0.000006959567, 0.000000000000),
        792: (0.000017966400, 0.000006487995, 0.000000000000),
        793: (0.000016749910, 0.000006048699, 0.000000000000),
        794: (0.000015616480, 0.000005639396, 0.000000000000),
        795: (0.000014559770, 0.000005257800, 0.000000000000),
        796: (0.000013573870, 0.000004901771, 0.000000000000),
        797: (0.000012654360, 0.000004569720, 0.000000000000),
        798: (0.000011797230, 0.000004260194, 0.000000000000),
        799: (0.000010998440, 0.000003971739, 0.000000000000),
        800: (0.000010253980, 0.000003702900, 0.000000000000),
        801: (0.000009559646, 0.000003452163, 0.000000000000),
        802: (0.000008912044, 0.000003218302, 0.000000000000),
        803: (0.000008308358, 0.000003000300, 0.000000000000),
        804: (0.000007745769, 0.000002797139, 0.000000000000),
        805: (0.000007221456, 0.000002607800, 0.000000000000),
        806: (0.000006732475, 0.000002431220, 0.000000000000),
        807: (0.000006276423, 0.000002266531, 0.000000000000),
        808: (0.000005851304, 0.000002113013, 0.000000000000),
        809: (0.000005455118, 0.000001969943, 0.000000000000),
        810: (0.000005085868, 0.000001836600, 0.000000000000),
        811: (0.000004741466, 0.000001712230, 0.000000000000),
        812: (0.000004420236, 0.000001596228, 0.000000000000),
        813: (0.000004120783, 0.000001488090, 0.000000000000),
        814: (0.000003841716, 0.000001387314, 0.000000000000),
        815: (0.000003581652, 0.000001293400, 0.000000000000),
        816: (0.000003339127, 0.000001205820, 0.000000000000),
        817: (0.000003112949, 0.000001124143, 0.000000000000),
        818: (0.000002902121, 0.000001048009, 0.000000000000),
        819: (0.000002705645, 0.000000977058, 0.000000000000),
        820: (0.000002522525, 0.000000910930, 0.000000000000),
        821: (0.000002351726, 0.000000849251, 0.000000000000),
        822: (0.000002192415, 0.000000791721, 0.000000000000),
        823: (0.000002043902, 0.000000738090, 0.000000000000),
        824: (0.000001905497, 0.000000688110, 0.000000000000),
        825: (0.000001776509, 0.000000641530, 0.000000000000),
        826: (0.000001656215, 0.000000598090, 0.000000000000),
        827: (0.000001544022, 0.000000557575, 0.000000000000),
        828: (0.000001439440, 0.000000519808, 0.000000000000),
        829: (0.000001341977, 0.000000484612, 0.000000000000),
        830: (0.000001251141, 0.000000451810, 0.000000000000)
    },
    'CIE 1964 10 Degree Standard Observer': {
        360: (0.000000122200, 0.000000013398, 0.000000535027),
        361: (0.000000185138, 0.000000020294, 0.000000810720),
        362: (0.000000278830, 0.000000030560, 0.000001221200),
        363: (0.000000417470, 0.000000045740, 0.000001828700),
        364: (0.000000621330, 0.000000068050, 0.000002722200),
        365: (0.000000919270, 0.000000100650, 0.000004028300),
        366: (0.000001351980, 0.000000147980, 0.000005925700),
        367: (0.000001976540, 0.000000216270, 0.000008665100),
        368: (0.000002872500, 0.000000314200, 0.000012596000),
        369: (0.000004149500, 0.000000453700, 0.000018201000),
        370: (0.000005958600, 0.000000651100, 0.000026143700),
        371: (0.000008505600, 0.000000928800, 0.000037330000),
        372: (0.000012068600, 0.000001317500, 0.000052987000),
        373: (0.000017022600, 0.000001857200, 0.000074764000),
        374: (0.000023868000, 0.000002602000, 0.000104870000),
        375: (0.000033266000, 0.000003625000, 0.000146220000),
        376: (0.000046087000, 0.000005019000, 0.000202660000),
        377: (0.000063472000, 0.000006907000, 0.000279230000),
        378: (0.000086892000, 0.000009449000, 0.000382450000),
        379: (0.000118246000, 0.000012848000, 0.000520720000),
        380: (0.000159952000, 0.000017364000, 0.000704776000),
        381: (0.000215080000, 0.000023327000, 0.000948230000),
        382: (0.000287490000, 0.000031150000, 0.001268200000),
        383: (0.000381990000, 0.000041350000, 0.001686100000),
        384: (0.000504550000, 0.000054560000, 0.002228500000),
        385: (0.000662440000, 0.000071560000, 0.002927800000),
        386: (0.000864500000, 0.000093300000, 0.003823700000),
        387: (0.001121500000, 0.000120870000, 0.004964200000),
        388: (0.001446160000, 0.000155640000, 0.006406700000),
        389: (0.001853590000, 0.000199200000, 0.008219300000),
        390: (0.002361600000, 0.000253400000, 0.010482200000),
        391: (0.002990600000, 0.000320200000, 0.013289000000),
        392: (0.003764500000, 0.000402400000, 0.016747000000),
        393: (0.004710200000, 0.000502300000, 0.020980000000),
        394: (0.005858100000, 0.000623200000, 0.026127000000),
        395: (0.007242300000, 0.000768500000, 0.032344000000),
        396: (0.008899600000, 0.000941700000, 0.039802000000),
        397: (0.010870900000, 0.001147800000, 0.048691000000),
        398: (0.013198900000, 0.001390300000, 0.059210000000),
        399: (0.015929200000, 0.001674000000, 0.071576000000),
        400: (0.019109700000, 0.002004400000, 0.086010900000),
        401: (0.022788000000, 0.002386000000, 0.102740000000),
        402: (0.027011000000, 0.002822000000, 0.122000000000),
        403: (0.031829000000, 0.003319000000, 0.144020000000),
        404: (0.037278000000, 0.003880000000, 0.168990000000),
        405: (0.043400000000, 0.004509000000, 0.197120000000),
        406: (0.050223000000, 0.005209000000, 0.228570000000),
        407: (0.057764000000, 0.005985000000, 0.263470000000),
        408: (0.066038000000, 0.006833000000, 0.301900000000),
        409: (0.075033000000, 0.007757000000, 0.343870000000),
        410: (0.084736000000, 0.008756000000, 0.389366000000),
        411: (0.095041000000, 0.009816000000, 0.437970000000),
        412: (0.105836000000, 0.010918000000, 0.489220000000),
        413: (0.117066000000, 0.012058000000, 0.542900000000),
        414: (0.128682000000, 0.013237000000, 0.598810000000),
        415: (0.140638000000, 0.014456000000, 0.656760000000),
        416: (0.152893000000, 0.015717000000, 0.716580000000),
        417: (0.165416000000, 0.017025000000, 0.778120000000),
        418: (0.178191000000, 0.018399000000, 0.841310000000),
        419: (0.191214000000, 0.019848000000, 0.906110000000),
        420: (0.204492000000, 0.021391000000, 0.972542000000),
        421: (0.217650000000, 0.022992000000, 1.038900000000),
        422: (0.230267000000, 0.024598000000, 1.103100000000),
        423: (0.242311000000, 0.026213000000, 1.165100000000),
        424: (0.253793000000, 0.027841000000, 1.224900000000),
        425: (0.264737000000, 0.029497000000, 1.282500000000),
        426: (0.275195000000, 0.031195000000, 1.338200000000),
        427: (0.285301000000, 0.032927000000, 1.392600000000),
        428: (0.295143000000, 0.034738000000, 1.446100000000),
        429: (0.304869000000, 0.036654000000, 1.499400000000),
        430: (0.314679000000, 0.038676000000, 1.553480000000),
        431: (0.324355000000, 0.040792000000, 1.607200000000),
        432: (0.333570000000, 0.042946000000, 1.658900000000),
        433: (0.342243000000, 0.045114000000, 1.708200000000),
        434: (0.350312000000, 0.047333000000, 1.754800000000),
        435: (0.357719000000, 0.049602000000, 1.798500000000),
        436: (0.364482000000, 0.051934000000, 1.839200000000),
        437: (0.370493000000, 0.054337000000, 1.876600000000),
        438: (0.375727000000, 0.056822000000, 1.910500000000),
        439: (0.380158000000, 0.059399000000, 1.940800000000),
        440: (0.383734000000, 0.062077000000, 1.967280000000),
        441: (0.386327000000, 0.064737000000, 1.989100000000),
        442: (0.387858000000, 0.067285000000, 2.005700000000),
        443: (0.388396000000, 0.069764000000, 2.017400000000),
        444: (0.387978000000, 0.072218000000, 2.024400000000),
        445: (0.386726000000, 0.074704000000, 2.027300000000),
        446: (0.384696000000, 0.077272000000, 2.026400000000),
        447: (0.382006000000, 0.079979000000, 2.022300000000),
        448: (0.378709000000, 0.082874000000, 2.015300000000),
        449: (0.374915000000, 0.086000000000, 2.006000000000),
        450: (0.370702000000, 0.089456000000, 1.994800000000),
        451: (0.366089000000, 0.092947000000, 1.981400000000),
        452: (0.361045000000, 0.096275000000, 1.965300000000),
        453: (0.355518000000, 0.099535000000, 1.946400000000),
        454: (0.349486000000, 0.102829000000, 1.924800000000),
        455: (0.342957000000, 0.106256000000, 1.900700000000),
        456: (0.335893000000, 0.109901000000, 1.874100000000),
        457: (0.328284000000, 0.113835000000, 1.845100000000),
        458: (0.320150000000, 0.118167000000, 1.813900000000),
        459: (0.311475000000, 0.122932000000, 1.780600000000),
        460: (0.302273000000, 0.128201000000, 1.745370000000),
        461: (0.292858000000, 0.133457000000, 1.709100000000),
        462: (0.283502000000, 0.138323000000, 1.672300000000),
        463: (0.274044000000, 0.143042000000, 1.634700000000),
        464: (0.264263000000, 0.147787000000, 1.595600000000),
        465: (0.254085000000, 0.152761000000, 1.554900000000),
        466: (0.243392000000, 0.158102000000, 1.512200000000),
        467: (0.232187000000, 0.163941000000, 1.467300000000),
        468: (0.220488000000, 0.170362000000, 1.419900000000),
        469: (0.208198000000, 0.177425000000, 1.370000000000),
        470: (0.195618000000, 0.185190000000, 1.317560000000),
        471: (0.183034000000, 0.193025000000, 1.262400000000),
        472: (0.170222000000, 0.200313000000, 1.205000000000),
        473: (0.157348000000, 0.207156000000, 1.146600000000),
        474: (0.144650000000, 0.213644000000, 1.088000000000),
        475: (0.132349000000, 0.219940000000, 1.030200000000),
        476: (0.120584000000, 0.226170000000, 0.973830000000),
        477: (0.109456000000, 0.232467000000, 0.919430000000),
        478: (0.099042000000, 0.239025000000, 0.867460000000),
        479: (0.089388000000, 0.245997000000, 0.818280000000),
        480: (0.080507000000, 0.253589000000, 0.772125000000),
        481: (0.072034000000, 0.261876000000, 0.728290000000),
        482: (0.063710000000, 0.270643000000, 0.686040000000),
        483: (0.055694000000, 0.279645000000, 0.645530000000),
        484: (0.048117000000, 0.288694000000, 0.606850000000),
        485: (0.041072000000, 0.297665000000, 0.570060000000),
        486: (0.034642000000, 0.306469000000, 0.535220000000),
        487: (0.028896000000, 0.315035000000, 0.502340000000),
        488: (0.023876000000, 0.323335000000, 0.471400000000),
        489: (0.019628000000, 0.331366000000, 0.442390000000),
        490: (0.016172000000, 0.339133000000, 0.415254000000),
        491: (0.013300000000, 0.347860000000, 0.390024000000),
        492: (0.010759000000, 0.358326000000, 0.366399000000),
        493: (0.008542000000, 0.370001000000, 0.344015000000),
        494: (0.006661000000, 0.382464000000, 0.322689000000),
        495: (0.005132000000, 0.395379000000, 0.302356000000),
        496: (0.003982000000, 0.408482000000, 0.283036000000),
        497: (0.003239000000, 0.421588000000, 0.264816000000),
        498: (0.002934000000, 0.434619000000, 0.247848000000),
        499: (0.003114000000, 0.447601000000, 0.232318000000),
        500: (0.003816000000, 0.460777000000, 0.218502000000),
        501: (0.005095000000, 0.474340000000, 0.205851000000),
        502: (0.006936000000, 0.488200000000, 0.193596000000),
        503: (0.009299000000, 0.502340000000, 0.181736000000),
        504: (0.012147000000, 0.516740000000, 0.170281000000),
        505: (0.015444000000, 0.531360000000, 0.159249000000),
        506: (0.019156000000, 0.546190000000, 0.148673000000),
        507: (0.023250000000, 0.561180000000, 0.138609000000),
        508: (0.027690000000, 0.576290000000, 0.129096000000),
        509: (0.032444000000, 0.591500000000, 0.120215000000),
        510: (0.037465000000, 0.606741000000, 0.112044000000),
        511: (0.042956000000, 0.622150000000, 0.104710000000),
        512: (0.049114000000, 0.637830000000, 0.098196000000),
        513: (0.055920000000, 0.653710000000, 0.092361000000),
        514: (0.063349000000, 0.669680000000, 0.087088000000),
        515: (0.071358000000, 0.685660000000, 0.082248000000),
        516: (0.079901000000, 0.701550000000, 0.077744000000),
        517: (0.088909000000, 0.717230000000, 0.073456000000),
        518: (0.098293000000, 0.732570000000, 0.069268000000),
        519: (0.107949000000, 0.747460000000, 0.065060000000),
        520: (0.117749000000, 0.761757000000, 0.060709000000),
        521: (0.127839000000, 0.775340000000, 0.056457000000),
        522: (0.138450000000, 0.788220000000, 0.052609000000),
        523: (0.149516000000, 0.800460000000, 0.049122000000),
        524: (0.161041000000, 0.812140000000, 0.045954000000),
        525: (0.172953000000, 0.823330000000, 0.043050000000),
        526: (0.185209000000, 0.834120000000, 0.040368000000),
        527: (0.197755000000, 0.844600000000, 0.037839000000),
        528: (0.210538000000, 0.854870000000, 0.035384000000),
        529: (0.223460000000, 0.865040000000, 0.032949000000),
        530: (0.236491000000, 0.875211000000, 0.030451000000),
        531: (0.249633000000, 0.885370000000, 0.028029000000),
        532: (0.262972000000, 0.895370000000, 0.025862000000),
        533: (0.276515000000, 0.905150000000, 0.023920000000),
        534: (0.290269000000, 0.914650000000, 0.022174000000),
        535: (0.304213000000, 0.923810000000, 0.020584000000),
        536: (0.318361000000, 0.932550000000, 0.019127000000),
        537: (0.332705000000, 0.940810000000, 0.017740000000),
        538: (0.347232000000, 0.948520000000, 0.016403000000),
        539: (0.361926000000, 0.955600000000, 0.015064000000),
        540: (0.376772000000, 0.961988000000, 0.013676000000),
        541: (0.391683000000, 0.967540000000, 0.012308000000),
        542: (0.406594000000, 0.972230000000, 0.011056000000),
        543: (0.421539000000, 0.976170000000, 0.009915000000),
        544: (0.436517000000, 0.979460000000, 0.008872000000),
        545: (0.451584000000, 0.982200000000, 0.007918000000),
        546: (0.466782000000, 0.984520000000, 0.007030000000),
        547: (0.482147000000, 0.986520000000, 0.006223000000),
        548: (0.497738000000, 0.988320000000, 0.005453000000),
        549: (0.513606000000, 0.990020000000, 0.004714000000),
        550: (0.529826000000, 0.991761000000, 0.003988000000),
        551: (0.546440000000, 0.993530000000, 0.003289000000),
        552: (0.563426000000, 0.995230000000, 0.002646000000),
        553: (0.580726000000, 0.996770000000, 0.002063000000),
        554: (0.598290000000, 0.998090000000, 0.001533000000),
        555: (0.616053000000, 0.999110000000, 0.001091000000),
        556: (0.633948000000, 0.999770000000, 0.000711000000),
        557: (0.651901000000, 1.000000000000, 0.000407000000),
        558: (0.669824000000, 0.999710000000, 0.000184000000),
        559: (0.687632000000, 0.998850000000, 0.000047000000),
        560: (0.705224000000, 0.997340000000, 0.000000000000),
        561: (0.722773000000, 0.995260000000, 0.000000000000),
        562: (0.740483000000, 0.992740000000, 0.000000000000),
        563: (0.758273000000, 0.989750000000, 0.000000000000),
        564: (0.776083000000, 0.986300000000, 0.000000000000),
        565: (0.793832000000, 0.982380000000, 0.000000000000),
        566: (0.811436000000, 0.977980000000, 0.000000000000),
        567: (0.828822000000, 0.973110000000, 0.000000000000),
        568: (0.845879000000, 0.967740000000, 0.000000000000),
        569: (0.862525000000, 0.961890000000, 0.000000000000),
        570: (0.878655000000, 0.955552000000, 0.000000000000),
        571: (0.894208000000, 0.948601000000, 0.000000000000),
        572: (0.909206000000, 0.940981000000, 0.000000000000),
        573: (0.923672000000, 0.932798000000, 0.000000000000),
        574: (0.937638000000, 0.924158000000, 0.000000000000),
        575: (0.951162000000, 0.915175000000, 0.000000000000),
        576: (0.964283000000, 0.905954000000, 0.000000000000),
        577: (0.977068000000, 0.896608000000, 0.000000000000),
        578: (0.989590000000, 0.887249000000, 0.000000000000),
        579: (1.001910000000, 0.877986000000, 0.000000000000),
        580: (1.014160000000, 0.868934000000, 0.000000000000),
        581: (1.026500000000, 0.860164000000, 0.000000000000),
        582: (1.038800000000, 0.851519000000, 0.000000000000),
        583: (1.051000000000, 0.842963000000, 0.000000000000),
        584: (1.062900000000, 0.834393000000, 0.000000000000),
        585: (1.074300000000, 0.825623000000, 0.000000000000),
        586: (1.085200000000, 0.816764000000, 0.000000000000),
        587: (1.095200000000, 0.807544000000, 0.000000000000),
        588: (1.104200000000, 0.797947000000, 0.000000000000),
        589: (1.112000000000, 0.787893000000, 0.000000000000),
        590: (1.118520000000, 0.777405000000, 0.000000000000),
        591: (1.123800000000, 0.766490000000, 0.000000000000),
        592: (1.128000000000, 0.755309000000, 0.000000000000),
        593: (1.131100000000, 0.743845000000, 0.000000000000),
        594: (1.133200000000, 0.732190000000, 0.000000000000),
        595: (1.134300000000, 0.720353000000, 0.000000000000),
        596: (1.134300000000, 0.708281000000, 0.000000000000),
        597: (1.133300000000, 0.696055000000, 0.000000000000),
        598: (1.131200000000, 0.683621000000, 0.000000000000),
        599: (1.128100000000, 0.671048000000, 0.000000000000),
        600: (1.123990000000, 0.658341000000, 0.000000000000),
        601: (1.118900000000, 0.645545000000, 0.000000000000),
        602: (1.112900000000, 0.632718000000, 0.000000000000),
        603: (1.105900000000, 0.619815000000, 0.000000000000),
        604: (1.098000000000, 0.606887000000, 0.000000000000),
        605: (1.089100000000, 0.593878000000, 0.000000000000),
        606: (1.079200000000, 0.580781000000, 0.000000000000),
        607: (1.068400000000, 0.567653000000, 0.000000000000),
        608: (1.056700000000, 0.554490000000, 0.000000000000),
        609: (1.044000000000, 0.541228000000, 0.000000000000),
        610: (1.030480000000, 0.527963000000, 0.000000000000),
        611: (1.016000000000, 0.514634000000, 0.000000000000),
        612: (1.000800000000, 0.501363000000, 0.000000000000),
        613: (0.984790000000, 0.488124000000, 0.000000000000),
        614: (0.968080000000, 0.474935000000, 0.000000000000),
        615: (0.950740000000, 0.461834000000, 0.000000000000),
        616: (0.932800000000, 0.448823000000, 0.000000000000),
        617: (0.914340000000, 0.435917000000, 0.000000000000),
        618: (0.895390000000, 0.423153000000, 0.000000000000),
        619: (0.876030000000, 0.410526000000, 0.000000000000),
        620: (0.856297000000, 0.398057000000, 0.000000000000),
        621: (0.836350000000, 0.385835000000, 0.000000000000),
        622: (0.816290000000, 0.373951000000, 0.000000000000),
        623: (0.796050000000, 0.362311000000, 0.000000000000),
        624: (0.775610000000, 0.350863000000, 0.000000000000),
        625: (0.754930000000, 0.339554000000, 0.000000000000),
        626: (0.733990000000, 0.328309000000, 0.000000000000),
        627: (0.712780000000, 0.317118000000, 0.000000000000),
        628: (0.691290000000, 0.305936000000, 0.000000000000),
        629: (0.669520000000, 0.294737000000, 0.000000000000),
        630: (0.647467000000, 0.283493000000, 0.000000000000),
        631: (0.625110000000, 0.272222000000, 0.000000000000),
        632: (0.602520000000, 0.260990000000, 0.000000000000),
        633: (0.579890000000, 0.249877000000, 0.000000000000),
        634: (0.557370000000, 0.238946000000, 0.000000000000),
        635: (0.535110000000, 0.228254000000, 0.000000000000),
        636: (0.513240000000, 0.217853000000, 0.000000000000),
        637: (0.491860000000, 0.207780000000, 0.000000000000),
        638: (0.471080000000, 0.198072000000, 0.000000000000),
        639: (0.450960000000, 0.188748000000, 0.000000000000),
        640: (0.431567000000, 0.179828000000, 0.000000000000),
        641: (0.412870000000, 0.171285000000, 0.000000000000),
        642: (0.394750000000, 0.163059000000, 0.000000000000),
        643: (0.377210000000, 0.155151000000, 0.000000000000),
        644: (0.360190000000, 0.147535000000, 0.000000000000),
        645: (0.343690000000, 0.140211000000, 0.000000000000),
        646: (0.327690000000, 0.133170000000, 0.000000000000),
        647: (0.312170000000, 0.126400000000, 0.000000000000),
        648: (0.297110000000, 0.119892000000, 0.000000000000),
        649: (0.282500000000, 0.113640000000, 0.000000000000),
        650: (0.268329000000, 0.107633000000, 0.000000000000),
        651: (0.254590000000, 0.101870000000, 0.000000000000),
        652: (0.241300000000, 0.096347000000, 0.000000000000),
        653: (0.228480000000, 0.091063000000, 0.000000000000),
        654: (0.216140000000, 0.086010000000, 0.000000000000),
        655: (0.204300000000, 0.081187000000, 0.000000000000),
        656: (0.192950000000, 0.076583000000, 0.000000000000),
        657: (0.182110000000, 0.072198000000, 0.000000000000),
        658: (0.171770000000, 0.068024000000, 0.000000000000),
        659: (0.161920000000, 0.064052000000, 0.000000000000),
        660: (0.152568000000, 0.060281000000, 0.000000000000),
        661: (0.143670000000, 0.056697000000, 0.000000000000),
        662: (0.135200000000, 0.053292000000, 0.000000000000),
        663: (0.127130000000, 0.050059000000, 0.000000000000),
        664: (0.119480000000, 0.046998000000, 0.000000000000),
        665: (0.112210000000, 0.044096000000, 0.000000000000),
        666: (0.105310000000, 0.041345000000, 0.000000000000),
        667: (0.098786000000, 0.038750700000, 0.000000000000),
        668: (0.092610000000, 0.036297800000, 0.000000000000),
        669: (0.086773000000, 0.033983200000, 0.000000000000),
        670: (0.081260600000, 0.031800400000, 0.000000000000),
        671: (0.076048000000, 0.029739500000, 0.000000000000),
        672: (0.071114000000, 0.027791800000, 0.000000000000),
        673: (0.066454000000, 0.025955100000, 0.000000000000),
        674: (0.062062000000, 0.024226300000, 0.000000000000),
        675: (0.057930000000, 0.022601700000, 0.000000000000),
        676: (0.054050000000, 0.021077900000, 0.000000000000),
        677: (0.050412000000, 0.019650500000, 0.000000000000),
        678: (0.047006000000, 0.018315300000, 0.000000000000),
        679: (0.043823000000, 0.017068600000, 0.000000000000),
        680: (0.040850800000, 0.015905100000, 0.000000000000),
        681: (0.038072000000, 0.014818300000, 0.000000000000),
        682: (0.035468000000, 0.013800800000, 0.000000000000),
        683: (0.033031000000, 0.012849500000, 0.000000000000),
        684: (0.030753000000, 0.011960700000, 0.000000000000),
        685: (0.028623000000, 0.011130300000, 0.000000000000),
        686: (0.026635000000, 0.010355500000, 0.000000000000),
        687: (0.024781000000, 0.009633200000, 0.000000000000),
        688: (0.023052000000, 0.008959900000, 0.000000000000),
        689: (0.021441000000, 0.008332400000, 0.000000000000),
        690: (0.019941300000, 0.007748800000, 0.000000000000),
        691: (0.018544000000, 0.007204600000, 0.000000000000),
        692: (0.017241000000, 0.006697500000, 0.000000000000),
        693: (0.016027000000, 0.006225100000, 0.000000000000),
        694: (0.014896000000, 0.005785000000, 0.000000000000),
        695: (0.013842000000, 0.005375100000, 0.000000000000),
        696: (0.012862000000, 0.004994100000, 0.000000000000),
        697: (0.011949000000, 0.004639200000, 0.000000000000),
        698: (0.011100000000, 0.004309300000, 0.000000000000),
        699: (0.010311000000, 0.004002800000, 0.000000000000),
        700: (0.009576880000, 0.003717740000, 0.000000000000),
        701: (0.008894000000, 0.003452620000, 0.000000000000),
        702: (0.008258100000, 0.003205830000, 0.000000000000),
        703: (0.007666400000, 0.002976230000, 0.000000000000),
        704: (0.007116300000, 0.002762810000, 0.000000000000),
        705: (0.006605200000, 0.002564560000, 0.000000000000),
        706: (0.006130600000, 0.002380480000, 0.000000000000),
        707: (0.005690300000, 0.002209710000, 0.000000000000),
        708: (0.005281900000, 0.002051320000, 0.000000000000),
        709: (0.004903300000, 0.001904490000, 0.000000000000),
        710: (0.004552630000, 0.001768470000, 0.000000000000),
        711: (0.004227500000, 0.001642360000, 0.000000000000),
        712: (0.003925800000, 0.001525350000, 0.000000000000),
        713: (0.003645700000, 0.001416720000, 0.000000000000),
        714: (0.003385900000, 0.001315950000, 0.000000000000),
        715: (0.003144700000, 0.001222390000, 0.000000000000),
        716: (0.002920800000, 0.001135550000, 0.000000000000),
        717: (0.002713000000, 0.001054940000, 0.000000000000),
        718: (0.002520200000, 0.000980140000, 0.000000000000),
        719: (0.002341100000, 0.000910660000, 0.000000000000),
        720: (0.002174960000, 0.000846190000, 0.000000000000),
        721: (0.002020600000, 0.000786290000, 0.000000000000),
        722: (0.001877300000, 0.000730680000, 0.000000000000),
        723: (0.001744100000, 0.000678990000, 0.000000000000),
        724: (0.001620500000, 0.000631010000, 0.000000000000),
        725: (0.001505700000, 0.000586440000, 0.000000000000),
        726: (0.001399200000, 0.000545110000, 0.000000000000),
        727: (0.001300400000, 0.000506720000, 0.000000000000),
        728: (0.001208700000, 0.000471110000, 0.000000000000),
        729: (0.001123600000, 0.000438050000, 0.000000000000),
        730: (0.001044760000, 0.000407410000, 0.000000000000),
        731: (0.000971560000, 0.000378962000, 0.000000000000),
        732: (0.000903600000, 0.000352543000, 0.000000000000),
        733: (0.000840480000, 0.000328001000, 0.000000000000),
        734: (0.000781870000, 0.000305208000, 0.000000000000),
        735: (0.000727450000, 0.000284041000, 0.000000000000),
        736: (0.000676900000, 0.000264375000, 0.000000000000),
        737: (0.000629960000, 0.000246109000, 0.000000000000),
        738: (0.000586370000, 0.000229143000, 0.000000000000),
        739: (0.000545870000, 0.000213376000, 0.000000000000),
        740: (0.000508258000, 0.000198730000, 0.000000000000),
        741: (0.000473300000, 0.000185115000, 0.000000000000),
        742: (0.000440800000, 0.000172454000, 0.000000000000),
        743: (0.000410580000, 0.000160678000, 0.000000000000),
        744: (0.000382490000, 0.000149730000, 0.000000000000),
        745: (0.000356380000, 0.000139550000, 0.000000000000),
        746: (0.000332110000, 0.000130086000, 0.000000000000),
        747: (0.000309550000, 0.000121290000, 0.000000000000),
        748: (0.000288580000, 0.000113106000, 0.000000000000),
        749: (0.000269090000, 0.000105501000, 0.000000000000),
        750: (0.000250969000, 0.000098428000, 0.000000000000),
        751: (0.000234130000, 0.000091853000, 0.000000000000),
        752: (0.000218470000, 0.000085738000, 0.000000000000),
        753: (0.000203910000, 0.000080048000, 0.000000000000),
        754: (0.000190350000, 0.000074751000, 0.000000000000),
        755: (0.000177730000, 0.000069819000, 0.000000000000),
        756: (0.000165970000, 0.000065222000, 0.000000000000),
        757: (0.000155020000, 0.000060939000, 0.000000000000),
        758: (0.000144800000, 0.000056942000, 0.000000000000),
        759: (0.000135280000, 0.000053217000, 0.000000000000),
        760: (0.000126390000, 0.000049737000, 0.000000000000),
        761: (0.000118100000, 0.000046491000, 0.000000000000),
        762: (0.000110370000, 0.000043464000, 0.000000000000),
        763: (0.000103150000, 0.000040635000, 0.000000000000),
        764: (0.000096427000, 0.000038000000, 0.000000000000),
        765: (0.000090151000, 0.000035540500, 0.000000000000),
        766: (0.000084294000, 0.000033244800, 0.000000000000),
        767: (0.000078830000, 0.000031100600, 0.000000000000),
        768: (0.000073729000, 0.000029099000, 0.000000000000),
        769: (0.000068969000, 0.000027230700, 0.000000000000),
        770: (0.000064525800, 0.000025486000, 0.000000000000),
        771: (0.000060376000, 0.000023856100, 0.000000000000),
        772: (0.000056500000, 0.000022333200, 0.000000000000),
        773: (0.000052880000, 0.000020910400, 0.000000000000),
        774: (0.000049498000, 0.000019580800, 0.000000000000),
        775: (0.000046339000, 0.000018338400, 0.000000000000),
        776: (0.000043389000, 0.000017177700, 0.000000000000),
        777: (0.000040634000, 0.000016093400, 0.000000000000),
        778: (0.000038060000, 0.000015080000, 0.000000000000),
        779: (0.000035657000, 0.000014133600, 0.000000000000),
        780: (0.000033411700, 0.000013249000, 0.000000000000),
        781: (0.000031315000, 0.000012422600, 0.000000000000),
        782: (0.000029355000, 0.000011649900, 0.000000000000),
        783: (0.000027524000, 0.000010927700, 0.000000000000),
        784: (0.000025811000, 0.000010251900, 0.000000000000),
        785: (0.000024209000, 0.000009619600, 0.000000000000),
        786: (0.000022711000, 0.000009028100, 0.000000000000),
        787: (0.000021308000, 0.000008474000, 0.000000000000),
        788: (0.000019994000, 0.000007954800, 0.000000000000),
        789: (0.000018764000, 0.000007468600, 0.000000000000),
        790: (0.000017611500, 0.000007012800, 0.000000000000),
        791: (0.000016532000, 0.000006585800, 0.000000000000),
        792: (0.000015521000, 0.000006185700, 0.000000000000),
        793: (0.000014574000, 0.000005810700, 0.000000000000),
        794: (0.000013686000, 0.000005459000, 0.000000000000),
        795: (0.000012855000, 0.000005129800, 0.000000000000),
        796: (0.000012075000, 0.000004820600, 0.000000000000),
        797: (0.000011345000, 0.000004531200, 0.000000000000),
        798: (0.000010659000, 0.000004259100, 0.000000000000),
        799: (0.000010017000, 0.000004004200, 0.000000000000),
        800: (0.000009413630, 0.000003764730, 0.000000000000),
        801: (0.000008847900, 0.000003539950, 0.000000000000),
        802: (0.000008317100, 0.000003329140, 0.000000000000),
        803: (0.000007819000, 0.000003131150, 0.000000000000),
        804: (0.000007351600, 0.000002945290, 0.000000000000),
        805: (0.000006913000, 0.000002770810, 0.000000000000),
        806: (0.000006501500, 0.000002607050, 0.000000000000),
        807: (0.000006115300, 0.000002453290, 0.000000000000),
        808: (0.000005752900, 0.000002308940, 0.000000000000),
        809: (0.000005412700, 0.000002173380, 0.000000000000),
        810: (0.000005093470, 0.000002046130, 0.000000000000),
        811: (0.000004793800, 0.000001926620, 0.000000000000),
        812: (0.000004512500, 0.000001814400, 0.000000000000),
        813: (0.000004248300, 0.000001708950, 0.000000000000),
        814: (0.000004000200, 0.000001609880, 0.000000000000),
        815: (0.000003767100, 0.000001516770, 0.000000000000),
        816: (0.000003548000, 0.000001429210, 0.000000000000),
        817: (0.000003342100, 0.000001346860, 0.000000000000),
        818: (0.000003148500, 0.000001269450, 0.000000000000),
        819: (0.000002966500, 0.000001196620, 0.000000000000),
        820: (0.000002795310, 0.000001128090, 0.000000000000),
        821: (0.000002634500, 0.000001063680, 0.000000000000),
        822: (0.000002483400, 0.000001003130, 0.000000000000),
        823: (0.000002341400, 0.000000946220, 0.000000000000),
        824: (0.000002207800, 0.000000892630, 0.000000000000),
        825: (0.000002082000, 0.000000842160, 0.000000000000),
        826: (0.000001963600, 0.000000794640, 0.000000000000),
        827: (0.000001851900, 0.000000749780, 0.000000000000),
        828: (0.000001746500, 0.000000707440, 0.000000000000),
        829: (0.000001647100, 0.000000667480, 0.000000000000),
        830: (0.000001553140, 0.000000629700, 0.000000000000)
    },
    'CIE 2012 2 Degree Standard Observer': {
        390: (3.769647e-03, 4.146161e-04, 1.847260e-02),
        391: (4.532416e-03, 5.028333e-04, 2.221101e-02),
        392: (5.446553e-03, 6.084991e-04, 2.669819e-02),
        393: (6.538868e-03, 7.344436e-04, 3.206937e-02),
        394: (7.839699e-03, 8.837389e-04, 3.847832e-02),
        395: (9.382967e-03, 1.059646e-03, 4.609784e-02),
        396: (1.120608e-02, 1.265532e-03, 5.511953e-02),
        397: (1.334965e-02, 1.504753e-03, 6.575257e-02),
        398: (1.585690e-02, 1.780493e-03, 7.822113e-02),
        399: (1.877286e-02, 2.095572e-03, 9.276013e-02),
        400: (2.214302e-02, 2.452194e-03, 1.096090e-01),
        401: (2.601285e-02, 2.852216e-03, 1.290077e-01),
        402: (3.043036e-02, 3.299115e-03, 1.512047e-01),
        403: (3.544325e-02, 3.797466e-03, 1.764441e-01),
        404: (4.109640e-02, 4.352768e-03, 2.049517e-01),
        405: (4.742986e-02, 4.971717e-03, 2.369246e-01),
        406: (5.447394e-02, 5.661014e-03, 2.725123e-01),
        407: (6.223612e-02, 6.421615e-03, 3.117820e-01),
        408: (7.070048e-02, 7.250312e-03, 3.547064e-01),
        409: (7.982513e-02, 8.140173e-03, 4.011473e-01),
        410: (8.953803e-02, 9.079860e-03, 4.508369e-01),
        411: (9.974848e-02, 1.005608e-02, 5.034164e-01),
        412: (1.104019e-01, 1.106456e-02, 5.586361e-01),
        413: (1.214566e-01, 1.210522e-02, 6.162734e-01),
        414: (1.328741e-01, 1.318014e-02, 6.760982e-01),
        415: (1.446214e-01, 1.429377e-02, 7.378822e-01),
        416: (1.566468e-01, 1.545004e-02, 8.013019e-01),
        417: (1.687901e-01, 1.664093e-02, 8.655573e-01),
        418: (1.808328e-01, 1.785302e-02, 9.295791e-01),
        419: (1.925216e-01, 1.907018e-02, 9.921293e-01),
        420: (2.035729e-01, 2.027369e-02, 1.051821e+00),
        421: (2.137531e-01, 2.144805e-02, 1.107509e+00),
        422: (2.231348e-01, 2.260041e-02, 1.159527e+00),
        423: (2.319245e-01, 2.374789e-02, 1.208869e+00),
        424: (2.403892e-01, 2.491247e-02, 1.256834e+00),
        425: (2.488523e-01, 2.612106e-02, 1.305008e+00),
        426: (2.575896e-01, 2.739923e-02, 1.354758e+00),
        427: (2.664991e-01, 2.874993e-02, 1.405594e+00),
        428: (2.753532e-01, 3.016909e-02, 1.456414e+00),
        429: (2.838921e-01, 3.165145e-02, 1.505960e+00),
        430: (2.918246e-01, 3.319038e-02, 1.552826e+00),
        431: (2.989200e-01, 3.477912e-02, 1.595902e+00),
        432: (3.052993e-01, 3.641495e-02, 1.635768e+00),
        433: (3.112031e-01, 3.809569e-02, 1.673573e+00),
        434: (3.169047e-01, 3.981843e-02, 1.710604e+00),
        435: (3.227087e-01, 4.157940e-02, 1.748280e+00),
        436: (3.288194e-01, 4.337098e-02, 1.787504e+00),
        437: (3.349242e-01, 4.517180e-02, 1.826609e+00),
        438: (3.405452e-01, 4.695420e-02, 1.863108e+00),
        439: (3.451688e-01, 4.868718e-02, 1.894332e+00),
        440: (3.482554e-01, 5.033657e-02, 1.917479e+00),
        441: (3.494153e-01, 5.187611e-02, 1.930529e+00),
        442: (3.489075e-01, 5.332218e-02, 1.934819e+00),
        443: (3.471746e-01, 5.470603e-02, 1.932650e+00),
        444: (3.446705e-01, 5.606335e-02, 1.926395e+00),
        445: (3.418483e-01, 5.743393e-02, 1.918437e+00),
        446: (3.390240e-01, 5.885107e-02, 1.910430e+00),
        447: (3.359926e-01, 6.030809e-02, 1.901224e+00),
        448: (3.324276e-01, 6.178644e-02, 1.889000e+00),
        449: (3.280157e-01, 6.326570e-02, 1.871996e+00),
        450: (3.224637e-01, 6.472352e-02, 1.848545e+00),
        451: (3.156225e-01, 6.614749e-02, 1.817792e+00),
        452: (3.078201e-01, 6.757256e-02, 1.781627e+00),
        453: (2.994771e-01, 6.904928e-02, 1.742514e+00),
        454: (2.909776e-01, 7.063280e-02, 1.702749e+00),
        455: (2.826646e-01, 7.238339e-02, 1.664439e+00),
        456: (2.747962e-01, 7.435960e-02, 1.629207e+00),
        457: (2.674312e-01, 7.659383e-02, 1.597360e+00),
        458: (2.605847e-01, 7.911436e-02, 1.568896e+00),
        459: (2.542749e-01, 8.195345e-02, 1.543823e+00),
        460: (2.485254e-01, 8.514816e-02, 1.522157e+00),
        461: (2.433039e-01, 8.872657e-02, 1.503611e+00),
        462: (2.383414e-01, 9.266008e-02, 1.486673e+00),
        463: (2.333253e-01, 9.689723e-02, 1.469595e+00),
        464: (2.279619e-01, 1.013746e-01, 1.450709e+00),
        465: (2.219781e-01, 1.060145e-01, 1.428440e+00),
        466: (2.151735e-01, 1.107377e-01, 1.401587e+00),
        467: (2.075619e-01, 1.155111e-01, 1.370094e+00),
        468: (1.992183e-01, 1.203122e-01, 1.334220e+00),
        469: (1.902290e-01, 1.251161e-01, 1.294275e+00),
        470: (1.806905e-01, 1.298957e-01, 1.250610e+00),
        471: (1.707154e-01, 1.346299e-01, 1.203696e+00),
        472: (1.604471e-01, 1.393309e-01, 1.154316e+00),
        473: (1.500244e-01, 1.440235e-01, 1.103284e+00),
        474: (1.395705e-01, 1.487372e-01, 1.051347e+00),
        475: (1.291920e-01, 1.535066e-01, 9.991789e-01),
        476: (1.189859e-01, 1.583644e-01, 9.473958e-01),
        477: (1.090615e-01, 1.633199e-01, 8.966222e-01),
        478: (9.951424e-02, 1.683761e-01, 8.473981e-01),
        479: (9.041850e-02, 1.735365e-01, 8.001576e-01),
        480: (8.182895e-02, 1.788048e-01, 7.552379e-01),
        481: (7.376817e-02, 1.841819e-01, 7.127879e-01),
        482: (6.619477e-02, 1.896559e-01, 6.725198e-01),
        483: (5.906380e-02, 1.952101e-01, 6.340976e-01),
        484: (5.234242e-02, 2.008259e-01, 5.972433e-01),
        485: (4.600865e-02, 2.064828e-01, 5.617313e-01),
        486: (4.006154e-02, 2.121826e-01, 5.274921e-01),
        487: (3.454373e-02, 2.180279e-01, 4.948809e-01),
        488: (2.949091e-02, 2.241586e-01, 4.642586e-01),
        489: (2.492140e-02, 2.307302e-01, 4.358841e-01),
        490: (2.083981e-02, 2.379160e-01, 4.099313e-01),
        491: (1.723591e-02, 2.458706e-01, 3.864261e-01),
        492: (1.407924e-02, 2.546023e-01, 3.650566e-01),
        493: (1.134516e-02, 2.640760e-01, 3.454812e-01),
        494: (9.019658e-03, 2.742490e-01, 3.274095e-01),
        495: (7.097731e-03, 2.850680e-01, 3.105939e-01),
        496: (5.571145e-03, 2.964837e-01, 2.948102e-01),
        497: (4.394566e-03, 3.085010e-01, 2.798194e-01),
        498: (3.516303e-03, 3.211393e-01, 2.654100e-01),
        499: (2.887638e-03, 3.344175e-01, 2.514084e-01),
        500: (2.461588e-03, 3.483536e-01, 2.376753e-01),
        501: (2.206348e-03, 3.629601e-01, 2.241211e-01),
        502: (2.149559e-03, 3.782275e-01, 2.107484e-01),
        503: (2.337091e-03, 3.941359e-01, 1.975839e-01),
        504: (2.818931e-03, 4.106582e-01, 1.846574e-01),
        505: (3.649178e-03, 4.277595e-01, 1.720018e-01),
        506: (4.891359e-03, 4.453993e-01, 1.596918e-01),
        507: (6.629364e-03, 4.635396e-01, 1.479415e-01),
        508: (8.942902e-03, 4.821376e-01, 1.369428e-01),
        509: (1.190224e-02, 5.011430e-01, 1.268279e-01),
        510: (1.556989e-02, 5.204972e-01, 1.176796e-01),
        511: (1.997668e-02, 5.401387e-01, 1.094970e-01),
        512: (2.504698e-02, 5.600208e-01, 1.020943e-01),
        513: (3.067530e-02, 5.800972e-01, 9.527993e-02),
        514: (3.674999e-02, 6.003172e-01, 8.890075e-02),
        515: (4.315171e-02, 6.206256e-01, 8.283548e-02),
        516: (4.978584e-02, 6.409398e-01, 7.700982e-02),
        517: (5.668554e-02, 6.610772e-01, 7.144001e-02),
        518: (6.391651e-02, 6.808134e-01, 6.615436e-02),
        519: (7.154352e-02, 6.999044e-01, 6.117199e-02),
        520: (7.962917e-02, 7.180890e-01, 5.650407e-02),
        521: (8.821473e-02, 7.351593e-01, 5.215121e-02),
        522: (9.726978e-02, 7.511821e-01, 4.809566e-02),
        523: (1.067504e-01, 7.663143e-01, 4.431720e-02),
        524: (1.166192e-01, 7.807352e-01, 4.079734e-02),
        525: (1.268468e-01, 7.946448e-01, 3.751912e-02),
        526: (1.374060e-01, 8.082074e-01, 3.446846e-02),
        527: (1.482471e-01, 8.213817e-01, 3.163764e-02),
        528: (1.593076e-01, 8.340701e-01, 2.901901e-02),
        529: (1.705181e-01, 8.461711e-01, 2.660364e-02),
        530: (1.818026e-01, 8.575799e-01, 2.438164e-02),
        531: (1.931090e-01, 8.682408e-01, 2.234097e-02),
        532: (2.045085e-01, 8.783061e-01, 2.046415e-02),
        533: (2.161166e-01, 8.879907e-01, 1.873456e-02),
        534: (2.280650e-01, 8.975211e-01, 1.713788e-02),
        535: (2.405015e-01, 9.071347e-01, 1.566174e-02),
        536: (2.535441e-01, 9.169947e-01, 1.429644e-02),
        537: (2.671300e-01, 9.269295e-01, 1.303702e-02),
        538: (2.811351e-01, 9.366731e-01, 1.187897e-02),
        539: (2.954164e-01, 9.459482e-01, 1.081725e-02),
        540: (3.098117e-01, 9.544675e-01, 9.846470e-03),
        541: (3.241678e-01, 9.619834e-01, 8.960687e-03),
        542: (3.384319e-01, 9.684390e-01, 8.152811e-03),
        543: (3.525786e-01, 9.738289e-01, 7.416025e-03),
        544: (3.665839e-01, 9.781519e-01, 6.744115e-03),
        545: (3.804244e-01, 9.814106e-01, 6.131421e-03),
        546: (3.940988e-01, 9.836669e-01, 5.572778e-03),
        547: (4.076972e-01, 9.852081e-01, 5.063463e-03),
        548: (4.213484e-01, 9.863813e-01, 4.599169e-03),
        549: (4.352003e-01, 9.875357e-01, 4.175971e-03),
        550: (4.494206e-01, 9.890228e-01, 3.790291e-03),
        551: (4.641616e-01, 9.910811e-01, 3.438952e-03),
        552: (4.794395e-01, 9.934913e-01, 3.119341e-03),
        553: (4.952180e-01, 9.959172e-01, 2.829038e-03),
        554: (5.114395e-01, 9.980205e-01, 2.565722e-03),
        555: (5.280233e-01, 9.994608e-01, 2.327186e-03),
        556: (5.448696e-01, 9.999930e-01, 2.111280e-03),
        557: (5.618898e-01, 9.997557e-01, 1.915766e-03),
        558: (5.790137e-01, 9.989839e-01, 1.738589e-03),
        559: (5.961882e-01, 9.979123e-01, 1.577920e-03),
        560: (6.133784e-01, 9.967737e-01, 1.432128e-03),
        561: (6.305897e-01, 9.957356e-01, 1.299781e-03),
        562: (6.479223e-01, 9.947115e-01, 1.179667e-03),
        563: (6.654866e-01, 9.935534e-01, 1.070694e-03),
        564: (6.833782e-01, 9.921156e-01, 9.718623e-04),
        565: (7.016774e-01, 9.902549e-01, 8.822531e-04),
        566: (7.204110e-01, 9.878596e-01, 8.010231e-04),
        567: (7.394495e-01, 9.849324e-01, 7.273884e-04),
        568: (7.586285e-01, 9.815036e-01, 6.606347e-04),
        569: (7.777885e-01, 9.776035e-01, 6.001146e-04),
        570: (7.967750e-01, 9.732611e-01, 5.452416e-04),
        571: (8.154530e-01, 9.684764e-01, 4.954847e-04),
        572: (8.337389e-01, 9.631369e-01, 4.503642e-04),
        573: (8.515493e-01, 9.571062e-01, 4.094455e-04),
        574: (8.687862e-01, 9.502540e-01, 3.723345e-04),
        575: (8.853376e-01, 9.424569e-01, 3.386739e-04),
        576: (9.011588e-01, 9.336897e-01, 3.081396e-04),
        577: (9.165278e-01, 9.242893e-01, 2.804370e-04),
        578: (9.318245e-01, 9.146707e-01, 2.552996e-04),
        579: (9.474524e-01, 9.052333e-01, 2.324859e-04),
        580: (9.638388e-01, 8.963613e-01, 2.117772e-04),
        581: (9.812596e-01, 8.883069e-01, 1.929758e-04),
        582: (9.992953e-01, 8.808462e-01, 1.759024e-04),
        583: (1.017343e+00, 8.736445e-01, 1.603947e-04),
        584: (1.034790e+00, 8.663755e-01, 1.463059e-04),
        585: (1.051011e+00, 8.587203e-01, 1.335031e-04),
        586: (1.065522e+00, 8.504295e-01, 1.218660e-04),
        587: (1.078421e+00, 8.415047e-01, 1.112857e-04),
        588: (1.089944e+00, 8.320109e-01, 1.016634e-04),
        589: (1.100320e+00, 8.220154e-01, 9.291003e-05),
        590: (1.109767e+00, 8.115868e-01, 8.494468e-05),
        591: (1.118438e+00, 8.007874e-01, 7.769425e-05),
        592: (1.126266e+00, 7.896515e-01, 7.109247e-05),
        593: (1.133138e+00, 7.782053e-01, 6.507936e-05),
        594: (1.138952e+00, 7.664733e-01, 5.960061e-05),
        595: (1.143620e+00, 7.544785e-01, 5.460706e-05),
        596: (1.147095e+00, 7.422473e-01, 5.005417e-05),
        597: (1.149464e+00, 7.298229e-01, 4.590157e-05),
        598: (1.150838e+00, 7.172525e-01, 4.211268e-05),
        599: (1.151326e+00, 7.045818e-01, 3.865437e-05),
        600: (1.151033e+00, 6.918553e-01, 3.549661e-05),
        601: (1.150002e+00, 6.791009e-01, 3.261220e-05),
        602: (1.148061e+00, 6.662846e-01, 2.997643e-05),
        603: (1.144998e+00, 6.533595e-01, 2.756693e-05),
        604: (1.140622e+00, 6.402807e-01, 2.536339e-05),
        605: (1.134757e+00, 6.270066e-01, 2.334738e-05),
        606: (1.127298e+00, 6.135148e-01, 2.150221e-05),
        607: (1.118342e+00, 5.998494e-01, 1.981268e-05),
        608: (1.108033e+00, 5.860682e-01, 1.826500e-05),
        609: (1.096515e+00, 5.722261e-01, 1.684667e-05),
        610: (1.083928e+00, 5.583746e-01, 1.554631e-05),
        611: (1.070387e+00, 5.445535e-01, 1.435360e-05),
        612: (1.055934e+00, 5.307673e-01, 1.325915e-05),
        613: (1.040592e+00, 5.170130e-01, 1.225443e-05),
        614: (1.024385e+00, 5.032889e-01, 1.133169e-05),
        615: (1.007344e+00, 4.895950e-01, 1.048387e-05),
        616: (9.895268e-01, 4.759442e-01, 0.000000e+00),
        617: (9.711213e-01, 4.623958e-01, 0.000000e+00),
        618: (9.523257e-01, 4.490154e-01, 0.000000e+00),
        619: (9.333248e-01, 4.358622e-01, 0.000000e+00),
        620: (9.142877e-01, 4.229897e-01, 0.000000e+00),
        621: (8.952798e-01, 4.104152e-01, 0.000000e+00),
        622: (8.760157e-01, 3.980356e-01, 0.000000e+00),
        623: (8.561607e-01, 3.857300e-01, 0.000000e+00),
        624: (8.354235e-01, 3.733907e-01, 0.000000e+00),
        625: (8.135565e-01, 3.609245e-01, 0.000000e+00),
        626: (7.904565e-01, 3.482860e-01, 0.000000e+00),
        627: (7.664364e-01, 3.355702e-01, 0.000000e+00),
        628: (7.418777e-01, 3.228963e-01, 0.000000e+00),
        629: (7.171219e-01, 3.103704e-01, 0.000000e+00),
        630: (6.924717e-01, 2.980865e-01, 0.000000e+00),
        631: (6.681600e-01, 2.861160e-01, 0.000000e+00),
        632: (6.442697e-01, 2.744822e-01, 0.000000e+00),
        633: (6.208450e-01, 2.631953e-01, 0.000000e+00),
        634: (5.979243e-01, 2.522628e-01, 0.000000e+00),
        635: (5.755410e-01, 2.416902e-01, 0.000000e+00),
        636: (5.537296e-01, 2.314809e-01, 0.000000e+00),
        637: (5.325412e-01, 2.216378e-01, 0.000000e+00),
        638: (5.120218e-01, 2.121622e-01, 0.000000e+00),
        639: (4.922070e-01, 2.030542e-01, 0.000000e+00),
        640: (4.731224e-01, 1.943124e-01, 0.000000e+00),
        641: (4.547417e-01, 1.859227e-01, 0.000000e+00),
        642: (4.368719e-01, 1.778274e-01, 0.000000e+00),
        643: (4.193121e-01, 1.699654e-01, 0.000000e+00),
        644: (4.018980e-01, 1.622841e-01, 0.000000e+00),
        645: (3.844986e-01, 1.547397e-01, 0.000000e+00),
        646: (3.670592e-01, 1.473081e-01, 0.000000e+00),
        647: (3.497167e-01, 1.400169e-01, 0.000000e+00),
        648: (3.326305e-01, 1.329013e-01, 0.000000e+00),
        649: (3.159341e-01, 1.259913e-01, 0.000000e+00),
        650: (2.997374e-01, 1.193120e-01, 0.000000e+00),
        651: (2.841189e-01, 1.128820e-01, 0.000000e+00),
        652: (2.691053e-01, 1.067113e-01, 0.000000e+00),
        653: (2.547077e-01, 1.008052e-01, 0.000000e+00),
        654: (2.409319e-01, 9.516653e-02, 0.000000e+00),
        655: (2.277792e-01, 8.979594e-02, 0.000000e+00),
        656: (2.152431e-01, 8.469044e-02, 0.000000e+00),
        657: (2.033010e-01, 7.984009e-02, 0.000000e+00),
        658: (1.919276e-01, 7.523372e-02, 0.000000e+00),
        659: (1.810987e-01, 7.086061e-02, 0.000000e+00),
        660: (1.707914e-01, 6.671045e-02, 0.000000e+00),
        661: (1.609842e-01, 6.277360e-02, 0.000000e+00),
        662: (1.516577e-01, 5.904179e-02, 0.000000e+00),
        663: (1.427936e-01, 5.550703e-02, 0.000000e+00),
        664: (1.343737e-01, 5.216139e-02, 0.000000e+00),
        665: (1.263808e-01, 4.899699e-02, 0.000000e+00),
        666: (1.187979e-01, 4.600578e-02, 0.000000e+00),
        667: (1.116088e-01, 4.317885e-02, 0.000000e+00),
        668: (1.047975e-01, 4.050755e-02, 0.000000e+00),
        669: (9.834835e-02, 3.798376e-02, 0.000000e+00),
        670: (9.224597e-02, 3.559982e-02, 0.000000e+00),
        671: (8.647506e-02, 3.334856e-02, 0.000000e+00),
        672: (8.101986e-02, 3.122332e-02, 0.000000e+00),
        673: (7.586514e-02, 2.921780e-02, 0.000000e+00),
        674: (7.099633e-02, 2.732601e-02, 0.000000e+00),
        675: (6.639960e-02, 2.554223e-02, 0.000000e+00),
        676: (6.206225e-02, 2.386121e-02, 0.000000e+00),
        677: (5.797409e-02, 2.227859e-02, 0.000000e+00),
        678: (5.412533e-02, 2.079020e-02, 0.000000e+00),
        679: (5.050600e-02, 1.939185e-02, 0.000000e+00),
        680: (4.710606e-02, 1.807939e-02, 0.000000e+00),
        681: (4.391411e-02, 1.684817e-02, 0.000000e+00),
        682: (4.091411e-02, 1.569188e-02, 0.000000e+00),
        683: (3.809067e-02, 1.460446e-02, 0.000000e+00),
        684: (3.543034e-02, 1.358062e-02, 0.000000e+00),
        685: (3.292138e-02, 1.261573e-02, 0.000000e+00),
        686: (3.055672e-02, 1.170696e-02, 0.000000e+00),
        687: (2.834146e-02, 1.085608e-02, 0.000000e+00),
        688: (2.628033e-02, 1.006476e-02, 0.000000e+00),
        689: (2.437465e-02, 9.333376e-03, 0.000000e+00),
        690: (2.262306e-02, 8.661284e-03, 0.000000e+00),
        691: (2.101935e-02, 8.046048e-03, 0.000000e+00),
        692: (1.954647e-02, 7.481130e-03, 0.000000e+00),
        693: (1.818727e-02, 6.959987e-03, 0.000000e+00),
        694: (1.692727e-02, 6.477070e-03, 0.000000e+00),
        695: (1.575417e-02, 6.027677e-03, 0.000000e+00),
        696: (1.465854e-02, 5.608169e-03, 0.000000e+00),
        697: (1.363571e-02, 5.216691e-03, 0.000000e+00),
        698: (1.268205e-02, 4.851785e-03, 0.000000e+00),
        699: (1.179394e-02, 4.512008e-03, 0.000000e+00),
        700: (1.096778e-02, 4.195941e-03, 0.000000e+00),
        701: (1.019964e-02, 3.902057e-03, 0.000000e+00),
        702: (9.484317e-03, 3.628371e-03, 0.000000e+00),
        703: (8.816851e-03, 3.373005e-03, 0.000000e+00),
        704: (8.192921e-03, 3.134315e-03, 0.000000e+00),
        705: (7.608750e-03, 2.910864e-03, 0.000000e+00),
        706: (7.061391e-03, 2.701528e-03, 0.000000e+00),
        707: (6.549509e-03, 2.505796e-03, 0.000000e+00),
        708: (6.071970e-03, 2.323231e-03, 0.000000e+00),
        709: (5.627476e-03, 2.153333e-03, 0.000000e+00),
        710: (5.214608e-03, 1.995557e-03, 0.000000e+00),
        711: (4.831848e-03, 1.849316e-03, 0.000000e+00),
        712: (4.477579e-03, 1.713976e-03, 0.000000e+00),
        713: (4.150166e-03, 1.588899e-03, 0.000000e+00),
        714: (3.847988e-03, 1.473453e-03, 0.000000e+00),
        715: (3.569452e-03, 1.367022e-03, 0.000000e+00),
        716: (3.312857e-03, 1.268954e-03, 0.000000e+00),
        717: (3.076022e-03, 1.178421e-03, 0.000000e+00),
        718: (2.856894e-03, 1.094644e-03, 0.000000e+00),
        719: (2.653681e-03, 1.016943e-03, 0.000000e+00),
        720: (2.464821e-03, 9.447269e-04, 0.000000e+00),
        721: (2.289060e-03, 8.775171e-04, 0.000000e+00),
        722: (2.125694e-03, 8.150438e-04, 0.000000e+00),
        723: (1.974121e-03, 7.570755e-04, 0.000000e+00),
        724: (1.833723e-03, 7.033755e-04, 0.000000e+00),
        725: (1.703876e-03, 6.537050e-04, 0.000000e+00),
        726: (1.583904e-03, 6.078048e-04, 0.000000e+00),
        727: (1.472939e-03, 5.653435e-04, 0.000000e+00),
        728: (1.370151e-03, 5.260046e-04, 0.000000e+00),
        729: (1.274803e-03, 4.895061e-04, 0.000000e+00),
        730: (1.186238e-03, 4.555970e-04, 0.000000e+00),
        731: (1.103871e-03, 4.240548e-04, 0.000000e+00),
        732: (1.027194e-03, 3.946860e-04, 0.000000e+00),
        733: (9.557493e-04, 3.673178e-04, 0.000000e+00),
        734: (8.891262e-04, 3.417941e-04, 0.000000e+00),
        735: (8.269535e-04, 3.179738e-04, 0.000000e+00),
        736: (7.689351e-04, 2.957441e-04, 0.000000e+00),
        737: (7.149425e-04, 2.750558e-04, 0.000000e+00),
        738: (6.648590e-04, 2.558640e-04, 0.000000e+00),
        739: (6.185421e-04, 2.381142e-04, 0.000000e+00),
        740: (5.758303e-04, 2.217445e-04, 0.000000e+00),
        741: (5.365046e-04, 2.066711e-04, 0.000000e+00),
        742: (5.001842e-04, 1.927474e-04, 0.000000e+00),
        743: (4.665005e-04, 1.798315e-04, 0.000000e+00),
        744: (4.351386e-04, 1.678023e-04, 0.000000e+00),
        745: (4.058303e-04, 1.565566e-04, 0.000000e+00),
        746: (3.783733e-04, 1.460168e-04, 0.000000e+00),
        747: (3.526892e-04, 1.361535e-04, 0.000000e+00),
        748: (3.287199e-04, 1.269451e-04, 0.000000e+00),
        749: (3.063998e-04, 1.183671e-04, 0.000000e+00),
        750: (2.856577e-04, 1.103928e-04, 0.000000e+00),
        751: (2.664108e-04, 1.029908e-04, 0.000000e+00),
        752: (2.485462e-04, 9.611836e-05, 0.000000e+00),
        753: (2.319529e-04, 8.973323e-05, 0.000000e+00),
        754: (2.165300e-04, 8.379694e-05, 0.000000e+00),
        755: (2.021853e-04, 7.827442e-05, 0.000000e+00),
        756: (1.888338e-04, 7.313312e-05, 0.000000e+00),
        757: (1.763935e-04, 6.834142e-05, 0.000000e+00),
        758: (1.647895e-04, 6.387035e-05, 0.000000e+00),
        759: (1.539542e-04, 5.969389e-05, 0.000000e+00),
        760: (1.438270e-04, 5.578862e-05, 0.000000e+00),
        761: (1.343572e-04, 5.213509e-05, 0.000000e+00),
        762: (1.255141e-04, 4.872179e-05, 0.000000e+00),
        763: (1.172706e-04, 4.553845e-05, 0.000000e+00),
        764: (1.095983e-04, 4.257443e-05, 0.000000e+00),
        765: (1.024685e-04, 3.981884e-05, 0.000000e+00),
        766: (9.584715e-05, 3.725877e-05, 0.000000e+00),
        767: (8.968316e-05, 3.487467e-05, 0.000000e+00),
        768: (8.392734e-05, 3.264765e-05, 0.000000e+00),
        769: (7.853708e-05, 3.056140e-05, 0.000000e+00),
        770: (7.347551e-05, 2.860175e-05, 0.000000e+00),
        771: (6.871576e-05, 2.675841e-05, 0.000000e+00),
        772: (6.425257e-05, 2.502943e-05, 0.000000e+00),
        773: (6.008292e-05, 2.341373e-05, 0.000000e+00),
        774: (5.620098e-05, 2.190914e-05, 0.000000e+00),
        775: (5.259870e-05, 2.051259e-05, 0.000000e+00),
        776: (4.926279e-05, 1.921902e-05, 0.000000e+00),
        777: (4.616623e-05, 1.801796e-05, 0.000000e+00),
        778: (4.328212e-05, 1.689899e-05, 0.000000e+00),
        779: (4.058715e-05, 1.585309e-05, 0.000000e+00),
        780: (3.806114e-05, 1.487243e-05, 0.000000e+00),
        781: (3.568818e-05, 1.395085e-05, 0.000000e+00),
        782: (3.346023e-05, 1.308528e-05, 0.000000e+00),
        783: (3.137090e-05, 1.227327e-05, 0.000000e+00),
        784: (2.941371e-05, 1.151233e-05, 0.000000e+00),
        785: (2.758222e-05, 1.080001e-05, 0.000000e+00),
        786: (2.586951e-05, 1.013364e-05, 0.000000e+00),
        787: (2.426701e-05, 9.509919e-06, 0.000000e+00),
        788: (2.276639e-05, 8.925630e-06, 0.000000e+00),
        789: (2.136009e-05, 8.377852e-06, 0.000000e+00),
        790: (2.004122e-05, 7.863920e-06, 0.000000e+00),
        791: (1.880380e-05, 7.381539e-06, 0.000000e+00),
        792: (1.764358e-05, 6.929096e-06, 0.000000e+00),
        793: (1.655671e-05, 6.505136e-06, 0.000000e+00),
        794: (1.553939e-05, 6.108221e-06, 0.000000e+00),
        795: (1.458792e-05, 5.736935e-06, 0.000000e+00),
        796: (1.369853e-05, 5.389831e-06, 0.000000e+00),
        797: (1.286705e-05, 5.065269e-06, 0.000000e+00),
        798: (1.208947e-05, 4.761667e-06, 0.000000e+00),
        799: (1.136207e-05, 4.477561e-06, 0.000000e+00),
        800: (1.068141e-05, 4.211597e-06, 0.000000e+00),
        801: (1.004411e-05, 3.962457e-06, 0.000000e+00),
        802: (9.446399e-06, 3.728674e-06, 0.000000e+00),
        803: (8.884754e-06, 3.508881e-06, 0.000000e+00),
        804: (8.356050e-06, 3.301868e-06, 0.000000e+00),
        805: (7.857521e-06, 3.106561e-06, 0.000000e+00),
        806: (7.386996e-06, 2.922119e-06, 0.000000e+00),
        807: (6.943576e-06, 2.748208e-06, 0.000000e+00),
        808: (6.526548e-06, 2.584560e-06, 0.000000e+00),
        809: (6.135087e-06, 2.430867e-06, 0.000000e+00),
        810: (5.768284e-06, 2.286786e-06, 0.000000e+00),
        811: (5.425069e-06, 2.151905e-06, 0.000000e+00),
        812: (5.103974e-06, 2.025656e-06, 0.000000e+00),
        813: (4.803525e-06, 1.907464e-06, 0.000000e+00),
        814: (4.522350e-06, 1.796794e-06, 0.000000e+00),
        815: (4.259166e-06, 1.693147e-06, 0.000000e+00),
        816: (4.012715e-06, 1.596032e-06, 0.000000e+00),
        817: (3.781597e-06, 1.504903e-06, 0.000000e+00),
        818: (3.564496e-06, 1.419245e-06, 0.000000e+00),
        819: (3.360236e-06, 1.338600e-06, 0.000000e+00),
        820: (3.167765e-06, 1.262556e-06, 0.000000e+00),
        821: (2.986206e-06, 1.190771e-06, 0.000000e+00),
        822: (2.814999e-06, 1.123031e-06, 0.000000e+00),
        823: (2.653663e-06, 1.059151e-06, 0.000000e+00),
        824: (2.501725e-06, 9.989507e-07, 0.000000e+00),
        825: (2.358723e-06, 9.422514e-07, 0.000000e+00),
        826: (2.224206e-06, 8.888804e-07, 0.000000e+00),
        827: (2.097737e-06, 8.386690e-07, 0.000000e+00),
        828: (1.978894e-06, 7.914539e-07, 0.000000e+00),
        829: (1.867268e-06, 7.470770e-07, 0.000000e+00),
        830: (1.762465e-06, 7.053860e-07, 0.000000e+00)
    },
    'CIE 2012 10 Degree Standard Observer': {
        390: (2.952420e-03, 4.076779e-04, 1.318752e-02),
        391: (3.577275e-03, 4.977769e-04, 1.597879e-02),
        392: (4.332146e-03, 6.064754e-04, 1.935758e-02),
        393: (5.241609e-03, 7.370040e-04, 2.343758e-02),
        394: (6.333902e-03, 8.929388e-04, 2.835021e-02),
        395: (7.641137e-03, 1.078166e-03, 3.424588e-02),
        396: (9.199401e-03, 1.296816e-03, 4.129467e-02),
        397: (1.104869e-02, 1.553159e-03, 4.968641e-02),
        398: (1.323262e-02, 1.851463e-03, 5.962964e-02),
        399: (1.579791e-02, 2.195795e-03, 7.134926e-02),
        400: (1.879338e-02, 2.589775e-03, 8.508254e-02),
        401: (2.226949e-02, 3.036799e-03, 1.010753e-01),
        402: (2.627978e-02, 3.541926e-03, 1.195838e-01),
        403: (3.087862e-02, 4.111422e-03, 1.408647e-01),
        404: (3.611890e-02, 4.752618e-03, 1.651644e-01),
        405: (4.204986e-02, 5.474207e-03, 1.927065e-01),
        406: (4.871256e-02, 6.285034e-03, 2.236782e-01),
        407: (5.612868e-02, 7.188068e-03, 2.582109e-01),
        408: (6.429866e-02, 8.181786e-03, 2.963632e-01),
        409: (7.319818e-02, 9.260417e-03, 3.381018e-01),
        410: (8.277331e-02, 1.041303e-02, 3.832822e-01),
        411: (9.295327e-02, 1.162642e-02, 4.316884e-01),
        412: (1.037137e-01, 1.289884e-02, 4.832440e-01),
        413: (1.150520e-01, 1.423442e-02, 5.379345e-01),
        414: (1.269771e-01, 1.564080e-02, 5.957740e-01),
        415: (1.395127e-01, 1.712968e-02, 6.568187e-01),
        416: (1.526661e-01, 1.871265e-02, 7.210459e-01),
        417: (1.663054e-01, 2.038394e-02, 7.878635e-01),
        418: (1.802197e-01, 2.212935e-02, 8.563391e-01),
        419: (1.941448e-01, 2.392985e-02, 9.253017e-01),
        420: (2.077647e-01, 2.576133e-02, 9.933444e-01),
        421: (2.207911e-01, 2.760156e-02, 1.059178e+00),
        422: (2.332355e-01, 2.945513e-02, 1.122832e+00),
        423: (2.452462e-01, 3.133884e-02, 1.184947e+00),
        424: (2.570397e-01, 3.327575e-02, 1.246476e+00),
        425: (2.688989e-01, 3.529554e-02, 1.308674e+00),
        426: (2.810677e-01, 3.742705e-02, 1.372628e+00),
        427: (2.933967e-01, 3.967137e-02, 1.437661e+00),
        428: (3.055933e-01, 4.201998e-02, 1.502449e+00),
        429: (3.173165e-01, 4.446166e-02, 1.565456e+00),
        430: (3.281798e-01, 4.698226e-02, 1.624940e+00),
        431: (3.378678e-01, 4.956742e-02, 1.679488e+00),
        432: (3.465097e-01, 5.221219e-02, 1.729668e+00),
        433: (3.543953e-01, 5.491387e-02, 1.776755e+00),
        434: (3.618655e-01, 5.766919e-02, 1.822228e+00),
        435: (3.693084e-01, 6.047429e-02, 1.867751e+00),
        436: (3.770107e-01, 6.332195e-02, 1.914504e+00),
        437: (3.846850e-01, 6.619271e-02, 1.961055e+00),
        438: (3.918591e-01, 6.906185e-02, 2.005136e+00),
        439: (3.980192e-01, 7.190190e-02, 2.044296e+00),
        440: (4.026189e-01, 7.468288e-02, 2.075946e+00),
        441: (4.052637e-01, 7.738452e-02, 2.098231e+00),
        442: (4.062482e-01, 8.003601e-02, 2.112591e+00),
        443: (4.060660e-01, 8.268524e-02, 2.121427e+00),
        444: (4.052283e-01, 8.538745e-02, 2.127239e+00),
        445: (4.042529e-01, 8.820537e-02, 2.132574e+00),
        446: (4.034808e-01, 9.118925e-02, 2.139093e+00),
        447: (4.025362e-01, 9.431041e-02, 2.144815e+00),
        448: (4.008675e-01, 9.751346e-02, 2.146832e+00),
        449: (3.979327e-01, 1.007349e-01, 2.142250e+00),
        450: (3.932139e-01, 1.039030e-01, 2.128264e+00),
        451: (3.864108e-01, 1.069639e-01, 2.103205e+00),
        452: (3.779513e-01, 1.099676e-01, 2.069388e+00),
        453: (3.684176e-01, 1.129992e-01, 2.030030e+00),
        454: (3.583473e-01, 1.161541e-01, 1.988178e+00),
        455: (3.482214e-01, 1.195389e-01, 1.946651e+00),
        456: (3.383830e-01, 1.232503e-01, 1.907521e+00),
        457: (3.288309e-01, 1.273047e-01, 1.870689e+00),
        458: (3.194977e-01, 1.316964e-01, 1.835578e+00),
        459: (3.103345e-01, 1.364178e-01, 1.801657e+00),
        460: (3.013112e-01, 1.414586e-01, 1.768440e+00),
        461: (2.923754e-01, 1.468003e-01, 1.735338e+00),
        462: (2.833273e-01, 1.524002e-01, 1.701254e+00),
        463: (2.739463e-01, 1.582021e-01, 1.665053e+00),
        464: (2.640352e-01, 1.641400e-01, 1.625712e+00),
        465: (2.534221e-01, 1.701373e-01, 1.582342e+00),
        466: (2.420135e-01, 1.761233e-01, 1.534439e+00),
        467: (2.299346e-01, 1.820896e-01, 1.482544e+00),
        468: (2.173617e-01, 1.880463e-01, 1.427438e+00),
        469: (2.044672e-01, 1.940065e-01, 1.369876e+00),
        470: (1.914176e-01, 1.999859e-01, 1.310576e+00),
        471: (1.783672e-01, 2.060054e-01, 1.250226e+00),
        472: (1.654407e-01, 2.120981e-01, 1.189511e+00),
        473: (1.527391e-01, 2.183041e-01, 1.129050e+00),
        474: (1.403439e-01, 2.246686e-01, 1.069379e+00),
        475: (1.283167e-01, 2.312426e-01, 1.010952e+00),
        476: (1.167124e-01, 2.380741e-01, 9.541809e-01),
        477: (1.056121e-01, 2.451798e-01, 8.995253e-01),
        478: (9.508569e-02, 2.525682e-01, 8.473720e-01),
        479: (8.518206e-02, 2.602479e-01, 7.980093e-01),
        480: (7.593120e-02, 2.682271e-01, 7.516389e-01),
        481: (6.733159e-02, 2.765005e-01, 7.082645e-01),
        482: (5.932018e-02, 2.850035e-01, 6.673867e-01),
        483: (5.184106e-02, 2.936475e-01, 6.284798e-01),
        484: (4.486119e-02, 3.023319e-01, 5.911174e-01),
        485: (3.836770e-02, 3.109438e-01, 5.549619e-01),
        486: (3.237296e-02, 3.194105e-01, 5.198843e-01),
        487: (2.692095e-02, 3.278683e-01, 4.862772e-01),
        488: (2.204070e-02, 3.365263e-01, 4.545497e-01),
        489: (1.773951e-02, 3.456176e-01, 4.249955e-01),
        490: (1.400745e-02, 3.554018e-01, 3.978114e-01),
        491: (1.082291e-02, 3.660893e-01, 3.730218e-01),
        492: (8.168996e-03, 3.775857e-01, 3.502618e-01),
        493: (6.044623e-03, 3.896960e-01, 3.291407e-01),
        494: (4.462638e-03, 4.021947e-01, 3.093356e-01),
        495: (3.446810e-03, 4.148227e-01, 2.905816e-01),
        496: (3.009513e-03, 4.273539e-01, 2.726773e-01),
        497: (3.090744e-03, 4.398206e-01, 2.555143e-01),
        498: (3.611221e-03, 4.523360e-01, 2.390188e-01),
        499: (4.491435e-03, 4.650298e-01, 2.231335e-01),
        500: (5.652072e-03, 4.780482e-01, 2.078158e-01),
        501: (7.035322e-03, 4.915173e-01, 1.930407e-01),
        502: (8.669631e-03, 5.054224e-01, 1.788089e-01),
        503: (1.060755e-02, 5.197057e-01, 1.651287e-01),
        504: (1.290468e-02, 5.343012e-01, 1.520103e-01),
        505: (1.561956e-02, 5.491344e-01, 1.394643e-01),
        506: (1.881640e-02, 5.641302e-01, 1.275353e-01),
        507: (2.256923e-02, 5.792416e-01, 1.163771e-01),
        508: (2.694456e-02, 5.944264e-01, 1.061161e-01),
        509: (3.199910e-02, 6.096388e-01, 9.682266e-02),
        510: (3.778185e-02, 6.248296e-01, 8.852389e-02),
        511: (4.430635e-02, 6.399656e-01, 8.118263e-02),
        512: (5.146516e-02, 6.550943e-01, 7.463132e-02),
        513: (5.912224e-02, 6.702903e-01, 6.870644e-02),
        514: (6.714220e-02, 6.856375e-01, 6.327834e-02),
        515: (7.538941e-02, 7.012292e-01, 5.824484e-02),
        516: (8.376697e-02, 7.171103e-01, 5.353812e-02),
        517: (9.233581e-02, 7.330917e-01, 4.914863e-02),
        518: (1.011940e-01, 7.489041e-01, 4.507511e-02),
        519: (1.104362e-01, 7.642530e-01, 4.131175e-02),
        520: (1.201511e-01, 7.788199e-01, 3.784916e-02),
        521: (1.303960e-01, 7.923410e-01, 3.467234e-02),
        522: (1.411310e-01, 8.048510e-01, 3.175471e-02),
        523: (1.522944e-01, 8.164747e-01, 2.907029e-02),
        524: (1.638288e-01, 8.273520e-01, 2.659651e-02),
        525: (1.756832e-01, 8.376358e-01, 2.431375e-02),
        526: (1.878114e-01, 8.474653e-01, 2.220677e-02),
        527: (2.001621e-01, 8.568868e-01, 2.026852e-02),
        528: (2.126822e-01, 8.659242e-01, 1.849246e-02),
        529: (2.253199e-01, 8.746041e-01, 1.687084e-02),
        530: (2.380254e-01, 8.829552e-01, 1.539505e-02),
        531: (2.507787e-01, 8.910274e-01, 1.405450e-02),
        532: (2.636778e-01, 8.989495e-01, 1.283354e-02),
        533: (2.768607e-01, 9.068753e-01, 1.171754e-02),
        534: (2.904792e-01, 9.149652e-01, 1.069415e-02),
        535: (3.046991e-01, 9.233858e-01, 9.753000e-03),
        536: (3.196485e-01, 9.322325e-01, 8.886096e-03),
        537: (3.352447e-01, 9.412862e-01, 8.089323e-03),
        538: (3.513290e-01, 9.502378e-01, 7.359131e-03),
        539: (3.677148e-01, 9.587647e-01, 6.691736e-03),
        540: (3.841856e-01, 9.665325e-01, 6.083223e-03),
        541: (4.005312e-01, 9.732504e-01, 5.529423e-03),
        542: (4.166669e-01, 9.788415e-01, 5.025504e-03),
        543: (4.325420e-01, 9.832867e-01, 4.566879e-03),
        544: (4.481063e-01, 9.865720e-01, 4.149405e-03),
        545: (4.633109e-01, 9.886887e-01, 3.769336e-03),
        546: (4.781440e-01, 9.897056e-01, 3.423302e-03),
        547: (4.927483e-01, 9.899849e-01, 3.108313e-03),
        548: (5.073315e-01, 9.899624e-01, 2.821650e-03),
        549: (5.221315e-01, 9.900731e-01, 2.560830e-03),
        550: (5.374170e-01, 9.907500e-01, 2.323578e-03),
        551: (5.534217e-01, 9.922826e-01, 2.107847e-03),
        552: (5.701242e-01, 9.943837e-01, 1.911867e-03),
        553: (5.874093e-01, 9.966221e-01, 1.734006e-03),
        554: (6.051269e-01, 9.985649e-01, 1.572736e-03),
        555: (6.230892e-01, 9.997775e-01, 1.426627e-03),
        556: (6.410999e-01, 9.999440e-01, 1.294325e-03),
        557: (6.590659e-01, 9.992200e-01, 1.174475e-03),
        558: (6.769436e-01, 9.978793e-01, 1.065842e-03),
        559: (6.947143e-01, 9.961934e-01, 9.673215e-04),
        560: (7.123849e-01, 9.944304e-01, 8.779264e-04),
        561: (7.299978e-01, 9.927831e-01, 7.967847e-04),
        562: (7.476478e-01, 9.911578e-01, 7.231502e-04),
        563: (7.654250e-01, 9.893925e-01, 6.563501e-04),
        564: (7.834009e-01, 9.873288e-01, 5.957678e-04),
        565: (8.016277e-01, 9.848127e-01, 5.408385e-04),
        566: (8.201041e-01, 9.817253e-01, 4.910441e-04),
        567: (8.386843e-01, 9.780714e-01, 4.459046e-04),
        568: (8.571936e-01, 9.738860e-01, 4.049826e-04),
        569: (8.754652e-01, 9.692028e-01, 3.678818e-04),
        570: (8.933408e-01, 9.640545e-01, 3.342429e-04),
        571: (9.106772e-01, 9.584409e-01, 3.037407e-04),
        572: (9.273554e-01, 9.522379e-01, 2.760809e-04),
        573: (9.432502e-01, 9.452968e-01, 2.509970e-04),
        574: (9.582244e-01, 9.374773e-01, 2.282474e-04),
        575: (9.721304e-01, 9.286495e-01, 2.076129e-04),
        576: (9.849237e-01, 9.187953e-01, 1.888948e-04),
        577: (9.970067e-01, 9.083014e-01, 1.719127e-04),
        578: (1.008907e+00, 8.976352e-01, 1.565030e-04),
        579: (1.021163e+00, 8.872401e-01, 1.425177e-04),
        580: (1.034327e+00, 8.775360e-01, 1.298230e-04),
        581: (1.048753e+00, 8.687920e-01, 1.182974e-04),
        582: (1.063937e+00, 8.607474e-01, 1.078310e-04),
        583: (1.079166e+00, 8.530233e-01, 9.832455e-05),
        584: (1.093723e+00, 8.452535e-01, 8.968787e-05),
        585: (1.106886e+00, 8.370838e-01, 8.183954e-05),
        586: (1.118106e+00, 8.282409e-01, 7.470582e-05),
        587: (1.127493e+00, 8.187320e-01, 6.821991e-05),
        588: (1.135317e+00, 8.086352e-01, 6.232132e-05),
        589: (1.141838e+00, 7.980296e-01, 5.695534e-05),
        590: (1.147304e+00, 7.869950e-01, 5.207245e-05),
        591: (1.151897e+00, 7.756040e-01, 4.762781e-05),
        592: (1.155582e+00, 7.638996e-01, 4.358082e-05),
        593: (1.158284e+00, 7.519157e-01, 3.989468e-05),
        594: (1.159934e+00, 7.396832e-01, 3.653612e-05),
        595: (1.160477e+00, 7.272309e-01, 3.347499e-05),
        596: (1.159890e+00, 7.145878e-01, 3.068400e-05),
        597: (1.158259e+00, 7.017926e-01, 2.813839e-05),
        598: (1.155692e+00, 6.888866e-01, 2.581574e-05),
        599: (1.152293e+00, 6.759103e-01, 2.369574e-05),
        600: (1.148163e+00, 6.629035e-01, 2.175998e-05),
        601: (1.143345e+00, 6.498911e-01, 1.999179e-05),
        602: (1.137685e+00, 6.368410e-01, 1.837603e-05),
        603: (1.130993e+00, 6.237092e-01, 1.689896e-05),
        604: (1.123097e+00, 6.104541e-01, 1.554815e-05),
        605: (1.113846e+00, 5.970375e-01, 1.431231e-05),
        606: (1.103152e+00, 5.834395e-01, 1.318119e-05),
        607: (1.091121e+00, 5.697044e-01, 1.214548e-05),
        608: (1.077902e+00, 5.558892e-01, 1.119673e-05),
        609: (1.063644e+00, 5.420475e-01, 1.032727e-05),
        610: (1.048485e+00, 5.282296e-01, 9.530130e-06),
        611: (1.032546e+00, 5.144746e-01, 8.798979e-06),
        612: (1.015870e+00, 5.007881e-01, 8.128065e-06),
        613: (9.984859e-01, 4.871687e-01, 7.512160e-06),
        614: (9.804227e-01, 4.736160e-01, 6.946506e-06),
        615: (9.617111e-01, 4.601308e-01, 6.426776e-06),
        616: (9.424119e-01, 4.467260e-01, 0.000000e+00),
        617: (9.227049e-01, 4.334589e-01, 0.000000e+00),
        618: (9.027804e-01, 4.203919e-01, 0.000000e+00),
        619: (8.828123e-01, 4.075810e-01, 0.000000e+00),
        620: (8.629581e-01, 3.950755e-01, 0.000000e+00),
        621: (8.432731e-01, 3.828894e-01, 0.000000e+00),
        622: (8.234742e-01, 3.709190e-01, 0.000000e+00),
        623: (8.032342e-01, 3.590447e-01, 0.000000e+00),
        624: (7.822715e-01, 3.471615e-01, 0.000000e+00),
        625: (7.603498e-01, 3.351794e-01, 0.000000e+00),
        626: (7.373739e-01, 3.230562e-01, 0.000000e+00),
        627: (7.136470e-01, 3.108859e-01, 0.000000e+00),
        628: (6.895336e-01, 2.987840e-01, 0.000000e+00),
        629: (6.653567e-01, 2.868527e-01, 0.000000e+00),
        630: (6.413984e-01, 2.751807e-01, 0.000000e+00),
        631: (6.178723e-01, 2.638343e-01, 0.000000e+00),
        632: (5.948484e-01, 2.528330e-01, 0.000000e+00),
        633: (5.723600e-01, 2.421835e-01, 0.000000e+00),
        634: (5.504353e-01, 2.318904e-01, 0.000000e+00),
        635: (5.290979e-01, 2.219564e-01, 0.000000e+00),
        636: (5.083728e-01, 2.123826e-01, 0.000000e+00),
        637: (4.883006e-01, 2.031698e-01, 0.000000e+00),
        638: (4.689171e-01, 1.943179e-01, 0.000000e+00),
        639: (4.502486e-01, 1.858250e-01, 0.000000e+00),
        640: (4.323126e-01, 1.776882e-01, 0.000000e+00),
        641: (4.150790e-01, 1.698926e-01, 0.000000e+00),
        642: (3.983657e-01, 1.623822e-01, 0.000000e+00),
        643: (3.819846e-01, 1.550986e-01, 0.000000e+00),
        644: (3.657821e-01, 1.479918e-01, 0.000000e+00),
        645: (3.496358e-01, 1.410203e-01, 0.000000e+00),
        646: (3.334937e-01, 1.341614e-01, 0.000000e+00),
        647: (3.174776e-01, 1.274401e-01, 0.000000e+00),
        648: (3.017298e-01, 1.208887e-01, 0.000000e+00),
        649: (2.863684e-01, 1.145345e-01, 0.000000e+00),
        650: (2.714900e-01, 1.083996e-01, 0.000000e+00),
        651: (2.571632e-01, 1.025007e-01, 0.000000e+00),
        652: (2.434102e-01, 9.684588e-02, 0.000000e+00),
        653: (2.302389e-01, 9.143944e-02, 0.000000e+00),
        654: (2.176527e-01, 8.628318e-02, 0.000000e+00),
        655: (2.056507e-01, 8.137687e-02, 0.000000e+00),
        656: (1.942251e-01, 7.671708e-02, 0.000000e+00),
        657: (1.833530e-01, 7.229404e-02, 0.000000e+00),
        658: (1.730097e-01, 6.809696e-02, 0.000000e+00),
        659: (1.631716e-01, 6.411549e-02, 0.000000e+00),
        660: (1.538163e-01, 6.033976e-02, 0.000000e+00),
        661: (1.449230e-01, 5.676054e-02, 0.000000e+00),
        662: (1.364729e-01, 5.336992e-02, 0.000000e+00),
        663: (1.284483e-01, 5.016027e-02, 0.000000e+00),
        664: (1.208320e-01, 4.712405e-02, 0.000000e+00),
        665: (1.136072e-01, 4.425383e-02, 0.000000e+00),
        666: (1.067579e-01, 4.154205e-02, 0.000000e+00),
        667: (1.002685e-01, 3.898042e-02, 0.000000e+00),
        668: (9.412394e-02, 3.656091e-02, 0.000000e+00),
        669: (8.830929e-02, 3.427597e-02, 0.000000e+00),
        670: (8.281010e-02, 3.211852e-02, 0.000000e+00),
        671: (7.761208e-02, 3.008192e-02, 0.000000e+00),
        672: (7.270064e-02, 2.816001e-02, 0.000000e+00),
        673: (6.806167e-02, 2.634698e-02, 0.000000e+00),
        674: (6.368176e-02, 2.463731e-02, 0.000000e+00),
        675: (5.954815e-02, 2.302574e-02, 0.000000e+00),
        676: (5.564917e-02, 2.150743e-02, 0.000000e+00),
        677: (5.197543e-02, 2.007838e-02, 0.000000e+00),
        678: (4.851788e-02, 1.873474e-02, 0.000000e+00),
        679: (4.526737e-02, 1.747269e-02, 0.000000e+00),
        680: (4.221473e-02, 1.628841e-02, 0.000000e+00),
        681: (3.934954e-02, 1.517767e-02, 0.000000e+00),
        682: (3.665730e-02, 1.413473e-02, 0.000000e+00),
        683: (3.412407e-02, 1.315408e-02, 0.000000e+00),
        684: (3.173768e-02, 1.223092e-02, 0.000000e+00),
        685: (2.948752e-02, 1.136106e-02, 0.000000e+00),
        686: (2.736717e-02, 1.054190e-02, 0.000000e+00),
        687: (2.538113e-02, 9.775050e-03, 0.000000e+00),
        688: (2.353356e-02, 9.061962e-03, 0.000000e+00),
        689: (2.182558e-02, 8.402962e-03, 0.000000e+00),
        690: (2.025590e-02, 7.797457e-03, 0.000000e+00),
        691: (1.881892e-02, 7.243230e-03, 0.000000e+00),
        692: (1.749930e-02, 6.734381e-03, 0.000000e+00),
        693: (1.628167e-02, 6.265001e-03, 0.000000e+00),
        694: (1.515301e-02, 5.830085e-03, 0.000000e+00),
        695: (1.410230e-02, 5.425391e-03, 0.000000e+00),
        696: (1.312106e-02, 5.047634e-03, 0.000000e+00),
        697: (1.220509e-02, 4.695140e-03, 0.000000e+00),
        698: (1.135114e-02, 4.366592e-03, 0.000000e+00),
        699: (1.055593e-02, 4.060685e-03, 0.000000e+00),
        700: (9.816228e-03, 3.776140e-03, 0.000000e+00),
        701: (9.128517e-03, 3.511578e-03, 0.000000e+00),
        702: (8.488116e-03, 3.265211e-03, 0.000000e+00),
        703: (7.890589e-03, 3.035344e-03, 0.000000e+00),
        704: (7.332061e-03, 2.820496e-03, 0.000000e+00),
        705: (6.809147e-03, 2.619372e-03, 0.000000e+00),
        706: (6.319204e-03, 2.430960e-03, 0.000000e+00),
        707: (5.861036e-03, 2.254796e-03, 0.000000e+00),
        708: (5.433624e-03, 2.090489e-03, 0.000000e+00),
        709: (5.035802e-03, 1.937586e-03, 0.000000e+00),
        710: (4.666298e-03, 1.795595e-03, 0.000000e+00),
        711: (4.323750e-03, 1.663989e-03, 0.000000e+00),
        712: (4.006709e-03, 1.542195e-03, 0.000000e+00),
        713: (3.713708e-03, 1.429639e-03, 0.000000e+00),
        714: (3.443294e-03, 1.325752e-03, 0.000000e+00),
        715: (3.194041e-03, 1.229980e-03, 0.000000e+00),
        716: (2.964424e-03, 1.141734e-03, 0.000000e+00),
        717: (2.752492e-03, 1.060269e-03, 0.000000e+00),
        718: (2.556406e-03, 9.848854e-04, 0.000000e+00),
        719: (2.374564e-03, 9.149703e-04, 0.000000e+00),
        720: (2.205568e-03, 8.499903e-04, 0.000000e+00),
        721: (2.048294e-03, 7.895158e-04, 0.000000e+00),
        722: (1.902113e-03, 7.333038e-04, 0.000000e+00),
        723: (1.766485e-03, 6.811458e-04, 0.000000e+00),
        724: (1.640857e-03, 6.328287e-04, 0.000000e+00),
        725: (1.524672e-03, 5.881375e-04, 0.000000e+00),
        726: (1.417322e-03, 5.468389e-04, 0.000000e+00),
        727: (1.318031e-03, 5.086349e-04, 0.000000e+00),
        728: (1.226059e-03, 4.732403e-04, 0.000000e+00),
        729: (1.140743e-03, 4.404016e-04, 0.000000e+00),
        730: (1.061495e-03, 4.098928e-04, 0.000000e+00),
        731: (9.877949e-04, 3.815137e-04, 0.000000e+00),
        732: (9.191847e-04, 3.550902e-04, 0.000000e+00),
        733: (8.552568e-04, 3.304668e-04, 0.000000e+00),
        734: (7.956433e-04, 3.075030e-04, 0.000000e+00),
        735: (7.400120e-04, 2.860718e-04, 0.000000e+00),
        736: (6.880980e-04, 2.660718e-04, 0.000000e+00),
        737: (6.397864e-04, 2.474586e-04, 0.000000e+00),
        738: (5.949726e-04, 2.301919e-04, 0.000000e+00),
        739: (5.535291e-04, 2.142225e-04, 0.000000e+00),
        740: (5.153113e-04, 1.994949e-04, 0.000000e+00),
        741: (4.801234e-04, 1.859336e-04, 0.000000e+00),
        742: (4.476245e-04, 1.734067e-04, 0.000000e+00),
        743: (4.174846e-04, 1.617865e-04, 0.000000e+00),
        744: (3.894221e-04, 1.509641e-04, 0.000000e+00),
        745: (3.631969e-04, 1.408466e-04, 0.000000e+00),
        746: (3.386279e-04, 1.313642e-04, 0.000000e+00),
        747: (3.156452e-04, 1.224905e-04, 0.000000e+00),
        748: (2.941966e-04, 1.142060e-04, 0.000000e+00),
        749: (2.742235e-04, 1.064886e-04, 0.000000e+00),
        750: (2.556624e-04, 9.931439e-05, 0.000000e+00),
        751: (2.384390e-04, 9.265512e-05, 0.000000e+00),
        752: (2.224525e-04, 8.647225e-05, 0.000000e+00),
        753: (2.076036e-04, 8.072780e-05, 0.000000e+00),
        754: (1.938018e-04, 7.538716e-05, 0.000000e+00),
        755: (1.809649e-04, 7.041878e-05, 0.000000e+00),
        756: (1.690167e-04, 6.579338e-05, 0.000000e+00),
        757: (1.578839e-04, 6.148250e-05, 0.000000e+00),
        758: (1.474993e-04, 5.746008e-05, 0.000000e+00),
        759: (1.378026e-04, 5.370272e-05, 0.000000e+00),
        760: (1.287394e-04, 5.018934e-05, 0.000000e+00),
        761: (1.202644e-04, 4.690245e-05, 0.000000e+00),
        762: (1.123502e-04, 4.383167e-05, 0.000000e+00),
        763: (1.049725e-04, 4.096780e-05, 0.000000e+00),
        764: (9.810596e-05, 3.830123e-05, 0.000000e+00),
        765: (9.172477e-05, 3.582218e-05, 0.000000e+00),
        766: (8.579861e-05, 3.351903e-05, 0.000000e+00),
        767: (8.028174e-05, 3.137419e-05, 0.000000e+00),
        768: (7.513013e-05, 2.937068e-05, 0.000000e+00),
        769: (7.030565e-05, 2.749380e-05, 0.000000e+00),
        770: (6.577532e-05, 2.573083e-05, 0.000000e+00),
        771: (6.151508e-05, 2.407249e-05, 0.000000e+00),
        772: (5.752025e-05, 2.251704e-05, 0.000000e+00),
        773: (5.378813e-05, 2.106350e-05, 0.000000e+00),
        774: (5.031350e-05, 1.970991e-05, 0.000000e+00),
        775: (4.708916e-05, 1.845353e-05, 0.000000e+00),
        776: (4.410322e-05, 1.728979e-05, 0.000000e+00),
        777: (4.133150e-05, 1.620928e-05, 0.000000e+00),
        778: (3.874992e-05, 1.520262e-05, 0.000000e+00),
        779: (3.633762e-05, 1.426169e-05, 0.000000e+00),
        780: (3.407653e-05, 1.337946e-05, 0.000000e+00),
        781: (3.195242e-05, 1.255038e-05, 0.000000e+00),
        782: (2.995808e-05, 1.177169e-05, 0.000000e+00),
        783: (2.808781e-05, 1.104118e-05, 0.000000e+00),
        784: (2.633581e-05, 1.035662e-05, 0.000000e+00),
        785: (2.469630e-05, 9.715798e-06, 0.000000e+00),
        786: (2.316311e-05, 9.116316e-06, 0.000000e+00),
        787: (2.172855e-05, 8.555201e-06, 0.000000e+00),
        788: (2.038519e-05, 8.029561e-06, 0.000000e+00),
        789: (1.912625e-05, 7.536768e-06, 0.000000e+00),
        790: (1.794555e-05, 7.074424e-06, 0.000000e+00),
        791: (1.683776e-05, 6.640464e-06, 0.000000e+00),
        792: (1.579907e-05, 6.233437e-06, 0.000000e+00),
        793: (1.482604e-05, 5.852035e-06, 0.000000e+00),
        794: (1.391527e-05, 5.494963e-06, 0.000000e+00),
        795: (1.306345e-05, 5.160948e-06, 0.000000e+00),
        796: (1.226720e-05, 4.848687e-06, 0.000000e+00),
        797: (1.152279e-05, 4.556705e-06, 0.000000e+00),
        798: (1.082663e-05, 4.283580e-06, 0.000000e+00),
        799: (1.017540e-05, 4.027993e-06, 0.000000e+00),
        800: (9.565993e-06, 3.788729e-06, 0.000000e+00),
        801: (8.995405e-06, 3.564599e-06, 0.000000e+00),
        802: (8.460253e-06, 3.354285e-06, 0.000000e+00),
        803: (7.957382e-06, 3.156557e-06, 0.000000e+00),
        804: (7.483997e-06, 2.970326e-06, 0.000000e+00),
        805: (7.037621e-06, 2.794625e-06, 0.000000e+00),
        806: (6.616311e-06, 2.628701e-06, 0.000000e+00),
        807: (6.219265e-06, 2.472248e-06, 0.000000e+00),
        808: (5.845844e-06, 2.325030e-06, 0.000000e+00),
        809: (5.495311e-06, 2.186768e-06, 0.000000e+00),
        810: (5.166853e-06, 2.057152e-06, 0.000000e+00),
        811: (4.859511e-06, 1.935813e-06, 0.000000e+00),
        812: (4.571973e-06, 1.822239e-06, 0.000000e+00),
        813: (4.302920e-06, 1.715914e-06, 0.000000e+00),
        814: (4.051121e-06, 1.616355e-06, 0.000000e+00),
        815: (3.815429e-06, 1.523114e-06, 0.000000e+00),
        816: (3.594719e-06, 1.435750e-06, 0.000000e+00),
        817: (3.387736e-06, 1.353771e-06, 0.000000e+00),
        818: (3.193301e-06, 1.276714e-06, 0.000000e+00),
        819: (3.010363e-06, 1.204166e-06, 0.000000e+00),
        820: (2.837980e-06, 1.135758e-06, 0.000000e+00),
        821: (2.675365e-06, 1.071181e-06, 0.000000e+00),
        822: (2.522020e-06, 1.010243e-06, 0.000000e+00),
        823: (2.377511e-06, 9.527779e-07, 0.000000e+00),
        824: (2.241417e-06, 8.986224e-07, 0.000000e+00),
        825: (2.113325e-06, 8.476168e-07, 0.000000e+00),
        826: (1.992830e-06, 7.996052e-07, 0.000000e+00),
        827: (1.879542e-06, 7.544361e-07, 0.000000e+00),
        828: (1.773083e-06, 7.119624e-07, 0.000000e+00),
        829: (1.673086e-06, 6.720421e-07, 0.000000e+00),
        830: (1.579199e-06, 6.345380e-07, 0.000000e+00)
    }
}

STANDARD_OBSERVERS_CMFS = CaseInsensitiveMapping({
    'CIE 1931 2 Degree Standard Observer':
        XYZ_ColourMatchingFunctions(
            STANDARD_OBSERVERS_CMFS_DATA[
                'CIE 1931 2 Degree Standard Observer'],
            name='CIE 1931 2 Degree Standard Observer',
            strict_name='CIE 1931 2$^\\circ$ Standard Observer'),
    'CIE 1964 10 Degree Standard Observer':
        XYZ_ColourMatchingFunctions(
            STANDARD_OBSERVERS_CMFS_DATA[
                'CIE 1964 10 Degree Standard Observer'],
            name='CIE 1964 10 Degree Standard Observer',
            strict_name='CIE 1964 10$^\\circ$ Standard Observer'),
    'CIE 2012 2 Degree Standard Observer':
        XYZ_ColourMatchingFunctions(
            STANDARD_OBSERVERS_CMFS_DATA[
                'CIE 2012 2 Degree Standard Observer'],
            name='CIE 2012 2 Degree Standard Observer',
            strict_name='CIE 2012 2$^\\circ$ Standard Observer'),
    'CIE 2012 10 Degree Standard Observer':
        XYZ_ColourMatchingFunctions(
            STANDARD_OBSERVERS_CMFS_DATA[
                'CIE 2012 10 Degree Standard Observer'],
            name='CIE 2012 10 Degree Standard Observer',
            strict_name='CIE 2012 10$^\\circ$ Standard Observer')
})
STANDARD_OBSERVERS_CMFS.__doc__ = """
*CIE* Standard Observers *XYZ* colour matching functions.

References
----------
:cite:`CVRLr`, :cite:`CVRLs`

STANDARD_OBSERVERS_CMFS : CaseInsensitiveMapping
    **{'CIE 1931 2 Degree Standard Observer',
    'CIE 1964 10 Degree Standard Observer',
    'CIE 2012 2 Degree Standard Observer',
    'CIE 2012 10 Degree Standard Observer'}**

Aliases:

-   'cie_2_1931': 'CIE 1931 2 Degree Standard Observer'
-   'cie_10_1964': 'CIE 1964 10 Degree Standard Observer'
"""
STANDARD_OBSERVERS_CMFS['cie_2_1931'] = (
    STANDARD_OBSERVERS_CMFS['CIE 1931 2 Degree Standard Observer'])
STANDARD_OBSERVERS_CMFS['cie_10_1964'] = (
    STANDARD_OBSERVERS_CMFS['CIE 1964 10 Degree Standard Observer'])

CMFS = CaseInsensitiveMapping(LMS_CMFS)
CMFS.__doc__ = """
Aggregated colour matching functions.

References
----------
:cite:`Broadbent2009a`, :cite:`CVRLr`, :cite:`CVRLs`, :cite:`CVRLt`,
:cite:`CVRLu`, :cite:`CVRLw`, :cite:`Machado2010a`

CMFS : CaseInsensitiveMapping
    **{'Stockman & Sharpe 10 Degree Cone Fundamentals',
    'Stockman & Sharpe 2 Degree Cone Fundamentals',
    'Wright & Guild 1931 2 Degree RGB CMFs',
    'Stiles & Burch 1955 2 Degree RGB CMFs',
    'Stiles & Burch 1959 10 Degree RGB CMFs',
    'CIE 1931 2 Degree Standard Observer',
    'CIE 1964 10 Degree Standard Observer',
    'CIE 2012 2 Degree Standard Observer',
    'CIE 2012 10 Degree Standard Observer'}**
"""
CMFS.update(RGB_CMFS)
CMFS.update(STANDARD_OBSERVERS_CMFS)
