# -*- coding: utf-8 -*-
"""
Luminous Efficiency Functions Spectral Power Distributions
==========================================================

Defines various luminous efficiency functions involved in spectral
computations.

The luminous efficiency data is in the form of a *dict* of
:class:`colour.SpectralPowerDistribution` classes as follows::

    {'name': SpectralPowerDistribution, ..., 'name': SpectralPowerDistribution}

The following luminous efficiency functions are available:

-   CIE 1924 Photopic Standard Observer
-   Judd Modified CIE 1951 Photopic Standard Observer
-   Judd-Vos Modified CIE 1978 Photopic Standard Observer
-   CIE 1964 Photopic 10 Degree Standard Observer
-   CIE 2008 2 Degree Physiologically Relevant LEF
-   CIE 2008 10 Degree Physiologically Relevant LEF
-   CIE 1951 Scotopic Standard Observer

See Also
--------
`Luminous Efficiency Functions Jupyter Notebook
<http://nbviewer.jupyter.org/github/colour-science/colour-notebooks/\
blob/master/notebooks/colorimetry/lefs.ipynb>`_

Notes
-----
-   The luminous efficiency functions are provided at 1 nm interval.
-   The mesopic luminous efficiency function is calculated using the
    *CIE 1924 Photopic Standard Observer* and
    *CIE 1951 Scotopic Standard Observer* luminous efficiency functions with
    the :func:`colour.mesopic_luminous_efficiency_function` definition and the
    data from :attr:`colour.colorimetry.dataset.lefs.MESOPIC_X_DATA`
    attribute that defines weighting factors dependent on the photopic
    luminance :math:`L_p`.

References
----------
-   :cite:`CVRLq` : CVRL. (n.d.). Luminous efficiency. Retrieved April 19,
    2014, from http://www.cvrl.org/lumindex.htm
-   :cite:`CVRLs` : CVRL. (n.d.). Older CIE Standards. Retrieved February 24,
    2014, from http://cvrl.ioo.ucl.ac.uk/cie.htm
-   :cite:`Wikipediacc` : Wikipedia. (n.d.). Mesopic weighting function.
    Retrieved June 20, 2014, from http://en.wikipedia.org/wiki/\
Mesopic_vision#Mesopic_weighting_function
"""

from __future__ import division, unicode_literals

from colour.colorimetry import SpectralPowerDistribution
from colour.utilities import CaseInsensitiveMapping

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013-2018 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = [
    'PHOTOPIC_LEFS_DATA', 'PHOTOPIC_LEFS', 'SCOTOPIC_LEFS_DATA',
    'SCOTOPIC_LEFS', 'LEFS', 'MESOPIC_X_DATA'
]

PHOTOPIC_LEFS_DATA = {
    'CIE 1924 Photopic Standard Observer': {
        360: 0.0000039170000,
        361: 0.0000043935810,
        362: 0.0000049296040,
        363: 0.0000055321360,
        364: 0.0000062082450,
        365: 0.0000069650000,
        366: 0.0000078132190,
        367: 0.0000087673360,
        368: 0.0000098398440,
        369: 0.0000110432300,
        370: 0.0000123900000,
        371: 0.0000138864100,
        372: 0.0000155572800,
        373: 0.0000174429600,
        374: 0.0000195837500,
        375: 0.0000220200000,
        376: 0.0000248396500,
        377: 0.0000280412600,
        378: 0.0000315310400,
        379: 0.0000352152100,
        380: 0.0000390000000,
        381: 0.0000428264000,
        382: 0.0000469146000,
        383: 0.0000515896000,
        384: 0.0000571764000,
        385: 0.0000640000000,
        386: 0.0000723442100,
        387: 0.0000822122400,
        388: 0.0000935081600,
        389: 0.0001061361000,
        390: 0.0001200000000,
        391: 0.0001349840000,
        392: 0.0001514920000,
        393: 0.0001702080000,
        394: 0.0001918160000,
        395: 0.0002170000000,
        396: 0.0002469067000,
        397: 0.0002812400000,
        398: 0.0003185200000,
        399: 0.0003572667000,
        400: 0.0003960000000,
        401: 0.0004337147000,
        402: 0.0004730240000,
        403: 0.0005178760000,
        404: 0.0005722187000,
        405: 0.0006400000000,
        406: 0.0007245600000,
        407: 0.0008255000000,
        408: 0.0009411600000,
        409: 0.0010698800000,
        410: 0.0012100000000,
        411: 0.0013620910000,
        412: 0.0015307520000,
        413: 0.0017203680000,
        414: 0.0019353230000,
        415: 0.0021800000000,
        416: 0.0024548000000,
        417: 0.0027640000000,
        418: 0.0031178000000,
        419: 0.0035264000000,
        420: 0.0040000000000,
        421: 0.0045462400000,
        422: 0.0051593200000,
        423: 0.0058292800000,
        424: 0.0065461600000,
        425: 0.0073000000000,
        426: 0.0080865070000,
        427: 0.0089087200000,
        428: 0.0097676800000,
        429: 0.0106644300000,
        430: 0.0116000000000,
        431: 0.0125731700000,
        432: 0.0135827200000,
        433: 0.0146296800000,
        434: 0.0157150900000,
        435: 0.0168400000000,
        436: 0.0180073600000,
        437: 0.0192144800000,
        438: 0.0204539200000,
        439: 0.0217182400000,
        440: 0.0230000000000,
        441: 0.0242946100000,
        442: 0.0256102400000,
        443: 0.0269585700000,
        444: 0.0283512500000,
        445: 0.0298000000000,
        446: 0.0313108300000,
        447: 0.0328836800000,
        448: 0.0345211200000,
        449: 0.0362257100000,
        450: 0.0380000000000,
        451: 0.0398466700000,
        452: 0.0417680000000,
        453: 0.0437660000000,
        454: 0.0458426700000,
        455: 0.0480000000000,
        456: 0.0502436800000,
        457: 0.0525730400000,
        458: 0.0549805600000,
        459: 0.0574587200000,
        460: 0.0600000000000,
        461: 0.0626019700000,
        462: 0.0652775200000,
        463: 0.0680420800000,
        464: 0.0709110900000,
        465: 0.0739000000000,
        466: 0.0770160000000,
        467: 0.0802664000000,
        468: 0.0836668000000,
        469: 0.0872328000000,
        470: 0.0909800000000,
        471: 0.0949175500000,
        472: 0.0990458400000,
        473: 0.1033674000000,
        474: 0.1078846000000,
        475: 0.1126000000000,
        476: 0.1175320000000,
        477: 0.1226744000000,
        478: 0.1279928000000,
        479: 0.1334528000000,
        480: 0.1390200000000,
        481: 0.1446764000000,
        482: 0.1504693000000,
        483: 0.1564619000000,
        484: 0.1627177000000,
        485: 0.1693000000000,
        486: 0.1762431000000,
        487: 0.1835581000000,
        488: 0.1912735000000,
        489: 0.1994180000000,
        490: 0.2080200000000,
        491: 0.2171199000000,
        492: 0.2267345000000,
        493: 0.2368571000000,
        494: 0.2474812000000,
        495: 0.2586000000000,
        496: 0.2701849000000,
        497: 0.2822939000000,
        498: 0.2950505000000,
        499: 0.3085780000000,
        500: 0.3230000000000,
        501: 0.3384021000000,
        502: 0.3546858000000,
        503: 0.3716986000000,
        504: 0.3892875000000,
        505: 0.4073000000000,
        506: 0.4256299000000,
        507: 0.4443096000000,
        508: 0.4633944000000,
        509: 0.4829395000000,
        510: 0.5030000000000,
        511: 0.5235693000000,
        512: 0.5445120000000,
        513: 0.5656900000000,
        514: 0.5869653000000,
        515: 0.6082000000000,
        516: 0.6293456000000,
        517: 0.6503068000000,
        518: 0.6708752000000,
        519: 0.6908424000000,
        520: 0.7100000000000,
        521: 0.7281852000000,
        522: 0.7454636000000,
        523: 0.7619694000000,
        524: 0.7778368000000,
        525: 0.7932000000000,
        526: 0.8081104000000,
        527: 0.8224962000000,
        528: 0.8363068000000,
        529: 0.8494916000000,
        530: 0.8620000000000,
        531: 0.8738108000000,
        532: 0.8849624000000,
        533: 0.8954936000000,
        534: 0.9054432000000,
        535: 0.9148501000000,
        536: 0.9237348000000,
        537: 0.9320924000000,
        538: 0.9399226000000,
        539: 0.9472252000000,
        540: 0.9540000000000,
        541: 0.9602561000000,
        542: 0.9660074000000,
        543: 0.9712606000000,
        544: 0.9760225000000,
        545: 0.9803000000000,
        546: 0.9840924000000,
        547: 0.9874182000000,
        548: 0.9903128000000,
        549: 0.9928116000000,
        550: 0.9949501000000,
        551: 0.9967108000000,
        552: 0.9980983000000,
        553: 0.9991120000000,
        554: 0.9997482000000,
        555: 1.0000000000000,
        556: 0.9998567000000,
        557: 0.9993046000000,
        558: 0.9983255000000,
        559: 0.9968987000000,
        560: 0.9950000000000,
        561: 0.9926005000000,
        562: 0.9897426000000,
        563: 0.9864444000000,
        564: 0.9827241000000,
        565: 0.9786000000000,
        566: 0.9740837000000,
        567: 0.9691712000000,
        568: 0.9638568000000,
        569: 0.9581349000000,
        570: 0.9520000000000,
        571: 0.9454504000000,
        572: 0.9384992000000,
        573: 0.9311628000000,
        574: 0.9234576000000,
        575: 0.9154000000000,
        576: 0.9070064000000,
        577: 0.8982772000000,
        578: 0.8892048000000,
        579: 0.8797816000000,
        580: 0.8700000000000,
        581: 0.8598613000000,
        582: 0.8493920000000,
        583: 0.8386220000000,
        584: 0.8275813000000,
        585: 0.8163000000000,
        586: 0.8047947000000,
        587: 0.7930820000000,
        588: 0.7811920000000,
        589: 0.7691547000000,
        590: 0.7570000000000,
        591: 0.7447541000000,
        592: 0.7324224000000,
        593: 0.7200036000000,
        594: 0.7074965000000,
        595: 0.6949000000000,
        596: 0.6822192000000,
        597: 0.6694716000000,
        598: 0.6566744000000,
        599: 0.6438448000000,
        600: 0.6310000000000,
        601: 0.6181555000000,
        602: 0.6053144000000,
        603: 0.5924756000000,
        604: 0.5796379000000,
        605: 0.5668000000000,
        606: 0.5539611000000,
        607: 0.5411372000000,
        608: 0.5283528000000,
        609: 0.5156323000000,
        610: 0.5030000000000,
        611: 0.4904688000000,
        612: 0.4780304000000,
        613: 0.4656776000000,
        614: 0.4534032000000,
        615: 0.4412000000000,
        616: 0.4290800000000,
        617: 0.4170360000000,
        618: 0.4050320000000,
        619: 0.3930320000000,
        620: 0.3810000000000,
        621: 0.3689184000000,
        622: 0.3568272000000,
        623: 0.3447768000000,
        624: 0.3328176000000,
        625: 0.3210000000000,
        626: 0.3093381000000,
        627: 0.2978504000000,
        628: 0.2865936000000,
        629: 0.2756245000000,
        630: 0.2650000000000,
        631: 0.2547632000000,
        632: 0.2448896000000,
        633: 0.2353344000000,
        634: 0.2260528000000,
        635: 0.2170000000000,
        636: 0.2081616000000,
        637: 0.1995488000000,
        638: 0.1911552000000,
        639: 0.1829744000000,
        640: 0.1750000000000,
        641: 0.1672235000000,
        642: 0.1596464000000,
        643: 0.1522776000000,
        644: 0.1451259000000,
        645: 0.1382000000000,
        646: 0.1315003000000,
        647: 0.1250248000000,
        648: 0.1187792000000,
        649: 0.1127691000000,
        650: 0.1070000000000,
        651: 0.1014762000000,
        652: 0.0961886400000,
        653: 0.0911229600000,
        654: 0.0862648500000,
        655: 0.0816000000000,
        656: 0.0771206400000,
        657: 0.0728255200000,
        658: 0.0687100800000,
        659: 0.0647697600000,
        660: 0.0610000000000,
        661: 0.0573962100000,
        662: 0.0539550400000,
        663: 0.0506737600000,
        664: 0.0475496500000,
        665: 0.0445800000000,
        666: 0.0417587200000,
        667: 0.0390849600000,
        668: 0.0365638400000,
        669: 0.0342004800000,
        670: 0.0320000000000,
        671: 0.0299626100000,
        672: 0.0280766400000,
        673: 0.0263293600000,
        674: 0.0247080500000,
        675: 0.0232000000000,
        676: 0.0218007700000,
        677: 0.0205011200000,
        678: 0.0192810800000,
        679: 0.0181206900000,
        680: 0.0170000000000,
        681: 0.0159037900000,
        682: 0.0148371800000,
        683: 0.0138106800000,
        684: 0.0128347800000,
        685: 0.0119200000000,
        686: 0.0110683100000,
        687: 0.0102733900000,
        688: 0.0095333110000,
        689: 0.0088461570000,
        690: 0.0082100000000,
        691: 0.0076237810000,
        692: 0.0070854240000,
        693: 0.0065914760000,
        694: 0.0061384850000,
        695: 0.0057230000000,
        696: 0.0053430590000,
        697: 0.0049957960000,
        698: 0.0046764040000,
        699: 0.0043800750000,
        700: 0.0041020000000,
        701: 0.0038384530000,
        702: 0.0035890990000,
        703: 0.0033542190000,
        704: 0.0031340930000,
        705: 0.0029290000000,
        706: 0.0027381390000,
        707: 0.0025598760000,
        708: 0.0023932440000,
        709: 0.0022372750000,
        710: 0.0020910000000,
        711: 0.0019535870000,
        712: 0.0018245800000,
        713: 0.0017035800000,
        714: 0.0015901870000,
        715: 0.0014840000000,
        716: 0.0013844960000,
        717: 0.0012912680000,
        718: 0.0012040920000,
        719: 0.0011227440000,
        720: 0.0010470000000,
        721: 0.0009765896000,
        722: 0.0009111088000,
        723: 0.0008501332000,
        724: 0.0007932384000,
        725: 0.0007400000000,
        726: 0.0006900827000,
        727: 0.0006433100000,
        728: 0.0005994960000,
        729: 0.0005584547000,
        730: 0.0005200000000,
        731: 0.0004839136000,
        732: 0.0004500528000,
        733: 0.0004183452000,
        734: 0.0003887184000,
        735: 0.0003611000000,
        736: 0.0003353835000,
        737: 0.0003114404000,
        738: 0.0002891656000,
        739: 0.0002684539000,
        740: 0.0002492000000,
        741: 0.0002313019000,
        742: 0.0002146856000,
        743: 0.0001992884000,
        744: 0.0001850475000,
        745: 0.0001719000000,
        746: 0.0001597781000,
        747: 0.0001486044000,
        748: 0.0001383016000,
        749: 0.0001287925000,
        750: 0.0001200000000,
        751: 0.0001118595000,
        752: 0.0001043224000,
        753: 0.0000973356000,
        754: 0.0000908458700,
        755: 0.0000848000000,
        756: 0.0000791466700,
        757: 0.0000738580000,
        758: 0.0000689160000,
        759: 0.0000643026700,
        760: 0.0000600000000,
        761: 0.0000559818700,
        762: 0.0000522256000,
        763: 0.0000487184000,
        764: 0.0000454474700,
        765: 0.0000424000000,
        766: 0.0000395610400,
        767: 0.0000369151200,
        768: 0.0000344486800,
        769: 0.0000321481600,
        770: 0.0000300000000,
        771: 0.0000279912500,
        772: 0.0000261135600,
        773: 0.0000243602400,
        774: 0.0000227246100,
        775: 0.0000212000000,
        776: 0.0000197785500,
        777: 0.0000184528500,
        778: 0.0000172168700,
        779: 0.0000160645900,
        780: 0.0000149900000,
        781: 0.0000139872800,
        782: 0.0000130515500,
        783: 0.0000121781800,
        784: 0.0000113625400,
        785: 0.0000106000000,
        786: 0.0000098858770,
        787: 0.0000092173040,
        788: 0.0000085923620,
        789: 0.0000080091330,
        790: 0.0000074657000,
        791: 0.0000069595670,
        792: 0.0000064879950,
        793: 0.0000060486990,
        794: 0.0000056393960,
        795: 0.0000052578000,
        796: 0.0000049017710,
        797: 0.0000045697200,
        798: 0.0000042601940,
        799: 0.0000039717390,
        800: 0.0000037029000,
        801: 0.0000034521630,
        802: 0.0000032183020,
        803: 0.0000030003000,
        804: 0.0000027971390,
        805: 0.0000026078000,
        806: 0.0000024312200,
        807: 0.0000022665310,
        808: 0.0000021130130,
        809: 0.0000019699430,
        810: 0.0000018366000,
        811: 0.0000017122300,
        812: 0.0000015962280,
        813: 0.0000014880900,
        814: 0.0000013873140,
        815: 0.0000012934000,
        816: 0.0000012058200,
        817: 0.0000011241430,
        818: 0.0000010480090,
        819: 0.0000009770578,
        820: 0.0000009109300,
        821: 0.0000008492513,
        822: 0.0000007917212,
        823: 0.0000007380904,
        824: 0.0000006881098,
        825: 0.0000006415300,
        826: 0.0000005980895,
        827: 0.0000005575746,
        828: 0.0000005198080,
        829: 0.0000004846123,
        830: 0.0000004518100
    },
    'Judd Modified CIE 1951 Photopic Standard Observer': {
        370: 0.0001,
        380: 0.0004,
        390: 0.0015,
        400: 0.0045,
        410: 0.0093,
        420: 0.0175,
        430: 0.0273,
        440: 0.0379,
        450: 0.0468,
        460: 0.0600,
        470: 0.0910,
        480: 0.1390,
        490: 0.2080,
        500: 0.3230,
        510: 0.5030,
        520: 0.7100,
        530: 0.8620,
        540: 0.9540,
        550: 0.9950,
        560: 0.9950,
        570: 0.9520,
        580: 0.8700,
        590: 0.7570,
        600: 0.6310,
        610: 0.5030,
        620: 0.3810,
        630: 0.2650,
        640: 0.1750,
        650: 0.1070,
        660: 0.0610,
        670: 0.0320,
        680: 0.0170,
        690: 0.0082,
        700: 0.0041,
        710: 0.0021,
        720: 0.0011,
        730: 0.0005,
        740: 0.0002,
        750: 0.0001,
        760: 0.0001,
        770: 0.0000
    },
    'Judd-Vos Modified CIE 1978 Photopic Standard Observer': {
        380: 0.0002000000,
        381: 0.0002282100,
        382: 0.0002610900,
        383: 0.0002993600,
        384: 0.0003438700,
        385: 0.0003955600,
        386: 0.0004554400,
        387: 0.0005246200,
        388: 0.0006042800,
        389: 0.0006956500,
        390: 0.0008000000,
        391: 0.0009163500,
        392: 0.0010477000,
        393: 0.0011955000,
        394: 0.0013611000,
        395: 0.0015457000,
        396: 0.0017508000,
        397: 0.0018776000,
        398: 0.0022273000,
        399: 0.0025011000,
        400: 0.0028000000,
        401: 0.0031159000,
        402: 0.0034576000,
        403: 0.0038268000,
        404: 0.0042256000,
        405: 0.0046562000,
        406: 0.0051216000,
        407: 0.0056248000,
        408: 0.0061695000,
        409: 0.0067597000,
        410: 0.0074000000,
        411: 0.0081451000,
        412: 0.0089555000,
        413: 0.0098322000,
        414: 0.0107740000,
        415: 0.0117790000,
        416: 0.0128420000,
        417: 0.0139560000,
        418: 0.0151110000,
        419: 0.0162970000,
        420: 0.0175000000,
        421: 0.0185820000,
        422: 0.0196450000,
        423: 0.0206830000,
        424: 0.0216940000,
        425: 0.0226780000,
        426: 0.0236360000,
        427: 0.0245720000,
        428: 0.0254900000,
        429: 0.0263970000,
        430: 0.0273000000,
        431: 0.0283350000,
        432: 0.0293830000,
        433: 0.0304420000,
        434: 0.0315100000,
        435: 0.0325840000,
        436: 0.0336610000,
        437: 0.0347350000,
        438: 0.0358030000,
        439: 0.0368600000,
        440: 0.0379000000,
        441: 0.0388380000,
        442: 0.0397520000,
        443: 0.0406460000,
        444: 0.0415240000,
        445: 0.0423910000,
        446: 0.0432520000,
        447: 0.0441160000,
        448: 0.0449900000,
        449: 0.0458810000,
        450: 0.0468000000,
        451: 0.0477430000,
        452: 0.0487330000,
        453: 0.0497850000,
        454: 0.0509100000,
        455: 0.0521220000,
        456: 0.0534350000,
        457: 0.0548640000,
        458: 0.0564240000,
        459: 0.0581310000,
        460: 0.0600000000,
        461: 0.0626019700,
        462: 0.0652775200,
        463: 0.0680420800,
        464: 0.0709110900,
        465: 0.0739000000,
        466: 0.0770160000,
        467: 0.0802664000,
        468: 0.0836668000,
        469: 0.0872328000,
        470: 0.0909800000,
        471: 0.0949175500,
        472: 0.0990458400,
        473: 0.1033674000,
        474: 0.1078846000,
        475: 0.1126000000,
        476: 0.1175320000,
        477: 0.1226744000,
        478: 0.1279928000,
        479: 0.1334528000,
        480: 0.1390200000,
        481: 0.1446764000,
        482: 0.1504693000,
        483: 0.1564619000,
        484: 0.1627177000,
        485: 0.1693000000,
        486: 0.1762431000,
        487: 0.1835581000,
        488: 0.1912735000,
        489: 0.1994180000,
        490: 0.2080200000,
        491: 0.2171199000,
        492: 0.2267345000,
        493: 0.2368571000,
        494: 0.2474812000,
        495: 0.2586000000,
        496: 0.2701849000,
        497: 0.2822939000,
        498: 0.2950505000,
        499: 0.3085780000,
        500: 0.3230000000,
        501: 0.3384021000,
        502: 0.3546858000,
        503: 0.3716986000,
        504: 0.3892875000,
        505: 0.4073000000,
        506: 0.4256299000,
        507: 0.4443096000,
        508: 0.4633944000,
        509: 0.4829395000,
        510: 0.5030000000,
        511: 0.5235693000,
        512: 0.5445120000,
        513: 0.5656900000,
        514: 0.5869653000,
        515: 0.6082000000,
        516: 0.6293456000,
        517: 0.6503068000,
        518: 0.6708752000,
        519: 0.6908424000,
        520: 0.7100000000,
        521: 0.7281852000,
        522: 0.7454636000,
        523: 0.7619694000,
        524: 0.7778368000,
        525: 0.7932000000,
        526: 0.8081104000,
        527: 0.8224962000,
        528: 0.8363068000,
        529: 0.8494916000,
        530: 0.8620000000,
        531: 0.8738108000,
        532: 0.8849624000,
        533: 0.8954936000,
        534: 0.9054432000,
        535: 0.9148501000,
        536: 0.9237348000,
        537: 0.9320924000,
        538: 0.9399226000,
        539: 0.9472252000,
        540: 0.9540000000,
        541: 0.9602561000,
        542: 0.9660074000,
        543: 0.9712606000,
        544: 0.9760225000,
        545: 0.9803000000,
        546: 0.9840924000,
        547: 0.9874182000,
        548: 0.9903128000,
        549: 0.9928116000,
        550: 0.9949501000,
        551: 0.9967108000,
        552: 0.9980983000,
        553: 0.9991120000,
        554: 0.9997482000,
        555: 1.0000000000,
        556: 0.9998567000,
        557: 0.9993046000,
        558: 0.9983255000,
        559: 0.9968987000,
        560: 0.9950000000,
        561: 0.9926005000,
        562: 0.9897426000,
        563: 0.9864444000,
        564: 0.9827241000,
        565: 0.9786000000,
        566: 0.9740837000,
        567: 0.9691712000,
        568: 0.9638568000,
        569: 0.9581349000,
        570: 0.9520000000,
        571: 0.9454504000,
        572: 0.9384992000,
        573: 0.9311628000,
        574: 0.9234576000,
        575: 0.9154000000,
        576: 0.9070064000,
        577: 0.8982772000,
        578: 0.8892048000,
        579: 0.8797816000,
        580: 0.8700000000,
        581: 0.8598613000,
        582: 0.8493920000,
        583: 0.8386220000,
        584: 0.8275813000,
        585: 0.8163000000,
        586: 0.8047947000,
        587: 0.7930820000,
        588: 0.7811920000,
        589: 0.7691547000,
        590: 0.7570000000,
        591: 0.7447541000,
        592: 0.7324224000,
        593: 0.7200036000,
        594: 0.7074965000,
        595: 0.6949000000,
        596: 0.6822192000,
        597: 0.6694716000,
        598: 0.6566744000,
        599: 0.6438448000,
        600: 0.6310000000,
        601: 0.6181555000,
        602: 0.6053144000,
        603: 0.5924756000,
        604: 0.5796379000,
        605: 0.5668000000,
        606: 0.5539611000,
        607: 0.5411372000,
        608: 0.5283528000,
        609: 0.5156323000,
        610: 0.5030000000,
        611: 0.4904688000,
        612: 0.4780304000,
        613: 0.4656776000,
        614: 0.4534032000,
        615: 0.4412000000,
        616: 0.4290800000,
        617: 0.4170360000,
        618: 0.4050320000,
        619: 0.3930320000,
        620: 0.3810000000,
        621: 0.3689184000,
        622: 0.3568272000,
        623: 0.3447768000,
        624: 0.3328176000,
        625: 0.3210000000,
        626: 0.3093381000,
        627: 0.2978504000,
        628: 0.2865936000,
        629: 0.2756245000,
        630: 0.2650000000,
        631: 0.2547632000,
        632: 0.2448896000,
        633: 0.2353344000,
        634: 0.2260528000,
        635: 0.2170000000,
        636: 0.2081616000,
        637: 0.1995488000,
        638: 0.1911552000,
        639: 0.1829744000,
        640: 0.1750000000,
        641: 0.1672235000,
        642: 0.1596464000,
        643: 0.1522776000,
        644: 0.1451259000,
        645: 0.1382000000,
        646: 0.1315003000,
        647: 0.1250248000,
        648: 0.1187792000,
        649: 0.1127691000,
        650: 0.1070000000,
        651: 0.1014762000,
        652: 0.0961886400,
        653: 0.0911229600,
        654: 0.0862648500,
        655: 0.0816000000,
        656: 0.0771206400,
        657: 0.0728255200,
        658: 0.0687100800,
        659: 0.0647697600,
        660: 0.0610000000,
        661: 0.0573962100,
        662: 0.0539550400,
        663: 0.0506737600,
        664: 0.0475496500,
        665: 0.0445800000,
        666: 0.0417587200,
        667: 0.0390849600,
        668: 0.0365638400,
        669: 0.0342004800,
        670: 0.0320000000,
        671: 0.0299626100,
        672: 0.0280766400,
        673: 0.0263293600,
        674: 0.0247080500,
        675: 0.0232000000,
        676: 0.0218007700,
        677: 0.0205011200,
        678: 0.0192810800,
        679: 0.0181206900,
        680: 0.0170000000,
        681: 0.0159037900,
        682: 0.0148371800,
        683: 0.0138106800,
        684: 0.0128347800,
        685: 0.0119200000,
        686: 0.0110683100,
        687: 0.0102733900,
        688: 0.0095333110,
        689: 0.0088461570,
        690: 0.0082100000,
        691: 0.0076237810,
        692: 0.0070854240,
        693: 0.0065914760,
        694: 0.0061384850,
        695: 0.0057230000,
        696: 0.0053430590,
        697: 0.0049957960,
        698: 0.0046764040,
        699: 0.0043800750,
        700: 0.0041020000,
        701: 0.0038384530,
        702: 0.0035890990,
        703: 0.0033542190,
        704: 0.0031340930,
        705: 0.0029290000,
        706: 0.0027381390,
        707: 0.0025598760,
        708: 0.0023932440,
        709: 0.0022372750,
        710: 0.0020910000,
        711: 0.0019535870,
        712: 0.0018245800,
        713: 0.0017035800,
        714: 0.0015901870,
        715: 0.0014840000,
        716: 0.0013844960,
        717: 0.0012912680,
        718: 0.0012040920,
        719: 0.0011227440,
        720: 0.0010470000,
        721: 0.0009765896,
        722: 0.0009111088,
        723: 0.0008501332,
        724: 0.0007932384,
        725: 0.0007400000,
        726: 0.0006900827,
        727: 0.0006433100,
        728: 0.0005994960,
        729: 0.0005584547,
        730: 0.0005200000,
        731: 0.0004839136,
        732: 0.0004500528,
        733: 0.0004183452,
        734: 0.0003887184,
        735: 0.0003611000,
        736: 0.0003353835,
        737: 0.0003114404,
        738: 0.0002891656,
        739: 0.0002684539,
        740: 0.0002492000,
        741: 0.0002313019,
        742: 0.0002146856,
        743: 0.0001992884,
        744: 0.0001850475,
        745: 0.0001719000,
        746: 0.0001597781,
        747: 0.0001486044,
        748: 0.0001383016,
        749: 0.0001287925,
        750: 0.0001200000,
        751: 0.0001118595,
        752: 0.0001043224,
        753: 0.0000973356,
        754: 0.0000908459,
        755: 0.0000848000,
        756: 0.0000791467,
        757: 0.0000738580,
        758: 0.0000689160,
        759: 0.0000643027,
        760: 0.0000600000,
        761: 0.0000559819,
        762: 0.0000522256,
        763: 0.0000487184,
        764: 0.0000454475,
        765: 0.0000424000,
        766: 0.0000395610,
        767: 0.0000369151,
        768: 0.0000344487,
        769: 0.0000321482,
        770: 0.0000300000,
        771: 0.0000279913,
        772: 0.0000261136,
        773: 0.0000243602,
        774: 0.0000227246,
        775: 0.0000212000,
        776: 0.0000197786,
        777: 0.0000184529,
        778: 0.0000172169,
        779: 0.0000160646,
        780: 0.0000149900
    },
    'CIE 1964 Photopic 10 Degree Standard Observer': {
        360: 0.000000013398,
        361: 0.000000020294,
        362: 0.000000030560,
        363: 0.000000045740,
        364: 0.000000068050,
        365: 0.000000100650,
        366: 0.000000147980,
        367: 0.000000216270,
        368: 0.000000314200,
        369: 0.000000453700,
        370: 0.000000651100,
        371: 0.000000928800,
        372: 0.000001317500,
        373: 0.000001857200,
        374: 0.000002602000,
        375: 0.000003625000,
        376: 0.000005019000,
        377: 0.000006907000,
        378: 0.000009449000,
        379: 0.000012848000,
        380: 0.000017364000,
        381: 0.000023327000,
        382: 0.000031150000,
        383: 0.000041350000,
        384: 0.000054560000,
        385: 0.000071560000,
        386: 0.000093300000,
        387: 0.000120870000,
        388: 0.000155640000,
        389: 0.000199200000,
        390: 0.000253400000,
        391: 0.000320200000,
        392: 0.000402400000,
        393: 0.000502300000,
        394: 0.000623200000,
        395: 0.000768500000,
        396: 0.000941700000,
        397: 0.001147800000,
        398: 0.001390300000,
        399: 0.001674000000,
        400: 0.002004400000,
        401: 0.002386000000,
        402: 0.002822000000,
        403: 0.003319000000,
        404: 0.003880000000,
        405: 0.004509000000,
        406: 0.005209000000,
        407: 0.005985000000,
        408: 0.006833000000,
        409: 0.007757000000,
        410: 0.008756000000,
        411: 0.009816000000,
        412: 0.010918000000,
        413: 0.012058000000,
        414: 0.013237000000,
        415: 0.014456000000,
        416: 0.015717000000,
        417: 0.017025000000,
        418: 0.018399000000,
        419: 0.019848000000,
        420: 0.021391000000,
        421: 0.022992000000,
        422: 0.024598000000,
        423: 0.026213000000,
        424: 0.027841000000,
        425: 0.029497000000,
        426: 0.031195000000,
        427: 0.032927000000,
        428: 0.034738000000,
        429: 0.036654000000,
        430: 0.038676000000,
        431: 0.040792000000,
        432: 0.042946000000,
        433: 0.045114000000,
        434: 0.047333000000,
        435: 0.049602000000,
        436: 0.051934000000,
        437: 0.054337000000,
        438: 0.056822000000,
        439: 0.059399000000,
        440: 0.062077000000,
        441: 0.064737000000,
        442: 0.067285000000,
        443: 0.069764000000,
        444: 0.072218000000,
        445: 0.074704000000,
        446: 0.077272000000,
        447: 0.079979000000,
        448: 0.082874000000,
        449: 0.086000000000,
        450: 0.089456000000,
        451: 0.092947000000,
        452: 0.096275000000,
        453: 0.099535000000,
        454: 0.102829000000,
        455: 0.106256000000,
        456: 0.109901000000,
        457: 0.113835000000,
        458: 0.118167000000,
        459: 0.122932000000,
        460: 0.128201000000,
        461: 0.133457000000,
        462: 0.138323000000,
        463: 0.143042000000,
        464: 0.147787000000,
        465: 0.152761000000,
        466: 0.158102000000,
        467: 0.163941000000,
        468: 0.170362000000,
        469: 0.177425000000,
        470: 0.185190000000,
        471: 0.193025000000,
        472: 0.200313000000,
        473: 0.207156000000,
        474: 0.213644000000,
        475: 0.219940000000,
        476: 0.226170000000,
        477: 0.232467000000,
        478: 0.239025000000,
        479: 0.245997000000,
        480: 0.253589000000,
        481: 0.261876000000,
        482: 0.270643000000,
        483: 0.279645000000,
        484: 0.288694000000,
        485: 0.297665000000,
        486: 0.306469000000,
        487: 0.315035000000,
        488: 0.323335000000,
        489: 0.331366000000,
        490: 0.339133000000,
        491: 0.347860000000,
        492: 0.358326000000,
        493: 0.370001000000,
        494: 0.382464000000,
        495: 0.395379000000,
        496: 0.408482000000,
        497: 0.421588000000,
        498: 0.434619000000,
        499: 0.447601000000,
        500: 0.460777000000,
        501: 0.474340000000,
        502: 0.488200000000,
        503: 0.502340000000,
        504: 0.516740000000,
        505: 0.531360000000,
        506: 0.546190000000,
        507: 0.561180000000,
        508: 0.576290000000,
        509: 0.591500000000,
        510: 0.606741000000,
        511: 0.622150000000,
        512: 0.637830000000,
        513: 0.653710000000,
        514: 0.669680000000,
        515: 0.685660000000,
        516: 0.701550000000,
        517: 0.717230000000,
        518: 0.732570000000,
        519: 0.747460000000,
        520: 0.761757000000,
        521: 0.775340000000,
        522: 0.788220000000,
        523: 0.800460000000,
        524: 0.812140000000,
        525: 0.823330000000,
        526: 0.834120000000,
        527: 0.844600000000,
        528: 0.854870000000,
        529: 0.865040000000,
        530: 0.875211000000,
        531: 0.885370000000,
        532: 0.895370000000,
        533: 0.905150000000,
        534: 0.914650000000,
        535: 0.923810000000,
        536: 0.932550000000,
        537: 0.940810000000,
        538: 0.948520000000,
        539: 0.955600000000,
        540: 0.961988000000,
        541: 0.967540000000,
        542: 0.972230000000,
        543: 0.976170000000,
        544: 0.979460000000,
        545: 0.982200000000,
        546: 0.984520000000,
        547: 0.986520000000,
        548: 0.988320000000,
        549: 0.990020000000,
        550: 0.991761000000,
        551: 0.993530000000,
        552: 0.995230000000,
        553: 0.996770000000,
        554: 0.998090000000,
        555: 0.999110000000,
        556: 0.999770000000,
        557: 1.000000000000,
        558: 0.999710000000,
        559: 0.998850000000,
        560: 0.997340000000,
        561: 0.995260000000,
        562: 0.992740000000,
        563: 0.989750000000,
        564: 0.986300000000,
        565: 0.982380000000,
        566: 0.977980000000,
        567: 0.973110000000,
        568: 0.967740000000,
        569: 0.961890000000,
        570: 0.955552000000,
        571: 0.948601000000,
        572: 0.940981000000,
        573: 0.932798000000,
        574: 0.924158000000,
        575: 0.915175000000,
        576: 0.905954000000,
        577: 0.896608000000,
        578: 0.887249000000,
        579: 0.877986000000,
        580: 0.868934000000,
        581: 0.860164000000,
        582: 0.851519000000,
        583: 0.842963000000,
        584: 0.834393000000,
        585: 0.825623000000,
        586: 0.816764000000,
        587: 0.807544000000,
        588: 0.797947000000,
        589: 0.787893000000,
        590: 0.777405000000,
        591: 0.766490000000,
        592: 0.755309000000,
        593: 0.743845000000,
        594: 0.732190000000,
        595: 0.720353000000,
        596: 0.708281000000,
        597: 0.696055000000,
        598: 0.683621000000,
        599: 0.671048000000,
        600: 0.658341000000,
        601: 0.645545000000,
        602: 0.632718000000,
        603: 0.619815000000,
        604: 0.606887000000,
        605: 0.593878000000,
        606: 0.580781000000,
        607: 0.567653000000,
        608: 0.554490000000,
        609: 0.541228000000,
        610: 0.527963000000,
        611: 0.514634000000,
        612: 0.501363000000,
        613: 0.488124000000,
        614: 0.474935000000,
        615: 0.461834000000,
        616: 0.448823000000,
        617: 0.435917000000,
        618: 0.423153000000,
        619: 0.410526000000,
        620: 0.398057000000,
        621: 0.385835000000,
        622: 0.373951000000,
        623: 0.362311000000,
        624: 0.350863000000,
        625: 0.339554000000,
        626: 0.328309000000,
        627: 0.317118000000,
        628: 0.305936000000,
        629: 0.294737000000,
        630: 0.283493000000,
        631: 0.272222000000,
        632: 0.260990000000,
        633: 0.249877000000,
        634: 0.238946000000,
        635: 0.228254000000,
        636: 0.217853000000,
        637: 0.207780000000,
        638: 0.198072000000,
        639: 0.188748000000,
        640: 0.179828000000,
        641: 0.171285000000,
        642: 0.163059000000,
        643: 0.155151000000,
        644: 0.147535000000,
        645: 0.140211000000,
        646: 0.133170000000,
        647: 0.126400000000,
        648: 0.119892000000,
        649: 0.113640000000,
        650: 0.107633000000,
        651: 0.101870000000,
        652: 0.096347000000,
        653: 0.091063000000,
        654: 0.086010000000,
        655: 0.081187000000,
        656: 0.076583000000,
        657: 0.072198000000,
        658: 0.068024000000,
        659: 0.064052000000,
        660: 0.060281000000,
        661: 0.056697000000,
        662: 0.053292000000,
        663: 0.050059000000,
        664: 0.046998000000,
        665: 0.044096000000,
        666: 0.041345000000,
        667: 0.038750700000,
        668: 0.036297800000,
        669: 0.033983200000,
        670: 0.031800400000,
        671: 0.029739500000,
        672: 0.027791800000,
        673: 0.025955100000,
        674: 0.024226300000,
        675: 0.022601700000,
        676: 0.021077900000,
        677: 0.019650500000,
        678: 0.018315300000,
        679: 0.017068600000,
        680: 0.015905100000,
        681: 0.014818300000,
        682: 0.013800800000,
        683: 0.012849500000,
        684: 0.011960700000,
        685: 0.011130300000,
        686: 0.010355500000,
        687: 0.009633200000,
        688: 0.008959900000,
        689: 0.008332400000,
        690: 0.007748800000,
        691: 0.007204600000,
        692: 0.006697500000,
        693: 0.006225100000,
        694: 0.005785000000,
        695: 0.005375100000,
        696: 0.004994100000,
        697: 0.004639200000,
        698: 0.004309300000,
        699: 0.004002800000,
        700: 0.003717740000,
        701: 0.003452620000,
        702: 0.003205830000,
        703: 0.002976230000,
        704: 0.002762810000,
        705: 0.002564560000,
        706: 0.002380480000,
        707: 0.002209710000,
        708: 0.002051320000,
        709: 0.001904490000,
        710: 0.001768470000,
        711: 0.001642360000,
        712: 0.001525350000,
        713: 0.001416720000,
        714: 0.001315950000,
        715: 0.001222390000,
        716: 0.001135550000,
        717: 0.001054940000,
        718: 0.000980140000,
        719: 0.000910660000,
        720: 0.000846190000,
        721: 0.000786290000,
        722: 0.000730680000,
        723: 0.000678990000,
        724: 0.000631010000,
        725: 0.000586440000,
        726: 0.000545110000,
        727: 0.000506720000,
        728: 0.000471110000,
        729: 0.000438050000,
        730: 0.000407410000,
        731: 0.000378962000,
        732: 0.000352543000,
        733: 0.000328001000,
        734: 0.000305208000,
        735: 0.000284041000,
        736: 0.000264375000,
        737: 0.000246109000,
        738: 0.000229143000,
        739: 0.000213376000,
        740: 0.000198730000,
        741: 0.000185115000,
        742: 0.000172454000,
        743: 0.000160678000,
        744: 0.000149730000,
        745: 0.000139550000,
        746: 0.000130086000,
        747: 0.000121290000,
        748: 0.000113106000,
        749: 0.000105501000,
        750: 0.000098428000,
        751: 0.000091853000,
        752: 0.000085738000,
        753: 0.000080048000,
        754: 0.000074751000,
        755: 0.000069819000,
        756: 0.000065222000,
        757: 0.000060939000,
        758: 0.000056942000,
        759: 0.000053217000,
        760: 0.000049737000,
        761: 0.000046491000,
        762: 0.000043464000,
        763: 0.000040635000,
        764: 0.000038000000,
        765: 0.000035540500,
        766: 0.000033244800,
        767: 0.000031100600,
        768: 0.000029099000,
        769: 0.000027230700,
        770: 0.000025486000,
        771: 0.000023856100,
        772: 0.000022333200,
        773: 0.000020910400,
        774: 0.000019580800,
        775: 0.000018338400,
        776: 0.000017177700,
        777: 0.000016093400,
        778: 0.000015080000,
        779: 0.000014133600,
        780: 0.000013249000,
        781: 0.000012422600,
        782: 0.000011649900,
        783: 0.000010927700,
        784: 0.000010251900,
        785: 0.000009619600,
        786: 0.000009028100,
        787: 0.000008474000,
        788: 0.000007954800,
        789: 0.000007468600,
        790: 0.000007012800,
        791: 0.000006585800,
        792: 0.000006185700,
        793: 0.000005810700,
        794: 0.000005459000,
        795: 0.000005129800,
        796: 0.000004820600,
        797: 0.000004531200,
        798: 0.000004259100,
        799: 0.000004004200,
        800: 0.000003764730,
        801: 0.000003539950,
        802: 0.000003329140,
        803: 0.000003131150,
        804: 0.000002945290,
        805: 0.000002770810,
        806: 0.000002607050,
        807: 0.000002453290,
        808: 0.000002308940,
        809: 0.000002173380,
        810: 0.000002046130,
        811: 0.000001926620,
        812: 0.000001814400,
        813: 0.000001708950,
        814: 0.000001609880,
        815: 0.000001516770,
        816: 0.000001429210,
        817: 0.000001346860,
        818: 0.000001269450,
        819: 0.000001196620,
        820: 0.000001128090,
        821: 0.000001063680,
        822: 0.000001003130,
        823: 0.000000946220,
        824: 0.000000892630,
        825: 0.000000842160,
        826: 0.000000794640,
        827: 0.000000749780,
        828: 0.000000707440,
        829: 0.000000667480,
        830: 0.000000629700
    },
    'CIE 2008 2 Degree Physiologically Relevant LEF': {
        390: 4.14616e-04,
        391: 5.02833e-04,
        392: 6.08499e-04,
        393: 7.34444e-04,
        394: 8.83739e-04,
        395: 1.05965e-03,
        396: 1.26553e-03,
        397: 1.50475e-03,
        398: 1.78049e-03,
        399: 2.09557e-03,
        400: 2.45219e-03,
        401: 2.85222e-03,
        402: 3.29912e-03,
        403: 3.79747e-03,
        404: 4.35277e-03,
        405: 4.97172e-03,
        406: 5.66101e-03,
        407: 6.42161e-03,
        408: 7.25031e-03,
        409: 8.14017e-03,
        410: 9.07986e-03,
        411: 1.00561e-02,
        412: 1.10646e-02,
        413: 1.21052e-02,
        414: 1.31801e-02,
        415: 1.42938e-02,
        416: 1.54500e-02,
        417: 1.66409e-02,
        418: 1.78530e-02,
        419: 1.90702e-02,
        420: 2.02737e-02,
        421: 2.14481e-02,
        422: 2.26004e-02,
        423: 2.37479e-02,
        424: 2.49125e-02,
        425: 2.61211e-02,
        426: 2.73992e-02,
        427: 2.87499e-02,
        428: 3.01691e-02,
        429: 3.16514e-02,
        430: 3.31904e-02,
        431: 3.47791e-02,
        432: 3.64149e-02,
        433: 3.80957e-02,
        434: 3.98184e-02,
        435: 4.15794e-02,
        436: 4.33710e-02,
        437: 4.51718e-02,
        438: 4.69542e-02,
        439: 4.86872e-02,
        440: 5.03366e-02,
        441: 5.18761e-02,
        442: 5.33222e-02,
        443: 5.47060e-02,
        444: 5.60634e-02,
        445: 5.74339e-02,
        446: 5.88511e-02,
        447: 6.03081e-02,
        448: 6.17864e-02,
        449: 6.32657e-02,
        450: 6.47235e-02,
        451: 6.61475e-02,
        452: 6.75726e-02,
        453: 6.90493e-02,
        454: 7.06328e-02,
        455: 7.23834e-02,
        456: 7.43596e-02,
        457: 7.65938e-02,
        458: 7.91144e-02,
        459: 8.19535e-02,
        460: 8.51482e-02,
        461: 8.87266e-02,
        462: 9.26601e-02,
        463: 9.68972e-02,
        464: 1.01375e-01,
        465: 1.06014e-01,
        466: 1.10738e-01,
        467: 1.15511e-01,
        468: 1.20312e-01,
        469: 1.25116e-01,
        470: 1.29896e-01,
        471: 1.34630e-01,
        472: 1.39331e-01,
        473: 1.44023e-01,
        474: 1.48737e-01,
        475: 1.53507e-01,
        476: 1.58364e-01,
        477: 1.63320e-01,
        478: 1.68376e-01,
        479: 1.73537e-01,
        480: 1.78805e-01,
        481: 1.84182e-01,
        482: 1.89656e-01,
        483: 1.95210e-01,
        484: 2.00826e-01,
        485: 2.06483e-01,
        486: 2.12183e-01,
        487: 2.18028e-01,
        488: 2.24159e-01,
        489: 2.30730e-01,
        490: 2.37916e-01,
        491: 2.45871e-01,
        492: 2.54602e-01,
        493: 2.64076e-01,
        494: 2.74249e-01,
        495: 2.85068e-01,
        496: 2.96484e-01,
        497: 3.08501e-01,
        498: 3.21139e-01,
        499: 3.34418e-01,
        500: 3.48354e-01,
        501: 3.62960e-01,
        502: 3.78228e-01,
        503: 3.94136e-01,
        504: 4.10658e-01,
        505: 4.27760e-01,
        506: 4.45399e-01,
        507: 4.63540e-01,
        508: 4.82138e-01,
        509: 5.01143e-01,
        510: 5.20497e-01,
        511: 5.40139e-01,
        512: 5.60021e-01,
        513: 5.80097e-01,
        514: 6.00317e-01,
        515: 6.20626e-01,
        516: 6.40940e-01,
        517: 6.61077e-01,
        518: 6.80813e-01,
        519: 6.99904e-01,
        520: 7.18089e-01,
        521: 7.35159e-01,
        522: 7.51182e-01,
        523: 7.66314e-01,
        524: 7.80735e-01,
        525: 7.94645e-01,
        526: 8.08207e-01,
        527: 8.21382e-01,
        528: 8.34070e-01,
        529: 8.46171e-01,
        530: 8.57580e-01,
        531: 8.68241e-01,
        532: 8.78306e-01,
        533: 8.87991e-01,
        534: 8.97521e-01,
        535: 9.07135e-01,
        536: 9.16995e-01,
        537: 9.26929e-01,
        538: 9.36673e-01,
        539: 9.45948e-01,
        540: 9.54468e-01,
        541: 9.61983e-01,
        542: 9.68439e-01,
        543: 9.73829e-01,
        544: 9.78152e-01,
        545: 9.81411e-01,
        546: 9.83667e-01,
        547: 9.85208e-01,
        548: 9.86381e-01,
        549: 9.87536e-01,
        550: 9.89023e-01,
        551: 9.91081e-01,
        552: 9.93491e-01,
        553: 9.95917e-01,
        554: 9.98021e-01,
        555: 9.99461e-01,
        556: 9.99993e-01,
        557: 9.99756e-01,
        558: 9.98984e-01,
        559: 9.97912e-01,
        560: 9.96774e-01,
        561: 9.95736e-01,
        562: 9.94711e-01,
        563: 9.93553e-01,
        564: 9.92116e-01,
        565: 9.90255e-01,
        566: 9.87860e-01,
        567: 9.84932e-01,
        568: 9.81504e-01,
        569: 9.77603e-01,
        570: 9.73261e-01,
        571: 9.68476e-01,
        572: 9.63137e-01,
        573: 9.57106e-01,
        574: 9.50254e-01,
        575: 9.42457e-01,
        576: 9.33690e-01,
        577: 9.24289e-01,
        578: 9.14671e-01,
        579: 9.05233e-01,
        580: 8.96361e-01,
        581: 8.88307e-01,
        582: 8.80846e-01,
        583: 8.73645e-01,
        584: 8.66376e-01,
        585: 8.58720e-01,
        586: 8.50430e-01,
        587: 8.41505e-01,
        588: 8.32011e-01,
        589: 8.22015e-01,
        590: 8.11587e-01,
        591: 8.00787e-01,
        592: 7.89652e-01,
        593: 7.78205e-01,
        594: 7.66473e-01,
        595: 7.54479e-01,
        596: 7.42247e-01,
        597: 7.29823e-01,
        598: 7.17252e-01,
        599: 7.04582e-01,
        600: 6.91855e-01,
        601: 6.79101e-01,
        602: 6.66285e-01,
        603: 6.53359e-01,
        604: 6.40281e-01,
        605: 6.27007e-01,
        606: 6.13515e-01,
        607: 5.99849e-01,
        608: 5.86068e-01,
        609: 5.72226e-01,
        610: 5.58375e-01,
        611: 5.44554e-01,
        612: 5.30767e-01,
        613: 5.17013e-01,
        614: 5.03289e-01,
        615: 4.89595e-01,
        616: 4.75944e-01,
        617: 4.62396e-01,
        618: 4.49015e-01,
        619: 4.35862e-01,
        620: 4.22990e-01,
        621: 4.10415e-01,
        622: 3.98036e-01,
        623: 3.85730e-01,
        624: 3.73391e-01,
        625: 3.60924e-01,
        626: 3.48286e-01,
        627: 3.35570e-01,
        628: 3.22896e-01,
        629: 3.10370e-01,
        630: 2.98086e-01,
        631: 2.86116e-01,
        632: 2.74482e-01,
        633: 2.63195e-01,
        634: 2.52263e-01,
        635: 2.41690e-01,
        636: 2.31481e-01,
        637: 2.21638e-01,
        638: 2.12162e-01,
        639: 2.03054e-01,
        640: 1.94312e-01,
        641: 1.85923e-01,
        642: 1.77827e-01,
        643: 1.69965e-01,
        644: 1.62284e-01,
        645: 1.54740e-01,
        646: 1.47308e-01,
        647: 1.40017e-01,
        648: 1.32901e-01,
        649: 1.25991e-01,
        650: 1.19312e-01,
        651: 1.12882e-01,
        652: 1.06711e-01,
        653: 1.00805e-01,
        654: 9.51665e-02,
        655: 8.97959e-02,
        656: 8.46904e-02,
        657: 7.98401e-02,
        658: 7.52337e-02,
        659: 7.08606e-02,
        660: 6.67104e-02,
        661: 6.27736e-02,
        662: 5.90418e-02,
        663: 5.55070e-02,
        664: 5.21614e-02,
        665: 4.89970e-02,
        666: 4.60058e-02,
        667: 4.31788e-02,
        668: 4.05075e-02,
        669: 3.79838e-02,
        670: 3.55998e-02,
        671: 3.33486e-02,
        672: 3.12233e-02,
        673: 2.92178e-02,
        674: 2.73260e-02,
        675: 2.55422e-02,
        676: 2.38612e-02,
        677: 2.22786e-02,
        678: 2.07902e-02,
        679: 1.93919e-02,
        680: 1.80794e-02,
        681: 1.68482e-02,
        682: 1.56919e-02,
        683: 1.46045e-02,
        684: 1.35806e-02,
        685: 1.26157e-02,
        686: 1.17070e-02,
        687: 1.08561e-02,
        688: 1.00648e-02,
        689: 9.33338e-03,
        690: 8.66128e-03,
        691: 8.04605e-03,
        692: 7.48113e-03,
        693: 6.95999e-03,
        694: 6.47707e-03,
        695: 6.02768e-03,
        696: 5.60817e-03,
        697: 5.21669e-03,
        698: 4.85179e-03,
        699: 4.51201e-03,
        700: 4.19594e-03,
        701: 3.90206e-03,
        702: 3.62837e-03,
        703: 3.37301e-03,
        704: 3.13432e-03,
        705: 2.91086e-03,
        706: 2.70153e-03,
        707: 2.50580e-03,
        708: 2.32323e-03,
        709: 2.15333e-03,
        710: 1.99556e-03,
        711: 1.84932e-03,
        712: 1.71398e-03,
        713: 1.58890e-03,
        714: 1.47345e-03,
        715: 1.36702e-03,
        716: 1.26895e-03,
        717: 1.17842e-03,
        718: 1.09464e-03,
        719: 1.01694e-03,
        720: 9.44727e-04,
        721: 8.77517e-04,
        722: 8.15044e-04,
        723: 7.57076e-04,
        724: 7.03376e-04,
        725: 6.53705e-04,
        726: 6.07805e-04,
        727: 5.65344e-04,
        728: 5.26005e-04,
        729: 4.89506e-04,
        730: 4.55597e-04,
        731: 4.24055e-04,
        732: 3.94686e-04,
        733: 3.67318e-04,
        734: 3.41794e-04,
        735: 3.17974e-04,
        736: 2.95744e-04,
        737: 2.75056e-04,
        738: 2.55864e-04,
        739: 2.38114e-04,
        740: 2.21745e-04,
        741: 2.06671e-04,
        742: 1.92747e-04,
        743: 1.79831e-04,
        744: 1.67802e-04,
        745: 1.56557e-04,
        746: 1.46017e-04,
        747: 1.36153e-04,
        748: 1.26945e-04,
        749: 1.18367e-04,
        750: 1.10393e-04,
        751: 1.02991e-04,
        752: 9.61184e-05,
        753: 8.97332e-05,
        754: 8.37969e-05,
        755: 7.82744e-05,
        756: 7.31331e-05,
        757: 6.83414e-05,
        758: 6.38704e-05,
        759: 5.96939e-05,
        760: 5.57886e-05,
        761: 5.21351e-05,
        762: 4.87218e-05,
        763: 4.55385e-05,
        764: 4.25744e-05,
        765: 3.98188e-05,
        766: 3.72588e-05,
        767: 3.48747e-05,
        768: 3.26477e-05,
        769: 3.05614e-05,
        770: 2.86018e-05,
        771: 2.67584e-05,
        772: 2.50294e-05,
        773: 2.34137e-05,
        774: 2.19091e-05,
        775: 2.05126e-05,
        776: 1.92190e-05,
        777: 1.80180e-05,
        778: 1.68990e-05,
        779: 1.58531e-05,
        780: 1.48724e-05,
        781: 1.39509e-05,
        782: 1.30853e-05,
        783: 1.22733e-05,
        784: 1.15123e-05,
        785: 1.08000e-05,
        786: 1.01336e-05,
        787: 9.50992e-06,
        788: 8.92563e-06,
        789: 8.37785e-06,
        790: 7.86392e-06,
        791: 7.38154e-06,
        792: 6.92910e-06,
        793: 6.50514e-06,
        794: 6.10822e-06,
        795: 5.73694e-06,
        796: 5.38983e-06,
        797: 5.06527e-06,
        798: 4.76167e-06,
        799: 4.47756e-06,
        800: 4.21160e-06,
        801: 3.96246e-06,
        802: 3.72867e-06,
        803: 3.50888e-06,
        804: 3.30187e-06,
        805: 3.10656e-06,
        806: 2.92212e-06,
        807: 2.74821e-06,
        808: 2.58456e-06,
        809: 2.43087e-06,
        810: 2.28679e-06,
        811: 2.15191e-06,
        812: 2.02566e-06,
        813: 1.90746e-06,
        814: 1.79679e-06,
        815: 1.69315e-06,
        816: 1.59603e-06,
        817: 1.50490e-06,
        818: 1.41925e-06,
        819: 1.33860e-06,
        820: 1.26256e-06,
        821: 1.19077e-06,
        822: 1.12303e-06,
        823: 1.05915e-06,
        824: 9.98951e-07,
        825: 9.42251e-07,
        826: 8.88880e-07,
        827: 8.38669e-07,
        828: 7.91454e-07,
        829: 7.47077e-07,
        830: 7.05386e-07
    },
    'CIE 2008 10 Degree Physiologically Relevant LEF': {
        390: 4.07678e-04,
        391: 4.97777e-04,
        392: 6.06475e-04,
        393: 7.37004e-04,
        394: 8.92939e-04,
        395: 1.07817e-03,
        396: 1.29682e-03,
        397: 1.55316e-03,
        398: 1.85146e-03,
        399: 2.19579e-03,
        400: 2.58977e-03,
        401: 3.03680e-03,
        402: 3.54193e-03,
        403: 4.11142e-03,
        404: 4.75262e-03,
        405: 5.47421e-03,
        406: 6.28503e-03,
        407: 7.18807e-03,
        408: 8.18179e-03,
        409: 9.26042e-03,
        410: 1.04130e-02,
        411: 1.16264e-02,
        412: 1.28988e-02,
        413: 1.42344e-02,
        414: 1.56408e-02,
        415: 1.71297e-02,
        416: 1.87127e-02,
        417: 2.03839e-02,
        418: 2.21294e-02,
        419: 2.39299e-02,
        420: 2.57613e-02,
        421: 2.76016e-02,
        422: 2.94551e-02,
        423: 3.13388e-02,
        424: 3.32758e-02,
        425: 3.52955e-02,
        426: 3.74271e-02,
        427: 3.96714e-02,
        428: 4.20200e-02,
        429: 4.44617e-02,
        430: 4.69823e-02,
        431: 4.95674e-02,
        432: 5.22122e-02,
        433: 5.49139e-02,
        434: 5.76692e-02,
        435: 6.04743e-02,
        436: 6.33220e-02,
        437: 6.61927e-02,
        438: 6.90619e-02,
        439: 7.19019e-02,
        440: 7.46829e-02,
        441: 7.73845e-02,
        442: 8.00360e-02,
        443: 8.26852e-02,
        444: 8.53875e-02,
        445: 8.82054e-02,
        446: 9.11893e-02,
        447: 9.43104e-02,
        448: 9.75135e-02,
        449: 1.00735e-01,
        450: 1.03903e-01,
        451: 1.06964e-01,
        452: 1.09968e-01,
        453: 1.12999e-01,
        454: 1.16154e-01,
        455: 1.19539e-01,
        456: 1.23250e-01,
        457: 1.27305e-01,
        458: 1.31696e-01,
        459: 1.36418e-01,
        460: 1.41459e-01,
        461: 1.46800e-01,
        462: 1.52400e-01,
        463: 1.58202e-01,
        464: 1.64140e-01,
        465: 1.70137e-01,
        466: 1.76123e-01,
        467: 1.82090e-01,
        468: 1.88046e-01,
        469: 1.94006e-01,
        470: 1.99986e-01,
        471: 2.06005e-01,
        472: 2.12098e-01,
        473: 2.18304e-01,
        474: 2.24669e-01,
        475: 2.31243e-01,
        476: 2.38074e-01,
        477: 2.45180e-01,
        478: 2.52568e-01,
        479: 2.60248e-01,
        480: 2.68227e-01,
        481: 2.76501e-01,
        482: 2.85004e-01,
        483: 2.93647e-01,
        484: 3.02332e-01,
        485: 3.10944e-01,
        486: 3.19410e-01,
        487: 3.27868e-01,
        488: 3.36526e-01,
        489: 3.45618e-01,
        490: 3.55402e-01,
        491: 3.66089e-01,
        492: 3.77586e-01,
        493: 3.89696e-01,
        494: 4.02195e-01,
        495: 4.14823e-01,
        496: 4.27354e-01,
        497: 4.39821e-01,
        498: 4.52336e-01,
        499: 4.65030e-01,
        500: 4.78048e-01,
        501: 4.91517e-01,
        502: 5.05422e-01,
        503: 5.19706e-01,
        504: 5.34301e-01,
        505: 5.49134e-01,
        506: 5.64130e-01,
        507: 5.79242e-01,
        508: 5.94426e-01,
        509: 6.09639e-01,
        510: 6.24830e-01,
        511: 6.39966e-01,
        512: 6.55094e-01,
        513: 6.70290e-01,
        514: 6.85638e-01,
        515: 7.01229e-01,
        516: 7.17110e-01,
        517: 7.33092e-01,
        518: 7.48904e-01,
        519: 7.64253e-01,
        520: 7.78820e-01,
        521: 7.92341e-01,
        522: 8.04851e-01,
        523: 8.16475e-01,
        524: 8.27352e-01,
        525: 8.37636e-01,
        526: 8.47465e-01,
        527: 8.56887e-01,
        528: 8.65924e-01,
        529: 8.74604e-01,
        530: 8.82955e-01,
        531: 8.91027e-01,
        532: 8.98949e-01,
        533: 9.06875e-01,
        534: 9.14965e-01,
        535: 9.23386e-01,
        536: 9.32232e-01,
        537: 9.41286e-01,
        538: 9.50238e-01,
        539: 9.58765e-01,
        540: 9.66532e-01,
        541: 9.73250e-01,
        542: 9.78842e-01,
        543: 9.83287e-01,
        544: 9.86572e-01,
        545: 9.88689e-01,
        546: 9.89706e-01,
        547: 9.89985e-01,
        548: 9.89962e-01,
        549: 9.90073e-01,
        550: 9.90750e-01,
        551: 9.92283e-01,
        552: 9.94384e-01,
        553: 9.96622e-01,
        554: 9.98565e-01,
        555: 9.99778e-01,
        556: 9.99944e-01,
        557: 9.99220e-01,
        558: 9.97879e-01,
        559: 9.96193e-01,
        560: 9.94430e-01,
        561: 9.92783e-01,
        562: 9.91158e-01,
        563: 9.89392e-01,
        564: 9.87329e-01,
        565: 9.84813e-01,
        566: 9.81725e-01,
        567: 9.78071e-01,
        568: 9.73886e-01,
        569: 9.69203e-01,
        570: 9.64055e-01,
        571: 9.58441e-01,
        572: 9.52238e-01,
        573: 9.45297e-01,
        574: 9.37477e-01,
        575: 9.28649e-01,
        576: 9.18795e-01,
        577: 9.08301e-01,
        578: 8.97635e-01,
        579: 8.87240e-01,
        580: 8.77536e-01,
        581: 8.68792e-01,
        582: 8.60747e-01,
        583: 8.53023e-01,
        584: 8.45253e-01,
        585: 8.37084e-01,
        586: 8.28241e-01,
        587: 8.18732e-01,
        588: 8.08635e-01,
        589: 7.98030e-01,
        590: 7.86995e-01,
        591: 7.75604e-01,
        592: 7.63900e-01,
        593: 7.51916e-01,
        594: 7.39683e-01,
        595: 7.27231e-01,
        596: 7.14588e-01,
        597: 7.01793e-01,
        598: 6.88887e-01,
        599: 6.75910e-01,
        600: 6.62904e-01,
        601: 6.49891e-01,
        602: 6.36841e-01,
        603: 6.23709e-01,
        604: 6.10454e-01,
        605: 5.97037e-01,
        606: 5.83440e-01,
        607: 5.69704e-01,
        608: 5.55889e-01,
        609: 5.42047e-01,
        610: 5.28230e-01,
        611: 5.14475e-01,
        612: 5.00788e-01,
        613: 4.87169e-01,
        614: 4.73616e-01,
        615: 4.60131e-01,
        616: 4.46726e-01,
        617: 4.33459e-01,
        618: 4.20392e-01,
        619: 4.07581e-01,
        620: 3.95076e-01,
        621: 3.82889e-01,
        622: 3.70919e-01,
        623: 3.59045e-01,
        624: 3.47162e-01,
        625: 3.35179e-01,
        626: 3.23056e-01,
        627: 3.10886e-01,
        628: 2.98784e-01,
        629: 2.86853e-01,
        630: 2.75181e-01,
        631: 2.63834e-01,
        632: 2.52833e-01,
        633: 2.42183e-01,
        634: 2.31890e-01,
        635: 2.21956e-01,
        636: 2.12383e-01,
        637: 2.03170e-01,
        638: 1.94318e-01,
        639: 1.85825e-01,
        640: 1.77688e-01,
        641: 1.69893e-01,
        642: 1.62382e-01,
        643: 1.55099e-01,
        644: 1.47992e-01,
        645: 1.41020e-01,
        646: 1.34161e-01,
        647: 1.27440e-01,
        648: 1.20889e-01,
        649: 1.14534e-01,
        650: 1.08400e-01,
        651: 1.02501e-01,
        652: 9.68459e-02,
        653: 9.14394e-02,
        654: 8.62832e-02,
        655: 8.13769e-02,
        656: 7.67171e-02,
        657: 7.22940e-02,
        658: 6.80970e-02,
        659: 6.41155e-02,
        660: 6.03398e-02,
        661: 5.67605e-02,
        662: 5.33699e-02,
        663: 5.01603e-02,
        664: 4.71241e-02,
        665: 4.42538e-02,
        666: 4.15421e-02,
        667: 3.89804e-02,
        668: 3.65609e-02,
        669: 3.42760e-02,
        670: 3.21185e-02,
        671: 3.00819e-02,
        672: 2.81600e-02,
        673: 2.63470e-02,
        674: 2.46373e-02,
        675: 2.30257e-02,
        676: 2.15074e-02,
        677: 2.00784e-02,
        678: 1.87347e-02,
        679: 1.74727e-02,
        680: 1.62884e-02,
        681: 1.51777e-02,
        682: 1.41347e-02,
        683: 1.31541e-02,
        684: 1.22309e-02,
        685: 1.13611e-02,
        686: 1.05419e-02,
        687: 9.77505e-03,
        688: 9.06196e-03,
        689: 8.40296e-03,
        690: 7.79746e-03,
        691: 7.24323e-03,
        692: 6.73438e-03,
        693: 6.26500e-03,
        694: 5.83009e-03,
        695: 5.42539e-03,
        696: 5.04763e-03,
        697: 4.69514e-03,
        698: 4.36659e-03,
        699: 4.06069e-03,
        700: 3.77614e-03,
        701: 3.51158e-03,
        702: 3.26521e-03,
        703: 3.03534e-03,
        704: 2.82050e-03,
        705: 2.61937e-03,
        706: 2.43096e-03,
        707: 2.25480e-03,
        708: 2.09049e-03,
        709: 1.93759e-03,
        710: 1.79560e-03,
        711: 1.66399e-03,
        712: 1.54220e-03,
        713: 1.42964e-03,
        714: 1.32575e-03,
        715: 1.22998e-03,
        716: 1.14173e-03,
        717: 1.06027e-03,
        718: 9.84885e-04,
        719: 9.14970e-04,
        720: 8.49990e-04,
        721: 7.89516e-04,
        722: 7.33304e-04,
        723: 6.81146e-04,
        724: 6.32829e-04,
        725: 5.88138e-04,
        726: 5.46839e-04,
        727: 5.08635e-04,
        728: 4.73240e-04,
        729: 4.40402e-04,
        730: 4.09893e-04,
        731: 3.81514e-04,
        732: 3.55090e-04,
        733: 3.30467e-04,
        734: 3.07503e-04,
        735: 2.86072e-04,
        736: 2.66072e-04,
        737: 2.47459e-04,
        738: 2.30192e-04,
        739: 2.14223e-04,
        740: 1.99495e-04,
        741: 1.85934e-04,
        742: 1.73407e-04,
        743: 1.61786e-04,
        744: 1.50964e-04,
        745: 1.40847e-04,
        746: 1.31364e-04,
        747: 1.22490e-04,
        748: 1.14206e-04,
        749: 1.06489e-04,
        750: 9.93144e-05,
        751: 9.26551e-05,
        752: 8.64722e-05,
        753: 8.07278e-05,
        754: 7.53872e-05,
        755: 7.04188e-05,
        756: 6.57934e-05,
        757: 6.14825e-05,
        758: 5.74601e-05,
        759: 5.37027e-05,
        760: 5.01893e-05,
        761: 4.69024e-05,
        762: 4.38317e-05,
        763: 4.09678e-05,
        764: 3.83012e-05,
        765: 3.58222e-05,
        766: 3.35190e-05,
        767: 3.13742e-05,
        768: 2.93707e-05,
        769: 2.74938e-05,
        770: 2.57308e-05,
        771: 2.40725e-05,
        772: 2.25170e-05,
        773: 2.10635e-05,
        774: 1.97099e-05,
        775: 1.84535e-05,
        776: 1.72898e-05,
        777: 1.62093e-05,
        778: 1.52026e-05,
        779: 1.42617e-05,
        780: 1.33795e-05,
        781: 1.25504e-05,
        782: 1.17717e-05,
        783: 1.10412e-05,
        784: 1.03566e-05,
        785: 9.71580e-06,
        786: 9.11632e-06,
        787: 8.55520e-06,
        788: 8.02956e-06,
        789: 7.53677e-06,
        790: 7.07442e-06,
        791: 6.64046e-06,
        792: 6.23344e-06,
        793: 5.85204e-06,
        794: 5.49496e-06,
        795: 5.16095e-06,
        796: 4.84869e-06,
        797: 4.55671e-06,
        798: 4.28358e-06,
        799: 4.02799e-06,
        800: 3.78873e-06,
        801: 3.56460e-06,
        802: 3.35428e-06,
        803: 3.15656e-06,
        804: 2.97033e-06,
        805: 2.79463e-06,
        806: 2.62870e-06,
        807: 2.47225e-06,
        808: 2.32503e-06,
        809: 2.18677e-06,
        810: 2.05715e-06,
        811: 1.93581e-06,
        812: 1.82224e-06,
        813: 1.71591e-06,
        814: 1.61636e-06,
        815: 1.52311e-06,
        816: 1.43575e-06,
        817: 1.35377e-06,
        818: 1.27671e-06,
        819: 1.20417e-06,
        820: 1.13576e-06,
        821: 1.07118e-06,
        822: 1.01024e-06,
        823: 9.52778e-07,
        824: 8.98622e-07,
        825: 8.47617e-07,
        826: 7.99605e-07,
        827: 7.54436e-07,
        828: 7.11962e-07,
        829: 6.72042e-07,
        830: 6.34538e-07
    }
}

PHOTOPIC_LEFS = CaseInsensitiveMapping({
    'CIE 1924 Photopic Standard Observer':
        SpectralPowerDistribution(
            PHOTOPIC_LEFS_DATA['CIE 1924 Photopic Standard Observer'],
            name='CIE 1924 Photopic Standard Observer'),
    'Judd Modified CIE 1951 Photopic Standard Observer':
        SpectralPowerDistribution(
            PHOTOPIC_LEFS_DATA[
                'Judd Modified CIE 1951 Photopic Standard Observer'],
            name='Judd Modified CIE 1951 Photopic Standard Observer'),
    'Judd-Vos Modified CIE 1978 Photopic Standard Observer':
        SpectralPowerDistribution(
            PHOTOPIC_LEFS_DATA[
                'Judd-Vos Modified CIE 1978 Photopic Standard Observer'],
            name='Judd-Vos Modified CIE 1978 Photopic Standard Observer'),
    'CIE 1964 Photopic 10 Degree Standard Observer':
        SpectralPowerDistribution(
            PHOTOPIC_LEFS_DATA[
                'CIE 1964 Photopic 10 Degree Standard Observer'],
            name='CIE 1964 Photopic 10 Degree Standard Observer',
            strict_name='CIE 1964 Photopic 10$^\\circ$ Standard Observer'),
    'CIE 2008 2 Degree Physiologically Relevant LEF':
        SpectralPowerDistribution(
            PHOTOPIC_LEFS_DATA[
                'CIE 2008 2 Degree Physiologically Relevant LEF'],
            name='CIE 2008 2 Degree Physiologically Relevant LEF',
            strict_name='CIE 2008 2$^\\circ$ Physiologically Relevant LEF'),
    'CIE 2008 10 Degree Physiologically Relevant LEF':
        SpectralPowerDistribution(
            PHOTOPIC_LEFS_DATA[
                'CIE 2008 10 Degree Physiologically Relevant LEF'],
            name='CIE 2008 10 Degree Physiologically Relevant LEF',
            strict_name='CIE 2008 10$^\\circ$ Physiologically Relevant LEF')
})
PHOTOPIC_LEFS.__doc__ = """
Photopic luminous efficiency functions.

References
----------
-   :cite:`CVRLq`
-   :cite:`CVRLs`

PHOTOPIC_LEFS : CaseInsensitiveMapping
    **{'CIE 1924 Photopic Standard Observer',
    'Judd Modified CIE 1951 Photopic Standard Observer',
    'Judd-Vos Modified CIE 1978 Photopic Standard Observer',
    'CIE 1964 Photopic 10 Degree Standard Observer',
    'CIE 2008 2 Degree Physiologically Relevant LEF',
    'CIE 2008 10 Degree Physiologically Relevant LEF'}**

Aliases:

-   'cie_2_1924': 'CIE 1931 2 Degree Standard Observer'
-   'cie_10_1964': 'CIE 1964 Photopic 10 Degree Standard Observer'
"""
PHOTOPIC_LEFS['cie_2_1924'] = (
    PHOTOPIC_LEFS['CIE 1924 Photopic Standard Observer'])
PHOTOPIC_LEFS['cie_10_1964'] = (
    PHOTOPIC_LEFS['CIE 1964 Photopic 10 Degree Standard Observer'])

SCOTOPIC_LEFS_DATA = {
    'CIE 1951 Scotopic Standard Observer': {
        380: 0.0005890000,
        381: 0.0006650000,
        382: 0.0007520000,
        383: 0.0008540000,
        384: 0.0009720000,
        385: 0.0011080000,
        386: 0.0012680000,
        387: 0.0014530000,
        388: 0.0016680000,
        389: 0.0019180000,
        390: 0.0022090000,
        391: 0.0025470000,
        392: 0.0029390000,
        393: 0.0033940000,
        394: 0.0039210000,
        395: 0.0045300000,
        396: 0.0052400000,
        397: 0.0060500000,
        398: 0.0069800000,
        399: 0.0080600000,
        400: 0.0092900000,
        401: 0.0107000000,
        402: 0.0123100000,
        403: 0.0141300000,
        404: 0.0161900000,
        405: 0.0185200000,
        406: 0.0211300000,
        407: 0.0240500000,
        408: 0.0273000000,
        409: 0.0308900000,
        410: 0.0348400000,
        411: 0.0391600000,
        412: 0.0439000000,
        413: 0.0490000000,
        414: 0.0545000000,
        415: 0.0604000000,
        416: 0.0668000000,
        417: 0.0736000000,
        418: 0.0808000000,
        419: 0.0885000000,
        420: 0.0966000000,
        421: 0.1052000000,
        422: 0.1141000000,
        423: 0.1235000000,
        424: 0.1334000000,
        425: 0.1436000000,
        426: 0.1541000000,
        427: 0.1651000000,
        428: 0.1764000000,
        429: 0.1879000000,
        430: 0.1998000000,
        431: 0.2119000000,
        432: 0.2243000000,
        433: 0.2369000000,
        434: 0.2496000000,
        435: 0.2625000000,
        436: 0.2755000000,
        437: 0.2886000000,
        438: 0.3017000000,
        439: 0.3149000000,
        440: 0.3281000000,
        441: 0.3412000000,
        442: 0.3543000000,
        443: 0.3673000000,
        444: 0.3803000000,
        445: 0.3931000000,
        446: 0.4060000000,
        447: 0.4180000000,
        448: 0.4310000000,
        449: 0.4430000000,
        450: 0.4550000000,
        451: 0.4670000000,
        452: 0.4790000000,
        453: 0.4900000000,
        454: 0.5020000000,
        455: 0.5130000000,
        456: 0.5240000000,
        457: 0.5350000000,
        458: 0.5460000000,
        459: 0.5570000000,
        460: 0.5670000000,
        461: 0.5780000000,
        462: 0.5880000000,
        463: 0.5990000000,
        464: 0.6100000000,
        465: 0.6200000000,
        466: 0.6310000000,
        467: 0.6420000000,
        468: 0.6530000000,
        469: 0.6640000000,
        470: 0.6760000000,
        471: 0.6870000000,
        472: 0.6990000000,
        473: 0.7100000000,
        474: 0.7220000000,
        475: 0.7340000000,
        476: 0.7450000000,
        477: 0.7570000000,
        478: 0.7690000000,
        479: 0.7810000000,
        480: 0.7930000000,
        481: 0.8050000000,
        482: 0.8170000000,
        483: 0.8280000000,
        484: 0.8400000000,
        485: 0.8510000000,
        486: 0.8620000000,
        487: 0.8730000000,
        488: 0.8840000000,
        489: 0.8940000000,
        490: 0.9040000000,
        491: 0.9140000000,
        492: 0.9230000000,
        493: 0.9320000000,
        494: 0.9410000000,
        495: 0.9490000000,
        496: 0.9570000000,
        497: 0.9640000000,
        498: 0.9700000000,
        499: 0.9760000000,
        500: 0.9820000000,
        501: 0.9860000000,
        502: 0.9900000000,
        503: 0.9940000000,
        504: 0.9970000000,
        505: 0.9980000000,
        506: 1.0000000000,
        507: 1.0000000000,
        508: 1.0000000000,
        509: 0.9980000000,
        510: 0.9970000000,
        511: 0.9940000000,
        512: 0.9900000000,
        513: 0.9860000000,
        514: 0.9810000000,
        515: 0.9750000000,
        516: 0.9680000000,
        517: 0.9610000000,
        518: 0.9530000000,
        519: 0.9440000000,
        520: 0.9350000000,
        521: 0.9250000000,
        522: 0.9150000000,
        523: 0.9040000000,
        524: 0.8920000000,
        525: 0.8800000000,
        526: 0.8670000000,
        527: 0.8540000000,
        528: 0.8400000000,
        529: 0.8260000000,
        530: 0.8110000000,
        531: 0.7960000000,
        532: 0.7810000000,
        533: 0.7650000000,
        534: 0.7490000000,
        535: 0.7330000000,
        536: 0.7170000000,
        537: 0.7000000000,
        538: 0.6830000000,
        539: 0.6670000000,
        540: 0.6500000000,
        541: 0.6330000000,
        542: 0.6160000000,
        543: 0.5990000000,
        544: 0.5810000000,
        545: 0.5640000000,
        546: 0.5480000000,
        547: 0.5310000000,
        548: 0.5140000000,
        549: 0.4970000000,
        550: 0.4810000000,
        551: 0.4650000000,
        552: 0.4480000000,
        553: 0.4330000000,
        554: 0.4170000000,
        555: 0.4020000000,
        556: 0.3864000000,
        557: 0.3715000000,
        558: 0.3569000000,
        559: 0.3427000000,
        560: 0.3288000000,
        561: 0.3151000000,
        562: 0.3018000000,
        563: 0.2888000000,
        564: 0.2762000000,
        565: 0.2639000000,
        566: 0.2519000000,
        567: 0.2403000000,
        568: 0.2291000000,
        569: 0.2182000000,
        570: 0.2076000000,
        571: 0.1974000000,
        572: 0.1876000000,
        573: 0.1782000000,
        574: 0.1690000000,
        575: 0.1602000000,
        576: 0.1517000000,
        577: 0.1436000000,
        578: 0.1358000000,
        579: 0.1284000000,
        580: 0.1212000000,
        581: 0.1143000000,
        582: 0.1078000000,
        583: 0.1015000000,
        584: 0.0956000000,
        585: 0.0899000000,
        586: 0.0845000000,
        587: 0.0793000000,
        588: 0.0745000000,
        589: 0.0699000000,
        590: 0.0655000000,
        591: 0.0613000000,
        592: 0.0574000000,
        593: 0.0537000000,
        594: 0.0502000000,
        595: 0.0469000000,
        596: 0.0438000000,
        597: 0.0409000000,
        598: 0.0381600000,
        599: 0.0355800000,
        600: 0.0331500000,
        601: 0.0308700000,
        602: 0.0287400000,
        603: 0.0267400000,
        604: 0.0248700000,
        605: 0.0231200000,
        606: 0.0214700000,
        607: 0.0199400000,
        608: 0.0185100000,
        609: 0.0171800000,
        610: 0.0159300000,
        611: 0.0147700000,
        612: 0.0136900000,
        613: 0.0126900000,
        614: 0.0117500000,
        615: 0.0108800000,
        616: 0.0100700000,
        617: 0.0093200000,
        618: 0.0086200000,
        619: 0.0079700000,
        620: 0.0073700000,
        621: 0.0068200000,
        622: 0.0063000000,
        623: 0.0058200000,
        624: 0.0053800000,
        625: 0.0049700000,
        626: 0.0045900000,
        627: 0.0042400000,
        628: 0.0039130000,
        629: 0.0036130000,
        630: 0.0033350000,
        631: 0.0030790000,
        632: 0.0028420000,
        633: 0.0026230000,
        634: 0.0024210000,
        635: 0.0022350000,
        636: 0.0020620000,
        637: 0.0019030000,
        638: 0.0017570000,
        639: 0.0016210000,
        640: 0.0014970000,
        641: 0.0013820000,
        642: 0.0012760000,
        643: 0.0011780000,
        644: 0.0010880000,
        645: 0.0010050000,
        646: 0.0009280000,
        647: 0.0008570000,
        648: 0.0007920000,
        649: 0.0007320000,
        650: 0.0006770000,
        651: 0.0006260000,
        652: 0.0005790000,
        653: 0.0005360000,
        654: 0.0004960000,
        655: 0.0004590000,
        656: 0.0004250000,
        657: 0.0003935000,
        658: 0.0003645000,
        659: 0.0003377000,
        660: 0.0003129000,
        661: 0.0002901000,
        662: 0.0002689000,
        663: 0.0002493000,
        664: 0.0002313000,
        665: 0.0002146000,
        666: 0.0001991000,
        667: 0.0001848000,
        668: 0.0001716000,
        669: 0.0001593000,
        670: 0.0001480000,
        671: 0.0001375000,
        672: 0.0001277000,
        673: 0.0001187000,
        674: 0.0001104000,
        675: 0.0001026000,
        676: 0.0000954000,
        677: 0.0000888000,
        678: 0.0000826000,
        679: 0.0000769000,
        680: 0.0000715000,
        681: 0.0000666000,
        682: 0.0000620000,
        683: 0.0000578000,
        684: 0.0000538000,
        685: 0.0000501000,
        686: 0.0000467000,
        687: 0.0000436000,
        688: 0.0000406000,
        689: 0.0000378900,
        690: 0.0000353300,
        691: 0.0000329500,
        692: 0.0000307500,
        693: 0.0000287000,
        694: 0.0000267900,
        695: 0.0000250100,
        696: 0.0000233600,
        697: 0.0000218200,
        698: 0.0000203800,
        699: 0.0000190500,
        700: 0.0000178000,
        701: 0.0000166400,
        702: 0.0000155600,
        703: 0.0000145400,
        704: 0.0000136000,
        705: 0.0000127300,
        706: 0.0000119100,
        707: 0.0000111400,
        708: 0.0000104300,
        709: 0.0000097600,
        710: 0.0000091400,
        711: 0.0000085600,
        712: 0.0000080200,
        713: 0.0000075100,
        714: 0.0000070400,
        715: 0.0000066000,
        716: 0.0000061800,
        717: 0.0000058000,
        718: 0.0000054400,
        719: 0.0000051000,
        720: 0.0000047800,
        721: 0.0000044900,
        722: 0.0000042100,
        723: 0.0000039510,
        724: 0.0000037090,
        725: 0.0000034820,
        726: 0.0000032700,
        727: 0.0000030700,
        728: 0.0000028840,
        729: 0.0000027100,
        730: 0.0000025460,
        731: 0.0000023930,
        732: 0.0000022500,
        733: 0.0000021150,
        734: 0.0000019890,
        735: 0.0000018700,
        736: 0.0000017590,
        737: 0.0000016550,
        738: 0.0000015570,
        739: 0.0000014660,
        740: 0.0000013790,
        741: 0.0000012990,
        742: 0.0000012230,
        743: 0.0000011510,
        744: 0.0000010840,
        745: 0.0000010220,
        746: 0.0000009620,
        747: 0.0000009070,
        748: 0.0000008550,
        749: 0.0000008060,
        750: 0.0000007600,
        751: 0.0000007160,
        752: 0.0000006750,
        753: 0.0000006370,
        754: 0.0000006010,
        755: 0.0000005670,
        756: 0.0000005350,
        757: 0.0000005050,
        758: 0.0000004770,
        759: 0.0000004500,
        760: 0.0000004250,
        761: 0.0000004010,
        762: 0.0000003790,
        763: 0.0000003580,
        764: 0.0000003382,
        765: 0.0000003196,
        766: 0.0000003021,
        767: 0.0000002855,
        768: 0.0000002699,
        769: 0.0000002552,
        770: 0.0000002413,
        771: 0.0000002282,
        772: 0.0000002159,
        773: 0.0000002042,
        774: 0.0000001932,
        775: 0.0000001829,
        776: 0.0000001731,
        777: 0.0000001638,
        778: 0.0000001551,
        779: 0.0000001468,
        780: 0.0000001390,
    }
}

SCOTOPIC_LEFS = CaseInsensitiveMapping({
    'CIE 1951 Scotopic Standard Observer':
        SpectralPowerDistribution(
            SCOTOPIC_LEFS_DATA['CIE 1951 Scotopic Standard Observer'],
            name='CIE 1951 Scotopic Standard Observer')
})
SCOTOPIC_LEFS.__doc__ = """
Scotopic luminous efficiency functions.

References
----------
-   :cite:`CVRLs`

SCOTOPIC_LEFS : CaseInsensitiveMapping
    **{'CIE 1951 Scotopic Standard Observer', }**

Aliases:

-   'cie_1951': 'CIE 1951 Scotopic Standard Observer'
"""
SCOTOPIC_LEFS['cie_1951'] = (
    SCOTOPIC_LEFS['CIE 1951 Scotopic Standard Observer'])

LEFS = CaseInsensitiveMapping(PHOTOPIC_LEFS)
LEFS.__doc__ = """
Aggregated luminous efficiency functions.

References
----------
-   :cite:`CVRLq`
-   :cite:`CVRLs`
-   :cite:`Wikipediacc`

LEFS : CaseInsensitiveMapping
    **{'CIE 1924 Photopic Standard Observer',
    'Judd Modified CIE 1951 Photopic Standard Observer',
    'Judd-Vos Modified CIE 1978 Photopic Standard Observer',
    'CIE 1964 Photopic 10 Degree Standard Observer',
    'CIE 2008 2 Degree Physiologically Relevant LEF',
    'CIE 2008 10 Degree Physiologically Relevant LEF',
    'CIE 1951 Scotopic Standard Observer'}**
"""
LEFS.update(SCOTOPIC_LEFS)

MESOPIC_X_DATA = {
    0.01:
        CaseInsensitiveMapping({
            'Blue Heavy': CaseInsensitiveMapping({
                'MOVE': 0.13,
                'LRC': 0.04
            }),
            'Red Heavy': CaseInsensitiveMapping({
                'MOVE': 0.00,
                'LRC': 0.01
            })
        }),
    0.1:
        CaseInsensitiveMapping({
            'Blue Heavy': CaseInsensitiveMapping({
                'MOVE': 0.42,
                'LRC': 0.28
            }),
            'Red Heavy': CaseInsensitiveMapping({
                'MOVE': 0.34,
                'LRC': 0.11
            })
        }),
    1.0:
        CaseInsensitiveMapping({
            'Blue Heavy': CaseInsensitiveMapping({
                'MOVE': 0.70,
                'LRC': 1.00
            }),
            'Red Heavy': CaseInsensitiveMapping({
                'MOVE': 0.68,
                'LRC': 1.00
            })
        }),
    10:
        CaseInsensitiveMapping({
            'Blue Heavy': CaseInsensitiveMapping({
                'MOVE': 0.98,
                'LRC': 1.00
            }),
            'Red Heavy': CaseInsensitiveMapping({
                'MOVE': 0.98,
                'LRC': 1.00
            })
        })
}
"""
Weighting factors for the mesopic luminous efficiency function calculation.

MESOPIC_X_DATA : CaseInsensitiveMapping
"""
