""" This module contains utility functions for finding the classes contained
    within package or module and its sub-packages or modules.

    Related to the search_package in function_tools
"""

# Standard libary imports
import os
import inspect
from glob import glob
import _ast
import logging


# local copy of Python 2.5 pkgutil.py
from blockcanvas.function_tools import _pkgutil

# Import from function_tools
from blockcanvas.function_tools.search_package import get_module_path, \
    find_package_sub_modules, find_path_sub_modules, python_path_from_file_path, \
    is_package, is_module

# Globals
logger = logging.getLogger(__name__)


##############################################################################
# Main function - this is probably the one you are interested in
##############################################################################

def find_classes(package, import_method=False):
    """ Recursively find all the classes in a package or module.

        By default, find_classes searches for classes by "scanning"
        the code for class definitions.  It does this by using Python's
        AST.  This prevents python from loading all of the modules into
        memory, but it also can miss some functions.  Setting
        import_method=True will actually import the modules in its search
        and may find classes that are not found using the other method.

        Parameters
        ----------
        package: str
            A string such as 'foo.bar' that specifies the package to
            search.  The package must be on the python path.
        import_method: bool
            Default is False.   When True, modules are imported when
            searching for classes.

        Returns
        -------
        classes: list of tuples
            A list of tuples with (module, name). module is a string for
            specifying the python module and the function name.  For example
            function 'foo.bar' would be returned as ('foo','bar')
    """

    if import_method:
        classes = find_classes_import(package)
    else:
        classes = find_classes_ast(package)

    return classes

##############################################################################
# Search helper functions/classes
##############################################################################

def find_classes_ast(package):
    """ Find classes by traversing an abstract syntax tree generated by the
        compiler module.
        fixme: expand docstring, possibly provide response about non-existant
        modules/packages
    """
    classes = []
    # It is a package (ie a directory)
    if is_package(package):
        package_path = get_module_path(package)
        file_paths = find_package_sub_modules(package)
        for file_path in file_paths:
            try:
                file = open(file_path)
                # Adding a new line to the source, so that compile wouldn't
                # throw a SyntaxError on EOF comment
                source = file.read().replace('\r\n','\n')+'\n'
                ast = compile(source, file_path,'exec', _ast.PyCF_ONLY_AST)
                file.close()
                python_path = python_path_from_file_path(package, file_path, package_path=package_path)
                classes.extend(visit_ast_node(ast, file_path, python_path))
            except SyntaxError:
                msg = 'SyntaxError in parsing file %s'% file_path
                logger.error(msg)

    # It is a module (ie a .py file)
    elif is_module(package):
        file_path = get_module_path(package)
        file = open(file_path)

        # Adding a new line to the source, so that compile wouldn't
        # throw a SyntaxError on EOF comment
        source = file.read().replace('\r\n','\n')+'\n'

        ast = compile(source, file_path, 'exec', _ast.PyCF_ONLY_AST)
        file.close()
        classes.extend(visit_ast_node(ast, file_path, package))

    return classes


def visit_ast_node(node, path, python_path):
    """ Given a and _ast node produced by 'compile' with the _ast.PyCF_ONLY_AST
        flag, returns a list of CallableObjects s from that node's toplevel
        functions.
    """
    classes = []

    for child in node.body:
        if isinstance(child, _ast.ClassDef):
            mod_and_name = (python_path, child.name)
            classes.append(mod_and_name)

    return classes


def find_classes_import(package):
    """ Find classes using an import statement. Sloppier and consumes more
        memory than find_classes_ast, but also get submodules of the modules,
        which the ast method cannot do. For example, giving it 'os', it would
        also find functions from os.path. It also follows import statements in
        the files, which may or may not be desirable.
    """
    classes = []
    if is_package(package):
        package_path = get_module_path(package)
        file_paths = find_package_sub_modules(package)

        for file_path in file_paths:
            python_path = python_path_from_file_path(package, file_path, package_path=package_path)
            results = find_classes_import_recurse(python_path)
            classes.extend(results)
    else:
        classes = find_classes_import_recurse(package)

    return classes

def find_classes_import_recurse(module_name):
    """ Search a module and all the modules within it for classes.

        The function imports the module and searches its __dict__ for
        fus.  It also search any module found within the module
        for classes.
    """
    classes = []
    try:
        exec "import " + module_name in globals()
        exec "mod_dict = " + module_name+".__dict__"
    except:
        # Skip the rest of processing.
        mod_dict = {}

    for name, item in mod_dict.items():

        if (inspect.isclass(item) or
            inspect.isbuiltin(item)):
            mod_and_name = (module_name, item.__name__)
            classes.append(mod_and_name)
        elif (inspect.ismodule(item) and
              item is not 'UserDict'):
            results = find_classes_import_recurse(module_name+'.'+name)
            classes.extend(results)

    return classes


if __name__ == "__main__":
    import doctest
    doctest.testmod()
