# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
Miscellaneous writers to be used by runtest and runtests
"""

import os
import sys
import re
from time import asctime, localtime, time

from apycot.interfaces import IWriter
from apycot.utils import SimpleOptionsManagerMixIn, REVERSE_SEVERITIES


class LoggerMixIn(object):
    """Abstract logger class with convenience methods to avoid importing
    level constrants"""

    def warning(self, path, line, msg):
        """log a warning"""
        self.log(WARNING, path, line, msg)

    def error(self, path, line, msg):
        """log an error"""
        self.log(ERROR, path, line, msg)

    def info(self, path, line, msg):
        """log an info"""
        self.log(INFO, path, line, msg)

    def fatal(self, path, line, msg):
        """log a fatal error"""
        self.log(FATAL, path, line, msg)

    def errors(self):
        """return the number of error logged"""
        return self.counts[ERROR]
    errors = property(errors, doc="the number of errors logged")
    
    def warnings(self):
        """return the number of warning logged"""
        return self.counts[WARNING]
    warnings = property(warnings, doc="the number of warnings logged")

    def fatals(self):
        """return the number of fatal error logged"""
        return self.counts[FATAL]
    fatals = property(fatals, doc="the number of fatals logged")

    def infos(self):
        """return the number of info logged"""
        return self.counts[INFO]
    infos = property(infos, doc="the number of infos logged")
   
class BaseMsgWriter(SimpleOptionsManagerMixIn, LoggerMixIn):
    """Abstract base class for PrintWriter and PrettyPrint Writer (implement
    prefix and timestamp)"""
    def __init__(self, verbosity=0, prefix=None, timestamp=False):
        LoggerMixIn.__init__(self)
        SimpleOptionsManagerMixIn.__init__(self)
        self.verbosity = verbosity
        self.prefix = prefix
        self.timestamp = timestamp

    def msg(self, verbose_level, msg, head=None):
        """print a message about what are we currently doing"""
        if self.verbosity >= verbose_level:
            if head is None:
                head = self._head()
            if head:
                msg = ' '.join((head, msg))
            print >> sys.stderr, msg
    def _head(self, sep=' '):
        head_part = []
        if self.timestamp:
            head_part.append(''.join(('[', asctime(localtime(time())), ']')))
        if self.prefix is not None:
            head_part.append(self.prefix)
        return sep.join(head_part)

    def line(self, verbose_level, char='-', length=80):
        if self.verbosity >= verbose_level:
            #if self.timestamp:
            #    start = ''.join(('[', asctime(localtime(time())), ']'))
            #else:
            #    start = ''
            #if self.prefix is not None:
            #    end = ''.join(('<', self.prefix, '>'))
            #else:
            #    end = ''
            start = self._head(char)
            end = ''
            length = max( length - len(start) - len(end), 0)
            line = ''.join((start, char * length, end))
            print >> sys.stderr, line

        

    
    
class PrintWriter(BaseMsgWriter):
    """a writer which print everything to stdout, in a easily parsable format
    """
    
    __implements__ = IWriter

        
    def notify(self, event, **kwargs):
        """ notify an event, with optional values

        possible events are :
        
        * 'open_test', take the module name as argument
        
        * 'open_check', take the checker name as argument
        
        * 'close_check', close the latest opened check, take a 'status'
          argument which should be 'succed', 'failed', 'error' or 'skipped'
          according to the check status
          
        * 'close_test', close the latest opened test (no additional argument)
        """
        attributes = []
        for key, val in kwargs.items():
            attributes.append('|%s=%s' % (key, val))
        print >> sys.stdout, 'NOTIFY|%s%s' % (event, ''.join(attributes))
        sys.stdout.flush()

        
    def log(self, severity, path, line, msg):
        """log a message of a given severity
        
        line may be None if unknown
        """
        print >> sys.stdout, 'LOG|%s|%s|%s|%s' % (REVERSE_SEVERITIES[severity], path,
                                   line or '', msg)
        sys.stdout.flush()

        
    def raw(self, name, value):
        """give some raw data"""
        print >> sys.stdout, 'RAW|%s=%s' % (name, value)
        sys.stdout.flush()

class PrettyPrintWriter(BaseMsgWriter):
    """a writer which print everything to stdout, in a readable format
    """
    
    __implements__ = IWriter

        
    def notify(self, event, **kwargs):
        """ notify an event, with optional values

        possible events are :
        
        * 'open_test', take the module name as argument
        
        * 'open_check', take the checker name as argument
        
        * 'close_check', close the latest opened check, take a 'status'
          argument which should be 'succed', 'failed', 'error' or 'skipped'
           according to the check status
          
        * 'close_test', close the latest opened test (no additional argument)
        """
        if event == 'open_test':
            print '*'*80
            print kwargs['name'].upper()
        elif event == 'open_check':
            print '-'*80
            print kwargs['name'].lower()
        elif event == 'close_check':
            print '[%s]' % unicode(kwargs['status']).upper()

        
    def log(self, severity, path, line, msg):
        """log a message of a given severity
        
        line may be None if unknown
        """
        print '%s:%s:%s: %s' % (REVERSE_SEVERITIES[severity][0], path,
                                line or '', msg)

        
    def raw(self, name, value):
        """give some raw data"""
        print 'RAW %s=%s' % (name, value)

NOTIFY_RGX = re.compile(r'NOTIFY\|([^|]*)\|?(.*)')
LOG_RGX = re.compile(r'LOG\|(INFO|WARNING|ERROR|FATAL)\|([^|]*)\|(\d*)\|(.*)')
RAW_RGX = re.compile(r'RAW\|([^=]*)=(.*)')

class PrintReader:
    """read output from a PrintWriter and give it to the real writer (XMLWriter)
    """
    def __init__(self, writer):
        self._writer = writer
        self._last_log = None
        
    def from_string(self, string):
        """regenerate event to a writer from command output"""
        for line in string.split(os.linesep):
            if line.strip():
                self.parse_line(line)

    def from_file(self, file):
        """regenerate event to a writer from command output"""
        for line in file:
            line = line.strip()
            if line:
                self.parse_line(line)

    def parse_line(self, line):
        """parse a line and dispatch detected event to writer"""
        match = LOG_RGX.match(line)
        if match:
            self._flush()
            severity, path, line, msg = match.groups()
            self._last_log = [severity, path, line, msg]
            return
        
        match = NOTIFY_RGX.match(line)
        if match:
            self._flush()
            event, attributes = match.groups()
            kwargs = {}
            if attributes:
                for attr_decl in attributes.split('|'):
                    key, val = attr_decl.split('=', 1)
                    kwargs[key.strip()] = val.strip()
            self._writer.notify(event, **kwargs)
            return
        
        match = RAW_RGX.match(line)
        if match:
            self._flush()
            key, val = match.groups()
            self._writer.raw(key, val)
            return
        
        # no regex match 
        if self._last_log is None:
            context = self._writer.context()
            test = context.get("test")
            msg = "Unexpected line"
            if test is not None :
                msg +=  " in test<%s>" % test
                check = context.get("check")
                if check is not None:
                    msg +=  " in check<%s>" % check
            self._writer.msg(0, '%s : %s' % (msg, line))
        else:
            self._last_log[3] += os.linesep
            self._last_log[3] += line
            
    def _flush(self):
        """flush the latest log if one found"""
        if self._last_log is not None:
            severity, path, line, msg = self._last_log
            self._writer.log(eval(severity), path, line or None, msg)
            self._last_log = None
