# pylint: disable-msg=W0622
#
# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""some common utilities shared by different APyCoT modules"""

import re
import os
import sys
from glob import glob
from os.path import isdir, exists, join, walk, sep, isabs, abspath, dirname, \
                     expanduser, expandvars, curdir, realpath
from warnings import warn
from ConfigParser import ConfigParser, NoSectionError, NoOptionError, Error

from logilab.common.textutils import get_csv, apply_units, TIME_UNITS

from apycot import ConfigError



DATE_RGX = re.compile(r'^\d{4}/\d\d/\d\d(/\d\d)?$')

# length of a day in seconds
DAY = 60 * 60 * 24
# length of an hour in seconds
HOUR = 60 * 60

INFO = 0
WARNING = 10
ERROR = 20
FATAL = 30

SEVERITIES = {
    'INFO' : INFO,
    'WARNING' : WARNING,
    'ERROR' : ERROR,
    'FATAL' : FATAL
    }

REVERSE_SEVERITIES = {
    INFO : 'INFO',
    WARNING : 'WARNING',
    ERROR : 'ERROR',
    FATAL : 'FATAL'
    }

__builtins__.update(SEVERITIES)



# data file utilities #########################################################

MAIN_SECT = 'MAIN'
DATA_SECT = 'DATA'

def init_directory(directory, date):
    """init the directory by creating <year>/<month>/day> subdirectories if
    they doesn't yet exist
    
    date is a 3-uple (Y, m, d)
    return the path to the created directory
    """
    assert isdir(directory), '%s should be an existing directory' % directory
    for part in date:
        if isinstance(part, int):
            part = '%02d' % part
        directory = join(directory, str(part))
        if not exists(directory):
            os.mkdir(directory)
    return directory

def get_latest(directory, level=3, alldates=None):
    """return the "latest directory" in a hierarchy of directories organized
    by date
    """
    if alldates is None:
        if not level:
            return directory
        alldates = existing_dates_index(directory, level)
    if not alldates:
        raise ConfigError('no data in directory %s' % directory) 
    return join(directory, *['%02d' % part for part in alldates[-1]])
    
def existing_dates_index(directory, level=3, datafile='tester_data.xml'):
    index = []
    def add_to_index(skip, dirname, fnames):
        if dirname != directory:
            try:
                key = [int(part) for part in dirname.split(sep)[skip:]]
            except ValueError:
                fnames[:] = []
            else:
                if len(key) == level and datafile in fnames:
                    index.append(tuple(key))
                    fnames[:] = []
    walk(directory, add_to_index, len(directory.split(sep)))
    return sorted(index)
    

# config utilities ############################################################

def load_addons(addons, verb=1):
    """load a list of modules/packages (<addons> is a csv string)"""
    for module in get_csv(addons):
        if verb:
            print >> sys.stderr, 'Loading ' + module
        try:
            __import__(module)
            #load_module_from_name(module)
        except ImportError:
            if verb:
                print >> sys.stderr, 'Failed to load ' + module
                if verb > 1:
                    import traceback
                    traceback.print_exc()
            continue
        if verb > 1:
            print >> sys.stderr, 'Loaded ' + module

_GLOBALS = None
def global_config(config, ignore_vars, verb=0):
    """return the global configuration (i.e. not specific to a test) as a
    dictionary
    """
    global _GLOBALS
    if _GLOBALS is not None:
        return _GLOBALS
    # load global add-on modules
    if config.has_option(MAIN_SECT, 'load'):
        load_addons(config.get(MAIN_SECT, 'load', verb))
    _GLOBALS = section_as_dict(config, MAIN_SECT, ignore_vars)
    return _GLOBALS

def parse_config(config_file, location=None, verb=None, config=False,
                                                                trace=False):
    """return the read config as a ConfigParser object"""
    config = ConfigParser()
    _, test_names = _rec_parse_config(config, config_file, trace=trace)
    for glob in ('MAIN','DATA'):
        test_names.discard(glob)
    for test_name in test_names:
        if not config.has_option(test_name,'is_test'):
            config.set(test_name,'is_test','1')

    # override location from configuration file ?
    if location is not None:
        config.set(DATA_SECT, 'location', location)
    if verb is None and config.has_option(MAIN_SECT, 'verbosity'):
        verb = config.getint(MAIN_SECT, 'verbosity')
    else:
        verb = verb and int(verb) or 0

    return config, verb


def _rec_parse_config(config, pattern, already_parsed=None, trace=False):
    """parse a config file and all file in [MAIN], include recursivly"""
    if already_parsed is None:
        already_parsed = []
        dir = abspath(curdir)
    else:
        
        dir = dirname(already_parsed[-1])
    
    target_pattern = expandvars(pattern)
    target_pattern = expanduser(target_pattern)
    if not isabs(target_pattern):
        target_pattern = join(dir, target_pattern)
    target_pattern = realpath(target_pattern)
    
    
    target_files = glob(target_pattern)
    test_names, includes_test_names = set(),set()
    if target_files:
        for target_file in target_files:
            if target_file in already_parsed:
                continue # ignore circular include
            if trace:
                print "parsing config file:", target_file
            new_config = ConfigParser(config.defaults())
            new_config.read([target_file])
            already_parsed.append(target_file)
            
            test_names.update( set(new_config.sections()) )

            includes = []
            if new_config.has_option(MAIN_SECT, 'includes'):
                try:
                    for pat in get_csv(new_config.get(MAIN_SECT, 'includes')):
                        includes.append((pat,False))
                except Error, ex:
                    warn("an error occurred while fetching includes in %s:  %s"%(target_file,ex))
                new_config.remove_option(MAIN_SECT, 'includes')
            
            if new_config.has_option(MAIN_SECT, 'includes-tests'):
                try:
                    for pat in get_csv(new_config.get(MAIN_SECT, 'includes-tests')):
                        includes.append((pat,True))
                except Error, ex:
                    warn("an error occurred while fetching includes in %s:  %s"%(target_file,ex))
                new_config.remove_option(MAIN_SECT, 'includes-tests')
            for include, test in includes:
                new_test_names = _rec_parse_config(new_config, include,
                                    list(already_parsed), trace=trace)
                if test:
                    for tns in new_test_names:
                        includes_test_names.update(tns)

            # merge
            for section in ['DEFAULT'] + list(new_config.sections()):
                if not config.has_section(section):
                    config.add_section(section)
                for option, value in new_config.items(section,raw=True):
                    if not config.has_option(section,option):
                        config.set(section,option,value)
                    
    elif already_parsed:
        warn('pattern "%s" (%s) include from %s doesn\'t match any file'
                %(target_pattern, pattern, already_parsed[-1]))
    else:
        warn('pattern "%s" (%s) doesn\'t match any file'%(target_pattern,
                                                                 pattern))
    return test_names, includes_test_names
   
def section_as_dict(config, section, ignore=()):
    """return a config parser section as a dictionnary"""
    result = {}
    try:
        for option in config.options(section):
            if option in ignore:
                continue
            result[option] = config.get(section, option)
    except NoSectionError, ex:
        raise ConfigError(str(ex))
    return result

def get_options_from_dict(options, prefix=None, ignore=()):
    """get a dictionary of options begining with <prefix>_ from the given
    dictionary
    
    <prefix>_ is removed from option's name in the returned dictionary and
    matching keys are removed from the original dictionany
    """
    new_options = {}
    for option, value in options.items():
        if prefix and not option.startswith(prefix + '_'):
            continue
        if option in ignore:
            continue
        if prefix:
            new_options[option[len(prefix)+1:]] = value
        else:
            new_options[option] = value
        del options[option]
    return new_options

MAIN_VARS = ('tests', 'verbosity', 'load', 'threads', 'includes')
def base_test_config(config, test_name, verb=0, do_load_addons=True):
    """get a test definition as a dictionnary"""
    test_def = global_config(config, MAIN_VARS, verb).copy()
    base_def = section_as_dict(config, test_name)
    
    # is there some group configuration ?
    for group in get_csv(base_def.get('groups', test_def.get('groups',''))):
        test_def.update(section_as_dict(config, group))
    
    test_def.update(base_def)
    
    if test_def.has_key('groups'):
        del test_def['groups']
    # load test add-on modules
    if do_load_addons and test_def.has_key('load'):
        load_addons(test_def['load'], verb)
        del test_def['load']
    return test_def


# others ######################################################################

_MARKER = ()

class SimpleOptionsManagerMixIn(object):
    """A very simple mixin to handle options"""
    required_options = ()
    
    def __init__(self):
        self.options = {}
        
    def set_options(self, options):
        """set a dictionary of options"""
        self.options = options

    def get_option(self, option, default=_MARKER, msg=None):
        """return option's value or None, raise ConfigError if no default is
        provided and the option is not defined
        """
        value = self.options.get(option, default)
        if value is _MARKER:
            raise ConfigError(msg or 'Missing %r option' % option)
        return value
    def set_option(self, option, value):
        """set and option's value
        """
        self.options[option] = value
    
    def check_options(self):
        """check options according to the required_options attributes
        """
        for opt, msg in self.required_options:
            if not self.get_option(opt, msg=msg):
                raise ConfigError(msg or 'Empty value for option %r' % opt)

    
class EnvironmentTrackerMixin:
    """track environment change to be able to restore it latter

    sys.path is synchronized with the PYTHONPATH environment variable
    """
    
    writer = None
    
    def __init__(self, writer=None):
        self._tracks = {}
        self.writer = writer
        
    def update_env(self, key, envvar, value, separator=None):
        """update an environment variable"""
        envvar = envvar.upper()
        orig_value = os.environ.get(envvar)
        if orig_value is None:
            orig_value = ''
        uid = self._make_key(key, envvar)
        assert not self._tracks.has_key(uid)
        if separator is not None:
            orig_values = orig_value.split(separator)
            if not value in orig_values:
                orig_values.insert(0, value)
                self._set_env(uid, envvar, separator.join(orig_values))
        elif orig_value != value:
            self._set_env(uid, envvar, value)
                
    def clean_env(self, key, envvar):
        """reinitialize an environment variable"""
        envvar = envvar.upper()
        uid = self._make_key(key, envvar)
        if self._tracks.has_key(uid):
            orig_value = self._tracks[uid]
            if envvar == 'PYTHONPATH':
                update_path(os.environ[envvar], orig_value)
            if self.writer:
                self.writer.msg(2, 'Reset %s=%r' % (envvar, orig_value))
            if orig_value is None:
                del os.environ[envvar]
            else:
                os.environ[envvar] = self._tracks[uid]
            del self._tracks[uid]
            
    def _make_key(self, key, envvar):
        """build a key for an environment variable"""
        return '%s-%s' % (key, envvar)

    def _set_env(self, uid, envvar, value):
        """set a new value for an environment variable
        """
        if self.writer:
            self.writer.msg(2, 'Set %s=%s' % (envvar, value))
        orig_value = os.environ.get(envvar)
        self._tracks[uid] = orig_value
        os.environ[envvar]  = value
        if envvar == 'PYTHONPATH':
            update_path(orig_value, value)


def clean_path(path):
    """remove trailing path separator from path"""
    if path and path[-1] == os.sep:
        return path[:-1]
    return path

def update_path(old_path, new_path):
    """update sys.path"""
    if old_path is not None:
        for path in old_path.split(os.pathsep):
            try:
                sys.path.remove(clean_path(path))
            except ValueError:
                continue
    if new_path is not None:
        new_path = new_path.split(os.pathsep)
        new_path.reverse()
        for path in new_path:
            sys.path.insert(0, clean_path(path))
