# -*- coding: iso-8859-1 -*-
"""unittest for apycot's tranports
"""

import tempfile
import os
import shutil
from os.path import join, exists

from logilab.common import testlib
from logilab.common.testlib import TestCase, unittest_main

from apycot import get_registered, list_registered
from apycot.transports import *


class TransportsTC(TestCase):
    def test(self):
        keys = list_registered('transport')
        keys.sort()
        self.assertEquals(keys, ['dailyfile',
                        'dailyzope', 'datedfile', 'datedzope', 'email', 'file'])


class BaseTransportTC(TestCase):
    TransportClass = BaseTransport
    def setUp(self):
        self.transport = self.TransportClass(0)
        self.open()
    
    def open(self):
        self.transport.open([2004, 2, 22])
        
    def test_base(self):
        self.assertEqual(self.transport.verbosity, 0)
        
    def test_open(self):
        transport = self.transport
        transport.open([2004, 2, 22])
        self.assertEquals(transport.date, ['2004', '02', '22'])
        self.assertEquals(transport.encoding, 'UTF-8')

    def test_base_url(self):
        self.assertRaises(NotSupported, self.transport.base_url)

    def test_encode(self):
        self.assertEquals(self.transport.encode(u'hhhh'),
                                    'h\xc3\xa9h\xc3\xa9h\xc3\xa9h\xc3\xb6')
        
##     def test_close(self):
##         # do nothing by default
##         self.transport.close()
        
class FileTransportTC(BaseTransportTC):
    TransportClass = FileTransport

    def setUp(self):
        self.transport = self.TransportClass(0)
        self.tempdir = tempfile.mktemp()
        os.mkdir(self.tempdir)
        self.open()
        
    def open(self):
        self.transport.options['destination'] = self.tempdir
        self.transport.open([2004, 2, 22])

    def test_base_path(self):
        self.assertEqual(self.transport.base_path(), self.tempdir)
        self.assertRaises(NotSupported, self.transport.base_path,
                                                        ['2004', '02', '22'])
        
    def test_base_url(self):
        transport = self.transport
        self.assertEquals(transport.base_url(), self.tempdir)
        transport.options['url_base'] = 'http://lahaut.dansla.montagne.com/'
        self.assertEquals(transport.base_url(),
                                        'http://lahaut.dansla.montagne.com/')
        self.assertRaises(NotSupported,
                                self.transport.base_url, ['2004', '02', '22'])

    def test_parts(self):
        partname = 'toto.txt'
        stream = self.transport.part_stream(partname)
        stream.write('roudoudou\n')
        self.transport.add_part(partname, stream)
        path = join(self.transport.base_path(), partname)
        self.assert_(exists(path), path)
        self.assertEquals(open(path).read(), 'roudoudou\n')
        partname = 'toto.xml'
        stream = StringIO('<xml>gouzigouzi\n</xml>')
        self.transport.add_part(partname, stream)
        path = join(self.transport.base_path(), partname)
        self.assert_(exists(path), path)
        self.assertEquals(open(path).read(), '<xml>gouzigouzi\n</xml>')
        
    def tearDown(self):
        shutil.rmtree(self.tempdir)

        
class DateFileTransportTC(FileTransportTC):
    TransportClass = DateFileTransport

    def test_base_path(self):
        transport = self.transport
        self.assertEqual(self.transport.base_path(),
                                            join(self.tempdir, *transport.date))
        self.assertEqual(self.transport.base_path(['2004', '02', '22']),
                                    join(self.tempdir, *['2004', '02', '22']))
        
    def test_base_url(self):
        transport = self.transport
        self.assertEquals(transport.base_url(),
                                            join(self.tempdir, *transport.date))
        self.assertEquals(transport.base_url(['2004', '02', '22']),
                                    join(self.tempdir, *['2004', '02', '22']))
        url = 'http://lahaut.dansla.montagne.com/'
        transport.options['url_base'] = url
        effective_url = url + '/'.join(transport.date)
        self.assertEquals(transport.base_url(), effective_url)
        effective_url = url + '/'.join(['2004', '02', '22'])
        self.assertEquals(transport.base_url(['2004', '02', '22']),
                                                                effective_url)


class EmailTransportTC(BaseTransportTC):
    TransportClass = EmailTransport
    
    def open(self):
        self.transport.smtp_server = testlib.MockSMTP
        self.transport.options['from'] = 'tester@logilab.fr'
        self.transport.options['dest'] = 'syt@logilab.fr'
        self.transport.open([2004, 2, 22])
        
    def test_parts(self):
        partname = 'toto.txt'
        stream = self.transport.part_stream(partname)
        stream.write('roudoudou\n')
        self.transport.add_part(partname, stream)
        self.transport.close()
        self.assertEquals(len(testlib.MAILBOX), 1)
        self.assertEquals(testlib.MAILBOX[0][0], 'tester@logilab.fr')
        self.assertEquals(testlib.MAILBOX[0][1], ['syt@logilab.fr'])

    def test_close(self):
        partname = 'toto.xml'
        stream = StringIO('<xml>gouzigouzi\n</xml>')
        self.transport.add_part(partname, stream)
        partname = 'toto.mng'
        stream = StringIO('a4dafz4fd5f.?:er/@@""^')
        self.transport.add_part(partname, stream)
        self.transport.close()
        self.assertEquals(len(testlib.MAILBOX), 1)
        self.assertEquals(testlib.MAILBOX[0][0], 'tester@logilab.fr')
        self.assertEquals(testlib.MAILBOX[0][1], ['syt@logilab.fr'])
                          
    

    
if __name__ == '__main__':
    unittest_main()
