"""unit tests for the apycot.datamanagers module"""

import shutil
from time import localtime
from cStringIO import StringIO
from os.path import join
import warnings
from warnings import filterwarnings

from logilab.common.testlib import TestCase, unittest_main
from logilab.common.interface import implements
from logilab.common.compat import set

from utils import MockConfigParser, etree_clean, clean

from apycot import IWriter
from apycot.datamanagers import *

class TestDecorator:
    def decorate(self, document, manager):
        document.set('decorated',"1")

DATA_DATE =  (2004, 02, 24)


class ExistingDatesIndexTC(TestCase):
    def test1(self):
        self.assertEquals(existing_dates_index('inputs'),
                          [(2004, 2, 23), (2004, 2, 24)])
    def test2(self):
        self.assertEquals(existing_dates_index('hourly', 4),
                          [(2004, 02, 24, 13)])
        
        
        
class XMLDataManagerTC(TestCase):
    """tests for the XML Data manager"""
    
    def setUp(self):
        self.config = MockConfigParser({'DATA': {'location':'inputs'}})
        self.hconfig = MockConfigParser({'DATA': {'location':'hourly',
                                                  'mode':'hourly'}})

    def tearDown(self):
        date = localtime()[:3]
        path = 'inputs'
        for i in range(len(date)):
            path = join(path, '%02d' % date[i])
            if date[i] != DATA_DATE[i]:
                try:
                    shutil.rmtree(path)
                except:
                    pass
                break
            
    def test_read_base(self):
        manager = XMLDataManager(self.config)
        self.assertEqual(manager.location, 'inputs')
        self.assertEqual(manager.encoding, 'UTF-8')
        self.assertEqual(manager.decorator, None)

    def test_config(self):
        self.config.set('DATA', 'encoding','ISO-8859-1')
        manager = XMLDataManager(self.config)
        self.assertEqual(manager.encoding, 'ISO-8859-1')
        self.config.set('DATA', 'location', 'labas')
        self.assertRaises(ConfigError, XMLDataManager, self.config)
        self.config.remove_option('DATA', 'location')
        self.assertRaises(ConfigError, XMLDataManager, self.config)
        
    def test_main_tree(self):
        manager = XMLDataManager(self.config)
        tree = manager.main_tree()
        self.assertEqual(tree.getroot().get('name'), 'testconfig')
    
    def test_tree_for_date(self):
        manager = XMLDataManager(self.config)
        tree = manager.tree_for_date(manager.date)
        self.assertEqual(tree.getroot().get('name'), 'testconfig')
        self.assertRaises(ConfigError, manager.tree_for_date, [2002, 1, 1])
        tree2 = manager.tree_for_date(manager.date)
        self.assert_(tree is tree2)
        
    def test_decorator(self):
        manager = XMLDataManager(self.config, decorator=TestDecorator())
        tree = manager.tree_for_date(manager.date)
        self.assert_(tree.getroot().get('decorated'))

    def test_writer(self):
        manager = XMLDataManager(self.config, rwmode='w')
        
        writer = manager.writer()
        self.assertIsInstance(writer,XMLWriter)
        self.assertEquals(writer._current.tag, 'testsdata') # XXX
        self.assertEquals(writer._current.getchildren(), [])
        writer = manager.writer(date=[2004,2,24], rewrite=1)
        
        self.assertIsInstance(writer,XMLWriter)
        self.assertEquals(writer._current.tag, 'testsdata')
        self.assertEquals(len(writer._current), 2)

        
    def test_read_daily(self):
        manager = XMLDataManager(self.config)
        self.assertEqual(manager.date, [2004, 02, 24])
        manager = XMLDataManager(self.config, date=[2001, 02, 24])
        self.assertEqual(manager.date, [2001, 02, 24])
            
    def test_read_hourly(self):
        manager = XMLDataManager(self.hconfig)
        self.assertEqual(manager.date, [2004, 02, 24, 13])
        manager = XMLDataManager(self.hconfig, date=[2001, 02, 24, 10])
        self.assertEqual(manager.date, [2001, 02, 24, 10])
        self.assertRaises(AssertionError,
                          XMLDataManager, self.hconfig, date=[2001, 02, 24])
        
    def test_write_daily(self):
        manager = XMLDataManager(self.config, rwmode='w')
        self.assertEqual(manager.date, localtime()[:3])
        manager = XMLDataManager(self.config, date=[2001, 02, 24], rwmode='w')
        self.assertEqual(manager.date, [2001, 02, 24])
        
    def test_write_hourly(self):
        manager = XMLDataManager(self.hconfig, rwmode='w')
        self.assertEqual(manager.date, localtime()[:4])
        manager = XMLDataManager(self.hconfig, date=[2001, 02, 24, 10], rwmode='w')
        self.assertEqual(manager.date, [2001, 02, 24, 10])
        
    def test_next_date_daily(self):
        manager = XMLDataManager(self.config)
        date = manager.next_date(manager.date)
        expected = localtime(time() + DAY)[:3]
        self.assertEquals(date, expected)
        date = manager.next_date([2004,2,23])
        self.assertEquals(date, (2004,2,24))
        date = manager.next_date(['2004','02','23'])
        self.assertEquals(date, (2004,02,24))
        date = manager.next_date([2004,2,20])
        self.assertEquals(date, (2004,2,23))
        date = manager.next_date(localtime())
        self.assertEquals(date, expected)
        
    def test_next_date_hourly(self):
        manager = XMLDataManager(self.hconfig)
        date = manager.next_date(manager.date)
        expected = localtime(time() + HOUR)[:4]
        self.assertEquals(date, expected)
        date = manager.next_date([2004, 2, 24, 12])
        self.assertEquals(date, (2004, 2, 24, 13))
        date = manager.next_date(['2004','02','24', '12'])
        self.assertEquals(date, (2004, 02, 24, 13))
        date = manager.next_date([2004, 2, 20, 1])
        self.assertEquals(date, (2004, 2, 24, 13))
        date = manager.next_date(localtime())
        self.assertEquals(date, expected)
        
    def test_previous_date_daily(self):
        manager = XMLDataManager(self.config)
        date = manager.previous_date(manager.date)
        self.assertEquals(date, (2004, 2, 23))
        date = manager.previous_date((2004, 2, 23))
        self.assertEquals(date, None)
        
        date = manager.previous_date([2004, 2, 25])
        self.assertEquals(date, (2004, 2, 24))
        date = manager.previous_date(['2004', '02', '25'])
        self.assertEquals(date, (2004, 02, 24))
        date = manager.previous_date([2004, 04, 25])
        self.assertEquals(date, (2004, 02, 24))
        
    def test_previous_date_hourly(self):
        manager = XMLDataManager(self.hconfig)
        date = manager.previous_date(manager.date)
        self.assertEquals(date, None)
        date = manager.previous_date([2004, 2, 25, 12])
        self.assertEquals(date, (2004, 2, 24, 13))
        date = manager.previous_date(['2004','02','25', '12'])
        self.assertEquals(date, (2004, 02, 24, 13))
        date = manager.previous_date([2004, 04, 25, 12])
        self.assertEquals(date, (2004, 02, 24, 13))
EXPECTED = """<?xml version='1.0' encoding='UTF-8'?>
<testsdata><test name="mytest"><check name="python_unittest" status="success"><log path="labas" line="1" severity="ERROR">erreur ligne 1</log><log path="autrepart" severity="ERROR">erreur chaipasou
multilines</log></check><check name="check2" status="failure"><raw class="name">value</raw></check></test><test name="myothertest"><check name="python_unittest" status="success"/></test></testsdata>"""

EXPECTED2 = """<?xml version='1.0' encoding='UTF-8'?>
<testsdata><test name="mytest"><check name="python_unittest" status="failure"><log path="autrepart" severity="ERROR">erreur chaitoujourspasou</log></check></test><test name="mytest2"/></testsdata>"""

EXPECTED3 = """<?xml version='1.0' encoding='UTF-8'?>
<testsdata><test name="mytest"><check name="python_unittest" status="error"><log path="labas" line="1" severity="ERROR">erreur ligne 1</log><log path="autrepart" severity="ERROR">erreur chaipasou</log></check><check name="check2" status="failure"><raw class="name">value</raw></check></test><test name="myothertest"><check name="python_unittest" status="success"/></test></testsdata>"""

def del_key(node, key):
    try:
        del node[key]
    except KeyError:
        pass
    
class XMLWriterTC(TestCase):
    data_file = 'tester_data.xml'

    def setUp(self,test=None):
        if test is None: # default set up
            root = Element('testsdata')
            self.tree = ElementTree(root)
            self.writer = XMLWriter(root=self.tree,data_file=self.data_file)
        if test == "test_rewrite":
            self.tree.getroot().append(Element('test', {'name':'mytest'}))
            self.tree.getroot().append(Element('test', {'name':'mytest2'}))
            self.writer = XMLWriter(root=self.tree,data_file=self.data_file,rewrite=1)
            return self.writer

    def tearDown(self):
        try:
            os.remove(self.data_file)
        except:
            pass


 
    def test_base(self):
        writer = self.writer
        writer.notify('open_test', name='mytest')
        writer.notify('open_check', name='python_unittest')
        writer.log(ERROR, 'labas', 1, 'erreur ligne 1')
        writer.log(ERROR, 'autrepart', None, 'erreur chaipasou\nmultilines')
        writer.notify('close_check', status="success")
        writer.notify('open_check', name='check2')
        writer.raw('name', 'value')
        writer.notify('close_check', status="failure")
        writer.notify('close_test')
        writer.notify('open_test', name='myothertest')
        writer.notify('open_check', name='python_unittest')
        writer.notify('close_check', status="success")
        writer.notify('close_test')
        etree_clean(writer)
        writer.flush(0)        
        self.assertTextEqual(open(self.data_file).read(), EXPECTED)
        
 
 
    def test_rewrite(self):
        writer = self.setUp("test_rewrite")
        shutil.copy('inputs/tester_data2.xml', self.data_file)
        writer.notify('open_test', name='mytest')
        writer.notify('open_check', name='python_unittest')
        writer.log(ERROR, 'autrepart', None, 'erreur chaitoujourspasou')
        writer.notify('close_check', status="failure")
        writer.notify('close_test')
        etree_clean(writer)
        writer.flush(0)
        self.assertTextEqual(open(self.data_file).read(), EXPECTED2)

    def test_robust(self):
        writer = self.writer
        writer.notify('open_test', name='mytest')
        
        writer.notify('open_check', name='python_unittest')
        writer.log(ERROR, 'labas', 1, 'erreur ligne 1')
        writer.log(ERROR, 'autrepart', None, 'erreur chaipasou')
        writer.notify('open_check', name='check2')
        writer.raw('name', 'value')
        writer.notify('close_check', status="failure")
        writer.notify('open_test', name='myothertest')

        writer.notify('open_check', name='python_unittest')
        writer.notify('close_check', status="success")
        writer.notify('close_test')

        etree_clean(writer)
        writer.flush(0)
        self.assertTextEqual(open(self.data_file).read(), EXPECTED3)

        
    def test_msg(self):
        buffer = sys.stderr = StringIO()
        try:
            self.writer.verbosity = 0
            self.writer.msg(1, 'yooo')
            self.assertEqual(buffer.getvalue(), '')
            self.writer.msg(0, 'yooo')
            self.assertEqual(buffer.getvalue(), 'yooo\n')
        finally:
            sys.stderr = sys.__stderr__



if __name__ == '__main__':
    unittest_main()
