# Copyright (c) 2004-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
unit tests for tests configuration file parsing
"""

import unittest
import os
from ConfigParser import ConfigParser
from logilab.common.compat import set
from logilab.common.testlib import TestCase

from apycot import ConfigError, IChecker
from apycot.runtest import *

from utils import input_path, MockWriter

config = ConfigParser()
config.read([input_path('tester.ini')])

class GetDepInfoTC(TestCase):

    def test_get_dependency_info(self):
        self.assertRaises(ConfigError, get_dependency_info, 'toto')
        self.assertRaises(ConfigError, get_dependency_info, 'toto@tag')
        self.assertEquals(str(get_dependency_info('cvs:repo;toto')), 'toto@HEAD in cvs repo')
        self.assertEquals(str(get_dependency_info('cvs:repo;toto@TAG')), 'toto@TAG in cvs repo')


class BaseTestConfigTC(TestCase):
    def setUp(self):
        self.value = base_test_config(config, 'rql', MockWriter())
        
    def test_base(self):
        self.assert_(len(self.value), 3)

    def test_test_def_values(self):
        test_def = self.value
        self.assertDictEquals(test_def, {'python_test_coverage_treshold': '70',
                                     'repository': 'cvs:/home/cvs/prive',
                                     'setup_install_match_checks': 'python_pkg, python_lint, python_unittest, python_test_coverage',
                                     'pythonpath': '',
                                     'checks': 'python_pkg,pkg_doc,python_syntax,python_lint,python_unittest,python_test_coverage',
                                     'dependencies': 'common,constraint',
                                     'python_lint_ignore': 'parser.py,parser_main.py,thirdparty',
                                     'build_debian_match_checks': 'debian_lint',
                                     'path': 'soft/rql',
                                     'python_lint_treshold': '7',
                                     'is_test':'1',
                                     'preprocessors':'setup_install'}
                          )
                
    
class BuildTestTC(TestCase):
    def setUp(self):
        self.test = build_test(config, 'rql', MockWriter(), checks=None,
                               keep_tmp=0, fspath=None, verb=0)
    def test_base(self):
        test = self.test
        self.assert_(isinstance(test, Test))
        for check in test.checkers:
            self.assertEqual(check.__implements__, IChecker)

    def common_values(self, test):
        #self.assertEqual(test.tag, '')
        self.assertEqual(len(test.checkers), 6)
        self.assertEqual([c.__name__ for c in test.checkers],
                         ['python_pkg', 'pkg_doc', 'python_syntax', 'python_lint',
                          'python_unittest', 'python_test_coverage'])
        self.assertEqual(len(test.preprocessors), 5)
        self.assertEqual([p.__name__ for p in test.preprocessors],
                         ['zope_install', 'setup_install', 'set_python_env',
                          'build_debian', 'make'])
        self.assertEqual(len(test.dependencies), 2)
        solutions = (('soft/logilab/common@HEAD in cvs /home/cvs/public', 'common'),
                     ('soft/logilab/constraint@HEAD in cvs /home/cvs/public', 'constraint'))
        for i in range(len(test.dependencies)):
            self.assertEquals(repr(test.dependencies[i][0]), solutions[i][0])
            self.assertEquals(test.dependencies[i][1], solutions[i][1])
        self.assertDictEqual(test.environ, {'pythonpath': '',
            'preprocessors':'setup_install'})
        self.assertEqual(test.keep_temp_dir, 0)
        
    def test_values(self):
        test = self.test
        self.common_values(test)
        self.assertEqual(repr(test.repo), 'soft/rql@HEAD in cvs /home/cvs/prive')
        self.assertEqual(test.repo.path, 'soft/rql')
        command = test.repo.checkout_command()
        self.assertEqual(command, 'cvs -d /home/cvs/prive -Q checkout -r HEAD soft/rql')
        
    def test_fspath_values(self):
        test = build_test(config, 'rql', MockWriter(), checks=None,
                          keep_tmp=0, fspath='XXX/TOTO', verb=0)
        self.common_values(test)
        path = '%s/XXX/TOTO' % os.getcwd()
        self.assertEqual(repr(test.repo), 'fs directory at %s' % path)
        self.assertEqual(test.repo.path, path)
        self.assertEqual(test.repo.env_path(), 'TOTO')
        command = test.repo.checkout_command()
        self.assert_(command.startswith('cp -R '))
        self.assert_(command.endswith('XXX/TOTO .'))
        
        
class DependenciesTC(TestCase):
    def test(self):
        deps = dependencies(config, 'rql,common', MockWriter(), _done=set())
        solutions = [('soft/logilab/common@HEAD in cvs /home/cvs/public', 'common'),
                     ('soft/logilab/constraint@HEAD in cvs /home/cvs/public', 'constraint'),
                     ('soft/rql@HEAD in cvs /home/cvs/prive', 'rql')]
        for i in range(len(deps)):
            self.assertEquals(repr(deps[i][0]), solutions[i][0])
            self.assertEquals(deps[i][1], solutions[i][1])
        
if __name__ == '__main__':
    unittest.main()
