"""
unit tests for checkers
"""

from logilab.common.testlib import unittest_main, TestCase
import unittest
import sys
import os
from os.path import join
zope_path = os.environ.get('SOFTWARE_HOME', '/usr/lib/zope/lib/python')
sys.path.insert(1, zope_path)

from unittest import TestSuite
#from apycot.repositories import FSRepository
from apycot import SUCCESS, FAILURE, PARTIAL, NODATA
from apycot.checkers.chks_python import *
from apycot.checkers.chks_xml import *
from apycot.checkers.chks_rest import *
from apycot.checkers.chks_pkg import *
from apycot.checkers import chks_pt 

from utils import *




class FileCheckerTest(TestCase):
    def __init__(self, checker, files, method_name, options=None):
        TestCase.__init__(self, method_name)
        self.checker = checker
        self.files = [input_path(file) for file in files]
        if options is not None:
            checker.set_options(options)

    def setUp(self):
        self.reporter = MockReporter()
    
    def check_file(self,file):
        return self.checker.check_file(file, self.reporter)
    def check_dir(self,file):
        return self.checker.run(MockTest(MockRepository(path=file)), self.reporter)

    def chks_test_(self, expected, func):
        for file in self.files:
            status = func(file)
            msg = '%s on %s status: %s expecting %s' % (self.checker.__name__, file, status, expected)
            if self.checker.options:
                msg = '\n    '.join([msg] + ['='.join(str(i) for i in pair) for pair in self.checker.options.iteritems()])
            self.failUnlessEqual(status, expected, msg+'\n-----\n'+self.reporter.stderr.getvalue())
        
    def chks_test_file_good(self):
        self.chks_test_(SUCCESS, self.check_file)
    
    def chks_test_file_bad(self):
        self.chks_test_(FAILURE, self.check_file)

    def chks_test_dir_good(self):
        self.chks_test_(SUCCESS, self.check_dir)
    
    def chks_test_dir_bad(self):
        self.chks_test_(FAILURE, self.check_dir)

    def chks_test_dir_partial(self):
        self.chks_test_(PARTIAL, self.check_dir)
    
    def chks_test_dir_nodata(self):
        self.chks_test_(NODATA, self.check_dir)
            
class ModuleCheckerTest(FileCheckerTest):
            
    def check_pkg(self, file):
        return self.checker.run(MockTest(MockRepository(path=file)), self.reporter)
    
    def chks_test_good(self):
        self.chks_test_(SUCCESS, self.check_pkg)
    
    def chks_test_bad(self):
        self.chks_test_(FAILURE, self.check_pkg)
    
    def chks_test_partial(self):
        self.chks_test_(PARTIAL, self.check_pkg)
    
    def chks_test_nodata(self):
        self.chks_test_(NODATA, self.check_dir)


def suite():
    """return the unitest suite"""
    testsuite = TestSuite()
    addTest = testsuite.addTest
    ##### FileChecker #####
    
    file_checker = PythonSyntaxChecker()
    addTest(FileCheckerTest(file_checker, ['empty_dir'], 'chks_test_dir_nodata'))
    
    ##### PythonSyntaxChecker #####
    python_syntax = PythonSyntaxChecker()
    addTest(FileCheckerTest(python_syntax, ['goodsyntax.py'], 'chks_test_file_good'))
    addTest(FileCheckerTest(python_syntax, ['badsyntax.py'], 'chks_test_file_bad'))
    addTest(FileCheckerTest(python_syntax, ['goodsyntax/'], 'chks_test_dir_good'))
    addTest(FileCheckerTest(python_syntax, ['badsyntax/'], 'chks_test_dir_bad'))
    addTest(FileCheckerTest(python_syntax, ['mixedsyntax/'], 'chks_test_dir_bad'))
    addTest(FileCheckerTest(python_syntax, ['extentionfilter/'], 'chks_test_dir_good'))
    addTest(FileCheckerTest(python_syntax, ['syntax_dir/badsyntax/'], 'chks_test_dir_bad'))
    
    
    addTest(FileCheckerTest(python_syntax, ['goodsyntax.py'], 'chks_test_file_good'))
    
    python_syntax = PythonSyntaxChecker()
    python_syntax.set_options({'ignore':'wrongsyntax.py'})
    addTest(FileCheckerTest(python_syntax, ['syntax_dir/badsyntax/'], 'chks_test_dir_good'))
    
    python_syntax = PythonSyntaxChecker()
    python_syntax.set_options({'ignore':'rootbadsyntax.py,badsyntax'})
    addTest(FileCheckerTest(python_syntax, ['syntax_dir/'], 'chks_test_dir_good'))

    python_syntax = PythonSyntaxChecker()
    python_syntax.set_options({'best_status':'partial'})
    addTest(FileCheckerTest(python_syntax, ['extentionfilter/'], 'chks_test_dir_partial'))

    ##### PythonUnittestChecker #####
    python_unit = PythonUnittestChecker()
    addTest(ModuleCheckerTest(python_unit, ['goodpkg'], 'chks_test_good'))
    addTest(ModuleCheckerTest(python_unit, ['badpkg1'], 'chks_test_nodata'))
    addTest(ModuleCheckerTest(python_unit, ['badpkg2'], 'chks_test_bad'))
    
    python_unit = PythonUnittestChecker()
    python_unit.set_options({'test_dirs':'dir_for_tetsing'}) # typo is intentional#
    addTest(ModuleCheckerTest(python_unit, ['test_dirs_test_pkg/'], 'chks_test_good'))

    python_unit = PythonUnittestChecker()
    python_unit.set_options({'use_pkginfo_python_versions':'0'}) #
    addTest(ModuleCheckerTest(python_unit, ['goodpkg2.4/'], 'chks_test_good'))

    python_unit = PythonUnittestChecker()
    python_unit.set_options({'use_pkginfo_python_versions':'1'})
    addTest(ModuleCheckerTest(python_unit, ['goodpkg2.4/'], 'chks_test_bad'))
    
    python_unit = PythonUnittestChecker()
    python_unit.set_options({'ignored_python_versions':'2.3'})  #
    addTest(ModuleCheckerTest(python_unit, ['goodpkg2.4/'], 'chks_test_partial'))
    
    python_unit = PythonUnittestChecker()
    python_unit.set_options({'ignored_python_versions':'2.3', 'tested_python_versions':'2.3'}) #
    addTest(ModuleCheckerTest(python_unit, ['goodpkg2.4/'], 'chks_test_partial'))
    
    python_unit = PythonUnittestChecker()
    python_unit.set_options({'tested_python_versions':'2.3', 'use_pkginfo_python_versions':'0'})
    addTest(ModuleCheckerTest(python_unit, ['goodpkg2.4/'], 'chks_test_bad'))
    
    python_unit = PythonUnittestChecker()
    python_unit.set_options({'ignored_python_versions':'2.3', 'tested_python_versions':'2.2'})
    addTest(ModuleCheckerTest(python_unit, ['goodpkg2.4/'], 'chks_test_bad'))

    ##### PythonTestCoverageChecker #####

    python_chks_test_coverage = PythonTestCoverageChecker()
    python_chks_test_coverage.set_options({'treshold':'0'})
    addTest(ModuleCheckerTest(python_chks_test_coverage, ['goodpkg'], 'chks_test_good'))
    
    ##### PythonLintChecker #####
    python_lint = PythonLintChecker()
    python_lint.set_options({'treshold': 7})
    addTest(ModuleCheckerTest(python_lint, ['pylint_ok.py'], 'chks_test_good'))
    addTest(ModuleCheckerTest(python_lint, ['pylint_bad.py'], 'chks_test_bad'))
    
    python_lint_rc = PythonLintChecker()
    python_lint_rc.set_options({'treshold': 7,'pylintrc':input_path("pylintrc")})
    addTest(ModuleCheckerTest(python_lint_rc, ['pylint_bad.py'], 'chks_test_good'))

    xml_syntax = XmlFormChecker()
    addTest(FileCheckerTest(xml_syntax, ['invalid.xml'], 'chks_test_file_good'))
    addTest(FileCheckerTest(xml_syntax, ['malformed.xml'], 'chks_test_file_bad'))
    
    xml_valid = XmlValidChecker()
    addTest(FileCheckerTest(xml_valid, ['invalid.xml'], 'chks_test_file_bad',
                                      options={'catalog': join(INPUTS_DIR,'logilab.cat')}))

    rest_syntax = ReSTChecker()
    addTest(FileCheckerTest(rest_syntax, ['goodrest.txt'], 'chks_test_file_good'))
    addTest(FileCheckerTest(rest_syntax, ['goodrest_2.txt'], 'chks_test_file_good'))
    addTest(FileCheckerTest(rest_syntax, ['goodrest_3.txt'], 'chks_test_file_good'))
    addTest(FileCheckerTest(rest_syntax, ['badrest.txt'], 'chks_test_file_bad'))
    addTest(FileCheckerTest(rest_syntax, ['badrest_2.txt'], 'chks_test_file_bad'))
    if hasattr(chks_pt, 'ZopePageTemplate'):
        pt_syntax = chks_pt.ZPTChecker()
        addTest(FileCheckerTest(pt_syntax, ['task_view.pt'], 'chks_test_file_good'))
        addTest(FileCheckerTest(pt_syntax, ['task_view_bad.pt'], 'chks_test_file_bad'))

    pkg_doc = PackageDocChecker()
    addTest(ModuleCheckerTest(pkg_doc, ['goodpkg'],'chks_test_good'))
    addTest(ModuleCheckerTest(pkg_doc, ['badpkg1', 'badpkg2'], 'chks_test_bad'))

    python_pytest = PythonPyTestChecker()
    addTest(ModuleCheckerTest(python_pytest, ['py_test/goodpkg'], 'chks_test_good'))
    addTest(ModuleCheckerTest(python_pytest, ['py_test/badpkg'], 'chks_test_bad'))    
    
    return testsuite

def Run(runner=None):
    """run tests"""
    testsuite = suite()
    if runner is None:
        runner = unittest.TextTestRunner()
    return runner.run(testsuite)

if __name__ == '__main__':
    Run()
    unittest_main()
