# Copyright (c) 2003-2004 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
unit tests for the Test.Test class
"""

__revision__ = "$Id: unittest_Test.py,v 1.6 2005-12-29 09:11:43 syt Exp $"

import unittest
import shutil
import os
from os.path import exists, join, abspath

from logilab.common.testlib import TestCase


from utils import MockWriter, MockRepository


from apycot import SUCCESS, FAILURE, PARTIAL, SKIPPED
from apycot.Test import *
from apycot.repositories import FSRepository


# mock objects ################################################################

class CleanRaisePreprocessor:
    __name__ = 'clean_raise_preprocessor'
    def match(self, name):
        return 1

    def dependancy_setup(self, test, path):
        return 0
    
    def dependancy_clean(self, test, path):
        return 1
    
    def check_setup(self, test, checker):
        return 1
            
    def check_clean(self, test, checker):
        raise Exception('whatever')
    
class SetupRaisePreprocessor:
    __name__ = 'setup_raise_preprocessor'
    def match(self, name):
        return 1
    
    def dependancy_setup(self, test, path):
        raise SetupException('%s failed on %r' % (self.__name__, path))
    
    def dependancy_clean(self, test, path):
        return 1
    
    def check_setup(self, test, checker):
        raise Exception('in test_preprocessor.check_setup')
            
    def check_clean(self, test, checker):
        return 1
        
class TouchTestPreprocessor:
    __name__ = 'touch_preprocessor'
    file = join(os.getcwd(), 'TestTC_pp')
    file2 = join(os.getcwd(), 'TestTC2_pp')
    
    def match(self, name):
        return 1
    
    def dependancy_setup(self, test, path):
        f = open(self.file, 'w')
        f.close()
        f = open(self.file2, 'w')
        f.close()
    check_setup = dependancy_setup
    
    def dependancy_clean(self, test, path):
        try:
            os.remove(self.file)
        except:
            pass
    check_clean = dependancy_clean


class DummyTest(object):

    def run(self, test, writer):
        return SUCCESS
    def check_options(self):
        pass

class SuccessTestChecker(DummyTest):
    __name__ = 'success_test_checker'
    options = {}

    def run(self, test, writer):
        return SUCCESS
    
class FailureTestChecker(DummyTest):
    __name__ = 'failure_test_checker'
    options = {}
    def run(self, test, writer):
        return FAILURE
    
class ErrorTestChecker(DummyTest):
    __name__ = 'error_test_checker'
    options = {}
    def run(self, test, writer):
        raise Exception('never succeed!')


# real tests ##################################################################

class TestTC(TestCase):
    
    # overridden from unitest.TestCase to return exception instance
    def failUnlessRaises(self, excClass, callableObj, *args, **kwargs):
        """Fail unless an exception of class excClass is thrown
           by callableObj when invoked with arguments args and keyword
           arguments kwargs. If a different type of exception is
           thrown, it will not be caught, and the test case will be
           deemed to have suffered an error, exactly as for an
           unexpected exception.
        """
        try:
            apply(callableObj, args, kwargs)
        except excClass, ex:
            return ex
        else:
            if hasattr(excClass, '__name__'): excName = excClass.__name__
            else: excName = str(excClass)
            raise self.failureException, excName
        
    assertRaises = failUnlessRaises

    def _test_setup_ex(self, test, msg):
        try:
            ex = self.assertRaises(SetupException, test.setup)        
            self.assertEquals(str(ex), msg)
        finally:
            shutil.rmtree('goodpkg')


        
    def test_str(self):
        test = Test('yo', [], MockRepository(),
                    [], None)
        self.assertEquals(str(test), '<MockRepository {}>')
        
    def test_setup(self):
        pp = TouchTestPreprocessor()
        test = Test('yo', [], MockRepository(repository='/home/cvs',
                    path='soft/goodpkg', command='cp -R inputs/goodpkg .'),
                    preprocessors=(pp, ), writer=MockWriter(),
                    environ={'ZIGUOUIGOUI': 'YOOOO'},
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                        'pypasax', []), ))
        try:
            test.setup()
            self.assert_(exists('goodpkg'))
            self.assert_(exists('badpkg2'))
            self.assert_(exists(pp.file))
            self.assertEquals(os.environ['ZIGUOUIGOUI'], 'YOOOO')
        finally:
            shutil.rmtree('goodpkg')
            shutil.rmtree('badpkg2')
            os.remove(pp.file)
            os.remove(pp.file2)
            del os.environ['ZIGUOUIGOUI']
        
    
    def test_setup_raise(self):
        # test bad checkout command
        test = Test('yo', [], MockRepository(repository='/home/cvs',
                                path='soft/goodpkg', command='false'), 
                    preprocessors=[], writer=MockWriter())
        ex = self.assertRaises(SetupException, test.setup)
        self.assertEquals(str(ex), "'false' returned status 256, aborting")
        # test bad dependencies checkout
        test = Test('yo', [], MockRepository(repository='/home/cvs',
                        path='soft/goodpkg', command='cp -R inputs/goodpkg .'),
                    [], MockWriter(),
                    dependencies=((MockRepository(command='false'), 'toto', []), ))
        self._test_setup_ex(test, "'false' returned status 256, aborting")
        # test bad preprocessing on dependency
        test = Test('yo', [], MockRepository(repository='/home/cvs',
                        path='soft/goodpkg', command='cp -R inputs/goodpkg .'),
                    (SetupRaisePreprocessor(), ), MockWriter(),
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                    'pypasax', []), ))
        try:
            self._test_setup_ex(test,
                                "setup_raise_preprocessor failed on 'badpkg2'")
        finally:
            shutil.rmtree('badpkg2')
        
        
    def test_clean(self):
        pp = TouchTestPreprocessor()
        test = Test(name='yo', checks=[], repository=FSRepository(
                                                    {'path':'inputs/goodpkg'}),
                    preprocessors=[pp], writer=MockWriter(),
                    environ={'ZIGUOUIGOUI': 'YOOOO'},
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                                                            'pypasax', tuple()), ),
                    )
        # clean should never fail
        test.clean()
        # but most interesting things occurs after setup...
        test.setup()
        test.clean()
        self.failIf(exists('goodpkg'))
        self.failIf(exists('badpkg2'))
        self.failIf(exists(pp.file))
        os.remove(pp.file2)
        self.assertEquals(os.environ.get('ZIGUOUIGOUI'), None)

    def test_pp_call(self):
        test = Test(name='yo', checks=[],
                    repository=MockRepository(repository='/home/cvs',
                                                        path='soft/goodpkg'),
                    preprocessors=[], writer=MockWriter())
        pp = SetupRaisePreprocessor()
        self.assertEqual(test.pp_call(SuccessTestChecker(), [pp], 'check_setup',
                                                                    verb=0), 0)
        self.assertEqual(test.pp_call(SuccessTestChecker(), [pp], 'check_clean',
                                                                    verb=0), 1)

    def test_run(self):
        pp = TouchTestPreprocessor()
        test = Test('yo', [SuccessTestChecker(), FailureTestChecker(),
                                                            ErrorTestChecker()],
                    FSRepository({'path':'inputs/goodpkg'}),
                    [pp], MockWriter(),
                    environ={'ZIGUOUIGOUI': 'YOOOO'},
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                                                        'pypasax', tuple()), ))
        test.run(verb=0)
        self.failIf(exists(pp.file))
        self.failIf(not exists(pp.file2))
        os.remove(pp.file2)
        
        test = Test('yo', [SuccessTestChecker(), FailureTestChecker(),
                                                            ErrorTestChecker()],
                    FSRepository({'path':'inputs/goodpkg'}),
                    [SetupRaisePreprocessor()], MockWriter(),
                    environ={'ZIGUOUIGOUI': 'YOOOO'},
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                                                       'pypasax', tuple()), ))
        test.run(verb=0)

    def test_execute(self):
        command = 'cp -R '+abspath('inputs/goodpkg')+' .'
        cwd = os.getcwd()
        test = Test(name='yo', checks=[],
               repository=MockRepository(repository='/home/cvs', path='goodpkg',
                                              command=command),
                    preprocessors=[], writer=MockWriter(),
                    environ={'ZIGUOUIGOUI': 'YOOOO'},
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                        'pypasax', tuple()), ))
        self.failIf(not exists(test.tmpdir))
        test.execute()
        self.failIf(exists(test.tmpdir))
        self.assertEquals(os.getcwd(), cwd)
        
        test = Test('yo', [], MockRepository(repository='/home/cvs',
                                    path='goodpkg', command=command),
                    [], MockWriter(),
                    environ={'ZIGUOUIGOUI': 'YOOOO'},
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}), 
                        'pypasax', []), ),
                    keep_temp_dir=1)
        self.failIf(not exists(test.tmpdir))
        test.execute()
        self.failIf(not exists(test.tmpdir))
        self.assertEquals(os.getcwd(), cwd)
        shutil.rmtree(test.tmpdir)
        
        test = Test('yo', [], FSRepository({'path':'inputs/goodpkg'}),
                    [SetupRaisePreprocessor(), ], MockWriter(),
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                        'pypasax',[]), ))
        self.failIf(not exists(test.tmpdir))
        test.execute()
        self.failIf(exists(test.tmpdir))
        self.assertEquals(os.getcwd(), cwd)
        
        test = Test('yo', [], FSRepository({'path':'inputs/goodpkg'}),
                    [CleanRaisePreprocessor(), ], MockWriter(),
                    dependencies=((FSRepository({'path':'inputs/badpkg2'}),
                        'pypasax',[]), ))
        self.failIf(not exists(test.tmpdir))
        test.execute()
        self.failIf(exists(test.tmpdir))
        self.assertEquals(os.getcwd(), cwd)

if __name__ == '__main__':
    unittest.main()
