# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""USAGE: %s [OPTIONS] <configuration_file> [report name]...

Make tester reports according to a configuration file

OPTIONS:
  --help / -h
    Print this help message and exit.
    
  --verbose / -v
    Set the verbosity level. You can set this option multiple times.

  --date / -d <DATE>
    Get report for the given date. Date should be YYYY/MM/DD in daily mode,
    YYYY/MM/DD/HH in hourly mode.
    
  --interval / -i <INTERVAL>
    Set the number of days (in daily mode) or hours (in hourly mode) to consider
    for evolution reports. The mode is specified in the configuration file.
    
  --location / -l <data file location>
    The location of the data file. It should be a directory where the
    tester_data.xml file may be found.
"""

import sys
from getopt import getopt
from os.path import basename

from apycot import get_registered, ConfigError, SkipReport
from apycot.utils import section_as_dict, get_options_from_dict, get_csv, \
     load_addons, MAIN_SECT, DATE_RGX, parse_config, global_config, ConfigError
from apycot.datamanagers import XMLDataManager

# load subpackages to trigger registering
import apycot.reporters
import apycot.decorators
import apycot.transports
import apycot.formatters


MAIN_VARS = ('reports', 'decorators', 'load', 'verbosity')
REPORTS_VARS = ('type', 'transport', 'format')


# utilities ###################################################################
def base_report_config(config, report_name, verb=0):
    """get a report definition as a dictionnary"""
    report_def = global_config(config, MAIN_VARS).copy()
    report_def.update(section_as_dict(config, report_name))
    # load report add-on modules
    if report_def.has_key('load'):
        load_addons(report_def['load'], verb)
        del report_def['load']        
    return report_def

def base_tf_config(config, tf_name):
    """get a transport / formater definition as a dictionnary"""
    tf_def = global_config(config, MAIN_VARS).copy()
    try:
        tf_def.update(section_as_dict(config, tf_name))
    except ConfigError:
        tf_def['type'] = tf_name
    return tf_def

def build_reporter(config, report_name, days, all_reporters, verb=0):
    """get a Reporter object for report_name, according to config"""
    report_def = base_report_config(config, report_name, verb)
    reporter_fact = get_registered('reporter', report_def['type'])
    del report_def['type']
    transport_def = base_tf_config(config, report_def['transport'])
    transport = get_registered('transport', transport_def['type'])(verb)
    del transport_def['type']
    transport.set_options(transport_def)
    formatter = None
    if report_def.has_key('format'):
        formatter_def = base_tf_config(config, report_def['format'])        
        formatter = get_registered('formatter', formatter_def['type'])()
        del formatter_def['type']
        formatter.set_options(formatter_def)
    reporter = reporter_fact(report_name, transport, formatter,
                             all_reporters, verb)
    report_def['interval'] = int(report_def.get('interval', days))
    reporter.set_options(report_def)
    return reporter

# access points ###############################################################

def run_from_config_file(config_file, report_names=None, date=None,
                         days=10, location=None, verb=None):
    """make reports defined in the given config file
    
    see the examples directory for a sample configuration file
    """
    config, verb = parse_config(config_file, location, verb)

    # build decoration chain and the data manager
    decorator = None
    if config.has_option(MAIN_SECT, 'decorators'):
        opts = global_config(config, MAIN_VARS)
        for name in get_csv(config.get(MAIN_SECT, 'decorators')):
            decorator = get_registered('decorator', name)(decorator)
            decorator.set_options(get_options_from_dict(opts, name))
            
    manager = XMLDataManager(config, decorator, date=date, verb=verb)

    # read the main section to get a list of reports definitions
    all_report_names = get_csv(config.get(MAIN_SECT, 'reports'))
    if not report_names:
        report_names = all_report_names
    # build reporter according to the configuration
    if verb:
        print 'building reporters'
    all_reporters = []
    reporters = {}
    for name in all_report_names:
        reporter = build_reporter(config, name, days, all_reporters, verb)
        all_reporters.append(reporter)
        reporters[name] = reporter
    # open all transports
    for reporter in reporters.values():
        reporter.transport.open(manager.date)
    # build reports
    for report_name in report_names:
        if verb:
            print '*' * 80
            print 'make %s report' % report_name
        try:
            reporter = reporters[report_name]
        except KeyError:
            raise ConfigError('No %r report in %s' % (report_name, config_file))
        try:
            reporter.make_reports(manager)
            reporter.transport.close()
        except SkipReport:
            print 'skiping %s report' % report_name
            continue
    
def run(args):
    """Main"""
    l_opts = ['help', 'version', 'date=', 'interval=', 'location=', 'verbose']
    s_opts = 'hVd:i:l:v'
    options, args = getopt(args, s_opts, l_opts)
    date = None
    verb = 0
    days = 10
    location = None
    for arg, val in options:
        if arg in ('-h', '--help'):
            print  (__doc__ % basename(sys.argv[0]))
            sys.exit(0)
        elif arg in ('-V', '--version'):
            from apycot.__pkginfo__ import version
            print 'APyCoT version', version
            print 'python', sys.version
            sys.exit(0)
        elif arg in ('-d', '--date'):
            val = val.strip()
            if not DATE_RGX.match(val):
                print >> sys.stderr, 'Bad date format, should be YYYY/mm/dd[/hh]'
                sys.exit(1)
            date = [int(part) for part in val.split('/')]
        elif arg in ('-i', '--interval'):
            days = int(val)
        elif arg in ('-v', '--verbose'):
            verb += 1
        elif arg in ('-l', '--location'):
            location = val
    if not args:
        print >> sys.stderr, __doc__
        print >> sys.stderr, 'You must specify a configuration file'
        sys.exit(1)
    run_from_config_file(args[0], args[1:], date, days, location, verb or None)
        
if __name__ == '__main__':
    run(sys.argv[1:])
