# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""simple reporters : minimal, synthetized, detailed, multi files
"""

from logilab.common.ureports import Section, Table, Span, VerbatimText, Text, \
     Link
            
from apycot import register, IReporter, ConfigError
from apycot.repositories import get_repository
from apycot.reporters import SimpleReporter, ColumnReporterMixIn, dict_as_table
from apycot.reporters.evolution import base_stats

def resume_section(node):
    """make a resume section with % active / sleep, and % success / failure /
    error
    """
    total, by_status = base_stats(node)
    section = Section(klass='resume')
    active, sleep = by_status.get('active', 0), by_status.get('sleep', 0)
    nb_projects = active + sleep
    if nb_projects:
        formated = {'active': '%.1f %%' % (100. * sleep / nb_projects),
                    'sleep': '%.1f %%' % (100. * active / nb_projects)}
        for status in ('success', 'partial', 'failure', 'error', 'skipped'):
            formated[status] = '%.1f %%' % (100. * by_status.get(status,
                                                                    0) / total)
    else:
        formated = {}
        for status in ('success', 'partial', 'failure', 'error', 'active',
                                                            'sleep', 'skipped'):
            formated[status] = '0'
    section.append(dict_as_table(formated, keys=('active', 'sleep')))
    section.append(dict_as_table(formated, keys=('success', 'partial',
                                                'failure', 'error', 'skipped')))
    return section

class MinimalReporter(SimpleReporter):
    """minimal reporter, outputing a simple table with check status
    """
    __implements__ = IReporter    
    __name__ = 'minimal_report'
    default_id = 'minimal'

    def __init__(self, name, transport, formatter, all_reporters, verbosity=0):
        SimpleReporter.__init__(self, name, transport, formatter,
                                all_reporters, verbosity)
        self.section = None
        self.test_table = None
        
    def cb_open_testsdata(self, node):
        """open a testsdata node (ie the root) hook"""
        self.section = Section(title=self.document_title(node), klass='result')
        self.navigation(self.section, node)
        self.field_info(self.section, node)
        self.section.append(resume_section(node))

    def cb_close_testsdata(self, node):
        """close a testsdata node (ie the root) hook"""
        self.navigation(self.section, node)
        self.post_layout(self.section, self.document_id(node))
        
    def cb_open_test(self, node):
        """open test node hook"""
        title = self.linked_title(node)
        nid = None
        if self.formatter.ulink_support():
            nid = self.relative_url(node)
        section = Section(title=title, id=nid, klass='test')
        self.section.append(section)
        self.field_info(section, node)
        self.test_table = Table(cols=2, klass="field")
        section.append(self.test_table)
        
    def cb_open_check(self, node):
        """open check node hook"""
        title = self.linked_title(node)
        self.test_table.append(title)
        self.test_table.append(Text(self.attr_value(node, 'status')))
    
        
register('reporter', MinimalReporter)


################################################################################

class SynthetizedReporter(ColumnReporterMixIn, SimpleReporter):
    """synthetize reporter, outputing a table with tests as row and check's
    status as columns
    """
    __implements__ = IReporter
    __name__ = 'synthetized_report'
    default_id = 'index'

    def cb_close_testsdata(self, node):
        """close a testsdata node (ie the root) hook"""
        section = Section(title=self.document_title(node), klass='result')
        self.navigation(section, node)
        self.field_info(section, node)
        section.append(resume_section(node))
        table = Table(cols=1+len(self._checks_in_order),
                      cheaders=1, rheaders=1, rrheaders=1)
        section.append(table)
        # headers
        table.append(Text('name'))
        for check_name in self._checks_in_order:
            table.append(Text(check_name.replace('_', ' ')))
        # table content
        test_names = self._tests.keys()
        test_names.sort()
        for test_name in test_names:
            test_def = self._tests[test_name]
            title = self.test_description(test_name, test_def)
            table.append(title)
            for column in self._checks_in_order:
                check_def = self._checks[column].get(test_name)
                if check_def:
                    status = check_def['status']
                    title = self.linked_title(check_def['node'], title=status)
                else:
                    title = status = 'nc'
                table.append(Span(children=(title, ), klass=status))
        # recall table headers
        table.append(Text('name'))
        for check_name in self._checks_in_order:
            table.append(Text(check_name.replace('_', ' ')))
        self.navigation(section, node)
        self.post_layout(section, self.document_id(node))
        
    def test_description(self, test_name, test_def):
        """format a test's description"""
        node = test_def['node']
        test = Span((self.linked_title(node), ), klass='testname')
        try:
            descr = repr(get_repository(dict(node.attrib)))
            descr = Span((Text(descr), ), klass='testinfo')
        except ConfigError:
            descr = repr(node.attrib)
        return Section(children=(test, Text(' '), descr))
    
register('reporter', SynthetizedReporter)


################################################################################

class DetailedReporter(SimpleReporter):
    """detailed reporter, including tests configuration, log messages with
    severity greater than a given treshold and raw values
    """
    __implements__ = IReporter
    __name__ = 'detailed_report'
    default_id = 'detailed'
        
    def __init__(self, name, transport, formatter, all_reporters, verbosity=0):
        SimpleReporter.__init__(self, name, transport, formatter,
                                all_reporters, verbosity)
        self.section = None
        self.test_section = None
        self._tresh = ERROR
        self._raws = None
        self._logs = None
        self._test_path = self._test_repo = self._test_tag = None


    def node_section_title(self, node):
        """return a title for a section associated to a node"""
        label = SimpleReporter.node_section_title(self, node)
        statut = node.get('status')
        if statut is None:
            return label
        else:
            return '%s (%s)' % (label, statut) 

    def reset(self):
        """reset reporting variables"""
        self._tresh = self.get_treshold()

    def cb_open_testsdata(self, node):
        """open a testsdata node (ie the root) hook"""
        self.reset()
        self.section = Section(title=self.document_title(node), klass='result')
        self.navigation(self.section, node)
        self.field_info(self.section, node)
        
    def cb_close_testsdata(self, node):
        """close a testsdata node (ie the root) hook"""
        self.navigation(self.section, node)
        summary = self.summary(self.section)
        if summary:
            self.section.insert(0, summary)
        self.post_layout(self.section, self.document_id(node))

    def cb_open_test(self, node):
        """open test node hook"""
        try:
            self._test_repo = get_repository(dict(node.attrib))
        except ConfigError:
            self._test_repo = None
        title = self.node_section_title(node)
        self.test_section = Section(title=title,
                                    id=self.relative_url(node),
                                    klass='test')
        self.test_section.node = node
        self.section.append(self.test_section)
        self.field_info(self.test_section, node)

    def cb_open_check(self, node):
        """open check node hook"""
        self._raws = []
        self._logs = []
        
    def cb_close_check(self, node):
        """close check node hook"""
        title = self.node_section_title(node)
        check_section = Section(title=title,  id=self.relative_url(node),
                                klass='check')
        check_section.node = node
        self.test_section.append(check_section)
        self.field_info(check_section, node)        
        if self._raws:
            table = Table(cols=2, klass="field", rheaders=1)
            for node in self._raws:
                table.append(Text(node.get('class')))
                table.append(Text(self.text_value(node)))
            check_section.append(table)            
        if self._logs:
            table = Table(cols=4, klass="result", rheaders=1, cheaders=1)
            table.append(Text('path'))
            table.append(Text('line'))
            table.append(Text('severity'))
            table.append(Text('message'))
            repo = self._test_repo
            for node in self._logs:
                path = node.get('path', '')
                vurl = repo and repo.view_url_for(path)
                if vurl:
                    path = Link(vurl, path)
                else:
                    path = Text(path)
                table.append(Span([path], klass="path"))
                table.append(Span([Text(node.get('line', ''))], klass="line"))
                table.append(Span([Text(node.get('severity'))],
                                                            klass="severity"))
                table.append(Span([VerbatimText(self.text_value(node).strip())],
                                  klass="msg"))
            check_section.append(table)

    def cb_open_log(self, node):
        """open log node hook"""
        severity = node.get('severity')
        if eval(severity) < self._tresh:
            return
        self._logs.append( (node) )

    def cb_open_raw(self, node):
        """open raw node hook"""
        self._raws.append( (node) )


register('reporter', DetailedReporter)


###############################################################################

class MultiFilesReporter(DetailedReporter):
    """detailed reporter writing report in a separated file for each test
    """
    __implements__ = IReporter
    __name__ = 'multi_files_detailed_report'    
    default_id = 'multi-index'

    def document_id(self, node=None):
        """return the document's id, optionaly in a context node"""
        while node is not None and node.tag != 'test':
            node = node.getparent()
        if node is None :
            return self.default_id  + self.formatter.output_ext           
        else:
            return node.get('name', '') + self.formatter.output_ext

    def cb_open_testsdata(self, node):
        """open a testsdata node (ie the root) hook"""
        self.reset()
        # dummy section
        self.section = Section()
        
    def cb_close_testsdata(self, node):
        """close a testsdata node (ie the root) hook"""
        
    def cb_open_test(self, node):
        """open test node hook"""
        DetailedReporter.cb_open_test(self, node)
        self.navigation(self.test_section, node)
        # hack: replace navigation
        self.test_section.append(self.test_section.children[1])
        del self.test_section.children[1]
        
    def cb_close_test(self, node):
        """close test node hook"""
        self.navigation(self.test_section, node)
        summary = self.summary(self.test_section)
        if summary:
            self.test_section.insert(2, summary)
        self.post_layout(self.test_section, self.document_id(node))
        
            
register('reporter', MultiFilesReporter)


class ContestReporter(SimpleReporter):
    """a reporter making a context between tested projects :)
    you shoud run test contest decorator to be able to use this report
    """
    __implements__ = IReporter    
    __name__ = 'contest_report'  
    default_id = 'contest'
    
    def __init__(self, name, transport, formatter, all_reporters, verbosity=0):
        SimpleReporter.__init__(self, name, transport, formatter,
                                all_reporters, verbosity)
        self.section = None
        self._contest_nodes = None
        
    def cb_open_testsdata(self, node):
        """open a testsdata node (ie the root) hook"""
        self.section = Section(title=self.document_title(node), klass='result')
        self.navigation(self.section, node)
        self.field_info(self.section, node)
        self._contest_nodes = []

    def cb_close_testsdata(self, node):
        """close a testsdata node (ie the root) hook"""
        data = []
        fields = []
        expression = None
        for node in self._contest_nodes:
            testnode = node.getparent()
            note = float(node.get('status'))
            values = {}
            test_data = [note, testnode.get('name'), testnode, values]
            for raw_node in node:
                name = raw_node.get('class')
                if name == 'expression':
                    if expression is None:
                        expression = raw_node.text

                    continue
                if name == 'fields':
                    continue
                value = float(raw_node.text)
                values[name] = value
                if not name in fields:
                    fields.append(name)
            data.append(test_data)
        if expression is not None: # may be the case during test...
            # add some information about the contest
            contest_info = dict_as_table({'expression': expression})
            self.section.append(contest_info)
        # create a table handling contest result
        data.sort()
        data.reverse()
        table = Table(cols=len(fields) + 3, cheaders=1)
        self.section.append(table)
        for header in ['num', 'project', 'note'] + fields:
            table.append(Text(header))
        # table lines
        for rang in range(len(data)):
            note, name, node, values = data[rang]
            table.append(Text(str(rang + 1)))
            table.append(self.linked_title(node))
            table.append(Text('%.2f' % note))
            for field in fields:
                table.append(Text(str(values.get(field, 'NC'))))
        self.navigation(self.section, node)
        self.post_layout(self.section, self.document_id(node))
        
    def cb_open_check(self, node):
        """open check node hook
        
        add contest check to the list
        """
        if node.get('name') == 'contest':
            self._contest_nodes.append(node)
    
register('reporter', ContestReporter)
