# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""evolution reporters : status_evolution, status_evolution_graph
"""

import tempfile

from logilab.common.ureports import Section, Table, List, Text

from apycot import IReporter, register, NotSupported, SkipReport
from apycot.reporters import StatsReporter
from apycot.utils import get_csv
from apycot.decorators.activity import STATUS


def get_changes(latest_data, date_data):
    """return a list of test.check string representing removed checks between
    the two given dates, and those where the status has changed
    """
    added, status_change = [], []
    tests_def = latest_data['tests']
    for test in tests_def.keys():
        resp_test = date_data['tests'].get(test, {})
        for check in tests_def[test].keys():
            if not resp_test.has_key(check):
                added.append('%s.%s' % (test, check))
            else:
                status = tests_def[test][check]
                resp_status = resp_test[check]
                if status != resp_status:
                    status_change.append( (status, resp_status,
                                           '%s.%s' % (test, check)))
    return added, status_change

def base_stats(data_tree):
    """given a data tree, return the total number of checks and a dictionary
    with check'status as key and number of checks in this status as value
    """
    total, by_status = 0, {}
    for test in data_tree:
        for node in test:
            if node.tag != 'check':
                continue
            total += 1
            status = node.get('status')
            try:
                by_status[status] += 1
            except KeyError:
                by_status[status] = 1
    return total, by_status


class StatusChangeReporter(StatsReporter):
    """base class for stats reporter analysing status changes"""

    def end_collect(self):
        """report building hook, once stats has been calculated"""
        data = self._datas[0][1]
        if len(self._datas) > 1:
            latest_data = self._datas[1][1]
            removed, status_change = get_changes(latest_data, data)
            added = get_changes(data, latest_data)[0]
            data['added'] = added
            data['removed'] = removed
            status_change.sort()
            data['changed'] = status_change
        else:
            data['added'] = data['removed'] = data['changed'] = ()
    
    def handle_data(self, date_tuple, data):
        """compute status statistics on a given data, and store result"""
        total, by_status = base_stats(data)
        tests_data = {}
        for test in data:
            tests_data[test.get('name')] = test_data = {}
            for node in test:
                if node.tag != 'check':
                    continue
                test_data[node.get('name')] = node.get('status')
        date_data = {'total': total,
                     'tests' : tests_data, 'by_status': by_status}
        self._datas.append( (date_tuple, date_data) )

    def get_keys(self, datas):
        """extract the different status returned"""
        columns = self.get_option('columns', None)
        if columns:
            return get_csv(columns)
        result = set()
        for date, data in datas:
            result.update(data['by_status'].keys())
        result = list(result)
        result.sort()
        return result
        

class EvolutionReporter(StatusChangeReporter):
    """status evolution report
    """
    __implements__ = IReporter
    __name__ = 'status_evolution_report'
    default_id = 'status_evolution'

    def end_collect(self):
        """write a simple status evolution report"""
        StatusChangeReporter.end_collect(self)
        layout = Section(self.title)
        self.navigation(layout)
        datas = self._datas
        datas.reverse()
        keys = self.get_keys(datas)
        labels = ['DATE', 'NB TESTS', 'NB CHECKS']
        for i in range(len(keys)):
            key = keys[i]
            if len(key) >= 5:
                key = key[:4] + '.'
            labels.append(key.upper())
        data = []
        table = Table(cols=len(labels), children=labels, cheaders=1)
        layout.append(table)
        for date, data in datas:
            date = ['%02d' % num for num in date]
            by_status = data['by_status']
            table.append(Text('/'.join(date)))
            nb_tests = sum(by_status.get(status,0) for status in STATUS)
            table.append(Text(str(nb_tests)))
            table.append(Text(str(data['total'])))
            for field in keys:
                table.append(Text(str(by_status.get(field, 0))))
        data = datas[-1][1]
        if data['added']:
            descr = List(data['added'])
            layout.append(Section('New checks', children=(descr, )))
        if data['removed']:
            descr = List(data['removed'])
            layout.append(Section('Removed checks', children=(descr, )))
        if data['changed']:
            descr = List(['%s -> %s: %s\n' % change_def
                          for change_def in data['changed']])
            layout.append(Section('Diff status checks', children=(descr, )))
        self.navigation(layout)
        self.post_layout(layout, self.default_id + self.formatter.output_ext)
                      
register('reporter', EvolutionReporter)


COLORS = {'success': 'green', 'failure':'red',
           'error': 'blue', 'activity': 'orange', 'skipped': 'orange'}

class EvolutionGraphReporter(StatusChangeReporter):
    """status evolution report as an image file
    """
    __implements__ = IReporter
    __name__ = 'status_evolution_graph'
    default_id = 'status_evolution'
    output_ext = '.png'
    
    def relative_url(self, node):
        """return the url for node relative to this document
        
        raise NotSupported if the report configuration doesn't permit relative
        urls
        """
        raise NotSupported()
        
    def document_id(self, node=None):
        """return the document's id, optionaly in a context node"""
        return self.default_id + self.output_ext

    def end_collect(self):
        """write a graphic evolution report"""
        datas = self._datas
        datas.reverse()
        plot = biggles.FramedPlot()
        plot.title = self.title
        plot.xlabel = 'date'
        plot.ylabel = 'number'
        ys_axis = {}
        x_axis = []
        x = 0
        keys = ['success', 'error', 'failure', 'activity']#self.get_keys(datas)
        for date, data in datas:
            x_axis.append(x)
            x -= 1
            data = data['by_status']
            for key in keys:
                ys_axis.setdefault(key, []).append(data.get(key, 0))
        for key in ys_axis.keys():
            color = COLORS[key]
            plot.add(biggles.Curve(x_axis, ys_axis[key], color=color))
        filename = tempfile.mktemp()
        plot.write_img(self.output_ext[1:], 500, 500, filename)
        self.transport.add_part(self.default_id + self.output_ext,
                                open(filename))

try:
    import biggles
except ImportError:
    pass
else:
    register('reporter', EvolutionGraphReporter)


class ImageReporterMixIn:
    """a mixin for reporter outputting an image file"""
    output_ext = '.png'
    
    def relative_url(self, node):
        """return the url for node relative to this document
        
        raise `NotSupported` if the report configuration doesn't handle
        relative urls
        """
        raise NotSupported()
        
    def document_id(self, node=None):
        """return the document's id, optionaly in a context node"""
        return self.default_id + self.output_ext

    def render_draw(self, draw):
        """render and transport a reportlab's draw"""
        # FIXME: remove tmp file
        filename = tempfile.mktemp()
        renderPM.drawToFile(draw, open(filename, 'w'),
                                                    self.output_ext[1:].upper())
        self.transport.add_part(self.document_id(), open(filename))


class StatusEvolutionGraphReporter(ImageReporterMixIn, StatusChangeReporter):
    """status evolution report as an image file
    """
    __implements__ = IReporter
    __name__ = 'status_evolution_graph'
    default_id = 'status_evolution'

    def end_collect(self):
        """write a graphic evolution report"""
        datas = self._datas
        datas.reverse()
        success_values = []
        partial_values = []
        failure_values = []
        error_values = []
        x_axis = []
        for date, data in datas:
            x_axis.append('/'.join([str(d) for d in date]))
            data = data['by_status']
            success = data.get('success', 0)
            partial = data.get('partial', 0)
            failure = data.get('failure', 0)
            error = data.get('error', 0)
            success_values.append(success)
            partial_values.append(partial)
            failure_values.append(failure)
            error_values.append(error)
        lines = [ [success_values, colors.green, 'success'],
                  [partial_values, colors.yellow, 'partial'],
                  [failure_values, colors.red, 'failure'],
                  [error_values, colors.blue, 'error'],
                  ]
        graph_type = self.get_option('graph_type', 'line')
        if graph_type != 'bar':
            draw = rl_graph(x_axis, lines, (600, 600),
                                                chart_class=VerticalBarChart)
            chart = draw.contents[0]
            chart.categoryAxis.style = 'stacked'
            chart.groupSpacing = 1
        else:
            for i in range(len(lines[0][0])):
                lines[1][0][i] += lines[0][0][i]
                lines[2][0][i] += lines[1][0][i]
                lines[3][0][i] += lines[2][0][i]
            lines[1][2] = 'success + partial'
            lines[2][2] = 'success + partial + failure'
            lines[3][2] = 'success + partial + failure + error'
            draw = rl_graph(x_axis, lines, (600, 600),
                                                chart_class=HorizontalLineChart)
        self.render_draw(draw)


class ActivityEvolutionGraphReporter(ImageReporterMixIn, StatusChangeReporter):
    """status evolution report as an image file
    """
    __implements__ = IReporter
    __name__ = 'activity_evolution_graph'
    default_id = 'activity_evolution'

    def end_collect(self):
        """write a graphic evolution report"""
        datas = self._datas
        datas.reverse()
        active_values = []
        sleep_values = []
        x_axis = []
        for date, data in datas:
            x_axis.append('/'.join([str(d) for d in date]))
            data = data['by_status']
            active = data.get('active', 0)
            sleep = data.get('sleep', 0)
            active_values.append(active)
            sleep_values.append(sleep + active)
        lines = [ (active_values, colors.green, 'active'),
                  (sleep_values, colors.blue, 'active + sleep') ]
        self.render_draw(rl_graph(x_axis, lines, (600, 400)))#, fill=True))



def rl_graph(categories, lines, size, chart_class=None):
    """return a reportlab drawing object"""
    color_name_pairs = []
    data = []
    for curve, color, label in lines:
        data.append(curve)
        color_name_pairs.append( (color, label) )
    chart = chart_class and chart_class() or HorizontalLineChart()
    chart.data = data
    for i, (curve, color, label) in enumerate(lines):
        chart.set_curve_color(i, color)
    chart.categoryAxis.categoryNames = categories
    chart.categoryAxis.labels.boxAnchor = 'ne'
    chart.categoryAxis.labels.dx = 8
    chart.categoryAxis.labels.dy = -2
    chart.categoryAxis.labels.angle = 50
    chart.categoryAxis.labels.fontName = 'Helvetica'
    chart.categoryAxis.labels.fontSize = 14
    chart.categoryAxis.labels.height = 100
    # FIXME: line chart specific
    #chart.categoryAxis.tickDown = 8
    chart.valueAxis.forceZero = 1
    chart.valueAxis.labels.fontName = 'Helvetica'
    chart.valueAxis.labels.fontSize = 14
    chart.x = 50
    chart.y = 70
    width, height = size
    chart.width = width - 100
    chart.height = height - 150
    legend = Legend()
    legend.alignment = 'right'
    legend.x = 5
    legend.y = height
    legend.dxTextSpace = 5
    legend.fontSize = 16
    legend.fontName = 'Helvetica'
    legend.colorNamePairs = color_name_pairs
    #legend.columnMaximum = 1
    draw = shapes.Drawing(width, height)
    draw.add(chart)
    draw.add(legend)
    return draw

        
try:
    from reportlab.graphics.charts.linecharts import HorizontalLineChart
#    from reportlab.graphics.charts.lineplots import LinePlot
    from reportlab.graphics.charts.barcharts import HorizontalBarChart, \
                                                             VerticalBarChart
    from reportlab.graphics.charts.legends import Legend
    from reportlab.graphics import shapes  #, String
    from reportlab.graphics import renderPM
    from reportlab.lib import colors
    shapes.STATE_DEFAULTS['fontName'] = 'Helvetica'

    def line_set_curve_color(self, curve_index, color):
        self.lines[curve_index].strokeColor = color
    HorizontalLineChart.set_curve_color = line_set_curve_color
#    LinePlot.set_curve_color = line_set_curve_color
    
    def bar_set_curve_color(self, curve_index, color):
        self.bars[curve_index].strokeColor = colors.black
        self.bars[curve_index].fillColor = color
    HorizontalBarChart.set_curve_color = bar_set_curve_color
    VerticalBarChart.set_curve_color = bar_set_curve_color
    
##     import os
##     from reportlab import rl_config
##     rl_config.warnOnMissingFontGlyphs = 0

##     def rl_draw_file(d, fn, fmt, dpi=72, bg=0xffffff, configPIL=None,
##                      showBoundary=rl_config.showBoundary):
##         w = int(d.width*dpi/72.0+0.5)
##         h = int(d.height*dpi/72.0+0.5)
##         c = renderPM.PMCanvas(w, h, dpi=dpi, bg=bg, configPIL=configPIL)
##         c.setFont('Helvetica', 12)
##         renderPM.draw(d, c, 0, 0)
##         renderPM.saveToFile(fn, fmt)
    
##     from reportlab.pdfbase import pdfmetrics
##     from reportlab.pdfbase.ttfonts import TTFont
    
##     def init_reportlab_fonts():
##         """load true type fonts"""
##         base = '/usr/share/fonts/truetype/'
##         for path in os.listdir(base):
##             abspath = base + path
##             if not os.path.isdir(abspath):
##                 continue
##             for fname in os.listdir(abspath):
##                 if fname.endswith('.ttf'):
##                     print 'registering', fname, abspath
##                     pdfmetrics.registerFont(TTFont(fname[:-4],
##                                                    '%s/%s' % (abspath,
##                                                                      fname)))
##     init_reportlab_fonts()
    
except ImportError:
    pass
else:
    register('reporter', StatusEvolutionGraphReporter)
    register('reporter', ActivityEvolutionGraphReporter)




class TestActivityReporter(StatsReporter):
    """test activity report: include test status change and vcs activity
    (require the activity decorator on)
    """
    __implements__ = IReporter
    __name__ = 'test_activity_report'
    default_id = 'test_activity'
    base_title = 'Test activity report from %s to %s'
    raw_classes = ('failed_test_cases', 'error_test_cases')
    
    def stats_interval(self):
        return 1
    
    
    def handle_data(self, date_tuple, data):
        """look for python_unittest checks for each test, record number of test
        success, failures
        """
        tests_data = {}
        for test in data:
            test_data = {}
            for node in test:
                if node.tag != 'check':
                    continue
                if node.get('name').endswith('test'):
                    for chkinfo in node:
                        if chkinfo.tag != 'raw' \
                               or not chkinfo.get('class') in self.raw_classes:
                            continue
                        test_data[chkinfo.get('class')] = int(self.text_value(
                                                                    chkinfo))
                if node.get('name') == 'activity':
                    test_data['activity'] = [self.text_value(log)
                                             for log in node.iterchildren(
                                                                    tag='log')]
            tests_data[test.get('name')] = test_data
        self._datas.append(tests_data)

    def end_collect(self):
        """write a simple status evolution report"""
        assert len(self._datas) <= 2
        if len(self._datas) != 2:
            raise SkipReport()
        # compute diff
        currentdata, previousdata = self._datas
        diffs = {}
        for project, testinfos in currentdata.items():
            if not project in previousdata:
                continue
            ptestinfos = previousdata[project]
            for key in self.raw_classes:
                try:
                    new, old = testinfos[key], ptestinfos[key]
                except KeyError:
                    continue
                if new > old:
                    diffs.setdefault(project, {})[key] = (old, new)
        if not diffs:
            raise SkipReport()
        layout = Section(self.title)
        data = ['project', 'status', 'previous', 'current']
        for project, pdiff in diffs.items():
            for status, (previous, current) in pdiff.items():
                status = status.split('_', 1)[0]
                data.extend([project, status, str(previous), str(current)])
        table = Table(cols=4, children=data, rheaders=1)
        layout.append(table)
        activity = Section('Projects activity')
        layout.append(activity)
        for project, testinfos in currentdata.items():
            activities = testinfos.get('activity')
            if not activities:
                continue
            descr = List(activities)
            activity.append(Section(project, children=(descr, )))
        self.navigation(layout)
        self.post_layout(layout, self.default_id + self.formatter.output_ext)

register('reporter', TestActivityReporter)
