# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""Common reporters utilities"""

import time

from logilab.common.ureports import Section, Table, Paragraph, Span, Link, \
     Text, List, Title, build_summary, layout_title

from apycot import ConfigError, NotSupported
from apycot.utils import SimpleOptionsManagerMixIn, get_csv, SEVERITIES, DAY

def dict_as_table(kwargs, ignore=(), keys=None):
    """return a table layout containing dictionary associations as a 2
    columns table
    """
    table = Table(cols=2, klass="field", rheaders=1)
    if keys is None:
        keys = kwargs.keys()
        keys.sort()
    for field in keys:
        if field in ignore:
            continue
        table.append(Text(field))
        table.append(Text(kwargs[field]))
    return table
    
    
class BaseReporter(SimpleOptionsManagerMixIn):
    """base class for reporter
    
    provide a number of usefull methods
    """
    
    default_id = None

    def __init__(self, name, transport, formatter, all_reporters, verbosity=0):
        SimpleOptionsManagerMixIn.__init__(self)
        # report's name
        self.name = name
        # ITransport object
        self.transport = transport
        # formatter
        self.formatter = formatter
        # manager, given to make_reports
        self.manager = None
        # pointer to all reporters (reports maybe linked togethers)
        self.all_reporters = all_reporters
        # building verbosity
        self.verbosity = verbosity

    def post_layout(self, layout, partname):
        """format the layout and send the result through the transport"""
        stream = self.transport.part_stream(partname)
        self.formatter.format(layout, stream)
        self.transport.add_part(partname, stream)
            
    def get_treshold(self):
        """return the current log treshold value as an integer"""
        treshold = self.get_option('treshold', None)
        if treshold:
            return SEVERITIES[treshold]
        else:
            return ERROR            

    def get_reporter(self, name):
        """return a reporter by its name"""
        for reporter in self.all_reporters:
            if reporter.name == name:
                return reporter
        raise ConfigError('No such report definition %r' % name)
    

    def relative_url(self, node):
        """return the url for node relative to this document
        
        raise NotSupported if the report configuration doesn't permit relative
        urls
        """
        try:
            return self.formatter.relative_url(node)
        except NotSupported:
            return None
    
    def absolute_url(self, node):
        """return the node's absolutprint nodee url
        
        raise NotSupported if the report configuration doesn't permit absolute
        urls
        """
        return '%s#%s' % (self.document_url(node),
                          self.formatter.relative_url(node))
        
    def document_url(self, node=None, date=None):
        """return the document's url for a node
        
        raise NotSupported if the report configuration doesn't permit absolute
        urls
        """
        return '%s/%s' % (self.transport.base_url(date), self.document_id(node))

    def document_id(self, node=None):
        """return the document's id, optionaly in a context node"""
        return self.default_id + self.formatter.output_ext

    def document_title(self, node=None):
        """return the document's title, optionaly in a context node"""
        title = self.get_option('title', None)
        if not (title or node is None):
            title = 'Report for %s' % self.attr_value(node, 'name')
        return title
    
    def text_value(self, node):
        """return text value of a node as an encoded string"""
        try:
            text = node.text
        except IndexError:
            text = 'error: no text value'
        if text is None:
            text = ''
        if isinstance(text, unicode):
            text = self.transport.encode(text)
        return text

    def attr_value(self, node, attribute):
        """return an attribute value of a node as an encoded string"""
        if attribute in node.attrib:
            return self.transport.encode(node.get(attribute))
        else:
            return None

    def linked_title(self, node, title=None):
        """create a title node, handling the link_to option"""
        link_to = None
        nid = None
        if self.formatter.ulink_support():
            link_to = self.get_option('link_to', None)
            nid = self.relative_url(node)
        if title is None:
            title = self.attr_value(node, 'name')
        if link_to:
            reporter = self.get_reporter(link_to)
            return Link(reporter.absolute_url(node), title,
                        id=nid, klass='title')
        else:
            return Text(title, id=nid, klass='title')
    
    def field_info(self, layout, node, ignore=('fake', 'name')):
        """add node fields as a 2 columns table to the layout (if any)"""
        kwargs = dict(node.attrib)
        for key in kwargs:
            kwargs[key]=kwargs[key].encode(self.transport.encoding)
        if kwargs:
            layout.append(dict_as_table(kwargs, ignore))
                
    def navigation(self, layout, node=None):
        """add navigation links to the layout"""
        section = Section(klass="navlinks")
        # previous / next
        try:
            previous = self.manager.previous_date(self.manager.date)
            if previous is not None:
                url = self.document_url(node, date=previous)
                section.append(Link(url, '<', klass='navigation'))
            next = self.manager.next_date(self.manager.date)
            url = self.document_url(node, date=next)
            section.append(Link(url, '>', klass='navigation'))
        except NotSupported:
            pass
        # optional top links
        top_links = self.get_option('top_links', None)
        if top_links:
            for link_to in get_csv(top_links):
                reporter = self.get_reporter(link_to)
                url = reporter.document_url(node)
                title = reporter.document_title(node)
                section.append(Link(url, title, klass='navigation'))
        if section.children:
            layout.append(section)
        
    def summary(self, layout, level=None):
        """return a summary or None according to the user configuration"""
        summary = level or int(self.get_option('summary', 2))
        if summary:
            return self.build_summary(layout, summary)
    
    def build_summary(self, layout, level=1):
        """make a summary for the report, including X level"""
        assert level > 0
        level -= 1
        summary = List(klass='summary')
        for child in layout.children:
            if not isinstance(child, Section):
                continue
            label = layout_title(child)
            if not label and not child.id:
                continue
            if not child.id:
                child.id = label.replace(' ', '-')
            node = Link('#'+child.id, label=label or child.id)
            if level and [n for n in child.children if isinstance(n, Section)]:
                node = Paragraph([node, self.build_summary(child, level)])
            summary.append(node)
        return summary
        
    def node_section_title(self, node):
        """return a title for a section associated to a node"""
        return node.get('name', '????')
        
class SimpleReporter(BaseReporter):
    """base class for reporters using the test data on a single date

    it provides facilities to produce events from an internal data tree
    """
    
    def __init__(self, name, transport, formatter, all_reporters, verbosity=0):
        BaseReporter.__init__(self, name, transport, formatter, all_reporters,
                              verbosity)
        self.test_nodes = []
        
    def make_reports(self, manager):
        """create reports from a data manager (i.e. load data file on demand)
        """
        self.manager = manager
        self.test_nodes = get_csv(self.get_option('filter', ''))
        tree = manager.main_tree()
        self.generate_events(tree.getroot())

    def _filter(self, node):
        """return true if the node shoud be processed"""
        # filter test nodes
        if self.test_nodes and (node.tagname == 'test' and
                                not node.get('name') in self.test_nodes):
            return 0
        return 1
        
    def generate_events(self, node):
        """call hooks according to the data tree, starting from node and
        recursing on its children. Call "cb_open_<node-name>" in preorder
        and "cb_close_<node-name>"  in postorder.
        """
        if not self._filter(node):
            return
        # pre order
        func = getattr(self, 'cb_open_%s' % node.tag, None)
        if func is not None:
            func(node)
        # recursion
        for child in node:
            # do not recurse on text node
            self.generate_events(child)
        # post order
        func = getattr(self, 'cb_close_%s' % node.tag, None)
        if func is not None:
            func(node)


def format_date(date_tuple):
    date = '/'.join([str(num) for num in date_tuple[:3]])
    if len(date_tuple) > 3:
        date += ' %02dh' % int(date_tuple[3])
    return date

class StatsReporter(BaseReporter):
    """base class for evolution reporters, e.g. reporters analyzing difference
    between multiple apycot runs
    """
    
    base_title = 'Evolution Report from %s to %s'
    
    def stats_interval(self):
        interval = int(self.get_option('interval', 10))
        assert interval, interval
        return interval
    
    def make_reports(self, manager):
        """call self.handle_data for each data file in the specified interval"""
        self.manager = manager
        interval = self.stats_interval()
        # get dates to include
        date = manager.date
        dates = [date]
        while interval:
            date = manager.previous_date(date)
            if date is None:
                break
            dates.append(date)
            interval -= 1
        title = self.get_option('title', None)
        if not title:
            title = self.base_title % (format_date(dates[-1]),
                                       format_date(dates[0]))
        self.begin_collect(title)
        for date_tuple in dates:
            data = manager.tree_for_date(date_tuple)
            root = data.getroot()
            self.handle_data(date_tuple, root)
        self.end_collect()
    
    def begin_collect(self, title):
        """called before stats computing

        by default store the title and create a _datas attribute to collect
        statistics
        """
        self.title = title
        self._datas = []
        
    def handle_data(self, date_tuple, data):
        """handle data tree for a given date (tuple Y,M,d). Called from the
        mosts recent date to the oldest considered date
        
        to overide in concrete classes
        """
        raise NotImplementedError()
    
    def end_collect(self):
        """notify the end of data parsing
        
        to overide in concrete classes
        """
        raise NotImplementedError()

    
class ColumnReporterMixIn:
    """a mixin class for column simple reporters"""
    def reset(self):
        """reset reporting variables"""
        self._checks = {}
        self._checks_in_order = []
        self._tests = {}
        self._current = None
                
    def cb_open_testsdata(self, node):
        """initialization hook"""
        self.reset()        

    def cb_open_test(self, node):
        """open test node hook"""
        name = self.attr_value(node, 'name')
        self._tests[name] = {'node': node}
        self._current = name

    def cb_open_check(self, node):
        """open check node hook"""
        name = self.attr_value(node, 'name')
        check_data = {'node' : node, 'status' : self.attr_value(node, 'status')}
        #check_data.update(kwargs)
        if not self._checks.has_key(name):
            self._checks[name] = {}
            # fake check, added by a decorator
            if node.get('fake'):
                self._checks_in_order.insert(0, name)
            else:
                self._checks_in_order.append(name)
        self._checks[name][self._current] = check_data
    

# import submodules to register reporters
import apycot.reporters.simple
import apycot.reporters.evolution

