
__revision__ = '$Id:$'

from logilab.common.tree import Node
import os
from os.path import exists

class NoSuchProcess(Exception): pass

def proc_exists(pid):
    """check the a pid is registered in /proc
    raise NoSuchProcess exception if not
    """
    if not exists('/proc/%s' % pid):
        raise NoSuchProcess()

PPID = 3
UTIME = 13
STIME = 14
CUTIME = 15
CSTIME = 16
VSIZE = 22

class ProcInfo(Node):
    """provide access to process information found in /proc
    """
    
    def __init__(self, pid):
        Node.__init__(self, pid)
        self.pid = pid
        proc_exists(pid)
        self.file = '/proc/%s/stat' % pid
        self.ppid = int(self.status()[PPID])
        
    def memory_usage(self):
        """return the memory usage of the process in Ko
        """
        try :
            return int(self.status()[VSIZE])
        except IOError:
            return 0
    
    def lineage_memory_usage(self):
        return self.memory_usage() + sum(child.lineage_memory_usage() for child in self.children)

    def time(self, children=0):
        """return the number of jiffies that this process has been scheduled
        in user and kernel mode
        """
        status = self.status()
        time = int(status[UTIME]) + int(status[STIME])
        if children:
            time += int(status[CUTIME]) + int(status[CSTIME])
        return time

    def status(self):
        """return the list of fields found in /proc/<pid>/stat
        """
        return open(self.file).read().split()

    
class ProcInfoLoader:
    """manage process information
    """
    def __init__(self):
        self._loaded = {}

    def list_pids(self):
        """return a list of existant process ids
        """
        for subdir in os.listdir('/proc'):
            if subdir.isdigit():
                yield int(subdir)

    def load(self, pid):
        """get a ProcInfo object for a given pid
        """
        pid = int(pid)
        try:
            return self._loaded[pid]
        except KeyError:
            procinfo = ProcInfo(pid)
            procinfo.manager = self
            self._loaded[pid] = procinfo
            return procinfo
        
    
    def load_all(self):
        """load all processes information
        """
        for pid in self.list_pids():
            try:
                procinfo = self.load(pid)
                if procinfo.parent is None and procinfo.ppid:
                    pprocinfo = self.load(procinfo.ppid)
                    pprocinfo.append(procinfo)
            except NoSuchProcess:
                pass
