# Copyright (c) 2003-2004 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""some common report formatters"""

__revision__ = '$Id: formatters.py,v 1.9 2005-04-21 17:39:49 nico Exp $'

from cStringIO import StringIO
from logilab.common.ureports import nodes, TextWriter, HTMLWriter

from apycot import register, IFormatter, NotSupported
from apycot.utils import SimpleOptionsManagerMixIn

class BaseFormatter(SimpleOptionsManagerMixIn):
    """base class for formatters"""
        
    def ulink_support(self):
        """return true if the format support links"""
        return 0
    
    def relative_url(self, node):
        """return the relative url for the node in the report
        may raise NotSupported
        """
        raise NotSupported()

        
class TextFormatter(BaseFormatter, TextWriter):
    """Text formatter, delegating most of the work (the formatting ;) to the
    Textwriter
    """
    __implements__ = IFormatter
    __name__ = 'text'
    output_ext = '.txt'

register('formatter', TextFormatter)


class HTMLFormatter(BaseFormatter, HTMLWriter):
    """HTML formatter, delegating most of the work (the formatting ;) to the
    HTMLwriter
    """
    __implements__ = IFormatter
    __name__ = 'html'
    output_ext = '.html'
    
    def __init__(self):
        BaseFormatter.__init__(self)
        HTMLWriter.__init__(self)
        self.section = 0
        
    def ulink_support(self):
        """HTML format support links, return always true"""
        return 1
    
    def relative_url(self, node):
        """return the relative url for the node in the report"""
        return anchor_name(node)

    def begin_format(self, layout):
        """begin to format a layout"""
        self.section = 0
        self.writeln('<html>')
        self.writeln('<head>')
        title = extract_title(layout)
        if title:
            out = StringIO()
            TextWriter().format(title, out)
            self.writeln('<title>%s</title>' % out.getvalue())
        css = self.get_option('css', None)
        if css is not None:
            self.writeln('<link rel="stylesheet" type="text/css" href="%s"/>'
                         % css)
        self.writeln('</head>')
        self.writeln('<body>')
        
register('formatter', HTMLFormatter)
        


def extract_title(layout):
    """try to get the report title by looking for the first Title layout"""
    if isinstance(layout, nodes.Title):
        return layout
    for child in layout:
        title = extract_title(child)
        if title:
            return title

def anchor_name(node, top='testsdata'):
    """return an unique anchor name for a node"""
    name = ''
    while node.tag != top:
        if name:
            name = "%s.%s" % (node.get('name').replace(' ', ''), name)
        else:
            name = node.get('name').replace(' ', '')
        node = node.getparent()
    return name
