# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
checkers for python source files
"""

import sys
import os
import re

from os.path import join, exists, abspath, dirname
from test import test_support

from logilab.common.testlib import find_tests
from logilab.common.modutils import get_module_files
from logilab.common.fileutils import norm_read
from logilab.common.shellutils import Execute


from pylint.lint import PyLinter
from pylint import checkers
from pylint.__pkginfo__ import version as pylint_version

from logilab.devtools.lib.pkginfo import PackageInfo
from logilab.devtools.__pkginfo__ import version as devtools_version

try:
    # development version
    from logilab.devtools.lib import coverage
except ImportError:
    try:
        # debian installed version
        import coverage
    except ImportError:
        coverage = None

from apycot import register, IChecker, SUCCESS, FAILURE, PARTIAL, merge_status,\
    NODATA
from apycot.utils import get_csv
from apycot.checkers import BaseChecker, AbstractFilteredFileChecker


def install_path(test):
    """return the project's installation path"""
    if hasattr(test, 'install_base'):
        return test.install_base
    return abspath(test.repo.env_path())
        
    
class PythonSyntaxChecker(AbstractFilteredFileChecker):
    """check syntax of python file
    """
    
    __implements__ = IChecker
    __name__ = 'python_syntax'

    def __init__(self):
        AbstractFilteredFileChecker.__init__(self, ('.py', ))
        
    def check_file(self, filepath, writer):
        """try to compile the given file to see if it's syntaxicaly correct
        """
        source = norm_read(filepath) + '\n'
        # Try to compile it. If compilation fails, then there's a 
        # SyntaxError
        try:
            compile(source, filepath, "exec")
            return SUCCESS
        except SyntaxError, error:
            writer.log(ERROR, filepath, error.lineno, error.msg)
            return FAILURE

register('checker', PythonSyntaxChecker)


    
class PythonUnittestChecker(BaseChecker):
    """check that unit tests of a python package succeed

    spawn unittest and parse output (expect a standard TextTestRunner)
    """
    
    __implements__ = IChecker
    __name__ = 'python_unittest'

    result_regex = re.compile(r''\
        'Ran (?P<total>[0-9]+) test cases'\
        ' in (?P<time>[0-9]+(.[0-9]+)?s)'\
        ' (?P<CPUTIME>\([0-9]+(.[0-9]+)?s CPU\))'\
        '(, (?P<errors>[0-9]+) errors)?'\
        '(, (?P<failures>[0-9]+) failures)?'\
        '(, (?P<skipped>[0-9]+) skipped)?')
    
    result_regex = re.compile(r''\
        '(OK|FAILED)'\
        '('\
        ' \('\
        '(failures=(?P<failures>[0-9]+))?'\
        '(, )?'\
        '(errors=(?P<errors>[0-9]+))?'\
        '(, )?'\
        '(skipped=(?P<skipped>[0-9]+))?'\
        '\)'\
        ')?'\
        
        '')
    #   ' in (?P<time>[0-9]+(.[0-9]+)?s)'\
    #   ' (?P<CPUTIME>\([0-9]+(.[0-9]+)?s CPU\))'\
    #   '(, (?P<errors>[0-9]+) errors)?'\
    #   '(, (?P<failures>[0-9]+) failures)?'\
    #   '(, (?P<skipped>[0-9]+) skipped)?')
    
    total_regex = re.compile(r''\
        'Ran (?P<total>[0-9]+) tests?'\
        ' in (?P<time>[0-9]+(.[0-9]+)?s)'\
        '')

    def __init__(self):
        BaseChecker.__init__(self)
        self._path = None
        self.test = None
        
    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)"""
        test_support.verbose = 0
        path = test.repo.env_path()
        try:
            pkginfodir = dirname(test.environ['pkginfo'])
        except KeyError:
            pkginfodir = path
        self.test = test
        status = SUCCESS
        if int(self.get_option("use_pkginfo_python_versions", True)):
            try:
                pkginfo = PackageInfo(directory=pkginfodir)
                pyversions = set(pkginfo.pyversions)
            except (NameError, ImportError):
                pyversions = set()
        else:
            pyversions = set()

        get_option = self.get_option
        tested_python_versions = get_option("tested_python_versions", None)
        if tested_python_versions:
            pyversions.update(set(get_csv(tested_python_versions)))
        ignored_python_versions = get_option("ignored_python_versions", None)
        if ignored_python_versions:
            ignored_python_versions = set(get_csv(ignored_python_versions))
            ignored_python_versions = pyversions.intersection(
                                                    ignored_python_versions)
            if ignored_python_versions:
                for py_ver in ignored_python_versions:
                    writer.log(INFO,None,None,"python's version %s ignored"
                        % py_ver)
                status = PARTIAL
            pyversions.difference_update(ignored_python_versions)
        
        if pyversions:
            writer.raw('python', ', '.join(pyversions))
        else:
            writer.raw('python', sys.executable)
        testdirs = get_csv(self.get_option("test_dirs", 'test, tests'))
        for testdir in testdirs:
            testdir = join(path, testdir)
            if exists(testdir):
                origdir = os.getcwd()
                os.chdir(testdir)
                self._path = testdir
                try:
                    _status = self.run_tests(writer, pyversions)
                    status = merge_status(status, _status)
                finally:
                    os.chdir(origdir)
                break
        else:
            writer.log(ERROR, path, None, 'No test directory !')
            status = NODATA
        return status

    def run_tests(self, writer, pyversions):
        """run a package test suite
        expect to be in the test directory
        """
        if pyversions:
            pyversions = ['python%s' % pyver for pyver in pyversions]
        else:
            pyversions = [sys.executable]
        tests = find_tests('.')
        if not tests:
            writer.log(ERROR, self._path, None, 'No test found !')
            return NODATA
        status = SUCCESS
        all_result = [0, 0, 0, 0]
        total = 0
        for python in pyversions:
            for test_file in tests:
                _status, result = self.run_test(test_file, writer, python)
                for i in (0, 1, 2, 3):
                    total += result[i]
                    all_result[i] += result[i]
                status = merge_status(status, _status)
        writer.raw('total_test_cases', total)
        writer.raw('succeeded_test_cases', all_result[0])
        writer.raw('failed_test_cases', all_result[1])
        writer.raw('error_test_cases', all_result[2])
        writer.raw('skipped_test_cases', all_result[3])
        return status
    
    def run_test(self, test_file, writer, executable='python'):
        """execute the given test file and parse output to detect failed /
        succeed test cases
        """
        status = SUCCESS
        coverageenabled = int(self.get_option('coverage', 1))
        if coverage is None or not coverageenabled or executable == 'python2.1':
            output = Execute('%s -W ignore %s.py' % (executable, test_file))
        else:
            covfile = coverage.__file__.replace('.pyc', '.py')
            output = Execute('%s -W ignore %s -p %s -x %s.py' % (
                executable, covfile, install_path(self.test), test_file))
        absfile = join(self._path, test_file) + '.py'
        total, failures, errors, skipped = 0, 0, 0, 0
        
        result = [0, 0, 0, 0]
        state = 0
        msgs, unk, lineunk = [], [], []

        for line in output.err.splitlines():
            match = self.total_regex.match(line)
            if match is not None:
                total = int(match.groupdict()['total'])
            match = self.result_regex.match(line)
            if match is not None:
                values = match.groupdict()
                failures = int(max(failures, values['failures']))
                errors = int(max(errors, values['errors']))
                skipped = int(max(skipped, values['skipped']))
            msgs.append(line)


        #for char in output.err:
        #    if state == 0:
        #        if char == '.':
        #            result[0] += 1
        #        elif char == 'F':
        #            result[1] += 1
        #            status = merge_status(status, FAILURE)
        #        elif char == 'E':
        #            result[2] += 1
        #            status = merge_status(status, FAILURE)
        #        elif char == 'S':
        #            result[3] += 1
        #            status = merge_status(status, PARTIAL)
        #        elif char == '\n':
        #            if lineunk:
        #                unk.append(char)
        #                lineunk = []
        #            else:
        #                state = 1
        #        else:
        #            unk.append(char)
        #            lineunk.append(char)
        #        continue
        #    msgs.append(char)
        if output.status:
            writer.log(ERROR, absfile, None,
                       '[%s] returned status %s' % (executable, output.status))
            status = FAILURE
        elif total == 0:
            writer.log(ERROR, absfile, None,
                       'File doesn\'t contains any test cases')
            status = NODATA      
        #if unk:
        #    unk = ''.join(unk)
        #    if unk != 'Traceback (most recent call last):':
        #        writer.log(WARNING, absfile, None,
        #                  '[%s] unparsed output :\n%s' % (executable, unk))
        if msgs:
            writer.log(INFO, absfile, None, '[%s]\n%s' % (executable,
                                                         '\n'.join(msgs)))
        
        if errors or failures:
            status = merge_status(FAILURE, status)
        if skipped:
            status = merge_status(PARTIAL, status)

        #failures, errors, skipped = result[1:]
        success = max(0, total - sum(( failures, errors, skipped,)))
        return status, (success, failures, errors, skipped)

register('checker', PythonUnittestChecker)

class PythonPyTestChecker(BaseChecker):
    """check that py.test based unit tests of a python package succeed

    spawn py.test and parse output (expect a standard TextTestRunner)
    """
    
    __implements__ = IChecker
    __name__ = 'python_pytest'


    def __init__(self):
        BaseChecker.__init__(self)
        self._path = None
        self.test = None
        
    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)"""
        test_support.verbose = 0
        path = test.repo.env_path()
        try:
            pkginfodir = dirname(test.environ['pkginfo'])
        except (KeyError, AttributeError):
            pkginfodir = path
        self.test = test
        status = SUCCESS
        try:
            pkginfo = PackageInfo(directory=pkginfodir)
            pyversions = pkginfo.pyversions
            writer.raw('python', ', '.join(pkginfo.pyversions))
        except (NameError, ImportError):
            writer.raw('python', sys.executable)
            pyversions = []
        origdir = os.getcwd()
        testdir = path
        os.chdir(testdir)        
        status = self.run_tests(writer, pyversions)
        os.chdir(origdir)        
        return status

    def run_tests(self, writer, pyversions):
        """run a package py.test suite
        expect to be in the test directory
        """
        if pyversions:
            pyversions = ['python%s' % pyver for pyver in pyversions]
        else:
            pyversions = [sys.executable]

        status = SUCCESS
        all_result = [0, 0, 0, 0]
        total = 0
        for python in pyversions:
            _status, result = self.run_test(writer, python)
            for i in (0, 1, 2, 3):
                total += result[i]
                all_result[i] += result[i]
            status = merge_status(status, _status)
        writer.raw('total_test_cases', total)
        writer.raw('succeeded_test_cases', all_result[0])
        writer.raw('failed_test_cases', all_result[1])
        writer.raw('error_test_cases', all_result[2])
        writer.raw('skipped_test_cases', all_result[3])
        return status
    
    def run_test(self, writer, executable):
        """execute the given test file and parse output to detect failed /
        succeed test cases
        """
        pyreg = re.compile(
            r'(?P<filename>\w+\.py)(\[(?P<ntests>\d+)\] | - )(?P<results>.*)')
        output = Execute('py.test --exec=%s --nomagic --tb=no' % executable)

        result = [0, 0, 0, 0]
        msgs = []
        for testname, _, _, results in pyreg.findall(output.out):
            if results == "FAILED TO LOAD MODULE":
                result[2] += 1
            else:
                result[0] += results.count('.')
                result[1] += results.count('F')
                result[2] += results.count('E')
                result[3] += results.count('s')

                status = SUCCESS
                if 'F' in results or 'E' in results:
                    status = FAILURE
                elif 's' in results:
                    status = PARTIAL
                
            msgs.append("%s: %s"%(testname, results))

        if msgs:
            writer.log(INFO, self._path, None, '[%s] %s' % (executable,
                                                            '\n'.join(msgs)))
        return status, result
                     
register('checker', PythonPyTestChecker)


class MyLintReporter(object):
    """a partial pylint writer (implements only the message method, not
    methods necessary to display layouts
    """
    from pylint.interfaces import IReporter
    __implements__ = IReporter

    def __init__(self, writer, basepath, categories):
        self.writer = writer
        self.categories = set(categories)
        self._to_remove = len(basepath) + 1 # +1 for the leading "/"
        
    def add_message(self, msg_id, location, msg):
        """ manage message of different type and in the context of path """
        if not msg_id[0] in self.categories:
            return
        path, line = location[0], location[-1]
        path = path[self._to_remove:]
        if msg_id[0] == 'I':
            self.writer.log(INFO, path, line, msg)
        elif msg_id[0] in ('F', 'E'):
            self.writer.log(ERROR, path, line, msg)
        else: # msg_id[0] in ('R', 'C', 'W')
            self.writer.log(WARNING, path, line, msg) 

        
class PythonLintChecker(BaseChecker):
    """check that the python package as a decent pylint evaluation
    """
    
    required_options = BaseChecker.required_options+(('treshold', ''), )
    __implements__ = IChecker
    __name__ = 'python_lint'
        
    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)"""
        writer.raw('pylint_version', pylint_version)
        treshold = int(self.get_option('treshold'))
        pylintrc_path = self.get_option('pylintrc', None)
        linter = PyLinter(pylintrc=pylintrc_path)
        linter.set_option('persistent', False)
        linter.set_option('reports', 0, action='store')
        linter.quiet = 1
        # register checkers
        checkers.initialize(linter)
        # load configuration
        package_wd_path = test.repo.env_path()
        if exists(join(package_wd_path, 'pylintrc')):
            linter.load_file_configuration(join(package_wd_path, 'pylintrc'))
        else:
            linter.load_file_configuration()
        # set file or dir to ignore
        for option in ('ignore', ):
            try:
                value = getattr(self, '%s_option' % option)()
            except AttributeError:
                value = self.get_option(option, None)
            if value is not None:
                linter.global_set_option(option.replace('_', '-'), value)
        # message categories to record
        categories = self._csv_option('show_categories', ('E', 'F'))
        linter.set_reporter(MyLintReporter(writer, test.tmpdir, categories))
        # run pylint
        linter.check(install_path(test))
        try:
            note = eval(linter.config.evaluation, {}, linter.stats)
            writer.raw('pylint_evaluation', '%.2f' % note)
        except ZeroDivisionError:
            writer.raw('pylint_evaluation', '0')
            note = 0
        except Exception, ex:
            writer.log(ERROR, test.repo.env_path(), None,
                       'Error while processing pylint evaluation %s' % ex)
            note = 0
        writer.raw('statements', '%i' % linter.stats['statement'])
        if note < treshold:
            return FAILURE
        return SUCCESS

    def additional_builtins_option(self):
        """return <additional_builtins> option value"""
        return self._csv_option('additional_builtins')

    def disable_msg_option(self):
        """return <disable_msg> option value"""
        return self._csv_option('disable_msg')

    def _csv_option(self, optname, default=None):
        """return the <optname> options value or default if not defined
        """
        value = self.get_option(optname, None)
        if value is not None:
            value = get_csv(value)
        else:
            value = default
        return value


register('checker', PythonLintChecker)


class PythonTestCoverageChecker(BaseChecker):
    """check the tests coverage of a python package
    if used, it must be after the python_unittest checker
    """
    
    required_options = BaseChecker.required_options+(('treshold', ''), )
    __implements__ = IChecker
    __name__ = 'python_test_coverage'
    
    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)"""
        writer.raw('devtools_version', devtools_version)
        treshold = self.get_option('treshold')
        assert treshold, 'no treshold defined'
        treshold = int(treshold)
        if os.environ.get('COVERAGE_FILE'):
            percent = self._get_cover_info(writer, install_path(test))
        else:
            # we have to chdir in the project's test directory
            env_path = test.repo.env_path()
            origdir = os.getcwd()
            for testdir in ('test', 'tests'):
                testdir = join(env_path, testdir)
                if exists(testdir):
                    os.chdir(testdir)
                    try:
                        if not exists('.coverage'):
                            writer.log(ERROR, testdir, None, 'No coverage \
    information ! Is the python_unittest executed before this one ?')
                            return NODATA
                        percent = self._get_cover_info(writer,
                                                             install_path(test))
                        break
                    finally:
                        os.chdir(origdir)
            else:
                writer.log(ERROR, env_path, None, 'No test directory !')
                return NODATA
        if percent < treshold:
            return FAILURE
        return SUCCESS

    def _get_cover_info(self, writer, inst_path):
        covertool = coverage.Coverage()
        covertool.restore()
        stats = covertool.report_stat(inst_path, ignore_errors=1)
        percent = stats[coverage.TOTAL_ENTRY][2]
        writer.raw('coverage', '%.3f' % (percent, ))
        result = []
        for name in stats.keys():
            if name == coverage.TOTAL_ENTRY:
                continue
            nb_stmts, nb_exec_stmts, pc, pc_missing, readable = stats[name]
            if pc == 100:
                continue
            result.append( (pc_missing, name, pc, readable) )
        result.sort()
        for _, name, pc_cover, readable in result:
            msg = '%d %% covered, missing %s' % (pc_cover, readable)
            writer.log(INFO, name, None, msg)
        return percent
    
if coverage is not None:
    register('checker', PythonTestCoverageChecker)


class PyCheckerChecker(BaseChecker):
    """check that unit tests of a python package succeed

    spawn unittest and parse output (expect a standard TextTestRunner)
    """
    
    __implements__ = IChecker
    __name__ = 'python_check'


    def __init__(self):
        BaseChecker.__init__(self)
        self._path = None

    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)"""
        inst_path = install_path(test)
        files = get_module_files(inst_path)
        cmdl = 'pychecker -Qqe %s' % ' '.join(files)
        writer.msg(2, cmdl)
        status = SUCCESS
        for line in Execute(cmdl).out.splitlines():
            if not line.strip():
                continue
            try:
                path, line, msg = line.split(':')
                writer.log(ERROR, path, line, msg)
                status = FAILURE
            except ValueError:
                writer.msg(1, 'python_check: unable to parse %r' % line)
        return status
        
register('checker', PyCheckerChecker)
