# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
Checkers for global package structure
"""

from os.path import exists, join

from apycot import register, IChecker, SUCCESS, FAILURE, SKIPPED, NODATA, merge_status
from apycot.writer import LoggerMixIn
from apycot.checkers import BaseChecker
from apycot.checkers.chks_rest import ReSTChecker, docutils_version
from apycot.checkers.chks_xml import XmlFormChecker
from apycot.checkers.chks_html import HTMLChecker, HAS_TIDY

REQUIRED_FILES = ['README']
RECOMMANDED_FILES = ['ChangeLog', 'INSTALL']

class DocTest:
    """wrap Test for documentation test"""
    def __init__(self, test):
        self.__dict__.update(test.__dict__)
        class Repository:
            """a dummy repository object"""
            path = join(test.repo.env_path(), 'doc')
        self.repo = Repository()
        
class PackageDocChecker(BaseChecker):
    """check standard package documentation :
    * presence of some required files (README...)
    * all plain text file are ReST files
    * xml file are well formed
    """
    
    __implements__ = IChecker
    __name__ = 'pkg_doc'

    def __init__(self):
        BaseChecker.__init__(self)
        self._checkers = []
        self._checkers.append(XmlFormChecker())
        if docutils_version is not None:
            self._rest_checker = ReSTChecker()
            self._checkers.append(self._rest_checker)
        else:
            self._rest_checker = None
        if HAS_TIDY:
            self._checkers.append(HTMLChecker())
            
    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)"""
        status = SUCCESS
        path = test.repo.env_path()
        ignore = self.get_option('ignore', '')
        for checker in self._checkers:
            checker.version_info(writer)
            checker.options['ignore'] = ignore
        for filename in REQUIRED_FILES:
            if not exists(join(path, filename)):
                writer.log(ERROR, path, None, 'No %s file' % filename)
                status = FAILURE
        
        for filename in RECOMMANDED_FILES:
            if not exists(join(path, filename)):
                writer.log(INFO, path, None, 'No %s file' % filename)

        if exists(join(path, 'doc')):
            doctest = DocTest(test)
            for checker in self._checkers:
                _res = checker.run(doctest, writer)
                if _res != NODATA:
                    status = merge_status(status, _res)
                else:
                    writer.log(INFO, doctest.path, None, 'No Data found for %s checker'
                               % checker.__name__)

            
        if self._rest_checker is None:
            return status
        
        for filename in REQUIRED_FILES + RECOMMANDED_FILES:
            filename = join(path, filename)
            if exists(filename):
                _res =  self._rest_checker.check_file(filename, writer)
                status = merge_status(status, _res)

        return status

register('checker', PackageDocChecker)


# devtools dependant checkers #################################################

try:
    from logilab.devtools.checkpackage import check_info_module, check_bin, \
         check_release_number, check_manifest_in, check_test, check_setup_py, \
         check_announce
except ImportError:
    if __debug__:
        import traceback
        traceback.print_exc()
    error = traceback.format_exc()
    class ZopePackageChecker(BaseChecker):
        __implements__ = IChecker
        __name__ = 'zope_pkg'
        error_trace = error
        def _run(self, test, writer):
            writer.log(FATAL,None,None,self.error_trace)
            return SKIPPED

    class PythonPackageChecker(ZopePackageChecker):
        __implements__ = IChecker
        __name__ = 'python_pkg'
        error_trace = error
        def _run(self,test,writer):
            writer.log(FATAL,None,None,self.error_trace)
            return SKIPPED
            
else:
    class ReporterWrapper(LoggerMixIn):
        """adapt devtools reporter to tester reporter
        """

        def __init__(self, writer):
            self.writer = writer
            self.counts = None
            self.reset()

        def reset(self):
            """reset counters"""
            self.counts = {}
            for sev in (INFO, WARNING, ERROR, FATAL):
                self.counts[sev] = 0

        def log(self, severity, path, line, msg):
            """log a message of a given severity

            line may be None if unknown
            """
            self.counts[severity] += 1
            self.writer.log(severity, path, line, msg)

        

    class ZopePackageChecker(BaseChecker):
        """check standard zope package :

        * __pkginfo__.py

        * release number mismatch

        * test directory structure

        * announce.txt template
        """
        __implements__ = IChecker
        __name__ = 'zope_pkg'
        __checks__ = (check_info_module, check_release_number, check_test,
                      check_announce)

        def _run(self, test, writer):
            """run the checker against <path> (usually a directory)"""
            status = SUCCESS
            reporter = ReporterWrapper(writer)
            for check_func in self.__checks__:
                reporter.reset()
                _status = check_func(reporter, test.repo.env_path())
                if _status:
                    _status = SUCCESS
                else:
                    _status = FAILURE
                status = merge_status(status, _status)
            return status


    class PythonPackageChecker(ZopePackageChecker):
        """check standard python package :

        * __pkginfo__.py

        * release number mismatch

        * test directory structure

        * MANIFEST.in

        * setup.py

        * executable scripts ('bin' directory)

        * announce.txt template
        """

        __implements__ = IChecker
        __name__ = 'python_pkg'

        __checks__ = (check_info_module, check_release_number,
                      check_manifest_in,
                      check_bin, check_test, check_setup_py, check_announce)
    register('checker', ZopePackageChecker)
    register('checker', PythonPackageChecker)
    

