# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""
checkers for debian packages, using lintian and/or piuparts
"""

import commands
import re
from subprocess import Popen, PIPE, STDOUT

from apycot import IChecker, register
from apycot.checkers import AbstractFilteredFileChecker
from apycot import SUCCESS, FAILURE, ERROR as ST_ERROR


class AbstractDebianChecker(AbstractFilteredFileChecker):
    """abstract checker for debian packages checkers using command line tools
    on generated debian files
    """
    __implements__ = IChecker

    def __init__(self):
        AbstractFilteredFileChecker.__init__(self, self.checked_extensions)

    def base_directory(self, test):
        try:
            return test.deb_packages_dir
        except AttributeError:
            raise ConfigError(
                'this check requires the build_debian preprocessor')
    
    def check_file(self, filepath, writer):
        """try to compile the given file to see if it's syntaxicaly correct
        """
        writer.msg(2, '%s checks %s' % (self.__name__, filepath))
        status = SUCCESS
        iter_line = self._get_output(filepath, writer)
        for line in iter_line:
            line_parts = line.split(':', 1)
            if len(line_parts) > 1:
                mtype, msg = line_parts
                if mtype == 'W':
                    writer.log(WARNING, filepath, None, msg)
                elif mtype == 'E':
                    writer.log(ERROR, filepath, None, msg)
                    status = FAILURE
                else:
                    writer.log(INFO, filepath, None, msg)

            else:
                writer.log(FATAL, filepath, None, "unexpected line %r" % line)
                for line in iter_line:
                    writer.log(INFO, filepath, None, "followed by: %r" % line)
                return ST_ERROR
        return status

    def _get_output(self, filepath):
        raise NotImplementedError()


class DebianLintianChecker(AbstractDebianChecker):
    """check debian packages by parsing lintian output"""
    __name__ = 'debian_lint'
    checked_extensions = ('.deb', '.dsc')
    
    def _get_output(self, filepath, writer):
        cmd = Popen(['lintian', filepath],stdout=PIPE,stdin=open('/dev/null','a'),stderr=STDOUT)
        for line in cmd.stdout:
            yield line
        cmd.wait()
    
    def version_info(self, writer):
        """hook for checkers to add their version information"""
        lintian = commands.getoutput("lintian -V")
        writer.raw('lintian', lintian)
    
register('checker', DebianLintianChecker)


_PIU_MSG = re.compile("^ *[IWEFN]:")

class DebianPiupartsChecker(AbstractDebianChecker):
    """check debian packages by parsing piuparts output"""
    __name__ = 'debian_piuparts'
    checked_extensions = ('.deb', )
        
    def _get_output(self, filepath, writer):
        cmd = Popen(['sudo','-S','piuparts','--pbuilder', filepath],stdout=PIPE,stdin=open('/dev/null','a'),stderr=STDOUT)
        for line in cmd.stdout:
            if _PIU_MSG.match(line):
                yield line.strip()
            else:
                writer.log(INFO,filepath,None,line)
        cmd.wait()
    
    def version_info(self, writer):
        """hook for checkers to add their version information"""
        version = commands.getoutput("piuparts --version").splitlines()[0]
        writer.raw('piuparts', version)

register('checker', DebianPiupartsChecker)
