# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""checkers register
"""

from os.path import walk, splitext, split

from apycot import merge_status, SUCCESS, MISSING, NODATA, TestStatus
from apycot.utils import get_csv, SimpleOptionsManagerMixIn, _MARKER
from apycot.interfaces import IChecker

class BaseChecker(SimpleOptionsManagerMixIn):
    

    def get_option(self, option, default=_MARKER, msg=None):
        """overridden to give a better message
        """
        msg = msg or 'Missing %r option for checker %s' % (option,
                                                        self.__name__)
        return SimpleOptionsManagerMixIn.get_option(self, option, default, msg)
        
    def ignore_option(self):
        """return files or directories to ignore
        
        (cut and paste from AbstractFilteredFileChecker)
        """
        ignore = self.get_option('ignore', '')
        if ignore:
            ignore = get_csv(ignore)
        else:
            ignore = []
        if not 'CVS ' in ignore:
            ignore.append('CVS')
        if not '.svn ' in ignore:
            ignore.append('.svn')
        return ignore

    def run(self, test, writer):
        status = self._run(test, writer)
        new_status = merge_status(status, self.best_status)
        if new_status is status:
            return status

        reason = self.get_option("status_cap_reason", '')
        writer.log(INFO, None, None,
        "Configuration's setting downgrade %s checker status from <%s> to'\
            '<%s>: %s" % (self.__name__, status, new_status, reason))
        return new_status

    def _get_best_status(self):
        best_status = self.get_option("best_status", None)
        if best_status is None:
            return None
        if not isinstance(best_status, TestStatus):
            best_status = TestStatus.get(best_status)
        return best_status
    def _set_best_status(self, value):
        if not isinstance(value, TestStatus):
            value = TestStatus.get(value)
        self.set_option("best_status", value)
    best_status = property(_get_best_status, _set_best_status)

    def _run(self, test, writer):
        raise NotImplementedError()

class MissingChecker(BaseChecker):

    __implements__ = IChecker

    def __init__(self, name, msgs= tuple() ):
        self.__name__ = name
        self.msgs = msgs

    def _run(self, test, writer):
        for msg in self.msgs:
            writer.log(FATAL,None,None,msg)
        return MISSING

class AbstractFilteredFileChecker(BaseChecker):
    """check a directory file by file, with an extension filter
    """
    def __init__(self, extensions=None):
        SimpleOptionsManagerMixIn.__init__(self)
        self.extensions = extensions
        self._res = None
        self._safe_dir = set()

    def base_directory(self, test):
        return test.repo.path
    
    def _run(self, test, writer):
        """run the checker against <path> (usually a directory)

        return true if the test succeeded, else false.
        """
        self.version_info(writer)
        self._res = SUCCESS
        self._nbanalyzed = 0
        ignored = set(self.ignore_option())
        def walk_handler(arg, directory, fnames):
            """walk callback handler"""
            for file in ignored: # fnamesneed to be replace in place
                if file in fnames:
                    fnames.remove(file)
            for filename in fnames:
                ext = splitext(filename)[1]
                if self.extensions is None or ext in self.extensions:
                    src = '%s/%s' % (directory, filename)
                    res = self.check_file(src, writer)
                    self._res = merge_status(self._res, res)
                    self._nbanalyzed += 1
        walk(self.base_directory(test), walk_handler, None)
        writer.raw('total analyzed', self._nbanalyzed)
        if self._nbanalyzed <= 0:
            self._res = merge_status(self._res, NODATA)
        return self._res

    def version_info(self, writer):
        """hook for checkers to add their version information"""
        
    # methods /attributes to overide in concrete classes ######################
    def check_file(self, filepath, writer):
        """check a single file
        return true if the test succeeded, else false.
        """
        raise NotImplementedError()


# import submodules
#
#  chks_pt should be imported explicitly, because it takes a long time to
#  import all the related Zope machinery
import apycot.checkers.chks_debian
import apycot.checkers.chks_pkg
import apycot.checkers.chks_python
import apycot.checkers.chks_rest
import apycot.checkers.chks_xml
import apycot.checkers.chks_html
import apycot.checkers.chks_pt
