# Copyright (c) 2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""USAGE: %s [OPTIONS] <configuration file> [test name]...

Report changes in repositories according to configuration file. If no
verbose option given, only exit with 0 status if some activity has
been detected, else 1. If verbose option is given changes are actually
printed on stdout.


OPTIONS:
  --help / -h
    Display this help message and exit.
  --version / -V
    Display version information and exit.
    
  --verbose / -v
    Set the verbosity level. You can set this option multiple times.

  --date / -d <DATE>
    Get report for the given date. Date should be YYYY/MM/DD in daily
    mode, YYYY/MM/DD/HH in hourly mode.
"""

import sys
from getopt import getopt
import threading
from time import localtime, mktime
from os.path import exists, basename

from apycot import NotSupported
from apycot.repositories import get_repository
from apycot.utils import MAIN_SECT, DATA_SECT, DAY, HOUR, get_csv, \
    parse_config, base_test_config, DATE_RGX

def run_from_config_file(config_file, testnames=None, to_date=None,
                         verbosity=None):
    config, verbosity = parse_config(config_file, verb=verbosity)
    if not testnames:
        # read the main section to get a list of test definitions
        testnames = set()
        for section in config.sections():
            if config.has_option(section,'is_test') and config.getboolean(section,'is_test'):
                testnames.add(section)
    try:
        mode = config.get(DATA_SECT, 'mode')
    except:
        mode = 'daily'
    if mode == 'daily':
        date_offset = DAY
    else: # mode == 'hourly'
        date_offset = HOUR
    if to_date is None:
        to_date = localtime()
    elif len(to_date) != 9:
        to_date = list(to_date) + [0] * (9 -len (to_date))
    from_date = localtime(mktime(to_date) - date_offset)
    status = 1
    for test in testnames:
        testdef = base_test_config(config, test, verbosity)
        repo = get_repository(testdef)
        try:
            logs = list(repo.log_info(from_date, to_date))
        except NotSupported:
            continue
        if not logs:
            continue
        if verbosity:
            print test
            print '*'*len(test)
        for chkininfo in logs:
            status = 0
            if not verbosity:
                return 0
            msg = chkininfo.message_summary().encode(sys.stdout.encoding)
            print '%s: %s' % (chkininfo.author, msg)
        if verbosity:
            print
    if verbosity and status:
        print 'no changes'
    return status


def run(args):
    l_opt = ['help', 'version', 'date=', 'verbose']
    opts, args = getopt(args, 'hVd:v', l_opt)
    verb = 0
    date = None
    
    if not args:
        print __doc__ % basename(sys.argv[0])
        return 0

    for arg, value in opts:
        if arg in ('-h', '--help'):
            print __doc__ % basename(sys.argv[0])
            return 0
        if arg in ('-V', '--version'):
            from apycot.__pkginfo__ import version
            print 'APyCoT version', version
            print 'python', sys.version
            return 0
        if arg in ('-v', '--verbose'):
            verb += 1
        elif arg in ('-d', '--date'):
            value = value.strip()
            if not DATE_RGX.match(value):
                print >> sys.stderr, '"%s":' % value, 'Bad date format, should be YYYY/mm/dd[/hh]'
                sys.exit(1)
            date = [int(part) for part in value.split('/')]
    verb = verb or None
    if not exists(args[0]):
        print 'Aborted: no such file', args[0]
        return 1
    return run_from_config_file(args[0], args[1:], date, verb)
        
if __name__ == '__main__':
    sys.exit(run(sys.argv[1:]))
