# Copyright (c) 2003-2007 LOGILAB S.A. (Paris, FRANCE).
# http://www.logilab.fr/ -- mailto:contact@logilab.fr
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""APyCoT, A Pythonic Code Tester

aims to run daily tests on a source repository
"""

from logilab.common.interface import implements

from apycot.interfaces import *


class TestStatus(object):
    __all = {}

    def __init__(self, name, order, bool):
        self.name = name
        self.order = order
        self.bool = bool
        self.__all[name]=self

    def __int__(self):
        return self.order

    def __nonzero__(self):
        return self.bool

    def __str__(self):
        return self.name

    def __repr__(self):
        return "<TestStatus %s>" % self.name

    def __cmp__(self, other):
        return cmp(int(self), int(other))
    @classmethod
    def get(class_, name):
        return class_.__all.get(name)

# keep order for bw compat
MISSING = TestStatus('missing', -10, False)
SKIPPED = TestStatus('skipped',  -5, False)
KILLED  = TestStatus('killed',   -3, False)
ERROR   = TestStatus('error',    -1, False) #XXX Collide with the builtins severity ERROR
FAILURE = TestStatus("failure",   0, False)
NODATA  = TestStatus('nodata',    2, False)
PARTIAL = TestStatus('partial',   5, True)
SUCCESS = TestStatus('success',  10, True)


def merge_status(global_status, status):
    if global_status is None:
        return status
    elif status is None:
        return global_status
    else:
        return min(global_status, status)
            
class Registry(dict):
    """a registry is a two level dictionnary to group together object in a
    same class
    """
    def __init__(self):
        dict.__init__(self)
        self['__interfaces'] = {}
        
    def get(self, category, name):
        """get a object by name"""
        try:
            return self[category][name]
        except KeyError:
            raise ConfigError('No object %r in category %r' % (name, category))

    def set(self, category, klass):
        """register a class"""
        assert self.check_implements(category, klass)
        name = klass.__dict__.get('__name__', klass.__name__)
        self[category][name] = klass

    def list(self, category):
        """list available object's names registered"""
        return self[category].keys()
        
    def define_category(self, category, interface=None):
        """define a new class of registered objects"""
        assert category != '__interfaces'
        assert not self.has_key(category)
        self[category] = {}
        self['__interfaces'][category] = interface
        
    def check_implements(self, category, klass):
        """check the class object satisfy the interface associated to <category>
        """
        assert implements(klass, self['__interfaces'][category])
        return 1

REGISTRY = Registry()

# testing classes
REGISTRY.define_category('repository', ISourceRepository)
REGISTRY.define_category('preprocessor', IPreProcessor)
REGISTRY.define_category('checker', IChecker)
# reporting classes
REGISTRY.define_category('decorator', IDecorator)
REGISTRY.define_category('reporter', IReporter)
REGISTRY.define_category('formatter', IFormatter)
REGISTRY.define_category('transport', ITransport)

# access point functions
register = REGISTRY.set
list_registered = REGISTRY.list
get_registered = REGISTRY.get

del REGISTRY

class ConfigError(Exception):
    """exception due to a wrong user configuration"""
    
class NotSupported(Exception):
    """a feature is not supported by an implementation"""

class SetupException(Exception):
    """raised in the setup step"""

class SkipReport(Exception):
    """raised to notice a report has to be skipped"""
