# -*- coding: utf-8 -*-
# Copyright: (c) 2017, Ansible Project
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

# Make coding more python3-ish
from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import os
import os.path
import pytest

from ansible.config.manager import ConfigManager, Setting, ensure_type, resolve_path, get_config_type
from ansible.errors import AnsibleOptionsError, AnsibleError
from ansible.module_utils.six import integer_types, string_types

curdir = os.path.dirname(__file__)
cfg_file = os.path.join(curdir, 'test.cfg')
cfg_file2 = os.path.join(curdir, 'test2.cfg')

expected_ini = {'CONFIG_FILE': Setting(name='CONFIG_FILE', value=cfg_file, origin='', type='string'),
                'config_entry': Setting(name='config_entry', value=u'fromini', origin=cfg_file, type='string'),
                'config_entry_bool': Setting(name='config_entry_bool', value=False, origin=cfg_file, type='bool'),
                'config_entry_list': Setting(name='config_entry_list', value=['fromini'], origin=cfg_file, type='list'),
                'config_entry_deprecated': Setting(name='config_entry_deprecated', value=u'fromini', origin=cfg_file, type='string'),
                'config_entry_multi': Setting(name='config_entry_multi', value=u'morefromini', origin=cfg_file, type='string'),
                'config_entry_multi_deprecated': Setting(name='config_entry_multi_deprecated', value=u'morefromini', origin=cfg_file, type='string'),
                'config_entry_multi_deprecated_source': Setting(name='config_entry_multi_deprecated_source', value=u'morefromini',
                                                                origin=cfg_file, type='string')}

ensure_test_data = [
    ('a,b', 'list', list),
    (['a', 'b'], 'list', list),
    ('y', 'bool', bool),
    ('yes', 'bool', bool),
    ('on', 'bool', bool),
    ('1', 'bool', bool),
    ('true', 'bool', bool),
    ('t', 'bool', bool),
    (1, 'bool', bool),
    (1.0, 'bool', bool),
    (True, 'bool', bool),
    ('n', 'bool', bool),
    ('no', 'bool', bool),
    ('off', 'bool', bool),
    ('0', 'bool', bool),
    ('false', 'bool', bool),
    ('f', 'bool', bool),
    (0, 'bool', bool),
    (0.0, 'bool', bool),
    (False, 'bool', bool),
    ('10', 'int', integer_types),
    (20, 'int', integer_types),
    ('0.10', 'float', float),
    (0.2, 'float', float),
    ('/tmp/test.yml', 'pathspec', list),
    ('/tmp/test.yml,/home/test2.yml', 'pathlist', list),
    ('a', 'str', string_types),
    ('a', 'string', string_types),
    ('Café', 'string', string_types),
    ('', 'string', string_types),
    ('None', 'none', type(None))
]


class TestConfigManager:
    @classmethod
    def setup_class(cls):
        cls.manager = ConfigManager(cfg_file, os.path.join(curdir, 'test.yml'))

    @classmethod
    def teardown_class(cls):
        cls.manager = None

    def test_initial_load(self):
        assert self.manager.data._global_settings == expected_ini

    @pytest.mark.parametrize("value, expected_type, python_type", ensure_test_data)
    def test_ensure_type(self, value, expected_type, python_type):
        assert isinstance(ensure_type(value, expected_type), python_type)

    def test_resolve_path(self):
        assert os.path.join(curdir, 'test.yml') == resolve_path('./test.yml', cfg_file)

    def test_resolve_path_cwd(self):
        assert os.path.join(os.getcwd(), 'test.yml') == resolve_path('{{CWD}}/test.yml')
        assert os.path.join(os.getcwd(), 'test.yml') == resolve_path('./test.yml')

    def test_value_and_origin_from_ini(self):
        assert self.manager.get_config_value_and_origin('config_entry') == ('fromini', cfg_file)

    def test_value_from_ini(self):
        assert self.manager.get_config_value('config_entry') == 'fromini'

    def test_value_and_origin_from_alt_ini(self):
        assert self.manager.get_config_value_and_origin('config_entry', cfile=cfg_file2) == ('fromini2', cfg_file2)

    def test_value_from_alt_ini(self):
        assert self.manager.get_config_value('config_entry', cfile=cfg_file2) == 'fromini2'

    def test_config_types(self):
        assert get_config_type('/tmp/ansible.ini') == 'ini'
        assert get_config_type('/tmp/ansible.cfg') == 'ini'
        assert get_config_type('/tmp/ansible.yaml') == 'yaml'
        assert get_config_type('/tmp/ansible.yml') == 'yaml'

    def test_config_types_negative(self):
        with pytest.raises(AnsibleOptionsError) as exec_info:
            get_config_type('/tmp/ansible.txt')
        assert "Unsupported configuration file extension for" in str(exec_info.value)

    def test_read_config_yaml_file(self):
        assert isinstance(self.manager._read_config_yaml_file(os.path.join(curdir, 'test.yml')), dict)

    def test_read_config_yaml_file_negative(self):
        with pytest.raises(AnsibleError) as exec_info:
            self.manager._read_config_yaml_file(os.path.join(curdir, 'test_non_existent.yml'))

        assert "Missing base YAML definition file (bad install?)" in str(exec_info.value)
