import os
from collections import namedtuple
from types import FunctionType

ERROR = object()  # represents a config value not accepted.


def _spinner_input_factory(default):
    from ..animations import spinner_compiler
    from ..styles.internal import SPINNERS
    return __style_input_factory(SPINNERS, spinner_compiler,
                                 'spinner_compiler_dispatcher_factory', default)


def _bar_input_factory():
    from ..animations import bars
    from ..styles.internal import BARS
    return __style_input_factory(BARS, bars, 'bar_assembler_factory', None)


def __style_input_factory(name_lookup, module_lookup, inner_name, default):
    def _input(x):
        return name_lookup(x) or func_lookup(x) or default

    name_lookup = __name_lookup_factory(name_lookup)
    func_lookup = __func_lookup_factory(module_lookup, inner_name)
    return _input


def __name_lookup_factory(name_lookup):
    def _input(x):
        if isinstance(x, str):
            return name_lookup.get(x) or ERROR

    return _input


def __func_lookup_factory(module_lookup, inner_name):
    def _input(x):
        if isinstance(x, FunctionType):
            func_file, _ = os.path.splitext(module_lookup.__file__)
            if x.__code__.co_name == inner_name \
                    and os.path.splitext(x.__code__.co_filename)[0] == func_file:
                return x
            return ERROR

    return _input


def _int_input_factory(lower, upper):
    def _input(x):
        if isinstance(x, int) and lower <= x <= upper:
            return int(x)
        return ERROR

    return _input


def _bool_input_factory():
    def _input(x):
        return bool(x)

    return _input


def _tristate_input_factory():
    def _input(x):
        return None if x is None else bool(x)

    return _input


def _text_input_factory():
    def _input(x):
        return None if x is None else str(x)

    return _input


Config = namedtuple('Config', 'title length spinner bar unknown force_tty manual enrich_print '
                              ' receipt_text monitor stats elapsed title_length spinner_length')


def create_config():
    def reset():
        """Resets global configuration to the default one."""
        set_global(  # this must have all available config vars.
            title=None,
            length=40,
            theme='smooth',  # includes spinner, bar and unknown.
            force_tty=None,
            manual=False,
            enrich_print=True,
            receipt_text=False,
            monitor=True,
            stats=True,
            elapsed=True,
            title_length=0,
            spinner_length=0,
        )

    def set_global(theme=None, **options):
        """Update the global configuration, to be used in subsequent alive bars.

        See Also:
            alive_progress#alive_bar(**options)

        """
        lazy_init()
        global_config.update(_parse(theme, options))

    def create_context(theme=None, **options):
        """Create an immutable copy of the current configuration, with optional customization."""
        lazy_init()
        local_config = {**global_config, **_parse(theme, options)}
        # noinspection PyArgumentList
        return Config(**{k: local_config[k] for k in Config._fields})

    def _parse(theme, options):
        """Validate and convert some configuration options."""

        def validator(key, value):
            try:
                result = validations[key](value)
                if result is ERROR:
                    raise ValueError
                return result
            except KeyError:
                raise ValueError(f'invalid config name: {key}')
            except Exception:
                raise ValueError(f'invalid config value: {key}={value!r}')

        from ..styles.internal import THEMES
        if theme:
            if theme not in THEMES:
                raise ValueError(f'invalid theme name={theme}')
            swap = options
            options = dict(THEMES[theme])
            options.update(swap)
        return {k: validator(k, v) for k, v in options.items()}

    def lazy_init():
        if validations:
            return

        validations.update(  # the ones the user can configure.
            title=_text_input_factory(),
            length=_int_input_factory(3, 300),
            spinner=_spinner_input_factory(None),  # accept empty.
            bar=_bar_input_factory(),
            unknown=_spinner_input_factory(ERROR),  # do not accept empty.
            force_tty=_tristate_input_factory(),
            manual=_bool_input_factory(),
            enrich_print=_bool_input_factory(),
            receipt_text=_bool_input_factory(),
            monitor=_bool_input_factory(),
            stats=_bool_input_factory(),
            elapsed=_bool_input_factory(),
            title_length=_int_input_factory(0, 100),
            spinner_length=_int_input_factory(0, 100),
            # title_effect=_enum_input_factory(),  # TODO someday.
        )
        assert all(k in validations for k in Config._fields)  # ensures all fields have validations.

        reset()
        assert all(k in global_config for k in Config._fields)  # ensures all fields have been set.

    global_config, validations = {}, {}
    create_context.set_global, create_context.reset = set_global, reset
    return create_context


config_handler = create_config()
