/* PSPP - a program for statistical analysis.
   Copyright (C) 2004, 2009, 2011 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef OUTPUT_PLOT_HIST_H
#define OUTPUT_PLOT_HIST_H

#include <gsl/gsl_histogram.h>
#include <stdbool.h>

#include "output/chart-item.h"

struct histogram_chart
  {
    struct chart_item chart_item;
    gsl_histogram *gsl_hist;
    double n;
    double mean;
    double stddev;
    bool show_normal;
  };

/* Creates and returns a new chart that depicts a histogram of
   the data in HIST with the given LABEL.  Labels the histogram
   with each of N, MEAN, and STDDEV that is not SYSMIS.  If all
   three are not SYSMIS and SHOW_NORMAL is true, also draws a
   normal curve on the histogram. */
struct chart_item *histogram_chart_create (const gsl_histogram *,
                                           const char *label, double n,
                                           double mean, double stddev,
                                           bool show_normal);

/* This boilerplate for histogram_chart, a subclass of chart_item, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct chart_item_class histogram_chart_class;

/* Returns true if SUPER is a histogram_chart, otherwise false. */
static inline bool
is_histogram_chart (const struct chart_item *super)
{
  return super->class == &histogram_chart_class;
}

/* Returns SUPER converted to histogram_chart.  SUPER must be a histogram_chart, as
   reported by is_histogram_chart. */
static inline struct histogram_chart *
to_histogram_chart (const struct chart_item *super)
{
  assert (is_histogram_chart (super));
  return UP_CAST (super, struct histogram_chart, chart_item);
}

/* Returns INSTANCE converted to chart_item. */
static inline struct chart_item *
histogram_chart_super (const struct histogram_chart *instance)
{
  return CONST_CAST (struct chart_item *, &instance->chart_item);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct histogram_chart *
histogram_chart_ref (const struct histogram_chart *instance)
{
  return to_histogram_chart (chart_item_ref (&instance->chart_item));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
histogram_chart_unref (struct histogram_chart *instance)
{
  chart_item_unref (&instance->chart_item);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
histogram_chart_is_shared (const struct histogram_chart *instance)
{
  return chart_item_is_shared (&instance->chart_item);
}

static inline void
histogram_chart_submit (struct histogram_chart *instance)
{
  chart_item_submit (&instance->chart_item);
}

#endif /* output/plot-hist.h */
