/*
 * psi-otr.cpp - off-the-record messaging plugin for psi
 *
 * Copyright (C) Timo Engel (timo-e@freenet.de), Berlin 2007.
 * This program was written as part of a diplom thesis advised by 
 * Prof. Dr. Ruediger Weis (PST Labor)
 * at the Technical University of Applied Sciences Berlin.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "PsiOtrPlugin.hpp"
#include "psiotrclosure.h"
#include "OtrMessaging.hpp"
#include "PsiOtrConfig.hpp"

namespace psiotr
{

// ---------------------------------------------------------------------------

namespace
{

// ---------------------------------------------------------------------------

/**
 * Removes the resource from a given JID. 
 * Example:
 * removeResource("user@jabber.org/Home")
 * returns "user@jabber.org"
 */
QString removeResource(const QString& aJid)
{
	QString addr(aJid);
	int pos = aJid.indexOf("/");
	if (pos > -1)
    {
		addr.truncate(pos);
		return addr;
	}
	return addr;
}

// ---------------------------------------------------------------------------

} // namespace

// ===========================================================================

PsiOtrPlugin::PsiOtrPlugin()
    : m_otrConnection(NULL),
      m_onlineUsers(),
      m_psiDataDir(),
      m_optionHost(NULL),
      m_senderHost(NULL)
{
}

// ---------------------------------------------------------------------------

PsiOtrPlugin::~PsiOtrPlugin()
{
}

// ---------------------------------------------------------------------------

QString PsiOtrPlugin::name() const
{
	return "Off-the-Record Messaging";
}

// ---------------------------------------------------------------------------

QString PsiOtrPlugin::shortName() const
{
	return "psi-otr";
}

// ---------------------------------------------------------------------------

QString PsiOtrPlugin::version() const
{
	return "0.5";
}

// ---------------------------------------------------------------------------

QWidget* PsiOtrPlugin::options() const
{
    return new ConfigDialog(m_otrConnection, m_optionHost);
} 

// ---------------------------------------------------------------------------

bool PsiOtrPlugin::enable()
{
    QVariant policyOption = m_optionHost->getGlobalOption(PSI_CONFIG_POLICY);
	m_otrConnection = new OtrMessaging(this,
                                       static_cast<OtrPolicy>(policyOption.toInt()));

    return true;
}

// ---------------------------------------------------------------------------

bool PsiOtrPlugin::disable()
{
    return true;
}

// ---------------------------------------------------------------------------

void PsiOtrPlugin::logout(int account)
{
	if (m_onlineUsers.contains(account))
	{
		foreach(QString jid, m_onlineUsers.value(account).keys())
		{
			m_otrConnection->endSession(QString::number(account), jid);
			m_onlineUsers[account][jid]->setIsLoggedIn(false);
			m_onlineUsers[account][jid]->updateMessageState();
		}
	}
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::contactOnline(int account, const QString& jid)
{
    if (!m_onlineUsers.value(account).contains(jid))
    {
        m_onlineUsers[account][jid] = new PsiOtrClosure(account,
                                                        jid, m_otrConnection);
    }

    m_onlineUsers[account][jid]->setIsLoggedIn(true);
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::contactOffline(int account, const QString& jid)
{
	if (m_onlineUsers.contains(account) && 
	    m_onlineUsers.value(account).contains(jid))
	{
		m_onlineUsers[account][jid]->setIsLoggedIn(false);
		m_onlineUsers[account][jid]->updateMessageState();
	}
}

//-----------------------------------------------------------------------------

QAction* PsiOtrPlugin::getChatDlgMenuEntries(QWidget* parent, int account,
                                             const QString& otherJid)
{
    Q_UNUSED(parent);
	QString toJ = removeResource(otherJid);

	if (!m_onlineUsers.value(account).contains(toJ))
	{
        m_onlineUsers[account][toJ] = new PsiOtrClosure(account,
                                                        toJ, m_otrConnection);
	}

	return m_onlineUsers[account][toJ]->getChatDlgMenu(parent);
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::setHomeDir(const QString& dir)
{
    m_psiDataDir = dir;
}

//-----------------------------------------------------------------------------

bool PsiOtrPlugin::processEvent(int account, const QDomElement& e)
{
    Q_UNUSED(account)
    Q_UNUSED(e)

    return false;
}

//-----------------------------------------------------------------------------

bool PsiOtrPlugin::processMessage(int account, const QString& fromJid,
                                  QString& body, QString&)
{
    QString contact = removeResource(fromJid);

    QString decrypted = m_otrConnection->decryptMessage(contact,
                                                        QString::number(account),
                                                        body);
	if (m_onlineUsers.contains(account) && 
	    m_onlineUsers.value(account).contains(contact))
	{
		m_onlineUsers[account][contact]->updateMessageState();
	}
	
    body = decrypted;

    return false;
}	

//-----------------------------------------------------------------------------

bool PsiOtrPlugin::processMessage(int account, const QString& fromJid,
                                  QDomElement& body, QString& subject)
{
    QString htmlBody;
    QTextStream ts(&htmlBody);
    body.save(ts, 0);

    // We have to add a new body tag for messages which were decrypted. Because
    // we don't know if a message is plaintext or encrypted we always remove
    // the body tag first. 
    QRegExp openTag("<body.*>", Qt::CaseInsensitive);
    QRegExp closeTag("</body.*>\r\n", Qt::CaseInsensitive);
    openTag.setMinimal(true);
    closeTag.setMinimal(true);
    htmlBody.replace(openTag, "");
    htmlBody.replace(closeTag, "");

    htmlBody.replace(QRegExp("\r"), "");
    htmlBody.replace(QRegExp("\n"), "");

    bool ret = processMessage(account, fromJid, htmlBody, subject);

    htmlBody = "<body>" + htmlBody + "</body>";

	QDomDocument doc;
	int errorLine, errorColumn;
	QString errorText;
	if (!doc.setContent(htmlBody, true, &errorText, &errorLine, &errorColumn))
	{
        // Pidgin with OTR plugin sends invalid XHTML messages. We replace
        // <br> by <br/> and remove all other html tags.
        QString htmlCleanup("<body>");
		bool bracket = false;
		for (int i = 0; i < htmlBody.length(); ++i)
		{
			QChar c = htmlBody.at(i);

			if (htmlBody.mid(i, 3).toLower() == "<br")
			{
			    htmlCleanup.append("<br/>");
			}

			if (c == '<')
			{
				bracket = true;
				continue;
			}
			if (c == '>')
			{
				bracket = false;
				continue;
			}

			if (!bracket)
			{
				htmlCleanup.append(c);
			}
		}
        htmlCleanup.append("</body>");

		if (!doc.setContent(htmlCleanup, true, &errorText, &errorLine, &errorColumn))
		{
			qWarning() << "XML parsing failed, message ignored: line:" << errorLine 
                       << " column:" << errorColumn << " " << errorText << endl
                       << "--" << endl << htmlCleanup << "\n--";
		    QString errorMessage = "<body>psi-otr: received invalid xhtml message</body>";
			doc.setContent(errorMessage, true, &errorText, &errorLine, &errorColumn);
		}
	}
			
    body = doc.documentElement().cloneNode(true).toElement();

    return ret;
}

//-----------------------------------------------------------------------------

bool PsiOtrPlugin::processOutgoingMessage(int account, const QString& toJid,
                                          QString& body, QString&)
{
    QString encrypted = m_otrConnection->encryptMessage(
        QString::number(account),
        removeResource(toJid),
        body);

    body = encrypted;

    return false;
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::setOptionAccessingHost(OptionAccessingHost* host)
{
    m_optionHost = host;
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::optionChanged(const QString&)
{
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::setStanzaSendingHost(StanzaSendingHost *host)
{
    m_senderHost = host;
}

//-----------------------------------------------------------------------------

void PsiOtrPlugin::sendMessage(int account, const QString& toJid,
                               const QString& message)
{
    m_senderHost->sendMessage(account, toJid, message, "", "chat");
}

//-----------------------------------------------------------------------------

QVariant PsiOtrPlugin::getPsiOption(const QString& option)
{
    return m_optionHost->getGlobalOption(option);
}

//-----------------------------------------------------------------------------

QString PsiOtrPlugin::psiDataDir()
{
    return m_psiDataDir;
}

//-----------------------------------------------------------------------------

bool PsiOtrPlugin::isLoggedIn(int account, QString jid)
{
	if (m_onlineUsers.contains(account) &&
	    m_onlineUsers.value(account).contains(jid))
	{
		return m_onlineUsers.value(account).value(jid)->isLoggedIn();
	}

    return false;
}

//-----------------------------------------------------------------------------

} // namespace psiotr

//-----------------------------------------------------------------------------

Q_EXPORT_PLUGIN2(psiOtrPlugin, psiotr::PsiOtrPlugin)

//-----------------------------------------------------------------------------
