/* type used for parsing and printing options.
   An array of string, int pairs. The idea is that
   for an enum of n values, the first n pairs give the 
   canonical name; then subsequent pairs can be used to
   give aliases. The array is terminated by a entry with null
   name; a name of the empty string shd be used for blank entries.
  */
typedef struct { char *name; int value; } OptEntry;

/* all the data types representing colours, cassettes, models, etc. 
 */

/* list of the colours available (to the user of the program). */
typedef enum {
  colBlack = 0x00,
  colCyan = 0x01,
  colMagenta = 0x02,
  colYellow = 0x03,
  colMetallicGold = 0x04,
  /* note that the next two colours are called
     metallic red and metallic blue in the Alps developer 
     documentation. However, the cartridges are actually
     magenta and cyan, and that's what they're called
     on the packaging. */
  colMetallicMagenta = 0x05,
  colMetallicCyan = 0x06,
  colMetallicSilver = 0x07,
  colLabecaBlack = 0x08,
  colLabecaRed = 0x09,
  colLabecaBlue = 0x0A,
  colWhite = 0x0B,
  colDyeSubOvercoat = 0x0C,
  colDyeSubGlossyFinish = 0x0D,
  colGlossyFinish = 0x0E,
  colReserved0F = 0x0F,
  colVPhotoPrimer = 0x10,
  colGoldFoil = 0x11,
  colReserved12 = 0x12,
  colReserved13 = 0x13,
  colSilverFoil = 0x14,
  colReserved15 = 0x15,
  colEconoBlack = 0x16,
  /* pseudo values for foil to be printed without undercoat */
  colRawGoldFoil = 0x17,
  colRawSilverFoil = 0x18,
  /* pseudo value for no colour to be printed */
  colNullSpot = 0xF0,
  colNullFoil = 0xF1,
} colCode;

extern const unsigned char colour_to_command[];

extern const OptEntry colour_table[];

/* list of the bar codes on cassettes */
typedef enum {
  barBlack = 0,
  barYellow = 1,
  barMagenta = 2,
  barCyan = 3,
  barMultiColour = 4,
  barGoldFoil = 5,
  barSilverFoil = 6,
  barReserved7 = 7,
  barMetallicGold = 8,
  barMetallicMagenta = 9,
  barMetallicCyan = 10,
  barMetallicSilver = 11,
  barReserved12 = 12,
  barOHPYellow = 13,
  barOHPMagenta = 14,
  barOHPCyan = 15,
  barWhite = 16,
  barFinish = 17,
  barVPhotoPrimer = 18,
  barFinishII = 19,
  barLabecaBlack = 20,
  barLabecaBlue = 21,
  barLabecaRed = 22,
  barEconoBlack = 23,
  barReserved24 = 24,
  barDyeSubYellow = 25,
  barDyeSubMagenta = 26,
  barDyeSubCyan = 27,
  barDyeSubOvercoat = 28,
  barReserved29 = 29,
  barDyeSubFinish = 30,
  barReserved31 = 31,
  // DP-7000 cartridges follow here
  barSpotRed = 32,
  barSpotGreen = 33,
  barSpotBlue = 34,
  barProcessBlack = 35, // actually this is called 35R, and has an unusually
                        // long lead-in
  barProcessYellow = 36,
  barProcessMagenta = 37,
  barProcessCyan = 38,
  //  barHighestCode = 32,
  barNoCassette = 127
} barCode;

extern const barCode colour_to_barcode[];
extern const barCode colour_to_barcode_dyesub[];
extern const char *cassette_names[];

/* media types. Important: these are NOT the PCL values, because
   the PCL values are two bytes. These are just enums.
   The prefix Fine here is called (Enhanced Quality) in the
   Alps docs, and means that everything is printed twice. 
   These are arranged in rows by first byte of media code */
typedef enum {
  mediaPlainPaper,  mediaFinePlainPaper,
  mediaFineSpecialOHP,  mediaSpecialOHP,
  mediaSpecialIron,  mediaIronSheet,
  mediaLabecaSheet,
  mediaThermalPaper,  mediaCDMaster,
  mediaCardboard,
  mediaPostCard,
  mediaLaserPaper,
  mediaFineOHP,  mediaOHP,
  mediaBackPrint,  mediaFineBackPrint,
  mediaDyeSubPaper,
  mediaReserved0B00,
  mediaDyeSubLabel,
  mediaReserved0D00,
  mediaGlossyLabel,
  mediaGlossyPaper,
  mediaVPhotoFilm,
  mediaVPhotoCard,
  mediaHighestType = mediaVPhotoCard
} mediaType;

extern const char media_byte1[];
extern const char media_byte2[];
extern const OptEntry media_table[];

/* paper sizes. Note: these values *are* the first byte
   of the data for the paper size command. */
typedef enum {
  paperCustom = 0x00,
  paperExecutive = 0x01,
  paperLetter = 0x02,
  paperLegal = 0x03,
  paperA4 = 0x04,
  paperB5 = 0x05,
  paperPostCard = 0x06,
  paperDyeSubLabel = 0x07,
  paperHighestCode = 0x07,
} paperSize;

extern const OptEntry paper_table[];

/* This is a list of known micro dry models.
   These enums are used as flag bit indices, so if
   we go above 31 we need to adjust some things below.
*/
typedef enum {
  /* citizen models; there will be no more of these */
  /* The Printiva600 was the first released in Europe, and is
     probably the same as the MD2000. It didn't understand
     the metallic silver ribbon */
  modelPrintiva600 = 0, /* unmodified Citizen Printiva 600C */
  /* 600s could be upgraded to understand silver. This is
     probably the MD2010 */
  modelPrintiva600U = 1,
  /* This is probably also the MD2010 */
  modelPrintiva700 = 2,
  /* This is the 700 with a scanner */
  modelPrintiva1700 = 3,
  /* Only sold in Japan: probably didn't understand silver ? */
  modelMD2000 = 4,
  /* The basic model */
  modelMD2010 = 5,
  /* a 2010 with a scanner */
  modelMD4000 = 6,
  /* This added dyesub printing */
  modelMD2300 = 7,
  /* This appears to be the 2010 with the ability to print
     in 1200x600 in colour, and the white and glossy-finish
     ribbons. */
  modelMD1000 = 8,
  /* Has dyesub. Difference between 1300 and 2300 is unknown to me.
     Probably the 2300 didn't support white. */
  modelMD1300 = 9,
  /* 1500 seems to have been japanese only */
  modelMD1500 = 10,
  /* the current (outside japan) model */
  modelMD5000 = 11,
  /* current japanese model. differences from 5000 unknown */
  modelMD5500 = 12,
  /* Oki models. These are probably identical in all relevant
     respects to the corresponding alps version */
  modelDP5000 = 20,
  modelHighestNum = 20
} model;

extern const OptEntry model_table[];

#define mbit(m) (1 << m)

/* now some properties... */
/* the main distinction is the 5000 and after, versus those before.
   This has to be a define, so that it's compile time constant */
#define model5k (mbit(modelMD5000) | mbit(modelMD5500) | mbit(modelDP5000))

extern const int modelHasWhite;
extern const int modelHasFinish;
extern const int modelHasDyeSub;
extern const int modelHasHighResColour;
extern const int modelHasVPhoto;
extern const int modelHasVPhotoPrimer;
extern const int modelHasFoil;
extern const int modelHasEconoBlack;
extern const int modelHas5000PageSizes;
extern const int modelHasPrintModes;
extern const int modelHasMultiRaster;

/* test model */
#define mtest(p,m) (p & mbit(m))

/* Mode in which colour planes are transferred. */
typedef enum {
  blackRaster = 0x00,
  econoblackRaster = 0x01,
  colourRaster = 0x02,
  /* 0x03 is not used */
  colourPlane = 0x04,
  cassetteRaster = 0x05,
  /* 0x06 is not used */
  cassettePlane = 0x07,
  multiPlane = 0x08,
  highestMode = 0x08
} transferMode;

extern const OptEntry transfer_mode_table[];

/* (for the 5000 series only) the printing mode, which can be
   set independently of the media type.
   Note that these are just enums.
*/
typedef enum {
  byMediaMode = 0,
  normalColourMode, normalMonoMode,
  vphotoColourFilmMode, vphotoMonoFilmMode,
  dyeSubColourMode, dyeSubMonoMode,
  vphotoColourPaperMode, vphotoMonoPaperMode,
  normalColourFilmMode, normalMonoFilmMode,
  vphotoColourPrimerMode, vphotoMonoPrimerMode
} printMode;

extern unsigned char print_mode_bytes[];
