/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.BeanDescriptor;
import java.beans.BeanInfo;
import java.beans.Customizer;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;
import java.beans.PropertyEditor;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

public class Inspector extends JPanel {
	
	private static final long serialVersionUID = -6527245644109661448L;
	
	Class<?> type;
	ArrayList<PropertyEditor> editors = new ArrayList<PropertyEditor>();
	ArrayList<PropertyDescriptor> properties=new ArrayList<PropertyDescriptor>();
	
	private boolean reading;
	private Object currObject;
	
	private List<ChangeListener> changeListeners = new CopyOnWriteArrayList<ChangeListener>();

	private Collection<String> excludedPropertyNames;
	private Customizer customizer = null;  //customizer for inspected class if existing
	
	public Inspector(Object o, Collection<String> excludedPropertyNames) throws IntrospectionException {
		setBorder(new EmptyBorder(10,10,10,0));
		
		if (o==null) throw new NullPointerException("object==null");
		type=o.getClass();
		currObject=o;
		
		this.excludedPropertyNames = excludedPropertyNames;
		
		boolean hasCustomizer=initWithCustomizer();
		if (!hasCustomizer)	initWithPropertyEditors();

		refresh();
	}




	public Object getObject() {
		return currObject;
	}
	
	public Collection<String> getExcludedPropertyNames() {
		return excludedPropertyNames;
	}
	

	public void refresh() {
		reading=true;
		try {
			if (customizer!=null)  //update existing customizer 
				customizer.setObject(currObject);
			else {  //update property editors
				for(int ix=0, n=properties.size(); ix<n; ix++) 
					try {
						PropertyDescriptor pd= properties.get(ix);
						Method readMethod=pd.getReadMethod();
						if(readMethod==null) continue;
						PropertyEditor pe= editors.get(ix);
						pe.setValue(readMethod.invoke(currObject, (Object[])null));
					} catch(Exception ex){
						ex.printStackTrace();
					}
			}
		} finally {
			reading=false;
		}
	}

	/** Change listeneres are informed whenever properties of the inspected (!) object change. 
	 * The source of the passed ChangeEvent is the inspected Object rather than the Inspector.
	 * (For change of the <code>Inspector</code> register a propertyChangeListener, which gets notified when
	 * <code>JPanel</code>-properties of the <code>Inspector</code> change.)  
	 * 
	 * @param listener
	 */
	public void addChangeListener(ChangeListener listener) {
		changeListeners.add(listener);
	}
	
	public void removeChangeListener(ChangeListener listener) {
		changeListeners.remove(listener);
	}

	private void fireStateChanged() {
		for (ChangeListener changeListener : changeListeners) {
			changeListener.stateChanged(new ChangeEvent(currObject));
		}
	}

	/** Get the BeanInfo of the current object (currObject). 
	 * 
	 * TODO: If the current object is a ProxyClass, it
	 * returns the BeanInfo of the first interface where instantiation of 
	 * a BeanInfo class succeeds, rather than all those interfaces.
	 * 
	 * @return the BeanInfo of <code>currObject</code>
	 * @throws IntrospectionException
	 */
	private BeanInfo getBeanInfo() throws IntrospectionException {
		BeanInfo bi=null;
		if (Proxy.isProxyClass(currObject.getClass())) {
			LinkedList<BeanInfo> bis = new LinkedList<BeanInfo>();
			for (Class<?> interf : currObject.getClass().getInterfaces()) {
				String name = interf.getName() + "BeanInfo";
		        try {
		        	bis.add((BeanInfo) currObject.getClass().getClassLoader().loadClass(name).newInstance());
		        } catch (Exception ex) {
			    // Just drop through
		        }
			}
			//if (bis.size() > 1) WARNING
			if (bis.size() > 0) bi=bis.getFirst();
		}
		if (bi == null) bi = Introspector.getBeanInfo(type);
		return bi;
	}
	
	/** Tries to init the Inspector with a Customizer. If this method returns falls, nothing was 
	 * added to the Inspector.
	 * 
	 * @return if the class has a Customizer and all neccessary introspection went well.  
	 * 
	 */
	private boolean initWithCustomizer() {
		try {
			BeanInfo bi = getBeanInfo();
			BeanDescriptor bd = bi.getBeanDescriptor();
			if (bd == null || bd.getCustomizerClass() == null) return false;		
			Class<?> customizerClass = getBeanInfo().getBeanDescriptor().getCustomizerClass();
			customizer = (Customizer)customizerClass.newInstance();
		} catch (Exception e) {
			return false;
		}
		setLayout(new BorderLayout());
		customizer.setObject(currObject);
		customizer.addPropertyChangeListener(new PropertyChangeListener() {
			@Override
			public void propertyChange(PropertyChangeEvent evt) {
				fireStateChanged();
			}
		});
		add((Component) customizer);
		return true;
	}

	/**
	 * @param excludedPropertyNames
	 * @throws IntrospectionException
	 */
	private void initWithPropertyEditors()
			throws IntrospectionException {

		PropertyDescriptor[] pd=getBeanInfo().getPropertyDescriptors();

		setLayout(new GridBagLayout());
		GridBagConstraints labelConstraints=new GridBagConstraints();
		labelConstraints.anchor=GridBagConstraints.EAST;
		labelConstraints.ipadx=5;
		GridBagConstraints editorConstraints=new GridBagConstraints();
		editorConstraints.fill=GridBagConstraints.BOTH;
		editorConstraints.anchor=GridBagConstraints.CENTER;
		editorConstraints.weightx=1;
		editorConstraints.gridwidth=GridBagConstraints.REMAINDER;

		for (int ix=0, num=pd.length; ix<num; ix++) {
			try {
				PropertyDescriptor descriptor=pd[ix];
				if (excludedPropertyNames==null || 
						!excludedPropertyNames.contains(descriptor.getName())) {
					final PropertyEditor pe=editor(descriptor);
					Component component=editorComponent(pe, descriptor);
					if(component==null) continue;

					JLabel label=new JLabel(descriptor.getDisplayName());
					label.setLabelFor(component);

					String sd = descriptor.getShortDescription();
					if(sd!=null && !sd.equals(descriptor.getDisplayName()))
						label.setToolTipText(sd);
					properties.add(pd[ix]);
					editors.add(pe);
					add(label, labelConstraints);
					add(component, editorConstraints);

					final Method m=descriptor.getWriteMethod();
					if(m!=null) {
						pe.addPropertyChangeListener(new PropertyChangeListener() {
							public void propertyChange(PropertyChangeEvent evt) {
								// do not set the property while refreshing 
								if(!reading) {
									try {
										m.invoke(currObject, new Object[]{ pe.getValue() });
									} catch (Exception e) {
										e.printStackTrace();
									} 
									fireStateChanged();
								}
							}
						});
					}
				}
			} catch(Exception ex) {
				ex.printStackTrace();
			}
		}

		//add something that fills the remaining vertical space
		editorConstraints.weighty=1;
		add(new JLabel(), editorConstraints);
	}

	@SuppressWarnings("unchecked") //the cast for the EnumEditor is checked
	private PropertyEditor editor(PropertyDescriptor descriptor) {
		PropertyEditor pe = EditorManager.findEditor(descriptor);
		if (pe == null) {
			if (descriptor.getPropertyType() != null) {
			if (descriptor.getPropertyType().isEnum()) {
				pe = new EnumEditor((Class<? extends Enum<?>>) descriptor.getPropertyType());
			}
			}
		}
		return pe;
	}
	
	private Component editorComponent(PropertyEditor pe, PropertyDescriptor pd) {
		if(pe==null) return null;
		final boolean editable=pd.getWriteMethod()!=null;
		Component c;
		if(pe.supportsCustomEditor() && editable) {
			c=pe.getCustomEditor();
			if(c instanceof JComponent) {
				((JComponent)c).setEnabled(editable);
			}
		} else {
			String[] tags=pe.getTags();
			if(tags!=null) c=choices(pe, tags, editable);
			else c=textual(pe, editable);
		}
		String sd = pd.getShortDescription();
	    if(sd!=null&&!sd.equals(pd.getDisplayName())&&
	      (c instanceof JComponent))
	      ((JComponent)c).setToolTipText(sd);
		return c;
	}
	
	private Component textual(final PropertyEditor pe, boolean editable) {
		final JTextField tf=new JTextField();
		pe.addPropertyChangeListener(new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				Object o=evt.getNewValue();
				if(o==null) o=pe.getValue();
				tf.setText(o==null? "": o.toString());
			}
		});
		tf.setEditable(editable);
		if(editable) tf.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				pe.setAsText(tf.getText());
			}
		});
		return tf;
	}
	
	private Component choices(final PropertyEditor pe, String[] tags, boolean editable) {
		final JComboBox cb=new JComboBox(tags);
		pe.addPropertyChangeListener(new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				String o=pe.getAsText();
				cb.setSelectedItem(o==null? "": o);
			}
		});
		cb.setEditable(false);
		cb.setEnabled(editable);
		if(editable) cb.getModel().addListDataListener(
				new ListDataListener() {
					public void contentsChanged(ListDataEvent e) {
						if(Math.min(e.getIndex0(), e.getIndex1())<0) {
							pe.setAsText((String)cb.getSelectedItem());
						}
					}
					public void intervalAdded(ListDataEvent e) {}      
					public void intervalRemoved(ListDataEvent e) {}
				});
		return cb;
	}	
}