/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.font.TextLayout;

import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class ColorDialog extends AbstractDialog implements ChangeListener {
	private static final long serialVersionUID = 6560372785543938761L;
	private JSlider slider;
	private boolean updating = false;
	private JColorChooser colorChooser;
	private static ColorDialog instance = null;
	private AlphaPreviewPanel previewPanel;
	
	private ColorDialog(String title, Color color) {
		super();
		setTitle(title);
		JPanel content = new JPanel(new BorderLayout());
		colorChooser = color != null ? new JColorChooser(color) : new JColorChooser();
		previewPanel = new AlphaPreviewPanel(color);
		colorChooser.getSelectionModel().addChangeListener(this);
		JPanel panel = new JPanel();
		panel.add(previewPanel);
		panel.setBorder(new EmptyBorder(0,0,5,0));
		colorChooser.setPreviewPanel(panel);
		content.add(colorChooser, BorderLayout.CENTER);
		slider = new JSlider(JSlider.HORIZONTAL, 0, 255, color != null ? color.getAlpha() : 255);
		slider.setMajorTickSpacing(85);
		slider.setMinorTickSpacing(17);
		slider.setPaintTicks(true);
		slider.setPaintLabels(true);
		slider.setBorder(BorderFactory.createTitledBorder("Alpha"));
		slider.addChangeListener(this);
		content.add(slider, BorderLayout.SOUTH);
		setMainComponent(content);
		pack();
	}
	
	static class AlphaPreviewPanel extends JButton {
		private static final long serialVersionUID = -7614992473405869635L;
		private static final String text="Black";
		private static final Font font=new Font("Sans Serif", Font.PLAIN, 30);
		
		private Color color;
		
		AlphaPreviewPanel(Color color) {
			setEnabled(false);
			setPreferredSize(new Dimension(350,50));
			setIcon(colorIcon);
			this.color=color;
		}
		
		private Icon colorIcon = new Icon() {
			
			public void paintIcon(Component cmp, Graphics g, int x, int y) {
				Graphics2D g2d = (Graphics2D) g;
				g.setColor(Color.white);
				g.fillRect(
						x,
						y,
						AlphaPreviewPanel.this.getWidth(),
						AlphaPreviewPanel.this.getHeight()
				);
				if (color != null) {
					g.setColor(Color.black);
					TextLayout tl = new TextLayout(
							text,font,
							g2d.getFontRenderContext()
					);
					Rectangle r = tl.getBounds().getBounds();
					
					int height = r.height;
					int width = r.width;
					
					g2d.setFont(font);
					
					int w=(getIconWidth()-width)/2;
					int h=(getIconHeight()-height)/2;
					g2d.drawString(text,x+w,y+height+h);
					
					g.setColor(color);
					g.fillRect(x, y, getIconWidth(), getIconHeight());
				}
				g.setColor(Color.black);
				g.drawRect(x, y, getIconWidth(), getIconHeight());
			}
			
			public int getIconWidth() {
				return AlphaPreviewPanel.this.getWidth();
			}
			
			public int getIconHeight() {
				return AlphaPreviewPanel.this.getHeight();
			}
		};

		public Color getColor() {
			return color;
		}

		public void setColor(Color color) {
			this.color = color;
			repaint();
		}
	}
	
	public static ColorDialog sharedInstance() {
		if( instance == null ) {
			instance = new ColorDialog("Choose Color",Color.RED);
		}
		return instance; 
	}
	
	public void setValue(Object o) {
		updating = true;
		Color color = o instanceof Color ? (Color)o : Color.red;
		colorChooser.setColor(color);
		previewPanel.setColor(color);
		slider.setValue( color.getAlpha() );
		updating = false;
	}
	
	public Color getValue() {
		Color chooser_color = colorChooser.getColor();
		return new Color(
				chooser_color.getRed(),
				chooser_color.getGreen(),
				chooser_color.getBlue(),
				slider.getValue()
		);
	}
	
	/**
	 * Handle change events from colorchooser or slider
	 */
	public void stateChanged(ChangeEvent e) {
		if (!updating) {
			previewPanel.setColor( getValue() );
			previewPanel.setColor( getValue() );
			fireStateChanged();
		}
	}
}

