#ifndef POAC_DATA_LOCKFILE_HPP_
#define POAC_DATA_LOCKFILE_HPP_

// std
#include <fstream>
#include <utility>

// external
#include <toml.hpp>

// internal
#include "poac/core/resolver/resolve.hpp"
#include "poac/data/manifest.hpp"
#include "poac/poac.hpp"

namespace poac::data::lockfile {

inline constexpr StringRef lockfile_name = "poac.lock";
inline constexpr StringRef lockfile_header =
    " This file is automatically generated by Poac.\n"
    "# It is not intended for manual editing.";

using InvalidLockfileVersion = Error<"invalid lockfile version found: {}", i64>;
using FailedToReadLockfile = Error<"failed to read lockfile:\n{}", String>;

inline fs::file_time_type
poac_lock_last_modified(const Path& base_dir) {
  return fs::last_write_time(base_dir / lockfile_name);
}

inline bool
is_outdated(const Path& base_dir) {
  if (!fs::exists(base_dir / lockfile_name)) {
    return true;
  }
  return poac_lock_last_modified(base_dir)
         < manifest::poac_toml_last_modified(base_dir);
}

} // namespace poac::data::lockfile

namespace poac::data::lockfile::inline v1 {

namespace resolver = core::resolver::resolve;

inline constexpr i64 lockfile_version = 1;

struct Package {
  String name;
  String version;
  Vec<String> dependencies;
};

struct Lockfile {
  i64 version = lockfile_version;
  Vec<Package> package;
};

// clang-format off
// to avoid reporting errors with inline namespace on only the dry-run mode. (IDK why)
} // namespace poac::data::lockfile::inline v1
// clang-format on

TOML11_DEFINE_CONVERSION_NON_INTRUSIVE(
    poac::data::lockfile::v1::Package, name, version, dependencies
)
TOML11_DEFINE_CONVERSION_NON_INTRUSIVE(
    poac::data::lockfile::v1::Lockfile, version, package
)

namespace poac::data::lockfile::inline v1 {

// -------------------- INTO LOCKFILE --------------------

[[nodiscard]] Result<toml::basic_value<toml::preserve_comments>>
convert_to_lock(const resolver::UniqDeps<resolver::WithDeps>& deps);

[[nodiscard]] Result<void>
overwrite(const resolver::UniqDeps<resolver::WithDeps>& deps);

[[nodiscard]] Result<void>
generate(const resolver::UniqDeps<resolver::WithDeps>& deps);

// -------------------- FROM LOCKFILE --------------------

[[nodiscard]] resolver::UniqDeps<resolver::WithDeps>
convert_to_deps(const Lockfile& lock);

[[nodiscard]] Result<Option<resolver::UniqDeps<resolver::WithDeps>>>
read(const Path& base_dir);

// clang-format off
// to avoid reporting errors with inline namespace on only the dry-run mode. (IDK why)
} // namespace poac::data::lockfile::inline v1
// clang-format on

#endif // POAC_DATA_LOCKFILE_HPP_
