//
//  MyDocument.m
//  Pixen
//
//  Created by Joe Osborn on Thu Sep 11 2003.
//  Copyright (c) 2003 Open Sword Group. All rights reserved.
//

#import "MyDocument.h"
#import "PXCanvasController.h"
#import "PXCanvas.h"
#import "PXPSDHandler.h"
#import "PXPalette.h"
#import "PXCanvasView.h"
#import "PXCanvasPrintView.h"
#import "gif_lib.h"
#import "PXGifExporter.h"

NSString * PXDocumentOpened = @"PXDocumentOpenedNotificationName";
NSString * PXDocumentClosed = @"PXDocumentClosedNotificationName";

@implementation MyDocument

- (BOOL)rescheduleAutosave
{
	NSTimeInterval repeatTime = [[NSUserDefaults standardUserDefaults] floatForKey:@"PXAutosaveInterval"];
	if (repeatTime == 0.0f) {
		[[NSUserDefaults standardUserDefaults] setFloat:180.0 forKey:@"PXAutosaveInterval"];
		repeatTime = 180.0f;
	}
	if (repeatTime < 0) {
		return NO;
	}
	
	[autosaveTimer release];
	autosaveTimer = [[NSTimer scheduledTimerWithTimeInterval:repeatTime target:self selector:@selector(autosave:) userInfo:nil repeats:NO] retain];
	return YES;
}

- (id)init
{
    [super init];
	canvas = [[PXCanvas alloc] init];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(cleanUpAutosaveFile:) name:NSApplicationWillTerminateNotification object:[NSApplication sharedApplication]];
 	[self autosave:nil];
    return self;
}

- (void)changeDirtyFileFromFilename:(NSString *)from toFilename:(NSString *)to
{
	NSMutableArray *dirtyFiles = [[[[NSUserDefaults standardUserDefaults] objectForKey:@"PXDirtyFiles"] mutableCopy] autorelease];
	
	if (dirtyFiles == nil) {
		dirtyFiles = [NSMutableArray arrayWithCapacity:8];
	}
	
	if (from != nil) {
		[dirtyFiles removeObject:from];
	}
	
	if (to != nil) {
		[dirtyFiles addObject:to];
	}
	
	[[NSUserDefaults standardUserDefaults] setObject:dirtyFiles forKey:@"PXDirtyFiles"];
	[[NSUserDefaults standardUserDefaults] synchronize];
}

- (void)cleanUpAutosaveFile:(NSNotification *)aNotification
{
	if (autosaveFilename != nil && ![[NSFileManager defaultManager] removeFileAtPath:autosaveFilename handler:nil]) {
		NSLog(@"Could not delete backup file \"%@\"", autosaveFilename);
	}
	[self changeDirtyFileFromFilename:autosaveFilename toFilename:nil]; // this will remove our dirty autosave file from the user defaults, so it doesn't complain on the next start of Pixen and freak out the user
}


- (void)updateAutosaveFilename
{
	NSString *oldFilename = autosaveFilename; // Don't worry, we release this at the end of the method
	
	if ([self fileName] != nil) {
		autosaveFilename = [[[[[self fileName] stringByDeletingPathExtension] stringByAppendingString:@"~"] stringByAppendingPathExtension:@"pxi"] retain];
	} else {
		autosaveFilename = [@"/tmp/PixenAutosave.pxi" retain];
	}
	if (![oldFilename isEqualToString:autosaveFilename]) {
		[self changeDirtyFileFromFilename:oldFilename toFilename:autosaveFilename];
	}
	
	[oldFilename release];
}

- (void)setFileName:(NSString *)path
{
	[super setFileName:path];
	[self updateAutosaveFilename];
}

- (void)autosave:(NSTimer *)timer
{
	if (![self rescheduleAutosave]) {
		return;
	}
	if (autosaveFilename == nil) {
		[self updateAutosaveFilename];
	}
	[[NSKeyedArchiver archivedDataWithRootObject:canvas] writeToFile:autosaveFilename atomically:YES];
}

- (void)dealloc
{
	[autosaveFilename release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[autosaveTimer invalidate];
	[autosaveTimer release];
    [[self windowControllers] makeObjectsPerformSelector:@selector(close)];
    [canvasController release];
    [canvas release];
    [super dealloc];
}

- (void)makeWindowControllers
{
    // Override returning the nib file name of the document
    // If you need to use a subclass of NSWindowController or if your document supports multiple NSWindowControllers, you should remove this method and override -makeWindowControllers instead.
    canvasController = [[PXCanvasController alloc] init];
    [canvasController setCanvas:canvas];
    [self addWindowController:canvasController];
    [canvasController window];
	[[NSNotificationCenter defaultCenter] postNotificationName:PXDocumentOpened object:self];
}

- (void)windowControllerDidLoadNib:aController
{
    [super windowControllerDidLoadNib:aController];
    // Add any code here that needs to be executed once the windowController has loaded the document's window.
}

- (void)close
{
	[[NSNotificationCenter defaultCenter] postNotificationName:PXDocumentClosed object:self];
	[autosaveTimer invalidate];
	[super close];
}

BOOL isPowerOfTwo(int num)
{
	double logResult = log2(num);
	return (logResult == (int)logResult);
}

- (NSData *)dataRepresentationOfType:(NSString *)aType
{
    // Insert code here to write your document from the given data.  You can also choose to override -fileWrapperRepresentationOfType: or -writeToFile:ofType: instead.
    if([aType isEqualToString:@"Pixen Image"])
    {
        return [NSKeyedArchiver archivedDataWithRootObject:canvas];
    }
    if([aType isEqualToString:@"Portable Network Graphic (PNG)"])
    {
        return [canvas imageDataWithType:NSPNGFileType properties:nil];
    }
    if([aType isEqualToString:@"Tagged Image File Format (TIFF)"])
    {
        return [canvas imageDataWithType:NSTIFFFileType properties:nil];
    }
    if([aType isEqualToString:@"Compuserve Graphic (GIF)"])
	{		
		id image = [[NSImage alloc] initWithSize:[canvas size]];
		[image lockFocus];
		[canvas drawRect:NSMakeRect(0,0,[canvas size].width,[canvas size].height) fixBug:YES];
		[image unlockFocus];
		return [PXGifExporter gifDataForImage:image];
	}
    if([aType isEqualToString:@"Windows Bitmap (BMP)"])
	{
		return [canvas imageDataWithType:NSBMPFileType properties:nil];
	}
	if([aType isEqualToString:@"Apple PICT Graphic"])
	{
		return [canvas PICTData];
	}
    if([aType isEqualToString:@"Encapsulated PostScript (EPS)"])
    {
        return [[(PXCanvasController *)canvasController view] dataWithEPSInsideRect:[[(PXCanvasController *)canvasController view] frame]];
    }

    return nil;
}

- (BOOL)checkSize:(NSSize)size
{
	if (size.width * size.height <= 256 * 256) {
		return YES;
	}
	return [[NSAlert alertWithMessageText:@"Large Image Warning" defaultButton:@"Yes" alternateButton:@"No" otherButton:nil informativeTextWithFormat:@"This image is %d by %d pixels in size, which is large enough that manipulation will be noticably slow.  Pixen is designed for images under 256 by 256 pixels.  Would you still like to open this image?", (int)size.width, (int)size.height] runModal] == NSOKButton;
}

- (BOOL)loadDataRepresentation:(NSData *)data ofType:(NSString *)aType
{
    // Insert code here to read your document from the given data.  You can also choose to override -loadFileWrapperRepresentation:ofType: or -readFromFile:ofType: instead.
    if([aType isEqualToString:@"Pixen Image"])
    {
		PXCanvas *tempCanvas = [NSKeyedUnarchiver unarchiveObjectWithData:data];
		if (![self checkSize:[tempCanvas size]]) {
			return NO;
		}
		[canvas release];
        canvas = [tempCanvas retain];
    }
/*	else if([aType isEqualToString:@"Photoshop Graphic (PSD)"])
	{
		canvas = [[PXCanvas alloc] initWithPSDData:data];
	} */
    else
    {
		NSImage *image = [[[NSImage alloc] initWithData:data] autorelease];
		if (![self checkSize:[image size]]) {
			return NO;
		}
		[canvas release];
        canvas = [[PXCanvas alloc] initWithImage:image];
    }
    if(canvas)
    {
        [canvasController setCanvas:canvas];
        return YES;
    }
    return NO;
}

- (void)setLayers:layers fromLayers:oldLayers
{
	[canvas setLayers:layers fromLayers:oldLayers];
	//[[[self undoManager] prepareWithInvocationTarget:self] setLayers:oldLayers fromLayers:layers];
	//[canvas setLayers:layers];
	//[canvas setSize:[[layers objectAtIndex:0] size]];
}

- (IBAction)cut:sender
{
	[[self undoManager] beginUndoGrouping];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];
	[self copy:sender];
	[self delete:sender];
	[[self undoManager] setActionName:@"Cut"];
	[[self undoManager] endUndoGrouping];
	[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (IBAction)copy:sender
{
	id board = [NSPasteboard generalPasteboard];	
	if(![[board types] containsObject:@"PXLayer"]) 
	{ 
		[board addTypes:[NSArray arrayWithObject:@"PXLayer"] owner:self]; 
	}
	[board declareTypes:[NSArray arrayWithObject:@"PXLayer"] owner:self];
	[board setData:[canvas selectionData] forType:@"PXLayer"];
}

- (IBAction)paste:sender
{
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Paste"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];
	[[[self undoManager] prepareWithInvocationTarget:canvasController] canvasSizeDidChange:nil];
	[[self undoManager] endUndoGrouping];
	id board = [NSPasteboard generalPasteboard];	
	if([[board types] containsObject:@"PXLayer"])
	{	
		[canvas pasteFromPasteboard:board type:@"PXLayer"];
	}
	id enumerator = [[NSImage imagePasteboardTypes] objectEnumerator], current;
	while (current = [enumerator nextObject])
	{
		if ([[board types] containsObject:current])
		{
			[canvas pasteFromPasteboard:board type:@"NSImage"];
		}
	}
	[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (IBAction)delete:sender
{
	if (![canvas hasSelection]) { return; }
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Delete"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];
	[[self undoManager] endUndoGrouping];
	[canvas deleteSelection];
	[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (IBAction)selectAll:sender
{
	[canvas selectAll];
	[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (IBAction)selectNone:sender
{
	[[self undoManager] beginUndoGrouping];
	[[self undoManager] setActionName:@"Deselect"];
	[self setLayers:[[canvas layers] deepMutableCopy] fromLayers:[canvas layers]];
	[[self undoManager] endUndoGrouping];
	[canvas deselect];
	[canvas changedInRect:NSMakeRect(0, 0, [canvas size].width, [canvas size].height)];
}

- (void)printShowingPrintPanel:(BOOL)showPanels 
{
	if(printableView == nil) { printableView = [[PXCanvasPrintView viewForCanvas:[self canvas]] retain]; }
	
	float scale = [[[[self printInfo] dictionary] objectForKey:NSPrintScalingFactor] floatValue];
	id transform = [NSAffineTransform transform];
	[transform scaleXBy:scale yBy:scale];
	[printableView setBoundsOrigin:[transform transformPoint:[printableView frame].origin]];
	[printableView setBoundsSize:[transform transformSize:[printableView frame].size]];
	
    NSPrintOperation *op = [NSPrintOperation printOperationWithView:printableView printInfo:[self printInfo]];
    [op setShowPanels:showPanels];
	
	[self runModalPrintOperation:op delegate:nil didRunSelector:NULL contextInfo:NULL];
}

- canvas
{
	return canvas;
}

@end
