/*
 * Layer-3 of MIDI support
 * 
 * (C) Christoph van Wullen, DL1YCF
 *
 *
 * In most cases, a certain action only makes sense for a specific
 * type. For example, changing the VFO frequency will only be implemeted
 * for MIDI_WHEEL, and TUNE off/on only with MIDI_KNOB.
 *
 * However, changing the volume makes sense both with MIDI_KNOB and MIDI_WHEEL.
 */
#include <gtk/gtk.h>

#include "radio.h"
#include "vfo.h"
#include "filter.h"
#include "band.h"
#include "mode.h"
#include "new_menu.h"
#include "sliders.h"
#include "ext.h"
#include "agc.h"
#include "midi.h"
#include "store.h"
#ifdef LOCALCW
#include "iambic.h"
#endif
#include "wdsp.h"
#include "diversity_menu.h"
#include "radio_menu.h"
#include "toolbar.h"
#include "noise_menu.h"
#include "zoompan.h"

//
// Virtually all MIDI actions at the end lead to functions that have
// to be invoked by GTK. Instead of having dozens of g_idle_add() in the
// code below, one can queue the "big switch statement" into the GTK
// idle queue and exectue all GUI functions directly.
//
// However, this is not wanted for CWKEYER, CWLEFT and CWRIGHT since
// these have to be processed with minimal delay (and do not call GUI functions).
//
// Therefore, these three cases are already handled in the MIDI callback
// function and everything else is put into the GTK queue
//

//
// This is the data struct generated by processing the MIDI event
// it is passed to the function DoTheMidi
//
struct _MIDIcmd {
   enum MIDIaction action;
   enum MIDItype type;
   int val;
};

typedef struct _MIDIcmd MIDIcmd;

static int DoTheRestOfTheMIDI(void *data);

void DoTheMidi(enum MIDIaction action, enum MIDItype type, int val) {
    if (action == MIDI_ACTION_CWKEYER) {
          //
          // This is a CW key-up/down which uses functions from the keyer
          // that by-pass the interrupt-driven standard action.
          // It is intended to
          // be used with external keyers that send key-up/down messages via
          // MIDI using this command.
          //
          // NO BREAK-IN! The keyer has to take care of sending MIDI PTT
          // on/off messages at appropriate times, or the operator has to
          // manually engage/disengage PTT
          //
          // Since this if for immediate key-down, it does not rely on LOCALCW
          //
          if (val != 0 && cw_keyer_internal == 0) {
            cw_key_down=960000;  // max. 20 sec to protect hardware
            cw_key_up=0;
            cw_key_hit=1;
          } else {
            cw_key_down=0;
            cw_key_up=0;
          }
          return;
    }
#ifdef LOCALCW
    if (action == MIDI_ACTION_CWLEFT) {
         keyer_event(1, val);
         return;
    }
    if (action == MIDI_ACTION_CWRIGHT) {
         keyer_event(0, val);
         return;
    }
#endif
    //
    // For all other MIDI actions, the parser is put into the idle queu
    //
    MIDIcmd *cmd=malloc(sizeof(MIDIcmd));
    cmd->action = action;
    cmd->type   = type;
    cmd->val    = val;
    g_idle_add(DoTheRestOfTheMIDI, cmd);
    return;
}

int DoTheRestOfTheMIDI(void *data) {
    MIDIcmd *cmd = (MIDIcmd *)data;
   
    int new;
    int id, c25;
    double dnew;

    enum MIDIaction action = cmd->action;
    enum MIDItype   type   = cmd->type;
    int             val    = cmd->val;

    free(data);

    //
    // Handle cases in alphabetical order of the key words in midi.props
    //
    switch (action) {
	/////////////////////////////////////////////////////////// "A2B"
	case MIDI_ACTION_VFO_A2B: // only key supported
            vfo_a_to_b();
	    break;
	/////////////////////////////////////////////////////////// "AFGAIN"
	case MIDI_ACTION_AF_GAIN: // knob or wheel supported
            switch (type) {
	      case MIDI_KNOB:
		active_receiver->volume = 0.01*val;
		break;
	      case MIDI_WHEEL:	
		dnew=active_receiver->volume += 0.01*val;
		if (dnew < 0.0) dnew=0.0; if (dnew > 1.0) dnew=1.0;
		active_receiver->volume = dnew;
		break;
	      default:
		// do not change volume
		// we should not come here anyway
		break;
	    }
            update_af_gain();
	    break;
	/////////////////////////////////////////////////////////// "AGCATTACK"
	case MIDI_ACTION_AGCATTACK: // only key supported
	    // cycle through fast/med/slow AGC attack
	    new=active_receiver->agc + 1;
	    if (new > AGC_FAST) new=0;
	    active_receiver->agc=new;
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "AGCVAL"
	case MIDI_ACTION_AGC: // knob or wheel supported
	    switch (type) {
	      case MIDI_KNOB:
		dnew = -20.0 + 1.4*val;
		break;
	      case MIDI_WHEEL:
		dnew=active_receiver->agc_gain + val;
		if (dnew < -20.0) dnew = -20.0;
                if (dnew > 120.0) dnew = 120.0;
		break;
	      default:
		// do not change value
		// we should not come here anyway
		dnew=active_receiver->agc_gain;
		break;
	    }
            set_agc_gain(active_receiver->id, dnew);
	    break;
	/////////////////////////////////////////////////////////// "ANF"
	case MIDI_ACTION_ANF:	// only key supported
            if (active_receiver->anf==0) {
              active_receiver->anf=1;
              mode_settings[vfo[active_receiver->id].mode].anf=1;
            } else {
              active_receiver->snb=0;
              mode_settings[vfo[active_receiver->id].mode].anf=0;
            }
            SetRXAANFRun(active_receiver->id, active_receiver->anf);
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "ATT"
	case MIDI_ACTION_ATT:	// Key for ALEX attenuator, wheel or knob for slider
	    switch(type) {
	      case MIDI_KEY:
		if (filter_board == ALEX && active_receiver->adc == 0) {
		  new=active_receiver->alex_attenuation + 1;
		  if (new > 3) new=0;
                  set_alex_attenuation(new);
                  update_att_preamp();
		}
		break;
	      case MIDI_WHEEL:
		dnew=adc[active_receiver->adc].attenuation + val;
                if(have_rx_gain) {
                  if(dnew < -12.0) {
                    dnew = -12.0;
                  } else if(dnew > 48.0) {
                    dnew=48.0;
                  }
                } else {
                  if (dnew < 0.0) {
                    dnew = 0.0;
                  } else if (dnew > 31.0) {
                    dnew = 31.0;
                  }
                }
                set_attenuation_value(dnew);
		break;
	      case MIDI_KNOB:
                if (have_rx_gain) {
		  dnew = -12.0 + 0.6*val;
                } else {
                  dnew  = 0.31*val;
                }
                set_attenuation_value(dnew);
		break;
	      default:
		// do nothing
		// we should not come here anyway
		break;
	    }
	    break;
	/////////////////////////////////////////////////////////// "B2A"
	case MIDI_ACTION_VFO_B2A: // only key supported
            vfo_b_to_a();
	    break;
        /////////////////////////////////////////////////////////// "BANDxxx"
        case MIDI_ACTION_BAND_10:
            vfo_band_changed(active_receiver->id, band10);
            break;
        case MIDI_ACTION_BAND_12:
            vfo_band_changed(active_receiver->id, band12);
            break;
#ifdef SOAPYSDR
        case MIDI_ACTION_BAND_1240:
            vfo_band_changed(active_receiver->id, band1240);
            break;
        case MIDI_ACTION_BAND_144:
            vfo_band_changed(active_receiver->id, band144);
            break;
#endif
        case MIDI_ACTION_BAND_15:
            vfo_band_changed(active_receiver->id, band15);
            break;
        case MIDI_ACTION_BAND_160:
            vfo_band_changed(active_receiver->id, band160);
            break;
        case MIDI_ACTION_BAND_17:
            vfo_band_changed(active_receiver->id, band17);
            break;
        case MIDI_ACTION_BAND_20:
            vfo_band_changed(active_receiver->id, band20);
            break;
#ifdef SOAPYSDR
        case MIDI_ACTION_BAND_220:
            vfo_band_changed(active_receiver->id, band220);
            break;
        case MIDI_ACTION_BAND_2300:
            vfo_band_changed(active_receiver->id, band2300);
            break;
#endif
        case MIDI_ACTION_BAND_30:
            vfo_band_changed(active_receiver->id, band30);
            break;
#ifdef SOAPYSDR
        case MIDI_ACTION_BAND_3400:
            vfo_band_changed(active_receiver->id, band3400);
            break;
#endif
        case MIDI_ACTION_BAND_40:
            vfo_band_changed(active_receiver->id, band40);
            break;
#ifdef SOAPYSDR
        case MIDI_ACTION_BAND_430:
            vfo_band_changed(active_receiver->id, band430);
            break;
#endif
        case MIDI_ACTION_BAND_6:
            vfo_band_changed(active_receiver->id, band6);
            break;
        case MIDI_ACTION_BAND_60:
            vfo_band_changed(active_receiver->id, band60);
            break;
#ifdef SOAPYSDR
        case MIDI_ACTION_BAND_70:
            vfo_band_changed(active_receiver->id, band70);
            break;
#endif
        case MIDI_ACTION_BAND_80:
            vfo_band_changed(active_receiver->id, band80);
            break;
#ifdef SOAPYSDR
        case MIDI_ACTION_BAND_902:
            vfo_band_changed(active_receiver->id, band902);
            break;
        case MIDI_ACTION_BAND_AIR:
            vfo_band_changed(active_receiver->id, bandAIR);
            break;
#endif
        case MIDI_ACTION_BAND_GEN:
            vfo_band_changed(active_receiver->id, bandGen);
            break;
        case MIDI_ACTION_BAND_WWV:
            vfo_band_changed(active_receiver->id, bandWWV);
            break;
	/////////////////////////////////////////////////////////// "BANDDOWN"
	/////////////////////////////////////////////////////////// "BANDUP"
	case MIDI_ACTION_BAND_DOWN:
	case MIDI_ACTION_BAND_UP:
	    switch (type) {
	      case MIDI_KEY:
		new=(action == MIDI_ACTION_BAND_UP) ? 1 : -1;
		break;
	      case MIDI_WHEEL:
		new=val > 0 ? 1 : -1;
		break;
	      case MIDI_KNOB:
		// cycle through the bands
		new = ((BANDS-1) * val) / 100 - vfo[active_receiver->id].band;
		break;
	      default:
		// do not change
		// we should not come here anyway
		new=0;
		break;
	    }
	    //
	    // If the band has not changed, do nothing. Otherwise
	    // vfo.c will loop through the band stacks
	    //
	    if (new != 0) {
	      new+=vfo[active_receiver->id].band;
	      if (new >= BANDS) new=0;
	      if (new < 0) new=BANDS-1;
              vfo_band_changed(active_receiver->id, new);
	    }
	    break;
	/////////////////////////////////////////////////////////// "COMPRESS"
	case MIDI_ACTION_COMPRESS: // wheel or knob
	    switch (type) {
	      case MIDI_WHEEL:
		dnew=transmitter->compressor_level + val;
		if (dnew > 20.0) dnew=20.0;
		if (dnew < 0 ) dnew=0;
		break;
	      case MIDI_KNOB:
		dnew=(20.0*val)/100.0;
		break;
	      default:
		// do not change
		// we should not come here anyway
		dnew=transmitter->compressor_level;
		break;
	    }
            if (can_transmit) {
	      transmitter->compressor_level=dnew;
	      // automatically engange compressor if level > 0.5
	      if (dnew < 0.5) transmitter->compressor=0;
	      if (dnew > 0.5) transmitter->compressor=1;
              set_compression(transmitter);
            }
	    break;
	/////////////////////////////////////////////////////////// "CTUN"
	case MIDI_ACTION_CTUN: // only key supported
	    // toggle CTUN
            id=active_receiver->id;
            vfo[id].ctun=vfo[id].ctun==1?0:1;
            if(!vfo[id].ctun) {
              vfo[id].offset=0;
            }
            vfo[id].ctun_frequency=vfo[id].frequency;
            set_offset(receiver[id],vfo[id].offset);
	    break;
	/////////////////////////////////////////////////////////// "CURRVFO"
	case MIDI_ACTION_VFO: // only wheel supported
            vfo_step(val);
	    break;
	/////////////////////////////////////////////////////////// "CWSPEED"
	case MIDI_ACTION_CWSPEED: // knob or wheel
            switch (type) {
              case MIDI_KNOB:
		// speed between 5 and 35 wpm
                new= (int) (5.0 + (double) val * 0.3);
                break;
              case MIDI_WHEEL:
		// here we allow from 1 to 60 wpm
                new = cw_keyer_speed + val;
		if (new <  1) new=1;
		if (new > 60) new=60;
                break;
              default:
                // do not change
                // we should not come here anyway
                new = cw_keyer_speed;
                break;
            }
	    cw_keyer_speed=new;
#ifdef LOCALCW
	    keyer_update();
#endif
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "DIVCOARSEGAIN"
	case MIDI_ACTION_DIV_COARSEGAIN:  // knob or wheel supported
	case MIDI_ACTION_DIV_FINEGAIN:    // knob or wheel supported
	case MIDI_ACTION_DIV_GAIN:        // knob or wheel supported
            switch (type) {
              case MIDI_KNOB:
                if (action == MIDI_ACTION_DIV_COARSEGAIN || action == MIDI_ACTION_DIV_GAIN) {
		  // -25 to +25 dB in steps of 0.5 dB
		  dnew = 10.0*(-25.0 + 0.5*val - div_gain);
		} else {
		  // round gain to a multiple of 0.5 dB and apply a +/- 0.5 dB update
                  new = (int) (2*div_gain + 1.0) / 2;
		  dnew = 10.0*((double) new + 0.01*val - 0.5 - div_gain);
		}
                break;
              case MIDI_WHEEL:
                // coarse: increaments in steps of 0.25 dB, medium: steps of 0.1 dB fine: in steps of 0.01 dB
                if (action == MIDI_ACTION_DIV_GAIN) {
		  dnew = val*0.5;
		} else if (action == MIDI_ACTION_DIV_COARSEGAIN) {
		  dnew = val*2.5;
		} else {
		  dnew = val * 0.1;
	 	}
                break;
              default:
                // do not change
                // we should not come here anyway
		dnew = 0.0;
                break;
            }
	    // dnew is the delta times 10
            update_diversity_gain(dnew);
            break;
        /////////////////////////////////////////////////////////// "DIVPHASE"
        case MIDI_ACTION_DIV_COARSEPHASE:   // knob or wheel supported
        case MIDI_ACTION_DIV_FINEPHASE:     // knob or wheel supported
	case MIDI_ACTION_DIV_PHASE:		// knob or wheel supported
            dnew=0.0;
            switch (type) {
              case MIDI_KNOB:
		// coarse: change phase from -180 to 180
                // fine: change from -5 to 5
                if (action == MIDI_ACTION_DIV_COARSEPHASE || action == MIDI_ACTION_DIV_PHASE) {
		  // coarse: change phase from -180 to 180 in steps of 3.6 deg
                  dnew = (-180.0 + 3.6*val - div_phase);
                } else {
		  // fine: round to multiple of 5 deg and apply a +/- 5 deg update
                  new = 5 * ((int) (div_phase+0.5) / 5);
                  dnew =  (double) new + 0.1*val -5.0 -div_phase;
                }
                break;
              case MIDI_WHEEL:
		if (action == MIDI_ACTION_DIV_PHASE) {
		  dnew = val*0.5; 
		} else if (action == MIDI_ACTION_DIV_COARSEPHASE) {
		  dnew = val*2.5;
		} else if (action == MIDI_ACTION_DIV_FINEPHASE) {
		  dnew = 0.1*val;
		}
                break;
              default:
                // do nothing
                break;
            }
            // dnew is the delta
            update_diversity_phase(dnew);
            break;
        /////////////////////////////////////////////////////////// "DIVTOGGLE"
        case MIDI_ACTION_DIV_TOGGLE:   // only key supported
            // enable/disable DIVERSITY
            diversity_enabled = diversity_enabled ? 0 : 1;
            vfo_update();
            break;
	/////////////////////////////////////////////////////////// "DUP"
        case MIDI_ACTION_DUP:
	    if (can_transmit && !isTransmitting()) {
	      duplex=duplex==1?0:1;
              setDuplex();
	    }
            break;
	/////////////////////////////////////////////////////////// "FILTERDOWN"
	/////////////////////////////////////////////////////////// "FILTERUP"
	case MIDI_ACTION_FILTER_DOWN:
	case MIDI_ACTION_FILTER_UP:
	    //
	    // In filter.c, the filters are sorted such that the widest one comes first
	    // Therefore let MIDI_ACTION_FILTER_UP move down.
	    //
	    switch (type) {
	      case MIDI_KEY:
		new=(action == MIDI_ACTION_FILTER_UP) ? -1 : 1;
		break;
	      case MIDI_WHEEL:
		new=val > 0 ? -1 : 1;
		break;
	      case MIDI_KNOB:
		// cycle through all the filters: val=100 maps to filter #0
		new = ((FILTERS-1) * (val-100)) / 100 - vfo[active_receiver->id].filter;
		break;
	      default:
		// do not change filter setting
		// we should not come here anyway
		new=0;
		break;
	    }
	    if (new != 0) {
	      new+=vfo[active_receiver->id].filter;
	      if (new >= FILTERS) new=0;
	      if (new <0) new=FILTERS-1;
              vfo_filter_changed(new);
	    }
	    break;
        /////////////////////////////////////////////////////////// "MENU_FILTER"
        case MIDI_ACTION_MENU_FILTER:
            start_filter();
            break;
        /////////////////////////////////////////////////////////// "MENU_MODE"
        case MIDI_ACTION_MENU_MODE:
            start_mode();
            break;
	/////////////////////////////////////////////////////////// "LOCK"
	case MIDI_ACTION_LOCK: // only key supported
	    locked=!locked;
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "MICGAIN"
	case MIDI_ACTION_MIC_VOLUME: // knob or wheel supported
	    // TODO: possibly adjust linein value if that is effective
	    switch (type) {
	      case MIDI_KNOB:
		dnew = -10.0 + 0.6*val;
		break;
	      case MIDI_WHEEL:
		dnew = mic_gain + val;
		if (dnew < -10.0) dnew = -10.0;
                if (dnew >  50.0) dnew =  50.0;
		break;
	      default:
		// do not change mic gain
		// we should not come here anyway
		dnew = mic_gain;
		break;
	    }
            set_mic_gain(dnew);
	    break;
	/////////////////////////////////////////////////////////// "MODEDOWN"
	/////////////////////////////////////////////////////////// "MODEUP"
	case MIDI_ACTION_MODE_DOWN:
	case MIDI_ACTION_MODE_UP:
	    switch (type) {
	      case MIDI_KEY:
		new=(action == MIDI_ACTION_MODE_UP) ? 1 : -1;
		break;
	      case MIDI_WHEEL:
		new=val > 0 ? 1 : -1;
		break;
	      case MIDI_KNOB:
		// cycle through all the modes
		new = ((MODES-1) * val) / 100 - vfo[active_receiver->id].mode;
		break;
	      default:
		// do not change
		// we should not come here anyway
		new=0;
		break;
	    }
	    if (new != 0) {
	      new+=vfo[active_receiver->id].mode;
	      if (new >= MODES) new=0;
	      if (new <0) new=MODES-1;
              vfo_mode_changed(new);
	    }
	    break;
	/////////////////////////////////////////////////////////// "MOX"
	case MIDI_ACTION_MOX: // only key supported
	    // Note this toggles the PTT state without knowing the
            // actual state. See MIDI_ACTION_PTTKEYER for actually
            // *setting* PTT
	    if (can_transmit) {
	        new = !mox;
                mox_update(new);
	    }
	    break;    
        /////////////////////////////////////////////////////////// "MUTE"
        case MIDI_ACTION_MUTE:
            active_receiver->mute_radio=!active_receiver->mute_radio;
            break;
	/////////////////////////////////////////////////////////// "NOISEBLANKER"
	case MIDI_ACTION_NB: // only key supported
	    // cycle through NoiseBlanker settings: OFF, NB, NB2
	    if (active_receiver->nb) {
	      active_receiver->nb = 0;
	      active_receiver->nb2= 1;
	    } else if (active_receiver->nb2) {
	      active_receiver->nb = 0;
	      active_receiver->nb2= 0;
	    } else {
	      active_receiver->nb = 1;
	      active_receiver->nb2= 0;
	    }
	    update_noise();
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "NOISEREDUCTION"
	case MIDI_ACTION_NR: // only key supported
	    // cycle through NoiseReduction settings: OFF, NR1, NR2
	    if (active_receiver->nr) {
	      active_receiver->nr = 0;
	      active_receiver->nr2= 1;
	    } else if (active_receiver->nr2) {
	      active_receiver->nr = 0;
	      active_receiver->nr2= 0;
	    } else {
	      active_receiver->nr = 1;
	      active_receiver->nr2= 0;
	    }
            update_noise();
            vfo_update();
	    break;
        /////////////////////////////////////////////////////////// "NUMPADxx"
        case MIDI_ACTION_NUMPAD_0:
            num_pad(0);
            break;
        case MIDI_ACTION_NUMPAD_1:
            num_pad(1);
            break;
        case MIDI_ACTION_NUMPAD_2:
            num_pad(2);
            break;
        case MIDI_ACTION_NUMPAD_3:
            num_pad(3);
            break;
        case MIDI_ACTION_NUMPAD_4:
            num_pad(4);
            break;
        case MIDI_ACTION_NUMPAD_5:
            num_pad(5);
            break;
        case MIDI_ACTION_NUMPAD_6:
            num_pad(6);
            break;
        case MIDI_ACTION_NUMPAD_7:
            num_pad(7);
            break;
        case MIDI_ACTION_NUMPAD_8:
            num_pad(9);
            break;
        case MIDI_ACTION_NUMPAD_9:
            num_pad(9);
            break;
        case MIDI_ACTION_NUMPAD_CL:
            num_pad(-1);
            break;
        case MIDI_ACTION_NUMPAD_ENTER:
            num_pad(-2);
            break;
	/////////////////////////////////////////////////////////// "PAN"
        case MIDI_ACTION_PAN:  // wheel and knob
	    switch (type) {
              case MIDI_WHEEL:
                update_pan(val*10.0);
                break;
	      case MIDI_KNOB:
                if(active_receiver->zoom>1) {
                  int pos=GPOINTER_TO_INT(data);
                  double pan=(double)((active_receiver->zoom-1)*active_receiver->width)*((double)pos/100.0);
                  set_pan(active_receiver->id,(double)pan);
                }
                break;
	      default:
		// no action for keys (we should not come here anyway)
		break;
            }
            break;
	/////////////////////////////////////////////////////////// "PANHIGH"
	case MIDI_ACTION_PAN_HIGH:  // wheel or knob
	    switch (type) {
	      case MIDI_WHEEL:
		if (mox) {
		    // TX panadapter affected
		    transmitter->panadapter_high += val;
		} else {
		    active_receiver->panadapter_high += val;
		}
		break;
	    case MIDI_KNOB:
		// Adjust "high water" in the range -50 ... 0 dBm
		new = -50 + val/2;
		if (mox) {
		    transmitter->panadapter_high = new;
		} else {
		    active_receiver->panadapter_high = new;
		}
		break;
	      default:
		// do nothing
		// we should not come here anyway
		break;
	    }
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "PANLOW"
	case MIDI_ACTION_PAN_LOW:  // wheel and knob
	    switch (type) {
	      case MIDI_WHEEL:
		if (isTransmitting()) {
		    // TX panadapter affected
		    transmitter->panadapter_low += val;
		} else {
		    active_receiver->panadapter_low += val;
		}
		break;
	      case MIDI_KNOB:
		if (isTransmitting()) {
		    // TX panadapter: use values -100 through -50
		    new = -100 + val/2;
		    transmitter->panadapter_low =new;
		} else {
		    // RX panadapter: use values -140 through -90
		    new = -140 + val/2;
		    active_receiver->panadapter_low = new;
		}
		break;
	      default:
		// do nothing
		// we should not come here anyway
		break;
	    }
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "PREAMP"
	case MIDI_ACTION_PRE:	// only key supported
	    //
	    // Normally on/off, but for CHARLY25, cycle through three
	    // possible states. Current HPSDR hardware does no have
	    // switch'able preamps.
	    //
	    c25= (filter_board == CHARLY25);
	    new = active_receiver->preamp + active_receiver->dither;
	    new++;
	    if (c25) {
	      if (new >2) new=0;
	    } else {
	      if (new >1) new=0;
	    }
	    switch (new) {
	      case 0:
		active_receiver->preamp=0;
		if (c25) active_receiver->dither=0;
		break;
	      case 1:
		active_receiver->preamp=1;
		if (c25) active_receiver->dither=0;
		break;
	      case 2:
		active_receiver->preamp=1;
		if (c25) active_receiver->dither=1;
		break;
	    }
            update_att_preamp();
	    break;
	/////////////////////////////////////////////////////////// "PTT(Keyer)"
        case MIDI_ACTION_PTTKEYER:  // key only
            // always use with "ONOFF"
	    if (can_transmit) {
                mox_update(val);
	    }
	    break;    
	/////////////////////////////////////////////////////////// "PURESIGNAL"
	case MIDI_ACTION_PS: // only key supported
#ifdef PURESIGNAL
	    // toggle PURESIGNAL
            if (can_transmit) {
	      new=!(transmitter->puresignal);
              tx_set_ps(transmitter, new);
            }
#endif
	    break;
	/////////////////////////////////////////////////////////// "RECALLMx"
	case MIDI_ACTION_MEM_RECALL_M0:
            recall_memory_slot(0);
            break;
	case MIDI_ACTION_MEM_RECALL_M1:
            recall_memory_slot(1);
            break;
	case MIDI_ACTION_MEM_RECALL_M2:
            recall_memory_slot(2);
            break;
	case MIDI_ACTION_MEM_RECALL_M3:
            recall_memory_slot(3);
            break;
	case MIDI_ACTION_MEM_RECALL_M4:
            recall_memory_slot(4);
            break;
	/////////////////////////////////////////////////////////// "RFGAIN"
        case MIDI_ACTION_RF_GAIN: // knob or wheel supported
            switch (type) {
              case MIDI_KNOB:
                dnew=val;
                break;
              case MIDI_WHEEL:
                dnew=adc[active_receiver->id].gain+val;
                break;
              default:
                // Arriving here means there is an error somewhere else
                dnew=0.0;
                break;
            }
            if (dnew <   0.0) dnew =   0.0;
            if (dnew > 100.0) dnew = 100.0;
            set_rf_gain(active_receiver->id, dnew);
	    break;
	/////////////////////////////////////////////////////////// "RFPOWER"
	case MIDI_ACTION_TX_DRIVE: // knob or wheel supported
	    switch (type) {
	      case MIDI_KNOB:
		dnew = val;
		break;
	      case MIDI_WHEEL:
		dnew=transmitter->drive + val;
		if (dnew < 0.0) dnew=0.0; if (dnew > 100.0) dnew=100.0;
		break;
	      default:
		// do not change value
		// we should not come here anyway
		dnew=transmitter->drive;
		break;
	    }
            set_drive(dnew);
	    break;
	/////////////////////////////////////////////////////////// "RITCLEAR"
	case MIDI_ACTION_RIT_CLEAR:	  // only key supported
	    // clear RIT value
	    vfo[active_receiver->id].rit = 0;
	    vfo[active_receiver->id].rit_enabled = 0;
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "RITSTEP"
        case MIDI_ACTION_RIT_STEP: // key or wheel supported
            // This cycles between RIT increments 1, 10, 100, 1, 10, 100, ...
            switch (type) {
              case MIDI_KEY:
                // key cycles through in upward direction
                val=1;
                /* FALLTHROUGH */
              case MIDI_WHEEL:
                // wheel cycles upward or downward
                if (val > 0) {
                  rit_increment=10*rit_increment;
                } else {
                  rit_increment=rit_increment/10;
                }
                if (rit_increment < 1) rit_increment=100;
                if (rit_increment > 100) rit_increment=1;
                break;
              default:
                // do nothing
                break;
            }
            vfo_update();
            break;
	/////////////////////////////////////////////////////////// "RITTOGGLE"
	case MIDI_ACTION_RIT_TOGGLE:  // only key supported
	    // enable/disable RIT
	    new=vfo[active_receiver->id].rit_enabled;
	    vfo[active_receiver->id].rit_enabled = new ? 0 : 1;
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "RITVAL"
	case MIDI_ACTION_RIT_VAL:	// wheel or knob
	    switch (type) {
	      case MIDI_WHEEL:
		// This changes the RIT value incrementally,
	  	// but we restrict the change to +/ 9.999 kHz
		new = vfo[active_receiver->id].rit + val*rit_increment;
		if (new >  9999) new=  9999;
		if (new < -9999) new= -9999;
		vfo[active_receiver->id].rit = new;
		break;
	      case MIDI_KNOB:
	 	// knob: adjust in the range +/ 50*rit_increment
		new = (val-50) * rit_increment;
		vfo[active_receiver->id].rit = new;
		break;
	      default:
		// do nothing
		// we should not come here anyway
		break;
	    }
	    // enable/disable RIT according to RIT value
	    vfo[active_receiver->id].rit_enabled = (vfo[active_receiver->id].rit == 0) ? 0 : 1;
            vfo_update();
	    break;
	/////////////////////////////////////////////////////////// "SAT"
        case MIDI_ACTION_SAT:
	    switch (sat_mode) {
		case SAT_NONE:
		  sat_mode=SAT_MODE;
		  break;
		case SAT_MODE:
		  sat_mode=RSAT_MODE;
		  break;
		case RSAT_MODE:
		default:
		  sat_mode=SAT_NONE;
		  break;
	    }
            vfo_update();
            break;
	/////////////////////////////////////////////////////////// "SNB"
	case MIDI_ACTION_SNB:	// only key supported
            if(active_receiver->snb==0) {
              active_receiver->snb=1;
              mode_settings[vfo[active_receiver->id].mode].snb=1;
            } else {
              active_receiver->snb=0;
              mode_settings[vfo[active_receiver->id].mode].snb=0;
            }
            update_noise();
	    break;
	/////////////////////////////////////////////////////////// "SPLIT"
	case MIDI_ACTION_SPLIT: // only key supported
	    // toggle split mode
            new= split ? 0:1;
            set_split(new);
	    break;
	/////////////////////////////////////////////////////////// "STOREMx"
	case MIDI_ACTION_MEM_STORE_M0:
            store_memory_slot(0);
	    break;
	case MIDI_ACTION_MEM_STORE_M1:
            store_memory_slot(1);
	    break;
	case MIDI_ACTION_MEM_STORE_M2:
            store_memory_slot(2);
	    break;
	case MIDI_ACTION_MEM_STORE_M3:
            store_memory_slot(3);
	    break;
	case MIDI_ACTION_MEM_STORE_M4:
            store_memory_slot(4);
	    break;
	/////////////////////////////////////////////////////////// "SWAPRX"
	case MIDI_ACTION_SWAP_RX:	// only key supported
	    if (receivers == 2) {
		new=active_receiver->id;	// 0 or 1
		new= (new == 1) ? 0 : 1;	// id of currently inactive receiver
		active_receiver=receiver[new];
		g_idle_add(menu_active_receiver_changed,NULL);
		g_idle_add(sliders_active_receiver_changed,NULL);
                vfo_update();
	    }
	    break;    
	/////////////////////////////////////////////////////////// "SWAPVFO"
	case MIDI_ACTION_SWAP_VFO:	// only key supported
            vfo_a_swap_b();
	    break;    
	/////////////////////////////////////////////////////////// "TUNE"
	case MIDI_ACTION_TUNE: // only key supported
	    if (can_transmit) {
	        new = !tune;
                tune_update(new);
	    }
	    break;    
	/////////////////////////////////////////////////////////// "VFOA"
	/////////////////////////////////////////////////////////// "VFOB"
	case MIDI_ACTION_VFOA: // only wheel supported
	case MIDI_ACTION_VFOB: // only wheel supported
	    if (!locked) {
		new = (action == MIDI_ACTION_VFOA) ? 0 : 1;
                vfo_id_step(new, val);
	    }
	    break;
	/////////////////////////////////////////////////////////// "VFOSTEPDOWN"
	/////////////////////////////////////////////////////////// "VFOSTEPUP"
        case MIDI_ACTION_VFO_STEP_DOWN: // key or wheel supported
        case MIDI_ACTION_VFO_STEP_UP:
	    switch (type) {
	      case MIDI_KEY:
		new =  (action == MIDI_ACTION_VFO_STEP_UP) ? 1 : -1;
                update_vfo_step(new);
		break;
	      case MIDI_WHEEL:
		new = (val > 0) ? 1 : -1;
                update_vfo_step(new);
		break;
	      default:
		// do nothing
		// we should not come here anyway
                break;
	    }
            break;
	/////////////////////////////////////////////////////////// "VOX"
	case MIDI_ACTION_VOX: // only key supported
	    // toggle VOX
            if (can_transmit) {
	      vox_enabled = !vox_enabled;
              vfo_update();
            }
	    break;
	/////////////////////////////////////////////////////////// "VOXLEVEL"
	case MIDI_ACTION_VOXLEVEL: // knob or wheel supported
            switch (type) {
              case MIDI_WHEEL:
                // This changes the value incrementally,
                // but stay within limits (0.0 through 1.0)
                vox_threshold += (double) val * 0.01;
		if (vox_threshold > 1.0) vox_threshold=1.0;
		if (vox_threshold < 0.0) vox_threshold=0.0;
                break;
              case MIDI_KNOB:
                vox_threshold = 0.01 * (double) val;
                break;
              default:
                // do nothing
                // we should not come here anyway
                break;
            }
	    // VOX level not shown on screen, hence no VFO update
	    break;
	/////////////////////////////////////////////////////////// "XITCLEAR"
        case MIDI_ACTION_XIT_CLEAR:  // only key supported
            // this clears the XIT value and disables XIT
            if(can_transmit) {
              transmitter->xit = 0;
              transmitter->xit_enabled = 0;
              vfo_update();
            }
            break;
	/////////////////////////////////////////////////////////// "XITVAL"
        case MIDI_ACTION_XIT_VAL:   // wheel and knob supported.
	    if (can_transmit) {
              switch (type) {
                case MIDI_WHEEL:
                  // This changes the XIT value incrementally,
                  // but we restrict the change to +/ 9.999 kHz
                  new = transmitter->xit + val*rit_increment;
                  if (new >  9999) new =  9999;
                  if (new < -9999) new = -9999;
                  transmitter->xit = new;
                  break;
                case MIDI_KNOB:
                  // knob: adjust in the range +/ 50*rit_increment
                  new = (val-50) * rit_increment;
                  transmitter->xit = new;
                  break;
                default:
                  // do nothing
                  // we should not come here anyway
                  break;
              }
              // enable/disable XIT according to XIT value
              transmitter->xit_enabled = (transmitter->xit == 0) ? 0 : 1;
              vfo_update();
	    }
            break;
	/////////////////////////////////////////////////////////// "ZOOM"
	/////////////////////////////////////////////////////////// "ZOOMDOWN"
	/////////////////////////////////////////////////////////// "ZOOMUP"
        //
        // The ZOOM key word is redundant but we leave it to maintain
        // backwards compatibility
        //
        case MIDI_ACTION_ZOOM:       // wheel, knob
        case MIDI_ACTION_ZOOM_UP:    // key, wheel, knob
        case MIDI_ACTION_ZOOM_DOWN:  // key, wheel, knob
	    switch (type) {
	      case MIDI_KEY:
                if (action == MIDI_ACTION_ZOOM) break;
		dnew =  (action == MIDI_ACTION_ZOOM_UP) ? 1.0 :-1.0;
                update_zoom(dnew);
		break;
	      case MIDI_WHEEL:
		dnew = (val > 0) ? 1.0 : -1.0;
                update_zoom(dnew);
		break;
              case MIDI_KNOB:
                dnew= 1.0+ 0.07*val;  // from 1.0 to 8.0
                if((int)dnew != active_receiver->zoom) {
                  set_zoom(active_receiver->id, dnew);
                }
                break;
	      default:
		// do nothing
		// we should not come here anyway
		break;
	    }
            break;

	case MIDI_ACTION_NONE:
	    // No error message, this is the "official" action for un-used controller buttons.
	    break;;
	default:
	    // This means we have forgotten to implement an action, so we inform on stderr.
	    fprintf(stderr,"Unimplemented MIDI action: A=%d\n", (int) action);
    }
    return 0;
}
