///
/// User management provider.
/// @file       users.h - pianod project
/// @author     Perette Barella
/// @date       2012-03-20
/// @copyright  Copyright 2012-2020 Devious Fish. All rights reserved.
///

#pragma once

#include <config.h>

#include <string>
#include <map>
#include <vector>
#include <functional>

#include <fb_public.h>
#include <football.h>

#include "user.h"

class PianodConnection;
class PianodService;

/// Container and manager class for users.
class UserManager : private std::map<std::string, User *> {
private:
    std::string shadow_user_name;
public:
    enum class WhichSources {
        User,           ///< Sources owned by a specific user.
        Listed,         ///< Sources that are publicly known
        Restorable      ///< Sources for automatic restore
    };
    using UserSelectionPredicate = std::function<bool (const User *user)>;
    using StoredSourcePair = std::pair<User *, UserData::DataStore *>;
    using StoredSourceList = std::vector<StoredSourcePair>;
    static inline time_t writeTime (void) { return User::write_time; };
    static inline bool needWrite (void) {
        return User::write_time != 0 && User::write_time > time(nullptr);
    };
    const std::string &shadowUserName () const;
    void shadowUserName (const std::string &who);

    User *addUser (User &user);
    User *get (const std::string &who);
    User *authenticate (const std::string &who, const std::string &password);
    void clearPrivilege (Privilege priv);
    bool validUserList (Football::Connection *event, char *const *username);

    std::vector<PianodConnection *> getUserConnections (PianodService &service,
                                                        const User *user) const;
    UserList getUsers (UserSelectionPredicate predicate = [] (const User *) { return true; }) const;
    UserList getUsersPresent (PianodService &service,
                              bool use_attribute = false) const;
    UserList allUsers (void) const { return getUsers (); };
    StoredSourceList getStoredSources (WhichSources selection,
                                       const User *visibleBy = nullptr) const;
    RESPONSE_CODE findStoredSource (const std::string &type,
                                    const std::string &name,
                                    User *forWho,
                                    UserData::StringDictionary **found,
                                    User **owner);
    void createDefaultUser ();
    void deleteUser (User *user);
    bool persist ();
    void restore ();
    float periodic (void);
    ~UserManager ();
};

extern /* Global */ UserManager *user_manager;

