///
/// Command handlers for all things user-related.
/// @file       usercommand.cpp - pianod project
/// @author     Perette Barella
/// @date       Initial: 2012-03-10.  C++: 2014-11-22.
/// @copyright  Copyright 2012-2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cstdio>
#include <cassert>

#include <football.h>

#include "fundamentals.h"
#include "enum.h"
#include "response.h"
#include "user.h"
#include "users.h"
#include "servicemanager.h"
#include "mediamanager.h"
#include "engine.h"

using namespace std;

/** Check a list of users, and respond if some don't exist.
    @param conn The connection to be checked.
    @return true if all users are valid, false otherwise. */
static bool validate_user_list (PianodConnection &conn, const char *startterm = "user") {
    std::vector<std::string> users = conn.argvFrom (startterm);
    bool found = true;
    for (string username : users) {
        if (user_manager->get (username) == NULL) {
            conn << Response (D_NOTFOUND, username);
            found = false;
        };
    }
    if (!found) {
        conn << E_NOTFOUND;
    }
    return found;
}

/** Set the requested privilege for a list of users.
    @param users A list of usernames.
    @param priv The privileges to set.
    @param setting Whether to enable (true) or disable (false).
 */
static void set_privileges (const vector<string> &users, Privilege priv, bool setting) {
    assert (Enum<Privilege>().isValid (priv));
    for (string username : users) {
        User *u = user_manager->get (username);
        assert (u);
        u->setPrivilege(priv, setting);
	}
 }


/** Transmit a user record, either just name or with details.
    @param there Where to send the information.
    @param user The user to send information of.
    @param details Whether to send privileges, etc. */
static void send_user (PianodConnection &there, const User *user, bool details) {
    there << S_DATA << Response (I_ID, user->username());
	if (details) {
        send_privileges (there, user);
	}
}

/** Send lists of users based on privilege or state criteria.
    @warning The pseudoenumeration values for `which` must not overlap.
    @param conn The destination of the information.
    @param which A list of users to send.
    @param details Whether to include privileged information in the list. */
static void send_users (PianodConnection &conn, UserList which, bool details) {
    for (auto user : which) {
        send_user (conn, user, details);
    }
    conn << S_DATA_END;
}

static void send_users (PianodConnection &conn, UserList which) {
    send_users (conn, which, conn.haveRank (Rank::Administrator));
}

/** Logoff users.
    @param service The service/room to logoff user in, or null to logoff everywhere.
    @param user The user to logoff, or nullptr to logoff visitors/unauthenticated users.
    @param message An optional message to send before disconnecting. */
static bool user_logoff (PianodService *service, User *user, const char *message = NULL) {
    bool any = false;
    if (!service) {
        for (auto const &svc : *service_manager) {
            any = user_logoff (svc.second, user, message) || any;
        }
    } else {
        for (auto conn : *service) {
            if (conn->user == user) {
                conn << Response (V_SERVER_STATUS, message ? message : "Logged off by an administrator");
                any = true;
                conn->close();
            }
        }
    }
    return any;
}


#define RANK_PATTERN "<rank:disabled|listener|user|admin>"
#define PRIV_PATTERN "<privilege:service|deejay|influence|tuner>"

typedef enum user_commands_t {
    AUTHENTICATE = CMD_RANGE_USER,
    AUTHANDEXEC,
    SETMYPASSWORD,
    USESHADOWPASSWORD,
    USERCREATE,
    USERSETPASSWORD,
    GETVISITORRANK,
    SETVISITORRANK,
    GETSHADOWUSERNAME,
    SETSHADOWUSERNAME,
    GETUSERRANK,
    USERSETRANK,
    USERDELETE,
    USERGRANT,
    USERREVOKE,
    USERLISTBYPRIVILEGE,
    USERLIST,
    USERKICK,
    USERKICKVISITORS,
    USERSONLINE,
    USERSINROOM,
    AUTOTUNEUSERS,
    AUTOTUNEUSERSLIST,
    AUTOTUNEADDREMOVE,
} COMMAND;

static const FB_PARSE_DEFINITION statementList[] = {
    { AUTHENTICATE,		"user {username} {password}" },                 // Authenticate
    { AUTHANDEXEC,		"as user {username} {password} [{command}] ..." },// Authenticate and execute command.
    { GETUSERRANK,		"get privileges" },                             // Request user's level/privileges
    { SETMYPASSWORD,	"set password {old} {new}" },                   // Let a user update their password
    { USESHADOWPASSWORD,"set shadow password {pianod} {system}" },      // Switch from custom to shadow password

    { GETVISITORRANK,	"get visitor rank" },                           // Visitor privilege level
    { SETVISITORRANK,	"set visitor rank " RANK_PATTERN },             // Visitor privilege level
    { GETSHADOWUSERNAME,"get shadow user name" },                       // Query the shadow user's username
    { SETSHADOWUSERNAME,"set shadow user name {username}" },            // Select a template for new shadow users

    { USERCREATE,		"create <rank:listener|user|admin> {user} {password}" },
                                                                        // Add a new user
    { USERSETPASSWORD,	"set user password {user} {password}" },		// Change a user's password
    { USERSETRANK,		"set user rank {user} " RANK_PATTERN },         // Alter rank
    { USERDELETE,		"delete user {user}" },							// Remove a user
    { USERGRANT,		"grant " PRIV_PATTERN " to {user} ..." },       // Grant privilege
    { USERREVOKE,		"revoke " PRIV_PATTERN " from {user} ..." },    // Revoke privilege
    { AUTOTUNEUSERS,	"autotune for [{user}] ..." },					// Tune for a list of users.
    { AUTOTUNEUSERSLIST,"autotune list users" },                        // List users considered by autotuning
    { AUTOTUNEADDREMOVE,"autotune <action:consider|disregard> {user} ..." },	// Add users to tuning list.
    { USERLISTBYPRIVILEGE, "users with <attribute:owner|service|influence|tuner|present>" },
                                                                        // List users with a privilege
    { USERLIST,			"users list [{user}]" },						// List all or a specific user
    { USERSONLINE,		"users online" },								// List users logged in
    { USERSINROOM,      "users in room" },                              // Users using current room
    { USERKICK,			"kick [all|room] user {user} [{message}]" },	// Log a user off
    { USERKICKVISITORS,	"kick [all|room] visitors [{message}]" },       // Disconnect unauthenticated
    { CMD_INVALID,      NULL }
};


/** Interpreter for user-related commands (CRUD functions, adjust privileges,
    view users online, etc.) */
class UserCommands : public Football::Interpreter<PianodConnection, COMMAND> {
private:
    virtual bool hasPermission (PianodConnection &conn, COMMAND command) override;
    virtual void handleCommand (PianodConnection &conn, COMMAND command) override;
    virtual const FB_PARSE_DEFINITION *statements (void) override {
        return statementList;
    };
};

bool UserCommands::hasPermission (PianodConnection &conn, COMMAND command) {
    switch (command) {
        case AUTHENTICATE:
        case AUTHANDEXEC:
        case GETUSERRANK:
        case SETMYPASSWORD:
        case USESHADOWPASSWORD:
            return true;

        case AUTOTUNEUSERSLIST:
            // If sharing user actions, anyone can see who influences autotuning
            if (conn.broadcastingActions()) return true;
            // FALLTHRU
        case AUTOTUNEADDREMOVE:
        case AUTOTUNEUSERS:
            return conn.havePrivilege (Privilege::Tuner);

        case USERSONLINE:
        case USERLIST:
            // If sharing user actions, anyone can view users since we announce them anyway.
            // But, never allow search by privilege since that would breach secrets.
            // Admins can always see users online and get their privileges.
            if (conn.broadcastingActions()) return true;
            // FALLTHRU
        default:
            return conn.haveRank (Rank::Administrator);
    }
};

void UserCommands::handleCommand (PianodConnection &conn, COMMAND command) {
    switch (command) {
        case AUTHENTICATE:
            conn.user = user_manager->authenticate (conn.argv("username"), conn.argv("password"));
            if (!conn.authenticated()) {
                conn << E_CREDENTIALS;
                return;
            }
            conn.announce (A_SIGNED_IN, NULL);
            conn.updateConnection ();
            conn << V_PLAYLISTRATING_CHANGED;
            conn.service().usersChangedNotification();
            // FALLTHRU
        case GETUSERRANK:
            conn << S_OK;
            conn.sendEffectivePrivileges();
            return;
        case AUTHANDEXEC:
            conn.user = user_manager->authenticate (conn.argv("username"), conn.argv("password"));
            if (!conn.authenticated()) {
                conn << E_CREDENTIALS;
            } else {
                conn.reinterpret("command");
            }
            conn.close_after_events();
            conn.user = nullptr;
            return;
        case USESHADOWPASSWORD:
#ifdef SHADOW_CAPABLE
            if (!conn.authenticated()) {
                conn << E_LOGINREQUIRED;
            } else if (conn.user->assumeShadowPassword (conn.argv ("pianod"), conn.argv("system"))) {
                conn << S_OK;
            } else {
                conn << E_CREDENTIALS;
            };
            return;
#else
            conn << E_UNSUPPORTED;
            return;
#endif
        case SETMYPASSWORD:
            if (!conn.authenticated()) {
                conn << E_LOGINREQUIRED;
            } else if (conn.user->changePassword(conn.argv ("old"), conn.argv("new"))) {
                conn << S_OK;
            } else {
                conn << E_CREDENTIALS;
            };
            return;


        case GETVISITORRANK:
            conn << S_DATA << Response (I_USER_PRIVILEGES, User::getRankName(User::getVisitorRank())) << S_DATA_END;
            return;
        case SETVISITORRANK:
        {
            bool rank_was_valid = User::setVisitorRank (conn.argv ("rank"));
            assert (rank_was_valid);
            conn << (rank_was_valid ? S_OK : E_NAK);
            if (rank_was_valid) {
                service_manager->broadcastEffectivePrivileges (nullptr);
            }
            return;
        }

        case GETSHADOWUSERNAME:
        {
            const string &shadow = user_manager->shadowUserName ();
            if (shadow.empty()) {
                conn << E_NOTFOUND;
            } else {
                conn << S_DATA << Response (I_NAME, shadow) << S_DATA_END;
            }
            return;
        }
        case SETSHADOWUSERNAME:
        {
            const char *name = conn ["username"];
            assert (name);
            if (!*name || user_manager->get (name)) {
                user_manager->shadowUserName (name);
                conn << S_OK;
            } else {
                conn << E_NOTFOUND;
            }
            return;
        }


            
        case USERCREATE:
        {
            // Create a new user with a rank and password
            User newuser (conn.argv ("user"), conn.argv ("password"), true);
            kept_assert (newuser.assignRank (conn.argv ("rank")));
            if (user_manager->addUser (newuser)) {
                conn << S_OK;
            } else {
                conn << E_DUPLICATE;
            }
            return;
        }
		case USERSETPASSWORD:
        {
            User *revise = user_manager->get (conn.argv ("user"));
            if (revise) {
                revise->setPassword (conn.argv ("password"));
                conn << S_OK;
            } else {
                conn << E_NOTFOUND;
            }
            return;
        }
        case USERSETRANK:
        {
            User *revise = user_manager->get (conn.argv ("user"));
            if (revise) {
                kept_assert (revise->assignRank (conn.argv ("rank")));
                conn << S_OK;
                service_manager->broadcastEffectivePrivileges (revise);
            } else {
                conn << E_NOTFOUND;
            }
            return;
        }
        case USERGRANT:
        case USERREVOKE:
            if (validate_user_list(conn)) {
                Privilege privilege = User::getPrivilege(conn.argv ("privilege"));
                set_privileges (conn.argvFrom ("user"), privilege, command == USERGRANT);
                if (privilege == Privilege::Influence) {
                    conn.service().usersChangedNotification();
                }
                service_manager->broadcastEffectivePrivileges();
                conn << S_OK;
            }
            return;



        case AUTOTUNEUSERSLIST:
            send_users (conn,
                        conn.service().audioEngine()->getAutotuneUsers());
            return;
        case AUTOTUNEUSERS:
        case AUTOTUNEADDREMOVE:
            if (validate_user_list(conn)) {
                if (command == AUTOTUNEUSERS) {
                    user_manager->clearPrivilege (Privilege::Present);
                }
                if (conn.argv ("user")) {
                    set_privileges (conn.argvFrom ("user"), Privilege::Present,
                                    conn.argvEquals ("action", "consider") ||
                                    command == AUTOTUNEUSERS);
                }
                conn << S_OK;
                conn.service().usersChangedNotification();
            }
            return;



        case USERSONLINE:
            send_users (conn,
                        user_manager->getUsers ([] (const User *user) -> bool {
                            return service_manager->userIsOnline (user);
                        }));
            break;
        case USERSINROOM:
        {
            PianodService &service = conn.service();
            send_users (conn,
                        user_manager->getUsers ([&service] (const User *user) -> bool {
                            return user->online (service);
                        }));
            break;
        }
        case USERLIST:
        {
            const char *who = conn.argv("user");
            if (who) {
                User *user = user_manager->get (who);
                if (user) {
                    send_user(conn, user, conn.haveRank (Rank::Administrator));
                    conn << S_DATA_END;
                } else {
                    conn << E_NOTFOUND;
                }
            } else {
                send_users (conn, user_manager->getUsers ());
            }
            return;
        }
        case USERLISTBYPRIVILEGE:
        {
            // haveRank: Only reveal privileges to administrators
            Privilege priv = User::getPrivilege(conn.argv ("attribute"));
            send_users (conn,
                        user_manager->getUsers ([priv] (const User *user) -> bool {
                            return user->havePrivilege (priv);
                        }));
            break;
        }

        case USERKICKVISITORS:
        case USERKICK:
        {
            User *target = nullptr;
            const char *whom = "the visitors";
            if (command == USERKICK) {
                whom = conn.argv ("user");
                target = user_manager->get (whom);
                if (!target) {
                    conn << E_NOTFOUND;
                    return;
                }
            }
            if (user_logoff (conn.argvEquals ("room", nullptr) ? nullptr : &conn.service(),
                             target, conn.argv ("message"))) {
                conn << S_OK;
                conn.announce (A_KICKED, whom);
            } else {
                conn << Response (E_WRONG_STATE, "Target not online.");
            }
            return;
        }
        case USERDELETE:
        {
            User *remove = user_manager->get (conn.argv ("user"));
            if (remove) {
                if (service_manager->userIsOnline (remove)) {
                    conn << Response (E_WRONG_STATE, "User is logged in.");
                } else {
                    // Disown the user's sources before deleting user
                    for (auto &src : *media_manager) {
                        if (src.second->isOwnedBy (remove)) {
                            src.second->abandon();
                        }
                    }
                    user_manager->deleteUser (remove);
                    conn << S_OK;
                }
            } else {
                conn << E_NOTFOUND;
            }
            return;
        }
        default:
            conn << E_NOT_IMPLEMENTED;
            flog (LOG_WHERE (LOG_WARNING), "Unimplemented command ", command);
            break;
    }
}

static UserCommands interpreter;

void register_user_commands (Football::ServiceBase *service) {
    service->addCommands (&interpreter);
}

