///
/// Method implementations for music library.
///	@file		musiclibrary.cpp - pianod
///	@author		Perette Barella
///	@date		2014-12-09
///	@copyright	Copyright 2014-2020 Devious Fish.  All rights reserved.
///

#include <config.h>

#include <math.h>

#include <string>
#include <unordered_map>
#include <functional>

#include <cstdio>

#include <football.h>
#include <parsnip.h>

#include "musiclibrary.h"
#include "musiclibraryparameters.h"
#include "fundamentals.h"
#include "utility.h"
#include "datastore.h"
#include "user.h"
#include "users.h"
#include "mediaunit.h"
#include "musickeys.h"
#include "fileio.h"
#include "musiclibraryhash.h"

using namespace std;

namespace MusicLibrary {
    Allocator <Artist, Foundation> artist_allocate;
    Allocator <Album, Artist> album_allocate;
    Allocator <Song, Album> song_allocate;
    Allocator <Playlist, Foundation> playlist_allocate;

    // Explicitly instantiate needed containers.
    template class ThingieContainer <Artist, Foundation>;
    template class ThingieContainer <Album, Artist>;
    template class ThingieContainer <Song, Album>;
    template class ThingieContainer <Playlist, Foundation>;

    /*
     *                  Library Foundation
     */
     
    /** Construct a new media library.
        @param owner The source to which the library belongs. */
    Foundation::Foundation (Media::Source *owner) : source (owner) {
    }

    /** Restore persisted library data from a file.
        @return True on success, false on failure. */
    bool Foundation::load() {
        bool ok;
        try {
            ok = restoreIndexFromFile (source->filename());
        } catch (const ios_base::failure &) {
            ok = false;
        }
        write_time = 0;
        return ok;
    }

    /** If memory is dirty, write library data to a file.
        @return True on success, false on failure. */
    bool Foundation::flush() {
        if (write_time == 0)
            return true;
        // In case of failure, try again in a while
        write_time = time (0) + 1800;
        bool status = writeIndexToFile (source->filename());
        if (status)
            write_time = 0;
        return status;
    }

    /** Do intermittent tasks, such as occasionally persisting
        the catalog to a file.
        @return Interval until function wants to be called again. */
    float Foundation::periodic() {
        if (write_time) {
            time_t now = time (nullptr);
            if (now >= write_time) {
                flush();
                return A_LONG_TIME;
            }
            return write_time - now;
        }
        return A_LONG_TIME;
    }

    using BiasType = double;

    /** Calculate bias values for some songs.
        @param candidates The list of songs for which to compute biases.
        @param [out] biases On return, the computed biases.
        @param users_ratings Users' song ratings to consider.
        @param overplay_ratings Users' overplay ratings to consider.
        @param settings Library parameters, for biasing levels. */
    static BiasType compute_biases (const SongList &candidates,
                                    vector<BiasType> &biases,
                                    const vector<UserData::Ratings *> &users_ratings,
                                    const vector<UserData::OverplayedList *> &overplay_ratings,
                                    const LibraryParameters &settings,
                                    const bool album_mode) {
        BiasType rating_scale = (BiasType (settings.rating_bias) * 0.09 + 1);
        BiasType recent_scale = (BiasType (settings.recent_bias) + 10) / 11;

        // Compute average age.  Matters for small numbers of songs;
        // If there are a large number number, assuming a week is good enough.
        time_t now = time (nullptr);
        BiasType average_age = 86400 * 7;  // One week
        if (candidates.size() < 2880) {
            // Small enough set to be worth doing the calculation.
            int count = 0;
            BiasType sum = 0;
            for (const auto song : candidates) {
                if (song->lastPlayed()) {
                    sum += (now - song->lastPlayed());
                    count++;
                }
            }
            if (count) {
                average_age = sum / BiasType (count);
            }
            // Don't use less than a minute for average song age
            if (average_age < 60)
                average_age = 60;
        }

        flog (LOG_FUNCTION (LOG_BIASING), "Computed biases follow:");

        biases.clear();
        biases.reserve (candidates.size());
        BiasType total_bias = 0;
        for (const auto song : candidates) {
            // Calculate the average rating for this song.
            RatingAverager average;
            for (auto ratings : users_ratings) {
                average.add (ratings->get (song->songId(), Rating::UNRATED));
            }
            BiasType average_rating = average.rating (Rating::NEUTRAL);
            if (average_rating < 0.000001)
                average_rating = 0.000001;
            time_t age = (song->lastPlayed() ? now - song->lastPlayed() : FAR_FUTURE);
            BiasType relative_age = BiasType (age) / average_age;
            if (relative_age > 10)
                relative_age = 10;
            else if (relative_age < 0.000001)
                relative_age = 0.000001;
            BiasType bias = (powf (rating_scale, (average_rating - static_cast<double> (Rating::NEUTRAL)) / 4)
                             * powf (recent_scale, log10f (relative_age)));
            assert (bias >= 0);
            // When choosing by album, divide by number of tracks so albums
            // are equally likely to be chosen, instead of more tracks=more likely.
            if (album_mode) {
                assert (static_cast<Song *> (song)->albumTrackCount());
                bias /= static_cast<Song *> (song)->albumTrackCount();
            }

            // If anyone has rated this overplayed, refuse to play song.
            bool overplayed = false;
            for (auto ratings : overplay_ratings) {
                time_t until = ratings->get (song->songId(), 0);
                if (until > now) {
                    overplayed = true;
                    break;
                }
            }

#ifndef NDEBUG
            if (logging_enabled (LOG_BIASING)) {
                fprintf (stderr,
                         "%-40.40s rate %5f age %5f bias %5f %s\n",
                         song->title().c_str(),
                         average_rating,
                         relative_age,
                         bias,
                         overplayed ? "(overplayed)" : "");
            }
#endif
            if (overplayed)
                bias = 0.0;
            // First song has range [0, biases [0])
            // Remaining have range [biases [n-1], biases [n])
            total_bias += bias;
            biases.push_back (total_bias);
        }
        return total_bias;
    }

    static SongList get_biased_selections (const SongList &candidates,
                                           const vector<BiasType> &biases,
                                           const BiasType total_bias,
                                           const int requested_count) {
        unsigned long retrieve = candidates.size();
        if (retrieve > requested_count)
            retrieve = requested_count;
        // Spin limit: Imagine 4 songs in the queue, 3 normal and one with a horrible bias.
        SongList selections;
        for (int i = 0, spin_limit = 8; i < retrieve && spin_limit > 0; i++, spin_limit--) {
            bool found = false;
            while (!found) {
                BiasType choice = fmod (random(), total_bias);
                // Binary search for the song with the chosen bias.
                SongList::size_type lower = 0;
                SongList::size_type upper = candidates.size() - 1;
                while (lower < upper) {
                    unsigned long index = (lower + upper) >> 1;  // Middle
                    assert (index >= 0 && index < candidates.size());
                    if (choice >= biases[index]) {
                        lower = index + 1;
                    } else {
                        upper = index;
                    }
                }
                // Check for duplicate choice.
                found = true;
                for (auto item : selections) {
                    if (item == candidates[lower]) {
                        /* Duplicate */
                        found = false;
                        break;
                    }
                }
                if (found) {
                    selections.push_back (candidates[lower]);
                    break;
                }
            }  // While unique item not found
        }      // For each request
        return selections;
    }

    /** Retrieve some random selections from the library for queue/playback.
        @param playlist The playlist selections should be from.
        @param users Users to consider when making selections.
        @param selectionMethod Manner in which to pick selections.
        @param settings Biasing parameters. */
    SongList Foundation::getRandomSongs (PianodPlaylist *playlist,
                                         const UserList &users,
                                         Media::SelectionMethod selectionMethod,
                                         const LibraryParameters &settings) {
        assert (settings.rating_bias >= 1 && settings.rating_bias <= 100);
        assert (settings.recent_bias >= 1 && settings.recent_bias <= 100);
        assert (selectionMethod == Media::SelectionMethod::Song ||
                selectionMethod == Media::SelectionMethod::Artist ||
                selectionMethod == Media::SelectionMethod::Album);

        // Collect ratings from relevant users
        vector<UserData::Ratings *> users_ratings;
        vector<UserData::OverplayedList *> overplayed_ratings;
        users_ratings.reserve (users.size());
        overplayed_ratings.reserve (users.size());
        for (auto user : users) {
            auto ratings = UserData::Ratings::retrieve (user,
                                                        UserData::Key::TrackRatings,
                                                        playlist->source()->key());
            if (ratings) {
                users_ratings.push_back (ratings);
            }
            auto overplays = UserData::OverplayedList::retrieve (user,
                                                                 playlist->source()->key());
            if (overplays) {
                overplayed_ratings.push_back (overplays);
            }
        }

        SongList candidates (getSongsForPlaylist (playlist));

        // Compute biases.
        vector<BiasType> biases;
        BiasType total_bias = compute_biases (candidates,
                                              biases,
                                              users_ratings,
                                              overplayed_ratings,
                                              settings,
                                              selectionMethod == Media::SelectionMethod::Album);

        SongList response = get_biased_selections (candidates,
                                                   biases,
                                                   total_bias,
                                                   selectionMethod == Media::SelectionMethod::Song ? 4 : 1);
        if (!response.empty() && selectionMethod != Media::SelectionMethod::Song) {
            Song *song = dynamic_cast<Song *> (response.front());
            assert (song);
            if (selectionMethod == Media::SelectionMethod::Album) {
                return song->_album->songs();
            }
            assert (selectionMethod == Media::SelectionMethod::Artist);
            candidates = song->_album->_artist->songs();
            total_bias = compute_biases (candidates, biases,
                                         users_ratings, overplayed_ratings,
                                         settings, false);
            response = get_biased_selections (candidates, biases, total_bias, 4);
        }
        return response;
    };

    /*
     *                  Library Parameters
     */

    const LookupTable <ScanFrequency> RescanModes ( {
        { "never",          SCAN_NEVER },
        { "once",           SCAN_ON_LOAD },
        { "always",         SCAN_EVERY_LOAD },
        { "periodically",   SCAN_DAILY }
    });

    const FB_PARSE_DEFINITION library_options [] = {
        { RECENT_BIAS, "recent bias {#level:1-100} ..." },
        { RATING_BIAS, "rating bias {#level:1-100} ..." },
        { RESCAN_BEHAVIOR,  "rescan {when:never|once|always|periodically} ..." },
        { 0, NULL }
    };

    LibraryParameters::LibraryParameters (Ownership::Type perm, User *user) : Media::SourceParameters (perm, user) {
    }
    LibraryParameters::LibraryParameters (const UserData::StringDictionary &src) : Media::SourceParameters (src) {
        rating_bias = stoi (src.get ("ratingBias", "1"));
        if (rating_bias < BIAS_MINIMUM || rating_bias > BIAS_MAXIMUM)
            throw invalid_argument ("Invalid ratingBias");

        recent_bias = stoi (src.get ("recentBias", "1"));
        if (recent_bias < BIAS_MINIMUM || recent_bias > BIAS_MAXIMUM)
            throw invalid_argument ("Invalid recentBias");

        string mode = src.get ("rescan", "");
        if (!mode.empty())
            scan_frequency = RescanModes[mode];
        if (scan_frequency == SCAN_ON_LOAD)
            scan_frequency = SCAN_NEVER;
    };

    bool LibraryParameters::persist (UserData::StringDictionary &dest) const {
        dest.add ("ratingBias", to_string (rating_bias));
        dest.add ("recentBias", to_string (recent_bias));
        dest.add ("rescan", RescanModes[scan_frequency == SCAN_ON_LOAD ? SCAN_NEVER : scan_frequency]);
        return Media::SourceParameters::persist (dest);
    };

    int interpretLibraryBasedParameters (Football::GenericOptionParser &parser, int cmd, LibraryParameters &result) {
        LibraryParameterId lpid = (LibraryParameterId) cmd;
        switch (lpid) {
            case RATING_BIAS:
                result.rating_bias = atoi (parser.argv ("level"));
                return FB_PARSE_SUCCESS;
            case RECENT_BIAS:
                result.recent_bias = atoi (parser.argv ("level"));
                return FB_PARSE_SUCCESS;
            case RESCAN_BEHAVIOR:
                result.scan_frequency = RescanModes[parser.argv ("when")];
                return FB_PARSE_SUCCESS;
        }
        return interpretCommonSourceParameters (parser, cmd, result);
    }
    ParameterParser<LibraryParameters, LibraryParameterId> parameterParser;

    /*
     *              Library Implementation
     */

    /** Construct a library.
        @owner The source to which the media belongs.
        @param song_allocator A function constructing songs of the source's type.
        @param album_allocator A function constructing albums of the source's type.
        @param artist_allocator A function constructing artists of the source's type.
        @param playlist_allocator A function constructing playlists of the source's type. */
    Library::Library (Media::Source *owner,
                      const SongAllocator &song_allocator,
                      const AlbumAllocator &album_allocator,
                      const ArtistAllocator &artist_allocator,
                      const PlaylistAllocator &playlist_allocator)
    : Foundation (owner),
      artists (artist_allocator),
      albums (album_allocator),
      songs (song_allocator),
      playlists (playlist_allocator){
              // Derived classes should invoke restoreIndexFromFile;
              // object is not fully constituted here.
      };

    /** Remove albums & artists from the library that don't have any
        songs & albums respectively. */
    void Library::purge (void) {
        albums.purge ([] (const Album *album) -> bool { return (album->getUseCount() == 1 && album->empty()); });
        artists.purge ([] (const Artist *artist) -> bool { return (artist->getUseCount() == 1 && artist->empty()); });
    }

    /** Remove a playlist from the library.
        @param play The playlist to remove. */
    bool Library::removePlaylist (Playlist *play) {
        auto it = playlists.find (play->playlistId());
        assert (it != playlists.end());
        playlists.erase (it);
        unpopulatePlaylist (play);
        play->release();
        return true;
    };

    /** Retrieve a list of seeds for a playlist.
        @param playlist The playlist for which to return data.
        @return A list containing song, album, and artist seeds. */
    ThingieList Library::seedsForPlaylist (const Playlist *playlist) {
        ThingieList results;
        for (auto &item : playlist->seeds) {
            assert (!item.empty());
            // Type code is the first character of the ID.
            const MusicThingie::Type type = (MusicThingie::Type) item.at (0);
            MusicThingie *thing = nullptr;
            switch (type) {
                case MusicThingie::Type::Song:
                    thing = songs.getById (item);
                    break;
                case MusicThingie::Type::Album:
                    thing = albums.getById (item);
                    break;
                case MusicThingie::Type::Artist:
                    thing = artists.getById (item);
                    break;
                default:
                    assert (0);
                    break;
            }
            if (thing) {
                results.push_back (thing);
            }
        }
        return results;
    }

    /// Get a list of all songs in the library.
    SongList Library::getAllSongs (void) {
        SongList list;
        list.reserve (songs.size());
        for (auto item : songs) {
            list.push_back (item.second);
        }
        return list;
    }

    /// Get a list of all songs matching a filter.
    SongList Library::getMatchingSongs (const Filter &criteria) {
        SongList list;
        list.reserve (songs.size());
        for (auto &item : songs) {
            if (criteria.matches (item.second)) {
                list.push_back (item.second);
            }
        }
        return list;
    }

    /** Retrieve suggestions from the library.  This returns a mix
        of artists, albums, and songs.
        @param criteria Specifies the criteria for which to select
        matching data.
        @param what Describes a search manner.
        - For deep searches, all matching items will be returned.
        - For shallow searches, matches on an artist or album
        will skip their respective albums or songs. */
    ThingieList Library::getSuggestions (const Filter &criteria, SearchRange what) {
        bool exhaustive = deepSearch (what);
        ThingieList list;
        for (auto &artist : artists) {
            bool matches_artist = criteria.matches (artist.second);
            if (matches_artist)
                list.push_back (artist.second);
            if (!matches_artist || exhaustive) {
                for (auto album : artist.second->getAlbums()) {
                    bool matches_album = criteria.matches (album);
                    if (matches_album)
                        list.push_back (album);
                    if (!matches_album || exhaustive) {
                        for (auto song : album->getSongs()) {
                            if (criteria.matches (song)) {
                                list.push_back (song);
                            }
                        }
                    }
                }
            }
        }
        return list;
    }

    /// Get a list of all songs belonging to enabled playlists.
    SongList Library::getMixSongs (void) {
        SongList list;
        list.reserve (songs.size());
        for (auto item : songs) {
            if (item.second->_playlist && item.second->_playlist->enabled) {
                list.push_back (item.second);
            }
        }
        return list;
    }

    /** Get a list of all songs assigned to a playlist.
        @param play The playlist for which to retrieve songs.
        @param reassess If false, only assigned songs are returned.
        If true, songs in other playlists are also considered.
        @return A list of songs belonging to the playlist. */
    SongList Library::getPlaylistSongs (const Playlist *play, bool reassess) {
        SongList list;
        list.reserve (songs.size());
        for (auto item : songs) {
            if ((item.second->_playlist == play)
                || (reassess && item.second->_playlist && play->appliesTo (item.second))) {
                list.push_back (item.second);
            }
        }
        return list;
    }

    /** Find a playlist for a song, preferring enabled playlists.
        First, aim for enabled playlists;
        if not found then recurse and aim for disabled playlists. */
    Playlist *Library::findPlaylistForSong (Song *song, bool enabled) {
        for (auto playlist : playlists) {
            if (playlist.second->enabled == enabled) {
                if (playlist.second->appliesTo (song)) {
                    return playlist.second;
                }
            }
        }
        if (enabled == false)
            return nullptr;
        return findPlaylistForSong (song, false);
    }

    /** Review songs and assign them to a new candidate if they match. This is
        applicable when a playlist has been created or just been added to the mix.
        @param play The playlist to populate with songs.
        @param aggressive If true, songs with enabled playlists are  considered for reassignment.
        If false, only songs without a playlist or with a disabled playlist are considered. */
    void Library::populatePlaylist (Playlist *play, bool aggressive) {
        for (auto item : songs) {
            // Only grab songs that aren't assigned or whose playlist isn't enabled, unless we're aggressive
            if (aggressive || !item.second->_playlist || !item.second->_playlist->enabled) {
                // Don't waste time if we already own it
                if (item.second->_playlist != play) {
                    if (play->appliesTo (item.second)) {
                        item.second->_playlist = play;
                    }
                }
            }
        }
    }

    /** Reassign all a playlists' songs to some other playlist.
        Applicable when song has been removed from the mix, or
        playlist is about to be deleted.
        @warning May reassign current playlists, if no better ones are found
        and the current playlist is still in the playlist set. */
    void Library::unpopulatePlaylist (Playlist *play) {
        for (auto &item : songs) {
            // Reassign only if it belongs to this playlist
            if (item.second->_playlist == play) {
                item.second->_playlist = findPlaylistForSong (item.second);
            }
        }
    }
    /** Retrieve anything stored in the library by its ID.
        @param type The type of the thing.
        @param id The ID of the thing to retrieve.
        @return The thing, or a nullptr if not found. */
    MusicThingie *Library::getById (MusicThingie::Type type, const std::string &id) {
        MusicThingie *thing = nullptr;
        switch (type) {
            case MusicThingie::Type::Playlist: {
                MusicLibrary::Playlist *play = playlists.getById (id);
                if (!play)
                    return nullptr;
                populatePlaylist (play, true);
                thing = play;
                break;
            }
            case MusicThingie::Type::Artist:
                thing = artists.getById (id);
                break;
            case MusicThingie::Type::Album:
                thing = albums.getById (id);
                break;
            case MusicThingie::Type::Song:
                thing = songs.getById (id);
                break;
            default:
                // Asked for a suggestion, seed, etc.
                // Not used by this source, may nevertheless be asked for one.
                break;
        }
        return thing;
    }

    /** Retrieve all songs for a playlist, which may be a meta playlist */
    SongList Library::getSongsForPlaylist (PianodPlaylist *playlist) {
        switch (playlist->playlistType()) {
            case PianodPlaylist::SINGLE:
            case PianodPlaylist::TRANSIENT:
                return playlist->songs();
            case PianodPlaylist::MIX:
                return getMixSongs();
            case PianodPlaylist::EVERYTHING:
                return getAllSongs();
        }
        assert (!"Unreachable");
        return SongList();
    }

    /** Construct a playlist with an initial seed.
        @param name The name for the new playlist.
        @param type The manner in which to interpret the initial seed.
        @param from An initial seed.
        @return The new playlist. */
    PianodPlaylist *Library::createPlaylist (const std::string &name, MusicThingie::Type type, MusicThingie *from) {
        assert (from);
        assert (from->source() == source);
        assert (!playlists.getByName (name, this));
        if (playlists.getByName (name, this))
            throw CommandError (E_DUPLICATE);
        Playlist *play = playlists.addOrGetItem (name, this);
        play->selector = Filter::None;
        populatePlaylist (play);
        markDirty (IMPORTANT);
        try {
            play->seed (type, from, true);
        } catch (...) {
            removePlaylist (play);
            throw;
        }
        return play;
    }

    /** Construct a new smart playlist.
        @param name A name for the new playlist.
        @param filter Search criteria with which to construct the playlist.
        @return The newly constructed playlist. */
    PianodPlaylist *Library::createPlaylist (const std::string &name, const Filter &filter) {
        assert (filter.canPersist());
        assert (!playlists.getByName (name, this));
        Playlist *play = playlists.addOrGetItem (name, this);
        play->selector = filter;
        populatePlaylist (play);
        markDirty (IMPORTANT);
        return play;
    }

    /** Construct a temporary playlist.
        @param filter Search criteria for the temporary playlist.
        @return The temporary playlist. */
    PianodPlaylist *Library::formTransientPlaylist (const Filter &criteria) {
        return new TransientPlaylist (this, criteria);
    }

    /** Persist the library's index into a file.
        This includes playlists and their seeds and match criteria.
        @param filename The name of the file. */
    bool Library::writeIndexToFile (const std::string &filename) const {
        try {
            Parsnip::Data all_artists{Parsnip::Data::List};
            for (const auto &artist : artists) {
                all_artists.push_back (artist.second->persist());
            }

            Parsnip::Data all_playlists{Parsnip::Data::List};
            for (const auto &playlist : playlists) {
                assert (playlist.second->selector.canPersist());
                all_playlists.push_back (playlist.second->persist());
            }

            Parsnip::Data media
                    = {Parsnip::Data::Dictionary, MusicStorage::MediaArtists, std::move (all_artists)};
            persist (media);

            Parsnip::Data doc{Parsnip::Data::Dictionary,
                                    MusicStorage::LibraryMedia,
                                    std::move (media),
                                    MusicStorage::LibraryPlaylists,
                                    std::move (all_playlists)};
            return carefullyWriteFile (filename, doc);
        } catch (const std::exception &e) {
            flog (LOG_WHERE (LOG_ERROR), "Could not serialize user data: ", e.what());
            return false;
        }
    }

    /** Restore the library's index, playlists, seeds and match
        criteria from a file.
        @param filename The name of the file from which to restore.
        @return True, or throws an exception. */
    bool Library::restoreIndexFromFile (const std::string &filename) {
        const Parsnip::Data index = retrieveJsonFile (filename);

        // Restore the regular tree
        MusicAutoReleasePool pool;
        for (const auto &artist : index[MusicStorage::LibraryMedia][MusicStorage::MediaArtists]) {
            auto newartist = artists.addOrGetItem (artist, this, MusicStorage::ArtistName, MusicStorage::ArtistId);
            if (!newartist)
                continue;
            for (const auto &album : artist[MusicStorage::ArtistAlbums]) {
                auto newalbum = albums.addOrGetItem (album, newartist, MusicStorage::AlbumName, MusicStorage::AlbumId);
                if (!newalbum)
                    continue;
                for (const auto &track : album[MusicStorage::AlbumSongs]) {
                    songs.addOrGetItem (track, newalbum, MusicStorage::SongName, MusicStorage::SongId);
                }
            }
        }
        // Restore compilations
        for (const auto &artist : index[MusicStorage::LibraryMedia][MusicStorage::MediaArtists]) {
            for (const auto &album : artist[MusicStorage::ArtistAlbums]) {
                if (isCompilationAlbum (album)) {
                    for (const auto &track : album[MusicStorage::AlbumSongs]) {
                        auto thesong = songs.getById (track, MusicStorage::SongId);
                        assert (thesong);
                        // Check if compilation song has an artist; if so, set it up.
                        if (track.contains (MusicStorage::ArtistId)) {
                            auto trackartist = artists.getById (track, MusicStorage::ArtistId);
                            if (!trackartist)
                                flog (LOG_WHERE (LOG_WARNING), "Could not find song artist for ", (*thesong)());
                            else
                                thesong->artist (trackartist);
                        }
                    }
                }
            }
        }
        // Restore the playlists
        for (const auto &playlist : index[MusicStorage::LibraryPlaylists]) {
            playlists.addOrGetItem (playlist, this, MusicStorage::PlaylistName, MusicStorage::PlaylistId);
        }
        return true;
    }

}  // namespace MusicLibrary
