///
/// Caches for music thingie types.
///	@file		musiccache.h - pianod
///	@author		Perette Barella
///	@date		2020-03-19
///	@copyright	Copyright (c) 2016-2020 Devious Fish. All rights reserved.
///

#pragma once

#include <retainer.h>

class ThingiePool;

/** Structure for tracking pool objects and their status for ThingiePool.
    @see ThingiePool. */
struct ThingieCache {
    Retainer <MusicThingie *> item;
    mutable time_t expiration = 0;
    void extend (int seconds = 600) const;
};


struct ThingiePoolParameters {
    int initial_duration {3600}; ///< Retain new or reinserted items for at least this amount of times.
    int reprieve_duration {900}; ///< Retain in-use objects at least this much longer.
    int minimum_retained_items {4000}; ///< Don't purge if we have less than this many.
    int maximum_retained_items {4800}; ///< If we hit this many items, try purging before adding.
    int purge_interval {307}; ///< Schedule periodic purges at this interval.
};


/** A cache/pool that retains all music thingies, mixed,
    and allows them to be retrieved from a single place by ID. */
class ThingiePool : protected std::unordered_map <std::string, ThingieCache> {
private:
    ThingiePoolParameters settings;
    int threshold {100};
    time_t next_purge { time (nullptr) + 4000 };
    time_t oldest_entry {time (nullptr)};

    void purge (void);
public:
    ThingiePool ();
    ThingiePool (const ThingiePoolParameters &params);
    void setParameters (const ThingiePoolParameters &params);
    void add (MusicThingie *thing);
    void add (const SongList &songs);
    void add (const ThingieList &things);
    MusicThingie *get (const std::string &id);
    ThingieList get (const Filter &filter);
    /// Periodically pare down the cache.
    inline void periodic (void) {
        if (next_purge <= time (nullptr))
            purge();
    }
};

