///
/// Handle commands controlling the overall pianod deamon/service.
/// Manages "rooms": separate services each with its own audio etc.
/// @file       servicemanager.cpp - pianod project
/// @author     Perette Barella
/// @date       Initial: 2012-03-10.  C++ Rework: 2014-10-27.
/// @copyright  Copyright 2012–2016 Devious Fish. All rights reserved.
///

#include <config.h>

#include <cstdio>
#include <cassert>
#include <ctime>

#include <functional>

#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#endif

#include <football.h>

#include "fundamentals.h"
#include "sources.h"
#include "connection.h"
#include "engine.h"
#include "response.h"
#include "user.h"
#include "users.h"
#include "servicemanager.h"
#include "mediamanager.h"
#include "audiooptionsparser.h"

/// Request all services (rooms) close so pianod can shutdown.
/// @param immediate If true, aborts playback in all rooms.
void ServiceManager::shutdown (bool immediate) {
    if (!shutdown_pending) {
        shutdown_pending = true;
        for (auto service : *this) {
            service.second->audioEngine()->shutdown (immediate);
        }
        broadcast (Response (F_SHUTDOWN, "Shutdown pending."));
    }
}

/// Persist data before shutdown
bool ServiceManager::flush(void) {
    bool status = user_manager->persist ();
    status = media_manager->flush() && status;
    return status;
}

/** Perform periodic duties, and determine when next ones are due.
    All sources, the audio engines (rooms) and users are invoked,
    allowing them to perform periodic duties (freeing resources,
    persisting data, background tasks). */
float ServiceManager::periodic (void) {
    float next_request = A_LONG_TIME;
    for (auto service : *this) {
        float next_req = service.second->engine->periodic();
        if (next_req < next_request) next_request = next_req;
    }
    // Defer media unit/user periodics if the audio engines want attention soon.
    if (next_request > 3) {
        float next_req = media_manager->periodic();
        if (next_req < next_request) next_request = next_req;
        next_req = user_manager->periodic();
        if (next_req < next_request) next_request = next_req;
    }
    // Deal with WAIT FOR event timeouts.
    time_t now = time (nullptr);
    if (now >= WaitEvent::nextTimeout) {
        WaitEvent::nextTimeout = FAR_FUTURE;
        for (auto service : *this) {
            for (auto conn : *(service.second)) {
                conn->checkTimeouts ();
            }
        }
    }
    if (WaitEvent::nextTimeout != FAR_FUTURE) {
        now = time (nullptr);
        if (WaitEvent::nextTimeout - now < next_request) {
            next_request = WaitEvent::nextTimeout - now;
        }
    }

    return (shutdown_pending ? 1 :
            next_request < 0 ? 0 : next_request);
}

/** Distribute an event to connections waiting on it.
    @param type The type of event.
    @param detail Pointer representing a specific event instance.
    @param reply Status code to report to connections waiting for the event. */
void ServiceManager::event (WaitEvent::Type type, const void *detail, RESPONSE_CODE reply) {
    assert (type != WaitEvent::Type::None);
    for (auto &svc : *this) {
        for (auto conn : *(svc.second)) {
            conn->event (type, detail, reply);
        }
    }
}

/// Send a message to everyone (or a target user) user on all services/rooms
/// @param target Target only this user
void ServiceManager::broadcast (const Response &message, User *target) {
    for (auto &svc : *this) {
        for (auto conn : *(svc.second)) {
            if (target == nullptr || target == conn->user) {
                conn << message;
            }
        }
    }
}

/// Broadcast a message to all services/rooms
void ServiceManager::broadcast (RESPONSE_CODE message, User *target) {
    for (auto &svc : *this) {
        for (auto conn : *(svc.second)) {
            if (target == nullptr || target == conn->user) {
                conn << message;
            }
        }
    }
}


/** Broadcast privileges to all users. */
void ServiceManager::broadcastEffectivePrivileges () {
    for (auto &svc : *this) {
        for (auto conn : *(svc.second)) {
            conn->sendEffectivePrivileges();
        }
    }
}

/** Broadcast privileges to a user.
    @param target The target user, or NULL to target visitors. */
void ServiceManager::broadcastEffectivePrivileges (User *target) {
    for (auto &svc : *this) {
        for (auto conn : *(svc.second)) {
            if (target == conn->user) {
                conn->sendEffectivePrivileges();
            }
        }
    }
}


/** Send events to those waiting on source ready, either a specific source,
    any source, or all sources.
    @param source The source that is transitioning from pending to ready or dead.
    @param result The status of the source's transition. */
void ServiceManager::sendSourceReadyEvents (const Media::Source *source,
                                            RESPONSE_CODE result) {
    // Waiting on specific source
    event (WaitEvent::Type::SourceReady, source, result);

    // Waiting on any source)
    event (WaitEvent::Type::SourceReady, media_manager, result);

    // Waiting on all sources ready
    if (!media_manager->areSourcesPending ()) {
        service_manager->event (WaitEvent::Type::SourceReady,
                               nullptr,
                               media_manager->areSourcesReady () ? S_OK : E_NAK);
    }
}


/** Prepare for a new source by alerting auto engines of it. */
void ServiceManager::sourceReady (const Media::Source *source) {
    sendSourceReadyEvents (source, S_OK);
}


/** Handle a source going offline by removing it from use.  Unlike
 invalidating sources, however, historical and queue references
 remain. */
void ServiceManager::sourceOffline (const Media::Source *source) {
    for (auto svc : *this) {
        // Remove source from any connections on the service
        for (auto conn : *(svc.second)) {
            if (conn->source() == source) {
                conn->source (media_manager);
            }
        }
    }
}


/** Handle a source being deleted. */
void ServiceManager::sourceRemoved (const Media::Source *source) {
    sendSourceReadyEvents (source, E_NAK);
}

/** Add a room name/service pair to our list.
    @param name The name of the room, so users can select it.
    @param audio Audio settings (presumably a certain audio out) for the room.
    @param options Football options for the room's service.
    Each room gets a separate service.*/
PianodService *ServiceManager::createRoom (const std::string &name, const AudioSettings &audio, FB_SERVICE_OPTIONS &options) {
    options.queue_size = 5;
    options.greeting_mode = FB_GREETING_ALLOW;
    options.name = (char *) name.c_str();
    PianodService *service = nullptr;
    try {
        service = new PianodService (options, name, master_service);
        service->engine = new AudioEngine (service, audio);


        // Add in the various command sets.
        service->addCommands (this);
        service->engine->registerWithInterpreter (service);

        register_user_commands (service);
        register_media_manager_commands (service);
        Sources::registerCommands (service);

        auto result = insert (make_pair (name, service));
        if (result.second) {
            if (!master_service) master_service = service;
            return service;
        }
    } catch (...) {
        // Continue on
    }
    if (service) delete service;
    return nullptr;
}


/** Remove a room.  Triggers a flush when the last room is removed.
    This is a callback function, invoked when Football calls the
    serviceShutdown() method.
    @param service The room to remove. */
void ServiceManager::removeRoom (PianodService *service) {
    for (iterator it = begin(); it != end(); it++) {
        if (it->second == service) {
            erase (it);
            if (empty())
                flush();
            return;
        }
    }
    assert (0);
    return;
}



/** Determine if a user is authenticated an connected in any room.
    @param user The user to look for.
    @return True if the user is online, false otherwise. */
bool ServiceManager::userIsOnline (const User *user) {
    for (auto const &svc : *this) {
        if (user->online (*svc.second)) return true;
    }
    return false;
}



static const FB_PARSE_DEFINITION statementList[] = {
    { NOP,				"# ..." },							// Comment
    { HELP,				"help [{command}]" },				// Request help
    { YELL,				"yell {announcement}" },			// Broadcast to all connected terminals
    { QUIT,				"quit" },							// Logoff terminal
    { SHUTDOWN,			"shutdown" },                       // Shutdown the player and quit
    { NEWROOM,          "room create {room} [{audio_options}] ..." }, // Create a room with audio options
    { DELETEROOM,       "room delete {room} [now]" },       // Delete a room.
    { CHOOSEROOM,       "room enter {room}" },              // Switch rooms/audio output channel
    { ROOMINFO,         "room <info|information> {room}" }, // Query room information
    { INROOMEXEC,       "in room {room} [{command}] ..." }, // Perform an action in a different room.
    { LISTROOMS,        "room list" },                     // Get a list of rooms
    { SHOWUSERACTIONS,	"announce user actions <switch:on|off>" },	// Whether to broadcast events 
    { SETLOGGINGFLAGS,	"set [football] logging flags {#logging-flags:0x0-0xffffff}" },
    { SYNC,             "sync [userdata]" },
    { UPTIME,           "uptime" }, // Unofficial
    { CMD_INVALID,      NULL }
};

const FB_PARSE_DEFINITION *ServiceManager::statements (void) {
    return statementList;
};

bool ServiceManager::hasPermission (PianodConnection &conn, SERVICECOMMAND command) {
    switch (command) {
        case NOP:
        case HELP:
        case QUIT:
            return true;
        case YELL:
        case CHOOSEROOM:
        case INROOMEXEC:
        case LISTROOMS:
        case UPTIME:
        case SYNC:
            return conn.haveRank (Rank::Listener);
        default:
            return conn.haveRank (Rank::Administrator);
    }
};

void ServiceManager::handleCommand (PianodConnection &conn, SERVICECOMMAND command) {
    // Command handlers, in alphabetical order. 
    switch (command) {
        case HELP:
        {
            const Football::HelpList help (conn.getHelp (conn.argv("command")));
            if (help.empty()) {
                conn << E_NOTFOUND;
            } else {
                conn << S_DATA;
                for (auto item : help) {
                    conn << Response (I_INFO, item);
                }
                conn << S_DATA_END;
            }
            return;
        }
        case QUIT:
            conn << S_SIGNOFF;
            conn.close();
            return;
        case SHOWUSERACTIONS:
            conn.broadcastUserActions (conn.argvEquals ("switch", "on"));
            // FALLTHRU
        case NOP:
            conn << S_OK;
            return;
        case SHUTDOWN:
            // Commence a server shutdown, which will take effect after the current song. 
            conn.announce (A_SHUTDOWN);
            shutdown (false);
            conn << S_OK;
            return;
        case YELL:
            conn.announce (V_YELL, conn.argv("announcement"));
            conn << S_OK;
            return;
        case CHOOSEROOM:
        {
            iterator new_room = find (conn.argv ("room"));
            if (new_room == end()) {
                conn << E_NOTFOUND;
            } else {
                PianodService *old_room = &conn.service();
                if (conn.transfer (new_room->second, true)) {
                    if (old_room != new_room->second) {
                        old_room->usersChangedNotification();
                        new_room->second->usersChangedNotification();
                    }
                } else {
                    conn << E_NAK;
                };
            }
            return;
        }
        case INROOMEXEC:
        {
            PianodService *original_room = &conn.service();
            iterator item = find (conn.argv ("room"));
            if (item == end()) {
                conn << E_NOTFOUND;
            } else if (!conn.transfer (item->second, true)) {
                conn << E_NAK;
            } else {
                conn.reinterpret("command");
                kept_assert (conn.transfer (item->second, original_room));
            }
            return;
        }
        case DELETEROOM:
        {
            iterator item = find (conn ["room"]);
            if (item == end()) {
                conn << E_NOTFOUND;
            } else if (conn.argvEquals ("room", "pianod")) {
                conn << Response (E_UNSUPPORTED, "Cannot remove initial room.");
            } else {
                item->second->audioEngine()->shutdown (conn.argv ("now"));
                conn << S_OK;
            }
            return;
        }
        case ROOMINFO:
        {
            iterator item = find (conn ["room"]);
            if (item == end()) {
                conn << E_NOTFOUND;
            } else {
                const AudioSettings &audio = item->second->audioEngine()->audioSettings();
                conn << S_DATA
                << Response (I_INFO, "library: " + audio.output_library)
                << Response (I_INFO, "device: " + audio.output_device)
                << Response (I_INFO, "driver: " + audio.output_driver)
                << Response (I_INFO, "options: " + audio.output_options)
                << Response (I_INFO, "server: " + audio.output_server)
                << Response (I_INFO, "options: " + audio.output_options)
                << Response (I_INFO, "crossfade level: " + std::to_string (audio.crossfade_level))
                << Response (I_INFO, "crossfade time: " + std::to_string (audio.crossfade_time))
                << Response (I_INFO, "options: " + audio.output_options)
                << S_DATA_END;
            }
            return;
        }
        case NEWROOM:
        {
            if (shutdown_pending) {
                conn << Response (E_NAK, "Shutdown is pending");
                return;
            }
            FB_SERVICE_OPTIONS options = { };
            options.transfer_only = true;
            AudioSettings audio = { };
            if (AudioOptions::parser.interpret (conn.argvFrom ("audio_options"), audio, conn) == FB_PARSE_SUCCESS) {
                conn << (createRoom (conn.argv ("room"), audio, options) ? S_OK : E_NAK);
            }
            return;
        }
        case LISTROOMS:
            if (!empty())
                conn << S_DATA;
            for (auto const &room : *this) {
                conn << Response (I_ROOM, room.second->roomName());
            }
            conn << S_DATA_END;
            return;
        case SETLOGGINGFLAGS:
        {
            long flags = strtol (conn.argv ("logging-flags"), NULL, 0);
            if (conn.argv ("football")) {
                fb_set_logging (flags, NULL);
            } else {
                set_logging (flags);
            }
            conn << S_OK;
            return;
        }
        case SYNC:
        {
            bool status = (conn ["userdata"] ? user_manager->persist() : flush());
            conn << (status ? S_OK : E_NAK);
            return;
        }
        case UPTIME:
        {
            time_t now = time (nullptr);;
            time_t duration = now - startup_time;
            int secs = duration % 60;
            duration /= 60;
            int mins = duration % 60;
            duration /= 60;
            int hours = duration % 24;
            duration /= 24;
            char *since = ctime (&startup_time);
            if (!since)
                throw CommandError (E_RESOURCE);
            conn << S_DATA;
            conn.printf ("%03d %s: Up since %24.24s (%dd+%02d:%02d:%02d)\n",
                         I_INFO, ResponseText (I_INFO), since,
                         duration, hours, mins, secs);
#if defined(HAVE_GETRUSAGE) && defined (HAVE_SYS_RESOURCE_H)
            struct rusage usage;
            if (getrusage (RUSAGE_SELF, &usage) == 0) {
                conn.printf ("%03d %s: Compute time=%dm%02d.%02ds (user), %dm%02d.%02ds (system)\n",
                             I_INFO, ResponseText (I_INFO),
                             usage.ru_utime.tv_sec / 60,
                             usage.ru_utime.tv_sec % 60,
                             usage.ru_utime.tv_usec / 10000,
                             usage.ru_stime.tv_sec / 60,
                             usage.ru_stime.tv_sec % 60,
                             usage.ru_stime.tv_usec / 10000);
            }
#endif
            conn.printf ("%03d %s: Current local time %s",
                         I_INFO, ResponseText (I_INFO),
                         ctime (&now));
            conn << S_DATA_END;
            return;
        }
        default:
            flog (LOG_WHERE (LOG_WARNING), "Unimplemented command ", command);
            break;
    }
}


ServiceManager::ServiceManager () {
    Media::Manager::Callbacks callbacks;
    callbacks.sourceReady = bind (&ServiceManager::sourceReady, this, std::placeholders::_1);
    callbacks.sourceOffline = bind (&ServiceManager::sourceOffline, this, std::placeholders::_1);
    callbacks.sourceRemoved = bind (&ServiceManager::sourceRemoved, this, std::placeholders::_1);
    media_manager->callback.subscribe (this, callbacks);
}

ServiceManager::~ServiceManager () {
    media_manager->callback.unsubscribe (this);
}


/* Global */ ServiceManager *service_manager {nullptr};


